/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.

*/

#ifndef _FONT_H
#define _FONT_H

/* IMPORTANT NOTE !
   Do not edit font.h, edit font.h.in instead !
   font.h gets created from font.h.in by configure !
*/

#ifdef HAVE_CONFIG
#include "config.h"
#endif

#include <freetype/freetype.h>

#include <string>

#include "surface.h"
#include "color.h"
#include "point.h"

namespace uta {

/**FreeType based surface font.
   This class uses the benefit of truetype fonts 
   (antialiasing, *many* free fonts) as well as the speed gain
   of surface fonts. When creating an object of Font the surfaces 
   for the characters are all rendered in best quality. 
   When the font-object is asked
   to create a surface from a string it just blits these character surfaces,
   which is quite fast.
*/
class Font 
{
 public:

  ///
  static Color defaultFontColor;
  ///
  static Color defaultFadeColor;

  /**Constructor.
     fontfilename has to be the name of a truetype font file.
     The font colors are read from Resources if available.
  */	
  Font(const std::string& fontfilename, int ptsize = 12);

   /**Constructor.
     fontfilename has to be the name of a truetype font file.
  */
  Font(const std::string& fontfilename, 
	int ptsize, const Color &fontCol, const Color &backCol);
   
  ///
  Font(const Font& );
  ///
  Font();

  ///this assumes a surface where all chars are drawn in one row
  Font(const Surface* img);
  ///
  ~Font();
  
  ///
  void create (const std::string& fontfilename, 
	       int ptsize = 12,
	       const Color &fontCol = Font::defaultFontColor,
	       const Color &backCol = Font::defaultFadeColor);

  /** return surfacerepresentation of a char.
      if no font is loaded, the surface returned is empty.
  */
  Surface *getChar(unsigned char ) const;

  /**return bitmaprepresentation of a string.
     if no font is loaded, the bitmap returned is empty.
  */
  Surface *getString(const std::string & ) const;
  ///
  int blitString(const std::string&, Surface* target, const Point& pos) const;
  ///
  int blitString(const std::string&, Surface* target, const Point& pos,
		  const std::vector<Color>& pal) const;
	
  ///return size of given character
  Point getSize(unsigned char c) const;

  ///maximum height of this font in pixels
  int getHeight() const; 

  ///
  const Color& fontColor() const { return fontColor_; };
  ///
  const Color& fadeColor() const { return backColor_; };

 protected:  

  ///
  bool initialized_;
  ///
  Surface* getCharPtr(unsigned char ) const;
  ///
  Surface* renderChar(unsigned char );
  ///
  void createPalette();
  ///
  void createFont();

  /** @name static members*/
  //@{
  ///truetype engine 
  static TT_Engine engine;
  ///
  static int engineLock;
  //@}

  ///size in points
  int ptsize_;
  ///max height in pixels
  int height_;
  ///
  Color fontColor_;
  ///
  Color backColor_;  
  ///
  Surface** surfaceTable_;

  /** @name FreeType related members*/
  //@{
  ///
  TT_Face face_;
  ///
  TT_Instance instance_;
  ///
  TT_Face_Properties properties_;
  ///
  TT_Glyph glyph_;
  ///
  Color *palette_;
  ///
  TT_Glyph_Metrics metric_;
  ///
  TT_UShort platform_;
  ///
  TT_UShort encoding_;
  ///
  TT_CharMap cmap_;
  ///
  TT_Instance_Metrics imetrics_;
  ///
  TT_Raster_Map rastermap_;   
  //@}

};



}


#endif



  
