#!/usr/bin/env perl
#
# ====================================================================
# Written by Andy Polyakov <appro@fy.chalmers.se> for the OpenSSL
# project. Rights for redistribution and usage in source and binary
# forms are granted according to the OpenSSL license.
# ====================================================================
#
# Version 2.0.
#
# You might fail to appreciate this module performance from the first
# try. If compared to "vanilla" linux-ia32-icc target, i.e. considered
# to be *the* best Intel C compiler without -KPIC, performance appears
# to be virtually identical... But try to re-configure with shared
# library support... Aha! Intel compiler "suddenly" lags behind by 30%
# [on P4, more on others]:-) And if compared to position-independent
# code generated by GNU C, this code performs *more* than *twice* as
# fast! Yes, all this buzz about PIC means that unlike other hand-
# coded implementations, this one was explicitly designed to be safe
# to use even in shared library context... This also means that this
# code isn't necessarily absolutely fastest "ever," because in order
# to achieve position independence an extra register has to be
# off-loaded to stack, which affects the benchmark result.
#
# Special note about instruction choice. Do you recall RC4_INT code
# performing poorly on P4? It might be the time to figure out why.
# RC4_INT code implies effective address calculations in base+offset*4
# form. Trouble is that it seems that offset scaling turned to be
# critical path... At least eliminating scaling resulted in 2.8x RC4
# performance improvement [as you might recall]. As AES code is hungry
# for scaling too, I [try to] avoid the latter by favoring off-by-2
# shifts and masking the result with 0xFF<<2 instead of "boring" 0xFF.
#
# As was shown by Dean Gaudet <dean@arctic.org>, the above note turned
# void. Performance improvement with off-by-2 shifts was observed on
# intermediate implementation, which was spilling yet another register
# to stack... Final offset*4 code below runs just a tad faster on P4,
# but exhibits up to 10% improvement on other cores.
#
# Second version is "monolithic" replacement for aes_core.c, which in
# addition to AES_[de|en]crypt implements AES_set_[de|en]cryption_key.
# This made it possible to implement little-endian variant of the
# algorithm without modifying the base C code. Motivating factor for
# the undertaken effort was that it appeared that in tight IA-32
# register window little-endian flavor could achieve slightly higher
# Instruction Level Parallelism, and it indeed resulted in up to 15%
# better performance on most recent -archs...
#
# Current ECB performance numbers for 128-bit key in cycles per byte
# [measure commonly used by AES benchmarkers] are:
#
#		small footprint		fully unrolled
# P4[-3]	23[24]			22[23]
# AMD K8	19			18
# PIII		26(*)			23
# Pentium	63(*)			52
#
# (*)	Performance difference between small footprint code and fully
#	unrolled in more commonly used CBC mode is not as big, 7% for
#	PIII and 15% for Pentium, which I consider tolerable.

push(@INC,"perlasm","../../perlasm");
require "x86asm.pl";

&asm_init($ARGV[0],"aes-586.pl",$ARGV[$#ARGV] eq "386");

$small_footprint=1;	# $small_footprint=1 code is ~5% slower [on
			# recent -archs], but ~5 times smaller!
			# I favor compact code, because it minimizes
			# cache contention...
$vertical_spin=0;	# shift "verticaly" defaults to 0, because of
			# its proof-of-concept status, see below...

$s0="eax";
$s1="ebx";
$s2="ecx";
$s3="edx";
$key="esi";
$acc="edi";

if ($vertical_spin) {
	# I need high parts of volatile registers to be accessible...
	$s1="esi";	$key="ebx";
	$s2="edi";	$acc="ecx";
}
# Note that there is no decvert(), as well as last encryption round is
# performed with "horizontal" shifts. This is because this "vertical"
# implementation [one which groups shifts on a given $s[i] to form a
# "column," unlike "horizontal" one, which groups shifts on different
# $s[i] to form a "row"] is work in progress. It was observed to run
# few percents faster on Intel cores, but not AMD. On AMD K8 core it's
# whole 12% slower:-( So we face a trade-off... Shall it be resolved
# some day? Till then the code is considered experimental and by
# default remains dormant...

sub encvert()
{ my ($te,@s) = @_;
  my $v0 = $acc, $v1 = $key;

	&mov	($v0,$s[3]);				# copy s3
	&mov	(&DWP(0,"esp"),$s[2]);			# save s2
	&mov	($v1,$s[0]);				# copy s0
	&mov	(&DWP(4,"esp"),$s[1]);			# save s1

	&movz	($s[2],&HB($s[0]));
	&and	($s[0],0xFF);
	&mov	($s[0],&DWP(1024*0,$te,$s[0],4));	# s0>>0
	&shr	($v1,16);
	&mov	($s[3],&DWP(1024*1,$te,$s[2],4));	# s0>>8
	&movz	($s[1],&HB($v1));
	&and	($v1,0xFF);
	&mov	($s[2],&DWP(1024*2,$te,$v1,4));		# s0>>16
	 &mov	($v1,$v0);
	&mov	($s[1],&DWP(1024*3,$te,$s[1],4));	# s0>>24

	&and	($v0,0xFF);
	&xor	($s[3],&DWP(1024*0,$te,$v0,4));		# s3>>0
	&movz	($v0,&HB($v1));
	&shr	($v1,16);
	&xor	($s[2],&DWP(1024*1,$te,$v0,4));		# s3>>8
	&movz	($v0,&HB($v1));
	&and	($v1,0xFF);
	&xor	($s[1],&DWP(1024*2,$te,$v1,4));		# s3>>16
	 &mov	($v1,&DWP(0,"esp"));			# restore s2
	&xor	($s[0],&DWP(1024*3,$te,$v0,4));		# s3>>24

	&mov	($v0,$v1);
	&and	($v1,0xFF);
	&xor	($s[2],&DWP(1024*0,$te,$v1,4));		# s2>>0
	&movz	($v1,&HB($v0));
	&shr	($v0,16);
	&xor	($s[1],&DWP(1024*1,$te,$v1,4));		# s2>>8
	&movz	($v1,&HB($v0));
	&and	($v0,0xFF);
	&xor	($s[0],&DWP(1024*2,$te,$v0,4));		# s2>>16
	 &mov	($v0,&DWP(4,"esp"));			# restore s1
	&xor	($s[3],&DWP(1024*3,$te,$v1,4));		# s2>>24

	&mov	($v1,$v0);
	&and	($v0,0xFF);
	&xor	($s[1],&DWP(1024*0,$te,$v0,4));		# s1>>0
	&movz	($v0,&HB($v1));
	&shr	($v1,16);
	&xor	($s[0],&DWP(1024*1,$te,$v0,4));		# s1>>8
	&movz	($v0,&HB($v1));
	&and	($v1,0xFF);
	&xor	($s[3],&DWP(1024*2,$te,$v1,4));		# s1>>16
	 &mov	($key,&DWP(12,"esp"));			# reincarnate v1 as key
	&xor	($s[2],&DWP(1024*3,$te,$v0,4));		# s1>>24
}

sub encstep()
{ my ($i,$te,@s) = @_;
  my $tmp = $key;
  my $out = $i==3?$s[0]:$acc;

	# lines marked with #%e?x[i] denote "reordered" instructions...
	if ($i==3)  {	&mov	($key,&DWP(12,"esp"));		}##%edx
	else        {	&mov	($out,$s[0]);
			&and	($out,0xFF);			}
	if ($i==1)  {	&shr	($s[0],16);			}#%ebx[1]
	if ($i==2)  {	&shr	($s[0],24);			}#%ecx[2]
			&mov	($out,&DWP(1024*0,$te,$out,4));

	if ($i==3)  {	$tmp=$s[1];				}##%eax
			&movz	($tmp,&HB($s[1]));
			&xor	($out,&DWP(1024*1,$te,$tmp,4));

	if ($i==3)  {	$tmp=$s[2]; &mov ($s[1],&DWP(0,"esp"));	}##%ebx
	else        {	&mov	($tmp,$s[2]);
			&shr	($tmp,16);			}
	if ($i==2)  {	&and	($s[1],0xFF);			}#%edx[2]
			&and	($tmp,0xFF);
			&xor	($out,&DWP(1024*2,$te,$tmp,4));

	if ($i==3)  {	$tmp=$s[3]; &mov ($s[2],&DWP(4,"esp"));	}##%ecx
	elsif($i==2){	&movz	($tmp,&HB($s[3]));		}#%ebx[2]
	else        {	&mov	($tmp,$s[3]); 
			&shr	($tmp,24)			}
			&xor	($out,&DWP(1024*3,$te,$tmp,4));
	if ($i<2)   {	&mov	(&DWP(4*$i,"esp"),$out);	}
	if ($i==3)  {	&mov	($s[3],$acc);			}
			&comment();
}

sub enclast()
{ my ($i,$te,@s)=@_;
  my $tmp = $key;
  my $out = $i==3?$s[0]:$acc;

	if ($i==3)  {	&mov	($key,&DWP(12,"esp"));		}##%edx
	else        {	&mov	($out,$s[0]);			}
			&and	($out,0xFF);
	if ($i==1)  {	&shr	($s[0],16);			}#%ebx[1]
	if ($i==2)  {	&shr	($s[0],24);			}#%ecx[2]
			&mov	($out,&DWP(1024*0,$te,$out,4));
			&and	($out,0x000000ff);

	if ($i==3)  {	$tmp=$s[1];				}##%eax
			&movz	($tmp,&HB($s[1]));
			&mov	($tmp,&DWP(0,$te,$tmp,4));
			&and	($tmp,0x0000ff00);
			&xor	($out,$tmp);

	if ($i==3)  {	$tmp=$s[2]; &mov ($s[1],&DWP(0,"esp"));	}##%ebx
	else        {	mov	($tmp,$s[2]);
			&shr	($tmp,16);			}
	if ($i==2)  {	&and	($s[1],0xFF);			}#%edx[2]
			&and	($tmp,0xFF);
			&mov	($tmp,&DWP(0,$te,$tmp,4));
			&and	($tmp,0x00ff0000);
			&xor	($out,$tmp);

	if ($i==3)  {	$tmp=$s[3]; &mov ($s[2],&DWP(4,"esp"));	}##%ecx
	elsif($i==2){	&movz	($tmp,&HB($s[3]));		}#%ebx[2]
	else        {	&mov	($tmp,$s[3]);
			&shr	($tmp,24);			}
			&mov	($tmp,&DWP(0,$te,$tmp,4));
			&and	($tmp,0xff000000);
			&xor	($out,$tmp);
	if ($i<2)   {	&mov	(&DWP(4*$i,"esp"),$out);	}
	if ($i==3)  {	&mov	($s[3],$acc);			}
}

# void AES_encrypt (const void *inp,void *out,const AES_KEY *key);
&public_label("AES_Te");
&function_begin("AES_encrypt");
	&mov	($acc,&wparam(0));		# load inp
	&mov	($key,&wparam(2));		# load key

        &call   (&label("pic_point"));          # make it PIC!
	&set_label("pic_point");
        &blindpop("ebp");
        &lea    ("ebp",&DWP(&label("AES_Te")."-".&label("pic_point"),"ebp"));

	# allocate aligned stack frame
	&mov	($s0,"esp");
	&sub	("esp",20);
	&and	("esp",-16);

	&mov	(&DWP(12,"esp"),$key);		# save key
	&mov	(&DWP(16,"esp"),$s0);		# save %esp

	&mov	($s0,&DWP(0,$acc));		# load input data
	&mov	($s1,&DWP(4,$acc));
	&mov	($s2,&DWP(8,$acc));
	&mov	($s3,&DWP(12,$acc));

	&xor	($s0,&DWP(0,$key));
	&xor	($s1,&DWP(4,$key));
	&xor	($s2,&DWP(8,$key));
	&xor	($s3,&DWP(12,$key));

	&mov	($acc,&DWP(240,$key));		# load key->rounds

	if ($small_footprint) {
	    &lea	($acc,&DWP(-2,$acc,$acc));
	    &lea	($acc,&DWP(0,$key,$acc,8));
	    &mov	(&DWP(8,"esp"),$acc);	# end of key schedule
	    &align	(4);
	    &set_label("loop");
		if ($vertical_spin) {
		    &encvert("ebp",$s0,$s1,$s2,$s3);
		} else {
		    &encstep(0,"ebp",$s0,$s1,$s2,$s3);
		    &encstep(1,"ebp",$s1,$s2,$s3,$s0);
		    &encstep(2,"ebp",$s2,$s3,$s0,$s1);
		    &encstep(3,"ebp",$s3,$s0,$s1,$s2);
		}
		&add	($key,16);			# advance rd_key
		&xor	($s0,&DWP(0,$key));
		&xor	($s1,&DWP(4,$key));
		&xor	($s2,&DWP(8,$key));
		&xor	($s3,&DWP(12,$key));
	    &cmp	($key,&DWP(8,"esp"));
	    &mov	(&DWP(12,"esp"),$key);
	    &jb		(&label("loop"));
	}
	else {
	    &cmp	($acc,10);
	    &jle	(&label("10rounds"));
	    &cmp	($acc,12);
	    &jle	(&label("12rounds"));

	&set_label("14rounds");
	    for ($i=1;$i<3;$i++) {
		if ($vertical_spin) {
		    &encvert("ebp",$s0,$s1,$s2,$s3);
		} else {
		    &encstep(0,"ebp",$s0,$s1,$s2,$s3);
		    &encstep(1,"ebp",$s1,$s2,$s3,$s0);
		    &encstep(2,"ebp",$s2,$s3,$s0,$s1);
		    &encstep(3,"ebp",$s3,$s0,$s1,$s2);
		}
		&xor	($s0,&DWP(16*$i+0,$key));
		&xor	($s1,&DWP(16*$i+4,$key));
		&xor	($s2,&DWP(16*$i+8,$key));
		&xor	($s3,&DWP(16*$i+12,$key));
	    }
	    &add	($key,32);
	    &mov	(&DWP(12,"esp"),$key);		# advance rd_key
	&set_label("12rounds");
	    for ($i=1;$i<3;$i++) {
		if ($vertical_spin) {
		    &encvert("ebp",$s0,$s1,$s2,$s3);
		} else {
		    &encstep(0,"ebp",$s0,$s1,$s2,$s3);
		    &encstep(1,"ebp",$s1,$s2,$s3,$s0);
		    &encstep(2,"ebp",$s2,$s3,$s0,$s1);
		    &encstep(3,"ebp",$s3,$s0,$s1,$s2);
		}
		&xor	($s0,&DWP(16*$i+0,$key));
		&xor	($s1,&DWP(16*$i+4,$key));
		&xor	($s2,&DWP(16*$i+8,$key));
		&xor	($s3,&DWP(16*$i+12,$key));
	    }
	    &add	($key,32);
	    &mov	(&DWP(12,"esp"),$key);		# advance rd_key
	&set_label("10rounds");
	    for ($i=1;$i<10;$i++) {
		if ($vertical_spin) {
		    &encvert("ebp",$s0,$s1,$s2,$s3);
		} else {
		    &encstep(0,"ebp",$s0,$s1,$s2,$s3);
		    &encstep(1,"ebp",$s1,$s2,$s3,$s0);
		    &encstep(2,"ebp",$s2,$s3,$s0,$s1);
		    &encstep(3,"ebp",$s3,$s0,$s1,$s2);
		}
		&xor	($s0,&DWP(16*$i+0,$key));
		&xor	($s1,&DWP(16*$i+4,$key));
		&xor	($s2,&DWP(16*$i+8,$key));
		&xor	($s3,&DWP(16*$i+12,$key));
	    }
	}

	&add	("ebp",4*1024);			# skip to Te4
	if ($vertical_spin) {
	    # "reincarnate" some registers for "horizontal" spin...
	    &mov	($s1="ebx",$key="esi");
	    &mov	($s2="ecx",$acc="edi");
	}
	&enclast(0,"ebp",$s0,$s1,$s2,$s3);
	&enclast(1,"ebp",$s1,$s2,$s3,$s0);
	&enclast(2,"ebp",$s2,$s3,$s0,$s1);
	&enclast(3,"ebp",$s3,$s0,$s1,$s2);

	&mov	("esp",&DWP(16,"esp"));		# restore %esp
	&add	($key,$small_footprint?16:160);
	&xor	($s0,&DWP(0,$key));
	&xor	($s1,&DWP(4,$key));
	&xor	($s2,&DWP(8,$key));
	&xor	($s3,&DWP(12,$key));

	&mov	($acc,&wparam(1));		# load out
	&mov	(&DWP(0,$acc),$s0);		# write output data
	&mov	(&DWP(4,$acc),$s1);
	&mov	(&DWP(8,$acc),$s2);
	&mov	(&DWP(12,$acc),$s3);

	&pop	("edi");
	&pop	("esi");
	&pop	("ebx");
	&pop	("ebp");
	&ret	();

&set_label("AES_Te",64);	# Yes! I keep it in the code segment!
	&data_word(0xa56363c6, 0x847c7cf8, 0x997777ee, 0x8d7b7bf6);
	&data_word(0x0df2f2ff, 0xbd6b6bd6, 0xb16f6fde, 0x54c5c591);
	&data_word(0x50303060, 0x03010102, 0xa96767ce, 0x7d2b2b56);
	&data_word(0x19fefee7, 0x62d7d7b5, 0xe6abab4d, 0x9a7676ec);
	&data_word(0x45caca8f, 0x9d82821f, 0x40c9c989, 0x877d7dfa);
	&data_word(0x15fafaef, 0xeb5959b2, 0xc947478e, 0x0bf0f0fb);
	&data_word(0xecadad41, 0x67d4d4b3, 0xfda2a25f, 0xeaafaf45);
	&data_word(0xbf9c9c23, 0xf7a4a453, 0x967272e4, 0x5bc0c09b);
	&data_word(0xc2b7b775, 0x1cfdfde1, 0xae93933d, 0x6a26264c);
	&data_word(0x5a36366c, 0x413f3f7e, 0x02f7f7f5, 0x4fcccc83);
	&data_word(0x5c343468, 0xf4a5a551, 0x34e5e5d1, 0x08f1f1f9);
	&data_word(0x937171e2, 0x73d8d8ab, 0x53313162, 0x3f15152a);
	&data_word(0x0c040408, 0x52c7c795, 0x65232346, 0x5ec3c39d);
	&data_word(0x28181830, 0xa1969637, 0x0f05050a, 0xb59a9a2f);
	&data_word(0x0907070e, 0x36121224, 0x9b80801b, 0x3de2e2df);
	&data_word(0x26ebebcd, 0x6927274e, 0xcdb2b27f, 0x9f7575ea);
	&data_word(0x1b090912, 0x9e83831d, 0x742c2c58, 0x2e1a1a34);
	&data_word(0x2d1b1b36, 0xb26e6edc, 0xee5a5ab4, 0xfba0a05b);
	&data_word(0xf65252a4, 0x4d3b3b76, 0x61d6d6b7, 0xceb3b37d);
	&data_word(0x7b292952, 0x3ee3e3dd, 0x712f2f5e, 0x97848413);
	&data_word(0xf55353a6, 0x68d1d1b9, 0x00000000, 0x2cededc1);
	&data_word(0x60202040, 0x1ffcfce3, 0xc8b1b179, 0xed5b5bb6);
	&data_word(0xbe6a6ad4, 0x46cbcb8d, 0xd9bebe67, 0x4b393972);
	&data_word(0xde4a4a94, 0xd44c4c98, 0xe85858b0, 0x4acfcf85);
	&data_word(0x6bd0d0bb, 0x2aefefc5, 0xe5aaaa4f, 0x16fbfbed);
	&data_word(0xc5434386, 0xd74d4d9a, 0x55333366, 0x94858511);
	&data_word(0xcf45458a, 0x10f9f9e9, 0x06020204, 0x817f7ffe);
	&data_word(0xf05050a0, 0x443c3c78, 0xba9f9f25, 0xe3a8a84b);
	&data_word(0xf35151a2, 0xfea3a35d, 0xc0404080, 0x8a8f8f05);
	&data_word(0xad92923f, 0xbc9d9d21, 0x48383870, 0x04f5f5f1);
	&data_word(0xdfbcbc63, 0xc1b6b677, 0x75dadaaf, 0x63212142);
	&data_word(0x30101020, 0x1affffe5, 0x0ef3f3fd, 0x6dd2d2bf);
	&data_word(0x4ccdcd81, 0x140c0c18, 0x35131326, 0x2fececc3);
	&data_word(0xe15f5fbe, 0xa2979735, 0xcc444488, 0x3917172e);
	&data_word(0x57c4c493, 0xf2a7a755, 0x827e7efc, 0x473d3d7a);
	&data_word(0xac6464c8, 0xe75d5dba, 0x2b191932, 0x957373e6);
	&data_word(0xa06060c0, 0x98818119, 0xd14f4f9e, 0x7fdcdca3);
	&data_word(0x66222244, 0x7e2a2a54, 0xab90903b, 0x8388880b);
	&data_word(0xca46468c, 0x29eeeec7, 0xd3b8b86b, 0x3c141428);
	&data_word(0x79dedea7, 0xe25e5ebc, 0x1d0b0b16, 0x76dbdbad);
	&data_word(0x3be0e0db, 0x56323264, 0x4e3a3a74, 0x1e0a0a14);
	&data_word(0xdb494992, 0x0a06060c, 0x6c242448, 0xe45c5cb8);
	&data_word(0x5dc2c29f, 0x6ed3d3bd, 0xefacac43, 0xa66262c4);
	&data_word(0xa8919139, 0xa4959531, 0x37e4e4d3, 0x8b7979f2);
	&data_word(0x32e7e7d5, 0x43c8c88b, 0x5937376e, 0xb76d6dda);
	&data_word(0x8c8d8d01, 0x64d5d5b1, 0xd24e4e9c, 0xe0a9a949);
	&data_word(0xb46c6cd8, 0xfa5656ac, 0x07f4f4f3, 0x25eaeacf);
	&data_word(0xaf6565ca, 0x8e7a7af4, 0xe9aeae47, 0x18080810);
	&data_word(0xd5baba6f, 0x887878f0, 0x6f25254a, 0x722e2e5c);
	&data_word(0x241c1c38, 0xf1a6a657, 0xc7b4b473, 0x51c6c697);
	&data_word(0x23e8e8cb, 0x7cdddda1, 0x9c7474e8, 0x211f1f3e);
	&data_word(0xdd4b4b96, 0xdcbdbd61, 0x868b8b0d, 0x858a8a0f);
	&data_word(0x907070e0, 0x423e3e7c, 0xc4b5b571, 0xaa6666cc);
	&data_word(0xd8484890, 0x05030306, 0x01f6f6f7, 0x120e0e1c);
	&data_word(0xa36161c2, 0x5f35356a, 0xf95757ae, 0xd0b9b969);
	&data_word(0x91868617, 0x58c1c199, 0x271d1d3a, 0xb99e9e27);
	&data_word(0x38e1e1d9, 0x13f8f8eb, 0xb398982b, 0x33111122);
	&data_word(0xbb6969d2, 0x70d9d9a9, 0x898e8e07, 0xa7949433);
	&data_word(0xb69b9b2d, 0x221e1e3c, 0x92878715, 0x20e9e9c9);
	&data_word(0x49cece87, 0xff5555aa, 0x78282850, 0x7adfdfa5);
	&data_word(0x8f8c8c03, 0xf8a1a159, 0x80898909, 0x170d0d1a);
	&data_word(0xdabfbf65, 0x31e6e6d7, 0xc6424284, 0xb86868d0);
	&data_word(0xc3414182, 0xb0999929, 0x772d2d5a, 0x110f0f1e);
	&data_word(0xcbb0b07b, 0xfc5454a8, 0xd6bbbb6d, 0x3a16162c);
#Te1:
	&data_word(0x6363c6a5, 0x7c7cf884, 0x7777ee99, 0x7b7bf68d);
	&data_word(0xf2f2ff0d, 0x6b6bd6bd, 0x6f6fdeb1, 0xc5c59154);
	&data_word(0x30306050, 0x01010203, 0x6767cea9, 0x2b2b567d);
	&data_word(0xfefee719, 0xd7d7b562, 0xabab4de6, 0x7676ec9a);
	&data_word(0xcaca8f45, 0x82821f9d, 0xc9c98940, 0x7d7dfa87);
	&data_word(0xfafaef15, 0x5959b2eb, 0x47478ec9, 0xf0f0fb0b);
	&data_word(0xadad41ec, 0xd4d4b367, 0xa2a25ffd, 0xafaf45ea);
	&data_word(0x9c9c23bf, 0xa4a453f7, 0x7272e496, 0xc0c09b5b);
	&data_word(0xb7b775c2, 0xfdfde11c, 0x93933dae, 0x26264c6a);
	&data_word(0x36366c5a, 0x3f3f7e41, 0xf7f7f502, 0xcccc834f);
	&data_word(0x3434685c, 0xa5a551f4, 0xe5e5d134, 0xf1f1f908);
	&data_word(0x7171e293, 0xd8d8ab73, 0x31316253, 0x15152a3f);
	&data_word(0x0404080c, 0xc7c79552, 0x23234665, 0xc3c39d5e);
	&data_word(0x18183028, 0x969637a1, 0x05050a0f, 0x9a9a2fb5);
	&data_word(0x07070e09, 0x12122436, 0x80801b9b, 0xe2e2df3d);
	&data_word(0xebebcd26, 0x27274e69, 0xb2b27fcd, 0x7575ea9f);
	&data_word(0x0909121b, 0x83831d9e, 0x2c2c5874, 0x1a1a342e);
	&data_word(0x1b1b362d, 0x6e6edcb2, 0x5a5ab4ee, 0xa0a05bfb);
	&data_word(0x5252a4f6, 0x3b3b764d, 0xd6d6b761, 0xb3b37dce);
	&data_word(0x2929527b, 0xe3e3dd3e, 0x2f2f5e71, 0x84841397);
	&data_word(0x5353a6f5, 0xd1d1b968, 0x00000000, 0xededc12c);
	&data_word(0x20204060, 0xfcfce31f, 0xb1b179c8, 0x5b5bb6ed);
	&data_word(0x6a6ad4be, 0xcbcb8d46, 0xbebe67d9, 0x3939724b);
	&data_word(0x4a4a94de, 0x4c4c98d4, 0x5858b0e8, 0xcfcf854a);
	&data_word(0xd0d0bb6b, 0xefefc52a, 0xaaaa4fe5, 0xfbfbed16);
	&data_word(0x434386c5, 0x4d4d9ad7, 0x33336655, 0x85851194);
	&data_word(0x45458acf, 0xf9f9e910, 0x02020406, 0x7f7ffe81);
	&data_word(0x5050a0f0, 0x3c3c7844, 0x9f9f25ba, 0xa8a84be3);
	&data_word(0x5151a2f3, 0xa3a35dfe, 0x404080c0, 0x8f8f058a);
	&data_word(0x92923fad, 0x9d9d21bc, 0x38387048, 0xf5f5f104);
	&data_word(0xbcbc63df, 0xb6b677c1, 0xdadaaf75, 0x21214263);
	&data_word(0x10102030, 0xffffe51a, 0xf3f3fd0e, 0xd2d2bf6d);
	&data_word(0xcdcd814c, 0x0c0c1814, 0x13132635, 0xececc32f);
	&data_word(0x5f5fbee1, 0x979735a2, 0x444488cc, 0x17172e39);
	&data_word(0xc4c49357, 0xa7a755f2, 0x7e7efc82, 0x3d3d7a47);
	&data_word(0x6464c8ac, 0x5d5dbae7, 0x1919322b, 0x7373e695);
	&data_word(0x6060c0a0, 0x81811998, 0x4f4f9ed1, 0xdcdca37f);
	&data_word(0x22224466, 0x2a2a547e, 0x90903bab, 0x88880b83);
	&data_word(0x46468cca, 0xeeeec729, 0xb8b86bd3, 0x1414283c);
	&data_word(0xdedea779, 0x5e5ebce2, 0x0b0b161d, 0xdbdbad76);
	&data_word(0xe0e0db3b, 0x32326456, 0x3a3a744e, 0x0a0a141e);
	&data_word(0x494992db, 0x06060c0a, 0x2424486c, 0x5c5cb8e4);
	&data_word(0xc2c29f5d, 0xd3d3bd6e, 0xacac43ef, 0x6262c4a6);
	&data_word(0x919139a8, 0x959531a4, 0xe4e4d337, 0x7979f28b);
	&data_word(0xe7e7d532, 0xc8c88b43, 0x37376e59, 0x6d6ddab7);
	&data_word(0x8d8d018c, 0xd5d5b164, 0x4e4e9cd2, 0xa9a949e0);
	&data_word(0x6c6cd8b4, 0x5656acfa, 0xf4f4f307, 0xeaeacf25);
	&data_word(0x6565caaf, 0x7a7af48e, 0xaeae47e9, 0x08081018);
	&data_word(0xbaba6fd5, 0x7878f088, 0x25254a6f, 0x2e2e5c72);
	&data_word(0x1c1c3824, 0xa6a657f1, 0xb4b473c7, 0xc6c69751);
	&data_word(0xe8e8cb23, 0xdddda17c, 0x7474e89c, 0x1f1f3e21);
	&data_word(0x4b4b96dd, 0xbdbd61dc, 0x8b8b0d86, 0x8a8a0f85);
	&data_word(0x7070e090, 0x3e3e7c42, 0xb5b571c4, 0x6666ccaa);
	&data_word(0x484890d8, 0x03030605, 0xf6f6f701, 0x0e0e1c12);
	&data_word(0x6161c2a3, 0x35356a5f, 0x5757aef9, 0xb9b969d0);
	&data_word(0x86861791, 0xc1c19958, 0x1d1d3a27, 0x9e9e27b9);
	&data_word(0xe1e1d938, 0xf8f8eb13, 0x98982bb3, 0x11112233);
	&data_word(0x6969d2bb, 0xd9d9a970, 0x8e8e0789, 0x949433a7);
	&data_word(0x9b9b2db6, 0x1e1e3c22, 0x87871592, 0xe9e9c920);
	&data_word(0xcece8749, 0x5555aaff, 0x28285078, 0xdfdfa57a);
	&data_word(0x8c8c038f, 0xa1a159f8, 0x89890980, 0x0d0d1a17);
	&data_word(0xbfbf65da, 0xe6e6d731, 0x424284c6, 0x6868d0b8);
	&data_word(0x414182c3, 0x999929b0, 0x2d2d5a77, 0x0f0f1e11);
	&data_word(0xb0b07bcb, 0x5454a8fc, 0xbbbb6dd6, 0x16162c3a);
#Te2:
	&data_word(0x63c6a563, 0x7cf8847c, 0x77ee9977, 0x7bf68d7b);
	&data_word(0xf2ff0df2, 0x6bd6bd6b, 0x6fdeb16f, 0xc59154c5);
	&data_word(0x30605030, 0x01020301, 0x67cea967, 0x2b567d2b);
	&data_word(0xfee719fe, 0xd7b562d7, 0xab4de6ab, 0x76ec9a76);
	&data_word(0xca8f45ca, 0x821f9d82, 0xc98940c9, 0x7dfa877d);
	&data_word(0xfaef15fa, 0x59b2eb59, 0x478ec947, 0xf0fb0bf0);
	&data_word(0xad41ecad, 0xd4b367d4, 0xa25ffda2, 0xaf45eaaf);
	&data_word(0x9c23bf9c, 0xa453f7a4, 0x72e49672, 0xc09b5bc0);
	&data_word(0xb775c2b7, 0xfde11cfd, 0x933dae93, 0x264c6a26);
	&data_word(0x366c5a36, 0x3f7e413f, 0xf7f502f7, 0xcc834fcc);
	&data_word(0x34685c34, 0xa551f4a5, 0xe5d134e5, 0xf1f908f1);
	&data_word(0x71e29371, 0xd8ab73d8, 0x31625331, 0x152a3f15);
	&data_word(0x04080c04, 0xc79552c7, 0x23466523, 0xc39d5ec3);
	&data_word(0x18302818, 0x9637a196, 0x050a0f05, 0x9a2fb59a);
	&data_word(0x070e0907, 0x12243612, 0x801b9b80, 0xe2df3de2);
	&data_word(0xebcd26eb, 0x274e6927, 0xb27fcdb2, 0x75ea9f75);
	&data_word(0x09121b09, 0x831d9e83, 0x2c58742c, 0x1a342e1a);
	&data_word(0x1b362d1b, 0x6edcb26e, 0x5ab4ee5a, 0xa05bfba0);
	&data_word(0x52a4f652, 0x3b764d3b, 0xd6b761d6, 0xb37dceb3);
	&data_word(0x29527b29, 0xe3dd3ee3, 0x2f5e712f, 0x84139784);
	&data_word(0x53a6f553, 0xd1b968d1, 0x00000000, 0xedc12ced);
	&data_word(0x20406020, 0xfce31ffc, 0xb179c8b1, 0x5bb6ed5b);
	&data_word(0x6ad4be6a, 0xcb8d46cb, 0xbe67d9be, 0x39724b39);
	&data_word(0x4a94de4a, 0x4c98d44c, 0x58b0e858, 0xcf854acf);
	&data_word(0xd0bb6bd0, 0xefc52aef, 0xaa4fe5aa, 0xfbed16fb);
	&data_word(0x4386c543, 0x4d9ad74d, 0x33665533, 0x85119485);
	&data_word(0x458acf45, 0xf9e910f9, 0x02040602, 0x7ffe817f);
	&data_word(0x50a0f050, 0x3c78443c, 0x9f25ba9f, 0xa84be3a8);
	&data_word(0x51a2f351, 0xa35dfea3, 0x4080c040, 0x8f058a8f);
	&data_word(0x923fad92, 0x9d21bc9d, 0x38704838, 0xf5f104f5);
	&data_word(0xbc63dfbc, 0xb677c1b6, 0xdaaf75da, 0x21426321);
	&data_word(0x10203010, 0xffe51aff, 0xf3fd0ef3, 0xd2bf6dd2);
	&data_word(0xcd814ccd, 0x0c18140c, 0x13263513, 0xecc32fec);
	&data_word(0x5fbee15f, 0x9735a297, 0x4488cc44, 0x172e3917);
	&data_word(0xc49357c4, 0xa755f2a7, 0x7efc827e, 0x3d7a473d);
	&data_word(0x64c8ac64, 0x5dbae75d, 0x19322b19, 0x73e69573);
	&data_word(0x60c0a060, 0x81199881, 0x4f9ed14f, 0xdca37fdc);
	&data_word(0x22446622, 0x2a547e2a, 0x903bab90, 0x880b8388);
	&data_word(0x468cca46, 0xeec729ee, 0xb86bd3b8, 0x14283c14);
	&data_word(0xdea779de, 0x5ebce25e, 0x0b161d0b, 0xdbad76db);
	&data_word(0xe0db3be0, 0x32645632, 0x3a744e3a, 0x0a141e0a);
	&data_word(0x4992db49, 0x060c0a06, 0x24486c24, 0x5cb8e45c);
	&data_word(0xc29f5dc2, 0xd3bd6ed3, 0xac43efac, 0x62c4a662);
	&data_word(0x9139a891, 0x9531a495, 0xe4d337e4, 0x79f28b79);
	&data_word(0xe7d532e7, 0xc88b43c8, 0x376e5937, 0x6ddab76d);
	&data_word(0x8d018c8d, 0xd5b164d5, 0x4e9cd24e, 0xa949e0a9);
	&data_word(0x6cd8b46c, 0x56acfa56, 0xf4f307f4, 0xeacf25ea);
	&data_word(0x65caaf65, 0x7af48e7a, 0xae47e9ae, 0x08101808);
	&data_word(0xba6fd5ba, 0x78f08878, 0x254a6f25, 0x2e5c722e);
	&data_word(0x1c38241c, 0xa657f1a6, 0xb473c7b4, 0xc69751c6);
	&data_word(0xe8cb23e8, 0xdda17cdd, 0x74e89c74, 0x1f3e211f);
	&data_word(0x4b96dd4b, 0xbd61dcbd, 0x8b0d868b, 0x8a0f858a);
	&data_word(0x70e09070, 0x3e7c423e, 0xb571c4b5, 0x66ccaa66);
	&data_word(0x4890d848, 0x03060503, 0xf6f701f6, 0x0e1c120e);
	&data_word(0x61c2a361, 0x356a5f35, 0x57aef957, 0xb969d0b9);
	&data_word(0x86179186, 0xc19958c1, 0x1d3a271d, 0x9e27b99e);
	&data_word(0xe1d938e1, 0xf8eb13f8, 0x982bb398, 0x11223311);
	&data_word(0x69d2bb69, 0xd9a970d9, 0x8e07898e, 0x9433a794);
	&data_word(0x9b2db69b, 0x1e3c221e, 0x87159287, 0xe9c920e9);
	&data_word(0xce8749ce, 0x55aaff55, 0x28507828, 0xdfa57adf);
	&data_word(0x8c038f8c, 0xa159f8a1, 0x89098089, 0x0d1a170d);
	&data_word(0xbf65dabf, 0xe6d731e6, 0x4284c642, 0x68d0b868);
	&data_word(0x4182c341, 0x9929b099, 0x2d5a772d, 0x0f1e110f);
	&data_word(0xb07bcbb0, 0x54a8fc54, 0xbb6dd6bb, 0x162c3a16);
#Te3:
	&data_word(0xc6a56363, 0xf8847c7c, 0xee997777, 0xf68d7b7b);
	&data_word(0xff0df2f2, 0xd6bd6b6b, 0xdeb16f6f, 0x9154c5c5);
	&data_word(0x60503030, 0x02030101, 0xcea96767, 0x567d2b2b);
	&data_word(0xe719fefe, 0xb562d7d7, 0x4de6abab, 0xec9a7676);
	&data_word(0x8f45caca, 0x1f9d8282, 0x8940c9c9, 0xfa877d7d);
	&data_word(0xef15fafa, 0xb2eb5959, 0x8ec94747, 0xfb0bf0f0);
	&data_word(0x41ecadad, 0xb367d4d4, 0x5ffda2a2, 0x45eaafaf);
	&data_word(0x23bf9c9c, 0x53f7a4a4, 0xe4967272, 0x9b5bc0c0);
	&data_word(0x75c2b7b7, 0xe11cfdfd, 0x3dae9393, 0x4c6a2626);
	&data_word(0x6c5a3636, 0x7e413f3f, 0xf502f7f7, 0x834fcccc);
	&data_word(0x685c3434, 0x51f4a5a5, 0xd134e5e5, 0xf908f1f1);
	&data_word(0xe2937171, 0xab73d8d8, 0x62533131, 0x2a3f1515);
	&data_word(0x080c0404, 0x9552c7c7, 0x46652323, 0x9d5ec3c3);
	&data_word(0x30281818, 0x37a19696, 0x0a0f0505, 0x2fb59a9a);
	&data_word(0x0e090707, 0x24361212, 0x1b9b8080, 0xdf3de2e2);
	&data_word(0xcd26ebeb, 0x4e692727, 0x7fcdb2b2, 0xea9f7575);
	&data_word(0x121b0909, 0x1d9e8383, 0x58742c2c, 0x342e1a1a);
	&data_word(0x362d1b1b, 0xdcb26e6e, 0xb4ee5a5a, 0x5bfba0a0);
	&data_word(0xa4f65252, 0x764d3b3b, 0xb761d6d6, 0x7dceb3b3);
	&data_word(0x527b2929, 0xdd3ee3e3, 0x5e712f2f, 0x13978484);
	&data_word(0xa6f55353, 0xb968d1d1, 0x00000000, 0xc12ceded);
	&data_word(0x40602020, 0xe31ffcfc, 0x79c8b1b1, 0xb6ed5b5b);
	&data_word(0xd4be6a6a, 0x8d46cbcb, 0x67d9bebe, 0x724b3939);
	&data_word(0x94de4a4a, 0x98d44c4c, 0xb0e85858, 0x854acfcf);
	&data_word(0xbb6bd0d0, 0xc52aefef, 0x4fe5aaaa, 0xed16fbfb);
	&data_word(0x86c54343, 0x9ad74d4d, 0x66553333, 0x11948585);
	&data_word(0x8acf4545, 0xe910f9f9, 0x04060202, 0xfe817f7f);
	&data_word(0xa0f05050, 0x78443c3c, 0x25ba9f9f, 0x4be3a8a8);
	&data_word(0xa2f35151, 0x5dfea3a3, 0x80c04040, 0x058a8f8f);
	&data_word(0x3fad9292, 0x21bc9d9d, 0x70483838, 0xf104f5f5);
	&data_word(0x63dfbcbc, 0x77c1b6b6, 0xaf75dada, 0x42632121);
	&data_word(0x20301010, 0xe51affff, 0xfd0ef3f3, 0xbf6dd2d2);
	&data_word(0x814ccdcd, 0x18140c0c, 0x26351313, 0xc32fecec);
	&data_word(0xbee15f5f, 0x35a29797, 0x88cc4444, 0x2e391717);
	&data_word(0x9357c4c4, 0x55f2a7a7, 0xfc827e7e, 0x7a473d3d);
	&data_word(0xc8ac6464, 0xbae75d5d, 0x322b1919, 0xe6957373);
	&data_word(0xc0a06060, 0x19988181, 0x9ed14f4f, 0xa37fdcdc);
	&data_word(0x44662222, 0x547e2a2a, 0x3bab9090, 0x0b838888);
	&data_word(0x8cca4646, 0xc729eeee, 0x6bd3b8b8, 0x283c1414);
	&data_word(0xa779dede, 0xbce25e5e, 0x161d0b0b, 0xad76dbdb);
	&data_word(0xdb3be0e0, 0x64563232, 0x744e3a3a, 0x141e0a0a);
	&data_word(0x92db4949, 0x0c0a0606, 0x486c2424, 0xb8e45c5c);
	&data_word(0x9f5dc2c2, 0xbd6ed3d3, 0x43efacac, 0xc4a66262);
	&data_word(0x39a89191, 0x31a49595, 0xd337e4e4, 0xf28b7979);
	&data_word(0xd532e7e7, 0x8b43c8c8, 0x6e593737, 0xdab76d6d);
	&data_word(0x018c8d8d, 0xb164d5d5, 0x9cd24e4e, 0x49e0a9a9);
	&data_word(0xd8b46c6c, 0xacfa5656, 0xf307f4f4, 0xcf25eaea);
	&data_word(0xcaaf6565, 0xf48e7a7a, 0x47e9aeae, 0x10180808);
	&data_word(0x6fd5baba, 0xf0887878, 0x4a6f2525, 0x5c722e2e);
	&data_word(0x38241c1c, 0x57f1a6a6, 0x73c7b4b4, 0x9751c6c6);
	&data_word(0xcb23e8e8, 0xa17cdddd, 0xe89c7474, 0x3e211f1f);
	&data_word(0x96dd4b4b, 0x61dcbdbd, 0x0d868b8b, 0x0f858a8a);
	&data_word(0xe0907070, 0x7c423e3e, 0x71c4b5b5, 0xccaa6666);
	&data_word(0x90d84848, 0x06050303, 0xf701f6f6, 0x1c120e0e);
	&data_word(0xc2a36161, 0x6a5f3535, 0xaef95757, 0x69d0b9b9);
	&data_word(0x17918686, 0x9958c1c1, 0x3a271d1d, 0x27b99e9e);
	&data_word(0xd938e1e1, 0xeb13f8f8, 0x2bb39898, 0x22331111);
	&data_word(0xd2bb6969, 0xa970d9d9, 0x07898e8e, 0x33a79494);
	&data_word(0x2db69b9b, 0x3c221e1e, 0x15928787, 0xc920e9e9);
	&data_word(0x8749cece, 0xaaff5555, 0x50782828, 0xa57adfdf);
	&data_word(0x038f8c8c, 0x59f8a1a1, 0x09808989, 0x1a170d0d);
	&data_word(0x65dabfbf, 0xd731e6e6, 0x84c64242, 0xd0b86868);
	&data_word(0x82c34141, 0x29b09999, 0x5a772d2d, 0x1e110f0f);
	&data_word(0x7bcbb0b0, 0xa8fc5454, 0x6dd6bbbb, 0x2c3a1616);
#Te4:
	&data_word(0x63636363, 0x7c7c7c7c, 0x77777777, 0x7b7b7b7b);
	&data_word(0xf2f2f2f2, 0x6b6b6b6b, 0x6f6f6f6f, 0xc5c5c5c5);
	&data_word(0x30303030, 0x01010101, 0x67676767, 0x2b2b2b2b);
	&data_word(0xfefefefe, 0xd7d7d7d7, 0xabababab, 0x76767676);
	&data_word(0xcacacaca, 0x82828282, 0xc9c9c9c9, 0x7d7d7d7d);
	&data_word(0xfafafafa, 0x59595959, 0x47474747, 0xf0f0f0f0);
	&data_word(0xadadadad, 0xd4d4d4d4, 0xa2a2a2a2, 0xafafafaf);
	&data_word(0x9c9c9c9c, 0xa4a4a4a4, 0x72727272, 0xc0c0c0c0);
	&data_word(0xb7b7b7b7, 0xfdfdfdfd, 0x93939393, 0x26262626);
	&data_word(0x36363636, 0x3f3f3f3f, 0xf7f7f7f7, 0xcccccccc);
	&data_word(0x34343434, 0xa5a5a5a5, 0xe5e5e5e5, 0xf1f1f1f1);
	&data_word(0x71717171, 0xd8d8d8d8, 0x31313131, 0x15151515);
	&data_word(0x04040404, 0xc7c7c7c7, 0x23232323, 0xc3c3c3c3);
	&data_word(0x18181818, 0x96969696, 0x05050505, 0x9a9a9a9a);
	&data_word(0x07070707, 0x12121212, 0x80808080, 0xe2e2e2e2);
	&data_word(0xebebebeb, 0x27272727, 0xb2b2b2b2, 0x75757575);
	&data_word(0x09090909, 0x83838383, 0x2c2c2c2c, 0x1a1a1a1a);
	&data_word(0x1b1b1b1b, 0x6e6e6e6e, 0x5a5a5a5a, 0xa0a0a0a0);
	&data_word(0x52525252, 0x3b3b3b3b, 0xd6d6d6d6, 0xb3b3b3b3);
	&data_word(0x29292929, 0xe3e3e3e3, 0x2f2f2f2f, 0x84848484);
	&data_word(0x53535353, 0xd1d1d1d1, 0x00000000, 0xedededed);
	&data_word(0x20202020, 0xfcfcfcfc, 0xb1b1b1b1, 0x5b5b5b5b);
	&data_word(0x6a6a6a6a, 0xcbcbcbcb, 0xbebebebe, 0x39393939);
	&data_word(0x4a4a4a4a, 0x4c4c4c4c, 0x58585858, 0xcfcfcfcf);
	&data_word(0xd0d0d0d0, 0xefefefef, 0xaaaaaaaa, 0xfbfbfbfb);
	&data_word(0x43434343, 0x4d4d4d4d, 0x33333333, 0x85858585);
	&data_word(0x45454545, 0xf9f9f9f9, 0x02020202, 0x7f7f7f7f);
	&data_word(0x50505050, 0x3c3c3c3c, 0x9f9f9f9f, 0xa8a8a8a8);
	&data_word(0x51515151, 0xa3a3a3a3, 0x40404040, 0x8f8f8f8f);
	&data_word(0x92929292, 0x9d9d9d9d, 0x38383838, 0xf5f5f5f5);
	&data_word(0xbcbcbcbc, 0xb6b6b6b6, 0xdadadada, 0x21212121);
	&data_word(0x10101010, 0xffffffff, 0xf3f3f3f3, 0xd2d2d2d2);
	&data_word(0xcdcdcdcd, 0x0c0c0c0c, 0x13131313, 0xecececec);
	&data_word(0x5f5f5f5f, 0x97979797, 0x44444444, 0x17171717);
	&data_word(0xc4c4c4c4, 0xa7a7a7a7, 0x7e7e7e7e, 0x3d3d3d3d);
	&data_word(0x64646464, 0x5d5d5d5d, 0x19191919, 0x73737373);
	&data_word(0x60606060, 0x81818181, 0x4f4f4f4f, 0xdcdcdcdc);
	&data_word(0x22222222, 0x2a2a2a2a, 0x90909090, 0x88888888);
	&data_word(0x46464646, 0xeeeeeeee, 0xb8b8b8b8, 0x14141414);
	&data_word(0xdededede, 0x5e5e5e5e, 0x0b0b0b0b, 0xdbdbdbdb);
	&data_word(0xe0e0e0e0, 0x32323232, 0x3a3a3a3a, 0x0a0a0a0a);
	&data_word(0x49494949, 0x06060606, 0x24242424, 0x5c5c5c5c);
	&data_word(0xc2c2c2c2, 0xd3d3d3d3, 0xacacacac, 0x62626262);
	&data_word(0x91919191, 0x95959595, 0xe4e4e4e4, 0x79797979);
	&data_word(0xe7e7e7e7, 0xc8c8c8c8, 0x37373737, 0x6d6d6d6d);
	&data_word(0x8d8d8d8d, 0xd5d5d5d5, 0x4e4e4e4e, 0xa9a9a9a9);
	&data_word(0x6c6c6c6c, 0x56565656, 0xf4f4f4f4, 0xeaeaeaea);
	&data_word(0x65656565, 0x7a7a7a7a, 0xaeaeaeae, 0x08080808);
	&data_word(0xbabababa, 0x78787878, 0x25252525, 0x2e2e2e2e);
	&data_word(0x1c1c1c1c, 0xa6a6a6a6, 0xb4b4b4b4, 0xc6c6c6c6);
	&data_word(0xe8e8e8e8, 0xdddddddd, 0x74747474, 0x1f1f1f1f);
	&data_word(0x4b4b4b4b, 0xbdbdbdbd, 0x8b8b8b8b, 0x8a8a8a8a);
	&data_word(0x70707070, 0x3e3e3e3e, 0xb5b5b5b5, 0x66666666);
	&data_word(0x48484848, 0x03030303, 0xf6f6f6f6, 0x0e0e0e0e);
	&data_word(0x61616161, 0x35353535, 0x57575757, 0xb9b9b9b9);
	&data_word(0x86868686, 0xc1c1c1c1, 0x1d1d1d1d, 0x9e9e9e9e);
	&data_word(0xe1e1e1e1, 0xf8f8f8f8, 0x98989898, 0x11111111);
	&data_word(0x69696969, 0xd9d9d9d9, 0x8e8e8e8e, 0x94949494);
	&data_word(0x9b9b9b9b, 0x1e1e1e1e, 0x87878787, 0xe9e9e9e9);
	&data_word(0xcececece, 0x55555555, 0x28282828, 0xdfdfdfdf);
	&data_word(0x8c8c8c8c, 0xa1a1a1a1, 0x89898989, 0x0d0d0d0d);
	&data_word(0xbfbfbfbf, 0xe6e6e6e6, 0x42424242, 0x68686868);
	&data_word(0x41414141, 0x99999999, 0x2d2d2d2d, 0x0f0f0f0f);
	&data_word(0xb0b0b0b0, 0x54545454, 0xbbbbbbbb, 0x16161616);
#rcon:
	&data_word(0x00000001, 0x00000002, 0x00000004, 0x00000008);
        &data_word(0x00000010, 0x00000020, 0x00000040, 0x00000080);
        &data_word(0x0000001b, 0x00000036);
&function_end_B("AES_encrypt");

#------------------------------------------------------------------#

$s0="eax";
$s1="ebx";
$s2="ecx";
$s3="edx";
$key="edi";
$acc="esi";

sub decstep()
{ my ($i,$td,@s) = @_;
  my $tmp = $key;
  my $out = $i==3?$s[0]:$acc;

	# no instructions are reordered, as performance appears
	# optimal... or rather that all attempts to reorder didn't
	# result in better performance [which by the way is not a
	# bit lower than ecryption].
	if($i==3)   {	&mov	($key,&DWP(12,"esp"));		}
	else        {	&mov	($out,$s[0]);			}
			&and	($out,0xFF);
			&mov	($out,&DWP(1024*0,$td,$out,4));

	if ($i==3)  {	$tmp=$s[1];				}
			&movz	($tmp,&HB($s[1]));
			&xor	($out,&DWP(1024*1,$td,$tmp,4));

	if ($i==3)  {	$tmp=$s[2]; &mov ($s[1],$acc);		}
	else        {	&mov	($tmp,$s[2]);			}
			&shr	($tmp,16);
			&and	($tmp,0xFF);
			&xor	($out,&DWP(1024*2,$td,$tmp,4));

	if ($i==3)  {	$tmp=$s[3]; &mov ($s[2],&DWP(4,"esp"));	}
	else        {	&mov	($tmp,$s[3]);			}
			&shr	($tmp,24);
			&xor	($out,&DWP(1024*3,$td,$tmp,4));
	if ($i<2)   {	&mov	(&DWP(4*$i,"esp"),$out);	}
	if ($i==3)  {	&mov	($s[3],&DWP(0,"esp"));		}
			&comment();
}

sub declast()
{ my ($i,$td,@s)=@_;
  my $tmp = $key;
  my $out = $i==3?$s[0]:$acc;

	if($i==3)   {	&mov	($key,&DWP(12,"esp"));		}
	else        {	&mov	($out,$s[0]);			}
			&and	($out,0xFF);
			&mov	($out,&DWP(0,$td,$out,4));
			&and	($out,0x000000ff);

	if ($i==3)  {	$tmp=$s[1];				}
			&movz	($tmp,&HB($s[1]));
			&mov	($tmp,&DWP(0,$td,$tmp,4));
			&and	($tmp,0x0000ff00);
			&xor	($out,$tmp);

	if ($i==3)  {	$tmp=$s[2]; &mov ($s[1],$acc);		}
	else        {	mov	($tmp,$s[2]);			}
			&shr	($tmp,16);
			&and	($tmp,0xFF);
			&mov	($tmp,&DWP(0,$td,$tmp,4));
			&and	($tmp,0x00ff0000);
			&xor	($out,$tmp);

	if ($i==3)  {	$tmp=$s[3]; &mov ($s[2],&DWP(4,"esp"));	}
	else        {	&mov	($tmp,$s[3]);			}
			&shr	($tmp,24);
			&mov	($tmp,&DWP(0,$td,$tmp,4));
			&and	($tmp,0xff000000);
			&xor	($out,$tmp);
	if ($i<2)   {	&mov	(&DWP(4*$i,"esp"),$out);	}
	if ($i==3)  {	&mov	($s[3],&DWP(0,"esp"));		}
}

# void AES_decrypt (const void *inp,void *out,const AES_KEY *key);
&public_label("AES_Td");
&function_begin("AES_decrypt");
	&mov	($acc,&wparam(0));		# load inp
	&mov	($key,&wparam(2));		# load key

        &call   (&label("pic_point"));          # make it PIC!
	&set_label("pic_point");
        &blindpop("ebp");
        &lea    ("ebp",&DWP(&label("AES_Td")."-".&label("pic_point"),"ebp"));

	# allocate aligned stack frame
	&mov	($s0,"esp");
	&sub	("esp",20);
	&and	("esp",-16);

	&mov	(&DWP(12,"esp"),$key);		# save key
	&mov	(&DWP(16,"esp"),$s0);		# save %esp

	&mov	($s0,&DWP(0,$acc));		# load input data
	&mov	($s1,&DWP(4,$acc));
	&mov	($s2,&DWP(8,$acc));
	&mov	($s3,&DWP(12,$acc));

	&xor	($s0,&DWP(0,$key));
	&xor	($s1,&DWP(4,$key));
	&xor	($s2,&DWP(8,$key));
	&xor	($s3,&DWP(12,$key));

	&mov	($acc,&DWP(240,$key));		# load key->rounds

	if ($small_footprint) {
	    &lea	($acc,&DWP(-2,$acc,$acc));
	    &lea	($acc,&DWP(0,$key,$acc,8));
	    &mov	(&DWP(8,"esp"),$acc);	# end of key schedule
	    &align	(4);
	    &set_label("loop");
		&decstep(0,"ebp",$s0,$s3,$s2,$s1);
		&decstep(1,"ebp",$s1,$s0,$s3,$s2);
		&decstep(2,"ebp",$s2,$s1,$s0,$s3);
		&decstep(3,"ebp",$s3,$s2,$s1,$s0);
		&add	($key,16);			# advance rd_key
		&xor	($s0,&DWP(0,$key));
		&xor	($s1,&DWP(4,$key));
		&xor	($s2,&DWP(8,$key));
		&xor	($s3,&DWP(12,$key));
	    &cmp	($key,&DWP(8,"esp"));
	    &mov	(&DWP(12,"esp"),$key);
	    &jb		(&label("loop"));
	}
	else {
	    &cmp	($acc,10);
	    &jle	(&label("10rounds"));
	    &cmp	($acc,12);
	    &jle	(&label("12rounds"));

	&set_label("14rounds");
	    for ($i=1;$i<3;$i++) {
		&decstep(0,"ebp",$s0,$s3,$s2,$s1);
		&decstep(1,"ebp",$s1,$s0,$s3,$s2);
		&decstep(2,"ebp",$s2,$s1,$s0,$s3);
		&decstep(3,"ebp",$s3,$s2,$s1,$s0);
		&xor	($s0,&DWP(16*$i+0,$key));
		&xor	($s1,&DWP(16*$i+4,$key));
		&xor	($s2,&DWP(16*$i+8,$key));
		&xor	($s3,&DWP(16*$i+12,$key));
	    }
	    &add	($key,32);
	    &mov	(&DWP(12,"esp"),$key);		# advance rd_key
	&set_label("12rounds");
	    for ($i=1;$i<3;$i++) {
		&decstep(0,"ebp",$s0,$s3,$s2,$s1);
		&decstep(1,"ebp",$s1,$s0,$s3,$s2);
		&decstep(2,"ebp",$s2,$s1,$s0,$s3);
		&decstep(3,"ebp",$s3,$s2,$s1,$s0);
		&xor	($s0,&DWP(16*$i+0,$key));
		&xor	($s1,&DWP(16*$i+4,$key));
		&xor	($s2,&DWP(16*$i+8,$key));
		&xor	($s3,&DWP(16*$i+12,$key));
	    }
	    &add	($key,32);
	    &mov	(&DWP(12,"esp"),$key);		# advance rd_key
	&set_label("10rounds");
	    for ($i=1;$i<10;$i++) {
		&decstep(0,"ebp",$s0,$s3,$s2,$s1);
		&decstep(1,"ebp",$s1,$s0,$s3,$s2);
		&decstep(2,"ebp",$s2,$s1,$s0,$s3);
		&decstep(3,"ebp",$s3,$s2,$s1,$s0);
		&xor	($s0,&DWP(16*$i+0,$key));
		&xor	($s1,&DWP(16*$i+4,$key));
		&xor	($s2,&DWP(16*$i+8,$key));
		&xor	($s3,&DWP(16*$i+12,$key));
	    }
	}

	&add	("ebp",4*1024);			# skip to Te4
	&declast(0,"ebp",$s0,$s3,$s2,$s1);
	&declast(1,"ebp",$s1,$s0,$s3,$s2);
	&declast(2,"ebp",$s2,$s1,$s0,$s3);
	&declast(3,"ebp",$s3,$s2,$s1,$s0);

	&mov	("esp",&DWP(16,"esp"));		# restore %esp
	&add	($key,$small_footprint?16:160);
	&xor	($s0,&DWP(0,$key));
	&xor	($s1,&DWP(4,$key));
	&xor	($s2,&DWP(8,$key));
	&xor	($s3,&DWP(12,$key));

	&mov	($key,&wparam(1));		# load out
	&mov	(&DWP(0,$key),$s0);		# write output data
	&mov	(&DWP(4,$key),$s1);
	&mov	(&DWP(8,$key),$s2);
	&mov	(&DWP(12,$key),$s3);

	&pop	("edi");
	&pop	("esi");
	&pop	("ebx");
	&pop	("ebp");
	&ret	();

&set_label("AES_Td",64);	# Yes! I keep it in the code segment!
	&data_word(0x50a7f451, 0x5365417e, 0xc3a4171a, 0x965e273a);
	&data_word(0xcb6bab3b, 0xf1459d1f, 0xab58faac, 0x9303e34b);
	&data_word(0x55fa3020, 0xf66d76ad, 0x9176cc88, 0x254c02f5);
	&data_word(0xfcd7e54f, 0xd7cb2ac5, 0x80443526, 0x8fa362b5);
	&data_word(0x495ab1de, 0x671bba25, 0x980eea45, 0xe1c0fe5d);
	&data_word(0x02752fc3, 0x12f04c81, 0xa397468d, 0xc6f9d36b);
	&data_word(0xe75f8f03, 0x959c9215, 0xeb7a6dbf, 0xda595295);
	&data_word(0x2d83bed4, 0xd3217458, 0x2969e049, 0x44c8c98e);
	&data_word(0x6a89c275, 0x78798ef4, 0x6b3e5899, 0xdd71b927);
	&data_word(0xb64fe1be, 0x17ad88f0, 0x66ac20c9, 0xb43ace7d);
	&data_word(0x184adf63, 0x82311ae5, 0x60335197, 0x457f5362);
	&data_word(0xe07764b1, 0x84ae6bbb, 0x1ca081fe, 0x942b08f9);
	&data_word(0x58684870, 0x19fd458f, 0x876cde94, 0xb7f87b52);
	&data_word(0x23d373ab, 0xe2024b72, 0x578f1fe3, 0x2aab5566);
	&data_word(0x0728ebb2, 0x03c2b52f, 0x9a7bc586, 0xa50837d3);
	&data_word(0xf2872830, 0xb2a5bf23, 0xba6a0302, 0x5c8216ed);
	&data_word(0x2b1ccf8a, 0x92b479a7, 0xf0f207f3, 0xa1e2694e);
	&data_word(0xcdf4da65, 0xd5be0506, 0x1f6234d1, 0x8afea6c4);
	&data_word(0x9d532e34, 0xa055f3a2, 0x32e18a05, 0x75ebf6a4);
	&data_word(0x39ec830b, 0xaaef6040, 0x069f715e, 0x51106ebd);
	&data_word(0xf98a213e, 0x3d06dd96, 0xae053edd, 0x46bde64d);
	&data_word(0xb58d5491, 0x055dc471, 0x6fd40604, 0xff155060);
	&data_word(0x24fb9819, 0x97e9bdd6, 0xcc434089, 0x779ed967);
	&data_word(0xbd42e8b0, 0x888b8907, 0x385b19e7, 0xdbeec879);
	&data_word(0x470a7ca1, 0xe90f427c, 0xc91e84f8, 0x00000000);
	&data_word(0x83868009, 0x48ed2b32, 0xac70111e, 0x4e725a6c);
	&data_word(0xfbff0efd, 0x5638850f, 0x1ed5ae3d, 0x27392d36);
	&data_word(0x64d90f0a, 0x21a65c68, 0xd1545b9b, 0x3a2e3624);
	&data_word(0xb1670a0c, 0x0fe75793, 0xd296eeb4, 0x9e919b1b);
	&data_word(0x4fc5c080, 0xa220dc61, 0x694b775a, 0x161a121c);
	&data_word(0x0aba93e2, 0xe52aa0c0, 0x43e0223c, 0x1d171b12);
	&data_word(0x0b0d090e, 0xadc78bf2, 0xb9a8b62d, 0xc8a91e14);
	&data_word(0x8519f157, 0x4c0775af, 0xbbdd99ee, 0xfd607fa3);
	&data_word(0x9f2601f7, 0xbcf5725c, 0xc53b6644, 0x347efb5b);
	&data_word(0x7629438b, 0xdcc623cb, 0x68fcedb6, 0x63f1e4b8);
	&data_word(0xcadc31d7, 0x10856342, 0x40229713, 0x2011c684);
	&data_word(0x7d244a85, 0xf83dbbd2, 0x1132f9ae, 0x6da129c7);
	&data_word(0x4b2f9e1d, 0xf330b2dc, 0xec52860d, 0xd0e3c177);
	&data_word(0x6c16b32b, 0x99b970a9, 0xfa489411, 0x2264e947);
	&data_word(0xc48cfca8, 0x1a3ff0a0, 0xd82c7d56, 0xef903322);
	&data_word(0xc74e4987, 0xc1d138d9, 0xfea2ca8c, 0x360bd498);
	&data_word(0xcf81f5a6, 0x28de7aa5, 0x268eb7da, 0xa4bfad3f);
	&data_word(0xe49d3a2c, 0x0d927850, 0x9bcc5f6a, 0x62467e54);
	&data_word(0xc2138df6, 0xe8b8d890, 0x5ef7392e, 0xf5afc382);
	&data_word(0xbe805d9f, 0x7c93d069, 0xa92dd56f, 0xb31225cf);
	&data_word(0x3b99acc8, 0xa77d1810, 0x6e639ce8, 0x7bbb3bdb);
	&data_word(0x097826cd, 0xf418596e, 0x01b79aec, 0xa89a4f83);
	&data_word(0x656e95e6, 0x7ee6ffaa, 0x08cfbc21, 0xe6e815ef);
	&data_word(0xd99be7ba, 0xce366f4a, 0xd4099fea, 0xd67cb029);
	&data_word(0xafb2a431, 0x31233f2a, 0x3094a5c6, 0xc066a235);
	&data_word(0x37bc4e74, 0xa6ca82fc, 0xb0d090e0, 0x15d8a733);
	&data_word(0x4a9804f1, 0xf7daec41, 0x0e50cd7f, 0x2ff69117);
	&data_word(0x8dd64d76, 0x4db0ef43, 0x544daacc, 0xdf0496e4);
	&data_word(0xe3b5d19e, 0x1b886a4c, 0xb81f2cc1, 0x7f516546);
	&data_word(0x04ea5e9d, 0x5d358c01, 0x737487fa, 0x2e410bfb);
	&data_word(0x5a1d67b3, 0x52d2db92, 0x335610e9, 0x1347d66d);
	&data_word(0x8c61d79a, 0x7a0ca137, 0x8e14f859, 0x893c13eb);
	&data_word(0xee27a9ce, 0x35c961b7, 0xede51ce1, 0x3cb1477a);
	&data_word(0x59dfd29c, 0x3f73f255, 0x79ce1418, 0xbf37c773);
	&data_word(0xeacdf753, 0x5baafd5f, 0x146f3ddf, 0x86db4478);
	&data_word(0x81f3afca, 0x3ec468b9, 0x2c342438, 0x5f40a3c2);
	&data_word(0x72c31d16, 0x0c25e2bc, 0x8b493c28, 0x41950dff);
	&data_word(0x7101a839, 0xdeb30c08, 0x9ce4b4d8, 0x90c15664);
	&data_word(0x6184cb7b, 0x70b632d5, 0x745c6c48, 0x4257b8d0);
#Td1:
	&data_word(0xa7f45150, 0x65417e53, 0xa4171ac3, 0x5e273a96);
	&data_word(0x6bab3bcb, 0x459d1ff1, 0x58faacab, 0x03e34b93);
	&data_word(0xfa302055, 0x6d76adf6, 0x76cc8891, 0x4c02f525);
	&data_word(0xd7e54ffc, 0xcb2ac5d7, 0x44352680, 0xa362b58f);
	&data_word(0x5ab1de49, 0x1bba2567, 0x0eea4598, 0xc0fe5de1);
	&data_word(0x752fc302, 0xf04c8112, 0x97468da3, 0xf9d36bc6);
	&data_word(0x5f8f03e7, 0x9c921595, 0x7a6dbfeb, 0x595295da);
	&data_word(0x83bed42d, 0x217458d3, 0x69e04929, 0xc8c98e44);
	&data_word(0x89c2756a, 0x798ef478, 0x3e58996b, 0x71b927dd);
	&data_word(0x4fe1beb6, 0xad88f017, 0xac20c966, 0x3ace7db4);
	&data_word(0x4adf6318, 0x311ae582, 0x33519760, 0x7f536245);
	&data_word(0x7764b1e0, 0xae6bbb84, 0xa081fe1c, 0x2b08f994);
	&data_word(0x68487058, 0xfd458f19, 0x6cde9487, 0xf87b52b7);
	&data_word(0xd373ab23, 0x024b72e2, 0x8f1fe357, 0xab55662a);
	&data_word(0x28ebb207, 0xc2b52f03, 0x7bc5869a, 0x0837d3a5);
	&data_word(0x872830f2, 0xa5bf23b2, 0x6a0302ba, 0x8216ed5c);
	&data_word(0x1ccf8a2b, 0xb479a792, 0xf207f3f0, 0xe2694ea1);
	&data_word(0xf4da65cd, 0xbe0506d5, 0x6234d11f, 0xfea6c48a);
	&data_word(0x532e349d, 0x55f3a2a0, 0xe18a0532, 0xebf6a475);
	&data_word(0xec830b39, 0xef6040aa, 0x9f715e06, 0x106ebd51);
	&data_word(0x8a213ef9, 0x06dd963d, 0x053eddae, 0xbde64d46);
	&data_word(0x8d5491b5, 0x5dc47105, 0xd406046f, 0x155060ff);
	&data_word(0xfb981924, 0xe9bdd697, 0x434089cc, 0x9ed96777);
	&data_word(0x42e8b0bd, 0x8b890788, 0x5b19e738, 0xeec879db);
	&data_word(0x0a7ca147, 0x0f427ce9, 0x1e84f8c9, 0x00000000);
	&data_word(0x86800983, 0xed2b3248, 0x70111eac, 0x725a6c4e);
	&data_word(0xff0efdfb, 0x38850f56, 0xd5ae3d1e, 0x392d3627);
	&data_word(0xd90f0a64, 0xa65c6821, 0x545b9bd1, 0x2e36243a);
	&data_word(0x670a0cb1, 0xe757930f, 0x96eeb4d2, 0x919b1b9e);
	&data_word(0xc5c0804f, 0x20dc61a2, 0x4b775a69, 0x1a121c16);
	&data_word(0xba93e20a, 0x2aa0c0e5, 0xe0223c43, 0x171b121d);
	&data_word(0x0d090e0b, 0xc78bf2ad, 0xa8b62db9, 0xa91e14c8);
	&data_word(0x19f15785, 0x0775af4c, 0xdd99eebb, 0x607fa3fd);
	&data_word(0x2601f79f, 0xf5725cbc, 0x3b6644c5, 0x7efb5b34);
	&data_word(0x29438b76, 0xc623cbdc, 0xfcedb668, 0xf1e4b863);
	&data_word(0xdc31d7ca, 0x85634210, 0x22971340, 0x11c68420);
	&data_word(0x244a857d, 0x3dbbd2f8, 0x32f9ae11, 0xa129c76d);
	&data_word(0x2f9e1d4b, 0x30b2dcf3, 0x52860dec, 0xe3c177d0);
	&data_word(0x16b32b6c, 0xb970a999, 0x489411fa, 0x64e94722);
	&data_word(0x8cfca8c4, 0x3ff0a01a, 0x2c7d56d8, 0x903322ef);
	&data_word(0x4e4987c7, 0xd138d9c1, 0xa2ca8cfe, 0x0bd49836);
	&data_word(0x81f5a6cf, 0xde7aa528, 0x8eb7da26, 0xbfad3fa4);
	&data_word(0x9d3a2ce4, 0x9278500d, 0xcc5f6a9b, 0x467e5462);
	&data_word(0x138df6c2, 0xb8d890e8, 0xf7392e5e, 0xafc382f5);
	&data_word(0x805d9fbe, 0x93d0697c, 0x2dd56fa9, 0x1225cfb3);
	&data_word(0x99acc83b, 0x7d1810a7, 0x639ce86e, 0xbb3bdb7b);
	&data_word(0x7826cd09, 0x18596ef4, 0xb79aec01, 0x9a4f83a8);
	&data_word(0x6e95e665, 0xe6ffaa7e, 0xcfbc2108, 0xe815efe6);
	&data_word(0x9be7bad9, 0x366f4ace, 0x099fead4, 0x7cb029d6);
	&data_word(0xb2a431af, 0x233f2a31, 0x94a5c630, 0x66a235c0);
	&data_word(0xbc4e7437, 0xca82fca6, 0xd090e0b0, 0xd8a73315);
	&data_word(0x9804f14a, 0xdaec41f7, 0x50cd7f0e, 0xf691172f);
	&data_word(0xd64d768d, 0xb0ef434d, 0x4daacc54, 0x0496e4df);
	&data_word(0xb5d19ee3, 0x886a4c1b, 0x1f2cc1b8, 0x5165467f);
	&data_word(0xea5e9d04, 0x358c015d, 0x7487fa73, 0x410bfb2e);
	&data_word(0x1d67b35a, 0xd2db9252, 0x5610e933, 0x47d66d13);
	&data_word(0x61d79a8c, 0x0ca1377a, 0x14f8598e, 0x3c13eb89);
	&data_word(0x27a9ceee, 0xc961b735, 0xe51ce1ed, 0xb1477a3c);
	&data_word(0xdfd29c59, 0x73f2553f, 0xce141879, 0x37c773bf);
	&data_word(0xcdf753ea, 0xaafd5f5b, 0x6f3ddf14, 0xdb447886);
	&data_word(0xf3afca81, 0xc468b93e, 0x3424382c, 0x40a3c25f);
	&data_word(0xc31d1672, 0x25e2bc0c, 0x493c288b, 0x950dff41);
	&data_word(0x01a83971, 0xb30c08de, 0xe4b4d89c, 0xc1566490);
	&data_word(0x84cb7b61, 0xb632d570, 0x5c6c4874, 0x57b8d042);
#Td2:
	&data_word(0xf45150a7, 0x417e5365, 0x171ac3a4, 0x273a965e);
	&data_word(0xab3bcb6b, 0x9d1ff145, 0xfaacab58, 0xe34b9303);
	&data_word(0x302055fa, 0x76adf66d, 0xcc889176, 0x02f5254c);
	&data_word(0xe54ffcd7, 0x2ac5d7cb, 0x35268044, 0x62b58fa3);
	&data_word(0xb1de495a, 0xba25671b, 0xea45980e, 0xfe5de1c0);
	&data_word(0x2fc30275, 0x4c8112f0, 0x468da397, 0xd36bc6f9);
	&data_word(0x8f03e75f, 0x9215959c, 0x6dbfeb7a, 0x5295da59);
	&data_word(0xbed42d83, 0x7458d321, 0xe0492969, 0xc98e44c8);
	&data_word(0xc2756a89, 0x8ef47879, 0x58996b3e, 0xb927dd71);
	&data_word(0xe1beb64f, 0x88f017ad, 0x20c966ac, 0xce7db43a);
	&data_word(0xdf63184a, 0x1ae58231, 0x51976033, 0x5362457f);
	&data_word(0x64b1e077, 0x6bbb84ae, 0x81fe1ca0, 0x08f9942b);
	&data_word(0x48705868, 0x458f19fd, 0xde94876c, 0x7b52b7f8);
	&data_word(0x73ab23d3, 0x4b72e202, 0x1fe3578f, 0x55662aab);
	&data_word(0xebb20728, 0xb52f03c2, 0xc5869a7b, 0x37d3a508);
	&data_word(0x2830f287, 0xbf23b2a5, 0x0302ba6a, 0x16ed5c82);
	&data_word(0xcf8a2b1c, 0x79a792b4, 0x07f3f0f2, 0x694ea1e2);
	&data_word(0xda65cdf4, 0x0506d5be, 0x34d11f62, 0xa6c48afe);
	&data_word(0x2e349d53, 0xf3a2a055, 0x8a0532e1, 0xf6a475eb);
	&data_word(0x830b39ec, 0x6040aaef, 0x715e069f, 0x6ebd5110);
	&data_word(0x213ef98a, 0xdd963d06, 0x3eddae05, 0xe64d46bd);
	&data_word(0x5491b58d, 0xc471055d, 0x06046fd4, 0x5060ff15);
	&data_word(0x981924fb, 0xbdd697e9, 0x4089cc43, 0xd967779e);
	&data_word(0xe8b0bd42, 0x8907888b, 0x19e7385b, 0xc879dbee);
	&data_word(0x7ca1470a, 0x427ce90f, 0x84f8c91e, 0x00000000);
	&data_word(0x80098386, 0x2b3248ed, 0x111eac70, 0x5a6c4e72);
	&data_word(0x0efdfbff, 0x850f5638, 0xae3d1ed5, 0x2d362739);
	&data_word(0x0f0a64d9, 0x5c6821a6, 0x5b9bd154, 0x36243a2e);
	&data_word(0x0a0cb167, 0x57930fe7, 0xeeb4d296, 0x9b1b9e91);
	&data_word(0xc0804fc5, 0xdc61a220, 0x775a694b, 0x121c161a);
	&data_word(0x93e20aba, 0xa0c0e52a, 0x223c43e0, 0x1b121d17);
	&data_word(0x090e0b0d, 0x8bf2adc7, 0xb62db9a8, 0x1e14c8a9);
	&data_word(0xf1578519, 0x75af4c07, 0x99eebbdd, 0x7fa3fd60);
	&data_word(0x01f79f26, 0x725cbcf5, 0x6644c53b, 0xfb5b347e);
	&data_word(0x438b7629, 0x23cbdcc6, 0xedb668fc, 0xe4b863f1);
	&data_word(0x31d7cadc, 0x63421085, 0x97134022, 0xc6842011);
	&data_word(0x4a857d24, 0xbbd2f83d, 0xf9ae1132, 0x29c76da1);
	&data_word(0x9e1d4b2f, 0xb2dcf330, 0x860dec52, 0xc177d0e3);
	&data_word(0xb32b6c16, 0x70a999b9, 0x9411fa48, 0xe9472264);
	&data_word(0xfca8c48c, 0xf0a01a3f, 0x7d56d82c, 0x3322ef90);
	&data_word(0x4987c74e, 0x38d9c1d1, 0xca8cfea2, 0xd498360b);
	&data_word(0xf5a6cf81, 0x7aa528de, 0xb7da268e, 0xad3fa4bf);
	&data_word(0x3a2ce49d, 0x78500d92, 0x5f6a9bcc, 0x7e546246);
	&data_word(0x8df6c213, 0xd890e8b8, 0x392e5ef7, 0xc382f5af);
	&data_word(0x5d9fbe80, 0xd0697c93, 0xd56fa92d, 0x25cfb312);
	&data_word(0xacc83b99, 0x1810a77d, 0x9ce86e63, 0x3bdb7bbb);
	&data_word(0x26cd0978, 0x596ef418, 0x9aec01b7, 0x4f83a89a);
	&data_word(0x95e6656e, 0xffaa7ee6, 0xbc2108cf, 0x15efe6e8);
	&data_word(0xe7bad99b, 0x6f4ace36, 0x9fead409, 0xb029d67c);
	&data_word(0xa431afb2, 0x3f2a3123, 0xa5c63094, 0xa235c066);
	&data_word(0x4e7437bc, 0x82fca6ca, 0x90e0b0d0, 0xa73315d8);
	&data_word(0x04f14a98, 0xec41f7da, 0xcd7f0e50, 0x91172ff6);
	&data_word(0x4d768dd6, 0xef434db0, 0xaacc544d, 0x96e4df04);
	&data_word(0xd19ee3b5, 0x6a4c1b88, 0x2cc1b81f, 0x65467f51);
	&data_word(0x5e9d04ea, 0x8c015d35, 0x87fa7374, 0x0bfb2e41);
	&data_word(0x67b35a1d, 0xdb9252d2, 0x10e93356, 0xd66d1347);
	&data_word(0xd79a8c61, 0xa1377a0c, 0xf8598e14, 0x13eb893c);
	&data_word(0xa9ceee27, 0x61b735c9, 0x1ce1ede5, 0x477a3cb1);
	&data_word(0xd29c59df, 0xf2553f73, 0x141879ce, 0xc773bf37);
	&data_word(0xf753eacd, 0xfd5f5baa, 0x3ddf146f, 0x447886db);
	&data_word(0xafca81f3, 0x68b93ec4, 0x24382c34, 0xa3c25f40);
	&data_word(0x1d1672c3, 0xe2bc0c25, 0x3c288b49, 0x0dff4195);
	&data_word(0xa8397101, 0x0c08deb3, 0xb4d89ce4, 0x566490c1);
	&data_word(0xcb7b6184, 0x32d570b6, 0x6c48745c, 0xb8d04257);
#Td3:
	&data_word(0x5150a7f4, 0x7e536541, 0x1ac3a417, 0x3a965e27);
	&data_word(0x3bcb6bab, 0x1ff1459d, 0xacab58fa, 0x4b9303e3);
	&data_word(0x2055fa30, 0xadf66d76, 0x889176cc, 0xf5254c02);
	&data_word(0x4ffcd7e5, 0xc5d7cb2a, 0x26804435, 0xb58fa362);
	&data_word(0xde495ab1, 0x25671bba, 0x45980eea, 0x5de1c0fe);
	&data_word(0xc302752f, 0x8112f04c, 0x8da39746, 0x6bc6f9d3);
	&data_word(0x03e75f8f, 0x15959c92, 0xbfeb7a6d, 0x95da5952);
	&data_word(0xd42d83be, 0x58d32174, 0x492969e0, 0x8e44c8c9);
	&data_word(0x756a89c2, 0xf478798e, 0x996b3e58, 0x27dd71b9);
	&data_word(0xbeb64fe1, 0xf017ad88, 0xc966ac20, 0x7db43ace);
	&data_word(0x63184adf, 0xe582311a, 0x97603351, 0x62457f53);
	&data_word(0xb1e07764, 0xbb84ae6b, 0xfe1ca081, 0xf9942b08);
	&data_word(0x70586848, 0x8f19fd45, 0x94876cde, 0x52b7f87b);
	&data_word(0xab23d373, 0x72e2024b, 0xe3578f1f, 0x662aab55);
	&data_word(0xb20728eb, 0x2f03c2b5, 0x869a7bc5, 0xd3a50837);
	&data_word(0x30f28728, 0x23b2a5bf, 0x02ba6a03, 0xed5c8216);
	&data_word(0x8a2b1ccf, 0xa792b479, 0xf3f0f207, 0x4ea1e269);
	&data_word(0x65cdf4da, 0x06d5be05, 0xd11f6234, 0xc48afea6);
	&data_word(0x349d532e, 0xa2a055f3, 0x0532e18a, 0xa475ebf6);
	&data_word(0x0b39ec83, 0x40aaef60, 0x5e069f71, 0xbd51106e);
	&data_word(0x3ef98a21, 0x963d06dd, 0xddae053e, 0x4d46bde6);
	&data_word(0x91b58d54, 0x71055dc4, 0x046fd406, 0x60ff1550);
	&data_word(0x1924fb98, 0xd697e9bd, 0x89cc4340, 0x67779ed9);
	&data_word(0xb0bd42e8, 0x07888b89, 0xe7385b19, 0x79dbeec8);
	&data_word(0xa1470a7c, 0x7ce90f42, 0xf8c91e84, 0x00000000);
	&data_word(0x09838680, 0x3248ed2b, 0x1eac7011, 0x6c4e725a);
	&data_word(0xfdfbff0e, 0x0f563885, 0x3d1ed5ae, 0x3627392d);
	&data_word(0x0a64d90f, 0x6821a65c, 0x9bd1545b, 0x243a2e36);
	&data_word(0x0cb1670a, 0x930fe757, 0xb4d296ee, 0x1b9e919b);
	&data_word(0x804fc5c0, 0x61a220dc, 0x5a694b77, 0x1c161a12);
	&data_word(0xe20aba93, 0xc0e52aa0, 0x3c43e022, 0x121d171b);
	&data_word(0x0e0b0d09, 0xf2adc78b, 0x2db9a8b6, 0x14c8a91e);
	&data_word(0x578519f1, 0xaf4c0775, 0xeebbdd99, 0xa3fd607f);
	&data_word(0xf79f2601, 0x5cbcf572, 0x44c53b66, 0x5b347efb);
	&data_word(0x8b762943, 0xcbdcc623, 0xb668fced, 0xb863f1e4);
	&data_word(0xd7cadc31, 0x42108563, 0x13402297, 0x842011c6);
	&data_word(0x857d244a, 0xd2f83dbb, 0xae1132f9, 0xc76da129);
	&data_word(0x1d4b2f9e, 0xdcf330b2, 0x0dec5286, 0x77d0e3c1);
	&data_word(0x2b6c16b3, 0xa999b970, 0x11fa4894, 0x472264e9);
	&data_word(0xa8c48cfc, 0xa01a3ff0, 0x56d82c7d, 0x22ef9033);
	&data_word(0x87c74e49, 0xd9c1d138, 0x8cfea2ca, 0x98360bd4);
	&data_word(0xa6cf81f5, 0xa528de7a, 0xda268eb7, 0x3fa4bfad);
	&data_word(0x2ce49d3a, 0x500d9278, 0x6a9bcc5f, 0x5462467e);
	&data_word(0xf6c2138d, 0x90e8b8d8, 0x2e5ef739, 0x82f5afc3);
	&data_word(0x9fbe805d, 0x697c93d0, 0x6fa92dd5, 0xcfb31225);
	&data_word(0xc83b99ac, 0x10a77d18, 0xe86e639c, 0xdb7bbb3b);
	&data_word(0xcd097826, 0x6ef41859, 0xec01b79a, 0x83a89a4f);
	&data_word(0xe6656e95, 0xaa7ee6ff, 0x2108cfbc, 0xefe6e815);
	&data_word(0xbad99be7, 0x4ace366f, 0xead4099f, 0x29d67cb0);
	&data_word(0x31afb2a4, 0x2a31233f, 0xc63094a5, 0x35c066a2);
	&data_word(0x7437bc4e, 0xfca6ca82, 0xe0b0d090, 0x3315d8a7);
	&data_word(0xf14a9804, 0x41f7daec, 0x7f0e50cd, 0x172ff691);
	&data_word(0x768dd64d, 0x434db0ef, 0xcc544daa, 0xe4df0496);
	&data_word(0x9ee3b5d1, 0x4c1b886a, 0xc1b81f2c, 0x467f5165);
	&data_word(0x9d04ea5e, 0x015d358c, 0xfa737487, 0xfb2e410b);
	&data_word(0xb35a1d67, 0x9252d2db, 0xe9335610, 0x6d1347d6);
	&data_word(0x9a8c61d7, 0x377a0ca1, 0x598e14f8, 0xeb893c13);
	&data_word(0xceee27a9, 0xb735c961, 0xe1ede51c, 0x7a3cb147);
	&data_word(0x9c59dfd2, 0x553f73f2, 0x1879ce14, 0x73bf37c7);
	&data_word(0x53eacdf7, 0x5f5baafd, 0xdf146f3d, 0x7886db44);
	&data_word(0xca81f3af, 0xb93ec468, 0x382c3424, 0xc25f40a3);
	&data_word(0x1672c31d, 0xbc0c25e2, 0x288b493c, 0xff41950d);
	&data_word(0x397101a8, 0x08deb30c, 0xd89ce4b4, 0x6490c156);
	&data_word(0x7b6184cb, 0xd570b632, 0x48745c6c, 0xd04257b8);
#Td4:
	&data_word(0x52525252, 0x09090909, 0x6a6a6a6a, 0xd5d5d5d5);
	&data_word(0x30303030, 0x36363636, 0xa5a5a5a5, 0x38383838);
	&data_word(0xbfbfbfbf, 0x40404040, 0xa3a3a3a3, 0x9e9e9e9e);
	&data_word(0x81818181, 0xf3f3f3f3, 0xd7d7d7d7, 0xfbfbfbfb);
	&data_word(0x7c7c7c7c, 0xe3e3e3e3, 0x39393939, 0x82828282);
	&data_word(0x9b9b9b9b, 0x2f2f2f2f, 0xffffffff, 0x87878787);
	&data_word(0x34343434, 0x8e8e8e8e, 0x43434343, 0x44444444);
	&data_word(0xc4c4c4c4, 0xdededede, 0xe9e9e9e9, 0xcbcbcbcb);
	&data_word(0x54545454, 0x7b7b7b7b, 0x94949494, 0x32323232);
	&data_word(0xa6a6a6a6, 0xc2c2c2c2, 0x23232323, 0x3d3d3d3d);
	&data_word(0xeeeeeeee, 0x4c4c4c4c, 0x95959595, 0x0b0b0b0b);
	&data_word(0x42424242, 0xfafafafa, 0xc3c3c3c3, 0x4e4e4e4e);
	&data_word(0x08080808, 0x2e2e2e2e, 0xa1a1a1a1, 0x66666666);
	&data_word(0x28282828, 0xd9d9d9d9, 0x24242424, 0xb2b2b2b2);
	&data_word(0x76767676, 0x5b5b5b5b, 0xa2a2a2a2, 0x49494949);
	&data_word(0x6d6d6d6d, 0x8b8b8b8b, 0xd1d1d1d1, 0x25252525);
	&data_word(0x72727272, 0xf8f8f8f8, 0xf6f6f6f6, 0x64646464);
	&data_word(0x86868686, 0x68686868, 0x98989898, 0x16161616);
	&data_word(0xd4d4d4d4, 0xa4a4a4a4, 0x5c5c5c5c, 0xcccccccc);
	&data_word(0x5d5d5d5d, 0x65656565, 0xb6b6b6b6, 0x92929292);
	&data_word(0x6c6c6c6c, 0x70707070, 0x48484848, 0x50505050);
	&data_word(0xfdfdfdfd, 0xedededed, 0xb9b9b9b9, 0xdadadada);
	&data_word(0x5e5e5e5e, 0x15151515, 0x46464646, 0x57575757);
	&data_word(0xa7a7a7a7, 0x8d8d8d8d, 0x9d9d9d9d, 0x84848484);
	&data_word(0x90909090, 0xd8d8d8d8, 0xabababab, 0x00000000);
	&data_word(0x8c8c8c8c, 0xbcbcbcbc, 0xd3d3d3d3, 0x0a0a0a0a);
	&data_word(0xf7f7f7f7, 0xe4e4e4e4, 0x58585858, 0x05050505);
	&data_word(0xb8b8b8b8, 0xb3b3b3b3, 0x45454545, 0x06060606);
	&data_word(0xd0d0d0d0, 0x2c2c2c2c, 0x1e1e1e1e, 0x8f8f8f8f);
	&data_word(0xcacacaca, 0x3f3f3f3f, 0x0f0f0f0f, 0x02020202);
	&data_word(0xc1c1c1c1, 0xafafafaf, 0xbdbdbdbd, 0x03030303);
	&data_word(0x01010101, 0x13131313, 0x8a8a8a8a, 0x6b6b6b6b);
	&data_word(0x3a3a3a3a, 0x91919191, 0x11111111, 0x41414141);
	&data_word(0x4f4f4f4f, 0x67676767, 0xdcdcdcdc, 0xeaeaeaea);
	&data_word(0x97979797, 0xf2f2f2f2, 0xcfcfcfcf, 0xcececece);
	&data_word(0xf0f0f0f0, 0xb4b4b4b4, 0xe6e6e6e6, 0x73737373);
	&data_word(0x96969696, 0xacacacac, 0x74747474, 0x22222222);
	&data_word(0xe7e7e7e7, 0xadadadad, 0x35353535, 0x85858585);
	&data_word(0xe2e2e2e2, 0xf9f9f9f9, 0x37373737, 0xe8e8e8e8);
	&data_word(0x1c1c1c1c, 0x75757575, 0xdfdfdfdf, 0x6e6e6e6e);
	&data_word(0x47474747, 0xf1f1f1f1, 0x1a1a1a1a, 0x71717171);
	&data_word(0x1d1d1d1d, 0x29292929, 0xc5c5c5c5, 0x89898989);
	&data_word(0x6f6f6f6f, 0xb7b7b7b7, 0x62626262, 0x0e0e0e0e);
	&data_word(0xaaaaaaaa, 0x18181818, 0xbebebebe, 0x1b1b1b1b);
	&data_word(0xfcfcfcfc, 0x56565656, 0x3e3e3e3e, 0x4b4b4b4b);
	&data_word(0xc6c6c6c6, 0xd2d2d2d2, 0x79797979, 0x20202020);
	&data_word(0x9a9a9a9a, 0xdbdbdbdb, 0xc0c0c0c0, 0xfefefefe);
	&data_word(0x78787878, 0xcdcdcdcd, 0x5a5a5a5a, 0xf4f4f4f4);
	&data_word(0x1f1f1f1f, 0xdddddddd, 0xa8a8a8a8, 0x33333333);
	&data_word(0x88888888, 0x07070707, 0xc7c7c7c7, 0x31313131);
	&data_word(0xb1b1b1b1, 0x12121212, 0x10101010, 0x59595959);
	&data_word(0x27272727, 0x80808080, 0xecececec, 0x5f5f5f5f);
	&data_word(0x60606060, 0x51515151, 0x7f7f7f7f, 0xa9a9a9a9);
	&data_word(0x19191919, 0xb5b5b5b5, 0x4a4a4a4a, 0x0d0d0d0d);
	&data_word(0x2d2d2d2d, 0xe5e5e5e5, 0x7a7a7a7a, 0x9f9f9f9f);
	&data_word(0x93939393, 0xc9c9c9c9, 0x9c9c9c9c, 0xefefefef);
	&data_word(0xa0a0a0a0, 0xe0e0e0e0, 0x3b3b3b3b, 0x4d4d4d4d);
	&data_word(0xaeaeaeae, 0x2a2a2a2a, 0xf5f5f5f5, 0xb0b0b0b0);
	&data_word(0xc8c8c8c8, 0xebebebeb, 0xbbbbbbbb, 0x3c3c3c3c);
	&data_word(0x83838383, 0x53535353, 0x99999999, 0x61616161);
	&data_word(0x17171717, 0x2b2b2b2b, 0x04040404, 0x7e7e7e7e);
	&data_word(0xbabababa, 0x77777777, 0xd6d6d6d6, 0x26262626);
	&data_word(0xe1e1e1e1, 0x69696969, 0x14141414, 0x63636363);
	&data_word(0x55555555, 0x21212121, 0x0c0c0c0c, 0x7d7d7d7d);
&function_end_B("AES_decrypt");

sub enckey()
{
	&movz	("esi",&LB("edx"));		# rk[i]>>0
	&mov	("ebx",&DWP(0,"ebp","esi",4));
	&movz	("esi",&HB("edx"));		# rk[i]>>8
	&and	("ebx",0xFF000000);
	&xor	("eax","ebx");

	&mov	("ebx",&DWP(0,"ebp","esi",4));
	&shr	("edx",16);
	&and	("ebx",0x000000FF);
	&movz	("esi",&LB("edx"));		# rk[i]>>16
	&xor	("eax","ebx");

	&mov	("ebx",&DWP(0,"ebp","esi",4));
	&movz	("esi",&HB("edx"));		# rk[i]>>24
	&and	("ebx",0x0000FF00);
	&xor	("eax","ebx");

	&mov	("ebx",&DWP(0,"ebp","esi",4));
	&and	("ebx",0x00FF0000);
	&xor	("eax","ebx");

	&xor	("eax",&DWP(1024,"ebp","ecx",4));	# rcon
}

# int AES_set_encrypt_key(const unsigned char *userKey, const int bits,
#                        AES_KEY *key)
&public_label("AES_Te");
&function_begin("AES_set_encrypt_key");
	&mov	("esi",&wparam(0));		# user supplied key
	&mov	("edi",&wparam(2));		# private key schedule

	&test	("esi",-1);
	&jz	(&label("badpointer"));
	&test	("edi",-1);
	&jz	(&label("badpointer"));

	&call	(&label("pic_point"));
	&set_label("pic_point");
	&blindpop("ebp");
	&lea	("ebp",&DWP(&label("AES_Te")."-".&label("pic_point"),"ebp"));
	&add	("ebp",1024*4);			# skip to Te4

	&mov	("ecx",&wparam(1));		# number of bits in key
	&cmp	("ecx",128);
	&je	(&label("10rounds"));
	&cmp	("ecx",192);
	&je	(&label("12rounds"));
	&cmp	("ecx",256);
	&je	(&label("14rounds"));
	&mov	("eax",-2);			# invalid number of bits
	&jmp	(&label("exit"));

    &set_label("10rounds");
	&mov	("eax",&DWP(0,"esi"));		# copy first 4 dwords
	&mov	("ebx",&DWP(4,"esi"));
	&mov	("ecx",&DWP(8,"esi"));
	&mov	("edx",&DWP(12,"esi"));
	&mov	(&DWP(0,"edi"),"eax");
	&mov	(&DWP(4,"edi"),"ebx");
	&mov	(&DWP(8,"edi"),"ecx");
	&mov	(&DWP(12,"edi"),"edx");

	&xor	("ecx","ecx");
	&jmp	(&label("10shortcut"));

	&align	(4);
	&set_label("10loop");
		&mov	("eax",&DWP(0,"edi"));		# rk[0]
		&mov	("edx",&DWP(12,"edi"));		# rk[3]
	&set_label("10shortcut");
		&enckey	();

		&mov	(&DWP(16,"edi"),"eax");		# rk[4]
		&xor	("eax",&DWP(4,"edi"));
		&mov	(&DWP(20,"edi"),"eax");		# rk[5]
		&xor	("eax",&DWP(8,"edi"));
		&mov	(&DWP(24,"edi"),"eax");		# rk[6]
		&xor	("eax",&DWP(12,"edi"));
		&mov	(&DWP(28,"edi"),"eax");		# rk[7]
		&inc	("ecx");
		&add	("edi",16);
		&cmp	("ecx",10);
	&jl	(&label("10loop"));

	&mov	(&DWP(80,"edi"),10);		# setup number of rounds
	&xor	("eax","eax");
	&jmp	(&label("exit"));
		
    &set_label("12rounds");
	&mov	("eax",&DWP(0,"esi"));		# copy first 6 dwords
	&mov	("ebx",&DWP(4,"esi"));
	&mov	("ecx",&DWP(8,"esi"));
	&mov	("edx",&DWP(12,"esi"));
	&mov	(&DWP(0,"edi"),"eax");
	&mov	(&DWP(4,"edi"),"ebx");
	&mov	(&DWP(8,"edi"),"ecx");
	&mov	(&DWP(12,"edi"),"edx");
	&mov	("ecx",&DWP(16,"esi"));
	&mov	("edx",&DWP(20,"esi"));
	&mov	(&DWP(16,"edi"),"ecx");
	&mov	(&DWP(20,"edi"),"edx");

	&xor	("ecx","ecx");
	&jmp	(&label("12shortcut"));

	&align	(4);
	&set_label("12loop");
		&mov	("eax",&DWP(0,"edi"));		# rk[0]
		&mov	("edx",&DWP(20,"edi"));		# rk[5]
	&set_label("12shortcut");
		&enckey	();

		&mov	(&DWP(24,"edi"),"eax");		# rk[6]
		&xor	("eax",&DWP(4,"edi"));
		&mov	(&DWP(28,"edi"),"eax");		# rk[7]
		&xor	("eax",&DWP(8,"edi"));
		&mov	(&DWP(32,"edi"),"eax");		# rk[8]
		&xor	("eax",&DWP(12,"edi"));
		&mov	(&DWP(36,"edi"),"eax");		# rk[9]

		&cmp	("ecx",7);
		&je	(&label("12break"));
		&inc	("ecx");

		&xor	("eax",&DWP(16,"edi"));
		&mov	(&DWP(40,"edi"),"eax");		# rk[10]
		&xor	("eax",&DWP(20,"edi"));
		&mov	(&DWP(44,"edi"),"eax");		# rk[11]

		&add	("edi",24);
	&jmp	(&label("12loop"));

	&set_label("12break");
	&mov	(&DWP(72,"edi"),12);		# setup number of rounds
	&xor	("eax","eax");
	&jmp	(&label("exit"));

    &set_label("14rounds");
	&mov	("eax",&DWP(0,"esi"));		# copy first 8 dwords
	&mov	("ebx",&DWP(4,"esi"));
	&mov	("ecx",&DWP(8,"esi"));
	&mov	("edx",&DWP(12,"esi"));
	&mov	(&DWP(0,"edi"),"eax");
	&mov	(&DWP(4,"edi"),"ebx");
	&mov	(&DWP(8,"edi"),"ecx");
	&mov	(&DWP(12,"edi"),"edx");
	&mov	("eax",&DWP(16,"esi"));
	&mov	("ebx",&DWP(20,"esi"));
	&mov	("ecx",&DWP(24,"esi"));
	&mov	("edx",&DWP(28,"esi"));
	&mov	(&DWP(16,"edi"),"eax");
	&mov	(&DWP(20,"edi"),"ebx");
	&mov	(&DWP(24,"edi"),"ecx");
	&mov	(&DWP(28,"edi"),"edx");

	&xor	("ecx","ecx");
	&jmp	(&label("14shortcut"));

	&align	(4);
	&set_label("14loop");
		&mov	("edx",&DWP(28,"edi"));		# rk[7]
	&set_label("14shortcut");
		&mov	("eax",&DWP(0,"edi"));		# rk[0]

		&enckey	();

		&mov	(&DWP(32,"edi"),"eax");		# rk[8]
		&xor	("eax",&DWP(4,"edi"));
		&mov	(&DWP(36,"edi"),"eax");		# rk[9]
		&xor	("eax",&DWP(8,"edi"));
		&mov	(&DWP(40,"edi"),"eax");		# rk[10]
		&xor	("eax",&DWP(12,"edi"));
		&mov	(&DWP(44,"edi"),"eax");		# rk[11]

		&cmp	("ecx",6);
		&je	(&label("14break"));
		&inc	("ecx");

		&mov	("edx","eax");
		&mov	("eax",&DWP(16,"edi"));		# rk[4]
		&movz	("esi",&LB("edx"));		# rk[11]>>0
		&mov	("ebx",&DWP(0,"ebp","esi",4));
		&movz	("esi",&HB("edx"));		# rk[11]>>8
		&and	("ebx",0x000000FF);
		&xor	("eax","ebx");

		&mov	("ebx",&DWP(0,"ebp","esi",4));
		&shr	("edx",16);
		&and	("ebx",0x0000FF00);
		&movz	("esi",&LB("edx"));		# rk[11]>>16
		&xor	("eax","ebx");

		&mov	("ebx",&DWP(0,"ebp","esi",4));
		&movz	("esi",&HB("edx"));		# rk[11]>>24
		&and	("ebx",0x00FF0000);
		&xor	("eax","ebx");

		&mov	("ebx",&DWP(0,"ebp","esi",4));
		&and	("ebx",0xFF000000);
		&xor	("eax","ebx");

		&mov	(&DWP(48,"edi"),"eax");		# rk[12]
		&xor	("eax",&DWP(20,"edi"));
		&mov	(&DWP(52,"edi"),"eax");		# rk[13]
		&xor	("eax",&DWP(24,"edi"));
		&mov	(&DWP(56,"edi"),"eax");		# rk[14]
		&xor	("eax",&DWP(28,"edi"));
		&mov	(&DWP(60,"edi"),"eax");		# rk[15]

		&add	("edi",32);
	&jmp	(&label("14loop"));

	&set_label("14break");
	&mov	(&DWP(48,"edi"),14);		# setup number of rounds
	&xor	("eax","eax");
	&jmp	(&label("exit"));

    &set_label("badpointer");
	&mov	("eax",-1);
    &set_label("exit");
&function_end("AES_set_encrypt_key");

sub deckey()
{ my ($i,$ptr,$te4,$td) = @_;

	&mov	("eax",&DWP($i,$ptr));
	&mov	("edx","eax");
	&movz	("ebx",&HB("eax"));
	&shr	("edx",16);
	&and	("eax",0xFF);
	&movz	("eax",&BP(0,$te4,"eax",4));
	&movz	("ebx",&BP(0,$te4,"ebx",4));
	&mov	("eax",&DWP(1024*0,$td,"eax",4));
	&xor	("eax",&DWP(1024*1,$td,"ebx",4));
	&movz	("ebx",&HB("edx"));
	&and	("edx",0xFF);
	&movz	("edx",&BP(0,$te4,"edx",4));
	&movz	("ebx",&BP(0,$te4,"ebx",4));
	&xor	("eax",&DWP(1024*2,$td,"edx",4));
	&xor	("eax",&DWP(1024*3,$td,"ebx",4));
	&mov	(&DWP($i,$ptr),"eax");
}

# int AES_set_decrypt_key(const unsigned char *userKey, const int bits,
#                        AES_KEY *key)
&public_label("AES_Td");
&public_label("AES_Te");
&function_begin_B("AES_set_decrypt_key");
	&mov	("eax",&wparam(0));
	&mov	("ecx",&wparam(1));
	&mov	("edx",&wparam(2));
	&sub	("esp",12);
	&mov	(&DWP(0,"esp"),"eax");
	&mov	(&DWP(4,"esp"),"ecx");
	&mov	(&DWP(8,"esp"),"edx");
	&call	("AES_set_encrypt_key");
	&add	("esp",12);
	&cmp	("eax",0);
	&je	(&label("proceed"));
	&ret	();

    &set_label("proceed");
	&push	("ebp");
	&push	("ebx");
	&push	("esi");
	&push	("edi");

	&mov	("esi",&wparam(2));
	&mov	("ecx",&DWP(240,"esi"));	# pull number of rounds
	&lea	("ecx",&DWP(0,"","ecx",4));
	&lea	("edi",&DWP(0,"esi","ecx",4));	# pointer to last chunk

	&align	(4);
	&set_label("invert");			# invert order of chunks
		&mov	("eax",&DWP(0,"esi"));
		&mov	("ebx",&DWP(4,"esi"));
		&mov	("ecx",&DWP(0,"edi"));
		&mov	("edx",&DWP(4,"edi"));
		&mov	(&DWP(0,"edi"),"eax");
		&mov	(&DWP(4,"edi"),"ebx");
		&mov	(&DWP(0,"esi"),"ecx");
		&mov	(&DWP(4,"esi"),"edx");
		&mov	("eax",&DWP(8,"esi"));
		&mov	("ebx",&DWP(12,"esi"));
		&mov	("ecx",&DWP(8,"edi"));
		&mov	("edx",&DWP(12,"edi"));
		&mov	(&DWP(8,"edi"),"eax");
		&mov	(&DWP(12,"edi"),"ebx");
		&mov	(&DWP(8,"esi"),"ecx");
		&mov	(&DWP(12,"esi"),"edx");
		&add	("esi",16);
		&sub	("edi",16);
		&cmp	("esi","edi");
	&jne	(&label("invert"));

	&call	(&label("pic_point"));
	&set_label("pic_point");
	blindpop("ebp");
	&lea	("edi",&DWP(&label("AES_Td")."-".&label("pic_point"),"ebp"));
	&lea	("ebp",&DWP(&label("AES_Te")."-".&label("pic_point"),"ebp"));
	&add	("ebp",1024*4);			# skip to Te4

	&mov	("esi",&wparam(2));
	&mov	("ecx",&DWP(240,"esi"));	# pull number of rounds
	&dec	("ecx");
	&align	(4);
	&set_label("permute");			# permute the key schedule
		&add	("esi",16);
		&deckey	(0,"esi","ebp","edi");
		&deckey	(4,"esi","ebp","edi");
		&deckey	(8,"esi","ebp","edi");
		&deckey	(12,"esi","ebp","edi");
		&dec	("ecx");
	&jnz	(&label("permute"));

	&xor	("eax","eax");			# return success
&function_end("AES_set_decrypt_key");

&asm_finish();
