/*
 *  Copyright (C) 2006  MakeHuman Project
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *  
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *  
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 *  File   : Rect.h
 *  Project: MakeHuman <info@makehuman.org>, http://www.makehuman.org/
 *  Library: MHGUI
 *
 *  For individual developers look into the AUTHORS file.
 *   
 */

#ifndef RECT_H
#define RECT_H 1

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <stdint.h>
#include <iostream>
#include "Point.h"
#include "Size.h"

/*!
 * Trivial class which stores a rectangle (pos, size) in integer notation. 
 * This is supposed to be used by the GUI.
 */

namespace mhgui {

class Rect
{
public:
  Rect(int32_t inX, int32_t inY, int32_t inWidth, int32_t inHeight);
  Rect(const Rect& inRect);

  ~Rect();

  Rect& operator=(const Rect& inPoint);

  const Point& getPos  () const {return pos;}
  const Size&  getSize () const {return size;}

  int32_t getX() const { return pos.getX(); }
  int32_t getY() const { return pos.getY(); }

  int32_t getWidth()  const { return size.getWidth(); }
  int32_t getHeight() const { return size.getHeight(); }

  bool isEmpty() const { return size.isEmpty(); }

  bool  operator== (const Rect& inRect) const;

  void moveTo   (const Point& inPos);
  void moveBy   (const Point& inDeltaPos);

  void resizeTo (const Size& inSize);
  void resizeBy (const Size& inDeltaSize);

  void inset(int32_t inInsetX, int32_t inInsetY);

  bool isHitBy(const Point& inPoint) const;

  /* ==========================================================================
   * Members
   * ========================================================================== */
private:
  Point  pos;
  Size   size;

  /* ==========================================================================
   * Static Methods
   * ========================================================================== */
public:

  /* ==========================================================================
   * Static Members
   * ========================================================================== */
public:
}; // class Rect

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline Rect::Rect(int32_t inX, int32_t inY, int32_t inWidth, int32_t inHeight)
    : pos(inX, inY), size(inWidth, inHeight)
{
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline Rect::Rect(const Rect& inRect)
    : pos(inRect.pos), size(inRect.size)
{
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline Rect::~Rect()
{
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline Rect& Rect::operator=(const Rect& inPoint)
{
  pos = inPoint.pos; size = inPoint.size; return *this;
}

/* ========================================================================== */
/** Check if two rectangles are identical
 */
/* ========================================================================== */
inline bool Rect::operator== (const Rect& inRect) const
{
  if (this == &inRect)  // The same object?
    return true;

  return (pos  == inRect.pos &&
          size == inRect.size);
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline void Rect::moveTo (const Point& inPos)
{
  pos = inPos;
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline void Rect::moveBy (const Point& inDeltaPos)
{
  pos.moveBy(inDeltaPos);
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline void Rect::resizeTo (const Size& inSize)
{
  size = inSize;
}

/* ========================================================================== */
/**
 */
/* ========================================================================== */
inline void Rect::resizeBy (const Size& inDeltaSize)
{
  size.resizeBy(inDeltaSize);
}

/* ========================================================================== */
/** Returns an copy of this Rectangle, which is inset by X and Y.
 */
/* ========================================================================== */
inline void Rect::inset(int32_t inInsetX, int32_t inInsetY)
{
  moveBy   (Point (     inInsetX,      inInsetY));
  resizeBy (Size  (-2 * inInsetX, -2 * inInsetY));
}

/* ========================================================================== */
/** Check if a particular point is located in this Rectangle
 */
/* ========================================================================== */
inline bool Rect::isHitBy(const Point& inPoint) const
{
  return (((inPoint.getX() >= getX()) &&
           (inPoint.getY() >= getY()))
          &&
          ((inPoint.getX() <  getX() + getWidth()) &&
           (inPoint.getY() <  getY() + getHeight())));
}

/* ==========================================================================
 * Inlines & Templates
 * ========================================================================== */

} // namespace mhgui

#endif // RECT_H
