<?php
/**
 * SyncML Backend for the Horde Application framework.
 *
 * The backend provides the following functionality:
 *
 * 1) handling of the actual data, i.e.
 *    a) add/replace/delete entries to and retrieve entries from the
 *       backend
 *    b) retrieve history to find out what entries have been changed
 * 2) managing of the map between cliend IDs and server IDs
 * 3) store information about sync anchors (timestamps) of previous
 *    successfuls sync sessions
 * 4) session handling (not yet, still to be done)
 * 5) authorisation (not yet, still to be done)
 * 6) logging
 *
 * Copyright 2005 Karsten Fourmont <karsten@horde.org>
 *
 * See the enclosed file COPYING for license information (LGPL). If you did not
 * receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * $Horde: framework/SyncML/SyncML/Backend.php,v 1.8.2.1 2005/07/07 14:59:00 chuck Exp $
 *
 * @author  Karsten Fourmont <karsten@horde.org>
 * @package Horde_SyncML
 */
class SyncMLBackend_Horde {

    /**
     * Retrieves an entry from the backend.
     *
     * @param string $database   database to scan. i.e.
     *                           calendar/tasks/contacts/notes
     * @param string $suid       server unique id of the entry: for horde
     *                           this is the guid.
     * @param string contentType contentType: the mime type in which the
     *                           function shall return the data
     *
     * @return mixed             a string with the data entry or
     *                           or a PEAR_Error object.
     */
    function retrieveEntry($database, $suid, $contentType)
    {
        global $registry;
        $c = $registry->call($database . '/export',
                                 array('guid' => $suid, 'contentType' => $contentType));
        return $c;
    }

    /**
     * Get entries that have been modified in the server database.
     *
     * @param string $syncIdentifier  Identifies the client device to allow the
     *                                user to sync with different devices.
     *                                Normally the SourceURI from the
     *                                SyncHeader
     * @param string $database        Database to scan. i.e.
     *                                calendar/tasks/contacts/notes
     * @param integer $from_ts        Start timestamp.
     * @param integer $to_ts          Exclusive end timestamp. Not yet
     *                                implemented.
     *
     * @return array  PEAR_Error or assoc array of changes with key=suid,
     *                value=cuid. If no cuid can be found for a suid, value is
     *                null. Then a Sync Add command has to be created to add
     *                this entry in the client database.
     */
    function getServerModifications($syncIdentifier, $database, $from_ts, $to_ts)
    {
        global $registry;

        // Get changes.
        $changes = $registry->call($database. '/listBy', array('action' => 'modify', 'timestamp' => $from_ts));
        if (is_a($changes, 'PEAR_Error')) {
            $this->logMessage("SyncML: $database/listBy failed for modify:"
                              . $changes->getMessage(),
                              __FILE__, __LINE__, PEAR_LOG_WARNING);
            return array();
        }

        $r = array();

        foreach ($changes as $suid) {
            $suid_ts = $registry->call($database . '/getActionTimestamp', array($suid, 'modify'));
            $sync_ts = $this->getChangeTS($syncIdentifier, $database, $suid);
            if ($sync_ts && $sync_ts >= $suid_ts) {
                // Change was done by us upon request of client.
                // Don't mirror that back to the client.
                $this->logMessage("change: $suid ignored, came from client", __FILE__, __LINE__, PEAR_LOG_DEBUG);
                continue;
            }
            $cuid = $this->getCuid($syncIdentifier, $database, $suid);
            if (!$cuid) {
                $this->logMessage("Unable to create change for $suid: client-id not found in map. Trying add instead.",
                                  __FILE__, __LINE__, PEAR_LOG_WARNING);
                $r[$suid] = null;
            } else {
                $r[$suid] = $cuid;
            }
        }

        return $r;
    }

    /**
     * Get entries that have been deleted from the server database.
     *
     * @param string $database  Database to scan. i.e.
     *                          calendar/tasks/contacts/notes
     * @param integer $from_ts
     * @param integer $to_ts    Exclusive
     *
     * @return array  PEAR_Error or assoc array of deletions with key=suid,
     *                value=cuid.
     */
    function getServerDeletions($syncIdentifier, $database, $from_ts, $to_ts)
    {
        global $registry;

        // Get deletions.
        $deletes = $registry->call($database. '/listBy', array('action' => 'delete', 'timestamp' => $from_ts));

        if (is_a($deletes, 'PEAR_Error')) {
            $this->logMessage("SyncML: $database/listBy failed for delete:"
                              . $deletes->getMessage(),
                              __FILE__, __LINE__, PEAR_LOG_WARNING);
            return array();
        }

        $r = array();

        foreach ($deletes as $suid) {
            $suid_ts = $registry->call($database. '/getActionTimestamp', array($suid, 'delete'));
            $sync_ts = $this->getChangeTS($syncIdentifier, $database, $suid);
            if ($sync_ts && $sync_ts >= $suid_ts) {
                // Change was done by us upon request of client.
                // Don't mirror that back to the client.
                $this->logMessage("SyncML: delete $suid ignored, came from client", __FILE__, __LINE__, PEAR_LOG_DEBUG);
                continue;
            }
            $cuid = $this->getCuid($syncIdentifier, $database, $suid);
            if (!$cuid) {
                $this->logMessage("Unable to create delete for $suid: locid not found in map",
                                  __FILE__, __LINE__, PEAR_LOG_WARNING);
                continue;
            }

            $r[$suid] = $cuid;
        }

        return $r;
    }

    /**
     * Get entries that have been added to the server database.
     *
     * @param string $database  Database to scan. i.e.
     *                          calendar/tasks/contacts/notes
     * @param integer $from_ts
     * @param integer $to_ts    Exclusive
     *
     * @return array  PEAR_Error or assoc array of deletions with key=suid,
     *                value=0. (array style is chosen to match change & del)
     */
    function getServerAdditions($syncIdentifier, $database, $from_ts, $to_ts)
    {
        global $registry;

        $adds = $registry->call($database. '/listBy', array('action' => 'add', 'timestamp' => $from_ts));
        if (is_a($adds, 'PEAR_Error')) {
            $this->logMessage("SyncML: $database/listBy failed for add:"
                              . $adds->getMessage(),
                              __FILE__, __LINE__, PEAR_LOG_WARNING);
            return array();
        }

        $r = array();

        foreach ($adds as $suid) {
            $suid_ts = $registry->call($database . '/getActionTimestamp', array($suid, 'add'));
            $sync_ts = $this->getChangeTS($syncIdentifier, $database, $suid);
            if ($sync_ts && $sync_ts >= $suid_ts) {
                // Change was done by us upon request of client.
                // Don't mirror that back to the client.
                $this->logMessage("add: $suid ignored, came from client", __FILE__, __LINE__, PEAR_LOG_DEBUG);
                continue;
            }
            // $this->logMessage("add: sync_ts: $sync_ts, suid_ts: $suid_ts", __FILE__, __LINE__, PEAR_LOG_DEBUG);

            $cuid = $this->getCuid($syncIdentifier, $database, $suid);

            if ($cuid && $from_ts == 0) {
                // For slow sync (ts=0): do not add data for which we
                // have a locid again.  This is a heuristic to avoid
                // duplication of entries.
                $this->logMessage("skipping add of guid $suid as there already is a cuid $cuid", __FILE__, __LINE__, PEAR_LOG_DEBUG);
                continue;
            }

            $r[$suid] = 0;
        }

        return $r;
    }

    /**
     * Adds an entry into the server database.
     *
     * @param string $database     Database where to add.
     *                             calendar/tasks/contacts/notes
     * @param string $content      The actual data
     * @param string $contentType  Mimetype of $content
     * @param string $cuid         Client ID of this entry (for map)
     *
     * @return array  PEAR_Error or suid (Horde guid) of new entry
     */
    function importEntry($syncIdentifier, $database, $content, $contentType, $cuid)
    {
        global $registry;

        $tasksandcalendarcombined = false;

        // Checks if the client sends us a vtodo in a calendar sync:
        if ($database == 'calendar'
             && preg_match('/(\r\n|\r|\n)BEGIN[^:]*:VTODO/', "\n" . $content)) {
            $serverdatabase = 'tasks';
        } else {
            $serverdatabase = $database;
        }

        $suid = $registry->call($serverdatabase . '/import',
                                array($content, $contentType));

        $this->logMessage("add to db $serverdatabase cuid $cuid -> suid $suid", __FILE__, __LINE__, PEAR_LOG_DEBUG);

        if (!is_a($suid, 'PEAR_Error')) {
            $ts = $registry->call($serverdatabase. '/getActionTimestamp', array($suid, 'add'));
            if (!$ts) {
                $this->logMessage('SyncML: unable to find add-ts for ' . $suid . ' at ' . $ts, __FILE__, __LINE__, PEAR_LOG_ERR);
            }
            $this->createUidMap($syncIdentifier, $database, $cuid, $suid, $ts);
        }

        return $suid;
    }

    /**
     * Deletes an entry from the server database.
     *
     * @param string $database  Database where to add.
     *                          calendar/tasks/contacts/notes
     * @param string $cuid      Client ID of the entry
     *
     * @return array  PEAR_Error or suid (Horde guid) of deleted entry.
     */
    function deleteEntry($syncIdentifier, $database, $cuid)
    {
        global $registry;

        // Find server ID for this entry:
        $suid = $this->getSuid($syncIdentifier, $database, $cuid);
        if (!is_a($suid, 'PEAR_Error')) {
            $registry->call($database. '/delete', array($suid));
            $ts = $registry->call($database . '/getActionTimestamp', array($suid, 'delete'));
            // We can't remove the mapping entry as we need to keep
            // the timestamp information.
            $this->createUidMap($syncIdentifier, $database, $cuid, $suid, $ts);
        }

        return $suid;
    }

    /**
     * Replaces an entry in the server database.
     *
     * @param string $database     Database where to replace.
     *                             calendar/tasks/contacts/notes
     * @param string $content      The actual data
     * @param string $contentType  Mimetype of $content
     * @param string $cuid         Client ID of this entry
     *
     * @return array  PEAR_Error or suid (Horde guid) of modified entry.
     */
    function replaceEntry($syncIdentifier, $database, $content, $contentType, $cuid)
    {
        global $registry;

        // Checks if the client sends us a vtodo in a calendar sync:
        if ($database == 'calendar'
             && preg_match('/(\r\n|\r|\n)BEGIN[^:]*:VTODO/', "\n" . $content)) {
            $serverdatabase = 'tasks';
        } else {
            $serverdatabase = $database;
        }

        $suid = $this->getSuid($syncIdentifier, $database, $cuid);

        $this->logMessage("replace in db $serverdatabase cuid $cuid suid $suid", __FILE__, __LINE__, PEAR_LOG_DEBUG);

        if ($suid) {
            // Entry exists: replace current one.
            $ok = $registry->call($serverdatabase . '/replace',
                                  array($suid, $content, $contentType));
            if (is_a($ok, 'PEAR_Error')) {
                return $ok;
            }
            $ts = $registry->call($serverdatabase . '/getActionTimestamp', array($suid, 'modify'));
            $this->createUidMap($syncIdentifier, $database, $cuid, $suid, $ts);
        } else {
            return PEAR::raiseError('No map entry found');
        }

        return $suid;
    }

    // Functions for handling anchors from previous syncs.

    /**
     * Retrieves information about the previous sync if any. Returns
     * false if no info found or a DateTreeObject with at least the
     * following attributes:
     *
     * ClientAnchor: the clients Next Anchor of the previous sync.
     * ServerAnchor: the Server Next Anchor of the previous sync.
     */
    function &getSyncSummary($syncIdentifier, $type)
    {
        $dt = &$this->getDataTree();

        $id = $dt->getId($syncIdentifier . ':summary:' . $type);
        if (is_a($id, 'PEAR_Error')) {
            return false;
        }

        return $dt->getObjectById($id);
    }

    /**
     * After a successful sync, the client and server's Next Anchors
     * are written to the database so they can be used to negotiate
     * upcoming syncs.
     */
    function writeSyncSummary($syncIdentifier,
                              $clientAnchorNext, $serverAnchorNext)
    {
        if (!isset($serverAnchorNext) || !is_array($serverAnchorNext)) {
            $this->logMessage('SyncML internal error: no anchor provided '
                              . 'in writeSyncSummary',
                              __FILE__, __LINE__, PEAR_LOG_ERR);
            die('SyncML internal error: no anchor provided in writeSyncSummary');
        }

        $dt = &$this->getDataTree();

        foreach (array_keys($serverAnchorNext) as $type) {
            $s = $syncIdentifier . ':summary:' . $type;

            // Set $locid.
            $info = &new DataTreeObject($s);
            $info->set('ClientAnchor', $clientAnchorNext);
            $info->set('ServerAnchor', $serverAnchorNext);
            $r = $dt->add($info);
            if (is_a($r, 'PEAR_Error')) {
                // Object already exists: update instead.
                $dt->updateData($info);
            }
        }
    }

    // Functions for handling cuid <-> suid maps.

    /**
     * Create a map entries to map between server and client IDs.
     *
     * Puts a given client $cuid and Horde server $suid pair into the
     * map table to allow mapping between the client's and server's
     * IDs.  Actually there are two maps: from the suid to the cuid
     * and vice versa.
     * If an entry already exists, it is overwritten.
     */
    function createUidMap($syncIdentifier, $type, $cuid, $suid, $ts=0)
    {
        $dt = &$this->getDataTree();

        // Set $cuid.
        $gid = &new DataTreeObject($syncIdentifier . ':' . $type
                                    . ':suid2cuid:' . $suid);
        $gid->set('datastore', $type);
        $gid->set('cuid', $cuid);
        $gid->set('ts', $ts);

        $r = $dt->add($gid);
        if (is_a($r, 'PEAR_Error')) {
            // Object already exists: update instead.
            $r = $dt->updateData($gid);
        }
        $this->dieOnError($r, __FILE__, __LINE__);

        // Set $globaluid
        $lid = &new DataTreeObject($syncIdentifier . ':' . $type
                                   . ':cuid2suid:' . $cuid);
        $lid->set('suid', $suid);
        $r = $dt->add($lid);
        if (is_a($r, 'PEAR_Error')) {
            // object already exists: update instead.
            $r = $dt->updateData($lid);
        }
        $this->dieOnError($r, __FILE__, __LINE__);

        // If tasks and events are handled at once, we need to store
        // the map entry in both databases:
        $session =& $_SESSION['SyncML.state'];
        $device =& $session->getDevice();

        if ($device->handleTasksInCalendar()
            && ($type == 'tasks' || $type == 'calendar') ) {
            $type = $type == 'tasks' ? 'calendar' : 'tasks' ; // the other one

            // Set $cuid.
            $gid = &new DataTreeObject($syncIdentifier . ':' . $type
                                       . ':suid2cuid:' . $suid);
            $gid->set('datastore', $type);
            $gid->set('cuid', $cuid);
            $gid->set('ts', $ts);

            $r = $dt->add($gid);
            if (is_a($r, 'PEAR_Error')) {
                // Object already exists: update instead.
                $r = $dt->updateData($gid);
            }
            $this->dieOnError($r, __FILE__, __LINE__);

            // Set $globaluid
            $lid = &new DataTreeObject($syncIdentifier . ':' . $type
                                       . ':cuid2suid:' . $cuid);
            $lid->set('suid', $suid);
            $r = $dt->add($lid);
            if (is_a($r, 'PEAR_Error')) {
                // Object already exists: update instead.
                $r = $dt->updateData($lid);
            }
            $this->dieOnError($r, __FILE__, __LINE__);
        }
    }

    /**
     * Returns the timestamp (if set) of the last change to the
     * obj:guid, that was caused by the client.
     *
     * @access private
     *
     * This is stored to
     * avoid mirroring these changes back to the client.
     */
    function getChangeTS($syncIdentifier, $database, $suid)
    {
        $dt = &$this->getDataTree();

        $id = $dt->getId($syncIdentifier . ':' . $database
                            . ':suid2cuid:' . $suid);
        if (is_a($id, 'PEAR_Error')) {
            return false;
        }

        $gid = $dt->getObjectById($id);
        if (is_a($gid, 'PEAR_Error')) {
            return false;
        }

        return $gid->get('ts');
    }

    /**
     * Retrieves the Horde server guid (like
     * kronolith:0d1b415fc124d3427722e95f0e926b75) for a given client
     * cuid. Returns false if no such id is stored yet.
     *
     * @access private
     *
     * Opposite of getLocId which returns the locid for a given guid.
     */
    function getSuid($syncIdentifier, $type, $cuid)
    {
        $dt = &$this->getDataTree();

        $id = $dt->getId($syncIdentifier . ':' . $type
                         . ':cuid2suid:' . $cuid);
        if (is_a($id, 'PEAR_Error')) {
            return false;
        }
        $lid = $dt->getObjectById($id);
        if (is_a($lid, 'PEAR_Error')) {
            return false;
        }

        return $lid->get('suid');
    }

    /**
     * Converts a suid server id (i.e. Horde GUID) to a cuid client ID
     * as used by the sync client (like 12) returns false if no such
     * id is stored yet.
     *
     * @access private
     */
    function getCuid($syncIdentifier, $database, $suid)
    {
        $dt = &$this->getDataTree();
        $id = $dt->getId($syncIdentifier . ':' . $database
                         . ':suid2cuid:' . $suid);
        if (is_a($id, 'PEAR_Error')) {
            return false;
        }

        $gid = $dt->getObjectById($id);
        if (is_a($gid, 'PEAR_Error')) {
            return false;
        }

        return $gid->get('cuid');
    }

    /**
     * Logs a message in the backend.
     *
     * @param mixed $message     Either a string or a PEAR_Error object.
     * @param string $file       What file was the log function called from
     *                           (e.g. __FILE__)?
     * @param integer $line      What line was the log function called from
     *                           (e.g. __LINE__)?
     * @param integer $priority  The priority of the message. One of:
     * <pre>
     * PEAR_LOG_EMERG
     * PEAR_LOG_ALERT
     * PEAR_LOG_CRIT
     * PEAR_LOG_ERR
     * PEAR_LOG_WARNING
     * PEAR_LOG_NOTICE
     * PEAR_LOG_INFO
     * PEAR_LOG_DEBUG
     * </pre>
     */
    function logMessage($message, $file = __FILE__, $line = __LINE__,
                        $priority = PEAR_LOG_INFO)
    {
        if (is_string($message)) {
            $message = "SyncML: " . $message;
        }
        Horde::logMessage($message, $file, $line, $priority);
    }

    // Various private functions.

    /**
     * Returns the DataTree used as persistence layer for SyncML.
     *
     * @access private
     *
     * @return DataTree  The DataTree object.
     */
    function &getDataTree()
    {
        $driver = $GLOBALS['conf']['datatree']['driver'];
        $params = Horde::getDriverConfig('datatree', $driver);
        $params = array_merge($params, array( 'group' => 'syncml' ));

        return DataTree::singleton($driver, $params);
    }

    /**
     * This is a small helper function that can be included to check
     * whether a given $obj is a PEAR_Error or not. If so, it logs
     * to debug, var_dumps the $obj and exits.
     */
    function dieOnError($obj, $file = __FILE__, $line = __LINE__)
    {
        if (!is_a($obj, 'PEAR_Error')) {
            return;
        }

        $this->logMessage('SyncML: PEAR Error: ' . $obj->getMessage(), $file, $line, PEAR_LOG_ERR);
        print "PEAR ERROR\n\n";
        var_dump($obj);
    }

}
