/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.core.rte.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.definition.DataModelElementFactory;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.RefreshManager;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;
import org.eclipse.wst.rdb.internal.models.sql.constraints.IncrementType;
import org.eclipse.wst.rdb.internal.models.sql.constraints.Index;
import org.eclipse.wst.rdb.internal.models.sql.constraints.IndexMember;
import org.eclipse.wst.rdb.internal.models.sql.constraints.SQLConstraintsPackage;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.PredefinedDataType;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.ReferentialActionType;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.tables.BaseTable;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;
import org.eclipse.wst.rdb.internal.models.sql.tables.impl.PersistentTableImpl;


public class JDBCTable extends PersistentTableImpl implements ICatalogObject {

	public synchronized void refresh() {
		this.columnsLoaded = false;

		if (this.constraintsLoaded) {
			this.constraintsLoaded = false;
			this.constraints.clear();
		}
		if (this.indexesLoaded) {
			this.indexesLoaded = false;
			this.index.clear();
		}

		RefreshManager.getInstance().referesh(this);

	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getCatalogDatabase();
		return ((JDBCDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getSchema().getDatabase();		
	}
	
	public EList getColumns(){
		if (!this.columnsLoaded) this.loadColumns();
		return this.columns;
	}
	
	public EList getConstraints() {
		if(!this.constraintsLoaded) this.loadConstraints();
		return this.constraints;
	}
	
	public EList getIndex() {
		if (!this.indexesLoaded) this.loadIndexes();
		return this.index;
	}
	
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLTablesPackage.PERSISTENT_TABLE__COLUMNS) {
			this.getColumns();
		} 
		else if(id == SQLTablesPackage.PERSISTENT_TABLE__CONSTRAINTS) {
			this.getConstraints();
		} 
		else if(id == SQLTablesPackage.PERSISTENT_TABLE__INDEX) {
			this.getIndex();
		} 
		return super.eIsSet(eFeature);
	}
	
	private synchronized void loadColumns() {
		if(this.columnsLoaded) return;
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);
		try {
			JDBCTable.loadColumns(this.getConnection(), super.getColumns(), this);
		}
		catch(Exception e) {
		}
		this.columnsLoaded = true;
		this.eSetDeliver(deliver);
	}

	private synchronized void loadIndexes() {
		if(this.indexesLoaded) return;

		this.indexesLoaded = true;
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);
		try {
			JDBCTable.loadIndexes(this.getConnection(), super.getIndex(), this);
		}
		catch(Exception e) {
		}
		this.eSetDeliver(deliver);
	}
	
	private synchronized void loadConstraints() {
		if(this.constraintsLoaded) return;
		this.constraintsLoaded = true;
		
		Connection connection = this.getConnection();
		if(connection == null) return;
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);	
		
		try {
			DatabaseMetaData metaData = connection.getMetaData();
			JDBCPrimaryKey  pk= this.loadPrimaryKey(metaData,super.getConstraints());
			
			this.loadForeignKey(metaData,super.getConstraints());
			
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		this.eSetDeliver(deliver);
	}
	
	public static void loadColumns(Connection connection, EList columnList, Table table) throws SQLException {
		final Schema schema = table.getSchema();
		final Database database = schema.getDatabase();
		final DatabaseDefinition databaseDefinition = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(database);
		final DataModelElementFactory factory = databaseDefinition.getDataModelElementFactory();

		Object[] list = columnList.toArray();
		columnList.clear();
		
		try {
			DatabaseMetaData metaData = connection.getMetaData();
			ResultSet r = metaData.getColumns(null, schema.getName(), table.getName(), null);
			while(r.next()) {
				final String columnName = r.getString(4);

				Object element = JDBCTable.findElement(list, columnName,SQLTablesPackage.eINSTANCE.getColumn());
				
				Column column;
				if (element != null){
					column = (Column) element;
					((ICatalogObject)element).refresh();
				} else {
					column = new JDBCColumn();
					column.setName(columnName);
				}	
				
				final String remarks = r.getString(12);
				column.setDescription(remarks);

				String defaultValue = r.getString(13);
				column.setDefaultValue(defaultValue);

				String typeName = r.getString(6);
				
				PredefinedDataTypeDefinition typeDefinition = databaseDefinition.getPredefinedDataTypeDefinition(typeName);
				if(typeDefinition != null) {
					PredefinedDataType type = databaseDefinition.getPredefinedDataType(typeDefinition);
					if(typeDefinition.isLengthSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("length");  //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(7)));
					}
					else if(typeDefinition.isPrecisionSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("precision"); //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(10)));
					}
					
					if(typeDefinition.isScaleSupported()) {
						EStructuralFeature feature = type.eClass().getEStructuralFeature("scale"); //$NON-NLS-1$
						type.eSet(feature, new Integer(r.getInt(9)));
					}
					column.setContainedType(type);
				}
				else {
					System.out.println("Unresolved datatype: " + typeName); //$NON-NLS-1$
					Iterator it = databaseDefinition.getPredefinedDataTypes();
					while (it.hasNext()) {
						PredefinedDataTypeDefinition datatype = (PredefinedDataTypeDefinition)it.next();
						PredefinedDataType type = databaseDefinition.getPredefinedDataType(datatype);
						column.setContainedType(type);
						break;
					}
				}

				final String nulls = r.getString(18);
				if(nulls.equals("YES")) column.setNullable(true);  //$NON-NLS-1$
				else column.setNullable(false);

				columnList.add(column);
			}
			r.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	
	public static void loadIndexes(Connection connection, EList indexList, Table table) throws SQLException {
		try {
			final Schema schema = table.getSchema();
			final Database database = schema.getDatabase();
			final DatabaseDefinition databaseDefinition = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(database);
			final DataModelElementFactory factory = databaseDefinition.getDataModelElementFactory();
			
			DatabaseMetaData metaData = connection.getMetaData();
			ResultSet r = metaData.getIndexInfo(null,schema.getName(),table.getName(),false,false);
			Index index = null;
			String indexName=""; //$NON-NLS-1$
			while(r.next()) {
				final String indName = r.getString(6);
				if (indName == null) continue;
				if (!indName.equals(indexName)){
					indexName = indName;
					index = new JDBCIndex();
					index.setName(indName);
	
					final String indSchema = r.getString(2);
					index.setSchema(JDBCTable.getSchema(table,indSchema));

					final boolean isUnqiue = r.getBoolean(4);
					index.setUnique(isUnqiue);

					final short type = r.getShort(7);
					if (type == DatabaseMetaData.tableIndexClustered) {
						index.setClustered(true);
					}
					
					indexList.add(index);
				}
				final String column_name= r.getString(9);
				if (column_name !=null){
					IndexMember member = (IndexMember) factory.create(SQLConstraintsPackage.eINSTANCE.getIndexMember()); 
					member.setColumn(JDBCTable.getColumn(table,column_name));
					final String order = r.getString(10);
					if (order != null) {
						if(order.equals("A"))  //$NON-NLS-1$
							member.setIncrementType(IncrementType.ASC_LITERAL);
						else if(order.equals("D"))  //$NON-NLS-1$
							member.setIncrementType(IncrementType.DESC_LITERAL);
					}
					index.getMembers().add(member);
				}
				
			}
			r.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	private synchronized JDBCPrimaryKey loadPrimaryKey(DatabaseMetaData metaData, EList constraintList){
		JDBCPrimaryKey key = null;
		try {
			ResultSet r = metaData.getPrimaryKeys(null, this.getSchema().getName(),this.getName());
	
			DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();

			KeyColumnCollection columns = new KeyColumnCollection(this);  
			
			while(r.next()) {
				if(key == null) {
					String name = r.getString(6);
					key = new JDBCPrimaryKey();
					key.setName(name);
					constraintList.add(key);
				}
				String columnName = r.getString(4);
				int seq = r.getInt(5);
				columns.add(seq, columnName);
			}
			r.close();
			Iterator it = columns.iterator();
			while(it.hasNext()) {
				Column column = (Column) it.next();
				key.getMembers().add(column); 
			}
		
		}catch (Exception e){
			e.printStackTrace();
		}
		
		return key;
	}
	

//	private synchronized void loadForeignKey(DatabaseMetaData metaData, UniqueConstraint uk){
//		try {
//			ResultSet r = metaData.getExportedKeys(null, this.getSchema().getName(),this.getName());
//	
//			DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();
//			
//			JDBCForeignKey fk = null;
//			Table fkTable=null;
//			String fkTableName=""; //$NON-NLS-1$
//			while(r.next()) {
//				final String fkSchema_Name= r.getString(6);
//				final String fkTable_Name = r.getString(7);
//				if(!fkTableName.equals(fkTable_Name)) {
//					fkTable= this.getTable(fkSchema_Name,fkTable_Name);
//					if (fkTable==null) continue;
//					fkTableName = fkTable_Name;
//					fk = new JDBCForeignKey();
//					final String fkName = r.getString(12);
//					fk.setName(fkName);
//					fk.setUniqueConstraint(uk);
//					short updateRule = r.getShort(10);
//					switch(updateRule) {
//						case DatabaseMetaData.importedKeyCascade:
//							fk.setOnUpdate(ReferentialActionType.CASCADE_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeyRestrict:
//							fk.setOnUpdate(ReferentialActionType.RESTRICT_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeySetDefault:
//							fk.setOnUpdate(ReferentialActionType.SET_DEFAULT_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeySetNull:
//							fk.setOnUpdate(ReferentialActionType.SET_NULL_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeyNoAction:
//							default:
//							fk.setOnUpdate(ReferentialActionType.NO_ACTION_LITERAL);
//							break;
//					}
//					short deleteRule = r.getShort(11);
//					switch(deleteRule) {
//						case DatabaseMetaData.importedKeyCascade:
//							fk.setOnDelete(ReferentialActionType.CASCADE_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeyRestrict:
//							fk.setOnDelete(ReferentialActionType.RESTRICT_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeySetDefault:
//							fk.setOnDelete(ReferentialActionType.SET_DEFAULT_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeySetNull:
//							fk.setOnDelete(ReferentialActionType.SET_NULL_LITERAL);
//							break;
//						case DatabaseMetaData.importedKeyNoAction:
//							default:
//							fk.setOnDelete(ReferentialActionType.NO_ACTION_LITERAL);
//							break;
//					}
//					short deferrability = r.getShort(14);
//					switch(deferrability) {
//						case DatabaseMetaData.importedKeyInitiallyDeferred:
//							fk.setDeferrable(true);
//							fk.setInitiallyDeferred(true);
//							break;
//						case DatabaseMetaData.importedKeyInitiallyImmediate:
//							fk.setDeferrable(true);
//							fk.setInitiallyDeferred(false);
//							break;
//						case DatabaseMetaData.importedKeyNotDeferrable:
//						default:
//							fk.setDeferrable(false);
//							break;
//					}
//					((BaseTable)fkTable).getConstraints().add(fk);
//				}
//				
//				String columnName = r.getString(8);
//				Column column = JDBCTable.getColumn(fkTable,columnName);
//				fk.getMembers().add(column);
//				/* Add EAnnotation */
//				EList tmpList = fk.getMembers();
//				boolean isIdentify = fk.isIdentifyingRelationship(tmpList);
//				EAnnotation eAnnotation = fk.addEAnnotation(RDBCorePlugin.FK_MODELING_RELATIONSHIP);
//				fk.addEAnnotationDetail(eAnnotation,RDBCorePlugin.FK_IS_IDENTIFYING_RELATIONSHIP,new Boolean(isIdentify).toString());
//
//				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_CHILD_MULTIPLICITY, RDBCorePlugin.MANY);
//				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_CHILD_ROLE_NAME, new String ());
//				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_PARENT_MULTIPLICITY, (fk.getMembers().size() > 0) ? RDBCorePlugin.ZERO_TO_ONE : RDBCorePlugin.ONE);
//				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_PARENT_ROLE_NAME, new String ());
//			}
//			r.close();
//		}catch (Exception e){
//			e.printStackTrace();
//		}
//	}
	
	
	private synchronized void loadForeignKey(DatabaseMetaData metaData, EList constraintList){
		try {
			ResultSet r = metaData.getImportedKeys(null, this.getSchema().getName(),this.getName());
	
			DataModelElementFactory factory = RDBCorePlugin.getDefault().getDatabaseDefinitionRegistry().getDefinition(this.getCatalogDatabase()).getDataModelElementFactory();
			
			JDBCForeignKey fk = null;
			BaseTable pkTable=null;
			String pkTableName=""; //$NON-NLS-1$
			while(r.next()) {
				final String pkSchema_Name = r.getString(2);
				final String pkTable_Name = r.getString(3);
				final int keySeq = r.getInt(9);
				if(!pkTableName.equals(pkTable_Name) || keySeq==1) {
					pkTable= this.getTable(pkSchema_Name,pkTable_Name);
					if (pkTable==null) continue;
					pkTableName = pkTable_Name;
					fk = new JDBCForeignKey();
					fk.setName(r.getString(12));
					fk.setUniqueConstraint(pkTable.getPrimaryKey());
					fk.setReferencedTable(pkTable);
					short updateRule = r.getShort(10);
					switch(updateRule) {
						case DatabaseMetaData.importedKeyCascade:
							fk.setOnUpdate(ReferentialActionType.CASCADE_LITERAL);
							break;
						case DatabaseMetaData.importedKeyRestrict:
							fk.setOnUpdate(ReferentialActionType.RESTRICT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetDefault:
							fk.setOnUpdate(ReferentialActionType.SET_DEFAULT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetNull:
							fk.setOnUpdate(ReferentialActionType.SET_NULL_LITERAL);
							break;
						case DatabaseMetaData.importedKeyNoAction:
							default:
							fk.setOnUpdate(ReferentialActionType.NO_ACTION_LITERAL);
							break;
					}
					short deleteRule = r.getShort(11);
					switch(deleteRule) {
						case DatabaseMetaData.importedKeyCascade:
							fk.setOnDelete(ReferentialActionType.CASCADE_LITERAL);
							break;
						case DatabaseMetaData.importedKeyRestrict:
							fk.setOnDelete(ReferentialActionType.RESTRICT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetDefault:
							fk.setOnDelete(ReferentialActionType.SET_DEFAULT_LITERAL);
							break;
						case DatabaseMetaData.importedKeySetNull:
							fk.setOnDelete(ReferentialActionType.SET_NULL_LITERAL);
							break;
						case DatabaseMetaData.importedKeyNoAction:
							default:
							fk.setOnDelete(ReferentialActionType.NO_ACTION_LITERAL);
							break;
					}
					short deferrability = r.getShort(14);
					switch(deferrability) {
						case DatabaseMetaData.importedKeyInitiallyDeferred:
							fk.setDeferrable(true);
							fk.setInitiallyDeferred(true);
							break;
						case DatabaseMetaData.importedKeyInitiallyImmediate:
							fk.setDeferrable(true);
							fk.setInitiallyDeferred(false);
							break;
						case DatabaseMetaData.importedKeyNotDeferrable:
						default:
							fk.setDeferrable(false);
							break;
					}
					constraintList.add(fk);
				}
				
				String columnName = r.getString(8);
				Column column = JDBCTable.getColumn(this,columnName);
				fk.getMembers().add(column);
				/* Add EAnnotation */
				EList tmpList = fk.getMembers();
				boolean isIdentify = fk.isIdentifyingRelationship(tmpList);
				EAnnotation eAnnotation = fk.addEAnnotation(RDBCorePlugin.FK_MODELING_RELATIONSHIP);
				fk.addEAnnotationDetail(eAnnotation,RDBCorePlugin.FK_IS_IDENTIFYING_RELATIONSHIP,new Boolean(isIdentify).toString());

				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_CHILD_MULTIPLICITY, RDBCorePlugin.MANY);
				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_CHILD_ROLE_NAME, new String ());
				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_PARENT_MULTIPLICITY, (fk.getMembers().size() > 0) ? RDBCorePlugin.ZERO_TO_ONE : RDBCorePlugin.ONE);
				fk.addEAnnotationDetail(eAnnotation, RDBCorePlugin.FK_PARENT_ROLE_NAME, new String ());
			}
			r.close();
		}catch (Exception e){
			e.printStackTrace();
		}
	}
	
	
	private static Object findElement(Object[] list, String name,EClass metaclass){
		Object object = null;
		for (int i = 0; i < list.length; i++){
			SQLObject sqlObject = (SQLObject) list[i];
			if (sqlObject.getName().equals(name) && sqlObject.eClass() == metaclass && sqlObject instanceof ICatalogObject){
				object = list[i];
				break;
			}
		}
		return object;
	}
	
	private static Schema getSchema(Table table, String schemaName) {
		Schema s = table.getSchema();
		if(s.getName().equals(schemaName)) return s;
		Database d = s.getDatabase();
		Iterator it = d.getSchemas().iterator();
		while(it.hasNext()) {
			s = (Schema) it.next();
			if(s.getName().equals(schemaName)) return s;
		}
		return null;
	}

	private BaseTable getTable(String schemaName, String tableName) {
		Schema schema = JDBCTable.getSchema(this,schemaName);
		Iterator it = schema.getTables().iterator();
		while(it.hasNext()) {
			Table table = (Table) it.next();
			if(table.getName().equals(tableName) && table instanceof BaseTable) return (BaseTable)table;			
		}
		return null;		
	}
	
	private static Column getColumn(Table table,String columnName) {
		Iterator it = table.getColumns().iterator();
		while(it.hasNext()) {
			Column c = (Column) it.next();
			if(c.getName().equals(columnName)) return c;
		}
		return null;
	}
	
	
	private static class KeyColumnCollection {
		public KeyColumnCollection(BaseTable table) {
			this.table = table;
		}
		
		public void add(int seq, String name) {
			Column column = this.getColumn(name);
			String key = "k" + seq; //$NON-NLS-1$
			this.keyMap.put(key, column);
		}
		
		public Iterator iterator() {
			return keyMap.values().iterator();
		}
		
		private Column getColumn(String name) {
			Iterator it = this.table.getColumns().iterator();
			while(it.hasNext()) {
				Column column = (Column) it.next();
				if(column.getName().equals(name)) return column;
			}
			return null;
		}
		
		private Map keyMap = new TreeMap();
		private BaseTable table;
	}
	
	private boolean columnsLoaded = false;
	private boolean constraintsLoaded = false;
	private boolean indexesLoaded = false;
}
