/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.core.rte.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.ibm.icu.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionFilter;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.rte.RefreshManager;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLObject;
import org.eclipse.wst.rdb.internal.models.sql.schema.SQLSchemaPackage;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.internal.models.sql.schema.impl.SchemaImpl;
import org.eclipse.wst.rdb.internal.models.sql.tables.SQLTablesPackage;
import org.eclipse.wst.rdb.internal.models.sql.tables.Table;

public class JDBCSchema extends SchemaImpl implements ICatalogObject {

	public synchronized void refresh() {
		this.tablesLoaded = false;

		RefreshManager.getInstance().referesh(this);
	}

	public boolean isSystemObject() {
		return false;
	}

	public Connection getConnection() {
		Database database = this.getDatabase();
		return ((JDBCDatabase) database).getConnection();
	}
	
	public Database getCatalogDatabase() {
		return this.getDatabase();
	}

	public EList getTables(){
		if (!this.tablesLoaded) this.loadTables();
		return this.tables;
	}
	
	
	public boolean eIsSet(EStructuralFeature eFeature) {
		int id = eDerivedStructuralFeatureID(eFeature);
		if(id == SQLSchemaPackage.SCHEMA__TABLES) {
			this.getTables();
		}
		return super.eIsSet(eFeature);
	}
	
	private synchronized void loadTables() {
		if(this.tablesLoaded) return;
 		EList tableList = super.getTables();
		
		boolean deliver = this.eDeliver();
		this.eSetDeliver(false);
		Connection connection = this.getConnection();
		try {
			
			JDBCSchema.loadTables(connection,tableList,this);
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		this.tablesLoaded = true;
		this.eSetDeliver(deliver);
	}

	

	
	public static void loadTables(Connection connection, EList tableList, Schema schema) throws SQLException {
		Object[] oldList = tableList.toArray();
		tableList.clear();
	
		try {
			String catalogName =null;
			DatabaseMetaData metaData = connection.getMetaData();
			if(metaData.supportsCatalogsInTableDefinitions()) {
				catalogName = connection.getCatalog();
			}
			
			ConnectionInfo connectionInfo = RDBCorePlugin.getDefault().getConnectionManager().getConnectionInfo(schema.getDatabase());
			ConnectionFilter filter = connectionInfo.getFilter(schema.getName()+"::"+ConnectionFilter.TABLE_FILTER); //$NON-NLS-1$
			if (filter == null) {  //if schema filter is null, then get default filter
				filter = connectionInfo.getFilter(ConnectionFilter.TABLE_FILTER);
			}
			String[] tableAry = null;
			String pattern = null;
			if (filter != null) {
				String tableFilter = filter.getPredicate();
				tableFilter = tableFilter.replaceAll(" ",""); //$NON-NLS-1$ //$NON-NLS-2$
				if(tableFilter.startsWith("IN(")) //$NON-NLS-1$
				{
					tableFilter = tableFilter.substring(3, tableFilter.length() - 1); // skip "IN(" and ")"
					tableFilter = tableFilter.replaceAll(",",""); //$NON-NLS-1$ //$NON-NLS-2$
					tableAry = parseINClause(tableFilter);
				}
				if(tableFilter.startsWith("LIKE")) //$NON-NLS-1$
				{
					pattern = parseLikeClause(filter.getPredicate()); 
				}
			}
			
			ResultSet r = null;
			if(pattern != null)
			{
				r = metaData.getTables(catalogName, schema.getName(), pattern, new String[] {"TABLE","VIEW","SYSTEM TABLE"}); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			}
			else
			{
				r = metaData.getTables(catalogName, schema.getName(), null, new String[] {"TABLE","VIEW", "TABLE_VIEW","SYSTEM TABLE"}); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
			while(r.next()) {
				String tableName = r.getString(3);
				String type      = r.getString(4);
				String remarks   = r.getString(5);
				Table table;
				
				if(tableAry != null)
				{
					boolean found = false;
					for(int i = 0; i < tableAry.length; i++)
					{
						if(tableName.equals(tableAry[i])) 
						{
							found = true;
							break; // for
						}
					}
					if(!found) continue; // while
				}
				EClass metaclass=null;
				if (type.equals("TABLE") || type.equals("SYSTEM TABLE")){ //$NON-NLS-1$ //$NON-NLS-2$
					metaclass = SQLTablesPackage.eINSTANCE.getPersistentTable();
				} else if (type.equals("VIEW")){ //$NON-NLS-1$
					metaclass = SQLTablesPackage.eINSTANCE.getViewTable();
				}

				Object element = JDBCSchema.findElement(oldList,tableName,metaclass);

				if (element != null) {
					table = (Table) element;
					((ICatalogObject)table).refresh();
				} else {
					if(type.equals("TABLE") || type.equals("SYSTEM TABLE")) { //$NON-NLS-1$ //$NON-NLS-2$
						table = new JDBCTable();
					}
					else if(type.equals("VIEW")) { //$NON-NLS-1$
						table = new JDBCView();
					}
					else continue;
					table.setName(tableName);
				}
				
				table.setDescription(remarks);
				tableList.add(table);
			}
			r.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}	
	
	private static Object findElement(Object[] list, String name,EClass metaclass){
		Object object = null;
		for (int i = 0; i < list.length; i++){
			SQLObject sqlObject = (SQLObject) list[i];
			if (sqlObject.getName().equals(name) && sqlObject.eClass() == metaclass && sqlObject instanceof ICatalogObject){
				object = list[i];
				break;
			}
		}
		return object;
	}
	/* This method can be moved to util package so that all others metadata
	 * processing would have access to it
	 */
		protected static String[] parseINClause(String toParse)
		{
	        StringTokenizer tokenizer = new StringTokenizer(toParse, "'"); //$NON-NLS-1$
	        Vector list = new Vector();
	        String result = null;
	        if(tokenizer.countTokens () >= 1) {
	        	while (tokenizer.hasMoreTokens()) {
	            	list.add(tokenizer.nextToken());
	            }
	        }
	        String[] retStrList = null;
	        if (list.size() > 0)
	        {
	        	retStrList = new String[list.size()];
	        	for(int i = 0; i < list.size(); i++){
	        		retStrList[i] = (String)list.get(i);
	        	}
	        }
	        return retStrList;
			
		}
		
		/* This method can be moved to util package so that all others metadata
		 * processing would have access to it
		 */
		protected static String parseLikeClause(String toParse)
		{
	        String retString = toParse.substring(toParse.indexOf("'") + 1, toParse.length() - 1); // Strip off begin LIKE' and end ' //$NON-NLS-1$
	        return retString;
		}
	
	
	private boolean tablesLoaded = false;
}
