\name{AIC-methods}
\docType{methods}
\alias{AIC-methods}
\alias{AIC,ANY-method}
\title{Methods for Function `AIC' in Package `stats4'}
\description{
  Calculate the Akaike information criterion for a fitted model object.
}
\section{Methods}{
  \describe{
    \item{object = "ANY"}{Generic function: see \code{\link[stats]{AIC}}.}
  }
}
\keyword{methods}
\eof
\name{BIC}
\docType{genericFunction}
\alias{BIC}
\alias{BIC,ANY-method}
\alias{BIC,logLik-method}
\title{Bayesian Information Criterion}
\description{
  This generic function calculates the Bayesian information criterion,
  also known as Schwarz's Bayesian criterion (SBC), for one or several
  fitted model objects for which a log-likelihood value can be obtained,
  according to the formula \eqn{-2 \mbox{log-likelihood} + n_{par}
    \log(n_{obs})}{-2*log-likelihood + npar*log(nobs)}, where
  \eqn{n_{par}}{npar}  represents the
  number of parameters and \eqn{n_{obs}}{nobs} the number of
  observations in the fitted model. 
}
\usage{
BIC(object, \dots)
}
\arguments{
  \item{object}{An object of a suitable class for the BIC to be
    calculated - usually a \code{"logLik"} object or an object for which
    a \code{\link[stats4:logLik-methods]{logLik}} method exists.
  }
  \item{\dots}{Some methods for this generic function may take
    additional, optional arguments.  At present none do.}
}
\value{
  Returns a numeric value with the corresponding BIC.
}
\references{
  Schwarz, G. (1978) "Estimating the Dimension of a Model", Annals of
  Statistics, 6, 461-464.
}
%\author{}
%\note{}
\seealso{\code{\link{logLik-methods}}, \code{\link{AIC-methods}}}
\examples{
data(swiss)
lm1 <- lm(Fertility ~ . , data = swiss)
AIC(lm1)
BIC(lm1)

\testonly{
## 2 equivalent ways of calculating the BIC:
stopifnot(all.equal(AIC(lm1, k=log(nrow(swiss))), BIC(lm1)))
}
}
\keyword{models}

\eof
\name{coef-methods}
\docType{methods}
\alias{coef-methods}
\alias{coef,ANY-method}
\alias{coef,mle-method}
\alias{coef,summary.mle-method}
\title{Methods for Function `coef' in Package `stats4'}
\description{
  Extract the coefficient vector from \code{"mle"} objects.
}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function: see \code{\link[stats]{coef}}.}

\item{object = "mle"}{Extract the full coefficient vector (including any
  fixed coefficients) from the fit.}
\item{object = "summary.mle"}{Extract the coefficient vector and standard
  errors from the summary of the fit.} 
}}
\keyword{methods}
\eof
\name{confint-methods}
\docType{methods}
\alias{confint-methods}
\alias{confint,ANY-method}
\alias{confint,profile.mle-method}
\alias{confint,mle-method}
\title{Methods for Function `confint' in Package `stats4'}
\description{Generate confidence intervals}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function: see \code{\link[stats]{confint}}.}

\item{object = "mle"}{First generate profile and then confidence
  intervals from the profile.}

\item{object = "profile.mle"}{Generate confidence intervals based on
  likelihood profile.}
}}
\keyword{methods}
\eof
\name{logLik-methods}
\docType{methods}
\alias{logLik-methods}
\alias{logLik,ANY-method}
\alias{logLik,mle-method}
\title{Methods for Function `logLik' in Package `stats4'}
\description{
  Extract the maximized log-likelihood from \code{"mle"} objects.
}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function: see \code{\link[stats]{logLik}}.}

\item{object = "mle"}{Extract log-likelihood from the fit.}
}}
\keyword{methods}
\eof
\name{mle-class}
\docType{class}
\alias{mle-class}
\title{Class "mle". Result of Maximum Likelihood Estimation.}
\description{This class encapsulates results of a generic maximum
  likelihood procedure.}
\section{Objects from the Class}{
  Objects can be created by calls of the form \code{new("mle", \dots)}, but
  most often as the result of a call to \code{\link{mle}}.
}
\section{Slots}{
  \describe{
    \item{\code{call}:}{Object of class \code{"language"}.
      The call to \code{\link{mle}}.}
    \item{\code{coef}:}{Object of class \code{"numeric"}.  Estimated
      parameters.}
    \item{\code{fullcoef}:}{Object of class \code{"numeric"}.
      Fixed and estimated parameters.}
    \item{\code{vcov}:}{Object of class \code{"matrix"}.  Approximate
      variance-covariance matrix.}
    \item{\code{min}:}{Object of class \code{"numeric"}.  Minimum value
      of objective function.}
    \item{\code{details}:}{Object of class \code{"list"}.  List returned
      from \code{\link{optim}}.}
    \item{\code{minuslogl}:}{Object of class \code{"function"}.  The
      negative loglikelihood function.}
    \item{\code{method}:}{Object of class \code{"character"}.  The
      optimization method used.}
  }
}
\section{Methods}{
  \describe{
    \item{confint}{\code{signature(object = "mle")}: Confidence
      intervals from likelihood profiles.}
    \item{logLik}{\code{signature(object = "mle")}: Extract maximized
      log-likelihood.}
    \item{profile}{\code{signature(fitted = "mle")}: Likelihood profile
      generation.}
    \item{show}{\code{signature(object = "mle")}: Display object briefly.}
    \item{summary}{\code{signature(object = "mle")}: Generate object summary.}
    \item{update}{\code{signature(object = "mle")}:  Update fit.}
    \item{vcov}{\code{signature(object = "mle")}: Extract
      variance-covariance matrix.}
  }
}
\keyword{classes}
\eof
\name{mle}
\alias{mle}
\title{Maximum Likelihood Estimation}
\description{
  Estimate parameters by the method of maximum likelihood.
}
\usage{
mle(minuslogl, start = formals(minuslogl), method = "BFGS",
    fixed = list(), \dots)
}
\arguments{
  \item{minuslogl}{Function to calculate negative log-likelihood.}
  \item{start}{Named list. Initial values for optimizer.}
  \item{method}{Optimization method to use. See \code{\link{optim}}.}
  \item{fixed}{Named list.  Parameter values to keep fixed during
    optimization.}
  \item{\dots}{Further arguments to pass to \code{\link{optim}}.}
}
\details{
  The \code{\link{optim}} optimizer is used to find the minimum of the
  negative log-likelihood.  An approximate covariance matrix for the
  parameters is obtained by inverting the Hessian matrix at the optimum.
}
\value{
  An object of class \code{"mle"}.
}
\note{
  Be careful to note that the argument is -log L (not -2 log L). It
  is for the user to ensure that the likelihood is correct, and that
  asymptotic likelihood inference is valid.
}
\seealso{
  \code{\link{mle-class}}
}
\examples{
x <- 0:10
y <- c(26, 17, 13, 12, 20, 5, 9, 8, 5, 4, 8)
ll <- function(ymax=15, xhalf=6)
    -sum(stats::dpois(y, lambda=ymax/(1+x/xhalf), log=TRUE))
(fit <- mle(ll))
mle(ll, fixed=list(xhalf=6))

summary(fit)
logLik(fit)
vcov(fit)
plot(profile(fit), absVal=FALSE)
confint(fit)

## use bounded optimization
## the lower bounds are really > 0, but we use >=0 to stress-test profiling
(fit1 <- mle(ll, method="L-BFGS-B", lower=c(0, 0)))
plot(profile(fit1), absVal=FALSE)

## a better parametrization:
ll2 <- function(lymax=log(15), lxhalf=log(6))
    -sum(stats::dpois(y, lambda=exp(lymax)/(1+x/exp(lxhalf)), log=TRUE))
(fit2 <- mle(ll2))
plot(profile(fit2), absVal=FALSE)
exp(confint(fit2))
}
\keyword{models}

\eof
\name{plot-methods}
\docType{methods}
\alias{plot-methods}
\alias{plot,ANY,ANY-method}
\alias{plot,profile.mle,missing-method}
\title{Methods for Function `plot' in Package `stats4'}
\description{
  Plot profile likelihoods for \code{"mle"} objects.
}
\usage{
\S4method{plot}{profile.mle,missing}(x, levels, conf = c(99, 95, 90, 80, 50)/100, nseg = 50,
     absVal = TRUE, \dots)
}
\arguments{
 \item{x}{an object of class \code{"profile.mle"} }
 \item{levels}{levels, on the scale of the absolute value of a t
     statistic, at which to interpolate intervals.  Usually \code{conf}
     is used instead of giving \code{levels} explicitly.}
 \item{conf}{a numeric vector of confidence levels for profile-based
     confidence intervals on the parameters.}
 \item{nseg}{an integer value giving the number of segments to use in
     the spline interpolation of the profile t curves.}
 \item{absVal}{a logical value indicating whether or not the plots
     should be on the scale of the absolute value of the profile t.
     Defaults to \code{TRUE}.} 
 \item{\dots}{other arguments to the \code{plot} function can be passed here.}
}
\section{Methods}{\describe{

\item{x = "ANY", y = "ANY"}{Generic function: see \code{\link[graphics]{plot}}.}

\item{x = "profile.mle", y = "missing"}{Plot likelihood profiles for \code{x}.}
}}
\keyword{methods}
\eof
\name{profile-methods}
\docType{methods}
\alias{profile-methods}
\alias{profile,ANY-method}
\alias{profile,mle-method}
\title{Methods for Function profile in Package `stats4'}
\description{Profile likelihood}
\section{Methods}{\describe{

\item{fitted = "ANY"}{Generic function}

\item{fitted = "mle"}{Profile the likelihood in the vicinity of the optimum}
}}
\keyword{methods}
\eof
\name{profile.mle-class}
\docType{class}
\alias{profile.mle-class}
\title{Class "profile.mle"; Profiling information for "mle" object}
\description{Likelihood profiles along each parameter of likelihood function}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("profile.mle",
  ...)}, but most often by invoking \code{profile} on an "mle" object.
}
\section{Slots}{
  \describe{
    \item{\code{profile}:}{Object of class \code{"list"}. List of
      profiles, one for each requested parameter. Each profile is a data
      frame with the first column called \code{z} being the signed square
      root of the -2 log likelihood ratio, and the others being the
      parameters with names prefixed by \code{par.vals.}}
    \item{\code{summary}:}{Object of class \code{"summary.mle"}. Summary
      of object being profiled.}
  }
}
\section{Methods}{
  \describe{
    \item{confint}{\code{signature(object = "profile.mle")}: Use profile
      to generate approximate confidence intervals for parameters.}
    \item{plot}{\code{signature(x = "profile.mle", y = "missing")}: Plot
      profiles for each parameter.}
  }
}
\seealso{
  \code{\link{mle}}, \code{\link{mle-class}}, \code{\link{summary.mle-class}} 
}
%% Do we want examples for classes??
%%\examples{
%%}
\keyword{classes}
\eof
\name{show-methods}
\docType{methods}
\alias{show-methods}
\alias{show,ANY-method}
\alias{show,mle-method}
\alias{show,summary.mle-method}
\title{Methods for Function show in Package `stats4'}
\description{Show objects of classes \code{mle} and \code{summary.mle}}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function}

\item{object = "mle"}{Print simple summary of \code{mle} object. Just
  the coefficients and the call.}

\item{object = "summary.mle"}{Shows call, table of coefficients and
  standard errors, and -2 log L}
}}
\keyword{methods}
\eof
\name{summary-methods}
\docType{methods}
\alias{summary-methods}
\alias{summary,ANY-method}
\alias{summary,mle-method}
\title{Methods for Function summary in Package `stats4'}
\description{Summarize objects}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function}

\item{object = "mle"}{Generate a summary as an object of class
  "summary.mle", containing estimates, asymptotic SE, and value of
  -2 log L}
}}
\keyword{methods}
\eof
\name{summary.mle-class}
\docType{class}
\alias{summary.mle-class}
\title{Class "summary.mle", summary of "mle" objects}
\description{Extract of "mle" object}
\section{Objects from the Class}{
Objects can be created by calls of the form \code{new("summary.mle",
  ...)}, but most often by invoking \code{summary} on an "mle" object.
They contain values meant for printing by \code{show}.
}
\section{Slots}{
  \describe{
    \item{\code{call}:}{Object of class \code{"language"} The call that
      generated the "mle" object.}
    \item{\code{coef}:}{Object of class \code{"matrix"}. Estimated
      coefficients and standard errors }
    \item{\code{m2logL}:}{Object of class \code{"numeric"}. Minus twice
      the log likelihood.}
  }
}
\section{Methods}{
  \describe{
    \item{show}{\code{signature(object = "summary.mle")}: Pretty-prints
      \code{object}  }
    \item{coef}{\code{signature(object = "summary.mle")}: Extracts the
    contents of the \code{coef} slot}
  }
}
\seealso{
  \code{\link{summary}}, \code{\link{mle}}, \code{\link{mle-class}} 
}
\keyword{classes}
\eof
\name{update-methods}
\docType{methods}
\alias{update-methods}
\alias{update,ANY-method}
\alias{update,mle-method}
\title{Methods for Function `update' in Package `stats4'}
\description{
  Update \code{"mle"} objects.
}
\usage{
\S4method{update}{mle}(object, \dots, evaluate = TRUE)
}
\arguments{
 \item{object}{An existing fit.}
 \item{\dots}{Additional arguments to the call, or arguments with
   changed values. Use \code{name=NULL} to remove the argument \code{name}.}
 \item{evaluate}{If true evaluate the new call else return the call.}
}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function: see \code{\link[stats]{update}}.}

\item{object = "mle"}{Update a fit.}
}}
\examples{
x <- 0:10
y <- c(26, 17, 13, 12, 20, 5, 9, 8, 5, 4, 8)
ll <- function(ymax=15, xhalf=6)
    -sum(stats::dpois(y, lambda=ymax/(1+x/xhalf), log=TRUE))
fit <- mle(ll)
## note the recorded call contains ..1, a problem with S4 dispatch
update(fit, fixed=list(xhalf=3))
}
\keyword{methods}
\eof
\name{vcov-methods}
\docType{methods}
\alias{vcov-methods}
\alias{vcov,ANY-method}
\alias{vcov,mle-method}
\title{Methods for Function `vcov' in Package `stats4'}
\description{
  Extract the approximate variance-covariance matrix from \code{"mle"} objects.
}
\section{Methods}{\describe{

\item{object = "ANY"}{Generic function: see \code{\link[stats]{vcov}}.}

\item{object = "mle"}{Extract the estimated variance-covariance matrix
  for the estimated parameters (if any).}
}}
\keyword{methods}
\eof
