\name{Grid}
\alias{Grid}
\title{Grid Graphics}
\description{
  General information about the grid graphics package.
}
\details{
  Grid graphics provides an alternative to the standard R graphics.
  The user is able to define arbitrary rectangular regions (called
  \emph{viewports}) on the graphics device and define a number of
  coordinate systems for each region.  Drawing can be specified to occur
  in any viewport using any of the available coordinate systems.
  
  Grid graphics and standard R graphics do not mix!

  Type \code{library(help = grid)} to see a list of (public)
  Grid graphics functions.
}
\author{Paul Murrell}
\seealso{
  \code{\link{viewport}},
  \code{\link{grid.layout}}, and
  \code{\link{unit}}.
}
\examples{
## Diagram of a simple layout 
grid.show.layout(grid.layout(4,2,
                     heights=unit(rep(1, 4),
                                  c("lines", "lines", "lines", "null")),
                     widths=unit(c(1, 1), "inches")))
## Diagram of a sample viewport
grid.show.viewport(viewport(x=0.6, y=0.6,
                            w=unit(1, "inches"), h=unit(1, "inches")))
## A flash plotting example
grid.multipanel(vp=viewport(0.5, 0.5, 0.8, 0.8))
}
\keyword{dplot}
\eof
\name{absolute.size}
\alias{absolute.size}
\title{ Absolute Size of a Grob }
\description{
  This function converts a unit object into absolute units.
  Absolute units are unaffected, but non-absolute units are
  converted into \code{"null"} units.
}
\usage{
absolute.size(unit)
}
\arguments{
  \item{unit}{ An object of class \code{"unit"}. }
}
\details{
  Absolute units are things like \code{"inches"}, \code{"cm"},
  and \code{"lines"}.  Non-absolute units are \code{"npc"}
  and \code{"native"}.

  This function is designed to be used in \code{widthDetails}
  and \code{heightDetails} methods.
}
\value{
  An object of class \code{"unit"}.
}
\author{ Paul Murrell }
\seealso{
  \code{\link{widthDetails}}
  and \code{\link{heightDetails}} methods.
}
\keyword{ dplot }
\eof
\name{convertNative}
\alias{convertNative}
\title{ Convert a Unit Object to Native units }
\description{
  \bold{This function is deprecated in grid version 0.8 and will be
    made defunct in grid version 1.9}

  You should use the \code{convertUnit()} function or one of its
  close allies instead.
  
  This function returns a numeric vector containing the specified
  x or y locations or dimensions, converted to "user" or "data"
  units, relative to the current viewport.
}
\usage{
convertNative(unit, dimension="x", type="location")
}
\arguments{
  \item{unit}{ A unit object. }
  \item{dimension}{ Either "x" or "y". }
  \item{type}{ Either "location" or "dimension". }
}
\value{
  A numeric vector.
}
\author{ Paul Murrell }
\section{WARNING }{ If you draw objects based on output from these conversion
    functions, then resize your device, the objects will be drawn 
    incorrectly -- the base R display list will not recalculate these
    conversions.  This means that you can only rely on the results of 
    these calculations if the size of your device is fixed.} 
\seealso{ \code{\link{grid.convert}}, \code{\link{unit}} }

\examples{
grid.newpage()
pushViewport(viewport(width=unit(.5, "npc"),
                       height=unit(.5, "npc")))
grid.rect()
w <- convertNative(unit(1, "inches"))
h <- convertNative(unit(1, "inches"), "y")
# This rectangle starts off life as 1in square, but if you
# resize the device it will no longer be 1in square
grid.rect(width=unit(w, "native"), height=unit(h, "native"),
          gp=gpar(col="red"))
popViewport(1)

# How to use grid.convert(), etc instead
convertNative(unit(1, "inches")) ==
  convertX(unit(1, "inches"), "native", valueOnly=TRUE)
convertNative(unit(1, "inches"), "y", "dimension") ==
  convertHeight(unit(1, "inches"), "native", valueOnly=TRUE)
}
\keyword{ dplot }
\eof
\name{Querying the Viewport Tree}
\alias{current.viewport}
\alias{current.vpTree}
\alias{current.transform}
\title{Get the Current Grid Viewport (Tree)}
\description{
  \code{current.viewport()}
  returns the viewport that Grid is going to draw into.

  \code{current.vpTree} returns the entire Grid viewport tree.

  \code{current.transform} returns the transformation
  matrix for the current viewport.
}
\usage{
current.viewport(vp=NULL)
current.vpTree(all=TRUE)
current.transform()
}
\arguments{
  \item{vp}{A Grid viewport object.  Use of this argument has been deprecated.}
  \item{all}{A logical value indicating whether the entire viewport
    tree should be returned.}
}
\details{
  If \code{all} is \code{FALSE} then
  \code{current.vpTree} only returns the subtree below
  the current viewport.
}
\value{
  A Grid viewport object.
}
\author{Paul Murrell}
\seealso{
  \code{\link{viewport}}
}
\examples{
grid.newpage()
pushViewport(viewport(width=0.8, height=0.8, name="A"))
pushViewport(viewport(x=0.1, width=0.3, height=0.6,
  just="left", name="B"))
upViewport(1)
pushViewport(viewport(x=0.5, width=0.4, height=0.8,
  just="left", name="C"))
pushViewport(viewport(width=0.8, height=0.8, name="D"))
upViewport(1)
current.vpTree()
current.viewport()
current.vpTree(all=FALSE)
}
\keyword{dplot}
\eof
\name{dataViewport}
\alias{dataViewport}
\title{ Create a Viewport with Scales based on Data }
\description{
  This is a convenience function for producing a viewport with
  x- and/or y-scales based on numeric values passed to the function.
}
\usage{
dataViewport(xData = NULL, yData = NULL, xscale = NULL, yscale = NULL,
             extension = 0.05, ...)
}
\arguments{
  \item{xData}{ A numeric vector of data. }
  \item{yData}{ A numeric vector of data. }
  \item{xscale}{ A numeric vector (length 2). }
  \item{yscale}{ A numeric vector (length 2). }
  \item{extension}{ A numeric. }
  \item{\dots}{ All other arguments will be passed to a call to
    the \code{viewport()} function. }
}
\details{
  If \code{xscale} is not specified then the values in \code{x} are
  used to generate an x-scale based on the range of \code{x}, extended
  by the proportion specified in \code{extension}.  Similarly for the
  y-scale.
}
\value{
  A grid viewport object.
}
\author{ Paul Murrell }
\seealso{
  \code{\link{viewport}} and
  \code{\link{plotViewport}}.
}
\keyword{dplot}
\eof
\name{drawDetails}
\alias{drawDetails}
\alias{preDrawDetails}
\alias{postDrawDetails}
\alias{draw.details}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Customising grid Drawing }
\description{
  These generic hook functions are called whenever a grid grob is drawn.
  They provide an opportunity for customising the drawing of a
  new class derived from grob (or gTree).
}
\usage{
drawDetails(x, recording)
draw.details(x, recording)
preDrawDetails(x)
postDrawDetails(x)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ A grid grob. }
  \item{recording}{ A logical value indicating whether a grob
  is being added to the display list or redrawn from the display list. }
}
\details{
  These functions are called by the \code{grid.draw} methods for grobs
  and gTrees.
  
  \code{preDrawDetails} is called first during the drawing of a grob.
  This is where any additional viewports should be pushed (see,
  for example, \code{grid:::preDrawDetails.frame}).  Note that
  the default behaviour for grobs is to push any viewports in the
  \code{vp} slot, and for gTrees is to also push and up any viewports in the
  \code{childrenvp} slot so there is typically nothing to do here.
  
  \code{drawDetails} is called next and is where any additional
  calculations and graphical output should occur (see, for example,
  \code{grid:::drawDetails.xaxis}.  Note that the default behaviour
  for gTrees is to draw all grobs in the \code{children} slot
  so there is typically nothing to do here.

  \code{postDrawDetails} is called last and should reverse anything
  done in \code{preDrawDetails} (i.e., pop or up any viewports that were
  pushed;  again, see,
  for example, \code{grid:::postDrawDetails.frame}).  Note that the
  default behaviour for grobs is to pop any viewports that were pushed
  so there is typically nothing to do here.

  Note that \code{preDrawDetails} and \code{postDrawDetails} are
  also called in the calculation of \code{"grobwidth"} and
  \code{"grobheight"} units.
}
\value{
  None of these functions are expected to return a value.
}
\author{ Paul Murrell }
\seealso{ \code{\link{grid.draw}} }
\keyword{ dplot }% at least one, from doc/KEYWORDS
\eof
\name{editDetails}
\alias{editDetails}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Customising grid Editing }
\description{
  This generic hook function is called whenever a grid grob is edited
  via \code{grid.edit} or \code{editGrob}.
  This provides an opportunity for customising the editing of a
  new class derived from grob (or gTree).
}
\usage{
editDetails(x, specs)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ A grid grob. }
  \item{specs}{ A list of named elements.  The names indicate the
    grob slots to modify and the values are the new values for the slots. }
}
\details{
  This function is called by \code{grid.edit} and \code{editGrob}.
  A method should be written for classes derived from grob or gTree
  if a change in a slot has an effect on other slots in the grob or
  children of a gTree (e.g., see \code{grid:::editDetails.xaxis}).

  Note that the slot already has the new value.
}
\value{
  The function MUST return the modified grob.
}
\author{ Paul Murrell }
\seealso{ \code{\link{grid.edit}} }
\keyword{ dplot }% at least one, from doc/KEYWORDS
\eof
\name{gPath}
\alias{gPath}
\title{ Concatenate Grob Names }
\description{
  This function can be used to generate a grob path for use
  in \code{grid.edit} and friends.

  A grob path is a list of nested grob names.
}
\usage{
gPath(...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{\dots}{ Character values which are grob names. }
}
\details{
  Grob names must only be unique amongst grobs which
  share the same parent in a gTree.

  This function can be used to generate a specification for
  a grob that includes the grob's parent's name
  (and the name of its parent and so on).

  For interactive use, it is possible to directly specify
  a path, but it is strongly recommended that this function
  is used otherwise in case the path separator is changed
  in future versions of grid.
}
\value{
  A \code{ gPath } object.
}
\seealso{
  \code{\link{grob}},
  \code{\link{editGrob}},
  \code{\link{addGrob}},
  \code{\link{removeGrob}},
  \code{\link{getGrob}},
  \code{\link{setGrob}}
}
\examples{
gPath("g1", "g2")
}
\keyword{ dplot }% at least one, from doc/KEYWORDS
\eof
\name{gpar}
\alias{gpar}
\alias{get.gpar}
\title{Handling Grid Graphical Parameters}
\description{
  \code{gpar()} should be used to create a set of graphical
  parameter settings.  It returns an object of class \code{"gpar"}.  This is
  basically a list of name-value pairs.

  \code{get.gpar()} can be used to query the current
  graphical parameter settings.
}
\usage{
gpar(...)
get.gpar(names = NULL)
}
\arguments{
  \item{\dots}{ Any number of named arguments. }
  \item{names}{A character vector of valid graphical parameter names.}
}
\details{
  All grid viewports and (predefined) graphical objects have a slot
  called \code{gp}, which contains a \code{"gpar"} object.  When
  a viewport is pushed onto the viewport stack and when a graphical object
  is drawn, the settings in the \code{"gpar"} object are enforced.
  In this way, the graphical output is modified by the \code{gp}
  settings until the graphical object has finished drawing, or until the
  viewport is popped off the viewport stack, or until some other
  viewport or graphical object is pushed or begins drawing.

  Valid parameter names are:
  \tabular{ll}{
    col \tab Colour for lines and borders. \cr
    fill \tab Colour for filling rectangles, polygons, ... \cr
    lty \tab Line type \cr
    lwd \tab Line width \cr
    fontsize \tab The size of text (in points) \cr
    cex \tab Multiplier applied to fontsize \cr
    fontfamily \tab The font family \cr
    fontface \tab The font face (bold, italic, ...) \cr
    lineheight \tab The height of a line as a multiple of the size of text \cr
    font \tab Font face (alias for fontface;  for
    backward compatibility) \cr
  }

  The size of text is \code{fontsize}*\code{cex}.  The size of a line
  is \code{fontsize}*\code{cex}*\code{lineheight}.
  
  For most devices, the \code{fontfamily} is specified when the device
  is first opened and may not be changed thereafter -- i.e., specifying
  a different font family via \code{fontfamily} will be ignored.  This
  will hopefully change in future versions of R.  Also, there is an
  important exception:  \code{fontfamily} may be used to specify one
  of the Hershey Font families (e.g., \code{HersheySerif})
  and this specification will be honoured
  on all devices.

  The specification of \code{fontface} follows the R base graphics
  standard:  1 = plain, 2 = bold, 3 = italic, 4 = bold italic.
  
  Specifying the value \code{NULL} for a parameter is the same as not
  specifying any value for that parameter, except for \code{col} and
  \code{fill}, where \code{NULL} indicates not to draw a border or
  not to fill an area (respectively).

  All parameter values can be vectors of multiple values.  (This will
  not always make sense -- for example, viewports will only take
  notice of the first parameter value.)

  \code{get.gpar()} returns all current graphical parameter settings.
}
\value{
  An object of class \code{"gpar"}.
}
\seealso{
  \code{\link{Hershey}}.
}
\author{Paul Murrell}
\examples{
get.gpar()
gpar(col = "red")
gpar(col = "blue", lty = "solid", lwd = 3, fontsize = 16)
get.gpar(c("col", "lty"))
grid.newpage()
vp <- viewport(w = .8, h = .8, gp = gpar(col="blue"))
grid.draw(gTree(children=gList(rectGrob(gp = gpar(col="red")),
                     textGrob(paste("The rect is its own colour (red)",
                                    "but this text is the colour",
                                    "set by the gTree (green)",
                                    sep = "\n"))),
      gp = gpar(col="green"), vp = vp))
grid.text("This text is the colour set by the viewport (blue)",
          y = 1, just = c("center", "bottom"),
          gp = gpar(fontsize=20), vp = vp)
grid.newpage()
## example with multiple values for a parameter
pushViewport(viewport())
grid.points(1:10/11, 1:10/11, gp = gpar(col=1:10))
popViewport()
}
\keyword{ dplot }
\eof
\name{grid-internal}
%% <FIXME>
%% These had aliases but no corresponding code object:
%% <LIST>
%% \alias{.unit}
%% \alias{.unit.arithmetic}
%% \alias{.unit.list}
%% \alias{clear.display.list}
%% \alias{clearpage}
%% \alias{common.draw.axis}
%% \alias{convert.gpar}
%% \alias{copy}
%% \alias{demo.locator}
%% \alias{editDetails.frame}
%% \alias{grid.height}
%% \alias{grid.init.viewport.stack}
%% \alias{height.default}
%% \alias{height.frame}
%% \alias{height.lines}
%% \alias{height.text}
%% \alias{justifyX}
%% \alias{justifyY}
%% \alias{pop.saved.Rpars}
%% \alias{push.saved.Rpars}
%% \alias{set.viewport}
%% \alias{stack.viewports}
%% \alias{unit.list.c}
%% \alias{unstack.viewports}
%% \alias{viewport.layout<-}
%% \alias{width.default}
%% \alias{width.frame}
%% \alias{width.lines}
%% \alias{width.text}
%% \alias{grid.width}
%% </LIST>
%% </FIXME>

%% These are now registered but unexported S3 methods:
%% <LIST>
%% \alias{[.unit}
%% \alias{[.unit.arithmetic}
%% \alias{[.unit.list}
%% \alias{Ops.unit}
%% \alias{Summary.unit}
%% \alias{as.character.unit}
%% \alias{as.character.unit.arithmetic}
%% \alias{as.character.unit.list}
%% \alias{print.grob}
%% \alias{print.unit}
%% \alias{print.viewport}
%% \alias{draw.details.arrows}
%% \alias{draw.details.circle}
%% \alias{draw.details.collection}
%% \alias{draw.details.default}
%% \alias{draw.details.frame}
%% \alias{draw.details.glist}
%% \alias{draw.details.lines}
%% \alias{draw.details.line.to}
%% \alias{draw.details.move.to}
%% \alias{draw.details.points}
%% \alias{draw.details.polygon}
%% \alias{draw.details.rect}
%% \alias{draw.details.segments}
%% \alias{draw.details.text}
%% \alias{draw.details.viewport}
%% \alias{draw.details.xaxis}
%% \alias{draw.details.yaxis}
%% \alias{editDetails.default}
%% \alias{editDetails.xaxis}
%% \alias{editDetails.yaxis}
%% \alias{height.details.default}
%% \alias{height.details.frame}
%% \alias{height.details.rect}
%% \alias{height.details.text}
%% \alias{height.details.viewport}
%% \alias{height.post.details.default}
%% \alias{height.pre.details.default}
%% \alias{width.details.default}
%% \alias{width.details.frame}
%% \alias{width.details.rect}
%% \alias{width.details.text}
%% \alias{width.details.viewport}
%% \alias{width.post.details.default}
%% \alias{width.pre.details.default}
%% </LIST>

%% These are no longer in the namespace:
%% <LIST>
%% \alias{.GRID.STATE}
%% \alias{.grid.gpar.names}
%% \alias{.grid.loaded}
%% \alias{.grid.unit.list}
%% \alias{absolute}
%% \alias{absolute.units}
%% \alias{absolute.units.unit}
%% \alias{absolute.units.unit.arithmetic}
%% \alias{absolute.units.unit.list}
%% \alias{col.spec}
%% \alias{draw.all}
%% \alias{draw.frame.child}
%% \alias{get.gpar}
%% \alias{get.value}
%% \alias{get.value.default}
%% \alias{get.value.grob}
%% \alias{grid.Call}
%% \alias{grid.Call.graphics}
%% \alias{grid.top.level.vp}
%% \alias{height}
%% \alias{height.post}
%% \alias{height.pre}
%% \alias{inc.display.list}
%% \alias{is.even}
%% \alias{is.gpar}
%% \alias{is.grob}
%% \alias{is.layout}
%% \alias{is.odd}
%% \alias{is.unit.arithmetic}
%% \alias{is.unit.list}
%% \alias{is.viewport}
%% \alias{layout.heights}
%% \alias{layout.ncol}
%% \alias{layout.nrow}
%% \alias{layout.respect}
%% \alias{layout.widths}
%% \alias{make.xaxis.labels}
%% \alias{make.xaxis.major}
%% \alias{make.xaxis.ticks}
%% \alias{make.yaxis.labels}
%% \alias{make.yaxis.major}
%% \alias{make.yaxis.ticks}
%% \alias{mod.dims}
%% \alias{new.col}
%% \alias{new.row}
%% \alias{num.col.specs}
%% \alias{num.row.specs}
%% \alias{origin.bottom}
%% \alias{origin.left}
%% \alias{origin.right}
%% \alias{origin.top}
%% \alias{pop.saved.gpars}
%% \alias{pop.vp}
%% \alias{push.saved.gpars}
%% \alias{push.vp}
%% \alias{record}
%% \alias{record.default}
%% \alias{record.grob}
%% \alias{record.viewport}
%% \alias{recycle.data}
%% \alias{row.spec}
%% \alias{saved.pars}
%% \alias{set.gpar}
%% \alias{set.value.grob}
%% \alias{swap.origin.horizontal}
%% \alias{swap.origin.vertical}
%% \alias{unit.arithmetic}
%% \alias{unit.arithmetic.rep}
%% \alias{unit.length.unit}
%% \alias{unit.length.unit.arithmetic}
%% \alias{unit.length.unit.list}
%% \alias{unit.list}
%% \alias{unit.list.from.list}
%% \alias{unit.list.rep}
%% \alias{unset.gpar}
%% \alias{updateCol}
%% \alias{updateRow}
%% \alias{valid.data}
%% \alias{valid.just}
%% \alias{valid.layout}
%% \alias{valid.origin}
%% \alias{valid.pch}
%% \alias{valid.unit}
%% \alias{valid.units}
%% \alias{valid.viewport}
%% \alias{validGP}
%% \alias{width}
%% \alias{width.post}
%% \alias{width.pre}
%% </LIST>

%% Part of the sensible export list but (currently?) documented as
%% internal (or waiting for documentation to be written):
%% <LIST>
\alias{grid.legend}
\alias{grid.multipanel}
\alias{grid.panel}
\alias{grid.strip}
\alias{is.unit}
\alias{layoutRegion}
\alias{viewport.layout}
\alias{viewport.transform}
%% </LIST>
%% Currently (?) exported as used in the grid.layout.Rd \example ...
%% <LIST>
\alias{layout.torture}
%% </LIST>

\title{Internal Grid Functions}
\description{
  Internal Grid functions
}
\details{
  These are not to be called by the user (or in some cases are just
  waiting for proper documentation to be written :).
}
\keyword{ internal }
\eof
\name{grid.add}
\alias{grid.add}
\alias{addGrob}
\title{Add a Grid Graphical Object}
\description{
  Add a grob to a gTree or a descendant of a gTree.
}
\usage{
grid.add(gPath, child, strict = FALSE, grep=FALSE, global=FALSE,
         allDevices=FALSE, redraw=TRUE) 
addGrob(gTree, child, gPath = NULL, strict = FALSE, grep=FALSE, global=FALSE) 
}
\arguments{
  \item{gTree}{A gTree object.}
  \item{gPath}{ A gPath object. For \code{grid.add} this
    specifyies a gTree on the display list.  For \code{addGrob} this
    specifies a descendant of the specified gTree. }
  \item{child}{ A grob object. }
  \item{strict}{ A boolean indicating whether the gPath must be
    matched exactly. }
  \item{grep}{A boolean indicating whether the \code{gPath} should
    be treated as a regular expression.  Values are recycled across
    elements of the \code{gPath} (e.g., \code{c(TRUE, FALSE)} means
    that every odd element of the \code{gPath} will be treated as
    a regular expression).
    NOT YET IMPLEMENTED.
  }
  \item{global}{ A boolean indicating whether the function should affect
    just the first match of the \code{gPath}, or whether all matches
    should be affected.
    NOT YET IMPLEMENTED.
  }
  \item{allDevices}{ A boolean indicating whether all open devices
    should
    be searched for matches, or just the current device.
    NOT YET IMPLEMENTED.
  }
  \item{redraw}{A logical value to indicate whether to redraw the grob. }
}
\details{
  \code{addGrob} copies the specified grob and returns a modified
  grob.
  
  \code{grid.add} destructively modifies a grob on the display list.
  If \code{redraw}
  is \code{TRUE} it then redraws everything to reflect the change.

}
\value{
  \code{addGrob} returns a grob object;  \code{grid.add} returns \code{NULL}.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grob}}, \code{\link{getGrob}},
  \code{\link{addGrob}}, \code{\link{removeGrob}}.
}
\examples{
}
\keyword{dplot}
\eof
\name{grid.arrows}
\alias{grid.arrows}
\alias{arrowsGrob}
\title{ Draw Arrows }
\description{
  Functions to create and draw arrows at either end of a line, or at
  either end of a 
  line.to, lines, or segments grob.
}
\usage{
grid.arrows(x = c(0.25, 0.75), y = 0.5, default.units = "npc",
            grob = NULL,
            angle = 30, length = unit(0.25, "inches"),
            ends = "last", type = "open", name = NULL,
            gp = gpar(), draw = TRUE, vp = NULL)
arrowsGrob(x = c(0.25, 0.75), y = 0.5, default.units = "npc",
            grob = NULL,
            angle = 30, length = unit(0.25, "inches"),
            ends = "last", type = "open", name = NULL,
            gp = gpar(), vp = NULL)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-values.}
  \item{y}{A numeric vector or unit object specifying y-values.}
  \item{default.units}{A string indicating the default units to use
    if \code{x} or \code{y} are only given as numeric vectors.}  
  \item{grob}{ A grob to add arrows to;  currently can only be
    a line.to, lines, or segments grob. }
  \item{angle}{ A numeric specifying (half) the width of the arrow head
    (in degrees). }
  \item{length}{ A unit object specifying the length of the arrow head. }
  \item{ends}{ One of \code{"first"}, \code{"last"}, or \code{"both"},
    indicating which end of the line to add arrow heads. }
  \item{type}{ Either \code{"open"} or \code{"closed"} to indicate
    the type of arrow head.   }
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create an arrows grob (a graphical object describing 
  arrows), but only \code{grid.arrows()}
  draws the arrows (and then only if \code{draw} is \code{TRUE}).
  
  If the grob argument is specified, this overrides any x
  and/or y arguments.
}
\value{
  An arrows grob.  \code{grid.arrows()} returns the value invisibly.
}
\author{ Paul Murrell }
\seealso{
  \link{Grid},
  \code{\link{viewport}},
  \code{\link{grid.line.to}},
  \code{\link{grid.lines}},
  \code{\link{grid.segments}}
}
\examples{
pushViewport(viewport(layout=grid.layout(2, 4)))
pushViewport(viewport(layout.pos.col=1,
                       layout.pos.row=1))
grid.rect(gp=gpar(col="grey"))
grid.arrows()
popViewport()
pushViewport(viewport(layout.pos.col=2,
                       layout.pos.row=1))
grid.rect(gp=gpar(col="grey"))
grid.arrows(angle=15, type="closed")
popViewport()
pushViewport(viewport(layout.pos.col=3,
                       layout.pos.row=1))
grid.rect(gp=gpar(col="grey"))
grid.arrows(angle=5, length=unit(0.1, "npc"),
            type="closed", gp=gpar(fill="white"))
popViewport()
pushViewport(viewport(layout.pos.col=4,
                       layout.pos.row=1))
grid.rect(gp=gpar(col="grey"))
grid.arrows(x=unit(0:80/100, "npc"),
            y=unit(1 - (0:80/100)^2, "npc"))
popViewport()
pushViewport(viewport(layout.pos.col=1,
                       layout.pos.row=2))
grid.rect(gp=gpar(col="grey"))
grid.arrows(ends="both")
popViewport()
pushViewport(viewport(layout.pos.col=2,
                       layout.pos.row=2))
grid.rect(gp=gpar(col="grey"))
# Recycling arguments
grid.arrows(x=unit(1:10/11, "npc"), y=unit(1:3/4, "npc"))
popViewport()
pushViewport(viewport(layout.pos.col=3,
                       layout.pos.row=2))
grid.rect(gp=gpar(col="grey"))
# Drawing arrows on a segments grob
gs <- segmentsGrob(x0=unit(1:4/5, "npc"),
                   x1=unit(1:4/5, "npc"))
grid.arrows(grob=gs, length=unit(0.1, "npc"),
            type="closed", gp=gpar(fill="white"))
popViewport()
pushViewport(viewport(layout.pos.col=4,
                       layout.pos.row=2))
grid.rect(gp=gpar(col="grey"))
# Arrows on a lines grob
# Name these because going to grid.edit them later
gl <- linesGrob(name="curve", x=unit(0:80/100, "npc"),
                y=unit((0:80/100)^2, "npc"))
grid.arrows(name="arrowOnLine", grob=gl, angle=15, type="closed",
            gp=gpar(fill="black"))
popViewport()
pushViewport(viewport(layout.pos.col=1,
                       layout.pos.row=2))
grid.move.to(x=0.5, y=0.8)
popViewport()
pushViewport(viewport(layout.pos.col=4,
                       layout.pos.row=1))
# Arrows on a line.to grob
glt <- lineToGrob(x=0.5, y=0.2, gp=gpar(lwd=3))
grid.arrows(grob=glt, ends="first", gp=gpar(lwd=3))
popViewport(2)
grid.edit(gPath("arrowOnLine", "curve"), y=unit((0:80/100)^3, "npc"))
}
\keyword{dplot}
\eof
\name{grid.circle}
\alias{grid.circle}
\alias{circleGrob}
\title{ Draw a Circle }
\description{
  Functions to create and draw a circle.  
}
\usage{
grid.circle(x=0.5, y=0.5, r=0.5, default.units="npc", name=NULL,
            gp=gpar(), draw=TRUE, vp=NULL)
circleGrob(x=0.5, y=0.5, r=0.5, default.units="npc", name=NULL,
            gp=gpar(), vp=NULL)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-locations.}
  \item{y}{A numeric vector or unit object specifying y-locations.}
  \item{r}{A numeric vector or unit object specifying radii.}
  \item{default.units}{A string indicating the default units to use
    if \code{x}, \code{y}, \code{width}, or \code{height}
    are only given as numeric vectors.}  
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create a circle grob (a graphical object describing a
  circle), but only \code{grid.circle()}
  draws the circle (and then only if \code{draw} is \code{TRUE}).
  
  The radius may be given in any
  units;  if the units are \emph{relative} (e.g., \code{"npc"} or
  \code{"native"}) then the radius will be different depending on
  whether it is interpreted as a width or as a height.  In such cases,
  the smaller of these two values will be the result.  To see the
  effect, type \code{grid.circle()} and adjust the size of the window.
}
\value{
  A circle grob.  \code{grid.circle()} returns the value invisibly.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}
\keyword{dplot}
\eof
\name{grid.collection}
\alias{grid.collection}
\title{Create a Coherent Group of Grid Graphical Objects}
\description{
  This function is deprecated; please use \code{gTree}.
  
  This function creates a graphical object which contains
  several other graphical objects.  When it is drawn, it draws
  all of its children.  

  It may be convenient to name the elements of the collection.
}
\usage{
grid.collection(..., gp=gpar(), draw=TRUE, vp=NULL)
}
\arguments{
  \item{\dots}{Zero or more objects of class \code{"grob"}.}
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value to indicate whether to produce
    graphical output.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\value{
  A collection grob.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grid.grob}}.
}
\examples{
}
\keyword{dplot}

\eof
\name{grid.convert}
\alias{grid.convert}
\alias{grid.convertX}
\alias{grid.convertY}
\alias{grid.convertWidth}
\alias{grid.convertHeight}
\alias{convertUnit}
\alias{convertX}
\alias{convertY}
\alias{convertWidth}
\alias{convertHeight}
\title{ Convert Between Different grid Coordinate Systems }
\description{
  These functions take a unit object and convert it to an
  equivalent
  unit object in a different coordinate system.
}
\usage{
convertX(x, unitTo, valueOnly = FALSE) 
convertY(x, unitTo, valueOnly = FALSE) 
convertWidth(x, unitTo, valueOnly = FALSE) 
convertHeight(x, unitTo, valueOnly = FALSE) 
convertUnit(x, unitTo,
             axisFrom = "x", typeFrom = "location",
             axisTo = axisFrom, typeTo = typeFrom,
             valueOnly = FALSE)
grid.convertX(x, unitTo, valueOnly = FALSE) 
grid.convertY(x, unitTo, valueOnly = FALSE) 
grid.convertWidth(x, unitTo, valueOnly = FALSE) 
grid.convertHeight(x, unitTo, valueOnly = FALSE) 
grid.convert(x, unitTo,
             axisFrom = "x", typeFrom = "location",
             axisTo = axisFrom, typeTo = typeFrom,
             valueOnly = FALSE)
}
\arguments{
  \item{x}{ A unit object. }
  \item{unitTo}{ The coordinate system to convert the unit to.
    See the \code{\link{unit}} function for valid coordinate systems.}
  \item{axisFrom}{ Either \code{"x"} or \code{"y"} to indicate
    whether the unit object represents a value in the x- or
    y-direction. }
  \item{typeFrom}{ Either \code{"location"} or \code{"dimension"}
    to indicate whether the unit object represents a location or a
    length. }
  \item{axisTo}{ Same as \code{axisFrom}, but applies to the unit object
    that is to be created. }
  \item{typeTo}{ Same as \code{typeFrom}, but applies to the unit object
    that is to be created. }
  \item{valueOnly}{ A logical indicating.  If \code{TRUE} then
    the function does not return a unit object, but rather only the
    converted numeric values. }
}
\details{
  The \code{convertUnit} function allows for general-purpose
  conversions.  The other four functions are just more convenient
  front-ends to it for the most common conversions.
  
  The conversions occur within the current viewport.
  
  It is not currently possible to convert to all valid coordinate systems
  (e.g., "strwidth" or "grobwidth").  I'm not sure if all of these
  are impossible, they just seem implausible at this stage.

  In normal usage of grid, these functions should not be necessary.
  If you want to express a location or dimension in inches rather
  than user coordinates then you should simply do something like
  \code{unit(1, "inches")} rather than something like
  \code{unit(0.134, "native")}.
  
  In some cases, however, it is necessary for the user to
  perform calculations on a unit value and this function becomes
  necessary.  In such cases, please take note of the warning below.

  The grid.* versions are just previous incarnations which have
  been deprecated.
}
\value{
  A unit object in the specified coordinate system
  (unless \code{valueOnly} is \code{TRUE} in which case
  the returned value is a numeric).
}
\author{ Paul Murrell }

\section{Warning}{
  The conversion is only valid for the current device size.
  If the device is resized then at least some conversions will
  become invalid.  For example, suppose that I create a unit
  object as follows: \code{oneinch <- convertUnit(unit(1, "inches"),
    "native"}.  Now if I resize the device, the unit object in
  oneinch no longer corresponds to a physical length of 1 inch.
  }
\seealso{ \code{\link{unit}} }
\examples{
## A tautology
convertX(unit(1, "inches"), "inches")
## The physical units
convertX(unit(2.54, "cm"), "inches")
convertX(unit(25.4, "mm"), "inches")
convertX(unit(72.27, "points"), "inches")
convertX(unit(1/12*72.27, "picas"), "inches")
convertX(unit(72, "bigpts"), "inches")
convertX(unit(1157/1238*72.27, "dida"), "inches")
convertX(unit(1/12*1157/1238*72.27, "cicero"), "inches")
convertX(unit(65536*72.27, "scaledpts"), "inches")
convertX(unit(1/2.54, "inches"), "cm")
convertX(unit(1/25.4, "inches"), "mm")
convertX(unit(1/72.27, "inches"), "points")
convertX(unit(1/(1/12*72.27), "inches"), "picas")
convertX(unit(1/72, "inches"), "bigpts")
convertX(unit(1/(1157/1238*72.27), "inches"), "dida")
convertX(unit(1/(1/12*1157/1238*72.27), "inches"), "cicero")
convertX(unit(1/(65536*72.27), "inches"), "scaledpts")

pushViewport(viewport(width=unit(1, "inches"),
                       height=unit(2, "inches"),
                       xscale=c(0, 1),
                       yscale=c(1, 3)))
  ## Location versus dimension
  convertY(unit(2, "native"), "inches")
  convertHeight(unit(2, "native"), "inches")
  ## From "x" to "y" (the conversion is via "inches")
  convertUnit(unit(1, "native"), "native",
               axisFrom="x", axisTo="y")
  ## Convert several values at once
  convertX(unit(c(0.5, 2.54), c("npc", "cm")),
                c("inches", "native"))
popViewport()
## Convert a complex unit
convertX(unit(1, "strwidth", "Hello"), "native")
}
\keyword{dplot}
\eof
\name{grid.copy}
\alias{grid.copy}
\title{Make a Copy of a Grid Graphical Object}
\description{
  This function is redundant and will disappear in future versions.
}
\usage{
grid.copy(grob)
}
\arguments{
  \item{grob}{A grob object.}
}
\value{
  A copy of the grob object.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grid.grob}}.
}
\examples{
}
\keyword{dplot}
\eof
\name{grid.display.list}
\alias{grid.display.list}
\alias{engine.display.list}
\title{Control the Grid Display List}
\description{
  Turn the Grid display list on or off.
}
\usage{
grid.display.list(on=TRUE)
engine.display.list(on=TRUE)
}
\arguments{
  \item{on}{A logical value to indicate whether the display list
    should be on or off.}
}
\details{
  All drawing and viewport-setting operations are (by default)
  recorded in the Grid display list.  This allows redrawing
  to occur following an editing operation.

  This display list could get very large so it may be useful to
  turn it off in some cases;  this will of course disable redrawing.

  All graphics output is also recorded on the main display list
  of the R graphics engine (by default).  This supports redrawing
  following a device resize and allows copying between devices.

  Turning off this display list means that grid will redraw from its
  own display list for device resizes and copies.  This will be slower
  than using the graphics engine display list.
}
\value{
  None.
}
\author{Paul Murrell}

\section{WARNING}{
  Turning the display list on causes the display list to be erased!

  Turning off both the grid display list and the graphics engine
  display list will result in no redrawing whatsoever.
}
\keyword{dplot}
\eof
\name{grid.draw}
\alias{grid.draw}
\title{Draw a grid grob}
\description{
  Produces graphical output from a graphical object.
}
\usage{
grid.draw(x, recording=TRUE)
}
\arguments{
  \item{x}{An object of class \code{"grob"} or NULL.}
  \item{recording}{A logical value to indicate whether the drawing
    operation should be recorded on the Grid display list.}
}
\details{
  This is a generic function with methods for grob and gTree objects.

  The grob and gTree methods automatically push any viewports in a
  \code{vp} slot and automatically apply any \code{gpar} settings
  in a \code{gp} slot.  In addition, the gTree method pushes and
  ups any viewports in a \code{childrenvp} slot and automatically
  calls \code{grid.draw} for any grobs in a \code{children} slot.

  The methods for grob and gTree call the generic hook functions
  \code{preDrawDetails}, \code{drawDetails}, and \code{postDrawDetails}
  to allow classes derived from grob or gTree to perform
  additional viewport pushing/popping and produce additional
  output beyond the default behaviour for grobs and gTrees.
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grob}}.
}

\examples{
grid.newpage()
## Create a graphical object, but don't draw it
l <- linesGrob()
## Draw it
grid.draw(l)
}
\keyword{dplot}
\eof
\name{grid.edit}
\alias{grid.edit}
\alias{editGrob}
\title{Edit the Description of a Grid Graphical Object}
\description{
  Changes the value of one of the slots of a grob
 and redraws the grob.
}
\usage{
grid.edit(gPath, ..., strict=FALSE, grep=FALSE, global=FALSE,
          allDevices=FALSE, redraw=TRUE) 
editGrob(grob, gPath=NULL, ..., strict=FALSE, grep=FALSE, global=FALSE) 
}
\arguments{
  \item{grob}{A grob object.}
  \item{\dots}{Zero or more named arguments specifying new slot values.}
  \item{gPath}{ A gPath object. For \code{grid.edit} this
    specifyies a grob on the display list.  For \code{editGrob} this
    specifies a descendant of the specified grob. }
  \item{strict}{ A boolean indicating whether the gPath must be
    matched exactly. }
  \item{grep}{A boolean indicating whether the \code{gPath} should
    be treated as a regular expression.  Values are recycled across
    elements of the \code{gPath} (e.g., \code{c(TRUE, FALSE)} means
    that every odd element of the \code{gPath} will be treated as
    a regular expression).
    NOT YET IMPLEMENTED.
  }
  \item{global}{ A boolean indicating whether the function should affect
    just the first match of the \code{gPath}, or whether all matches
    should be affected.
    NOT YET IMPLEMENTED.
  }
  \item{allDevices}{ A boolean indicating whether all open devices
    should
    be searched for matches, or just the current device.
    NOT YET IMPLEMENTED.
  }
  \item{redraw}{A logical value to indicate whether to redraw the grob. }
}
\details{
  \code{editGrob} copies the specified grob and returns a modified
  grob.
  
  \code{grid.edit} destructively modifies a grob on the display list.
  If \code{redraw}
  is \code{TRUE} it then redraws everything to reflect the change.

  Both functions call \code{editDetails} to allow a grob to perform
  custom actions and \code{validDetails} to check that the modified grob
  is still coherent.
  
}
\value{
  \code{editGrob} returns a grob object;  \code{grid.edit} returns \code{NULL}.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grob}}, \code{\link{getGrob}},
  \code{\link{addGrob}}, \code{\link{removeGrob}}.
}
\examples{
grid.newpage()
grid.xaxis(name="xa", vp=viewport(width=.5, height=.5))
grid.edit("xa", gp=gpar(col="red"))
# won't work because no ticks (at is NULL)
try(grid.edit(gPath("xa", "ticks"), gp=gpar(col="green")))
grid.edit("xa", at=1:4/5)
# Now it should work
try(grid.edit(gPath("xa", "ticks"), gp=gpar(col="green")))
}
\keyword{dplot}
\eof
\name{grid.frame}
\alias{grid.frame}
\alias{frameGrob}
\title{ Create a Frame for Packing Objects }
\description{
  These functions, together with \code{grid.pack},
  \code{grid.place}, \code{packGrob}, and \code{placeGrob} are part of a
  GUI-builder-like interface to constructing graphical images.
  The idea is that you create a frame with this function then
  use \code{grid.pack} or whatever to pack/place objects into the frame.
}
\usage{
grid.frame(layout=NULL, name=NULL, gp=gpar(), vp=NULL, draw=TRUE)
frameGrob(layout=NULL, name=NULL, gp=gpar(), vp=NULL)
}
\arguments{
  \item{layout}{ A Grid layout, or NULL. This can be used to initialise
    the frame with a number of rows and columns, with initial widths
    and heights, etc. }
  \item{name}{ A character identifier. }
  \item{vp}{ An object of class \code{viewport}, or NULL. }
  \item{gp}{ An object of class \code{gpar};  typically the output from
    a call to the function \code{gpar}.}
\item{draw}{ Should the frame be drawn.  }
}
\details{
  Both functions create a frame grob (a graphical object describing a
  frame), but only \code{grid.frame()}
  draws the frame (and then only if \code{draw} is \code{TRUE}).
  Nothing will actually be
  drawn, but it will put the frame on the display list, which means
  that the output will be dynamically updated as objects are packed
  into the frame.  Possibly useful for debugging.
}
\value{
  A frame grob.  \code{grid.frame()} returns the value invisibly.
}
\author{ Paul Murrell }
\seealso{
  \code{\link{grid.pack}}
}
\examples{
grid.newpage()
grid.frame(name="gf", draw=TRUE)
grid.pack("gf", rectGrob(gp=gpar(fill="grey")))
grid.pack("gf", textGrob("hi there"), side="right")
}
\keyword{dplot}
\eof
\name{grid.get}
\alias{grid.get}
\alias{getGrob}
\title{Get a Grid Graphical Object}
\description{
  Retrieve a grob or a descendant of a grob.
}
\usage{
grid.get(gPath, strict=FALSE, grep=FALSE, global=FALSE, allDevices=FALSE) 
getGrob(gTree, gPath, strict=FALSE, grep=FALSE, global=FALSE) 
}
\arguments{
  \item{gTree}{A gTree object.}
  \item{gPath}{ A gPath object. For \code{grid.get} this
    specifyies a grob on the display list.  For \code{getGrob} this
    specifies a descendant of the specified gTree. }
  \item{strict}{ A boolean indicating whether the gPath must be
    matched exactly. }
  \item{grep}{A boolean indicating whether the \code{gPath} should
    be treated as a regular expression.  Values are recycled across
    elements of the \code{gPath} (e.g., \code{c(TRUE, FALSE)} means
    that every odd element of the \code{gPath} will be treated as
    a regular expression).
    NOT YET IMPLEMENTED.
  }
  \item{global}{ A boolean indicating whether the function should affect
    just the first match of the \code{gPath}, or whether all matches
    should be affected.
    NOT YET IMPLEMENTED.
  }
  \item{allDevices}{ A boolean indicating whether all open devices
    should
    be searched for matches, or just the current device.
    NOT YET IMPLEMENTED.
  }
}
\value{
  A grob object.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grob}}, \code{\link{getGrob}},
  \code{\link{addGrob}}, \code{\link{removeGrob}}.
}
\examples{
grid.xaxis(name="xa")
grid.get("xa")
grid.get(gPath("xa", "ticks"))

grid.draw(gTree(name="gt", children=gList(xaxisGrob(name="axis"))))
grid.get(gPath("gt", "axis", "ticks"))
}
\keyword{dplot}
\eof
\name{grid.grill}
\alias{grid.grill}
\title{Draw a Grill}
\description{
  This function draws a grill within a Grid viewport.
}
\usage{
grid.grill(h = unit(seq(0.25, 0.75, 0.25), "npc"),
           v = unit(seq(0.25, 0.75, 0.25), "npc"),
           default.units = "npc", gp=gpar(col = "grey"), vp = NULL)
}
\arguments{
  \item{h}{A numeric vector or unit object
    indicating the horizontal location of the
    vertical grill lines.}
  \item{v}{A numeric vector or unit object
    indicating the vertical location of the
    horizontal grill lines.}
  \item{default.units}{A string indicating the default units to use
    if \code{h} or \code{v} are only given as numeric vectors.}
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{vp}{A Grid viewport object.}
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}.
}
\keyword{dplot}
\eof
\name{grid.grob}
\alias{grid.grob}
\alias{grob}
\alias{gTree}
\alias{gList}
\title{Create a Grid Graphical Object}
\description{
  These functions create grid graphical objects.
}
\usage{
grid.grob(list.struct, cl = NULL, draw = TRUE)
grob(..., name = NULL, gp = NULL, vp = NULL, cl = NULL) 
gTree(..., name = NULL, gp = NULL, vp = NULL, children = NULL, 
      childrenvp = NULL, cl = NULL)
gList(...)
}
\arguments{
  \item{...}{ For \code{grob} and \code{gTree}, the
    named slots describing important features of
    the graphical object. For \code{gList}, a series of grob objects. }
  \item{list.struct}{A list (preferably with each element named).}
  \item{name}{ A character identifier for the grob.  Used to find the
    grob on the display list and/or as a child of another grob. }
  \item{children}{ A gList object. }
  \item{childrenvp}{ A viewport object (or \code{NULL}).}
  \item{gp}{A gpar object, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{vp}{A viewport object (or \code{NULL}).}
  \item{cl}{A string giving the class attribute for the \code{list.struct}}
  \item{draw}{A logical value to indicate whether to produce
    graphical output.}
}
\details{
  These functions can be used to create a basic grob, gTree, or gList
  object, or a new class derived from one of these.
  
  A grid graphical object (grob) is a description of a graphical
  item.  These basic classes provide default behaviour for
  validating, drawing,
  and modifying graphical objects.  Both call the function
  \code{validDetails} to check that the object returned is coherent.

  A gTree can have other grobs as children;  when a gTree is drawn, it
  draws all of its children.  Before drawing its children, a gTree
  pushes its childrenvp slot and then navigates back up (calls
  \code{upViewport}) so that the children can specify their location
  within the childrenvp via a vpPath.
  
  All grid primitives (\code{grid.lines}, \code{grid.rect}, ...) and
  some higher-level grid components (e.g., \code{grid.xaxis} and
  \code{grid.yaxis}) are derived from these classes.

  \code{grid.grob} is deprecated.

}
\value{
  A grob object.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grid.draw}},
  \code{\link{grid.edit}},
  \code{\link{grid.get}}.
}

\examples{
}
\keyword{dplot}
\eof
\name{grid.layout}
\alias{grid.layout}
\title{Create a Grid Layout}
\description{
  This function returns a Grid layout, which describes a subdivision
  of a rectangular region.
}
\usage{
grid.layout(nrow = 1, ncol = 1,
        widths = unit(rep(1, ncol), "null"),
        heights = unit(rep(1, nrow), "null"),
        default.units = "null", respect = FALSE,
        just="centre")
}
\arguments{
  \item{nrow}{An integer describing the number of rows in the layout.}
  \item{ncol}{An integer describing the number of columns in the layout.}
  \item{widths}{A numeric vector or unit object
    describing the widths of the columns
    in the layout.}
  \item{heights}{A numeric vector or unit object
    describing the heights of the rows
    in the layout.}
  \item{default.units}{A string indicating the default units to use
    if \code{widths} or \code{heights} are only given as numeric vectors.}  
  \item{respect}{A logical value indicating whether
    row heights and column widths should respect each other.}
  \item{just}{A string vector indicating how the layout should be
    justified if it is not the same size as its parent viewport.
    If there are two values, the first
    value specifes horizontal justification and the second value specifies
    vertical justification.  Possible values are: \code{"left"},
    \code{"right"}, \code{"centre"}, \code{"center"}, \code{"bottom"},
    and \code{"top"}.  NOTE that in this context,
    \code{"left"}, for example, means align the left
    edge of the left-most layout column with the left edge of the
    parent viewport.}
}
\details{
  The unit objects given for the \code{widths} and \code{heights}
  of a layout may use a special \code{units} that only has
  meaning for layouts.  This is the \code{"null"} unit, which
  indicates what relative fraction of the available width/height the
  column/row occupies.  See the reference for a better description
  of relative widths and heights in layouts.
}
\section{WARNING}{
  This function must NOT be confused with the base R graphics function
  \code{layout}.  In particular, do not use \code{layout} in
  combination with Grid graphics.  The documentation for
  \code{layout} may provide some useful information and this
  function should behave identically in comparable situations.  The
  \code{grid.layout} 
  function has \emph{added} the ability to specify a broader range
  of units for row heights and column widths, and allows for nested
  layouts (see \code{viewport}).
}
\value{
  A Grid layout object.
}
\references{Murrell, P. R. (1999), Layouts: A Mechanism for Arranging
  Plots on a Page, \emph{Journal of Computational and Graphical
  Statistics}, \bold{8}, 121--134.}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{grid.show.layout}},
  \code{\link{viewport}},
  \code{\link{layout}}}

\examples{
## A variety of layouts (some a bit mid-bending ...)
layout.torture()
## Demonstration of layout justification
grid.newpage()
testlay <- function(just="centre") {
  pushViewport(viewport(layout=grid.layout(1, 1, widths=unit(1, "inches"),
                          height=unit(0.25, "npc"),
                          just=just)))
  pushViewport(viewport(layout.pos.col=1, layout.pos.row=1))
  grid.rect()
  grid.text(paste(just, collapse="-"))
  popViewport(2)
}
testlay()
testlay(c("left", "top"))
testlay(c("right", "top"))
testlay(c("right", "bottom"))
testlay(c("left", "bottom"))
testlay(c("left"))
testlay(c("right"))
testlay(c("bottom"))
testlay(c("top"))
}
\keyword{dplot}
\eof
\name{grid.lines}
\alias{grid.lines}
\alias{linesGrob}
\title{Draw Lines in a Grid Viewport}
\description{
  These functions create and draw a series of lines.
}
\usage{
grid.lines(x = unit(c(0, 1), "npc", units.per.obs),
       y = unit(c(0, 1), "npc", units.per.obs), 
       default.units = "npc", units.per.obs = FALSE, name = NULL,
       gp=gpar(), draw = TRUE, vp = NULL)
linesGrob(x = unit(c(0, 1), "npc", units.per.obs),
       y = unit(c(0, 1), "npc", units.per.obs), 
       default.units = "npc", units.per.obs = FALSE, name = NULL,
       gp=gpar(), vp = NULL)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-values.}
  \item{y}{A numeric vector or unit object specifying y-values.}
  \item{default.units}{A string indicating the default units to use
    if \code{x} or \code{y} are only given as numeric vectors.}  
  \item{units.per.obs}{A logical value to indicate whether each
    individual (x, y) location has its own unit(s) specified.}
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create a lines grob (a graphical object describing
  lines), but only \code{grid.lines}
  draws the lines (and then only if \code{draw} is \code{TRUE}).
  
}
\value{
  A lines grob.  \code{grid.lines} returns the value invisibly.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}
\keyword{dplot}
\eof
\name{grid.locator}
\alias{grid.locator}
\title{ Capture a Mouse Click }
\description{
  Allows the user to click the mouse once within the current
  graphics
  device and returns the location of the mouse click within the
  current viewport, in the specified coordinate system.
}
\usage{
grid.locator(unit = "native")
}
\arguments{
  \item{unit}{ The coordinate system in which to return the location
    of the mouse click.
        See the \code{\link{unit}} function for valid coordinate systems.}
}
\details{
  This function is modal (like the base function \code{locator}) so
  the command line and graphics drawing is blocked until
  the use has clicked the mouse  in the current device.
}
\value{
  A unit object representing the location of the mouse click within the
  current viewport, in the specified coordinate system.
}
\author{ Paul Murrell }
\seealso{ \code{\link{viewport}}, \code{\link{unit}},
  \code{\link{locator}} }
\examples{
if (interactive()) {
  ## Need to write a more sophisticated unit as.character method
  unittrim <- function(unit) {
    sub("^([0-9]+|[0-9]+[.][0-9])[0-9]*", "\\\\1", as.character(unit))
  }
  do.click <- function(unit) {
    click.locn <- grid.locator(unit)
    grid.segments(unit.c(click.locn$x, unit(0, "npc")),
                  unit.c(unit(0, "npc"), click.locn$y),
                  click.locn$x, click.locn$y,
                  gp=gpar(lty="dashed", col="grey"))
    grid.points(click.locn$x, click.locn$y, pch=16, size=unit(1, "mm"))
    clickx <- unittrim(click.locn$x)
    clicky <- unittrim(click.locn$y)
    grid.text(paste("(", clickx, ", ", clicky, ")", sep=""),
              click.locn$x + unit(2, "mm"), click.locn$y,
              just="left")
  }
  do.click("inches")
  pushViewport(viewport(width=0.5, height=0.5,
                         xscale=c(0, 100), yscale=c(0, 10)))
  grid.rect()
  grid.xaxis()
  grid.yaxis()
  do.click("native")
  popViewport()  
}
}
\keyword{dplot}
\eof
\name{grid.move.to}
\alias{grid.move.to}
\alias{grid.line.to}
\alias{moveToGrob}
\alias{lineToGrob}
\title{ Move or Draw to a Specified Position }
\description{
  Grid has the notion of a current location.  These functions
  sets that location.
}
\usage{
grid.move.to(x=0, y=0, default.units="npc", name=NULL, draw=TRUE, vp=NULL)
moveToGrob(x=0, y=0, default.units="npc", name=NULL, vp=NULL)
grid.line.to(x=1, y=1, default.units="npc", name=NULL, gp=gpar(),
             draw=TRUE, vp=NULL)
lineToGrob(x=1, y=1, default.units="npc", name=NULL, gp=gpar(), vp=NULL)
}
\arguments{
  \item{x}{A numeric value or a unit object specifying an x-value.}
  \item{y}{A numeric value or a unit object specifying a y-value.}
  \item{default.units}{A string indicating the default units to use
    if \code{x} or \code{y} are only given as numeric values.}  
  \item{name}{ A character identifier. }
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create a move.to/line.to grob (a graphical object describing a
  move-to/line-to), but only \code{grid.move.to/line.to()}
  draws the move.to/line.to (and then only if \code{draw} is \code{TRUE}).  
}
\value{
  A move.to/line.to grob.
  \code{grid.move.to/line.to()} returns the value invisibly.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}

\examples{
grid.newpage()
grid.move.to(0.5, 0.5)
grid.line.to(1, 1)
grid.line.to(0.5, 0)
pushViewport(viewport(x=0, y=0, w=0.25, h=0.25, just=c("left", "bottom")))
grid.rect()
grid.grill()
grid.line.to(0.5, 0.5)
popViewport()
}
\keyword{dplot}
\eof
\name{grid.newpage}
\alias{grid.newpage}
\title{Move to a New Page on a Grid Device}
\description{
  This function erases the current device or moves to a new page.
}
\usage{
grid.newpage(recording = TRUE)
}
\arguments{
  \item{recording}{A logical value to indicate whether the new-page
    operation should be saved onto the Grid display list.}
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \link{Grid}
}
\keyword{dplot}
\eof
\name{grid.pack}
\alias{grid.pack}
\alias{packGrob}
\title{ Pack an Object within a Frame }
\description{
  This functions, together with \code{grid.frame} and \code{frameGrob}
  are part of a
  GUI-builder-like interface to constructing graphical images.
  The idea is that you create a frame with \code{grid.frame} or
  \code{frameGrob} then
  use this functions to pack objects into the frame.
}
\usage{
grid.pack(gPath, grob, redraw=TRUE, side=NULL,
          row=NULL, row.before=NULL, row.after=NULL,
          col=NULL, col.before=NULL, col.after=NULL,
          width=NULL, height=NULL,
          force.width=FALSE, force.height=FALSE, border=NULL,
          dynamic=FALSE)
packGrob(frame, grob, side=NULL,
          row=NULL, row.before=NULL, row.after=NULL,
          col=NULL, col.before=NULL, col.after=NULL,
          width=NULL, height=NULL,
          force.width=FALSE, force.height=FALSE, border=NULL,
          dynamic=FALSE)
}
\arguments{
  \item{gPath}{ A gPath object, which specifies a frame on the
    display list.} 
  \item{frame}{ An object of class \code{frame}, typically the output
  from a call to \code{grid.frame}. }
  \item{grob}{ An object of class \code{grob}.  The object to be
    packed. }
  \item{redraw}{ A boolean indicating whether the output should be updated. }
  \item{side}{ One of \code{"left"}, \code{"top"}, \code{"right"},
    \code{"bottom"} to indicate which side to pack the object on. }
  \item{row}{ Which row to add the object to.  Must be between 1 and
    the-number-of-rows-currently-in-the-frame + 1, or \code{NULL} in
    which case the object occupies all rows. }
  \item{row.before}{ Add the object to a new row just before this row. }
  \item{row.after}{ Add the object to a new row just after this row. }
  \item{col}{ Which col to add the object to.  Must be between 1 and
    the-number-of-cols-currently-in-the-frame + 1, or \code{NULL} in
    which case the object occupies all cols.  }
  \item{col.before}{Add the object to a new col just before this col.}
  \item{col.after}{ Add the object to a new col just after this col. }
  \item{width}{ Specifies the width of the column that the object is
    added to (rather than allowing the width to be taken from the object). }
  \item{height}{ Specifies the height of the row that the object is
    added to (rather than allowing the height to be taken from the
    object). }
  \item{force.width}{ A logical value indicating whether the width of
    the column that the grob is being packed into should be EITHER the
    width specified in the call to \code{grid.pack} OR the maximum of
    that width and the pre-existing width. }
  \item{force.height}{ A logical value indicating whether the height of
    the column that the grob is being packed into should be EITHER the
    height specified in the call to \code{grid.pack} OR the maximum of
    that height and the pre-existing height. }
  \item{border}{ A \code{unit} object of length 4 indicating the borders
    around the object. }
  \item{dynamic}{ If the width/height is taken from the grob being
    packed, this boolean flag indicates whether the grobwidth/height
    unit refers directly to the grob, or uses a gPath to the grob.
    In the latter case, changes to the grob will trigger a
    recalculation of the width/height.}
}
\details{
  \code{packGrob} modifies the given frame grob and returns the modified
  frame grob.
  
  \code{grid.pack} destructively modifies a frame grob on the display
  list (and redraws the display list if \code{redraw} is \code{TRUE}).
  
  These are (meant to be) very flexible functions.  There are many
  different
  ways to specify where the new object is to be added relative to the
  objects already in the frame.  The function checks that the
  specification is not self-contradictory.

  NOTE that the width/height of the row/col that the object is added to
  is taken from the object itself unless the \code{width}/\code{height}
  is specified.
}
\value{
  \code{packGrob} returns a frame grob, but \code{grid.pack} returns
  \code{NULL}.
}
\author{ Paul Murrell }
\seealso{
  \code{\link{grid.frame}},
    \code{\link{grid.place}},
    \code{\link{grid.edit}}, and \code{\link{gPath}}.
}
\keyword{dplot}
\eof
\name{grid.place}
\alias{grid.place}
\alias{placeGrob}
\title{ Place an Object within a Frame }
\description{
  These functions provide a simpler (and faster) alternative
  to the \code{grid.pack()} and \code{packGrob}
  functions.  They can be used to place objects within the existing
  rows and columns of a frame layout.  They do not provide the ability to
  add new rows and columns nor do they affect the
  heights and widths of the rows and columns.
}
\usage{
grid.place(gPath, grob, row = 1, col = 1, redraw = TRUE)
placeGrob(frame, grob, row = NULL, col = NULL)
}
\arguments{
  \item{gPath}{ A gPath object, which specifies a frame on the
    display list.} 
  \item{frame}{ An object of class \code{frame}, typically the output
  from a call to \code{grid.frame}. }
  \item{grob}{ An object of class \code{grob}.  The object to be
    placed. }
  \item{row}{ Which row to add the object to.  Must be between 1 and
    the-number-of-rows-currently-in-the-frame. }
  \item{col}{ Which col to add the object to.  Must be between 1 and
    the-number-of-cols-currently-in-the-frame.  }
  \item{redraw}{ A boolean indicating whether the output should be updated. }
}
\details{
  \code{placeGrob} modifies the given frame grob and returns the modified
  frame grob.
  
  \code{grid.place} destructively modifies a frame grob on the display
  list (and redraws the display list if \code{redraw} is \code{TRUE}).
  
}
\value{
  \code{placeGrob} returns a frame grob, but \code{grid.place} returns
  \code{NULL}.
}
\author{ Paul Murrell }
\seealso{
  \code{\link{grid.frame}},
  \code{\link{grid.pack}},
  \code{\link{grid.edit}}, and \code{\link{gPath}}.
}
\keyword{dplot}
\eof
\name{grid.plot.and.legend}
\alias{grid.plot.and.legend}
\title{ A Simple Plot and Legend Demo }
\description{
  This function is just a wrapper for a simple demonstration of how
  a basic plot and legend can be drawn from scratch using grid.
}
\usage{
grid.plot.and.legend()
}
\author{ Paul Murrell }
\examples{
grid.plot.and.legend()
}
\keyword{dplot}
\eof
\name{grid.points}
\alias{grid.points}
\alias{pointsGrob}
\title{Draw Data Symbols}
\description{
  These functions create and draw data symbols.
}
\usage{
grid.points(x = runif(10),
            y = runif(10),
            pch = 1, size = unit(1, "char"),
            default.units = "native", name = NULL,
            gp=gpar(), draw = TRUE, vp = NULL)
pointsGrob(x = runif(10),
            y = runif(10),
            pch = 1, size = unit(1, "char"),
            default.units = "native", name = NULL,
            gp=gpar(), vp = NULL)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-values.}
  \item{y}{A numeric vector or unit object specifying y-values.}
  \item{pch}{A numeric or character vector
    indicating what sort of plotting symbol to use.}
  \item{size}{A unit object specifying the size of the plotting symbols.}
  \item{default.units}{A string indicating the default units to use
    if \code{x} or \code{y} are only given as numeric vectors.}  
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create a points grob (a graphical object describing
  points), but only \code{grid.points}
  draws the points (and then only if \code{draw} is \code{TRUE}).
  
}
\value{
  A points grob.  \code{grid.points} returns the value invisibly.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}
\keyword{dplot}
\eof
\name{grid.polygon}
\alias{grid.polygon}
\alias{polygonGrob}
\title{ Draw a Polygon }
\description{
  These functions create and draw a polygon.
  The final point will automatically
  be connected to the initial point.
}
\usage{
grid.polygon(x=c(0, 0.5, 1, 0.5), y=c(0.5, 1, 0.5, 0),
             id=NULL, id.lengths=NULL,
             default.units="npc", name=NULL,
             gp=gpar(), draw=TRUE, vp=NULL)
polygonGrob(x=c(0, 0.5, 1, 0.5), y=c(0.5, 1, 0.5, 0),
             id=NULL, id.lengths=NULL,
             default.units="npc", name=NULL,
             gp=gpar(), vp=NULL)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-locations.}
  \item{y}{A numeric vector or unit object specifying y-locations.}
  \item{id}{A numeric vector used to separate locations in \code{x} and
    \code{y} into multiple polygons.  All locations with the same
    \code{id} belong to the same polygon.}
  \item{id.lengths}{A numeric vector used to separate locations in \code{x} and
    \code{y} into multiple polygons.  Specifies consecutive blocks of
    locations which make up separate polygons.}
  \item{default.units}{A string indicating the default units to use
    if \code{x}, \code{y}, \code{width}, or \code{height}
    are only given as numeric vectors.}  
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\value{
  A grob object.
}
\details{
  Both functions create a polygon grob (a graphical object describing a
  polygon), but only \code{grid.polygon}
  draws the polygon (and then only if \code{draw} is \code{TRUE}).  
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}
\examples{
grid.polygon()
# Using id (NOTE: locations are not in consecutive blocks)
grid.newpage()
grid.polygon(x=c((0:4)/10, rep(.5, 5), (10:6)/10, rep(.5, 5)),
             y=c(rep(.5, 5), (10:6/10), rep(.5, 5), (0:4)/10),
             id=rep(1:5, 4),
             gp=gpar(fill=1:5))
# Using id.lengths
grid.newpage()
grid.polygon(x=outer(c(0, .5, 1, .5), 5:1/5),
             y=outer(c(.5, 1, .5, 0), 5:1/5),
             id.lengths=rep(4, 5),
             gp=gpar(fill=1:5))
}
\keyword{dplot}
\eof
\name{grid.pretty}
\alias{grid.pretty}
\title{ Generate a Sensible Set of Breakpoints }
\description{
  Produces a pretty set of breakpoints within the range given.
}
\usage{
grid.pretty(range)
}
\arguments{
  \item{range}{ A numeric vector }
}
\value{
  A numeric vector of breakpoints.
}
\author{ Paul Murrell }
\keyword{dplot}
\eof
\name{grid.rect}
\alias{grid.rect}
\alias{rectGrob}
\title{Draw rectangles }
\description{
  These functions create and draw rectangles.
}
\usage{
grid.rect(x = unit(0.5, "npc"), y = unit(0.5, "npc"),
      width = unit(1, "npc"), height = unit(1, "npc"),
      just = "centre", default.units = "npc", name = NULL,
      gp=gpar(), draw = TRUE, vp = NULL)
rectGrob(x = unit(0.5, "npc"), y = unit(0.5, "npc"),
      width = unit(1, "npc"), height = unit(1, "npc"),
      just = "centre", default.units = "npc", name = NULL,
      gp=gpar(), vp = NULL)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-location.}
  \item{y}{A numeric vector or unit object specifying y-location.}
  \item{width}{A numeric vector or unit object specifying width.}
  \item{height}{A numeric vector or unit object specifying height.}
  \item{just}{The justification of the rectangle about its (x, y)
    location.  If two values are given, the first specifies
    horizontal justification and the second specifies vertical
    justification.}
  \item{default.units}{A string indicating the default units to use
    if \code{x}, \code{y}, \code{width}, or \code{height}
    are only given as numeric vectors.}  
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create a rect grob (a graphical object describing
  rectangles), but only \code{grid.rect}
  draws the rectangles (and then only if \code{draw} is \code{TRUE}).
  
}
\value{
  A rect grob.  \code{grid.rect} returns the value invisibly.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}
\keyword{dplot}
\eof
\name{grid.remove}
\alias{grid.remove}
\alias{removeGrob}
\title{Remove a Grid Graphical Object}
\description{
  Remove a grob from a gTree or a descendant of a gTree.
}
\usage{
grid.remove(gPath, warn=TRUE, strict = FALSE, grep=FALSE, global=FALSE,
            allDevices=FALSE, redraw=TRUE) 
removeGrob(gTree, gPath, strict=FALSE, grep=FALSE, global=FALSE, warn=TRUE) 
}
\arguments{
  \item{gTree}{A gTree object.}
  \item{gPath}{ A gPath object. For \code{grid.remove} this
    specifyies a gTree on the display list.  For \code{removeGrob} this
    specifies a descendant of the specified gTree. }
  \item{strict}{ A boolean indicating whether the gPath must be
    matched exactly. }
  \item{grep}{A boolean indicating whether the \code{gPath} should
    be treated as a regular expression.  Values are recycled across
    elements of the \code{gPath} (e.g., \code{c(TRUE, FALSE)} means
    that every odd element of the \code{gPath} will be treated as
    a regular expression).
    NOT YET IMPLEMENTED.
  }
  \item{global}{ A boolean indicating whether the function should affect
    just the first match of the \code{gPath}, or whether all matches
    should be affected.
    NOT YET IMPLEMENTED.
  }
  \item{allDevices}{ A boolean indicating whether all open devices
    should
    be searched for matches, or just the current device.
    NOT YET IMPLEMENTED.
  }
  \item{warn}{A logical to indicate whether failing to find the
    specified grob should trigger an error.  }
  \item{redraw}{A logical value to indicate whether to redraw the grob. }
}
\details{
  \code{removeGrob} copies the specified grob and returns a modified
  grob.
  
  \code{grid.remove} destructively modifies a grob on the display list.
  If \code{redraw}
  is \code{TRUE} it then redraws everything to reflect the change.

}
\value{
  \code{removeGrob} returns a grob object;  \code{grid.remove} returns \code{NULL}.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grob}}, \code{\link{getGrob}},
  \code{\link{removeGrob}}, \code{\link{removeGrob}}.
}
\examples{
}
\keyword{dplot}
\eof
\name{grid.segments}
\alias{grid.segments}
\alias{segmentsGrob}
\title{ Draw Line Segments }
\description{
  These functions create and draw line segments.
}
\usage{
grid.segments(x0 = unit(0, "npc"), y0 = unit(0, "npc"),
              x1 = unit(1, "npc"), y1 = unit(1, "npc"),
              default.units = "npc", units.per.obs = FALSE, name = NULL,
              gp = gpar(), draw = TRUE, vp = NULL)
segmentsGrob(x0 = unit(0, "npc"), y0 = unit(0, "npc"),
              x1 = unit(1, "npc"), y1 = unit(1, "npc"),
              default.units = "npc", units.per.obs = FALSE, name = NULL,
              gp = gpar(), vp = NULL)
}
\arguments{
  \item{x0}{ Numeric indicating the starting x-values of the line segments. }
  \item{y0}{ Numeric indicating the starting y-values of the line
    segments. }
  \item{x1}{ Numeric indicating the stopping x-values of the line
    segments. }
  \item{y1}{ Numeric indicating the stopping y-values of the line segments.}
  \item{default.units}{ A string. }
  \item{units.per.obs}{ A boolean indicating whether distinct units are
    given for each x/y-value. }
  \item{name}{ A character identifier. }
  \item{gp}{ An object of class \code{gpar}. }
  \item{draw}{ A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL)}
}
\details{
  Both functions create a segments grob (a graphical object describing
  segments), but only \code{grid.segments}
  draws the segments (and then only if \code{draw} is \code{TRUE}).
  
}
\value{
  A segments grob.  \code{grid.segments} returns the value invisibly.
}
\author{ Paul Murrell                }
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}
\keyword{dplot}
\eof
\name{grid.set}
\alias{grid.set}
\alias{setGrob}
\title{Set a Grid Graphical Object}
\description{
  Replace a grob or a descendant of a grob.

}
\usage{
grid.set(gPath, newGrob, strict = FALSE, redraw = TRUE) 
setGrob(gTree, gPath, newGrob, strict = FALSE) 
}
\arguments{
  \item{gTree}{A gTree object.}
  \item{gPath}{ A gPath object. For \code{grid.set} this
    specifyies a grob on the display list.  For \code{setGrob} this
    specifies a descendant of the specified gTree. }
  \item{strict}{ A boolean indicating whether the gPath must be
    matched exactly. }
  \item{newGrob}{ A grob object. }
  \item{redraw}{A logical value to indicate whether to redraw the grob. }
}
\details{
  \code{setGrob} copies the specified grob and returns a modified
  grob.
  
  \code{grid.set} destructively replaces a grob on the display list.
  If \code{redraw}
  is \code{TRUE} it then redraws everything to reflect the change.

  These functions should not normally be called by the user.
}
\value{
  \code{setGrob} returns a grob object;  \code{grid.set} returns \code{NULL}.
}
\author{Paul Murrell}
\seealso{
  \code{\link{grid.grob}}.
}
\keyword{dplot}
\eof
\name{grid.show.layout}
\alias{grid.show.layout}
\title{Draw a Diagram of a Grid Layout}
\description{
  This function uses Grid graphics to draw a diagram of a
  Grid layout.
}
\usage{
grid.show.layout(l, newpage=TRUE,
             cell.border = "blue", cell.fill = "light blue",
             cell.label = TRUE, vp = NULL) 
}
\arguments{
  \item{l}{A Grid layout object.}
  \item{newpage}{A logical value indicating whether to move on
    to a new page before drawing the diagram.}
  \item{cell.border}{The colour used to draw the borders of the cells
    in the layout.}
  \item{cell.fill}{The colour used to fill the cells in the layout.}
  \item{cell.label}{A logical indicating whether the layout cells
    should be labelled.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  A viewport is created within \code{vp} to provide a margin for
  annotation, and the layout is drawn within that new viewport.
  The margin is filled with light grey, the new viewport is filled
  with white and framed with a black border, and the layout regions
  are filled with light blue and framed with a blue border.
  The diagram is annotated with the widths and heights (including units)
  of the columns and rows of the layout using red text.
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}},
  \code{\link{grid.layout}}
}

\examples{
## Diagram of a simple layout 
grid.show.layout(grid.layout(4,2,
                     heights=unit(rep(1, 4),
                                  c("lines", "lines", "lines", "null")),
                     widths=unit(c(1, 1), "inches")))
}
\keyword{dplot}
\eof
\name{grid.show.viewport}
\alias{grid.show.viewport}
\title{Draw a Diagram of a Grid Viewport}
\description{
  This function uses Grid graphics to draw a diagram of
  a Grid viewport.
}
\usage{
grid.show.viewport(v, parent.layout = NULL, newpage = TRUE, vp = NULL)
}
\arguments{
  \item{v}{A Grid viewport object.}
  \item{parent.layout}{A grid layout object.  If this is not NULL and
    the viewport given in \code{v} has its location specified relative
    to the layout, then the diagram shows the layout and which cells
    \code{v} occupies within the layout.}
  \item{newpage}{A logical value to indicate whether to move to
    a new page before drawing the diagram.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  A viewport is created within \code{vp} to provide a margin for
  annotation, and the diagram is drawn within that new viewport.
  The margin is filled with light grey, the new viewport is filled
  with white and framed with a black border, and the viewport region
  is filled with light blue and framed with a blue border.
  The diagram is annotated with the width and height (including units)
  of the viewport, the (x, y) location of the viewport, and
  the x- and y-scales of the viewport, using red lines and text.
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}

\examples{
## Diagram of a sample viewport
grid.show.viewport(viewport(x=0.6, y=0.6,
                            w=unit(1, "inches"), h=unit(1, "inches")))
grid.show.viewport(viewport(layout.pos.row=2, layout.pos.col=2:3),
                   grid.layout(3, 4))
}
\keyword{dplot}
\eof
\name{grid.text}
\alias{grid.text}
\alias{textGrob}
\title{Draw Text}
\description{
  These functions create and draw text.
}
\usage{
grid.text(label, x = unit(0.5, "npc"), y = unit(0.5, "npc"), 
          just = "centre", rot = 0,
          check.overlap = FALSE, default.units = "npc", name = NULL,
          gp=gpar(), draw = TRUE, vp = NULL)
textGrob(label, x = unit(0.5, "npc"), y = unit(0.5, "npc"), 
          just = "centre", rot = 0,
          check.overlap = FALSE, default.units = "npc", name = NULL,
          gp=gpar(), vp = NULL)
}
\arguments{
  \item{label}{A vector of strings or expressions to draw.}
  \item{x}{A numeric vector or unit object specifying x-values.}
  \item{y}{A numeric vector or unit object specifying y-values.}
  \item{just}{The justification of the text about its (x, y)
    location.  If two values are given, the first specifies
    horizontal justification and the second specifies vertical
    justification.}
  \item{rot}{The angle to rotate the text.}
  \item{check.overlap}{A logical value to indicate whether to
    check for and omit overlapping text.}
  \item{default.units}{A string indicating the default units to use
    if \code{x} or \code{y}
    are only given as numeric vectors.}  
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport object (or NULL).}
}
\details{
  Both functions create a text grob (a graphical object describing
  text), but only \code{grid.text}
  draws the text (and then only if \code{draw} is \code{TRUE}).
  
  If the \code{label} argument is an expression, the output is
  formatted as a mathematical annotation, as for base graphics text.
  
}
\value{
  A text grob.  \code{grid.text} returns the value invisibly.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}}
}

\examples{
grid.newpage()
x <- stats::runif(20)
y <- stats::runif(20)
rot <- stats::runif(20, 0, 360)
grid.text("SOMETHING NICE AND BIG", x=x, y=y, rot=rot,
          gp=gpar(fontsize=20, col="grey"))
grid.text("SOMETHING NICE AND BIG", x=x, y=y, rot=rot,
          gp=gpar(fontsize=20), check=TRUE)
grid.newpage()
draw.text <- function(just, i, j) {
  grid.text("ABCD", x=x[j], y=y[i], just=just)
  grid.text(deparse(substitute(just)), x=x[j], y=y[i] + unit(2, "lines"),
            gp=gpar(col="grey", fontsize=8))
}
x <- unit(1:4/5, "npc")
y <- unit(1:4/5, "npc")
grid.grill(h=y, v=x, gp=gpar(col="grey"))
draw.text(c("bottom"), 1, 1)
draw.text(c("left", "bottom"), 2, 1)
draw.text(c("right", "bottom"), 3, 1)
draw.text(c("centre", "bottom"), 4, 1)
draw.text(c("centre"), 1, 2)
draw.text(c("left", "centre"), 2, 2)
draw.text(c("right", "centre"), 3, 2)
draw.text(c("centre", "centre"), 4, 2)
draw.text(c("top"), 1, 3)
draw.text(c("left", "top"), 2, 3)
draw.text(c("right", "top"), 3, 3)
draw.text(c("centre", "top"), 4, 3)
draw.text(c(), 1, 4)
draw.text(c("left"), 2, 4)
draw.text(c("right"), 3, 4)
draw.text(c("centre"), 4, 4)
}
\keyword{dplot}
\eof
\name{grid.xaxis}
\alias{grid.xaxis}
\alias{xaxisGrob}
\title{Draw an X-Axis}
\description{
  These functions create and draw an x-axis.
}
\usage{
grid.xaxis(at = NULL, label = TRUE, main = TRUE, name = NULL, gp = gpar(),
           draw = TRUE, vp = NULL)
xaxisGrob(at = NULL, label = TRUE, main = TRUE, name = NULL, gp = gpar(),
         vp = NULL)
}
\arguments{
  \item{at}{A numeric vector of x-value locations for the tick marks.}
  \item{label}{A logical value indicating whether to draw the labels
    on the tick marks.}
  \item{main}{A logical value indicating whether to draw the axis
    at the bottom (\code{TRUE}) or at the top (\code{FALSE}) of the
    viewport.}
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport obect (or \code{NULL}).}
}
\details{
  Both functions create an xaxis grob (a graphical object describing an
  xaxis), but only \code{grid.xaxis}
  draws the xaxis (and then only if \code{draw} is \code{TRUE}).
  
}
\value{
  An xaxis grob.  \code{grid.xaxis} returns the value invisibly.
}
\section{Children}{
  If the \code{at} slot of an xaxis grob is not \code{NULL} then
  the xaxis will have the following children:
  \describe{
    \item{major}{ representing the line at the base of the
    tick marks.}
    \item{ticks}{representing the tick marks.}
    \item{labels}{representing the tick labels.}
  }

  If the \code{at} slot is \code{NULL} then there are no children
  and ticks are drawn based on the current viewport scale.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}},
  \code{\link{grid.yaxis}}
}
\keyword{dplot}
\eof
\name{grid.yaxis}
\alias{grid.yaxis}
\alias{yaxisGrob}
\title{Draw a Y-Axis}
\description{
  These functions create and draw a y-axis.
}
\usage{
grid.yaxis(at = NULL, label = TRUE, main = TRUE, name = NULL, gp  =gpar(),
           draw = TRUE, vp = NULL)
yaxisGrob(at = NULL, label = TRUE, main = TRUE, name = NULL, gp  =gpar(),
          vp = NULL)
}
\arguments{
  \item{at}{A numeric vector of y-value locations for the tick marks.}
  \item{label}{A logical value indicating whether to draw the labels
    on the tick marks.}
  \item{main}{A logical value indicating whether to draw the axis
    at the left (\code{TRUE}) or at the right (\code{FALSE}) of the
    viewport.}
  \item{name}{ A character identifier. }
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{draw}{A logical value indicating whether graphics output
    should be produced.}
  \item{vp}{A Grid viewport obect (or \code{NULL}).}
}
\details{
  Both functions create a yaxis grob (a graphical object describing a
  yaxis), but only \code{grid.yaxis}
  draws the yaxis (and then only if \code{draw} is \code{TRUE}).
  
}
\value{
  A yaxis grob.  \code{grid.yaxis} returns the value invisibly.
}
\section{Children}{
  If the \code{at} slot of an xaxis grob is not \code{NULL} then
  the xaxis will have the following children:
  \describe{
    \item{major}{ representing the line at the base of the
    tick marks.}
    \item{ticks}{representing the tick marks.}
    \item{labels}{representing the tick labels.}
  }

  If the \code{at} slot is \code{NULL} then there are no children
  and ticks are drawn based on the current viewport scale.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{viewport}},
  \code{\link{grid.xaxis}}
}
\keyword{dplot}
\eof
\name{grobWidth}
\alias{grobWidth}
\alias{grobHeight}
\title{Create a Unit Describing the Width of a Grob}
\description{
  These functions create a unit object describing the width or height of
  a grob.  They are generic.
}
\usage{
grobWidth(x)
grobHeight(x)
}
\arguments{
  \item{x}{A grob object.}
}
\value{
  A unit object. 
}
\author{Paul Murrell}
\seealso{
  \code{\link{unit}} and
  \code{\link{stringWidth}}
}
\keyword{dplot}
\eof
\name{plotViewport}
\alias{plotViewport}
\title{ Create a Viewport with a Standard Plot Layout }
\description{
  This is a convenience function for producing a viewport with
  the common S-style plot layout -- i.e., a central
  plot region surrounded by margins given in terms of a
  number of lines of text.
}
\usage{
plotViewport(margins, ...)
}
\arguments{
  \item{margins}{ A numeric vector interpreted in the same way
    as \code{par(mar)} in base graphics. }
  \item{\dots}{ All other arguments will be passed to a call to
  the \code{viewport()} function. }
}
\value{
  A grid viewport object.
}
\author{ Paul Murrell }
\seealso{
  \code{\link{viewport}} and
  \code{\link{dataViewport}}.
}
\keyword{dplot}
\eof
\name{pop.viewport}
\alias{pop.viewport}
\title{Pop a Viewport off the Grid Viewport Stack}
\description{
  Grid maintains a viewport stack --- a list of nested drawing
  contexts.
  
  This function makes the parent of the specified viewport the
  new default viewport.
}
\section{Warning}{
  This function has been deprecated.  Please use \code{popViewport}
  instead.
}
\usage{
pop.viewport(n=1, recording=TRUE)
}
\arguments{
  \item{n}{An integer giving the number of viewports to pop.
    Defaults to 1.}
  \item{recording}{A logical value to indicate whether the set-viewport
    operation should be recorded on the Grid display list.}
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \code{\link{push.viewport}}.
}
\keyword{dplot}
\eof
\name{push.viewport}
\alias{push.viewport}
\title{Push a Viewport onto the Grid Viewport Stack}
\description{
  Grid maintains a viewport stack --- a list of nested drawing
  contexts.
  
  This function makes the specified viewport the default viewport
  and makes its parent the previous default viewport (i.e.,
  nests the specified context within the previous default context).
}
\section{Warning}{
  This function has been deprecated.  Please use \code{pushViewport}
  instead.
}
\usage{
push.viewport(..., recording=TRUE)
}
\arguments{
  \item{\dots}{One or more objects of class \code{"viewport"}, or
    \code{NULL}.}
  \item{recording}{A logical value to indicate whether the set-viewport
    operation should be recorded on the Grid display list.}
}
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \code{\link{pop.viewport}}.
}
\keyword{dplot}
\eof
\name{stringWidth}
\alias{stringWidth}
\alias{stringHeight}
\title{Create a Unit Describing the Width of a String}
\description{
  These functions create a unit object describing the width or height of
  a string.
}
\usage{
stringWidth(string)
stringHeight(string)
}
\arguments{
  \item{string}{A character vector.}
}
\value{
  A unit object. 
}
\author{Paul Murrell}
\seealso{
  \code{\link{unit}} and
  \code{\link{grobWidth}}
}
\keyword{dplot}
\eof
\name{unit}
\alias{unit}
\title{Function to Create a Unit Object}
\description{
  This function creates a unit object --- a vector of unit values.
  A unit value is typically just a single numeric value with an
  associated unit.
}
\usage{
unit(x, units, data=NULL)
}
\arguments{
  \item{x}{A numeric vector.}
  \item{units}{A character vector specifying the units for the
    corresponding numeric values.}
  \item{data}{This argument is used to supply extra information
    for special \code{unit} types.}
}
\details{
  Unit objects allow the user to specify locations and dimensions
  in a large number of different coordinate systems.  All drawing
  occurs relative to a viewport and the \code{units} specifies
  what coordinate system to use within that viewport.
  
  Possible \code{units} (coordinate systems) are:
  \describe{
    \item{\code{"npc"}}{Normalised Parent Coordinates (the default).
      The origin of the viewport is (0, 0) and the viewport has a width
      and height of 1 unit.  For example, (0.5, 0.5) is the centre of
      the viewport.}
    \item{\code{"cm"}}{Centimetres.}
    \item{\code{"inches"}}{Inches.  1 in = 2.54 cm.}
    \item{\code{"mm"}}{Millimetres.  10 mm = 1 cm.}
    \item{\code{"points"}}{Points.  72.27 pt = 1 in.}
    \item{\code{"picas"}}{Picas.  1 pc = 12 pt.}
    \item{\code{"bigpts"}}{Big Points.  72 bp = 1 in.}
    \item{\code{"dida"}}{Dida. 1157 dd = 1238 pt.}
    \item{\code{"cicero"}}{Cicero.  1 cc = 12 dd.}
    \item{\code{"scaledpts"}}{Scaled Points.  65536 sp = 1 pt.}
    \item{\code{"lines"}}{Lines of text.  Locations and dimensions
      are in terms of multiples of the default text size of the viewport
      (as specified by the viewport's \code{fontsize} and \code{lineheight}).}
    \item{\code{"char"}}{Multiples of nominal font height of the
      viewport (as specified by the viewport's \code{fontsize}).}
    \item{\code{"native"}}{Locations and dimensions are relative to
      the viewport's \code{xscale} and \code{yscale}.}
    \item{\code{"snpc"}}{Square Normalised Parent Coordinates.
      Same as Normalised Parent Coordinates, except gives the same answer for
      horizontal and vertical locations/dimensions.  It uses the
      \emph{lesser} of npc-width and npc-height.  This is useful for making
      things which are a proportion of the viewport, but have to be square 
      (or have a fixed aspect ratio).}
    \item{\code{"strwidth"}}{Multiples of the width of the string
      specified in the \code{data} argument.  The font size is
      determined by the pointsize of the viewport.}
    \item{\code{"strheight"}}{Multiples of the height of the string
      specified in the \code{data} argument.  The font size is
      determined by the pointsize of the viewport.}
    \item{\code{"grobwidth"}}{Multiples of the width of the grob
      specified in the \code{data} argument.}
    \item{\code{"grobheight"}}{Multiples of the height of the grob
      specified in the \code{data} argument.}
  }

  The \code{data} argument must be a list when the \code{unit.length()}
  is greater than 1.  For example,
  \code{unit(rep(1, 3), c("npc", "strwidth", "inches"),
        data=list(NULL, "my string", NULL))}.
      
  It is possible to subset unit objects in the normal way
  (e.g., \code{unit(1:5, "npc")[2:4]}), but a special function
  \code{unit.c} is provided for combining unit objects.

  Certain arithmetic and summary operations are defined for
  unit objects.  In particular, it is possible to add and subtract
  unit objects (e.g., \code{unit(1, "npc") - unit(1, "inches")}),
  and to specify the minimum or maximum of a list
  of unit objects (e.g., \code{min(unit(0.5, "npc"), unit(1,
    "inches"))}).
}
\section{WARNING}{
  A special function \code{unit.length} is provided for
  determining the number of unit values in a unit object.

  The \code{length} function will work in some cases, but in general
  will not give the right answer.

  There is also a special function \code{unit.c} for concatenating
  several unit objects.

  The \code{c} function will not give the right answer.

  There used to be \code{"mylines"}, \code{"mychar"}, \code{"mystrwidth"},
  \code{"mystrheight"} units.  These will still be accepted, but
  work exactly the same as
  \code{"lines"}, \code{"char"}, \code{"strwidth"},
  \code{"strheight"}.
}
\value{
  An object of class \code{"unit"}. 
}
\author{Paul Murrell}
\seealso{
  \code{\link{unit.c}} and
  \code{\link{unit.length}}
}

\examples{
unit(1, "npc")
unit(1:3/4, "npc")
unit(1:3/4, "npc") + unit(1, "inches")
min(unit(0.5, "npc"), unit(1, "inches"))
unit.c(unit(0.5, "npc"), unit(2, "inches") + unit(1:3/4, "npc"),
       unit(1, "strwidth", "hi there"))
}
\keyword{dplot}
\eof
\name{unit.c}
\alias{unit.c}
\title{Combine Unit Objects}
\description{
  This function produces a new unit object by combining the
  unit objects specified as arguments.
}
\usage{
unit.c(...)
}
\arguments{
  \item{\dots}{An arbitrary number of unit objects.}
}
\value{
  An object of class \code{unit}.
}
\author{Paul Murrell}
\seealso{
  \code{\link{unit}}.
}
\keyword{dplot}
\eof
\name{unit.length}
\alias{unit.length}
\title{Length of a Unit Object}
\description{
  The length of a unit object is defined as the number of unit
  values in the unit object.
}
\usage{
unit.length(unit)
}
\arguments{
  \item{unit}{A unit object.}
}
\value{
  An object of class \code{unit}.
}
\author{Paul Murrell}
\seealso{
  \code{\link{unit}}
}
\keyword{dplot}
\eof
\name{unit.pmin}
\alias{unit.pmin}
\alias{unit.pmax}
\title{ Parallel Unit Minima and Maxima }
\description{
  Returns a unit object whose i'th value is the minimum (or maximum)
  of the i'th values of the arguments.  
}
\usage{
unit.pmin(...)
unit.pmax(...)
}
\arguments{
  \item{\dots}{ One or more unit objects. }
}
\details{
  The length of the result is the maximum of the lengths of the
  arguments;  shorter arguments are recycled in the usual manner.
}
\value{
  A unit object.
}
\author{ Paul Murrell }
\examples{
max(unit(1:3, "cm"), unit(0.5, "npc"))
unit.pmax(unit(1:3, "cm"), unit(0.5, "npc"))
}
\keyword{dplot}
\eof
\name{unit.rep}
\alias{unit.rep}
\title{ Replicate Elements of Unit Objects }
\description{
  Replicates the units according to the values given in \code{times} and
  \code{length.out}.
}
\usage{
unit.rep(x, times, length.out)
}
\arguments{
  \item{x}{ An object of class \code{"unit"}. }
  \item{times}{integer.  A vector giving the number of times to repeat
    each element.  Either of length 1 or \code{length(x)}. }
  \item{length.out}{integer.  (Optional.)  The desired length of the
    output vector.}
}
\value{
  An object of class \code{"unit"}.
}
\author{ Paul Murrell }
 
\seealso{
  \code{\link{rep}}
}

\examples{
unit.rep(unit(1:3, "npc"), 3)
unit.rep(unit(1:3, "npc"), 1:3)
unit.rep(unit(1:3, "npc") + unit(1, "inches"), 3)
unit.rep(max(unit(1:3, "npc") + unit(1, "inches")), 3)
unit.rep(max(unit(1:3, "npc") + unit(1, "strwidth", "a"))*4, 3)
unit.rep(unit(1:3, "npc") + unit(1, "strwidth", "a")*4, 3)
}
\keyword{dplot}
\eof
\name{validDetails}
\alias{validDetails}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Customising grid grob Validation }
\description{
  This generic hook function is called whenever a grid grob is created
  or
  edited
  via \code{grob}, \code{gTree}, \code{grid.edit} or \code{editGrob}.
  This provides an opportunity for customising the validation of a
  new class derived from grob (or gTree).
}
\usage{
validDetails(x)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ A grid grob. }
}
\details{
  This function is called by \code{grob}, \code{gTree},
  \code{grid.edit} and \code{editGrob}.
  A method should be written for classes derived from grob or gTree
  to validate the values of slots specific to the new class.
  (e.g., see \code{grid:::validDetails.axis}).

  Note that the standard slots for grobs and gTrees are automatically
  validated (e.g., \code{vp}, \code{gp} slots for grobs and,
  in addition, \code{children}, and \code{childrenvp} slots for
  gTrees) so only slots specific to a new class need to be addressed.
}
\value{
  The function MUST return the validated grob.
}
\author{ Paul Murrell }
\seealso{ \code{\link{grid.edit}} }
\keyword{ dplot }% at least one, from doc/KEYWORDS
\eof
\name{Grid Viewports}
\alias{viewport}
\alias{vpList}
\alias{vpStack}
\alias{vpTree}
\title{Create a Grid Viewport}
\description{
  These functions create viewports, which describe a rectangular regions
  on a graphics device and define a number of coordinate systems within
  those regions.
}
\usage{
viewport(x = unit(0.5, "npc"), y = unit(0.5, "npc"),
         width = unit(1, "npc"), height = unit(1, "npc"),
         default.units = "npc", just = "centre", 
         gp = gpar(), clip = "inherit",
         xscale = c(0, 1), yscale = c(0, 1),
         angle = 0,
         layout = NULL, layout.pos.row = NULL, layout.pos.col = NULL,
         name=NULL)
vpList(...)
vpStack(...)
vpTree(parent, children)
}
\arguments{
  \item{x}{A numeric vector or unit object specifying x-location.}
  \item{y}{A numeric vector or unit object specifying y-location.}
  \item{width}{A numeric vector or unit object specifying width.}
  \item{height}{A numeric vector or unit object specifying height.}
  \item{default.units}{A string indicating the default units to use
    if \code{x}, \code{y}, \code{width}, or \code{height}
    are only given as numeric vectors.}  
  \item{just}{A string vector specifying the justification of the viewport
    relative to its (x, y) location.  If there are two values, the first
    value specifes horizontal justification and the second value specifies
    vertical justification.  Possible values are: \code{"left"},
    \code{"right"}, \code{"centre"}, \code{"center"}, \code{"bottom"},
    and \code{"top"}.}
  \item{gp}{An object of class \code{gpar}, typically the output
    from a call to the function \code{gpar}.  This is basically
    a list of graphical parameter settings.}
  \item{clip}{One of \code{"on"}, \code{"inherit"}, or
    \code{"off"}, indicating whether to
    clip to the extent of this viewport, inherit the clipping region
    from the parent viewport, or turn clipping off altogether.
    For back-compatibility, a logical value of \code{TRUE} corresponds
    to \code{"on"} and \code{FALSE} corresponds to \code{"inherit"}.}
  \item{xscale}{A numeric vector of length two indicating the minimum and
    maximum on the x-scale.}
  \item{yscale}{A numeric vector of length two indicating the minimum
    and maximum on the y-scale.}
  \item{angle}{A numeric value indicating the angle of rotation of the
    viewport.  Positive values indicate the amount of rotation, in
    degrees, anitclockwise from the positive x-axis.}
  \item{layout}{A Grid layout object which splits the viewport into
    subregions.}
  \item{layout.pos.row}{A numeric vector giving the
    rows occupied by this viewport in its
    parent's layout.}
  \item{layout.pos.col}{A numeric vector giving the
    columns occupied by this viewport in its
    parent's layout.}
  \item{name}{A character value to uniquely identify the viewport
    once it has been pushed onto the viewport tree.  }
  \item{...}{Any number of grid viewport objects.}
  \item{parent}{A grid viewport object.}
  \item{children}{A vpList object.}
}
\details{
  The location and size of a viewport are relative to the coordinate
  systems defined by the viewport's parent (either a graphical device
  or another viewport).  The location and size can be specified in a
  very flexible way by specifying them with unit objects.
  When specifying the location of a viewport, specifying
  both \code{layout.pos.row} and \code{layout.pos.col} as \code{NULL}
  indicates that
  the viewport ignores its parent's layout and specifies its own
  location and size (via its \code{locn}).  If only one of
  \code{layout.pos.row} and \code{layout.pos.col} is \code{NULL}, this
  means occupy ALL of the appropriate row(s)/column(s).  For example,
  \code{layout.pos.row = 1} and \code{layout.pos.col = NULL} means
  occupy all of row 1.  Specifying non-\code{NULL} values for both
  \code{layout.pos.row} and \code{layout.pos.col} means occupy the
  intersection of the appropriate rows and columns.  If a vector of
  length two is
  specified for \code{layout.pos.row} or \code{layout.pos.col}, this
  indicates a range of rows or columns to occupy.  For example,
  \code{layout.pos.row = c(1, 3)} and \code{layout.pos.col = c(2, 4)}
  means occupy cells in the intersection of rows 1, 2, and 3, and
  columns, 2, 3, and 4.

    Clipping obeys only the most recent viewport clip setting.
    For example, if you clip to viewport1, then clip to viewport2,
    the clipping region is determined wholly by viewport2, the
    size and shape of viewport1 is irrelevant (until viewport2
    is popped of course).

    If a viewport is rotated (because of its own \code{angle} setting
    or because it is within another viewport which is rotated) then
    the \code{clip} flag is ignored.

    Viewport names need not be unique.  When pushed, viewports
    sharing the same parent must have unique names, which means that
    if you push a viewport with the same name as an existing viewport,
    the existing viewport will be replaced in the viewport tree.
    A viewport name can be any valid \R symbol name, but
    grid uses the
    reserved name \code{"ROOT"} for the top-level viewport.  Also,
    when specifying a viewport name in \code{downViewport}
    and \code{seekViewport}, it is possible to provide a viewport
    path, which consists of several names concatenated using the
    separator  (currently \code{::}).  Consequently, it is not
    advisable to use this separator in viewport names.

The viewports in a \code{vpList} are pushed in parallel.  The
    viewports in a \code{vpStack} are pushed in series.  When a
    \code{vpTree} is pushed, the parent is pushed first, then the
    children are pushed in parallel.
}
\value{
  An R object of class \code{viewport}.
}
\author{Paul Murrell}
\seealso{
  \link{Grid},
  \code{\link{pushViewport}},
  \code{\link{popViewport}},
  \code{\link{downViewport}},
  \code{\link{seekViewport}},
  \code{\link{upViewport}},
  \code{\link{unit}},
  \code{\link{grid.layout}},
  \code{\link{grid.show.layout}}.
}

\examples{
# Diagram of a sample viewport
grid.show.viewport(viewport(x=0.6, y=0.6,
                   w=unit(1, "inches"), h=unit(1, "inches")))
# Demonstrate viewport clipping
clip.demo <- function(i, j, clip1, clip2) {
  pushViewport(viewport(layout.pos.col=i,
                         layout.pos.row=j))
  pushViewport(viewport(width=0.6, height=0.6, clip=clip1))
  grid.rect(gp=gpar(fill="white"))
  grid.circle(r=0.55, gp=gpar(col="red", fill="pink"))
  popViewport()
  pushViewport(viewport(width=0.6, height=0.6, clip=clip2))
  grid.polygon(x=c(0.5, 1.1, 0.6, 1.1, 0.5, -0.1, 0.4, -0.1),
               y=c(0.6, 1.1, 0.5, -0.1, 0.4, -0.1, 0.5, 1.1),
               gp=gpar(col="blue", fill="light blue"))
  popViewport(2)
}

grid.newpage()
grid.rect(gp=gpar(fill="grey"))
pushViewport(viewport(layout=grid.layout(2, 2)))
clip.demo(1, 1, FALSE, FALSE)
clip.demo(1, 2, TRUE, FALSE)
clip.demo(2, 1, FALSE, TRUE)
clip.demo(2, 2, TRUE, TRUE)
popViewport()
# Demonstrate turning clipping off
grid.newpage()
pushViewport(viewport(w=.5, h=.5, clip="on"))
grid.rect()
grid.circle(r=.6, gp=gpar(lwd=10))
pushViewport(viewport(clip="inherit"))
grid.circle(r=.6, gp=gpar(lwd=5, col="grey"))
pushViewport(viewport(clip="off"))
grid.circle(r=.6)
popViewport(3)
# Demonstrate vpList, vpStack, and vpTree
grid.newpage()
tree <- vpTree(viewport(w=0.8, h=0.8, name="A"),
               vpList(vpStack(viewport(x=0.1, y=0.1, w=0.5, h=0.5,
                                       just=c("left", "bottom"), name="B"),
                              viewport(x=0.1, y=0.1, w=0.5, h=0.5, 
                                       just=c("left", "bottom"), name="C"),
                              viewport(x=0.1, y=0.1, w=0.5, h=0.5, 
                                       just=c("left", "bottom"), name="D")),
                      viewport(x=0.5, w=0.4, h=0.9,
                               just="left", name="E")))
pushViewport(tree)
for (i in LETTERS[1:5]) {
  seekViewport(i)
  grid.rect()
  grid.text(current.vpTree(FALSE),
            x=unit(1, "mm"), y=unit(1, "npc") - unit(1, "mm"),
            just=c("left", "top"),
            gp=gpar(fontsize=8))
}
}
\keyword{dplot}
\eof
\name{Working with Viewports}
\alias{pushViewport}
\alias{downViewport}
\alias{seekViewport}
\alias{popViewport}
\alias{upViewport}
\title{Maintaining and Navigating the Grid Viewport Tree}
\description{
  Grid maintains a tree of viewports --- nested drawing
  contexts. 

  These functions provide ways to add or remove viewports
  and to navigate amongst viewports in the tree.
}
\usage{
pushViewport(..., recording=TRUE)
popViewport(n, recording=TRUE)
downViewport(name, strict=FALSE, recording=TRUE)
seekViewport(name, recording=TRUE)
upViewport(n, recording=TRUE)
}
\arguments{
  \item{\dots}{One or more objects of class \code{"viewport"}.}
  \item{n}{An integer value indicating how many viewports to
    pop or navigate up.  The special value \code{0} indicates to
    pop or navigate viewports right up to the root viewport.}
  \item{name}{A character value to identify a viewport in the tree.}
  \item{strict}{ A boolean indicating whether the vpPath must be
    matched exactly. }
  \item{recording}{A logical value to indicate whether the viewport
    operation should be recorded on the Grid display list.}
}
\details{
  Objects created by the \code{viewport()} function are only
  descriptions
  of a drawing context.  A viewport object must be pushed onto the
  viewport tree before it has any effect on drawing.
  
  The viewport tree always has a single root viewport (created by the
  system) which corresponds to the entire device (and default
  graphical parameter settings).  Viewports may be added to the tree
  using \code{pushViewport()} and removed from the tree using
  \code{popViewport()}.  

  There is only ever one current viewport, which is the current
  position within the viewport tree.  All drawing and
  viewport operations are relative to the current viewport.
  When a viewport is pushed it becomes the current viewport.
  When a viewport is popped, the parent viewport becomes
  the current viewport.  Use \code{upViewport} to navigate to the parent
  of the current viewport, without removing the current viewport
  from the viewport tree.  Use \code{downViewport} to navigate to
  a viewport further down the viewport tree and \code{seekViewport}
  to navigate to a viewport anywhere else in the tree.

  If a viewport is pushed and it has the same \code{name} as a
  viewport at the same level in the tree, then it replaces the
  existing viewport in the tree.
  }
\value{
  None.
}
\author{Paul Murrell}
\seealso{
  \code{\link{viewport}} and \code{\link{vpPath}}.
}
\examples{
# push the same viewport several times
grid.newpage()
vp <- viewport(width=0.5, height=0.5)
pushViewport(vp)
grid.rect(gp=gpar(col="blue"))
grid.text("Quarter of the device",
  y=unit(1, "npc") - unit(1, "lines"), gp=gpar(col="blue"))
pushViewport(vp)
grid.rect(gp=gpar(col="red"))
grid.text("Quarter of the parent viewport",
  y=unit(1, "npc") - unit(1, "lines"), gp=gpar(col="red"))
popViewport(2)
# push several viewports then navigate amongst them
grid.newpage()
grid.rect(gp=gpar(col="grey"))
grid.text("Top-level viewport",
  y=unit(1, "npc") - unit(1, "lines"), gp=gpar(col="grey"))
if (interactive()) Sys.sleep(1.0)
pushViewport(viewport(width=0.8, height=0.7, name="A"))
grid.rect(gp=gpar(col="blue"))
grid.text("1. Push Viewport A",
  y=unit(1, "npc") - unit(1, "lines"), gp=gpar(col="blue"))
if (interactive()) Sys.sleep(1.0)
pushViewport(viewport(x=0.1, width=0.3, height=0.6,
  just="left", name="B"))
grid.rect(gp=gpar(col="red"))
grid.text("2. Push Viewport B (in A)",
  y=unit(1, "npc") - unit(1, "lines"), gp=gpar(col="red"))
if (interactive()) Sys.sleep(1.0)
upViewport(1)
grid.text("3. Up from B to A",
  y=unit(1, "npc") - unit(2, "lines"), gp=gpar(col="blue"))
if (interactive()) Sys.sleep(1.0)
pushViewport(viewport(x=0.5, width=0.4, height=0.8,
  just="left", name="C"))
grid.rect(gp=gpar(col="green"))
grid.text("4. Push Viewport C (in A)",
  y=unit(1, "npc") - unit(1, "lines"), gp=gpar(col="green"))
if (interactive()) Sys.sleep(1.0)
pushViewport(viewport(width=0.8, height=0.6, name="D"))
grid.rect()
grid.text("5. Push Viewport D (in C)",
  y=unit(1, "npc") - unit(1, "lines"))
if (interactive()) Sys.sleep(1.0)
upViewport(0)
grid.text("6. Up from D to top-level",
  y=unit(1, "npc") - unit(2, "lines"), gp=gpar(col="grey"))
if (interactive()) Sys.sleep(1.0)
downViewport("D")
grid.text("7. Down from top-level to D",
  y=unit(1, "npc") - unit(2, "lines"))
if (interactive()) Sys.sleep(1.0)
seekViewport("B")
grid.text("8. Seek from D to B",
  y=unit(1, "npc") - unit(2, "lines"), gp=gpar(col="red"))
pushViewport(viewport(width=0.9, height=0.5, name="A"))
grid.rect()
grid.text("9. Push Viewport A (in B)",
  y=unit(1, "npc") - unit(1, "lines"))
if (interactive()) Sys.sleep(1.0)
seekViewport("A")
grid.text("10. Seek from B to A (in ROOT)",
  y=unit(1, "npc") - unit(3, "lines"), gp=gpar(col="blue"))
if (interactive()) Sys.sleep(1.0)
seekViewport(vpPath("B", "A"))
grid.text("11. Seek from\nA (in ROOT)\nto A (in B)")
popViewport(0)
}
\keyword{dplot}
\eof
\name{vpPath}
\alias{vpPath}
\title{ Concatenate Viewport Names }
\description{
  This function can be used to generate a viewport path for use
  in \code{downViewport} or \code{seekViewport}.

  A viewport path is a list of nested viewport names.
}
\usage{
vpPath(...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{\dots}{ Character values which are viewport names. }
}
\details{
  Viewport names must only be unique amongst viewports which
  share the same parent in the viewport tree.

  This function can be used to generate a specification for
  a viewport that includes the viewport's parent's name
  (and the name of its parent and so on).

  For interactive use, it is possible to directly specify
  a path, but it is strongly recommended that this function
  is used otherwise in case the path separator is changed
  in future versions of grid.
}
\value{
  A \code{ vpPath } object.
}
\seealso{
  \code{\link{viewport}},
  \code{\link{pushViewport}},
  \code{\link{popViewport}},
  \code{\link{downViewport}},
  \code{\link{seekViewport}},
  \code{\link{upViewport}}
}
\examples{
vpPath("vp1", "vp2")
}
\keyword{ dplot }% at least one, from doc/KEYWORDS
\eof
\name{widthDetails}
\alias{widthDetails}
\alias{heightDetails}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Width and Height of a grid grob }
\description{
  These generic functions are used to determine the size of grid grobs.
}
\usage{
widthDetails(x)
heightDetails(x)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ A grid grob. }
}
\details{
  These functions are called in the calculation of
  \code{"grobwidth"} and \code{"grobheight"} units.
  Methods should be written for classes derived from grob or gTree
  where the size of the grob can be determined (see, for example
  \code{grid:::widthDetails.frame}).
}
\value{
  A unit object.
}
\seealso{
  \code{\link{absolute.size}}.
}
\author{ Paul Murrell }
\keyword{ dplot }% at least one, from doc/KEYWORDS
\eof
