#copyright ReportLab Inc. 2001
#see license.txt for license details
#history http://cvs.sourceforge.net/cgi-bin/cvsweb.cgi/reportlab/pdfbase/_fontdata.py?cvsroot=reportlab
#$Header $
__version__=''' $Id: _fontdata.py,v 1.13 2003/07/07 19:38:31 rgbecker Exp $ '''
__doc__="""
    database of font related things
    standardFonts       tuple of the 14 standard string font names
    standardEncodings   tuple of the known standard font names
    encodings           a mapping object from standard encoding names (and minor variants)
                        to the encoding vectors ie the tuple of string glyph names
    widthsByFontGlyph   fontname x glyphname --> width of glyph
    widthVectorsByFont  fontName -> vector of widths
"""
import string, UserDict, os, sys

# mapping of name to width vector, starts empty until fonts are added
# e.g. widths['Courier'] = [...600,600,600,...]
widthVectorsByFont = {}
fontsByName = {}
fontsByBaseEnc = {}
# this is a list of the standard 14 font names in Acrobat Reader
standardFonts = (
    'Courier', 'Courier-Bold', 'Courier-Oblique', 'Courier-BoldOblique',
    'Helvetica', 'Helvetica-Bold', 'Helvetica-Oblique', 'Helvetica-BoldOblique',
    'Times-Roman', 'Times-Bold', 'Times-Italic', 'Times-BoldItalic',
    'Symbol','ZapfDingbats')

#this maps fontnames to the equivalent filename root.
_font2fnrMapWin32 = {
                    'symbol':                   'Sy______',
                    'zapfdingbats':             'Zd______',
                    'helvetica':                '_a______',
                    'helvetica-bold':           '_ab_____',
                    'helvetica-boldoblique':    '_abi____',
                    'helvetica-oblique':        '_ai_____',
                    'times-bold':               '_eb_____',
                    'times-bolditalic':         '_ebi____',
                    'times-italic':             '_ei_____',
                    'times-roman':              '_er_____',
                    'courier-bold':             'cob_____',
                    'courier-boldoblique':      'cobo____',
                    'courier':                  'com_____',
                    'courier-oblique':          'coo_____',
                    }
if sys.platform in ('linux2',):
    _font2fnrMapLinux2 ={
                'symbol': 'Symbol',
                'zapfdingbats': 'ZapfDingbats',
                'helvetica': 'Arial',
                'helvetica-bold': 'Arial-Bold',
                'helvetica-boldoblique': 'Arial-BoldItalic',
                'helvetica-oblique': 'Arial-Italic',
                'times-bold': 'TimesNewRoman-Bold',
                'times-bolditalic':'TimesNewRoman-BoldItalic',
                'times-italic': 'TimesNewRoman-Italic',
                'times-roman': 'TimesNewRoman',
                'courier-bold': 'Courier-Bold',
                'courier-boldoblique': 'Courier-BoldOblique',
                'courier': 'Courier',
                'courier-oblique': 'Courier-Oblique',
                }
    _font2fnrMap = _font2fnrMapLinux2
    _revmap = None
else:
    _font2fnrMap = _font2fnrMapWin32

def _findFNR(fontName):
    return _font2fnrMap[string.lower(fontName)]

def _searchT1Dirs(n):
    from reportlab.rl_config import T1SearchPath
    assert T1SearchPath!=[], "No Type-1 font search path"
    for d in T1SearchPath:
        f = os.path.join(d,n)
        if os.path.isfile(f): return f
    return None

def findT1File(fontName,ext='.pfb'):
    if sys.platform in ('linux2',) and ext=='.pfb':
        try:
            f = _searchT1Dirs(_findFNR(fontName))
            if f: return f
        except:
            pass
        global _revmap
        if not _revmap:
            for k, v in _font2fnrMap.items():
                if k in _font2fnrMapWin32.keys():
                    _font2fnrMapWin32[string.lower(v)] = _font2fnrMapWin32[k]
            revmap = 1
        try:
            f = _searchT1Dirs(_font2fnrMapWin32[string.lower(fontName)]+ext)
            if f: return f
        except:
            pass

    return _searchT1Dirs(_findFNR(fontName)+ext)

# this lists the predefined font encodings - WinAnsi and MacRoman.  We have
# not added MacExpert - it's possible, but would complicate life and nobody
# is asking.  StandardEncoding means something special.
standardEncodings = ('WinAnsiEncoding','MacRomanEncoding','StandardEncoding','SymbolEncoding','ZapfDingbatsEncoding','PDFDocEncoding', 'MacExpertEncoding')

#this is the global mapping of standard encodings to name vectors
class _Name2StandardEncodingMap(UserDict.UserDict):
    '''Trivial fake dictionary with some [] magic'''
    _XMap = {'winansi':'WinAnsiEncoding','macroman': 'MacRomanEncoding','standard':'StandardEncoding','symbol':'SymbolEncoding', 'zapfdingbats':'ZapfDingbatsEncoding','pdfdoc':'PDFDocEncoding', 'macexpert':'MacExpertEncoding'}
    def __setitem__(self,x,v):
        y = string.lower(x)
        if y[-8:]=='encoding': y = y[:-8]
        y = self._XMap[y]
        if y in self.keys(): raise IndexError, 'Encoding %s is already set' % y
        self.data[y] = v

    def __getitem__(self,x):
        y = string.lower(x)
        if y[-8:]=='encoding': y = y[:-8]
        y = self._XMap[y]
        return self.data[y]

encodings = _Name2StandardEncodingMap()
encodings['WinAnsiEncoding'] = (
                None, None, None, None, None, None, None, None, None, None, None, None,
                None, None, None, None, None, None, None, None, None, None, None, None,
                None, None, None, None, None, None, None, None, 'space', 'exclam',
                'quotedbl', 'numbersign', 'dollar', 'percent', 'ampersand',
                'quotesingle', 'parenleft', 'parenright', 'asterisk', 'plus', 'comma',
                'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three', 'four',
                'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon', 'less',
                'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F',
                'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
                'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright',
                'asciicircum', 'underscore', 'grave', 'a', 'b', 'c', 'd', 'e', 'f',
                'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
                'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright',
                'asciitilde', 'bullet', 'Euro', 'bullet', 'quotesinglbase', 'florin',
                'quotedblbase', 'ellipsis', 'dagger', 'daggerdbl', 'circumflex',
                'perthousand', 'Scaron', 'guilsinglleft', 'OE', 'bullet', 'Zcaron',
                'bullet', 'bullet', 'quoteleft', 'quoteright', 'quotedblleft',
                'quotedblright', 'bullet', 'endash', 'emdash', 'tilde', 'trademark',
                'scaron', 'guilsinglright', 'oe', 'bullet', 'zcaron', 'Ydieresis',
                'space', 'exclamdown', 'cent', 'sterling', 'currency', 'yen', 'brokenbar',
                'section', 'dieresis', 'copyright', 'ordfeminine', 'guillemotleft',
                'logicalnot', 'hyphen', 'registered', 'macron', 'degree', 'plusminus',
                'twosuperior', 'threesuperior', 'acute', 'mu', 'paragraph', 'periodcentered',
                'cedilla', 'onesuperior', 'ordmasculine', 'guillemotright', 'onequarter',
                'onehalf', 'threequarters', 'questiondown', 'Agrave', 'Aacute',
                'Acircumflex', 'Atilde', 'Adieresis', 'Aring', 'AE', 'Ccedilla',
                'Egrave', 'Eacute', 'Ecircumflex', 'Edieresis', 'Igrave', 'Iacute',
                'Icircumflex', 'Idieresis', 'Eth', 'Ntilde', 'Ograve', 'Oacute',
                'Ocircumflex', 'Otilde', 'Odieresis', 'multiply', 'Oslash', 'Ugrave',
                'Uacute', 'Ucircumflex', 'Udieresis', 'Yacute', 'Thorn', 'germandbls',
                'agrave', 'aacute', 'acircumflex', 'atilde', 'adieresis', 'aring', 'ae',
                'ccedilla', 'egrave', 'eacute', 'ecircumflex', 'edieresis', 'igrave',
                'iacute', 'icircumflex', 'idieresis', 'eth', 'ntilde', 'ograve', 'oacute',
                'ocircumflex', 'otilde', 'odieresis', 'divide', 'oslash', 'ugrave', 'uacute',
                'ucircumflex', 'udieresis', 'yacute', 'thorn', 'ydieresis')

encodings['MacRomanEncoding'] = (
                 None, None, None, None, None, None, None, None, None, None, None, None,
                 None, None, None, None, None, None, None, None, None, None, None, None,
                 None, None, None, None, None, None, None, None, 'space', 'exclam',
                 'quotedbl', 'numbersign', 'dollar', 'percent', 'ampersand',
                 'quotesingle', 'parenleft', 'parenright', 'asterisk', 'plus', 'comma',
                 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three', 'four',
                 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon', 'less',
                 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F',
                 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
                 'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright',
                 'asciicircum', 'underscore', 'grave', 'a', 'b', 'c', 'd', 'e', 'f',
                 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
                 'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright',
                 'asciitilde', None, 'Adieresis', 'Aring', 'Ccedilla', 'Eacute',
                 'Ntilde', 'Odieresis', 'Udieresis', 'aacute', 'agrave', 'acircumflex',
                 'adieresis', 'atilde', 'aring', 'ccedilla', 'eacute', 'egrave',
                 'ecircumflex', 'edieresis', 'iacute', 'igrave', 'icircumflex',
                 'idieresis', 'ntilde', 'oacute', 'ograve', 'ocircumflex', 'odieresis',
                 'otilde', 'uacute', 'ugrave', 'ucircumflex', 'udieresis', 'dagger',
                 'degree', 'cent', 'sterling', 'section', 'bullet', 'paragraph',
                 'germandbls', 'registered', 'copyright', 'trademark', 'acute',
                 'dieresis', None, 'AE', 'Oslash', None, 'plusminus', None, None, 'yen',
                 'mu', None, None, None, None, None, 'ordfeminine', 'ordmasculine', None,
                 'ae', 'oslash', 'questiondown', 'exclamdown', 'logicalnot', None, 'florin',
                 None, None, 'guillemotleft', 'guillemotright', 'ellipsis', 'space', 'Agrave',
                 'Atilde', 'Otilde', 'OE', 'oe', 'endash', 'emdash', 'quotedblleft',
                 'quotedblright', 'quoteleft', 'quoteright', 'divide', None, 'ydieresis',
                 'Ydieresis', 'fraction', 'currency', 'guilsinglleft', 'guilsinglright',
                 'fi', 'fl', 'daggerdbl', 'periodcentered', 'quotesinglbase',
                 'quotedblbase', 'perthousand', 'Acircumflex', 'Ecircumflex', 'Aacute',
                 'Edieresis', 'Egrave', 'Iacute', 'Icircumflex', 'Idieresis', 'Igrave',
                 'Oacute', 'Ocircumflex', None, 'Ograve', 'Uacute', 'Ucircumflex',
                 'Ugrave', 'dotlessi', 'circumflex', 'tilde', 'macron', 'breve',
                 'dotaccent', 'ring', 'cedilla', 'hungarumlaut', 'ogonek', 'caron')
encodings['SymbolEncoding']=(None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None,
                    None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, 'space',
                    'exclam', 'universal', 'numbersign', 'existential', 'percent', 'ampersand', 'suchthat',
                    'parenleft', 'parenright', 'asteriskmath', 'plus', 'comma', 'minus', 'period', 'slash', 'zero',
                    'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon',
                    'less', 'equal', 'greater', 'question', 'congruent', 'Alpha', 'Beta', 'Chi', 'Delta', 'Epsilon',
                    'Phi', 'Gamma', 'Eta', 'Iota', 'theta1', 'Kappa', 'Lambda', 'Mu', 'Nu', 'Omicron', 'Pi', 'Theta',
                    'Rho', 'Sigma', 'Tau', 'Upsilon', 'sigma1', 'Omega', 'Xi', 'Psi', 'Zeta', 'bracketleft',
                    'therefore', 'bracketright', 'perpendicular', 'underscore', 'radicalex', 'alpha', 'beta', 'chi',
                    'delta', 'epsilon', 'phi', 'gamma', 'eta', 'iota', 'phi1', 'kappa', 'lambda', 'mu', 'nu',
                    'omicron', 'pi', 'theta', 'rho', 'sigma', 'tau', 'upsilon', 'omega1', 'omega', 'xi', 'psi', 'zeta',
                    'braceleft', 'bar', 'braceright', 'similar', None, None, None, None, None, None, None, None, None,
                    None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None,
                    None, None, None, None, None, None, None, None, 'Euro', 'Upsilon1', 'minute', 'lessequal',
                    'fraction', 'infinity', 'florin', 'club', 'diamond', 'heart', 'spade', 'arrowboth', 'arrowleft',
                    'arrowup', 'arrowright', 'arrowdown', 'degree', 'plusminus', 'second', 'greaterequal', 'multiply',
                    'proportional', 'partialdiff', 'bullet', 'divide', 'notequal', 'equivalence', 'approxequal',
                    'ellipsis', 'arrowvertex', 'arrowhorizex', 'carriagereturn', 'aleph', 'Ifraktur', 'Rfraktur',
                    'weierstrass', 'circlemultiply', 'circleplus', 'emptyset', 'intersection', 'union',
                    'propersuperset', 'reflexsuperset', 'notsubset', 'propersubset', 'reflexsubset', 'element',
                    'notelement', 'angle', 'gradient', 'registerserif', 'copyrightserif', 'trademarkserif', 'product',
                    'radical', 'dotmath', 'logicalnot', 'logicaland', 'logicalor', 'arrowdblboth', 'arrowdblleft',
                    'arrowdblup', 'arrowdblright', 'arrowdbldown', 'lozenge', 'angleleft', 'registersans',
                    'copyrightsans', 'trademarksans', 'summation', 'parenlefttp', 'parenleftex', 'parenleftbt',
                    'bracketlefttp', 'bracketleftex', 'bracketleftbt', 'bracelefttp', 'braceleftmid', 'braceleftbt',
                    'braceex', None, 'angleright', 'integral', 'integraltp', 'integralex', 'integralbt',
                    'parenrighttp', 'parenrightex', 'parenrightbt', 'bracketrighttp', 'bracketrightex',
                    'bracketrightbt', 'bracerighttp', 'bracerightmid', 'bracerightbt', None)
encodings['ZapfDingbatsEncoding'] = (   None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None,
                    None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None,
                    'space', 'a1', 'a2', 'a202', 'a3', 'a4', 'a5', 'a119', 'a118', 'a117', 'a11', 'a12', 'a13', 'a14',
                    'a15', 'a16', 'a105', 'a17', 'a18', 'a19', 'a20', 'a21', 'a22', 'a23', 'a24', 'a25', 'a26', 'a27',
                    'a28', 'a6', 'a7', 'a8', 'a9', 'a10', 'a29', 'a30', 'a31', 'a32', 'a33', 'a34', 'a35', 'a36',
                    'a37', 'a38', 'a39', 'a40', 'a41', 'a42', 'a43', 'a44', 'a45', 'a46', 'a47', 'a48', 'a49', 'a50',
                    'a51', 'a52', 'a53', 'a54', 'a55', 'a56', 'a57', 'a58', 'a59', 'a60', 'a61', 'a62', 'a63', 'a64',
                    'a65', 'a66', 'a67', 'a68', 'a69', 'a70', 'a71', 'a72', 'a73', 'a74', 'a203', 'a75', 'a204', 'a76',
                    'a77', 'a78', 'a79', 'a81', 'a82', 'a83', 'a84', 'a97', 'a98', 'a99', 'a100', None, 'a89', 'a90',
                    'a93', 'a94', 'a91', 'a92', 'a205', 'a85', 'a206', 'a86', 'a87', 'a88', 'a95', 'a96', None, None,
                    None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None,
                    None, 'a101', 'a102', 'a103', 'a104', 'a106', 'a107', 'a108', 'a112', 'a111', 'a110', 'a109',
                    'a120', 'a121', 'a122', 'a123', 'a124', 'a125', 'a126', 'a127', 'a128', 'a129', 'a130', 'a131',
                    'a132', 'a133', 'a134', 'a135', 'a136', 'a137', 'a138', 'a139', 'a140', 'a141', 'a142', 'a143',
                    'a144', 'a145', 'a146', 'a147', 'a148', 'a149', 'a150', 'a151', 'a152', 'a153', 'a154', 'a155',
                    'a156', 'a157', 'a158', 'a159', 'a160', 'a161', 'a163', 'a164', 'a196', 'a165', 'a192', 'a166',
                    'a167', 'a168', 'a169', 'a170', 'a171', 'a172', 'a173', 'a162', 'a174', 'a175', 'a176', 'a177',
                    'a178', 'a179', 'a193', 'a180', 'a199', 'a181', 'a200', 'a182', None, 'a201', 'a183', 'a184',
                    'a197', 'a185', 'a194', 'a198', 'a186', 'a195', 'a187', 'a188', 'a189', 'a190', 'a191', None)
encodings['StandardEncoding']=(None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,"space","exclam",
    "quotedbl","numbersign","dollar","percent","ampersand","quoteright","parenleft","parenright","asterisk","plus",
    "comma","hyphen","period","slash","zero","one","two","three","four","five","six","seven","eight","nine","colon",
    "semicolon","less","equal","greater","question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O",
    "P","Q","R","S","T","U","V","W","X","Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore",
    "quoteleft","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y",
    "z","braceleft","bar","braceright","asciitilde",None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,
    None,None,None,"exclamdown","cent","sterling","fraction","yen","florin","section","currency","quotesingle","quotedblleft",
    "guillemotleft","guilsinglleft","guilsinglright","fi","fl",None,"endash","dagger","daggerdbl","periodcentered",None,
    "paragraph","bullet","quotesinglbase","quotedblbase","quotedblright","guillemotright","ellipsis","perthousand",
    None,"questiondown",None,"grave","acute","circumflex","tilde","macron","breve","dotaccent","dieresis",None,"ring",
    "cedilla",None,"hungarumlaut","ogonek","caron","emdash",None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,"AE",None,"ordfeminine",
    None,None,None,None,"Lslash","Oslash","OE","ordmasculine",None,None,None,None,None,"ae",None,None,None,"dotlessi",None,None,"lslash","oslash",
    "oe","germandbls",None,None,None,None)
encodings['PDFDocEncoding']=(None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,None,
    None,None,None,None,None,"breve","caron","circumflex",
    "dotaccent","hungarumlaut","ogonek","ring","tilde","space","exclam","quotedbl","numbersign","dollar","percent",
    "ampersand","quotesingle","parenleft","parenright","asterisk","plus","comma","hyphen","period","slash","zero",
    "one","two","three","four","five","six","seven","eight","nine","colon","semicolon","less","equal","greater",
    "question","at","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X",
    "Y","Z","bracketleft","backslash","bracketright","asciicircum","underscore","grave","a","b","c","d","e","f","g",
    "h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","braceleft","bar","braceright",
    "asciitilde",None,"bullet","dagger","daggerdbl","ellipsis","emdash","endash","florin","fraction","guilsinglleft",
    "guilsinglright","minus","perthousand","quotedblbase","quotedblleft","quotedblright","quoteleft","quoteright",
    "quotesinglbase","trademark","fi","fl","Lslash","OE","Scaron","Ydieresis","Zcaron","dotlessi","lslash","oe",
    "scaron","zcaron",None,"Euro","exclamdown","cent","sterling","currency","yen","brokenbar","section","dieresis",
    "copyright","ordfeminine","guillemotleft","logicalnot",None,"registered","macron","degree","plusminus","twosuperior",
    "threesuperior","acute","mu","paragraph","periodcentered","cedilla","onesuperior","ordmasculine","guillemotright",
    "onequarter","onehalf","threequarters","questiondown","Agrave","Aacute","Acircumflex","Atilde","Adieresis","Aring",
    "AE","Ccedilla","Egrave","Eacute","Ecircumflex","Edieresis","Igrave","Iacute","Icircumflex","Idieresis","Eth",
    "Ntilde","Ograve","Oacute","Ocircumflex","Otilde","Odieresis","multiply","Oslash","Ugrave","Uacute","Ucircumflex",
    "Udieresis","Yacute","Thorn","germandbls","agrave","aacute","acircumflex","atilde","adieresis","aring","ae",
    "ccedilla","egrave","eacute","ecircumflex","edieresis","igrave","iacute","icircumflex","idieresis","eth","ntilde",
    "ograve","oacute","ocircumflex","otilde","odieresis","divide","oslash","ugrave","uacute","ucircumflex","udieresis",
    "yacute","thorn","ydieresis")
encodings['MacExpertEncoding'] =  (None, None, None, None, None, None, None, None, None, None, None, None, None, None,
    None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None,
    'space', 'exclamsmall', 'Hungarumlautsmall', 'centoldstyle', 'dollaroldstyle', 'dollarsuperior', 'ampersandsmall',
    'Acutesmall', 'parenleftsuperior', 'parenrightsuperior', 'twodotenleader', 'onedotenleader', 'comma', 'hyphen',
    'period', 'fraction', 'zerooldstyle', 'oneoldstyle', 'twooldstyle', 'threeoldstyle', 'fouroldstyle',
    'fiveoldstyle', 'sixoldstyle', 'sevenoldstyle', 'eightoldstyle', 'nineoldstyle', 'colon', 'semicolon', None,
    'threequartersemdash', None, 'questionsmall', None, None, None, None, 'Ethsmall', None, None, 'onequarter',
    'onehalf', 'threequarters', 'oneeighth', 'threeeighths', 'fiveeighths', 'seveneighths', 'onethird', 'twothirds',
    None, None, None, None, None, None, 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior', None,
    'parenrightinferior', 'Circumflexsmall', 'hypheninferior', 'Gravesmall', 'Asmall', 'Bsmall', 'Csmall', 'Dsmall',
    'Esmall', 'Fsmall', 'Gsmall', 'Hsmall', 'Ismall', 'Jsmall', 'Ksmall', 'Lsmall', 'Msmall', 'Nsmall', 'Osmall',
    'Psmall', 'Qsmall', 'Rsmall', 'Ssmall', 'Tsmall', 'Usmall', 'Vsmall', 'Wsmall', 'Xsmall', 'Ysmall', 'Zsmall',
    'colonmonetary', 'onefitted', 'rupiah', 'Tildesmall', None, None, 'asuperior', 'centsuperior', None, None, None,
    None, 'Aacutesmall', 'Agravesmall', 'Acircumflexsmall', 'Adieresissmall', 'Atildesmall', 'Aringsmall',
    'Ccedillasmall', 'Eacutesmall', 'Egravesmall', 'Ecircumflexsmall', 'Edieresissmall', 'Iacutesmall', 'Igravesmall',
    'Icircumflexsmall', 'Idieresissmall', 'Ntildesmall', 'Oacutesmall', 'Ogravesmall', 'Ocircumflexsmall',
    'Odieresissmall', 'Otildesmall', 'Uacutesmall', 'Ugravesmall', 'Ucircumflexsmall', 'Udieresissmall', None,
    'eightsuperior', 'fourinferior', 'threeinferior', 'sixinferior', 'eightinferior', 'seveninferior', 'Scaronsmall',
    None, 'centinferior', 'twoinferior', None, 'Dieresissmall', None, 'Caronsmall', 'osuperior', 'fiveinferior', None,
    'commainferior', 'periodinferior', 'Yacutesmall', None, 'dollarinferior', None, None, 'Thornsmall', None,
    'nineinferior', 'zeroinferior', 'Zcaronsmall', 'AEsmall', 'Oslashsmall', 'questiondownsmall', 'oneinferior',
    'Lslashsmall', None, None, None, None, None, None, 'Cedillasmall', None, None, None, None, None, 'OEsmall',
    'figuredash', 'hyphensuperior', None, None, None, None, 'exclamdownsmall', None, 'Ydieresissmall', None,
    'onesuperior', 'twosuperior', 'threesuperior', 'foursuperior', 'fivesuperior', 'sixsuperior', 'sevensuperior',
    'ninesuperior', 'zerosuperior', None, 'esuperior', 'rsuperior', 'tsuperior', None, None, 'isuperior', 'ssuperior',
    'dsuperior', None, None, None, None, None, 'lsuperior', 'Ogoneksmall', 'Brevesmall', 'Macronsmall', 'bsuperior',
    'nsuperior', 'msuperior', 'commasuperior', 'periodsuperior', 'Dotaccentsmall', 'Ringsmall', None, None, None, None)
ascent_descent = {
    'Courier': (629, -157),
    'Courier-Bold': (626, -142),
    'Courier-BoldOblique': (626, -142),
    'Courier-Oblique': (629, -157),
    'Helvetica': (718, -207),
    'Helvetica-Bold': (718, -207),
    'Helvetica-BoldOblique': (718, -207),
    'Helvetica-Oblique': (718, -207),
    'Times-Roman': (683, -217),
    'Times-Bold': (676, -205),
    'Times-BoldItalic': (699, -205),
    'Times-Italic': (683, -205),
    'Symbol': (0, 0),
    'ZapfDingbats': (0, 0)
    }

# nuild this up one entry at a time to stay under JPython's 64k limit.
widthsByFontGlyph = {}
widthsByFontGlyph['Helvetica'] = {'A': 667,
 'AE': 1000,
 'Aacute': 667,
 'Acircumflex': 667,
 'Adieresis': 667,
 'Agrave': 667,
 'Aring': 667,
 'Atilde': 667,
 'B': 667,
 'C': 722,
 'Ccedilla': 722,
 'D': 722,
 'E': 667,
 'Eacute': 667,
 'Ecircumflex': 667,
 'Edieresis': 667,
 'Egrave': 667,
 'Eth': 722,
 'Euro': 556,
 'F': 611,
 'G': 778,
 'H': 722,
 'I': 278,
 'Iacute': 278,
 'Icircumflex': 278,
 'Idieresis': 278,
 'Igrave': 278,
 'J': 500,
 'K': 667,
 'L': 556,
 'Lslash': 556,
 'M': 833,
 'N': 722,
 'Ntilde': 722,
 'O': 778,
 'OE': 1000,
 'Oacute': 778,
 'Ocircumflex': 778,
 'Odieresis': 778,
 'Ograve': 778,
 'Oslash': 778,
 'Otilde': 778,
 'P': 667,
 'Q': 778,
 'R': 722,
 'S': 667,
 'Scaron': 667,
 'T': 611,
 'Thorn': 667,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 667,
 'W': 944,
 'X': 667,
 'Y': 667,
 'Yacute': 667,
 'Ydieresis': 667,
 'Z': 611,
 'Zcaron': 611,
 'a': 556,
 'aacute': 556,
 'acircumflex': 556,
 'acute': 333,
 'adieresis': 556,
 'ae': 889,
 'agrave': 556,
 'ampersand': 667,
 'aring': 556,
 'asciicircum': 469,
 'asciitilde': 584,
 'asterisk': 389,
 'at': 1015,
 'atilde': 556,
 'b': 556,
 'backslash': 278,
 'bar': 260,
 'braceleft': 334,
 'braceright': 334,
 'bracketleft': 278,
 'bracketright': 278,
 'breve': 333,
 'brokenbar': 260,
 'bullet': 350,
 'c': 500,
 'caron': 333,
 'ccedilla': 500,
 'cedilla': 333,
 'cent': 556,
 'circumflex': 333,
 'colon': 278,
 'comma': 278,
 'copyright': 737,
 'currency': 556,
 'd': 556,
 'dagger': 556,
 'daggerdbl': 556,
 'degree': 400,
 'dieresis': 333,
 'divide': 584,
 'dollar': 556,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 556,
 'eacute': 556,
 'ecircumflex': 556,
 'edieresis': 556,
 'egrave': 556,
 'eight': 556,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 556,
 'equal': 584,
 'eth': 556,
 'exclam': 278,
 'exclamdown': 333,
 'f': 278,
 'fi': 500,
 'five': 556,
 'fl': 500,
 'florin': 556,
 'four': 556,
 'fraction': 167,
 'g': 556,
 'germandbls': 611,
 'grave': 333,
 'greater': 584,
 'guillemotleft': 556,
 'guillemotright': 556,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 556,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 222,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 222,
 'k': 500,
 'l': 222,
 'less': 584,
 'logicalnot': 584,
 'lslash': 222,
 'm': 833,
 'macron': 333,
 'minus': 584,
 'mu': 556,
 'multiply': 584,
 'n': 556,
 'nine': 556,
 'ntilde': 556,
 'numbersign': 556,
 'o': 556,
 'oacute': 556,
 'ocircumflex': 556,
 'odieresis': 556,
 'oe': 944,
 'ogonek': 333,
 'ograve': 556,
 'one': 556,
 'onehalf': 834,
 'onequarter': 834,
 'onesuperior': 333,
 'ordfeminine': 370,
 'ordmasculine': 365,
 'oslash': 611,
 'otilde': 556,
 'p': 556,
 'paragraph': 537,
 'parenleft': 333,
 'parenright': 333,
 'percent': 889,
 'period': 278,
 'periodcentered': 278,
 'perthousand': 1000,
 'plus': 584,
 'plusminus': 584,
 'q': 556,
 'question': 556,
 'questiondown': 611,
 'quotedbl': 355,
 'quotedblbase': 333,
 'quotedblleft': 333,
 'quotedblright': 333,
 'quoteleft': 222,
 'quoteright': 222,
 'quotesinglbase': 222,
 'quotesingle': 191,
 'r': 333,
 'registered': 737,
 'ring': 333,
 's': 500,
 'scaron': 500,
 'section': 556,
 'semicolon': 278,
 'seven': 556,
 'six': 556,
 'slash': 278,
 'space': 278,
 'sterling': 556,
 't': 278,
 'thorn': 556,
 'three': 556,
 'threequarters': 834,
 'threesuperior': 333,
 'tilde': 333,
 'trademark': 1000,
 'two': 556,
 'twosuperior': 333,
 'u': 556,
 'uacute': 556,
 'ucircumflex': 556,
 'udieresis': 556,
 'ugrave': 556,
 'underscore': 556,
 'v': 500,
 'w': 722,
 'x': 500,
 'y': 500,
 'yacute': 500,
 'ydieresis': 500,
 'yen': 556,
 'z': 500,
 'zcaron': 500,
 'zero': 556}

widthsByFontGlyph['Helvetica-Bold'] = {'A': 722,
 'AE': 1000,
 'Aacute': 722,
 'Acircumflex': 722,
 'Adieresis': 722,
 'Agrave': 722,
 'Aring': 722,
 'Atilde': 722,
 'B': 722,
 'C': 722,
 'Ccedilla': 722,
 'D': 722,
 'E': 667,
 'Eacute': 667,
 'Ecircumflex': 667,
 'Edieresis': 667,
 'Egrave': 667,
 'Eth': 722,
 'Euro': 556,
 'F': 611,
 'G': 778,
 'H': 722,
 'I': 278,
 'Iacute': 278,
 'Icircumflex': 278,
 'Idieresis': 278,
 'Igrave': 278,
 'J': 556,
 'K': 722,
 'L': 611,
 'Lslash': 611,
 'M': 833,
 'N': 722,
 'Ntilde': 722,
 'O': 778,
 'OE': 1000,
 'Oacute': 778,
 'Ocircumflex': 778,
 'Odieresis': 778,
 'Ograve': 778,
 'Oslash': 778,
 'Otilde': 778,
 'P': 667,
 'Q': 778,
 'R': 722,
 'S': 667,
 'Scaron': 667,
 'T': 611,
 'Thorn': 667,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 667,
 'W': 944,
 'X': 667,
 'Y': 667,
 'Yacute': 667,
 'Ydieresis': 667,
 'Z': 611,
 'Zcaron': 611,
 'a': 556,
 'aacute': 556,
 'acircumflex': 556,
 'acute': 333,
 'adieresis': 556,
 'ae': 889,
 'agrave': 556,
 'ampersand': 722,
 'aring': 556,
 'asciicircum': 584,
 'asciitilde': 584,
 'asterisk': 389,
 'at': 975,
 'atilde': 556,
 'b': 611,
 'backslash': 278,
 'bar': 280,
 'braceleft': 389,
 'braceright': 389,
 'bracketleft': 333,
 'bracketright': 333,
 'breve': 333,
 'brokenbar': 280,
 'bullet': 350,
 'c': 556,
 'caron': 333,
 'ccedilla': 556,
 'cedilla': 333,
 'cent': 556,
 'circumflex': 333,
 'colon': 333,
 'comma': 278,
 'copyright': 737,
 'currency': 556,
 'd': 611,
 'dagger': 556,
 'daggerdbl': 556,
 'degree': 400,
 'dieresis': 333,
 'divide': 584,
 'dollar': 556,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 556,
 'eacute': 556,
 'ecircumflex': 556,
 'edieresis': 556,
 'egrave': 556,
 'eight': 556,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 556,
 'equal': 584,
 'eth': 611,
 'exclam': 333,
 'exclamdown': 333,
 'f': 333,
 'fi': 611,
 'five': 556,
 'fl': 611,
 'florin': 556,
 'four': 556,
 'fraction': 167,
 'g': 611,
 'germandbls': 611,
 'grave': 333,
 'greater': 584,
 'guillemotleft': 556,
 'guillemotright': 556,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 611,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 278,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 278,
 'k': 556,
 'l': 278,
 'less': 584,
 'logicalnot': 584,
 'lslash': 278,
 'm': 889,
 'macron': 333,
 'minus': 584,
 'mu': 611,
 'multiply': 584,
 'n': 611,
 'nine': 556,
 'ntilde': 611,
 'numbersign': 556,
 'o': 611,
 'oacute': 611,
 'ocircumflex': 611,
 'odieresis': 611,
 'oe': 944,
 'ogonek': 333,
 'ograve': 611,
 'one': 556,
 'onehalf': 834,
 'onequarter': 834,
 'onesuperior': 333,
 'ordfeminine': 370,
 'ordmasculine': 365,
 'oslash': 611,
 'otilde': 611,
 'p': 611,
 'paragraph': 556,
 'parenleft': 333,
 'parenright': 333,
 'percent': 889,
 'period': 278,
 'periodcentered': 278,
 'perthousand': 1000,
 'plus': 584,
 'plusminus': 584,
 'q': 611,
 'question': 611,
 'questiondown': 611,
 'quotedbl': 474,
 'quotedblbase': 500,
 'quotedblleft': 500,
 'quotedblright': 500,
 'quoteleft': 278,
 'quoteright': 278,
 'quotesinglbase': 278,
 'quotesingle': 238,
 'r': 389,
 'registered': 737,
 'ring': 333,
 's': 556,
 'scaron': 556,
 'section': 556,
 'semicolon': 333,
 'seven': 556,
 'six': 556,
 'slash': 278,
 'space': 278,
 'sterling': 556,
 't': 333,
 'thorn': 611,
 'three': 556,
 'threequarters': 834,
 'threesuperior': 333,
 'tilde': 333,
 'trademark': 1000,
 'two': 556,
 'twosuperior': 333,
 'u': 611,
 'uacute': 611,
 'ucircumflex': 611,
 'udieresis': 611,
 'ugrave': 611,
 'underscore': 556,
 'v': 556,
 'w': 778,
 'x': 556,
 'y': 556,
 'yacute': 556,
 'ydieresis': 556,
 'yen': 556,
 'z': 500,
 'zcaron': 500,
 'zero': 556}

widthsByFontGlyph['Helvetica-Oblique'] = {'A': 667,
 'AE': 1000,
 'Aacute': 667,
 'Acircumflex': 667,
 'Adieresis': 667,
 'Agrave': 667,
 'Aring': 667,
 'Atilde': 667,
 'B': 667,
 'C': 722,
 'Ccedilla': 722,
 'D': 722,
 'E': 667,
 'Eacute': 667,
 'Ecircumflex': 667,
 'Edieresis': 667,
 'Egrave': 667,
 'Eth': 722,
 'Euro': 556,
 'F': 611,
 'G': 778,
 'H': 722,
 'I': 278,
 'Iacute': 278,
 'Icircumflex': 278,
 'Idieresis': 278,
 'Igrave': 278,
 'J': 500,
 'K': 667,
 'L': 556,
 'Lslash': 556,
 'M': 833,
 'N': 722,
 'Ntilde': 722,
 'O': 778,
 'OE': 1000,
 'Oacute': 778,
 'Ocircumflex': 778,
 'Odieresis': 778,
 'Ograve': 778,
 'Oslash': 778,
 'Otilde': 778,
 'P': 667,
 'Q': 778,
 'R': 722,
 'S': 667,
 'Scaron': 667,
 'T': 611,
 'Thorn': 667,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 667,
 'W': 944,
 'X': 667,
 'Y': 667,
 'Yacute': 667,
 'Ydieresis': 667,
 'Z': 611,
 'Zcaron': 611,
 'a': 556,
 'aacute': 556,
 'acircumflex': 556,
 'acute': 333,
 'adieresis': 556,
 'ae': 889,
 'agrave': 556,
 'ampersand': 667,
 'aring': 556,
 'asciicircum': 469,
 'asciitilde': 584,
 'asterisk': 389,
 'at': 1015,
 'atilde': 556,
 'b': 556,
 'backslash': 278,
 'bar': 260,
 'braceleft': 334,
 'braceright': 334,
 'bracketleft': 278,
 'bracketright': 278,
 'breve': 333,
 'brokenbar': 260,
 'bullet': 350,
 'c': 500,
 'caron': 333,
 'ccedilla': 500,
 'cedilla': 333,
 'cent': 556,
 'circumflex': 333,
 'colon': 278,
 'comma': 278,
 'copyright': 737,
 'currency': 556,
 'd': 556,
 'dagger': 556,
 'daggerdbl': 556,
 'degree': 400,
 'dieresis': 333,
 'divide': 584,
 'dollar': 556,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 556,
 'eacute': 556,
 'ecircumflex': 556,
 'edieresis': 556,
 'egrave': 556,
 'eight': 556,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 556,
 'equal': 584,
 'eth': 556,
 'exclam': 278,
 'exclamdown': 333,
 'f': 278,
 'fi': 500,
 'five': 556,
 'fl': 500,
 'florin': 556,
 'four': 556,
 'fraction': 167,
 'g': 556,
 'germandbls': 611,
 'grave': 333,
 'greater': 584,
 'guillemotleft': 556,
 'guillemotright': 556,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 556,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 222,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 222,
 'k': 500,
 'l': 222,
 'less': 584,
 'logicalnot': 584,
 'lslash': 222,
 'm': 833,
 'macron': 333,
 'minus': 584,
 'mu': 556,
 'multiply': 584,
 'n': 556,
 'nine': 556,
 'ntilde': 556,
 'numbersign': 556,
 'o': 556,
 'oacute': 556,
 'ocircumflex': 556,
 'odieresis': 556,
 'oe': 944,
 'ogonek': 333,
 'ograve': 556,
 'one': 556,
 'onehalf': 834,
 'onequarter': 834,
 'onesuperior': 333,
 'ordfeminine': 370,
 'ordmasculine': 365,
 'oslash': 611,
 'otilde': 556,
 'p': 556,
 'paragraph': 537,
 'parenleft': 333,
 'parenright': 333,
 'percent': 889,
 'period': 278,
 'periodcentered': 278,
 'perthousand': 1000,
 'plus': 584,
 'plusminus': 584,
 'q': 556,
 'question': 556,
 'questiondown': 611,
 'quotedbl': 355,
 'quotedblbase': 333,
 'quotedblleft': 333,
 'quotedblright': 333,
 'quoteleft': 222,
 'quoteright': 222,
 'quotesinglbase': 222,
 'quotesingle': 191,
 'r': 333,
 'registered': 737,
 'ring': 333,
 's': 500,
 'scaron': 500,
 'section': 556,
 'semicolon': 278,
 'seven': 556,
 'six': 556,
 'slash': 278,
 'space': 278,
 'sterling': 556,
 't': 278,
 'thorn': 556,
 'three': 556,
 'threequarters': 834,
 'threesuperior': 333,
 'tilde': 333,
 'trademark': 1000,
 'two': 556,
 'twosuperior': 333,
 'u': 556,
 'uacute': 556,
 'ucircumflex': 556,
 'udieresis': 556,
 'ugrave': 556,
 'underscore': 556,
 'v': 500,
 'w': 722,
 'x': 500,
 'y': 500,
 'yacute': 500,
 'ydieresis': 500,
 'yen': 556,
 'z': 500,
 'zcaron': 500,
 'zero': 556}



widthsByFontGlyph['Helvetica-BoldOblique'] = {'A': 722,
 'AE': 1000,
 'Aacute': 722,
 'Acircumflex': 722,
 'Adieresis': 722,
 'Agrave': 722,
 'Aring': 722,
 'Atilde': 722,
 'B': 722,
 'C': 722,
 'Ccedilla': 722,
 'D': 722,
 'E': 667,
 'Eacute': 667,
 'Ecircumflex': 667,
 'Edieresis': 667,
 'Egrave': 667,
 'Eth': 722,
 'Euro': 556,
 'F': 611,
 'G': 778,
 'H': 722,
 'I': 278,
 'Iacute': 278,
 'Icircumflex': 278,
 'Idieresis': 278,
 'Igrave': 278,
 'J': 556,
 'K': 722,
 'L': 611,
 'Lslash': 611,
 'M': 833,
 'N': 722,
 'Ntilde': 722,
 'O': 778,
 'OE': 1000,
 'Oacute': 778,
 'Ocircumflex': 778,
 'Odieresis': 778,
 'Ograve': 778,
 'Oslash': 778,
 'Otilde': 778,
 'P': 667,
 'Q': 778,
 'R': 722,
 'S': 667,
 'Scaron': 667,
 'T': 611,
 'Thorn': 667,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 667,
 'W': 944,
 'X': 667,
 'Y': 667,
 'Yacute': 667,
 'Ydieresis': 667,
 'Z': 611,
 'Zcaron': 611,
 'a': 556,
 'aacute': 556,
 'acircumflex': 556,
 'acute': 333,
 'adieresis': 556,
 'ae': 889,
 'agrave': 556,
 'ampersand': 722,
 'aring': 556,
 'asciicircum': 584,
 'asciitilde': 584,
 'asterisk': 389,
 'at': 975,
 'atilde': 556,
 'b': 611,
 'backslash': 278,
 'bar': 280,
 'braceleft': 389,
 'braceright': 389,
 'bracketleft': 333,
 'bracketright': 333,
 'breve': 333,
 'brokenbar': 280,
 'bullet': 350,
 'c': 556,
 'caron': 333,
 'ccedilla': 556,
 'cedilla': 333,
 'cent': 556,
 'circumflex': 333,
 'colon': 333,
 'comma': 278,
 'copyright': 737,
 'currency': 556,
 'd': 611,
 'dagger': 556,
 'daggerdbl': 556,
 'degree': 400,
 'dieresis': 333,
 'divide': 584,
 'dollar': 556,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 556,
 'eacute': 556,
 'ecircumflex': 556,
 'edieresis': 556,
 'egrave': 556,
 'eight': 556,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 556,
 'equal': 584,
 'eth': 611,
 'exclam': 333,
 'exclamdown': 333,
 'f': 333,
 'fi': 611,
 'five': 556,
 'fl': 611,
 'florin': 556,
 'four': 556,
 'fraction': 167,
 'g': 611,
 'germandbls': 611,
 'grave': 333,
 'greater': 584,
 'guillemotleft': 556,
 'guillemotright': 556,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 611,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 278,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 278,
 'k': 556,
 'l': 278,
 'less': 584,
 'logicalnot': 584,
 'lslash': 278,
 'm': 889,
 'macron': 333,
 'minus': 584,
 'mu': 611,
 'multiply': 584,
 'n': 611,
 'nine': 556,
 'ntilde': 611,
 'numbersign': 556,
 'o': 611,
 'oacute': 611,
 'ocircumflex': 611,
 'odieresis': 611,
 'oe': 944,
 'ogonek': 333,
 'ograve': 611,
 'one': 556,
 'onehalf': 834,
 'onequarter': 834,
 'onesuperior': 333,
 'ordfeminine': 370,
 'ordmasculine': 365,
 'oslash': 611,
 'otilde': 611,
 'p': 611,
 'paragraph': 556,
 'parenleft': 333,
 'parenright': 333,
 'percent': 889,
 'period': 278,
 'periodcentered': 278,
 'perthousand': 1000,
 'plus': 584,
 'plusminus': 584,
 'q': 611,
 'question': 611,
 'questiondown': 611,
 'quotedbl': 474,
 'quotedblbase': 500,
 'quotedblleft': 500,
 'quotedblright': 500,
 'quoteleft': 278,
 'quoteright': 278,
 'quotesinglbase': 278,
 'quotesingle': 238,
 'r': 389,
 'registered': 737,
 'ring': 333,
 's': 556,
 'scaron': 556,
 'section': 556,
 'semicolon': 333,
 'seven': 556,
 'six': 556,
 'slash': 278,
 'space': 278,
 'sterling': 556,
 't': 333,
 'thorn': 611,
 'three': 556,
 'threequarters': 834,
 'threesuperior': 333,
 'tilde': 333,
 'trademark': 1000,
 'two': 556,
 'twosuperior': 333,
 'u': 611,
 'uacute': 611,
 'ucircumflex': 611,
 'udieresis': 611,
 'ugrave': 611,
 'underscore': 556,
 'v': 556,
 'w': 778,
 'x': 556,
 'y': 556,
 'yacute': 556,
 'ydieresis': 556,
 'yen': 556,
 'z': 500,
 'zcaron': 500,
 'zero': 556}

# Courier can be expressed more compactly!
_w = {}
for charname in widthsByFontGlyph['Helvetica'].keys():
    _w[charname] = 600
widthsByFontGlyph['Courier'] = _w
widthsByFontGlyph['Courier-Bold'] = _w
widthsByFontGlyph['Courier-Oblique'] = _w
widthsByFontGlyph['Courier-BoldOblique'] = _w

widthsByFontGlyph['Times-Roman'] = {'A': 722,
 'AE': 889,
 'Aacute': 722,
 'Acircumflex': 722,
 'Adieresis': 722,
 'Agrave': 722,
 'Aring': 722,
 'Atilde': 722,
 'B': 667,
 'C': 667,
 'Ccedilla': 667,
 'D': 722,
 'E': 611,
 'Eacute': 611,
 'Ecircumflex': 611,
 'Edieresis': 611,
 'Egrave': 611,
 'Eth': 722,
 'Euro': 500,
 'F': 556,
 'G': 722,
 'H': 722,
 'I': 333,
 'Iacute': 333,
 'Icircumflex': 333,
 'Idieresis': 333,
 'Igrave': 333,
 'J': 389,
 'K': 722,
 'L': 611,
 'Lslash': 611,
 'M': 889,
 'N': 722,
 'Ntilde': 722,
 'O': 722,
 'OE': 889,
 'Oacute': 722,
 'Ocircumflex': 722,
 'Odieresis': 722,
 'Ograve': 722,
 'Oslash': 722,
 'Otilde': 722,
 'P': 556,
 'Q': 722,
 'R': 667,
 'S': 556,
 'Scaron': 556,
 'T': 611,
 'Thorn': 556,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 722,
 'W': 944,
 'X': 722,
 'Y': 722,
 'Yacute': 722,
 'Ydieresis': 722,
 'Z': 611,
 'Zcaron': 611,
 'a': 444,
 'aacute': 444,
 'acircumflex': 444,
 'acute': 333,
 'adieresis': 444,
 'ae': 667,
 'agrave': 444,
 'ampersand': 778,
 'aring': 444,
 'asciicircum': 469,
 'asciitilde': 541,
 'asterisk': 500,
 'at': 921,
 'atilde': 444,
 'b': 500,
 'backslash': 278,
 'bar': 200,
 'braceleft': 480,
 'braceright': 480,
 'bracketleft': 333,
 'bracketright': 333,
 'breve': 333,
 'brokenbar': 200,
 'bullet': 350,
 'c': 444,
 'caron': 333,
 'ccedilla': 444,
 'cedilla': 333,
 'cent': 500,
 'circumflex': 333,
 'colon': 278,
 'comma': 250,
 'copyright': 760,
 'currency': 500,
 'd': 500,
 'dagger': 500,
 'daggerdbl': 500,
 'degree': 400,
 'dieresis': 333,
 'divide': 564,
 'dollar': 500,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 444,
 'eacute': 444,
 'ecircumflex': 444,
 'edieresis': 444,
 'egrave': 444,
 'eight': 500,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 500,
 'equal': 564,
 'eth': 500,
 'exclam': 333,
 'exclamdown': 333,
 'f': 333,
 'fi': 556,
 'five': 500,
 'fl': 556,
 'florin': 500,
 'four': 500,
 'fraction': 167,
 'g': 500,
 'germandbls': 500,
 'grave': 333,
 'greater': 564,
 'guillemotleft': 500,
 'guillemotright': 500,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 500,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 278,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 278,
 'k': 500,
 'l': 278,
 'less': 564,
 'logicalnot': 564,
 'lslash': 278,
 'm': 778,
 'macron': 333,
 'minus': 564,
 'mu': 500,
 'multiply': 564,
 'n': 500,
 'nine': 500,
 'ntilde': 500,
 'numbersign': 500,
 'o': 500,
 'oacute': 500,
 'ocircumflex': 500,
 'odieresis': 500,
 'oe': 722,
 'ogonek': 333,
 'ograve': 500,
 'one': 500,
 'onehalf': 750,
 'onequarter': 750,
 'onesuperior': 300,
 'ordfeminine': 276,
 'ordmasculine': 310,
 'oslash': 500,
 'otilde': 500,
 'p': 500,
 'paragraph': 453,
 'parenleft': 333,
 'parenright': 333,
 'percent': 833,
 'period': 250,
 'periodcentered': 250,
 'perthousand': 1000,
 'plus': 564,
 'plusminus': 564,
 'q': 500,
 'question': 444,
 'questiondown': 444,
 'quotedbl': 408,
 'quotedblbase': 444,
 'quotedblleft': 444,
 'quotedblright': 444,
 'quoteleft': 333,
 'quoteright': 333,
 'quotesinglbase': 333,
 'quotesingle': 180,
 'r': 333,
 'registered': 760,
 'ring': 333,
 's': 389,
 'scaron': 389,
 'section': 500,
 'semicolon': 278,
 'seven': 500,
 'six': 500,
 'slash': 278,
 'space': 250,
 'sterling': 500,
 't': 278,
 'thorn': 500,
 'three': 500,
 'threequarters': 750,
 'threesuperior': 300,
 'tilde': 333,
 'trademark': 980,
 'two': 500,
 'twosuperior': 300,
 'u': 500,
 'uacute': 500,
 'ucircumflex': 500,
 'udieresis': 500,
 'ugrave': 500,
 'underscore': 500,
 'v': 500,
 'w': 722,
 'x': 500,
 'y': 500,
 'yacute': 500,
 'ydieresis': 500,
 'yen': 500,
 'z': 444,
 'zcaron': 444,
 'zero': 500}

widthsByFontGlyph['Times-Bold'] = {'A': 722,
 'AE': 1000,
 'Aacute': 722,
 'Acircumflex': 722,
 'Adieresis': 722,
 'Agrave': 722,
 'Aring': 722,
 'Atilde': 722,
 'B': 667,
 'C': 722,
 'Ccedilla': 722,
 'D': 722,
 'E': 667,
 'Eacute': 667,
 'Ecircumflex': 667,
 'Edieresis': 667,
 'Egrave': 667,
 'Eth': 722,
 'Euro': 500,
 'F': 611,
 'G': 778,
 'H': 778,
 'I': 389,
 'Iacute': 389,
 'Icircumflex': 389,
 'Idieresis': 389,
 'Igrave': 389,
 'J': 500,
 'K': 778,
 'L': 667,
 'Lslash': 667,
 'M': 944,
 'N': 722,
 'Ntilde': 722,
 'O': 778,
 'OE': 1000,
 'Oacute': 778,
 'Ocircumflex': 778,
 'Odieresis': 778,
 'Ograve': 778,
 'Oslash': 778,
 'Otilde': 778,
 'P': 611,
 'Q': 778,
 'R': 722,
 'S': 556,
 'Scaron': 556,
 'T': 667,
 'Thorn': 611,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 722,
 'W': 1000,
 'X': 722,
 'Y': 722,
 'Yacute': 722,
 'Ydieresis': 722,
 'Z': 667,
 'Zcaron': 667,
 'a': 500,
 'aacute': 500,
 'acircumflex': 500,
 'acute': 333,
 'adieresis': 500,
 'ae': 722,
 'agrave': 500,
 'ampersand': 833,
 'aring': 500,
 'asciicircum': 581,
 'asciitilde': 520,
 'asterisk': 500,
 'at': 930,
 'atilde': 500,
 'b': 556,
 'backslash': 278,
 'bar': 220,
 'braceleft': 394,
 'braceright': 394,
 'bracketleft': 333,
 'bracketright': 333,
 'breve': 333,
 'brokenbar': 220,
 'bullet': 350,
 'c': 444,
 'caron': 333,
 'ccedilla': 444,
 'cedilla': 333,
 'cent': 500,
 'circumflex': 333,
 'colon': 333,
 'comma': 250,
 'copyright': 747,
 'currency': 500,
 'd': 556,
 'dagger': 500,
 'daggerdbl': 500,
 'degree': 400,
 'dieresis': 333,
 'divide': 570,
 'dollar': 500,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 444,
 'eacute': 444,
 'ecircumflex': 444,
 'edieresis': 444,
 'egrave': 444,
 'eight': 500,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 500,
 'equal': 570,
 'eth': 500,
 'exclam': 333,
 'exclamdown': 333,
 'f': 333,
 'fi': 556,
 'five': 500,
 'fl': 556,
 'florin': 500,
 'four': 500,
 'fraction': 167,
 'g': 500,
 'germandbls': 556,
 'grave': 333,
 'greater': 570,
 'guillemotleft': 500,
 'guillemotright': 500,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 556,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 278,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 333,
 'k': 556,
 'l': 278,
 'less': 570,
 'logicalnot': 570,
 'lslash': 278,
 'm': 833,
 'macron': 333,
 'minus': 570,
 'mu': 556,
 'multiply': 570,
 'n': 556,
 'nine': 500,
 'ntilde': 556,
 'numbersign': 500,
 'o': 500,
 'oacute': 500,
 'ocircumflex': 500,
 'odieresis': 500,
 'oe': 722,
 'ogonek': 333,
 'ograve': 500,
 'one': 500,
 'onehalf': 750,
 'onequarter': 750,
 'onesuperior': 300,
 'ordfeminine': 300,
 'ordmasculine': 330,
 'oslash': 500,
 'otilde': 500,
 'p': 556,
 'paragraph': 540,
 'parenleft': 333,
 'parenright': 333,
 'percent': 1000,
 'period': 250,
 'periodcentered': 250,
 'perthousand': 1000,
 'plus': 570,
 'plusminus': 570,
 'q': 556,
 'question': 500,
 'questiondown': 500,
 'quotedbl': 555,
 'quotedblbase': 500,
 'quotedblleft': 500,
 'quotedblright': 500,
 'quoteleft': 333,
 'quoteright': 333,
 'quotesinglbase': 333,
 'quotesingle': 278,
 'r': 444,
 'registered': 747,
 'ring': 333,
 's': 389,
 'scaron': 389,
 'section': 500,
 'semicolon': 333,
 'seven': 500,
 'six': 500,
 'slash': 278,
 'space': 250,
 'sterling': 500,
 't': 333,
 'thorn': 556,
 'three': 500,
 'threequarters': 750,
 'threesuperior': 300,
 'tilde': 333,
 'trademark': 1000,
 'two': 500,
 'twosuperior': 300,
 'u': 556,
 'uacute': 556,
 'ucircumflex': 556,
 'udieresis': 556,
 'ugrave': 556,
 'underscore': 500,
 'v': 500,
 'w': 722,
 'x': 500,
 'y': 500,
 'yacute': 500,
 'ydieresis': 500,
 'yen': 500,
 'z': 444,
 'zcaron': 444,
 'zero': 500}

widthsByFontGlyph['Times-Italic'] = {'A': 611,
 'AE': 889,
 'Aacute': 611,
 'Acircumflex': 611,
 'Adieresis': 611,
 'Agrave': 611,
 'Aring': 611,
 'Atilde': 611,
 'B': 611,
 'C': 667,
 'Ccedilla': 667,
 'D': 722,
 'E': 611,
 'Eacute': 611,
 'Ecircumflex': 611,
 'Edieresis': 611,
 'Egrave': 611,
 'Eth': 722,
 'Euro': 500,
 'F': 611,
 'G': 722,
 'H': 722,
 'I': 333,
 'Iacute': 333,
 'Icircumflex': 333,
 'Idieresis': 333,
 'Igrave': 333,
 'J': 444,
 'K': 667,
 'L': 556,
 'Lslash': 556,
 'M': 833,
 'N': 667,
 'Ntilde': 667,
 'O': 722,
 'OE': 944,
 'Oacute': 722,
 'Ocircumflex': 722,
 'Odieresis': 722,
 'Ograve': 722,
 'Oslash': 722,
 'Otilde': 722,
 'P': 611,
 'Q': 722,
 'R': 611,
 'S': 500,
 'Scaron': 500,
 'T': 556,
 'Thorn': 611,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 611,
 'W': 833,
 'X': 611,
 'Y': 556,
 'Yacute': 556,
 'Ydieresis': 556,
 'Z': 556,
 'Zcaron': 556,
 'a': 500,
 'aacute': 500,
 'acircumflex': 500,
 'acute': 333,
 'adieresis': 500,
 'ae': 667,
 'agrave': 500,
 'ampersand': 778,
 'aring': 500,
 'asciicircum': 422,
 'asciitilde': 541,
 'asterisk': 500,
 'at': 920,
 'atilde': 500,
 'b': 500,
 'backslash': 278,
 'bar': 275,
 'braceleft': 400,
 'braceright': 400,
 'bracketleft': 389,
 'bracketright': 389,
 'breve': 333,
 'brokenbar': 275,
 'bullet': 350,
 'c': 444,
 'caron': 333,
 'ccedilla': 444,
 'cedilla': 333,
 'cent': 500,
 'circumflex': 333,
 'colon': 333,
 'comma': 250,
 'copyright': 760,
 'currency': 500,
 'd': 500,
 'dagger': 500,
 'daggerdbl': 500,
 'degree': 400,
 'dieresis': 333,
 'divide': 675,
 'dollar': 500,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 444,
 'eacute': 444,
 'ecircumflex': 444,
 'edieresis': 444,
 'egrave': 444,
 'eight': 500,
 'ellipsis': 889,
 'emdash': 889,
 'endash': 500,
 'equal': 675,
 'eth': 500,
 'exclam': 333,
 'exclamdown': 389,
 'f': 278,
 'fi': 500,
 'five': 500,
 'fl': 500,
 'florin': 500,
 'four': 500,
 'fraction': 167,
 'g': 500,
 'germandbls': 500,
 'grave': 333,
 'greater': 675,
 'guillemotleft': 500,
 'guillemotright': 500,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 500,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 278,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 278,
 'k': 444,
 'l': 278,
 'less': 675,
 'logicalnot': 675,
 'lslash': 278,
 'm': 722,
 'macron': 333,
 'minus': 675,
 'mu': 500,
 'multiply': 675,
 'n': 500,
 'nine': 500,
 'ntilde': 500,
 'numbersign': 500,
 'o': 500,
 'oacute': 500,
 'ocircumflex': 500,
 'odieresis': 500,
 'oe': 667,
 'ogonek': 333,
 'ograve': 500,
 'one': 500,
 'onehalf': 750,
 'onequarter': 750,
 'onesuperior': 300,
 'ordfeminine': 276,
 'ordmasculine': 310,
 'oslash': 500,
 'otilde': 500,
 'p': 500,
 'paragraph': 523,
 'parenleft': 333,
 'parenright': 333,
 'percent': 833,
 'period': 250,
 'periodcentered': 250,
 'perthousand': 1000,
 'plus': 675,
 'plusminus': 675,
 'q': 500,
 'question': 500,
 'questiondown': 500,
 'quotedbl': 420,
 'quotedblbase': 556,
 'quotedblleft': 556,
 'quotedblright': 556,
 'quoteleft': 333,
 'quoteright': 333,
 'quotesinglbase': 333,
 'quotesingle': 214,
 'r': 389,
 'registered': 760,
 'ring': 333,
 's': 389,
 'scaron': 389,
 'section': 500,
 'semicolon': 333,
 'seven': 500,
 'six': 500,
 'slash': 278,
 'space': 250,
 'sterling': 500,
 't': 278,
 'thorn': 500,
 'three': 500,
 'threequarters': 750,
 'threesuperior': 300,
 'tilde': 333,
 'trademark': 980,
 'two': 500,
 'twosuperior': 300,
 'u': 500,
 'uacute': 500,
 'ucircumflex': 500,
 'udieresis': 500,
 'ugrave': 500,
 'underscore': 500,
 'v': 444,
 'w': 667,
 'x': 444,
 'y': 444,
 'yacute': 444,
 'ydieresis': 444,
 'yen': 500,
 'z': 389,
 'zcaron': 389,
 'zero': 500}

widthsByFontGlyph['Times-BoldItalic'] = {'A': 667,
 'AE': 944,
 'Aacute': 667,
 'Acircumflex': 667,
 'Adieresis': 667,
 'Agrave': 667,
 'Aring': 667,
 'Atilde': 667,
 'B': 667,
 'C': 667,
 'Ccedilla': 667,
 'D': 722,
 'E': 667,
 'Eacute': 667,
 'Ecircumflex': 667,
 'Edieresis': 667,
 'Egrave': 667,
 'Eth': 722,
 'Euro': 500,
 'F': 667,
 'G': 722,
 'H': 778,
 'I': 389,
 'Iacute': 389,
 'Icircumflex': 389,
 'Idieresis': 389,
 'Igrave': 389,
 'J': 500,
 'K': 667,
 'L': 611,
 'Lslash': 611,
 'M': 889,
 'N': 722,
 'Ntilde': 722,
 'O': 722,
 'OE': 944,
 'Oacute': 722,
 'Ocircumflex': 722,
 'Odieresis': 722,
 'Ograve': 722,
 'Oslash': 722,
 'Otilde': 722,
 'P': 611,
 'Q': 722,
 'R': 667,
 'S': 556,
 'Scaron': 556,
 'T': 611,
 'Thorn': 611,
 'U': 722,
 'Uacute': 722,
 'Ucircumflex': 722,
 'Udieresis': 722,
 'Ugrave': 722,
 'V': 667,
 'W': 889,
 'X': 667,
 'Y': 611,
 'Yacute': 611,
 'Ydieresis': 611,
 'Z': 611,
 'Zcaron': 611,
 'a': 500,
 'aacute': 500,
 'acircumflex': 500,
 'acute': 333,
 'adieresis': 500,
 'ae': 722,
 'agrave': 500,
 'ampersand': 778,
 'aring': 500,
 'asciicircum': 570,
 'asciitilde': 570,
 'asterisk': 500,
 'at': 832,
 'atilde': 500,
 'b': 500,
 'backslash': 278,
 'bar': 220,
 'braceleft': 348,
 'braceright': 348,
 'bracketleft': 333,
 'bracketright': 333,
 'breve': 333,
 'brokenbar': 220,
 'bullet': 350,
 'c': 444,
 'caron': 333,
 'ccedilla': 444,
 'cedilla': 333,
 'cent': 500,
 'circumflex': 333,
 'colon': 333,
 'comma': 250,
 'copyright': 747,
 'currency': 500,
 'd': 500,
 'dagger': 500,
 'daggerdbl': 500,
 'degree': 400,
 'dieresis': 333,
 'divide': 570,
 'dollar': 500,
 'dotaccent': 333,
 'dotlessi': 278,
 'e': 444,
 'eacute': 444,
 'ecircumflex': 444,
 'edieresis': 444,
 'egrave': 444,
 'eight': 500,
 'ellipsis': 1000,
 'emdash': 1000,
 'endash': 500,
 'equal': 570,
 'eth': 500,
 'exclam': 389,
 'exclamdown': 389,
 'f': 333,
 'fi': 556,
 'five': 500,
 'fl': 556,
 'florin': 500,
 'four': 500,
 'fraction': 167,
 'g': 500,
 'germandbls': 500,
 'grave': 333,
 'greater': 570,
 'guillemotleft': 500,
 'guillemotright': 500,
 'guilsinglleft': 333,
 'guilsinglright': 333,
 'h': 556,
 'hungarumlaut': 333,
 'hyphen': 333,
 'i': 278,
 'iacute': 278,
 'icircumflex': 278,
 'idieresis': 278,
 'igrave': 278,
 'j': 278,
 'k': 500,
 'l': 278,
 'less': 570,
 'logicalnot': 606,
 'lslash': 278,
 'm': 778,
 'macron': 333,
 'minus': 606,
 'mu': 576,
 'multiply': 570,
 'n': 556,
 'nine': 500,
 'ntilde': 556,
 'numbersign': 500,
 'o': 500,
 'oacute': 500,
 'ocircumflex': 500,
 'odieresis': 500,
 'oe': 722,
 'ogonek': 333,
 'ograve': 500,
 'one': 500,
 'onehalf': 750,
 'onequarter': 750,
 'onesuperior': 300,
 'ordfeminine': 266,
 'ordmasculine': 300,
 'oslash': 500,
 'otilde': 500,
 'p': 500,
 'paragraph': 500,
 'parenleft': 333,
 'parenright': 333,
 'percent': 833,
 'period': 250,
 'periodcentered': 250,
 'perthousand': 1000,
 'plus': 570,
 'plusminus': 570,
 'q': 500,
 'question': 500,
 'questiondown': 500,
 'quotedbl': 555,
 'quotedblbase': 500,
 'quotedblleft': 500,
 'quotedblright': 500,
 'quoteleft': 333,
 'quoteright': 333,
 'quotesinglbase': 333,
 'quotesingle': 278,
 'r': 389,
 'registered': 747,
 'ring': 333,
 's': 389,
 'scaron': 389,
 'section': 500,
 'semicolon': 333,
 'seven': 500,
 'six': 500,
 'slash': 278,
 'space': 250,
 'sterling': 500,
 't': 278,
 'thorn': 500,
 'three': 500,
 'threequarters': 750,
 'threesuperior': 300,
 'tilde': 333,
 'trademark': 1000,
 'two': 500,
 'twosuperior': 300,
 'u': 556,
 'uacute': 556,
 'ucircumflex': 556,
 'udieresis': 556,
 'ugrave': 556,
 'underscore': 500,
 'v': 444,
 'w': 667,
 'x': 500,
 'y': 444,
 'yacute': 444,
 'ydieresis': 444,
 'yen': 500,
 'z': 389,
 'zcaron': 389,
 'zero': 500}

widthsByFontGlyph['Symbol'] = {'Alpha': 722,
 'Beta': 667,
 'Chi': 722,
 'Delta': 612,
 'Epsilon': 611,
 'Eta': 722,
 'Euro': 750,
 'Gamma': 603,
 'Ifraktur': 686,
 'Iota': 333,
 'Kappa': 722,
 'Lambda': 686,
 'Mu': 889,
 'Nu': 722,
 'Omega': 768,
 'Omicron': 722,
 'Phi': 763,
 'Pi': 768,
 'Psi': 795,
 'Rfraktur': 795,
 'Rho': 556,
 'Sigma': 592,
 'Tau': 611,
 'Theta': 741,
 'Upsilon': 690,
 'Upsilon1': 620,
 'Xi': 645,
 'Zeta': 611,
 'aleph': 823,
 'alpha': 631,
 'ampersand': 778,
 'angle': 768,
 'angleleft': 329,
 'angleright': 329,
 'apple': 790,
 'approxequal': 549,
 'arrowboth': 1042,
 'arrowdblboth': 1042,
 'arrowdbldown': 603,
 'arrowdblleft': 987,
 'arrowdblright': 987,
 'arrowdblup': 603,
 'arrowdown': 603,
 'arrowhorizex': 1000,
 'arrowleft': 987,
 'arrowright': 987,
 'arrowup': 603,
 'arrowvertex': 603,
 'asteriskmath': 500,
 'bar': 200,
 'beta': 549,
 'braceex': 494,
 'braceleft': 480,
 'braceleftbt': 494,
 'braceleftmid': 494,
 'bracelefttp': 494,
 'braceright': 480,
 'bracerightbt': 494,
 'bracerightmid': 494,
 'bracerighttp': 494,
 'bracketleft': 333,
 'bracketleftbt': 384,
 'bracketleftex': 384,
 'bracketlefttp': 384,
 'bracketright': 333,
 'bracketrightbt': 384,
 'bracketrightex': 384,
 'bracketrighttp': 384,
 'bullet': 460,
 'carriagereturn': 658,
 'chi': 549,
 'circlemultiply': 768,
 'circleplus': 768,
 'club': 753,
 'colon': 278,
 'comma': 250,
 'congruent': 549,
 'copyrightsans': 790,
 'copyrightserif': 790,
 'degree': 400,
 'delta': 494,
 'diamond': 753,
 'divide': 549,
 'dotmath': 250,
 'eight': 500,
 'element': 713,
 'ellipsis': 1000,
 'emptyset': 823,
 'epsilon': 439,
 'equal': 549,
 'equivalence': 549,
 'eta': 603,
 'exclam': 333,
 'existential': 549,
 'five': 500,
 'florin': 500,
 'four': 500,
 'fraction': 167,
 'gamma': 411,
 'gradient': 713,
 'greater': 549,
 'greaterequal': 549,
 'heart': 753,
 'infinity': 713,
 'integral': 274,
 'integralbt': 686,
 'integralex': 686,
 'integraltp': 686,
 'intersection': 768,
 'iota': 329,
 'kappa': 549,
 'lambda': 549,
 'less': 549,
 'lessequal': 549,
 'logicaland': 603,
 'logicalnot': 713,
 'logicalor': 603,
 'lozenge': 494,
 'minus': 549,
 'minute': 247,
 'mu': 576,
 'multiply': 549,
 'nine': 500,
 'notelement': 713,
 'notequal': 549,
 'notsubset': 713,
 'nu': 521,
 'numbersign': 500,
 'omega': 686,
 'omega1': 713,
 'omicron': 549,
 'one': 500,
 'parenleft': 333,
 'parenleftbt': 384,
 'parenleftex': 384,
 'parenlefttp': 384,
 'parenright': 333,
 'parenrightbt': 384,
 'parenrightex': 384,
 'parenrighttp': 384,
 'partialdiff': 494,
 'percent': 833,
 'period': 250,
 'perpendicular': 658,
 'phi': 521,
 'phi1': 603,
 'pi': 549,
 'plus': 549,
 'plusminus': 549,
 'product': 823,
 'propersubset': 713,
 'propersuperset': 713,
 'proportional': 713,
 'psi': 686,
 'question': 444,
 'radical': 549,
 'radicalex': 500,
 'reflexsubset': 713,
 'reflexsuperset': 713,
 'registersans': 790,
 'registerserif': 790,
 'rho': 549,
 'second': 411,
 'semicolon': 278,
 'seven': 500,
 'sigma': 603,
 'sigma1': 439,
 'similar': 549,
 'six': 500,
 'slash': 278,
 'space': 250,
 'spade': 753,
 'suchthat': 439,
 'summation': 713,
 'tau': 439,
 'therefore': 863,
 'theta': 521,
 'theta1': 631,
 'three': 500,
 'trademarksans': 786,
 'trademarkserif': 890,
 'two': 500,
 'underscore': 500,
 'union': 768,
 'universal': 713,
 'upsilon': 576,
 'weierstrass': 987,
 'xi': 493,
 'zero': 500,
 'zeta': 494}

widthsByFontGlyph['ZapfDingbats'] = {'a1': 974,
 'a10': 692,
 'a100': 668,
 'a101': 732,
 'a102': 544,
 'a103': 544,
 'a104': 910,
 'a105': 911,
 'a106': 667,
 'a107': 760,
 'a108': 760,
 'a109': 626,
 'a11': 960,
 'a110': 694,
 'a111': 595,
 'a112': 776,
 'a117': 690,
 'a118': 791,
 'a119': 790,
 'a12': 939,
 'a120': 788,
 'a121': 788,
 'a122': 788,
 'a123': 788,
 'a124': 788,
 'a125': 788,
 'a126': 788,
 'a127': 788,
 'a128': 788,
 'a129': 788,
 'a13': 549,
 'a130': 788,
 'a131': 788,
 'a132': 788,
 'a133': 788,
 'a134': 788,
 'a135': 788,
 'a136': 788,
 'a137': 788,
 'a138': 788,
 'a139': 788,
 'a14': 855,
 'a140': 788,
 'a141': 788,
 'a142': 788,
 'a143': 788,
 'a144': 788,
 'a145': 788,
 'a146': 788,
 'a147': 788,
 'a148': 788,
 'a149': 788,
 'a15': 911,
 'a150': 788,
 'a151': 788,
 'a152': 788,
 'a153': 788,
 'a154': 788,
 'a155': 788,
 'a156': 788,
 'a157': 788,
 'a158': 788,
 'a159': 788,
 'a16': 933,
 'a160': 894,
 'a161': 838,
 'a162': 924,
 'a163': 1016,
 'a164': 458,
 'a165': 924,
 'a166': 918,
 'a167': 927,
 'a168': 928,
 'a169': 928,
 'a17': 945,
 'a170': 834,
 'a171': 873,
 'a172': 828,
 'a173': 924,
 'a174': 917,
 'a175': 930,
 'a176': 931,
 'a177': 463,
 'a178': 883,
 'a179': 836,
 'a18': 974,
 'a180': 867,
 'a181': 696,
 'a182': 874,
 'a183': 760,
 'a184': 946,
 'a185': 865,
 'a186': 967,
 'a187': 831,
 'a188': 873,
 'a189': 927,
 'a19': 755,
 'a190': 970,
 'a191': 918,
 'a192': 748,
 'a193': 836,
 'a194': 771,
 'a195': 888,
 'a196': 748,
 'a197': 771,
 'a198': 888,
 'a199': 867,
 'a2': 961,
 'a20': 846,
 'a200': 696,
 'a201': 874,
 'a202': 974,
 'a203': 762,
 'a204': 759,
 'a205': 509,
 'a206': 410,
 'a21': 762,
 'a22': 761,
 'a23': 571,
 'a24': 677,
 'a25': 763,
 'a26': 760,
 'a27': 759,
 'a28': 754,
 'a29': 786,
 'a3': 980,
 'a30': 788,
 'a31': 788,
 'a32': 790,
 'a33': 793,
 'a34': 794,
 'a35': 816,
 'a36': 823,
 'a37': 789,
 'a38': 841,
 'a39': 823,
 'a4': 719,
 'a40': 833,
 'a41': 816,
 'a42': 831,
 'a43': 923,
 'a44': 744,
 'a45': 723,
 'a46': 749,
 'a47': 790,
 'a48': 792,
 'a49': 695,
 'a5': 789,
 'a50': 776,
 'a51': 768,
 'a52': 792,
 'a53': 759,
 'a54': 707,
 'a55': 708,
 'a56': 682,
 'a57': 701,
 'a58': 826,
 'a59': 815,
 'a6': 494,
 'a60': 789,
 'a61': 789,
 'a62': 707,
 'a63': 687,
 'a64': 696,
 'a65': 689,
 'a66': 786,
 'a67': 787,
 'a68': 713,
 'a69': 791,
 'a7': 552,
 'a70': 785,
 'a71': 791,
 'a72': 873,
 'a73': 761,
 'a74': 762,
 'a75': 759,
 'a76': 892,
 'a77': 892,
 'a78': 788,
 'a79': 784,
 'a8': 537,
 'a81': 438,
 'a82': 138,
 'a83': 277,
 'a84': 415,
 'a85': 509,
 'a86': 410,
 'a87': 234,
 'a88': 234,
 'a89': 390,
 'a9': 577,
 'a90': 390,
 'a91': 276,
 'a92': 276,
 'a93': 317,
 'a94': 317,
 'a95': 334,
 'a96': 334,
 'a97': 392,
 'a98': 392,
 'a99': 668,
 'space': 278}
