<?php

// Sort types
/** @const MNEMO_SORT_DESC Sort by memo description.    */ define('MNEMO_SORT_DESC', 0);
/** @const MNEMO_SORT_CATEGORY Sort by memo category.   */ define('MNEMO_SORT_CATEGORY', 1);
/** @const MNEMO_SORT_ASCEND Sort in ascending order.   */ define('MNEMO_SORT_ASCEND', 0);
/** @const MNEMO_SORT_DESCEND Sort in descending order. */ define('MNEMO_SORT_DESCEND', 1);

/**
 * Mnemo Base Class.
 *
 * $Horde: mnemo/lib/Mnemo.php,v 1.52.2.1 2005/01/14 14:12:26 jan Exp $
 *
 * @author  Jon Parise <jon@horde.org>
 * @package Mnemo
 */
class Mnemo {

    /**
     * Retrieves the current user's memo list from storage. This
     * function will also sort the resulting list, if requested.
     *
     * @param constant $sortby   (optional) The field by which to sort.
     *                           (MNEMO_SORT_DESC, MNEMO_SORT_CATEGORY)
     * @param constant $sortdir  (optional) The direction by which to sort.
     *                           (MNEMO_SORT_ASC, MNEMO_SORT_DESC)
     *
     * @return array        Returns a list of the requested memos.
     *
     * @see Mnemo_Driver::listMemos()
     */
    function listMemos($sortby = MNEMO_SORT_DESC,
                       $sortdir = MNEMO_SORT_ASCEND)
    {
        global $conf, $display_notepads;
        $memos = array();

        /* Sort the memo list. */
        $sort_functions = array(
            MNEMO_SORT_DESC => 'ByDesc',
            MNEMO_SORT_CATEGORY => 'ByCategory'
        );

        foreach ($display_notepads as $notepad) {
            /* Create a Mnemo storage instance. */
            $storage = &Mnemo_Driver::singleton($notepad);
            $storage->retrieve();

            /* Retrieve the memo list from storage. */
            $newmemos = $storage->listMemos();
            $memos = array_merge($memos, $newmemos);
        }

        /* Sort the array if we have a sort function defined for this
         * field. */
        if (isset($sort_functions[$sortby])) {
            $prefix = ($sortdir == MNEMO_SORT_DESCEND) ? '_rsort' : '_sort';
            uasort($memos, array('Mnemo', $prefix . $sort_functions[$sortby]));
        }

        return $memos;
    }

    /**
     * Retrieve a specific memo from storage.
     *
     * @param string $memolist  The notepad to retrieve the memo from.
     * @param string $memo      The Id of the memo to retrieve.
     *
     * @return array  The memo.
     */
    function getMemo($notepad, $memo)
    {
        $storage = &Mnemo_Driver::singleton($notepad);
        return $storage->get($memo);
    }

    /**
     * List all memo lists a user has access to.
     *
     * @param optional boolean $owneronly  Only return memo lists that this
     *                                     user owns? Defaults to false.
     * @param optional integer $permission The permission to filter notepads by.
     *
     * @return array  The memo lists.
     */
    function listNotepads($owneronly = false, $permission = PERMS_SHOW)
    {
        $notepads = $GLOBALS['mnemo_shares']->listShares(Auth::getAuth(), $permission, $owneronly ? Auth::getAuth() : null);
        if (is_a($notepads, 'PEAR_Error')) {
            Horde::logMessage($notepads, __FILE__, __LINE__, PEAR_LOG_ERR);
            return array();
        }

        return $notepads;
    }

    /**
     * Get the default notepad for the current user at the specified
     * permissions level.
     */
    function getDefaultNotepad($permission = PERMS_SHOW)
    {
        global $prefs;

        $default_notepad = $prefs->getValue('default_notepad');
        $notepads = Mnemo::listNotepads(false, $permission);

        if (isset($notepads[$default_notepad])) {
            return $default_notepad;
        } elseif ($prefs->isLocked('default_notepad')) {
            return '';
        } elseif (count($notepads)) {
            return key($notepads);
        }

        return false;
    }

    /**
     * Comparison function for sorting memos by description.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  1 if memo one is greater, -1 if memo two is greater; 0 if they are equal.
     */
    function _sortByDesc($a, $b)
    {
        return strcasecmp($a['desc'], $b['desc']);
    }

    /**
     * Comparison function for reverse sorting memos by description.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  -1 if memo one is greater, 1 if memo two is greater; 0 if they are equal.
     */
    function _rsortByDesc($a, $b)
    {
        return strcasecmp($b['desc'], $a['desc']);
    }

    /**
     * Comparison function for sorting memos by category.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  1 if memo one is greater, -1 if memo two is greater; 0 if they are equal.
     */
    function _sortByCategory($a, $b)
    {
        return strcasecmp($a['category'], $b['category']);
    }

    /**
     * Comparison function for reverse sorting memos by category.
     *
     * @param array $a  Memo one.
     * @param array $b  Memo two.
     *
     * @return integer  -1 if memo one is greater, 1 if memo two is greater; 0 if they are equal.
     */
    function _rsortByCategory($a, $b)
    {
        return strcasecmp($b['category'], $a['category']);
    }

    /**
     * Build Mnemo's list of menu items.
     *
     * @access public
     */
    function getMenu($returnType = 'object')
    {
        global $conf, $registry, $browser, $print_link;

        require_once 'Horde/Menu.php';

        $menu = &new Menu();
        $menu->add(Horde::applicationUrl('list.php'), _("_List Notes"), 'mnemo.png', null, null, null, basename($_SERVER['PHP_SELF']) == 'index.php' ? 'current' : null);
        if (Mnemo::getDefaultNotepad(PERMS_EDIT)) {
            $menu->add(Horde::applicationUrl('memo.php?actionID=add_memo'), _("_New Note"), 'add.png');
        }
        if (Mnemo::getDefaultNotepad(PERMS_READ)) {
            $menu->add(Horde::applicationUrl('search.php'), _("_Search"), 'search.png', $registry->getImageDir('horde'));
        }

        if (Auth::getAuth()) {
            $menu->add(Horde::applicationUrl('notepads.php'), _("_My Notepads"), 'notepads.png');
        }

        /* Import/Export */
        if ($conf['menu']['import_export']) {
            $menu->add(Horde::applicationUrl('data.php'), _("_Import/Export"), 'data.png', $registry->getImageDir('horde'));
        }

        /* Print */
        if ($conf['menu']['print'] && isset($print_link) && $browser->hasFeature('javascript')) {
            $menu->add('#', _("_Print"), 'print.png', $registry->getImageDir('horde'), '', "popup('$print_link'); return false;");
        }

        if ($returnType == 'object') {
            return $menu;
        } else {
            return $menu->render();
        }
    }

}
