\name{SOM}
\alias{SOM}
\title{
Self-Organizing Maps: Online Algorithm
}
\description{
Kohonen's Self-Organizing Maps are a crude form of multidimensional scaling.
}
\usage{
SOM(data, grid = somgrid(), rlen = 10000, alpha, radii, init)
}
\arguments{
\item{data}{
a matrix or data frame of observations, scaled so that Euclidean
distance is appropriate.
}
\item{grid}{
A grid for the representatives: see \code{\link{somgrid}}.
}
\item{rlen}{
the number of updates: used only in the defaults for \code{alpha} and \code{radii}.
}
\item{alpha}{
the amount of change: one update is done for each element of \code{alpha}.
Default is to decline linearly from 0.05 to 0 over \code{rlen} updates.
}
\item{radii}{
the radii of the neighbourhood to be used for each update: must be the
same length as \code{alpha}.  Default is to decline linearly from 4 to 1
over \code{rlen} updates.
}
\item{init}{
the initial representatives.  If missing, chosen (without replacement)
randomly from \code{data}.
}}
\value{
an object of class \code{"SOM"} with components

\item{grid}{
the grid, an object of class \code{"somgrid"}.
}
\item{codes}{
a matrix of representatives.
}}
\details{
\code{alpha} and \code{radii} can also be lists, in which case each component is
used in turn, allowing two- or more phase training.
}
\seealso{
  \code{\link{somgrid}}, \code{\link{batchSOM}}
}
\references{
  Kohonen, T. (1995) \emph{Self-Organizing Maps.} Springer-Verlag

  Kohonen, T., Hynninen, J., Kangas, J. and Laaksonen, J. (1996)
  \emph{SOM PAK: The self-organizing map program package.}
  Laboratory of Computer and Information Science, Helsinki University
  of Technology, Technical Report A31.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\examples{
data(crabs, package = "MASS")

lcrabs <- log(crabs[, 4:8])
crabs.grp <- factor(c("B", "b", "O", "o")[rep(1:4, rep(50,4))])
gr <- somgrid(topo = "hexagonal")
crabs.som <- SOM(lcrabs, gr)
plot(crabs.som)

## 2-phase training
crabs.som2 <- SOM(lcrabs, gr,
    alpha = list(seq(0.05, 0, len = 1e4), seq(0.02, 0, len = 1e5)),
    radii = list(seq(8, 1, len = 1e4), seq(4, 1, len = 1e5)))
plot(crabs.som2)
}
\keyword{classif}

\eof
\name{batchSOM}
\alias{batchSOM}
\title{
Self-Organizing Maps: Batch Algorithm
}
\description{
Kohonen's Self-Organizing Maps are a crude form of multidimensional scaling.
}
\usage{
batchSOM(data, grid = somgrid(), radii, init)
}
\arguments{
  \item{data}{
    a matrix or data frame of observations, scaled so that Euclidean
    distance is appropriate.
  }
  \item{grid}{
    A grid for the representatives: see \code{\link{somgrid}}.
  }
  \item{radii}{
    the radii of the neighbourhood to be used for each pass: one pass is
    run for each element of \code{radii}.
  }
  \item{init}{
    the initial representatives.  If missing, chosen (without replacement)
    randomly from \code{data}.
  }
}
\value{
  an object of class \code{"SOM"} with components
  \item{grid}{the grid, an object of class \code{"somgrid"}.}
  \item{codes}{a matrix of representatives.}
}
\details{
  The batch SOM algorithm of Kohonen(1995, section 3.14) is used.
}
\seealso{
  \code{\link{somgrid}}, \code{\link{SOM}}
}
\references{
  Kohonen, T. (1995) \emph{Self-Organizing Maps.} Springer-Verlag.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\examples{
data(crabs, package = "MASS")

lcrabs <- log(crabs[, 4:8])
crabs.grp <- factor(c("B", "b", "O", "o")[rep(1:4, rep(50,4))])
gr <- somgrid(topo = "hexagonal")
crabs.som <- batchSOM(lcrabs, gr, c(4, 4, 2, 2, 1, 1, 1, 0, 0))
plot(crabs.som)

bins <- as.numeric(knn1(crabs.som$code, lcrabs, 0:47))
plot(crabs.som$grid, type = "n")
symbols(crabs.som$grid$pts[, 1], crabs.som$grid$pts[, 2],
        circles = rep(0.4, 48), inches = FALSE, add = TRUE)
text(crabs.som$grid$pts[bins, ] + rnorm(400, 0, 0.1),
     as.character(crabs.grp))
}
\keyword{classif}

\eof
% file class/condense.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{condense}
\alias{condense}
\title{
Condense training set for k-NN classifier
}
\description{
Condense training set for k-NN classifier
}
\usage{
condense(train, class, store, trace = TRUE)
}
\arguments{
\item{train}{
matrix for training set
}
\item{class}{
vector of classifications for test set
}
\item{store}{
initial store set. Default one randomly chosen element of the set.
}
\item{trace}{
logical. Trace iterations?
}}
\details{
The store set is used to 1-NN classify the rest, and misclassified
patterns are added to the store set. The whole set is checked until
no additions occur.
}
\value{
index vector of cases to be retained (the final store set).
}
\references{
P. A. Devijver and J. Kittler (1982)
\emph{Pattern Recognition. A Statistical Approach.}
Prentice-Hall, pp. 119--121.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{reduce.nn}}, \code{\link{multiedit}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
keep <- condense(train, cl)
knn(train[keep, , drop=FALSE], test, cl[keep])
keep2 <- reduce.nn(train, keep, cl)
knn(train[keep2, , drop=FALSE], test, cl[keep2])
}
\keyword{classif}

\eof
% file class/knn.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{knn}
\alias{knn}
\title{
k-Nearest Neighbour Classification
}
\description{
k-nearest neighbour classification for test set from training set. For
each row of the test set, the \code{k} nearest (in Euclidean distance)
training set vectors are found, and the classification is decided by
majority vote, with ties broken at random. If there are ties for the
\code{k}th nearest vector, all candidates are included in the vote.
}
\usage{
knn(train, test, cl, k = 1, l = 0, prob = FALSE, use.all = TRUE)
}
\arguments{
\item{train}{
matrix or data frame of training set cases.
}
\item{test}{
matrix or data frame of test set cases. A vector will be interpreted
as a row vector for a single case.
}
\item{cl}{
factor of true classifications of training set
}
\item{k}{
number of neighbours considered.
}
\item{l}{
minimum vote for definite decision, otherwise \code{doubt}. (More
precisely, less than \code{k-l} dissenting votes are allowed, even if \code{k}
is increased by ties.)
}
\item{prob}{
If this is true, the proportion of the votes for the winning class
are returned as attribute \code{prob}.
}
\item{use.all}{
controls handling of ties. If true, all distances equal to the \code{k}th
largest are included. If false, a random selection of distances
equal to the \code{k}th is chosen to use exactly \code{k} neighbours.
}}
\value{
factor of classifications of test set. \code{doubt} will be returned as \code{NA}.
}
\references{
  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{knn1}}, \code{\link{knn.cv}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
knn(train, test, cl, k = 3, prob=TRUE)
attributes(.Last.value)
}
\keyword{classif}

\eof
% file class/knn.cv.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{knn.cv}
\alias{knn.cv}
\title{
k-Nearest Neighbour Cross-Validatory Classification
}
\description{
k-nearest neighbour cross-validatory classification from training set.
}
\usage{
knn.cv(train, cl, k = 1, l = 0, prob = FALSE, use.all = TRUE)
}
\arguments{
\item{train}{
matrix or data frame of training set cases.
}
\item{cl}{
factor of true classifications of training set
}
\item{k}{
number of neighbours considered.
}
\item{l}{
minimum vote for definite decision, otherwise \code{doubt}. (More
precisely, less than \code{k-l} dissenting votes are allowed, even
if \code{k} is increased by ties.)
}
\item{prob}{
If this is true, the proportion of the votes for the winning class
are returned as attribute \code{prob}.
}
\item{use.all}{
controls handling of ties. If true, all distances equal to the \code{k}th
largest are included. If false, a random selection of distances
equal to the \code{k}th is chosen to use exactly \code{k} neighbours.
}}
\details{
This uses leave-one-out cross validation.
For each row of the training set \code{train}, the \code{k} nearest
(in Euclidean distance) other
training set vectors are found, and the classification is decided by
majority vote, with ties broken at random. If there are ties for the
\code{k}th nearest vector, all candidates are included in the vote.
}
\value{
factor of classifications of training set. \code{doubt} will be returned as \code{NA}.
}
\references{
  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{knn}}
}
\examples{
data(iris3)
train <- rbind(iris3[,,1], iris3[,,2], iris3[,,3])
cl <- factor(c(rep("s",50), rep("c",50), rep("v",50)))
knn.cv(train, cl, k = 3, prob = TRUE)
attributes(.Last.value)
}
\keyword{classif}

\eof
% file class/knn1.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{knn1}
\alias{knn1}
\title{
1-nearest neighbour classification
}
\description{
Nearest neighbour classification for test set from training set. For
each row of the test set, the nearest (by Euclidean distance) training
set vector is found, and its classification used. If there is more than
one nearest, a majority vote is used with ties broken at random.
}
\usage{
knn1(train, test, cl)
}
\arguments{
\item{train}{
matrix or data frame of training set cases.
}
\item{test}{
matrix or data frame of test set cases. A vector will be interpreted
as a row vector for a single case.
}
\item{cl}{
factor of true classification of training set.
}}
\value{
factor of classifications of test set.
}
\references{
  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{knn}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
knn1(train, test, cl)
}
\keyword{classif}

\eof
% file class/lvq1.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{lvq1}
\alias{lvq1}
\title{
Learning Vector Quantization 1
}
\description{
Moves examples in a codebook to better represent the training set.
}
\usage{
lvq1(x, cl, codebk, niter = 100 * nrow(codebk$x), alpha = 0.03)
}
\arguments{
\item{x}{
a matrix or data frame of examples
}
\item{cl}{
a vector or factor of classifications for the examples
}
\item{codebk}{
a codebook
}
\item{niter}{
number of iterations
}
\item{alpha}{
constant for training
}}
\value{
A codebook, represented as a list with components \code{x} and \code{cl}
giving the examples and classes.
}
\details{
Selects \code{niter} examples at random  with replacement, and adjusts the nearest
example in the codebook for each.
}
\references{
Kohonen, T. (1990) The self-organizing map.
\emph{Proc. IEEE }
\bold{78}, 1464--1480.

Kohonen, T. (1995)
\emph{Self-Organizing Maps.}
Springer, Berlin.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{lvqinit}}, \code{\link{olvq1}}, \code{\link{lvq2}}, \code{\link{lvq3}}, \code{\link{lvqtest}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
cd <- lvqinit(train, cl, 10)
lvqtest(cd, train)
cd0 <- olvq1(train, cl, cd)
lvqtest(cd0, train)
cd1 <- lvq1(train, cl, cd0)
lvqtest(cd1, train)
}
\keyword{classif}

\eof
% file class/lvq2.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{lvq2}
\alias{lvq2}
\title{
Learning Vector Quantization 2.1
}
\description{
Moves examples in a codebook to better represent the training set.
}
\usage{
lvq2(x, cl, codebk, niter = 100 * nrow(codebk$x), alpha = 0.03, win = 0.3)
}
\arguments{
\item{x}{
a matrix or data frame of examples
}
\item{cl}{
a vector or factor of classifications for the examples
}
\item{codebk}{
a codebook
}
\item{niter}{
number of iterations
}
\item{alpha}{
constant for training
}
\item{win}{
a tolerance for the closeness of the two nearest vectors.
}}
\value{
A codebook, represented as a list with components \code{x} and \code{cl}
giving the examples and classes.
}
\details{
Selects \code{niter} examples at random  with replacement, and adjusts the nearest
two examples in the codebook if one is correct and the other incorrect.
}
\references{
Kohonen, T. (1990) The self-organizing map.
\emph{Proc. IEEE}
\bold{78}, 1464--1480.

Kohonen, T. (1995)
\emph{Self-Organizing Maps.}
Springer, Berlin.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
  \code{\link{lvqinit}}, \code{\link{lvq1}}, \code{\link{olvq1}},
  \code{\link{lvq3}}, \code{\link{lvqtest}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
cd <- lvqinit(train, cl, 10)
lvqtest(cd, train)
cd0 <- olvq1(train, cl, cd)
lvqtest(cd0, train)
cd2 <- lvq2(train, cl, cd0)
lvqtest(cd2, train)
}
\keyword{classif}

\eof
% file class/lvq3.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{lvq3}
\alias{lvq3}
\title{
Learning Vector Quantization 3
}
\description{
Moves examples in a codebook to better represent the training set.
}
\usage{
lvq3(x, cl, codebk, niter = 100*nrow(codebk$x), alpha = 0.03,
     win = 0.3, epsilon = 0.1)
}
\arguments{
\item{x}{
a matrix or data frame of examples
}
\item{cl}{
a vector or factor of classifications for the examples
}
\item{codebk}{
a codebook
}
\item{niter}{
number of iterations
}
\item{alpha}{
constant for training
}
\item{win}{
a tolerance for the closeness of the two nearest vectors.
}
\item{epsilon}{
proportion of move for correct vectors
}}
\value{
A codebook, represented as a list with components \code{x} and \code{cl}
giving the examples and classes.
}
\details{
Selects \code{niter} examples at random  with replacement, and adjusts the nearest
two examples in the codebook for each.
}
\references{
Kohonen, T. (1990) The self-organizing map.
\emph{Proc. IEEE}
\bold{78}, 1464--1480.

Kohonen, T. (1995)
\emph{Self-Organizing Maps.}
Springer, Berlin.

Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
  \code{\link{lvqinit}}, \code{\link{lvq1}}, \code{\link{olvq1}},
  \code{\link{lvq2}}, \code{\link{lvqtest}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
cd <- lvqinit(train, cl, 10)
lvqtest(cd, train)
cd0 <- olvq1(train, cl, cd)
lvqtest(cd0, train)
cd3 <- lvq3(train, cl, cd0)
lvqtest(cd3, train)
}
\keyword{classif}

\eof
% file class/lvqinit.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{lvqinit}
\alias{lvqinit}
\title{
Initialize a LVQ Codebook
}
\description{
Construct an initial codebook for LVQ methods.
}
\usage{
lvqinit(x, cl, size, prior, k = 5)
}
\arguments{
\item{x}{
a matrix or data frame of training examples, \code{n} by \code{p}.
}
\item{cl}{
the classifications for the training examples. A vector or factor of
length \code{n}.
}
\item{size}{
the size of the codebook. Defaults to \code{min(round(0.4*ng*(ng-1 + p/2),0), n)}
where \code{ng} is the number of classes.
}
\item{prior}{
Probabilities to represent classes in the codebook. Default proportions in the
training set.
}
\item{k}{
k used for k-NN test of correct classification. Default is 5.
}}
\value{
A codebook, represented as a list with components \code{x} and \code{cl} giving
the examples and classes.
}
\details{
Selects \code{size} examples from the training set without replacement with
proportions proportional to the prior or the original proportions.
}
\references{
Kohonen, T. (1990) The self-organizing map.
\emph{Proc. IEEE }
\bold{78}, 1464--1480.

Kohonen, T. (1995)
\emph{Self-Organizing Maps.}
Springer, Berlin.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{lvq1}}, \code{\link{lvq2}}, \code{\link{lvq3}}, \code{\link{olvq1}}, \code{\link{lvqtest}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
cd <- lvqinit(train, cl, 10)
lvqtest(cd, train)
cd1 <- olvq1(train, cl, cd)
lvqtest(cd1, train)
}
\keyword{classif}

\eof
% file class/lvqtest.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{lvqtest}
\alias{lvqtest}
\title{
Classify Test Set from LVQ Codebook
}
\description{
Classify a test set by 1-NN from a specified LVQ codebook.
}
\usage{
lvqtest(codebk, test)
}
\arguments{
\item{codebk}{
codebook object returned by other LVQ software
}
\item{test}{
matrix of test examples
}}
\value{
factor of classification for each row of \code{x}
}
\details{
uses 1-NN to classify each test example against the codebook.
}
\references{
  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{lvqinit}}, \code{\link{olvq1}}
}
\examples{
# The function is currently defined as
function(codebk, test) knn1(codebk$x, test, codebk$cl)
}
\keyword{classif}

\eof
% file class/multiedit.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{multiedit}
\alias{multiedit}
\title{
Multiedit for k-NN Classifier
}
\description{
Multiedit for k-NN classifier
}
\usage{
multiedit(x, class, k = 1, V = 3, I = 5, trace = TRUE)
}
\arguments{
\item{x}{
matrix of training set.
}
\item{class}{
vector of classification of training set.
}
\item{k}{
number of neighbours used in k-NN.
}
\item{V}{
divide training set into V parts.
}
\item{I}{
number of null passes before quitting.
}
\item{trace}{
logical for statistics at each pass.
}}
\value{
index vector of cases to be retained.
}
\references{
P. A. Devijver and J. Kittler (1982)
\emph{Pattern Recognition. A Statistical Approach.}
Prentice-Hall, p. 115.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{condense}}, \code{\link{reduce.nn}}
}
\examples{
data(iris3)
tr <- sample(1:50, 25)
train <- rbind(iris3[tr,,1], iris3[tr,,2], iris3[tr,,3])
test <- rbind(iris3[-tr,,1], iris3[-tr,,2], iris3[-tr,,3])
cl <- factor(c(rep(1,25),rep(2,25), rep(3,25)), labels=c("s", "c", "v"))
table(cl, knn(train, test, cl, 3))
ind1 <- multiedit(train, cl, 3)
length(ind1)
table(cl, knn(train[ind1, , drop=FALSE], test, cl[ind1], 1))
ntrain <- train[ind1,]; ncl <- cl[ind1]
ind2 <- condense(ntrain, ncl)
length(ind2)
table(cl, knn(ntrain[ind2, , drop=FALSE], test, ncl[ind2], 1))
}
\keyword{classif}

\eof
% file class/olvq1.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{olvq1}
\alias{olvq1}
\title{
Optimized Learning Vector Quantization 1
}
\description{
Moves examples in a codebook to better represent the training set.
}
\usage{
olvq1(x, cl, codebk, niter = 40 * nrow(codebk$x), alpha = 0.3)
}
\arguments{
\item{x}{
a matrix or data frame of examples
}
\item{cl}{
a vector or factor of classifications for the examples
}
\item{codebk}{
a codebook
}
\item{niter}{
number of iterations
}
\item{alpha}{
constant for training
}}
\value{
A codebook, represented as a list with components \code{x} and \code{cl} giving
the examples and classes.
}
\details{
Selects \code{niter} examples at random with replacement, and adjusts the
nearest example in the codebook for each.
}
\references{
Kohonen, T. (1990) The self-organizing map.
\emph{Proc. IEEE}
\bold{78}, 1464--1480.

Kohonen, T. (1995)
\emph{Self-Organizing Maps.}
Springer, Berlin.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{lvqinit}}, \code{\link{lvqtest}}, \code{\link{lvq1}}, \code{\link{lvq2}}, \code{\link{lvq3}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
cd <- lvqinit(train, cl, 10)
lvqtest(cd, train)
cd1 <- olvq1(train, cl, cd)
lvqtest(cd1, train)
}
\keyword{classif}

\eof
% file class/reduce.nn.d copyright (C) 1994-9 W. N. Venables and B. D. Ripley
%
\name{reduce.nn}
\alias{reduce.nn}
\title{
Reduce Training Set for a k-NN Classifier
}
\description{
Reduce training set for a k-NN classifier. Used after \code{condense}.
}
\usage{
reduce.nn(train, ind, class)
}
\arguments{
\item{train}{
matrix for training set
}
\item{ind}{
Initial list of members of the training set (from \code{condense}).
}
\item{class}{
vector of classifications for test set
}}
\details{
All the members of the training set are tried in random order.
Any which when dropped do not cause any members of the training set to
be wrongly classified are dropped.
}
\value{
index vector of cases to be retained.
}
\references{
Gates, G.W. (1972) The reduced nearest neighbor rule.
\emph{IEEE Trans. Information Theory}
\bold{IT-18}, 431--432.

  Ripley, B. D. (1996)
  \emph{Pattern Recognition and Neural Networks.} Cambridge.

  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
\code{\link{condense}}, \code{\link{multiedit}}
}
\examples{
data(iris3)
train <- rbind(iris3[1:25,,1], iris3[1:25,,2], iris3[1:25,,3])
test <- rbind(iris3[26:50,,1], iris3[26:50,,2], iris3[26:50,,3])
cl <- factor(c(rep("s",25), rep("c",25), rep("v",25)))
keep <- condense(train, cl)
knn(train[keep,], test, cl[keep])
keep2 <- reduce.nn(train, keep, cl)
knn(train[keep2,], test, cl[keep2])
}
\keyword{classif}

\eof
\name{somgrid}
\alias{somgrid}
\alias{plot.somgrid}
\alias{plot.SOM}
\title{
Plot SOM Fits
}
\description{
  Plotting functions for SOM results.
}
\usage{
somgrid(xdim = 8, ydim = 6, topo = c("rectangular", "hexagonal"))

\method{plot}{somgrid}(x, type = "p", ...)

\method{plot}{SOM}(x, ...)
}
\arguments{
  \item{xdim, ydim}{dimensions of the grid}
  \item{topo}{the topology of the grid.}
  \item{x}{an object inheriting from class \code{"somgrid"} or \code{"SOM"}.}
  \item{type, \dots}{graphical parameters.}
}
\value{
  For \code{somgrid}, an object of class \code{"somgrid"}, a list with
  components
  \item{pts}{a two-column matrix giving locations for the grid points.}
  \item{xdim, ydim, topo}{as in the arguments to \code{somgrid}.}
}
\details{
  The class \code{"somgrid"} records the coordinates of the grid to be
  used for (batch or on-line) SOM: this has a plot method.

  The plot method for class \code{"SOM"} plots a \code{\link{stars}}
  plot of the representative at each grid point.
}
\references{
  Venables, W. N. and Ripley, B. D. (2002)
  \emph{Modern Applied Statistics with S.} Fourth edition.  Springer.
}
\seealso{
  \code{\link{batchSOM}}, \code{\link{SOM}}
}
\keyword{classif}

\eof
