\name{PkgUtils}
\alias{build}
\alias{check}
\title{Utilities for Building and Checking Add-on Packages}
\description{
  Utilities for checking whether the sources of an \R add-on package
  work correctly, and for building a source or binary package from
  them.
}
\usage{
R CMD build [options] pkgdirs
R CMD check [options] pkgdirs
}
\arguments{
  \item{pkgdirs}{a list of names of directories with sources of \R
    add-on packages.}
  \item{options}{further options to control the processing, or for
    obtaining information about usage and version of the utility.}
}
\details{
  \code{R CMD check}
  checks \R add-on packages from their sources, performing a wide
  variety of diagnostic checks.

  \code{R CMD build}
  builds \R source or binary packages from their sources.  It will
  create index files in the sources if necessary, so it is often helpful
  to run \code{build} before \code{check}.

  Use
  \code{R CMD foo --help}
  to obtain usage information on utility \code{foo}.

  Several of the options to \code{build --binary} are passed to
  \code{\link{INSTALL}} so consult its help for the details.
}
#ifdef windows
\note{
  These may not work correctly under Windows 95/98/ME because of
  problems Perl has launching programs on those limited OSes.

  They make use of a temporary directory specified by the enironment
  variable \code{TMPDIR} and defaulting to \code{c:/TEMP}.  Do ensure
  that if set forward slashes are used.
}
#endif
\seealso{
  The chapter \dQuote{Processing Rd format} in \dQuote{Writing \R
    Extensions}
#ifdef unix
  (see the \file{doc/manual} subdirectory of the \R source tree).
#endif
#ifdef windows
  (see the Manuals sub-menu of the Help menu on the console).
#endif

  \code{\link{INSTALL}} is called by \code{build --binary}.
}
\keyword{utilities}
\eof
\name{Rprof}
\alias{Rprof}
\title{Enable Profiling of R's Execution}
\description{
  Enable or disable profiling of the execution of \R expressions.
}
\usage{
Rprof(filename = "Rprof.out", append = FALSE, interval = 0.02)
}
\arguments{
  \item{filename}{
    The file to be used for recording the profiling results.
    Set to \code{NULL} or \code{""} to disable profiling.
  }
  \item{append}{
    logical: should the file be over-written or appended to?
  }
  \item{interval}{
    real: time interval between samples.
  }
}
\details{
  Enabling profiling automatically disables any existing profiling to
  another or the same file.
  
  Profiling works by writing out the call stack every \code{interval}
  seconds, to the file specified. Either the \code{\link{summaryRprof}}
  function or the Perl script \code{R CMD Rprof} can be used to process
  the output file to produce a summary of the
  usage; use \code{R CMD Rprof --help} for usage information.

#ifdef windows
  Exactly what the time interval measures is subtle: it is time that the
  \R process is running and executing an \R command.  It is not however just
  CPU time, for if \code{readline()} is waiting for input, that counts
  (on Windows, but not on Unix).

  Note that the timing interval cannot be too small, for the time spent
  in each profiling step is added to the interval.  What is feasible is
  machine-dependent, but 10ms seems as small as advisable on a 1GHz machine.
#endif
#ifdef unix
  Note that the timing interval cannot be too small: once the timer goes
  off, the information is not recorded until the next clock tick
  (probably every 10msecs).  Thus the interval is rounded to the nearest
  integer number of clock ticks, and is made to be at least one clock
  tick (at which resolution the total time spent is liable to be
  underestimated).
#endif
}
\note{
#ifdef windows
  Using \code{R CMD Rprof} needs Windows Perl to be installed.
#endif
#ifdef unix
  Profiling is not available on all platforms.  By default, it is
  attempted to compile support for profiling.  Configure \R with
  \option{--disable-R-profiling} to change this.

  As \R profiling uses the same mechanisms as C profiling, the two
  cannot be used together, so do not use \code{Rprof} in an executable
  built for profiling.
#endif
}
\seealso{
  The chapter on \dQuote{Tidying and profiling R code} in
  \dQuote{Writing \R Extensions} (see the \file{doc/manual} subdirectory
  of the \R source tree).

  \code{\link{summaryRprof}}
}
\examples{
\dontrun{Rprof()
## some code to be profiled
Rprof(NULL)
## some code NOT to be profiled
Rprof(append=TRUE)
## some code to be profiled
Rprof(NULL)
\dots
## Now post-process the output as described in Details
}}
\keyword{utilities}
\eof
\name{Rtangle}
\alias{Rtangle}
\alias{RtangleSetup}
\title{R Driver for Stangle}
\description{
  A driver for \code{\link{Stangle}} that extracts R code chunks.
}
\usage{
Rtangle()
RtangleSetup(file, syntax, output=NULL, annotate=TRUE, split=FALSE,
             prefix=TRUE, quiet=FALSE)
}
\arguments{
  \item{file}{Name of Sweave source file.}
  \item{syntax}{An object of class \code{SweaveSyntax}.}
  \item{output}{Name of output file, default is to remove extension \file{.nw},
    \file{.Rnw} or \file{.Snw} and to add extension \file{.R}. Any
    directory names in \code{file} are also removed such that the output
    is created in the current working directory.}
  \item{annotate}{By default, code chunks are seperated by comment
    lines specifying the names and numbers of the code chunks. If
    \code{FALSE}, only the code chunks without any decorating comments
    are extracted.}
  \item{split}{Split output in single files per code chunk?}
  \item{prefix}{If \code{split=TRUE}, prefix the chunk labels by the
    basename of the input file to get output file names?}
  \item{quiet}{If \code{TRUE} all progress messages are suppressed.}
}
\author{Friedrich Leisch}
\references{
  Friedrich Leisch: Sweave User Manual, 2002\cr
  \url{http://www.ci.tuwien.ac.at/~leisch/Sweave}
}
\seealso{\code{\link{Sweave}}, \code{\link{RweaveLatex}}}
\keyword{utilities}

\eof
\name{RweaveLatex}
\alias{RweaveLatex}
\alias{RweaveLatexSetup}
\title{R/LaTeX Driver for Sweave}
\description{
  A driver for \code{\link{Sweave}} that translates R code chunks in
  LaTeX files.
}
\usage{
RweaveLatex()
RweaveLatexSetup(file, syntax, output=NULL, quiet=FALSE, debug=FALSE,
                 echo=TRUE, eval = TRUE, split=FALSE, stylepath=TRUE,
                 pdf=TRUE, eps=TRUE)
}
\arguments{
  \item{file}{Name of Sweave source file.}
  \item{syntax}{An object of class \code{SweaveSyntax}.}
  \item{output}{Name of output file, default is to remove extension
    \file{.nw}, \file{.Rnw} or \file{.Snw} and to add extension
    \file{.tex}. Any 
    directory names in \code{file} are also removed such that the output
    is created in the current working directory.}
  \item{quiet}{If \code{TRUE} all progress messages are suppressed.}
  \item{debug}{If \code{TRUE}, input and output of all code
    chunks is copied to the console.}
  \item{stylepath}{If \code{TRUE}, a hard path to the file \file{Sweave.sty}
    installed with this package is set, if \code{FALSE}, only
    \code{\\usepackage\{Sweave\}} is written. The hard path makes the
    TeX file less
    portable, but avoids the problem of installing the current
    version of \file{Sweave.sty} to some place in your TeX input path.
    The argument is ignored if a \code{\\usepackage\{Sweave\}} is
    already present in the Sweave source file.
  }
  \item{echo}{set default for option \code{echo}, see details below.}
  \item{eval}{set default for option \code{eval}, see details below.}
  \item{split}{set default for option \code{split}, see details below.}
  \item{pdf}{set default for option \code{pdf}, see details below.}
  \item{eps}{set default for option \code{eps}, see details below.}
}
\section{Supported Options}{
  RweaveLatex supports the following options for code chunks (the values
  in parentheses show the default values):
  \describe{
    \item{echo:}{logical (\code{TRUE}). Include S code in the
      output file?}
    \item{eval:}{logical (\code{TRUE}). If \code{FALSE}, the code chunk is not
      evaluated, and hence no text or graphical output produced.}
    \item{results:}{character string (\code{verbatim}).
      If \code{verbatim}, the output of S commands is
      included in the verbatim-like Soutput environment. If
      \code{tex}, the output is taken to be already proper latex markup
      and included as is. If \code{hide} then all output is
      completely suppressed (but the code executed during the weave).}
    \item{print:}{logical (\code{FALSE})
      If \code{TRUE}, each expression in the
      code chunk is wrapped into a \code{print()} statement before evaluation,
      such that the values of all expressions become visible.}
    \item{term:}{logical (\code{TRUE}). If \code{TRUE}, visibility of values
      emulates an interactive R session: values of assignments are not
      printed, values of single objects are printed. If \code{FALSE},
      output comes only from explicit \code{\link{print}} or
      \code{\link{cat}} statements.}
    \item{split:}{logical (\code{FALSE}). If \code{TRUE}, text output is
      written to separate files 
      for each code chunk.}
    \item{strip.white:}{logical (\code{TRUE}). If \code{TRUE}, blank
      lines at the beginning and end of output are removed.}
    \item{prefix:}{logical (\code{TRUE}). If \code{TRUE} generated filenames of
      figures and output have a common prefix.}
    \item{prefix.string:}{a character string, default is the name of the
      \file{.Snw} source file.}
    \item{include:}{logical (\code{TRUE}), indicating whether input
      statements for text output and includegraphics statements
      for figures should be auto-generated. Use \code{include=FALSE} if
      the output should appear in a different place than the code chunk
      (by placing the input line manually).}
    \item{fig:}{logical (\code{FALSE}), indicating whether the code
      chunk produces 
      graphical output. Note that only one figure per code chunk can be
      processed this way.}
    \item{eps:}{logical (\code{TRUE}), indicating whether EPS figures shall be
      generated. Ignored if \code{fig=FALSE}.} 
    \item{pdf:}{logical (\code{TRUE}), indicating whether PDF figures shall be
      generated. Ignored if \code{fig=FALSE}.}
    \item{width:}{numeric (6), width of figures in inch.}
    \item{height:}{numeric (6), height of figures in inch.}
  }
}
\author{Friedrich Leisch}
\references{
  Friedrich Leisch: Sweave User Manual, 2002\cr
  \url{http://www.ci.tuwien.ac.at/~leisch/Sweave}
}
\seealso{\code{\link{Sweave}}, \code{\link{Rtangle}}}
\keyword{utilities}
\eof
\name{Sweave}
\alias{Sweave}
\alias{Stangle}
\alias{SweaveSyntaxLatex}
\alias{SweaveSyntaxNoweb}
\title{Automatic Generation of Reports}
\description{
  \code{Sweave} provides a flexible framework for mixing text and S code
  for automatic report generation. The basic idea is to replace the S
  code with its output, such that the final document only contains the
  text and the output of the statistical anlysis. 
}
\usage{
Sweave(file, driver=RweaveLatex(), syntax=getOption("SweaveSyntax"), ...)
Stangle(file, driver=Rtangle(), syntax=getOption("SweaveSyntax"), ...)
}
\arguments{
  \item{file}{Name of Sweave source file.}
  \item{driver}{The actual workhorse, see details below.}
  \item{syntax}{An object of class \code{SweaveSyntax} or a character
    string with its name. The default installation provides
    \code{SweaveSyntaxNoweb} and \code{SweaveSyntaxLatex}.}
  \item{\dots}{Further arguments passed to the driver's setup function.}
}
\details{
  Automatic generation of reports by mixing word processing markup (like
  latex) and S code. The S code gets replaced by its output (text or
  graphs) in the final markup file. This allows to re-generate a report
  if the input data change and documents the code to reproduce the
  analysis in the same file that also produces the report.

  \code{Sweave} combines the documentation and code chunks together
  (or their output) into a single document. \code{Stangle} extracts only
  the code from the Sweave file creating a valid S source file (that can
  be run using \code{\link{source}}). Code inside \code{\\Sexpr\{\}}
  statements is ignored by \code{Stangle}.

  \code{Stangle} is just a
  frontend to \code{Sweave} using a simple driver by default, which
  discards the documentation and concatenates all code chunks the
  current S engine understands.
}   
\section{Hook Functions}{
  Before each code chunk is evaluated, a number of hook functions can be
  executed. If \code{getOption("SweaveHooks")} is set,
  it is taken to be a collection of hook functions. For each logical
  option of a code chunk (\code{echo}, \code{print}, \ldots) a hook can
  be specified, which is executed if and only if the respective option
  is \code{TRUE}. Hooks must be named elements of the list returned by
  \code{getOption("SweaveHooks")} and be functions taking no
  arguments. E.g., if option
  \code{"SweaveHooks"} is defined as \code{list(fig = foo)}, and
  \code{foo} is a function, then
  it would be executed before the code in each
  figure chunk. This is especially useful to set defaults for the
  graphical parameters in a series of figure chunks.

  Note that the user is free to define new Sweave options and associate
  arbitrary hooks with them. E.g., one could define a hook function for
  option \code{clean} that removes all objects in the global
  environment. Then all code chunks with \code{clean=TRUE} would start
  operating on an empty workspace.
}
\section{Syntax Definition}{
  Sweave allows a very flexible syntax framework for marking
  documentation and text chunks. The default is a noweb-style syntax, as
  alternative a latex-style syntax can be used. See the user manual for
  details.
}
\author{Friedrich Leisch}
\references{
  Friedrich Leisch: Sweave User Manual, 2002\cr
  \url{http://www.ci.tuwien.ac.at/~leisch/Sweave}

  Friedrich Leisch: Dynamic generation of statistical reports using
  literate data analysis. In W. Hrdle and B. Rnz,
  editors, Compstat 2002 - Proceedings in Computational
  Statistics, pages 575-580. Physika Verlag, Heidelberg, Germany,
  2002. ISBN 3-7908-1517-9.
}
\seealso{\code{\link{RweaveLatex}}, \code{\link{Rtangle}}}
\examples{
testfile <- system.file("Sweave", "Sweave-test-1.Rnw", package = "utils")

## create a LaTeX file
Sweave(testfile)

## create an S source file from the code chunks
Stangle(testfile)
## which can be simply sourced
source("Sweave-test-1.R")

\dontshow{if(!interactive()) unlink("Sweave-test-1*")}
}
\keyword{utilities}
\eof
\name{SweaveSyntConv}
\alias{SweaveSyntConv}
\title{Convert Sweave Syntax}
\description{
  This function converts the syntax of files in \code{\link{Sweave}}
  format to another Sweave syntax definition. 
}
\usage{
SweaveSyntConv(file, syntax, output=NULL)
}
\arguments{
  \item{file}{Name of Sweave source file.}
  \item{syntax}{An object of class \code{SweaveSyntax} or a character
    string with its name giving the target syntax to which the file is
    converted.}
  \item{output}{Name of output file, default is to remove the extension
    from the input file and to add the default extension of the target
    syntax. Any directory names in \code{file} are also removed such
    that the output is created in the current working directory.}}
\author{Friedrich Leisch}
\references{
  Friedrich Leisch: Sweave User Manual, 2002\cr
  \url{http://www.ci.tuwien.ac.at/~leisch/Sweave}
}
\seealso{\code{\link{RweaveLatex}}, \code{\link{Rtangle}}}
\examples{
testfile <- system.file("Sweave", "Sweave-test-1.Rnw", package = "utils")

## convert the file to latex syntax
SweaveSyntConv(testfile, SweaveSyntaxLatex)

## and run it through Sweave
Sweave("Sweave-test-1.Stex")

\dontshow{if(!interactive()) unlink("Sweave-test-1*")}
}
\keyword{utilities}
\eof
\name{SweaveUtils}
\alias{SweaveUtils}
\alias{RtangleWritedoc}
\alias{RweaveLatexOptions}
\alias{RweaveChunkPrefix}
\alias{SweaveHooks}
\alias{RweaveTryStop}
\alias{RweaveEvalWithOpt}
\title{Sweave Driver Utilities}
\description{
  These functions are handy for writing Sweave drivers and currently not
  documented. Look at the source code of the Sweave Latex driver (in this
  package) or the HTML driver (in the R2HTML package from CRAN) to see
  how they can be used.
}
\usage{
RtangleWritedoc(object, chunk)
RweaveChunkPrefix(options)
RweaveEvalWithOpt(expr, options)
RweaveLatexOptions(options)
RweaveTryStop(err, options)
SweaveHooks(options, run = FALSE, envir = .GlobalEnv)
}
\author{Friedrich Leisch}
\keyword{utilities}
\keyword{internal}
\eof
\name{apropos}
\title{Find Objects by (Partial) Name}
\usage{
apropos(what, where = FALSE, mode = "any")

find(what, mode = "any", numeric. = FALSE, simple.words = TRUE)
}
\alias{apropos}
\alias{find}
\arguments{
  \item{what}{name of an object, or \link{regular expression} to match against}
  \item{where, numeric.}{a logical indicating whether positions in the
    search list should also be returned}
  \item{mode}{character; if not \code{"any"}, only objects who's
    \code{\link{mode}} equals \code{mode} are searched.}
  \item{simple.words}{logical; if \code{TRUE}, the \code{what} argument is
    only searched as whole word.}
}
\description{
  \code{apropos} returns a character vector giving the names of
  all objects in the search list matching \code{what}.

  \code{find} is a different user interface to the same task as \code{apropos}.
}
\details{
  If \code{mode != "any"} only those objects which are of mode \code{mode}
  are considered.
  If \code{where} is \code{TRUE}, the positions in the search list are
  returned as the names attribute.

  \code{find} is a different user interface for the same task as \code{apropos}.
  However, by default (\code{simple.words == TRUE}), only full words are
  searched with \code{grep(fixed = TRUE)}.
}
\author{Kurt Hornik and Martin Maechler (May 1997).}
\seealso{
  \code{\link{objects}} for listing objects from one place,
  \code{\link{help.search}} for searching the help system,
  \code{\link{search}} for the search path.
}
\examples{
%% some of these have enormous output that varies a lot by version
\dontrun{apropos("lm")}
apropos(ls)
apropos("lq")

cor <- 1:pi
find(cor)        #> ".GlobalEnv"   "package:stats"
find(cor, num=TRUE) # numbers with these names
find(cor, num=TRUE, mode="function")# only the second one
rm(cor)

\dontrun{apropos(".", mode="list") # a long list}

# need a DOUBLE backslash '\\\\' (in case you don't see it anymore)
apropos("\\\\[")

\dontrun{# everything % not diff-able
length(apropos("."))

# those starting with 'pr'
apropos("^pr")

# the 1-letter things
apropos("^.$")
# the 1-2-letter things
apropos("^..?$")
# the 2-to-4 letter things
apropos("^.{2,4}$")

# the 8-and-more letter things
apropos("^.{8,}$")
table(nchar(apropos("^.{8,}$")))
}}
\keyword{data}
\keyword{documentation}
\keyword{environment}
\eof
\name{browseEnv}
\alias{browseEnv}
\alias{wsbrowser}
\title{Browse Objects in Environment}
\description{
  The \code{browseEnv} function opens a browser with list of objects
  currently in \code{sys.frame()} environment.
}
\usage{
browseEnv(envir = .GlobalEnv, pattern, excludepatt = "^last\\.warning",
          html = .Platform$OS.type != "mac",
          expanded = TRUE, properties = NULL,
          main = NULL, debugMe = FALSE)
}
\arguments{
  \item{envir}{an \code{\link{environment}} the objects of which are to
    be browsed.}
  \item{pattern}{a \link{regular expression} for object subselection
    is passed to the internal \code{\link{ls}()} call.}
  \item{excludepatt}{a regular expression for \emph{dropping} objects
    with matching names.}
  \item{html}{is used on non Macintosh machines to display the workspace
    on a HTML page in your favorite browser.}
  \item{expanded}{whether to show one level of recursion.  It can be useful
    to switch it to \code{FALSE} if your workspace is large.  This
    option is ignored if \code{html} is set to \code{FALSE}.}
  \item{properties}{a named list of global properties (of the objects chosen)
    to be showed in the browser;  when \code{NULL} (as per default),
    user, date, and machine information is used.}
  \item{main}{a title string to be used in the browser; when \code{NULL}
    (as per default) a title is constructed.}
  \item{debugMe}{logical switch; if true, some diagnostic output is produced.}
}

\details{
  Very experimental code.  Only allows one level of recursion into
  object structures. The HTML version is not dynamic.

  It can be generalized.  See sources
  (\file{..../library/base/R/databrowser.R}) for details.

  \code{wsbrowser()} is currently just an internally used function;
  its argument list will certainly change.

  Most probably, this should rather work through using the \file{tkWidget}
  package (from \url{www.Bioconductor.org}).
}
\seealso{
  \code{\link{str}}, \code{\link{ls}}.
}
\examples{
if(interactive()) {
   ## create some interesting objects :
   ofa <- ordered(4:1)
   ex1 <- expression(1+ 0:9)
   ex3 <- expression(u,v, 1+ 0:9)
   example(factor, echo = FALSE)
   example(table, echo = FALSE)
   example(ftable, echo = FALSE)
   example(lm, echo = FALSE)
   example(str, echo = FALSE)

   ## and browse them:
   browseEnv()

   ## a (simple) function's environment:
   af12 <- approxfun(1:2, 1:2, method = "const")
   browseEnv(envir = environment(af12))
 }
}
\keyword{interface}
\eof
\name{browseURL}
\alias{browseURL}
\title{Load URL into a WWW Browser}
\description{
  Load a given URL into a WWW browser.
}
\usage{
browseURL(url, browser = getOption("browser"))
}
\arguments{
  \item{url}{a non-empty character string giving the URL to be loaded.}
  \item{browser}{a non-empty character string giving the name of the
    program to be used as hypertext browser.  It should be in the PATH,
    or a full path specified.
#ifdef windows
    
    Under Windows \code{NULL} is also allowed (and is the default), and
    implies that the file association mechanism will be used.
#endif
  }
}
#ifdef unix
\details{
  If \code{browser} supports remote control and \R knows how to perform
  it, the URL is opened in any already running browser or a new one if
  necessary.  This mechanism currently is available for browsers which
  support the \code{"-remote openURL(...)"} interface (which includes
  Netscape 4.x, 6.2.x (but not 6.0/1), 7.1, Opera 5/6, Mozilla >= 0.9.5
  and Mozilla Firefox),
  Galeon, KDE konqueror (via kfmclient) and the GNOME interface to
  Mozilla.  Netscape 7.0 and Opera 7 behave slightly differently,
  and you will need
  to open them first.  Note that the type of browser is determined from
  its name, so this mechanism will only be used if the browser is
  installed under its canonical name.

  Because \code{"-remote"} will use any browser displaying on the X
  server (whatever machine it is running on), the remote control
  mechanism is only used if \code{DISPLAY} points to the local host.
  This may not allow displaying more than one URL at a time from a
  remote host.
}
#endif
#ifdef windows
\examples{
\dontrun{browseURL("http://www.r-project.org")
browseURL("file://c|/R/rw1060/doc/html/index.html",
          browser="C:/Program Files/Netscape/Netscape/netscp.exe")
}}
#endif
\keyword{file}
\eof
\name{bug.report}
\alias{bug.report}
\title{Send a Bug Report}
\usage{
bug.report(subject = "", ccaddress = Sys.getenv("USER"),
           method = getOption("mailer"), address = "r-bugs@r-project.org",
#ifdef windows
           file = "R.bug.report", wait = TRUE)
#endif
#ifdef unix
           file = "R.bug.report")
#endif
}
\arguments{
  \item{subject}{Subject of the email. Please do not use single quotes
    (\code{'}) in the subject! File separate bug reports for multiple bugs}
  \item{ccaddress}{Optional email address for copies (default is current
    user).  Use \code{ccaddress = FALSE} for no copies.}
  \item{method}{Submission method, one of \code{"mailx"},
    \code{"gnudoit"}, \code{"none"}, or \code{"ess"}.}
  \item{address}{Recipient's email address.}
  \item{file}{File to use for setting up the email (or storing it when
    method is \code{"none"} or sending mail fails).}
#ifdef windows
  \item{wait}{logical. Should \R wait for the editor to return?}
#endif
}
\description{
  Invokes an editor to write a bug report and optionally mail it to the
  automated r-bugs repository at \email{r-bugs@r-project.org}.  Some standard
  information on the current version and configuration of \R are 
  included automatically.
}
\details{
  Currently direct submission of bug reports works only on Unix systems.
  If the submission method is \code{"mailx"}, then the default editor is
  used to write the bug report. Which editor is used can be controlled
  using \code{\link{options}}, type \code{getOption("editor")} to see what
  editor is currently defined. Please use the help pages of the
  respective editor for details of usage. After saving the bug report
  (in the temporary file opened) and exiting the editor
  the report is mailed using a Unix command line mail utility such as
  \code{mailx}.  A copy of the mail is sent to the current user.

  If method is \code{"gnudoit"}, then an emacs mail buffer is opened
  and used for sending the email.

  If method is \code{"none"} or \code{NULL} (which is the default on
  Windows systems), then only an editor is opened to help writing the
  bug report.  The report can then be copied to your favorite email
  program and be sent to the r-bugs list.

  If method is \code{"ess"} the body of the mail is simply sent to
  stdout.
}
\value{Nothing useful.}
\section{When is there a bug?}{
  If \R executes an illegal instruction, or dies with an operating
  system error message that indicates a problem in the program (as
  opposed to something like \dQuote{disk full}), then it is certainly a
  bug.

  Taking forever to complete a command can be a bug, but you must make
  certain that it was really \R's fault.  Some commands simply take a
  long time.  If the input was such that you KNOW it should have been
  processed quickly, report a bug.  If you don't know whether the
  command should take a long time, find out by looking in the manual or
  by asking for assistance.

  If a command you are familiar with causes an \R error message in a
  case where its usual definition ought to be reasonable, it is probably
  a bug.  If a command does the wrong thing, that is a bug.  But be sure
  you know for certain what it ought to have done.  If you aren't
  familiar with the command, or don't know for certain how the command
  is supposed to work, then it might actually be working right.  Rather
  than jumping to conclusions, show the problem to someone who knows for
  certain.

   Finally, a command's intended definition may not be best for
   statistical analysis.  This is a very important sort of problem, but
   it is also a matter of judgment.  Also, it is easy to come to such a
   conclusion out of ignorance of some of the existing features.  It is
   probably best not to complain about such a problem until you have
   checked the documentation in the usual ways, feel confident that you
   understand it, and know for certain that what you want is not
   available. The mailing list \code{r-devel@r-project.org} is a better
   place for discussions of this sort than the bug list.

   If you are not sure what the command is supposed to do
   after a careful reading of the manual this indicates a bug in the
   manual.  The manual's job is to make everything clear.  It is just as
   important to report documentation bugs as program bugs.

   If the online argument list of a function disagrees with the manual,
   one of them must be wrong, so report the bug.
}
\section{How to report a bug}{
  When you decide that there is a bug, it is important to report it and
  to report it in a way which is useful.  What is most useful is an
  exact description of what commands you type, from when you start \R
  until the problem happens.  Always include the version of \R, machine,
  and operating system that you are using; type \kbd{version} in \R to
  print this.  To help us keep track of which bugs have been fixed and
  which are still open please send a separate report for each bug.

  The most important principle in reporting a bug is to report FACTS,
  not hypotheses or categorizations.  It is always easier to report the
  facts, but people seem to prefer to strain to posit explanations and
  report them instead.  If the explanations are based on guesses about
  how \R is implemented, they will be useless; we will have to try to 
  figure out what the facts must have been to lead to such
  speculations.  Sometimes this is impossible.  But in any case, it is
  unnecessary work for us.

  For example, suppose that on a data set which you know to be quite
  large the command \code{data.frame(x, y, z, monday, tuesday)} never
  returns.  Do not report that \code{data.frame()} fails for large data
  sets.  Perhaps it fails when a variable name is a day of the week.  If
  this is so then when we got your report we would try out the
  \code{data.frame()} command on a large data set, probably with no day
  of the week variable name, and not see any problem. There is no way in
  the world that we could guess that we should try a day of the week
  variable name.

  Or perhaps the command fails because the last command you used was a
  \code{[} method that had a bug causing \R's internal data structures
  to be corrupted and making the \code{data.frame()} command fail from
  then on.  This is why we need to know what other commands you have
  typed (or read from your startup file).

  It is very useful to try and find simple examples that produce
  apparently the same bug, and somewhat useful to find simple examples
  that might be expected to produce the bug but actually do not.  If you
  want to debug the problem and find exactly what caused it, that is
  wonderful.  You should still report the facts as well as any
  explanations or solutions.

  Invoking \R with the \option{--vanilla} option may help in isolating a
  bug.  This ensures that the site profile and saved data files are not
  read.

  A bug report can be generated using the
  \code{bug.report()} function.  This automatically includes the version
  information and sends the bug to the correct address.  Alternatively
  the bug report can be emailed to \email{r-bugs@r-project.org} or
  submitted to the Web page at \url{http://bugs.r-project.org}.

  Bug reports on \strong{contributed packages} should perhaps be sent to the
  package maintainer rather than to r-bugs.
}
\seealso{R FAQ} 
\author{This help page is adapted from the Emacs manual and the R FAQ}
\keyword{utilities}
\keyword{error}
\eof
\name{capture.output}
\alias{capture.output}
\title{Send output to a character string or file}
\description{
  Evaluates its arguments with the output being returned as a character
  string or sent to a file. Related to \code{\link{sink}} in the same
  way that \code{\link{with}} is related to \code{\link{attach}}.
}
\usage{
capture.output(\dots, file = NULL, append = FALSE)
}
\arguments{
  \item{\dots}{Expressions to be evaluated}
  \item{file}{A file name or a connection, or \code{NULL} to return
    the output as a string. If the connection is not open it will be
    opened and then closed on exit.}
  \item{append}{Append or overwrite the file? }
}
\value{
 A character string, or \code{NULL} if a \code{file} argument was
 supplied.  
}

\seealso{ \code{\link{sink}}, \code{\link{textConnection}} }

\examples{
require(stats)
glmout <- capture.output(example(glm))
glmout[1:5]
capture.output(1+1, 2+2)
capture.output({1+1; 2+2})
\dontrun{
## on Unix with enscript available
ps <- pipe("enscript -o tempout.ps","w")
capture.output(example(glm), file=ps)
close(ps)
}
}
\keyword{utilities}
\eof
\name{close.socket}
\title{Close a Socket}
\usage{
close.socket(socket, \dots)
}
\alias{close.socket}
\arguments{
  \item{socket}{A \code{socket} object}
  \item{\dots}{further arguments passed to or from other methods.}
}
\description{
  Closes the socket and frees the space in the file descriptor table.  The
  port may not be freed immediately.
}
\value{
  logical indicating success or failure
}
\author{Thomas Lumley}
\seealso{
  \code{\link{make.socket}}, \code{\link{read.socket}}
}

\keyword{misc}
\eof
\name{dataentry}
\title{Spreadsheet Interface for Entering Data}
\usage{
data.entry(\dots, Modes = NULL, Names = NULL)
dataentry(data, modes)
de(\dots, Modes = list(), Names = NULL)
}
\alias{data.entry}
\alias{dataentry}
\alias{de}
\alias{de.ncols}
\alias{de.restore}
\alias{de.setup}
\description{
  A spreadsheet-like editor for entering or editing data.
}
\arguments{
  \item{\dots}{A list of variables: currently these should be numeric or
    character vectors or list containing such vectors.}
  \item{Modes}{The modes to be used for the variables.}
  \item{Names}{The names to be used for the variables.}
  \item{data}{A list of numeric and/or character vectors.}
  \item{modes}{A list of length up to that of \code{data} giving the
    modes of (some of) the variables. \code{list()} is allowed.}
}
\details{
  The data entry editor is only available on some platforms and GUIs.
  Where available it provides a means to visually edit a matrix or
  a collection of variables (including a data frame) as described in the
  \dQuote{Notes} section.
  
  \code{data.entry} has side
  effects, any changes made in the spreadsheet are reflected in the
  variables.  The functions \code{de}, \code{de.ncols}, \code{de.setup}
  and \code{de.restore} are designed to help achieve these side effects.
  If the user passes in a matrix, \code{X} say, then the matrix is
  broken into columns before \code{dataentry} is called. Then on return
  the columns are collected and glued back together and the result
  assigned to the variable \code{X}.  If you don't want this behaviour
  use dataentry directly.

  The primitive function is \code{dataentry}. It takes a list of
  vectors of possibly different lengths and modes (the second argument)
  and opens a spreadsheet with these variables being the columns.
  The columns of the dataentry window are returned as vectors in a
  list when the spreadsheet is closed.

  \code{de.ncols} counts the number of columns which are supplied as arguments
  to \code{data.entry}. It attempts to count columns in lists, matrices
  and vectors.  \code{de.setup} sets things up so that on return the
  columns can be regrouped and reassigned to the correct name. This
  is handled by \code{de.restore}.
}
\value{
  \code{de} and \code{dataentry} return the edited value of their
  arguments. \code{data.entry} invisibly returns a vector of variable
  names but its main value is its side effect of assigning new version
  of those variables in the user's workspace.
}
\note{
  The details of interface to the data grid may differ by platform and
  GUI.  The following description applies to
#ifdef unix
  the X11-based implementation under Unix.
#endif
#ifdef windows
  the GraphApp-based implementation under Windows.
#endif

  You can navigate around the grid using the cursor keys or by clicking
  with the (left) mouse button on any cell.  The active cell is
  highlighted by thickening the surrounding rectangle.  Moving to the
  right or down will scroll the grid as needed: there is no constraint
  to the rows or columns currently in use.

  There are alternative ways to navigate using the keys.  Return and
  (keypad) Enter and LineFeed all move down. Tab moves right and
  Shift-Tab move left.  Home moves to the top left.

  PageDown or Control-F moves down a page, and PageUp or
  Control-B up by a page.  End will show the last used column and the
  last few rows used (in any column).

  Using any other key starts an editing process on the currently
  selected cell: moving away from that cell enters the edited value
  whereas Esc cancels the edit and restores the previous value.  When
  the editing process starts the cell is cleared.
#ifdef windows
  The cursor changes to an I-beam to indicate that the cell is in enter mode.
#endif
  In numerical columns
  (the default) only letters making up a valid number (including
  \code{-.eE}) are accepted, and entering an invalid edited value (such
  as blank) enters \code{NA} in that cell.  The last entered value can
  be deleted using the  BackSpace or Del(ete) key.  Only a limited
  number of characters (currently 29) can be entered in a cell, and if
  necessary only the start or end of the string will be displayed, with the
  omissions indicated by \code{>} or \code{<}.  (The start is shown
  except when editing.)

#ifdef windows
  Double-clicking on a cell selects the cell and makes it into an
  editable field (a cursor will appear at the end of the text and it
  will change to the text highlight colour).  The edited text is
  entered by selecting another cell, for example by hitting Return.
  There is no way to cancel the edits.  The field will be expanded to
  the right if necessary to accommodate existing long strings, so it is
  preferable not to edit in the right-most displayed column.  (The
  editable field is itself scrollable.)
#endif

  Entering a value in a cell further down a column than the last used
  cell extends the variable and fills the gap (if any) by \code{NA}s (not
  shown on screen).

  The column names can only be selected by clicking in them.  This gives
  a popup menu to select the column type (currently Real (numeric) or
  Character) or to change the name.  Changing the type converts the
  current contents of the column (and converting from Character to Real
  may generate \code{NA}s.)
#ifdef unix
  If changing the name is selected the
  header cell becomes editable (and is cleared).  As with all cells, the
  value is entered by moving away from the cell by clicking elsewhere or
  by any of the keys for moving down (only).
#endif
#ifdef windows
  Enter the changes made in the popup window by clicking on its close box.
#endif

  New columns are created by entering values in them (and not by just
  assigning a new name).  The mode of the column is auto-detected from
  the first value entered: if this is a valid number it gives a numeric
  column.  Unused columns are ignored, so
  adding data in \code{var5} to a three-column grid adds one extra
  variable, not two.

#ifdef windows
  There is a popup-menu accessed by right-clicking anywhere in the window
  that refers to the currently selected cell. This can copy the value to
  or paste from the clipboard, or paste in common values in that column.
  Copying and pasting can also be accessed by the usual keyboard shortcuts
  Control-C and Control-V.

  Columns can be resized by selecting and dragging a line (the cursor
  will change) within limits: columns must be between 4 and 50 chars wide.
  The Autosize item on the popup menu will resize the currently selected
  column.
#endif
#ifdef unix
  The \code{Copy} button copies the currently selected cell:
  \code{paste} copies the last copied value to the current cell, and
  right-clicking selects a cell \emph{and} copies in the value.
  Initially the value is blank, and attempts to paste a blank value will
  have no effect.
#endif
  
  Control-L will refresh the display, recalculating field widths to fit
  the current entries.

  In the default mode the column widths are chosen to fit the contents
  of each column, with a default of 10 characters for empty columns.
  you can specify fixed column widths by setting option
  \code{de.cellwidth} to the required fixed width (in characters).
  (set it to zero to return to variable widths).  The displayed
  width of any field is limited to
#ifdef unix
  600 pixels (and by the window width).
#endif
#ifdef windows
  50 characters (and by the window width).
#endif
}
\seealso{
  \code{\link{vi}}, \code{\link{edit}}: \code{edit} uses
  \code{dataentry} to edit data frames.
}
\examples{
# call data entry with variables x and y
\dontrun{data.entry(x,y)}
}
\keyword{utilities}
\keyword{file}
\eof
\name{debugger}
\alias{debugger}
\alias{dump.frames}
\title{Post-Mortem Debugging}
\description{
  Functions to dump the evaluation environments (frames) and to examine
  dumped frames.
}
\usage{
dump.frames(dumpto = "last.dump", to.file = FALSE)
debugger(dump = last.dump)
}
\arguments{
  \item{dumpto}{a character string. The name of the object or file to
    dump to.}
  \item{to.file}{logical. Should the dump be to an \R object or to a
    file?}
  \item{dump}{An \R dump object created by \code{dump.frames}.}
}
\details{
  To use post-mortem debugging, set the option \code{error} to be a call
  to \code{dump.frames}.  By default this dumps to an \R object
  \code{"last.dump"} in the workspace, but it can be set to dump to a
  file (as dump of the object produced by a call to \code{\link{save}}).
  The dumped object contain the call stack, the active environments and
  the last error message as returned by \code{\link{geterrmessage}}.

  When dumping to file, \code{dumpto} gives the name of the dumped
  object and the file name has \code{.rda} appended.

  A dump object of class \code{"dump.frames"} can be examined
  by calling \code{debugger}. This will give the error message and a
  list of environments from which to select repeatedly. When an
  environment is selected, it is copied and the \code{browser} called
  from within the copy.

  If \code{dump.frames} is installed as the error handler, execution
  will continue even in non-interactive sessions. See the examples for
  how to dump and then quit.
}
\value{
  None.
}
\note{
  Functions such as \code{\link{sys.parent}} and
  \code{\link{environment}} applied to closures will not work correctly
  inside \code{debugger}.

  Of course post-mortem debugging will not work if \R is too damaged to
  produce and save the dump, for example if it has run out of workspace.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{options}} for setting \code{error} options;
  \code{\link{recover}} is an interactive debugger working similarly to
  \code{debugger} but directly after the error occurs.
}
\examples{
\dontrun{
options(error=quote(dump.frames("testdump", TRUE)))

f <- function() {
    g <- function() stop("test dump.frames")
    g()
}
f()   # will generate a dump on file "testdump.rda"
options(error=NULL)

## possibly in another R session
load("testdump.rda")
debugger(testdump)
Available environments had calls:
1: f()
2: g()
3: stop("test dump.frames")

Enter an environment number, or 0 to exit
Selection: 1
Browsing in the environment with call:
f()
Called from: debugger.look(ind)
Browse[1]> ls()
[1] "g"
Browse[1]> g
function() stop("test dump.frames")
<environment: 759818>
Browse[1]> 
Available environments had calls:
1: f()
2: g()
3: stop("test dump.frames")

Enter an environment number, or 0 to exit
Selection: 0

## A possible setting for non-interactive sessions
options(error=quote({dump.frames(to.file=TRUE); q()}))
}}
\keyword{utilities}
\keyword{error}
\eof
\name{demo}
\alias{demo}
\title{Demonstrations of R Functionality}
\description{
  \code{demo} is a user-friendly interface to running some demonstration
  \R scripts.  \code{demo()} gives the list of available topics.
}
\usage{
demo(topic, device = getOption("device"),
     package = .packages(), lib.loc = NULL,
     character.only = FALSE, verbose = getOption("verbose"))
}
\arguments{
  \item{topic}{the topic which should be demonstrated, given as a
    \link{name} or literal character string, or a character string,
    depending on whether \code{character.only} is \code{FALSE} (default)
    or \code{TRUE}.  If omitted, the list of available topics is
    displayed.}
  \item{device}{the graphics device to be used.}
  \item{package}{a character vector giving the packages to look into for
    demos.  By default, all packages in the search path are used.}
  \item{lib.loc}{a character vector of directory names of \R libraries,
    or \code{NULL}.  The default value of \code{NULL} corresponds to all
    libraries currently known.  If the default is used, the loaded
    packages are searched before the libraries.}
  \item{character.only}{logical; if \code{TRUE}, use \code{topic} as
    character string.}
  \item{verbose}{a logical.  If \code{TRUE}, additional diagnostics are
    printed.}
}
\details{
  If no topics are given, \code{demo} lists the available demos.  The
  corresponding information is returned in an object of class
  \code{"packageIQR"}.  The structure of this class is experimental.  In
  earlier versions of R, an empty character vector was returned along
  with listing available demos.
}
\seealso{\code{\link{source}} which is called by \code{demo}.}
\examples{
demo() # for attached packages

## All available demos:
demo(package = .packages(all.available = TRUE))

demo(lm.glm, package="stats")
\dontrun{
 ch <- "scoping"
 demo(ch, character = TRUE)
}}
\keyword{documentation}
\keyword{utilities}
\eof
\name{download.file}
\alias{download.file}
\title{Download File from the Internet}
\description{
  This function can be used to download a file from the Internet.
}
\usage{
download.file(url, destfile, method, quiet = FALSE, mode="w",
              cacheOK = TRUE)
}
\arguments{
  \item{url}{A character string naming the URL of a resource to be
    downloaded.}

  \item{destfile}{A character string with the name where the downloaded
    file is saved.  Tilde-expansion is performed.}

  \item{method}{Method to be used for downloading files.  Currently
    download methods \code{"internal"}, \code{"wget"} and \code{"lynx"}
    are available.  The default is to choose the first of these which
    will be \code{"internal"}. The method can also be set through the
    option \code{"download.file.method"}: see \code{\link{options}()}.}

  \item{quiet}{If \code{TRUE}, suppress status messages (if any).}

  \item{mode}{character.  The mode with which to write the file. Useful
    values are \code{"w"}, \code{"wb"} (binary), \code{"a"} (append) and
    \code{"ab"}.  Only used for the \code{"internal"} method.}

  \item{cacheOK}{logical. Is a server-side cached value acceptable?
    Implemented for the \code{"internal"} and \code{"wget"} methods.}
}
\details{
  The function \code{download.file} can be used to download a single
  file as described by \code{url} from the internet and store it in
  \code{destfile}.
  The \code{url} must start with a scheme such as
  \code{"http://"}, \code{"ftp://"} or \code{"file://"}.

  \code{cacheOK = FALSE} is useful for \code{"http://"} URLs, and will
  attempt to get a copy directly from the site rather than from an
  intermediate cache.  (Not all platforms support it.)
  It is used by \code{\link{CRAN.packages}}.

  The remaining details apply to method \code{"internal"} only.

  The timeout for many parts of the transfer can be set by the option
  \code{timeout} which defaults to 60 seconds.

  The level of detail provided during transfer can be set by the
  \code{quiet} argument and the \code{internet.info} option.  The
  details depend on the platform and scheme, but setting
  \code{internet.info} to 0 gives all available details, including
  all server responses. Using 2 (the default) gives only serious
  messages, and 3 or more suppresses all messages.

#ifdef windows
  A progress bar tracks the transfer. If the file length is known, the
  full width of the bar is the known length.  Otherwise the initial
  width represents 100Kbytes and is doubled whenever the current width
  is exceeded.

  There is an alternative method if you have Internet Explorer 4 or
  later installed.   You can use the flag \option{--internet2}, when
  the \sQuote{Internet Options} of the system are used to choose proxies
  and so on; these are set in the Control Panel and are those used for
  Internet Explorer.
  This version does not support \code{cacheOK = FALSE}.

#endif
  Method \code{"wget"} can be used with proxy firewalls which require
  user/password authentication if proper values are stored in the
  configuration file for \code{wget}.
}
\note{
  Methods \code{"wget"} and \code{"lynx"} are for historical
  compatibility. They will block all other activity on the \R process.

  For methods \code{"wget"} and \code{"lynx"} a system call is made to
  the tool given by \code{method}, and the respective program must be
  installed on your system and be in the search path for executables.
}
\section{Setting Proxies}{
  This applies to the internal code only.

  Proxies can be specified via environment variables.
  Setting \code{"no_proxy"} stops any proxy being tried.
  Otherwise the setting of \code{"http_proxy"} or \code{"ftp_proxy"}
  (or failing that, the all upper-case version) is consulted and if
  non-empty used as a proxy site.  For FTP transfers, the username
  and password on the proxy can be specified by \code{"ftp_proxy_user"}
  and \code{"ftp_proxy_password"}.  The form of \code{"http_proxy"}
  should be \code{"http://proxy.dom.com/"} or
  \code{"http://proxy.dom.com:8080/"} where the port defaults to
  \code{80} and the trailing slash may be omitted. For
  \code{"ftp_proxy"} use the form \code{"ftp://proxy.dom.com:3128/"}
  where the default port is \code{21}.  These environment variables
  must be set before the download code is first used: they cannot be
  altered later by calling \code{Sys.putenv}.

  Usernames and passwords can be set for HTTP proxy transfers via
  environment variable \env{http\_proxy\_user} in the form
  \code{user:passwd}.  Alternatively, \code{"http_proxy"} can be of the
  form \code{"http://user:pass@proxy.dom.com:8080/"} for compatibility
  with \code{wget}.  Only the HTTP/1.0 basic authentication scheme is
  supported.
#ifdef windows
  Under Windows, if \code{"http_proxy_user"} is set to \code{"ask"} then
  a dialog box will come up for the user to enter the username and
  password.  \bold{NB:} you will be given only one opportunity to enter this,
  but if proxy authentication is required and fails there will be one
  further prompt per download.
#endif
}
\seealso{
  \code{\link{options}} to set the \code{timeout} and
  \code{internet.info} options.

  \code{\link{url}} for a finer-grained way to read data from URLs.

  \code{\link{url.show}}, \code{\link{CRAN.packages}},
  \code{\link{download.packages}} for applications
}
\keyword{utilities}
\eof
\name{edit}
\alias{edit}
\alias{edit.default}
\alias{vi}
\alias{emacs}
\alias{pico}
\alias{xemacs}
\alias{xedit}
\title{Invoke a Text Editor}
\description{
  Invoke a text editor on an \R object.
}
\usage{
\method{edit}{default}(name = NULL, file = "", editor = getOption("editor"), \dots)
vi(name = NULL, file = "")
emacs(name = NULL, file = "")
pico(name = NULL, file = "")
xemacs(name = NULL, file = "")
xedit(name = NULL, file = "")
}
\arguments{
  \item{name}{a named object that you want to edit. If name is missing
    then the file specified by \code{file} is opened for editing.}
  \item{file}{a string naming the file to write the edited version to.}
  \item{editor}{a string naming the text editor you want to use.  On
    Unix the default is set from the environment variables \env{EDITOR}
    or \env{VISUAL} if either is set, otherwise \code{vi} is used.  On
    Windows it defaults to \code{notepad}.}
  \item{\dots}{further arguments to be passed to or from methods.}
}
\details{
  \code{edit} invokes the text editor specified by \code{editor} with
  the object \code{name} to be edited.  It is a generic function,
  currently with a default method and one for data frames and matrices.

  \code{data.entry} can be used to edit data, and is used by \code{edit}
  to edit matrices and data frames on systems for which
  \code{data.entry} is available.

  It is important to realize that \code{edit} does not change the object
  called \code{name}. Instead, a copy of name is made and it is that copy
  which is changed. Should you want the changes to apply to the object
  \code{name} you must assign the result of \code{edit} to
  \code{name}. (Try \code{\link{fix}} if you want to make permanent
  changes to an object.)
  
  In the form \code{edit(name)},
  \code{edit} deparses \code{name} into a temporary file and invokes the
  editor \code{editor} on this file. Quitting from the editor causes
  \code{file} to be parsed and that value returned.
  Should an error occur in parsing, possibly due to incorrect syntax, no
  value is returned. Calling \code{edit()}, with no arguments, will
  result in the temporary file being reopened for further editing.
}
\seealso{
  \code{\link{edit.data.frame}},
  \code{\link{data.entry}},
  \code{\link{fix}}.
}
\note{
  The functions \code{vi}, \code{emacs}, \code{pico}, \code{xemacs},
  \code{xedit} rely on the corresponding editor being available and
  being on the path. This is system-dependent.
}
\examples{
\dontrun{
# use xedit on the function mean and assign the changes
mean <- edit(mean, editor = "xedit")

# use vi on mean and write the result to file mean.out
vi(mean, file = "mean.out")
}
}
\keyword{utilities}
\eof
\name{edit.data.frame}
\alias{edit.data.frame}
\alias{edit.matrix}
\title{Edit Data Frames and Matrices}
\description{
  Use data editor on data frame or matrix contents.
}
\usage{
\method{edit}{data.frame}(name, factor.mode = c("character", "numeric"),
     edit.row.names = any(row.names(name) != 1:nrow(name)), \dots)

\method{edit}{matrix}(name, edit.row.names = any(rownames(name) != 1:nrow(name)), \dots)
}
\arguments{
  \item{name}{A data frame or matrix.}
  \item{factor.mode}{How to handle factors (as integers or using
    character levels) in a data frame.}
  \item{edit.row.names}{logical. Show the row names be displayed as
    a separate editable column?}
  \item{\dots}{further arguments passed to or from other methods.}
}
\details{
  At present, this only works on simple data frames containing numeric, 
  logical or character vectors and factors. Factors are represented in the
  spreadsheet as either numeric vectors (which is more suitable for data
  entry) or character vectors (better for browsing). After editing,
  vectors are padded with \code{NA} to have the same length and
  factor attributes are restored.  The set of factor levels can not be
  changed by editing in numeric mode; invalid levels are changed to
  \code{NA} and a warning is issued.  If new factor levels are
  introduced in character mode, they are added at the end of the list of
  levels in the order in which they encountered.

  It is possible to use the data-editor's facilities to select the mode
  of columns to swap between numerical and factor columns in a data
  frame.  Changing any column in a numerical matrix to character will
  cause the result to be coerced to a character matrix.  Changing 
  the mode of logical columns is not supported.
}
\value{
  The edited data frame.
}
\author{ Peter Dalgaard }
\note{
  \code{fix(dataframe)} works for in-place editing by calling this
  function.

  If the data editor is not available, a dump of the object is presented
  for editing using the default method of \code{edit}.

  At present the data editor is limited to 65535 rows.
}
\seealso{\code{\link{data.entry}}, \code{\link{edit}}}
\examples{
\dontrun{
data(InsectSprays)
edit(InsectSprays)
edit(InsectSprays, factor.mode="numeric")
}
}

\keyword{utilities}
\eof
\name{example}
\alias{example}
\title{Run an Examples Section from the Online Help}
\description{
  Run all the \R code from the \bold{Examples} part of \R's online help
  topic \code{topic} with two possible exceptions, \code{dontrun} and
  \code{dontshow}, see Details below.
}
\usage{
example(topic, package = .packages(), lib.loc = NULL,
        local = FALSE, echo = TRUE, verbose = getOption("verbose"),
        setRNG = FALSE,
        prompt.echo = paste(abbreviate(topic, 6),"> ", sep=""))
}
\arguments{
  \item{topic}{name or literal character string: the online
    \code{\link{help}} topic the examples of which should be run.}
  \item{package}{a character vector with package names.  By default,
    all packages in the search path are used.}
  \item{lib.loc}{a character vector of directory names of \R libraries,
    or \code{NULL}.  The default value of \code{NULL} corresponds to all
    libraries currently known.  If the default is used, the loaded
    packages are searched before the libraries.}
  \item{local}{logical: if \code{TRUE} evaluate locally, if \code{FALSE}
    evaluate in the workspace.}
  \item{echo}{logical;  if \code{TRUE}, show the \R input when sourcing.}
  \item{verbose}{logical;  if \code{TRUE}, show even more when running
    example code.}
  \item{setRNG}{logical or expression;  if not \code{FALSE}, the random
    number generator state is saved, then initialized to a specified state,
    the example is run and the (saved) state is restored.  \code{setRNG
      = TRUE} sets the same state as \code{R CMD \link{check}} does for
    running a package's examples.  This is currently equivalent to
    \code{setRNG = \{RNGkind("default", "default"); set.seed(1)\}}.}
  \item{prompt.echo}{character; gives the prompt to be used if
    \code{echo = TRUE}.}
}
\value{(the value of the last evaluated expression).}
\details{
  If \code{lib.loc} is not specified, the packages are searched for
  amongst those already loaded, then in the specified libraries.
  If \code{lib.loc} is specified, they are searched for only
  in the specified libraries, even if they are already loaded from
  another library.

  An attempt is made to load the package before running the examples,
  but this will not replace a package loaded from another location.

  If \code{local=TRUE} objects are not created in the workspace and so
  not available for examination after \code{example} completes: on the
  other hand they cannot clobber objects of the same name in the
  workspace.

  As detailed in the manual \emph{Writing \R Extensions}, the author of
  the help page can markup parts of the examples for two exception rules
  \describe{
    \item{\code{dontrun}}{encloses code that should not be run.}
    \item{\code{dontshow}}{encloses code that is invisible on help
      pages, but will be run both by the package checking tools,
      and the \code{example()} function.  This was previously
      \code{testonly}, and that form is still accepted.}
    }
}
\note{
  The examples can be many small files.  On some file systems it is
  desirable to save space, and the files in the \file{R-ex} directory
  of an installed package can be zipped up as a zip archive
  \file{Rex.zip}.
}
\author{Martin Maechler and others}
\seealso{\code{\link{demo}}}
\examples{
example(InsectSprays)
## force use of the standard package 'stats':
example("smooth", package="stats", lib.loc=.Library)

## set RNG *before* example as when R CMD check is run:

r1 <- example(quantile, setRNG = TRUE)
x1 <- rnorm(1)
u <- runif(1)
## identical randum numbers
r2 <- example(quantile, setRNG = TRUE)
x2 <- rnorm(1)
stopifnot(identical(r1, r2))
## but x1 and x2 differ since the RNG state from before example()
## differs and is restored!
x1; x2
}
\keyword{documentation}
\keyword{utilities}
\eof
\name{fix}
\alias{fix}
\title{Fix an Object}
\description{
  \code{fix} invokes \code{\link{edit}} on \code{x} and then assigns the new
  (edited) version of \code{x} in the user's workspace.
}
\usage{
fix(x, \dots)
}
\arguments{
  \item{x}{the name of an \R object, as a name or a character string.}
  \item{\dots}{arguments to pass to editor: see \code{\link{edit}}.}
}
\details{
  The name supplied as \code{x} need not exist as an \R object, when a
  function with no arguments and an empty body is supplied for editing.
}
\seealso{
  \code{\link{edit}},
  \code{\link{edit.data.frame}}
}
\examples{
\dontrun{
 ## Assume 'my.fun' is a user defined function :
 fix(my.fun)
 ## now my.fun is changed
 ## Also,
 fix(my.data.frame) # calls up data editor
 fix(my.data.frame, factor.mode="char") # use of ...
}}
\keyword{utilities}
\eof
\name{getAnywhere}
\alias{getAnywhere}
\alias{[.getAnywhere}
\alias{print.getAnywhere}
\title{Retrieve an R Object, Including from a Namespace}
\description{
  This functions locates all objects with name matching its argument,
  whether visible on the search path, registered as an S3 method or in a
  namespace but not exported.
}
\usage{
getAnywhere(x)
}
\arguments{
  \item{x}{a character string or name.}
}
\details{
  The function looks at all loaded namespaces, whether or not they are
  associated with a package on the search list.

  Where functions are found as an S3 method, an attempt is made to find
  which namespace registered them.  This may not be correct, especially
  if a namespace is unloaded.
}
\value{
  An object of class \code{"getAnywhere"}.  This is a list with components
  \item{name}{the name searched for.}
  \item{funs}{a list of objects found}
  \item{where}{a character vector explaining where the object(s) were found}
  \item{visible}{logical: is the object visible}
  \item{dups}{logical: is the object identical to one earlier in the
    list.}

  Normally the structure will be hidden by the \code{print} method.
  There is a \code{[} method to extract one or more of the objects found.
}
\seealso{
  \code{\link{get}},  \code{\link{getFromNamespace}}
}
\examples{
getAnywhere("format.dist")
getAnywhere("simpleLoess") # not exported from stats
}
\keyword{data}
\eof
\name{getFromNamespace}
\alias{assignInNamespace}
\alias{getFromNamespace}
\alias{fixInNamespace}
\title{Utility functions for Developing Namespaces}
\description{
  Utility functions to access and replace the non-exported functions in
  a namespace, for use in developing packages with namespaces.
}
\usage{
getFromNamespace(x, ns, pos = -1, envir = as.environment(pos))
assignInNamespace(x, value, ns, pos = -1, envir = as.environment(pos))
fixInNamespace(x, ns, pos = -1, envir = as.environment(pos), \dots)
}
\arguments{
  \item{x}{an object name (given as a character string).}
  \item{value}{an \R object.}
  \item{ns}{a namespace, or character string giving the namespace.}
  \item{pos}{where to look for the object: see \code{\link{get}}.}
  \item{envir}{an alternative way to specify an environment to look in.}
  \item{\dots}{arguments to pass to the editor: see \code{\link{edit}}.}
}
\details{
  The namespace can be specified in several ways.  Using, for example,
  \code{ns = "stats"} is the most direct, but a loaded package with a
  namespace can be specified via any of the methods used for
  \code{\link{get}}: \code{ns} can also be the environment printed as
  \code{<namespace:foo>}.

  \code{getFromNamespace} is similar to (but predates) the
  \code{\link{:::}} operator, but is more flexible in how the namespace
  is specified.
  
  \code{fixInNamespace} invokes \code{\link{edit}} on the object named
  \code{x} and assigns the revised object in place of the original
  object.  For compatibility with \code{fix}, \code{x} can be unquoted.
}
\note{
  \code{assignInNamespace} and \code{fixInNamespace} change the copy in
  the namespace, but not any copies already exported from the namespace,
  in particular an object of that name in the package (if already
  attached) and any copies already imported into other namespaces.
  They are really intended to be used \emph{only} for objects which are
  not exported from the namespace.  They do attempt to alter a copy
  registered as an S3 method if one is found.
}
\value{
  \code{getFromNamespace} returns the object found (or gives an error).

  \code{assignInNamespace} and \code{fixInNamespace} are invoked for
  their side effect of changing the object in the namespace.
}

\seealso{
  \code{\link{get}}, \code{\link{fix}}, \code{\link{getS3method}}
}
\examples{
getFromNamespace("findGeneric", "utils")
\dontrun{
fixInNamespace("predict.ppr", "stats")
stats:::predict.ppr
getS3method("predict", "ppr")
## alternatively
fixInNamespace("predict.ppr", pos = 3)
fixInNamespace("predict.ppr", pos = "package:stats")
}}
\keyword{data}
\eof
\name{getS3method}
\alias{getS3method}
\title{Get An S3 Method}
\description{
  Get a method for an S3 generic, possibly from a namespace. 
}
\usage{
getS3method(f, class, optional = FALSE)
}
\arguments{
  \item{f}{character: name of the generic.}
  \item{class}{character: name of the class.}
  \item{optional}{logical: should failure to find the generic or a
    method be allowed?}
}
\details{
  S3 methods may be hidden in packages with namespaces, and will not
  then be found by \code{\link{get}}: this function can retrieve
  such functions, primarily for debugging purposes.
}
\value{
  The function found, or \code{NULL} if no function is found and
  \code{optional = TRUE}.
}
\seealso{
  \code{\link{methods}}, \code{\link{get}}
}
\examples{
require(stats)
exists("predict.ppr") # false
getS3method("predict", "ppr")
}
\keyword{data}

\eof
\name{head}
\alias{head}
\alias{head.default}
\alias{head.data.frame}
\alias{head.function}
\alias{head.matrix}
\alias{tail}
\alias{tail.default}
\alias{tail.data.frame}
\alias{tail.function}
\alias{tail.matrix}

\title{
  Return the First or Last Part of an Object
}
\description{
  Returns the first or last parts of a vector, matrix, data frame or function.
}
\usage{
head(x, \dots)
\method{head}{default}(x, n=6, \dots)
\method{head}{data.frame}(x, n=6, \dots)

tail(x, \dots)
\method{tail}{default}(x, n=6, \dots)
\method{tail}{data.frame}(x, n=6, \dots)
}
\arguments{
  \item{x}{an object}
  \item{n}{size for the resulting object: number of elements for a
    vector (including lists), rows for a matrix or data frame or
    lines for a function.}
  \item{\dots}{arguments to be passed to or from other methods.}
}
\details{
  For matrices and data frames, the first/last \code{n} rows are returned.
  For functions, the first/last \code{n} lines of the deparsed function are
  returned as character strings.
}
\value{
  An object (usually) like \code{x} but generally smaller.
}
\author{
  Patrick Burns, improved and corrected by R-core
}

\examples{
data(freeny)
head(freeny.x, n=10)
head(freeny.y)

tail(freeny.x)
tail(freeny.y)

tail(library)
}
\keyword{ manip }
\eof
\name{help}
\alias{help}
\alias{?}
\alias{topicName}               % internal
\alias{.helpForCall}            % internal
\alias{.tryHelp}                % internal
\title{Documentation}
\description{
  These functions provide access to documentation.
  Documentation on a topic with name \code{name} (typically, an \R
  object or a data set) can be printed with either \code{help(name)} or
  \code{?name}.
}
\usage{
help(topic, offline = FALSE, package = .packages(),
     lib.loc = NULL, verbose = getOption("verbose"),
     try.all.packages = getOption("help.try.all.packages"),
#ifdef unix
     htmlhelp = getOption("htmlhelp"),
#endif
#ifdef windows
     chmhelp = getOption("chmhelp"), htmlhelp = getOption("htmlhelp"),
#endif
     pager = getOption("pager"))
?topic
type?topic
}
\arguments{
 \item{topic}{usually, the name on  which documentation is sought.
   The name may be quoted or unquoted (but note that if \code{topic}
   is the name of a
   variable containing a character string  documentation is provided for
   the name, not for the character string).

   The \code{topic} argument may also be a function call, to ask for
   documentation on a corresponding method.  See the section on method
   documentation.
 }
 \item{offline}{a logical indicating whether documentation should be
   displayed on-line to the screen (the default) or hardcopy of it
   should be produced.}
 \item{package}{a name or character vector giving the packages to look
   into for documentation.  By default, all packages in the search path
   are used.}
 \item{lib.loc}{a character vector of directory names of \R libraries,
   or \code{NULL}.  The default value of \code{NULL} corresponds to all
   libraries currently known.  If the default is used, the loaded
   packages are searched before the libraries.}
 \item{verbose}{logical; if \code{TRUE}, the file name is reported.}
 \item{try.all.packages}{logical; see \code{Notes}.}
#ifdef unix
 \item{htmlhelp}{logical (or \code{NULL}).  If \code{TRUE} (which is the
   default after \code{\link{help.start}} has been called), the HTML
   version of the help will be shown in the browser specified by
   \code{options("browser")}.  See \code{\link{browseURL}} for details
   of the browsers that are supported.  Where possible an existing
   browser window is re-used.}
#endif
#ifdef windows
  \item{chmhelp}{logical (or \code{NULL}).  If \code{TRUE} the
    Compiled HTML version of the help will be shown in a help viewer.}
  \item{htmlhelp}{logical (or \code{NULL}).  If \code{TRUE}, the HTML
    version of the help will be shown in a browser specified by
    \code{options("browser")} or the file-association mechanism.
  }
#endif
  \item{pager}{the pager to be used for \code{\link{file.show}}.}
  \item{type}{the special type of documentation to use for this topic;
      for example, if the type is \code{class}, documentation is
      provided for the class with name \code{topic}.  The function
      \code{topicName} returns the actual name used in this case.
    See the section on method documentation for the uses of
      \code{type} to get help on formal methods.}
}
\details{
  In the case of unary and binary operators and control-flow special
  forms (including \code{if}, \code{for} and \code{function}), the topic
  may need to be quoted.

#ifdef unix
  If \code{offline} is \code{TRUE}, hardcopy of the documentation is
  produced by running the LaTeX version of the help page through
  \code{latex} (note that LaTeX 2e is needed) and \code{dvips}.
  Depending on your \code{dvips} configuration, hardcopy will be sent to
  the printer or saved in a file.  If the programs are in non-standard
  locations and hence were not found at compile time, you can either set
  the options \code{latexcmd} and \code{dvipscmd}, or the environment
  variables \env{R\_LATEXCMD} and \env{R\_DVIPSCMD} appropriately.
  The appearance of the output can be customized through a file
  \file{Rhelp.cfg} somewhere in your LaTeX search path.
#endif
#ifdef windows
  If \code{offline} is \code{TRUE}, hardcopy of the documentation is
  produced by running the LaTeX version of the help page through
  \code{latex} (note that LaTeX 2e is needed).  You need to customize
  the file \file{R\_HOME/bin/helpPRINT.bat} which contains an example.
  The appearance of the output can be customized through a file
  \file{Rhelp.cfg} somewhere in your LaTeX search path.
#endif

}
\note{
  Unless \code{lib.loc} is specified explicitly, the loaded packages are
  searched before those in the specified libraries.  This ensures that
  if a library is loaded from a library not in the known library trees,
  then the help from the loaded library is used.  If \code{lib.loc} is
  specified explicitly, the loaded packages are \emph{not} searched.

  If this search fails and argument \code{try.all.packages} is
  \code{TRUE} and neither \code{packages} nor \code{lib.loc} is
  specified, then all the packages in the known library trees are
  searched for help on \code{topic} and a list of (any) packages where
  help may be found is printed (but no help is shown).
  \bold{N.B.} searching all packages can be slow.

  The help files can be many small files.  On some file systems it is
  desirable to save space, and the text files in the \file{help} directory
  of an installed package can be zipped up as a zip archive
  \file{Rhelp.zip}. Ensure that file \file{AnIndex} remains un-zipped.
  Similarly, all the files in the \file{latex} directory can be zipped
  to \file{Rhelp.zip}.
#ifdef windows
  This is done on Windows for the larger packages.
#endif
}

\section{Method Documentation.}{
  The authors of formal (\sQuote{S4}) methods can provide documentation
  on specific methods, as well as overall documentation on the methods
  of a particular function.  The \code{"?"} operator allows access to
  this documentation in three ways.

  The expression \code{methods ? f} will look for the overall
  documentation methods for the function \code{f}.  Currently, this
  means the documentation file containing the alias \code{f-methods}.

  There are two different ways to look for documentation on a
  particular method.  The first is to supply the \code{topic} argument
  in the form of a function call, omitting the \code{type} argument.
  The effect is to look for documentation on the method that would be
  used if this function call were actually evaluated. See the examples
  below.  If the function is not a generic (no S4 methods are defined
  for it), the help reverts to documentation on the function name.

  The \code{"?"} operator can also be called with \code{type} supplied
  as \code{"method"}; in this case also, the \code{topic} argument is
  a function call, but the arguments are now interpreted as specifying
  the class of the argument, not the actual expression that will
  appear in a real call to the function.  See the examples below.

  The first approach will be tedious if the actual call involves
  complicated expressions, and may be slow if the arguments take a
  long time to evaluate.  The second approach avoids these
  difficulties, but you do have to know what the classes of the actual
  arguments will be when they are evaluated.

  Both approaches make use of any inherited methods; the signature of
  the method to be looked up is found by using \code{selectMethod}
 (see the documentation for  \code{\link[methods]{getMethod}}).


}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\seealso{
  \code{\link{help.search}()} for finding help pages on a \dQuote{vague}
  topic;
  \code{\link{help.start}()} which opens the HTML version of the \R
  help pages;
  \code{\link{library}()} for listing available packages and the
  user-level objects they contain;
  \code{\link{data}()} for listing available data sets;
  \code{\link{methods}()}.

  See \code{\link{prompt}()} to get a prototype for writing \code{help}
  pages of private packages.
}
\examples{
help()
help(help)              # the same

help(lapply)
?lapply                 # the same

help("for")             # or ?"for", but the quotes are needed
?"+"

help(package="splines") # get help even when package is not loaded

data()                  # list all available data sets
?women                  # information about data set "women"

topi <- "women"
\dontrun{help(topi) ##--> Error: No documentation for 'topi'}

try(help("bs", try.all.packages=FALSE)) # reports not found (an error)
help("bs", try.all.packages=TRUE) # reports can be found in package 'splines'

\dontshow{require(methods)}

\dontrun{
## define a generic function and some methods
combo <- function(x, y) c(x, y)
setGeneric("combo")
setMethod("combo", c("numeric", "numeric"),
   function(x, y) x+y)

## assume we have written some documentation for combo, and its methods ....

?combo  ## produces the function documentation

methods?combo  ## looks for the overall methods documentation

method?combo("numeric", "numeric")  ## documentation for the method above

?combo(1:10, rnorm(10))  ## ... the same method, selected according to
                         ## the arguments (one integer, the other numeric)

?combo(1:10, letters)    ## documentation for the default method

}

}

\keyword{documentation}
\eof
\name{help.search}
\alias{help.search}
\alias{print.hsearch}
\title{Search the Help System}
\description{
  Allows for searching the help system for documentation matching a
  given character string in the (file) name, alias, title, concept or
  keyword entries (or any combination thereof), using either
  \link{fuzzy matching} or \link{regular expression} matching.  Names
  and titles of the matched help entries are displayed nicely.
}
\usage{
help.search(pattern, fields = c("alias", "concept", "title"),
            apropos, keyword, whatis, ignore.case = TRUE,
            package = NULL, lib.loc = NULL,
            help.db = getOption("help.db"),
            verbose = getOption("verbose"),
            rebuild = FALSE, agrep = NULL)
}
\arguments{
  \item{pattern}{a character string to be matched in the specified
    fields.  If this is given, the arguments \code{apropos},
    \code{keyword}, and \code{whatis} are ignored.}
  \item{fields}{a character vector specifying the fields of the help
    data bases to be searched.  The entries must be abbreviations of
    \code{"name"}, \code{"title"}, \code{"alias"}, \code{"concept"}, and
    \code{"keyword"}, corresponding to the help page's (file) name, its
    title, the topics and concepts it provides documentation for, and
    the keywords it can be classified to.}
  \item{apropos}{a character string to be matched in the help page
    topics and title.}
  \item{keyword}{a character string to be matched in the help page
    \sQuote{keywords}. \sQuote{Keywords} are really categories: the
    standard categories are listed in file \file{RHOME/doc/KEYWORDS}
    (see also the example) and some package writers have defined their
    own.  If \code{keyword} is specified, \code{agrep} defaults to
    \code{FALSE}.}  \item{whatis}{a character string to be matched in
    the help page topics.}
  \item{ignore.case}{a logical.  If \code{TRUE}, case is ignored during
    matching; if \code{FALSE}, pattern matching is case sensitive.}
  \item{package}{a character vector with the names of packages to
    search through, or \code{NULL} in which case \emph{all} available
    packages in the library trees specified by \code{lib.loc} are
    searched.}
  \item{lib.loc}{a character vector describing the location of \R
    library trees to search through, or \code{NULL}.  The default value
    of \code{NULL} corresponds to all libraries currently known.}
  \item{help.db}{a character string giving the file path to a previously
    built and saved help data base, or \code{NULL}.}
  \item{verbose}{logical; if \code{TRUE}, the search process is traced.}
  \item{rebuild}{a logical indicating whether the help data base should
    be rebuilt.}
  \item{agrep}{if \code{NULL} (the default unless \code{keyword} is
    used) and the character string to
    be matched consists of alphanumeric characters, whitespace or a dash
    only, approximate (fuzzy) matching via \code{\link{agrep}} is used
    unless the string has fewer than 5 characters; otherwise, it is
    taken to contain a \link{regular expression} to be matched via
    \code{\link{grep}}.  If \code{FALSE}, approximate matching is not
    used.  Otherwise, one can give a numeric or a list specifying the
    maximal distance for the approximate match, see argument
    \code{max.distance} in the documentation for \code{\link{agrep}}.}
}
\details{
  Upon installation of a package, a contents data base which contains
  the information on name, title, aliases and keywords and, concepts
  starting with \R 1.8.0, is computed from the Rd files in the package and
  serialized as \file{Rd.rds} in the \file{Meta} subdirectory of the
  top-level package installation directory (or, prior to \R 1.7.0, as
  \file{CONTENTS} in Debian Control Format with aliases and keywords
  collapsed to character strings in the top-level package installation
  directory).  This, or a pre-built help.search index serialized as
  \file{hsearch.rds} in the \file{Meta} directory, is the data base
  searched by \code{help.search()}.

  The arguments \code{apropos} and \code{whatis} play a role similar to
  the Unix commands with the same names.

  If possible, the help data base is saved to the file \file{help.db} in
  the \file{.R} subdirectory of the user's home directory or the current
  working directory.

  Note that currently, the aliases in the matching help files are not
  displayed.
}
\value{
  The results are returned in an object of class \code{"hsearch"}, which
  has a print method for nicely displaying the results of the query.
  This mechanism is experimental, and may change in future versions of
  R.
}
\seealso{
  \code{\link{help}};
  \code{\link{help.start}} for starting the hypertext (currently HTML)
  version of \R's online documentation, which offers a similar search
  mechanism.

  \code{\link{apropos}} uses regexps and has nice examples.
}
\examples{
help.search("linear models")    # In case you forgot how to fit linear
                                # models
help.search("non-existent topic")
\dontrun{
help.search("print")            # All help pages with topics or title
                                # matching 'print'
help.search(apropos = "print")  # The same

help.search(keyword = "hplot")  # All help pages documenting high-level
                                # plots.
file.show(file.path(R.home(), "doc", "KEYWORDS"))  # show all keywords

## Help pages with documented topics starting with 'try'.
help.search("\\\\btry", fields = "alias")
## Do not use '^' or '$' when matching aliases or keywords
## (unless all packages were installed using R 1.7 or newer).
}
}
\keyword{documentation}
\eof
\name{help.start}
\title{Hypertext Documentation}
\alias{help.start}
\description{
  Start the hypertext (currently HTML) version of \R's online
  documentation.
}
\usage{
#ifdef unix  
help.start(gui = "irrelevant", browser = getOption("browser"),
           remote = NULL)
#endif
#ifdef windows  
help.start(update = TRUE, gui = "irrelevant",
           browser = getOption("browser"))
#endif
}
\arguments{
  \item{gui}{just for compatibility with S-PLUS.}
  \item{browser}{the name of the program to be used as hypertext
    browser. It should be in the PATH, or a full path specified.
  }
#ifdef unix  
  \item{remote}{A character giving a valid URL for the \file{\$R\_HOME}
    directory on a remote location.}
#endif
#ifdef windows  
  \item{update}{logical: should this attempt to update the indices to
    reflect the currently installed packages.}
#endif
}
\details{
#ifdef unix
  All the packages in the known library trees are linked to directory
  \file{.R} in the per-session temporary directory.  The links are
  re-made each time \code{help.start} is run, which should be done
  after packages are installed, updated or removed.

  If the browser given by the \code{browser} argument is different from
  the default browser as specified by \code{options("browser")}, the
  default is changed to the given browser so that it gets used for all
  future help requests.
#endif
#ifdef windows
  The Windows file association mechanism is used to send the HTML file
  to a browser, launching one if necessary.

  Unlike Unix systems, running \code{help.start} does not send all
  future help requests to the browser: use \code{options(htmlhelp=TRUE)}
  to set that.

  Alternatively, a specific browser can be started if \code{browser} is
  specified, but \code{help} will still use the file association mechanism.

  \code{help.start} and the Java-based search engine have been tested
  under Internet Explorer 6, Netscape 6.2.3/7.0/7.1, Opera 7.23,
  Mozilla 1.6 and MozillaFirefox 0.8.
#endif
}
\note{
  There is a Java-based search facility available from the HTML page
  that \code{help.start} brings up.  Should this not work, please
  consult the \sQuote{R Installation and Administration} manual which is
  linked from that page.
}

\seealso{
  \code{\link{help}()} for on- and off-line help in ASCII/Editor or
  PostScript format.

  \code{\link{browseURL}} for how the help file is displayed.
}
\examples{
\dontrun{help.start()
#ifdef windows
help.start(browser="C:\\\\Program Files\\\\Internet Explorer\\\\IEXPLORE.EXE")
help.start(browser="C:/Program Files/Netscape/Netscape/netscp.exe")
help.start(browser="C:/Program Files/Opera7/Opera.exe")
help.start(browser="C:/Program Files/mozilla.org/Mozilla/mozilla.exe")
help.start(browser="C:/Program Files/Mozilla Firefox/firefox.exe")
#endif
}}
\keyword{documentation}
\eof
\name{make.socket}
\alias{make.socket}
\alias{print.socket}
\title{Create a Socket Connection}
\usage{
make.socket(host = "localhost", port, fail = TRUE, server = FALSE)
}
\arguments{
  \item{host}{name of remote host}
  \item{port}{port to connect to/listen on}
  \item{fail}{failure to connect is an error?}
  \item{server}{a server socket?}
}
\description{
  With \code{server = FALSE} attempts to open a client socket to the
  specified port and host. With \code{server = TRUE} listens on the
  specified port for a connection and then returns a server socket. It is
  a good idea to use \code{\link{on.exit}} to ensure that a socket is
  closed, as you only get 64 of them.
}
\value{
  An object of class \code{"socket"}.
  \item{socket}{socket number. This is for internal use}
  \item{port}{port number of the connection}
  \item{host}{name of remote computer}
}
\author{Thomas Lumley}
\references{
  Adapted from Luke Tierney's code for \code{XLISP-Stat}, in turn
  based on code from Robbins and Robbins "Practical UNIX Programming"
}
\section{Warning}{
  I don't know if the connecting host name returned
  when \code{server = TRUE} can be trusted. I suspect not.
} 

\seealso{
  \code{\link{close.socket}}, \code{\link{read.socket}}
}

\examples{
daytime <- function(host = "localhost"){
    a <- make.socket(host, 13)
    on.exit(close.socket(a))
    read.socket(a)
}
## Offical time (UTC) from US Naval Observatory
\dontrun{daytime("tick.usno.navy.mil")}
}
\keyword{misc}
\eof
\name{menu}
\alias{menu}
\title{Menu Interaction Function}
\description{
  \code{menu} presents the user with a menu of choices labelled from 1
  to the number of choices.  To exit without choosing an item one can
  select \samp{0}.
}
\usage{
menu(choices, graphics = FALSE, title = "")
}
\arguments{
  \item{choices}{a character vector of choices}
  \item{graphics}{a logical indicating whether a graphics menu should be
    used.  Currently unused.}
  \item{title}{a character string to be used as the title of the menu}
}
\value{
  The number corresponding to the selected item, or 0 if no choice was
  made.
}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
\dontrun{
switch(menu(c("List letters", "List LETTERS")) + 1,
       cat("Nothing done\n"), letters, LETTERS)
}
}
\keyword{utilities}
\keyword{programming}
\eof
\name{methods}
\title{List Methods for S3 Generic Functions or Classes}
\alias{methods}
\alias{print.MethodsFunction}
\description{
  List all available methods for an S3 generic function, or all
  methods for a class.
}
\usage{
methods(generic.function, class)
}
\arguments{
  \item{generic.function}{a generic function, or a character string naming a
    generic function.}
  \item{class}{a symbol or character string naming a class: only used if
    \code{generic.function} is not supplied.}
}
\value{
  An object of class \code{"MethodsFunction"}, a
  character vector of function names with an \code{"info"} attribute.
  There is a \code{print} method which marks with an asterisk any
  methods which are not visible: such functions can be examined by
  \code{\link{getS3method}} or \code{\link{getAnywhere}}.

  The \code{"info"} attribute is a data frame, currently with a
  logical column, \code{visible} and a factor column \code{from}
  (indicating where the methods were found).
}
\details{
  Function \code{methods} can be used to find out about the methods for
  a particular generic function or class.  The functions listed are those
  which \emph{are named like methods} and may not actually be methods
  (known exceptions are discarded in the code).  Note that the listed
  methods may not be user-visible objects, but often help will be
  available for them.

  If \code{class} is used, we check that a matching generic can be found
  for each user-visible object named.
}
\note{
  This scheme is called \emph{S3} (S version 3).  For new projects,
  it is recommended to use the more flexible and robust \emph{S4} scheme
  provided in the \pkg{methods} package.  Functions can have both S3
  and S4 methods, and function \code{\link[methods]{showMethods}} will
  list the S4 methods (possibly none).

  The original \code{methods} function was written by Martin Maechler.
}
\seealso{
  \code{\link{S3Methods}}, \code{\link[base]{class}}, \code{\link{getS3method}}
}
\references{
  Chambers, J. M. (1992)
  \emph{Classes and methods: object-oriented programming in S.}
  Appendix A of \emph{Statistical Models in S}
  eds J. M. Chambers and T. J. Hastie, Wadsworth \& Brooks/Cole.
}
\examples{
methods(summary)
methods(class = "aov")
methods("[[")    ##- does not list the C-internal ones...
methods("$")     # currently none
methods("$<-")   # replacement function
methods("+")     # binary operator
methods("Math")  # group generic
\dontrun{methods(print)
}}
\keyword{methods}
\eof
\name{object.size}
\alias{object.size}
\title{Report the Space Allocated for an Object}
\description{
  Provides an estimate of the memory that is being used to store an \R object.
}
\usage{
object.size(x)
}
\arguments{
  \item{x}{An \R object.}
}
\details{
  Exactly which parts of the memory allocation should be attributed
  to which object is not clear-cut.  This function merely provides
  a rough indication.  For example, it will not detect if character
  storage for character strings are shared between identical elements
  (which it will be if \code{rep} was used, for example).

  The calculation is of the size of the object, and excludes the space
  needed to store its name in the symbol table.

}
\value{
  An estimate of the memory allocation attributable to the object, in bytes.
}

\examples{
object.size(letters)
object.size(ls)
## find the 10 largest objects in base
z <- sapply(ls("package:base"), function(x) object.size(get(x, envir=NULL)))
as.matrix(rev(sort(z))[1:10])
}
\keyword{utilities}
\eof
\name{package.skeleton}
\alias{package.skeleton}
\title{Create a skeleton for a new package}
\description{
  \code{package.skeleton} automates some of the setup for a new
  package.  It creates directories, saves functions and data to
  appropriate places, and creates skeleton help files and \file{README}
  files describing further steps in packaging.
}
\usage{
package.skeleton(name = "anRpackage", list, environment = .GlobalEnv, 
	         path = ".", force = FALSE)
}
\arguments{
  \item{name}{character string: the directory name for your package.}
  \item{list}{character vector naming the R objects to put in the package}
  \item{environment}{if \code{list} is omitted, the contents of this
    environment are packaged.}
  \item{path}{path to put the package directories in.}
  \item{force}{If \code{FALSE} will not overwrite an existing directory.}
}
\value{
  used for its side-effects.
}
\details{
  This tries to create filenames valid for all OSes known to run R.
  Invalid characters are replaced by \code{_}, invalid names are
  preceeded by \code{zz}, and finally the converted names are made
  unique by \code{\link{make.unique}(sep = "_")}.  This can be done for
  code and help files but not data files (which are looked for by name).
}

\references{Read the \emph{Writing R Extensions} manual for more details }
\seealso{
  \code{\link{prompt}}, \code{\link{install.packages}}
}

\examples{
\dontrun{
    f <- function(x,y) x+y
    g <- function(x,y) x-y
    d <- data.frame(a=1,b=2)
    e <- rnorm(1000)
    package.skeleton(list=c("f","g","d","e"), name="AnExample")
}
}

\keyword{file}
\keyword{utilities}
\eof
\name{packageStatus}
\title{Package Management Tools}
\alias{newestVersion}
\alias{packageStatus}
\alias{print.packageStatus}
\alias{summary.packageStatus}
\alias{update.packageStatus}
\alias{upgrade.packageStatus}
\alias{upgrade}
\description{Summarize information about installed packages and packages
  available at various repositories, and automatically upgrade outdated
  packages. These tools will replace \code{\link{update.packages}} and friends
  in the future and are currently work in progress.}
\usage{
packageStatus(lib.loc = NULL, repositories = getOption("repositories")())

\method{summary}{packageStatus}(object, \dots)

\method{update}{packageStatus}(object, lib.loc = levels(object$inst$LibPath),
       repositories = levels(object$avail$Repository), \dots)

\method{upgrade}{packageStatus}(object, ask = TRUE, \dots)
}
\arguments{
  \item{lib.loc}{a character vector describing the location of \R
    library trees to search through, or \code{NULL}.  The default value
    of \code{NULL} corresponds to all libraries currently known.}
  \item{repositories}{a character vector of URLs describing the location of \R
    package repositories on the Internet or on the local machine.}
  \item{object}{return value of \code{packageStatus}.}
  \item{ask}{if \code{TRUE}, the user is prompted which packages should
    be upgraded and which not.}
  \item{\dots}{currently not used.}
}
\examples{
\dontrun{
x <- packageStatus()
print(x)
summary(x)
upgrade(x)
x <- update(x)
print(x)
}}
\keyword{utilities}
\eof
\name{page}
\alias{page}
\title{Invoke a Pager on an R Object}
\usage{
page(x, method = c("dput", "print"), \dots)
}
\arguments{
  \item{x}{the name of an \R object.}
  \item{method}{The default method is to dump the object \emph{via}
    \code{\link{dput}}. An alternative is to print to a file.}
  \item{\dots}{additional arguments for \code{\link{file.show}}.
    Intended for setting \code{pager} as \code{title} and
    \code{delete.file} are already used.}
}
\description{
  Displays a representation of the object named by \code{x} in a pager.
}
\seealso{
  \code{\link{file.show}}, \code{\link{edit}}, \code{\link{fix}}.

  To go to a new page when graphing, see \code{\link[graphics]{frame}}.
}
\keyword{utilities}
\eof
\name{prompt}
\title{Produce Prototype of an R Documentation File}
\alias{prompt}
\alias{prompt.default}
\alias{prompt.data.frame}
\description{
  Facilitate the constructing of files documenting \R objects.
}
\usage{
prompt(object, filename = NULL, name = NULL, \dots)
\method{prompt}{default}(object, filename = NULL, name = NULL,
       force.function = FALSE, \dots)
\method{prompt}{data.frame}(object, filename = NULL, name = NULL, \dots)
}
\arguments{
  \item{object}{an \R object, typically a function for the default
    method.}
  \item{filename}{usually, a connection or a character string giving the
    name of the file to which the documentation shell should be written.
    The default corresponds to a file whose name is \code{name} followed
    by \code{".Rd"}.  Can also be \code{NA} (see below).}
  \item{name}{a character string specifying the name of the object.}
  \item{force.function}{a logical.  If \code{TRUE}, treat \code{object}
    as function in any case.}
  \item{\dots}{further arguments passed to or from other methods.}
}
\value{
  If \code{filename} is \code{NA}, a list-style representation of the
  documentation shell.  Otherwise, the name of the file written to is
  returned invisibly.
}
\details{
  Unless \code{filename} is \code{NA}, a documentation shell for
  \code{object} is written to the file specified by \code{filename}, and
  a message about this is given.  For function objects, this shell
  contains the proper function and argument names.  R documentation
  files thus created still need to be edited and moved into the
  \file{man} subdirectory of the package containing the object to be
  documented.

  If \code{filename} is \code{NA}, a list-style representation of the
  documentation shell is created and returned.  Writing the shell to a
  file amounts to \code{cat(unlist(x), file = filename, sep = "\\n")},
  where \code{x} is the list-style representation.

  When \code{prompt} is used in \code{\link{for}} loops or scripts, the
  explicit \code{name} specification will be useful.
}
\note{
  The documentation file produced by \code{prompt.data.frame} does not
  have the same format as many of the data frame documentation files in
  the \pkg{base} package.  We are trying to settle on a preferred
  format for the documentation.
}
\section{Warning}{
  The default filename may not be a valid filename under limited file
  systems (e.g. those on Windows).
  
  Currently, calling \code{prompt} on a non-function object assumes that
  the object is in fact a data set and hence documents it as such.  This
  may change in future versions of \R.  Use \code{\link{promptData}} to
  create documentation skeletons for data sets.
}
\seealso{
  \code{\link{promptData}}, \code{\link{help}} and the chapter on
  \dQuote{Writing \R documentation} in \dQuote{Writing \R Extensions}
  (see the \file{doc/manual} subdirectory of the \R source tree).

  To prompt the user for input, see \code{\link{readline}}.
}
\author{Douglas Bates for \code{prompt.data.frame}}
\references{
  Becker, R. A., Chambers, J. M. and Wilks, A. R. (1988)
  \emph{The New S Language}.
  Wadsworth \& Brooks/Cole.
}
\examples{
require(graphics)
prompt(plot.default)
prompt(interactive, force.function = TRUE)
unlink("plot.default.Rd")
unlink("interactive.Rd")

data(women) # data.frame
prompt(women)
unlink("women.Rd")

data(sunspots, package="stats") # non-data.frame data
prompt(sunspots)
unlink("sunspots.Rd")
}
\keyword{documentation}
\eof
\name{promptData}
\alias{promptData}
\title{Generate a Shell for Documentation of Data Sets}
\description{
  Generates a shell of documentation for a data set.
}
\usage{
promptData(object, filename = NULL, name = NULL)
}
\arguments{
  \item{object}{an \R object to be documented as a data set.}
  \item{filename}{usually, a connection or a character string giving the
    name of the file to which the documentation shell should be written.
    The default corresponds to a file whose name is \code{name} followed
    by \code{".Rd"}.  Can also be \code{NA} (see below).}
  \item{name}{a character string specifying the name of the object.}
}
\value{
  If \code{filename} is \code{NA}, a list-style representation of the
  documentation shell.  Otherwise, the name of the file written to is
  returned invisibly.
}
\details{
  Unless \code{filename} is \code{NA}, a documentation shell for
  \code{object} is written to the file specified by \code{filename}, and
  a message about this is given.

  If \code{filename} is \code{NA}, a list-style representation of the
  documentation shell is created and returned.  Writing the shell to a
  file amounts to \code{cat(unlist(x), file = filename, sep = "\\n")},
  where \code{x} is the list-style representation.
  
  Currently, only data frames are handled explicitly by the code.
}
\section{Warning}{
  This function is still experimental.  Both interface and value might
  change in future versions.  In particular, it may be preferable to use
  a character string naming the data set and optionally a specification
  of where to look for it instead of using \code{object}/\code{name} as
  we currently do.  This would be different from \code{\link{prompt}},
  but consistent with other prompt-style functions in package
  \pkg{methods}, and also allow prompting for data set documentation
  without explicitly having to load the data set.
}
\seealso{\code{\link{prompt}}}
\examples{
data(sunspots, package="stats")
promptData(sunspots)
unlink("sunspots.Rd")
}
\keyword{documentation}
\eof
\name{read.fwf}
\alias{read.fwf}
\title{Read Fixed Width Format Files}
\description{
  Read a \dQuote{table} of \bold{f}ixed \bold{w}idth \bold{f}ormatted
  data into a \code{\link{data.frame}}.
}
\usage{
read.fwf(file, widths, header = FALSE, sep = "\t", as.is = FALSE,
         skip = 0, row.names, col.names, n = -1, \dots)
}
\arguments{
  \item{file}{
    the name of the file which the data are to be read from.

    Alternatively, \code{file} can be a \code{\link{connection}}, which
    will be opened if necessary, and if so closed at the end of the
    function call.
  }
  \item{widths}{integer vector, giving the widths of the fixed-width
    fields (of one line).}
  \item{header}{a logical value indicating whether the file contains the
    names of the variables as its first line.}
  \item{sep}{character; the separator used internally; should be a
    character that does not occur in the file.}
  \item{as.is}{see \code{\link{read.table}}.}
  \item{skip}{number of initial lines to skip; see
    \code{\link{read.table}}.}
  \item{row.names}{see \code{\link{read.table}}.}
  \item{col.names}{see \code{\link{read.table}}.}
  \item{n}{the maximum number of records (lines) to be read, defaulting
    to no limit.}
  \item{\dots}{further arguments to be passed to
    \code{\link{read.table}}.}
}
\value{
  A \code{\link{data.frame}} as produced by \code{\link{read.table}}
  which is called internally.
}
\details{
  Fields that are of zero-width or are wholly beyond the end of the line
  in \code{file} are replaced by \code{NA}.
}
\author{
  Brian Ripley for \R version: original \code{Perl} by Kurt Hornik.
}
\seealso{\code{\link{scan}} and \code{\link{read.table}}.}
\examples{
ff <- tempfile()
cat(file=ff, "123456", "987654", sep="\n")
read.fwf(ff, width=c(1,2,3))    #> 1 23 456 \\ 9 87 654
unlink(ff)
cat(file=ff, "123", "987654", sep="\n")
read.fwf(ff, width=c(1,0, 2,3))    #> 1 NA 23 NA \\ 9 NA 87 654
unlink(ff)
}
\keyword{file}
\keyword{connection}
\eof
\name{read.socket}
\title{ Read from or Write to a Socket}
\usage{
read.socket(socket, maxlen=256, loop=FALSE)
write.socket(socket, string)
}
\alias{read.socket}
\alias{write.socket}
\arguments{
    \item{socket}{a socket object}
    \item{maxlen}{maximum length of string to read}
    \item{loop}{wait for ever if there is nothing to read?}
    \item{string}{string to write to socket}
}
\description{
    \code{read.socket} reads a string from the specified socket,
    \code{write.socket} writes to the specified socket.  There is very
    little error checking done by either.
}
\value{
    \code{read.socket} returns the string read.
}
\author{Thomas Lumley}

\seealso{
    \code{\link{close.socket}}, \code{\link{make.socket}}
}

\examples{
finger <- function(user, host = "localhost", port = 79, print = TRUE)
{
    if (!is.character(user))
        stop("user name must be a string")
    user <- paste(user,"\r\n")
    socket <- make.socket(host, port)
    on.exit(close.socket(socket))
    write.socket(socket, user)
    output <- character(0)
    repeat{
        ss <- read.socket(socket)
        if (ss == "") break
        output <- paste(output, ss)
    }
    close.socket(socket)
    if (print) cat(output)
    invisible(output)
}
\dontrun{finger("root")  ## only works if your site provides a finger daemon}
}
\keyword{misc}
\eof
\name{recover}
\alias{recover}
\alias{limitedLabels}
\title{Browsing after an Error}
\description{
  This function allows the user to browse directly on any of the
  currently active function calls, and is suitable as an error option.
  The expression \code{options(error=recover)} will make this
  the error option.
}
\usage{
recover()
}
\details{
  When called, \code{recover} prints the list of current calls, and
  prompts the user to select one of them.  The standard \R
  \code{\link{browser}} is then invoked from the corresponding
  environment; the user can type ordinary S language expressions to be
  evaluated in that environment.

  When finished browsing in this call, type \code{c} to return to
  \code{recover} from the browser.  Type another frame number to browse
  some more, or type \code{0} to exit \code{recover}.

  The use of \code{recover} largely supersedes \code{\link{dump.frames}}
  as an error option, unless you really want to wait to look at the
  error.  If \code{recover} is called in non-interactive mode, it
  behaves like \code{dump.frames}.  For computations involving large
  amounts of data, \code{recover} has the advantage that it does not
  need to copy out all the environments in order to browse in them.  If
  you do decide to quit interactive debugging, call
  \code{\link{dump.frames}} directly while browsing in any frame (see
  the examples).

  \emph{WARNING}: The special \code{Q} command to go directly from the
  browser to the prompt level of the evaluator currently interacts with
  \code{recover} to effectively turn off the error option for the next
  error (on subsequent errors, \code{recover} will be called normally).
}
\value{
  Nothing useful is returned.  However, you \emph{can} invoke
  \code{recover} directly from a function, rather than through the error
  option shown in the examples.  In this case, execution continues
  after you type \code{0} to exit \code{recover}.
}

\section{Compatibility Note}{
  The \R \code{recover} function can be used in the same way as the
  S-Plus function of the same name; therefore, the error option shown is
  a compatible way to specify the error action.  However, the actual
  functions are essentially unrelated and interact quite differently
  with the user.  The navigating commands \code{up} and \code{down} do
  not exist in the \R version; instead, exit the browser and select
  another frame.
}
\references{
  John M. Chambers (1998).
  \emph{Programming with Data}; Springer. \cr
  See the compatibility note above, however.
}
\seealso{
  \code{\link{browser}} for details about the interactive computations;
  \code{\link{options}} for setting the error option;
  \code{\link{dump.frames}} to save the current environments for later
  debugging.
}
\examples{
\dontrun{

options(error = recover) # setting the error option

### Example of interaction

> myFit <- lm(y ~ x, data = xy, weights = w)
Error in lm.wfit(x, y, w, offset = offset, ...) :
        missing or negative weights not allowed

Enter a frame number, or 0 to exit
1:lm(y ~ x, data = xy, weights = w)
2:lm.wfit(x, y, w, offset = offset, ...)
Selection: 2
Called from: eval(expr, envir, enclos)
Browse[1]> objects() # all the objects in this frame
[1] "method" "n"      "ny"     "offset" "tol"    "w"
[7] "x"      "y"
Browse[1]> w
[1] -0.5013844  1.3112515  0.2939348 -0.8983705 -0.1538642
[6] -0.9772989  0.7888790 -0.1919154 -0.3026882
Browse[1]> dump.frames() # save for offline debugging
Browse[1]> c # exit the browser

Enter a frame number, or 0 to exit
1:lm(y ~ x, data = xy, weights = w)
2:lm.wfit(x, y, w, offset = offset, ...)
Selection: 0 # exit recover
>

}%dontrun
}
\keyword{programming}
\keyword{debugging}
\eof
\name{remove.packages}
\alias{remove.packages}
\title{Remove Installed Packages}
\description{
  Removes installed packages and updates index information as necessary.
}
\usage{
remove.packages(pkgs, lib, version)
}
\arguments{
  \item{pkgs}{a character vector with the names of the packages to be
    removed.}
  \item{lib}{a character string giving the library directory to move the
    packages from.}
  \item{version}{A character string specifying a version of the
    package to remove.  If none is provided, the system will remove an
    unversioned install of the package.}
}
\seealso{
#ifdef unix
  \code{\link{REMOVE}} for a command line version;
#endif
  \code{\link{install.packages}} for installing packages.
}
\keyword{utilities}
\eof
\name{savehistory}
\alias{loadhistory}
\alias{savehistory}
\alias{history}
\title{Load or Save or Display the Commands History}
\description{
  Load or save or display the commands history.
}
\usage{
loadhistory(file = ".Rhistory")
savehistory(file = ".Rhistory")
history(max.show = 25, reverse = FALSE)
}
\arguments{
  \item{file}{The name of the file in which to save the history, or
    from which to load it. The path is relative to the current
    working directory.}
  \item{max.show}{The maximum number of lines to show. \code{Inf} will
    give all of the currently available history.}
  \item{reverse}{logical. If true, the lines are shown in reverse
    order. Note: this is not useful when there are continuation lines.}
}
\details{
#ifdef windows
  This works in Rgui and interactive Rterm but not in batch use of Rterm
  nor in embedded/DCOM versions.
#endif
#ifdef unix
  This works under the \code{readline} and GNOME interfaces, but not if
  \code{readline} is not available (for example, in batch use).
#endif
}
\note{
  If you want to save the history (almost) every session, you can put a
  call to \code{savehistory()} in \code{\link{.Last}}.
}
\examples{\dontrun{
.Last <- function()
    if(interactive()) try(savehistory("~/.Rhistory"))
}}
\keyword{utilities}
\eof
\name{shQuote}
\alias{shQuote}
\title{Quote Strings for Use in OS Shells}
\description{
  Quote a string to be passed to an operating system shell.
}
\usage{
shQuote(string, type = c("sh", "csh", "cmd"))
}
\arguments{
  \item{string}{a character vector, usually of length one.}
  \item{type}{character: the type of shell.  Partial matching is
    supported. \code{"cmd"} refers to the Windows NT shell.}
}
\details{
  The default type of quoting supported is that for the Bourne shell
  \code{sh}.  If the string does not contain single quotes, we can just
  surround it with single quotes.  Otherwise, the string is surrounded in
  double quotes, which suppresses all special meanings of metacharacters
  except dollar, backquote and backslash, so these (and of course double
  quote) are preceded by backslash.  This type of quoting is also
  appropriate for \code{ksh} and \code{bash}.

  The other type of quoting is for the C-shell (\code{csh} and
  \code{tcsh}).  Once again, if the string does not contain single
  quotes, we can just surround it with single quotes.  If it does
  contain single quotes, we can use double quotes provided it does not
  contain dollar or backquote (and we need to escape backslash,
  exclamation mark and double quote).  As a last resort, we need to
  split the string into pieces not containing single quotes and surround
  each with single quotes, and the single quotes with double quotes.

  The Windows shell supports only double quoting.  All this
  implementation does is to surround by double quotes and escape
  internal double quotes by a backslash.
}
\references{
  Loukides, M. et al (2002)
  \emph{Unix Power Tools} Third Edition.  O'Reilly.  Section 27.12.

  \url{http://www.mhuffman.com/notes/dos/bash_cmd.htm}
}
\seealso{
  \code{\link{sQuote}} for quoting in text.
}
\examples{
test <- "abc$def`gh`i\\j"
cat(shQuote(test), "\n")
\dontrun{system(paste("echo", shQuote(test)))}
test <- "don't do it!"
cat(shQuote(test), "\n")

tryit <- "use the `-c' switch\nlike this"
cat(shQuote(tryit), "\n")
\dontrun{system(paste("echo", shQuote(tryit)))}
cat(shQuote(tryit, type="csh"), "\n")

## Windows-only example.
perlcmd <- 'print "Hello World\\n";'
\dontrun{shell(paste("perl -e", shQuote(perlcmd, type="cmd")))}
}
\keyword{utilities}
\eof
\name{str}
\title{Compactly Display the Structure of an Arbitrary R Object}
\usage{
str(object, \dots)

\method{str}{data.frame}(object, \dots)

\method{str}{default}(object, max.level = 0, vec.len = 4, digits.d = 3,
    nchar.max = 128, give.attr = TRUE, give.length = TRUE,
    wid = getOption("width"), nest.lev = 0,
    indent.str = paste(rep(" ", max(0, nest.lev + 1)), collapse = ".."),
    \dots)

 ls.str(pos = 1, pattern, \dots,  envir = as.environment(pos), mode = "any",
        max.level = 1, give.attr = FALSE)
lsf.str(pos = 1, \dots, envir = as.environment(pos))
}
\alias{str}
\alias{str.default}
\alias{str.data.frame}
\alias{ls.str}
\alias{lsf.str}
\description{
  Compactly display the internal \bold{str}ucture of an \R object, a
  \dQuote{diagnostic} function and an alternative to \code{\link{summary}}
  (and to some extent, \code{\link{dput}}).  Ideally, only one line for
  each \dQuote{basic} structure is displayed.  It is especially well suited
  to compactly display the (abbreviated) contents of (possibly nested)
  lists.  The idea is to give reasonable output for \bold{any} \R
  object.  It calls \code{\link{args}} for (non-primitive) function objects.

  \code{\link{ls.str}} and \code{\link{lsf.str}} are useful \dQuote{versions}
  of \code{\link{ls}}, calling \code{str} on each object.  They are not
  foolproof and should rather not be used for programming, but are
  provided for their usefulness.
}
\arguments{
  \item{object}{any \R object about which you want to have some
    information.}
  \item{max.level}{maximal level of nesting which is applied for
    displaying nested structures, e.g., a list containing sub lists.
    Default 0: Display all nesting levels.}
  \item{vec.len}{numeric (>= 0) indicating how many \dQuote{first few} elements
    are displayed of each vector.  The number is multiplied by different
    factors (from .5 to 3) depending on the kind of vector.  Default 4.}
  \item{digits.d}{number of digits for numerical components (as for
    \code{\link{print}}).}
  \item{nchar.max}{maximal number of characters to show for
    \code{\link{character}} strings.  Longer strings are truncated, see
    \code{longch} example below.}
  \item{give.attr}{logical; if \code{TRUE} (default), show attributes
    as sub structures.}
  \item{give.length}{logical; if \code{TRUE} (default), indicate
    length (as \code{[1:\dots]}).}
  \item{wid}{the page width to be used.  The default is the currently
    active \code{\link{options}("width")}.}
  \item{nest.lev}{current nesting level in the recursive calls to
    \code{str}.}
  \item{indent.str}{the indentation string to use.}
  \item{\dots}{potential further arguments (required for Method/Generic reasons).}

%% The following line is swallowed, unfortunately..  fixme??
  For \code{ls.str} and \code{lsf.str}:

  \item{pos}{integer indicating \code{\link{search}} path position.}
  \item{envir}{environment to use, see \code{\link{ls}}.}
  \item{pattern}{a \link{regular expression} passed to \code{\link{ls}}.
    Only names matching \code{pattern} are considered.}
  \item{mode}{character specifying the \code{\link{mode}} of objects to
    consider.  Passed to \code{\link{exists}} and \code{\link{get}}.}
}
\value{
  \code{str} does not return anything, for efficiency reasons.
  The obvious side effect is output to the terminal.

  \code{ls.str} and \code{lsf.str} invisibly return a character vector
  of the matching names, similarly to \code{\link{ls}}.
}
\author{Martin Maechler \email{maechler@stat.math.ethz.ch} since 1990.
}
\seealso{
  \code{\link{summary}}, \code{\link{args}}.
}
\examples{
require(stats)
## The following examples show some of 'str' capabilities
str(1:12)
str(ls)
str(args) #- more useful than  args(args) !
data(freeny); str(freeny)
str(str)
str(.Machine, digits = 20)
str( lsfit(1:9,1:9))
str( lsfit(1:9,1:9),  max = 1)
op <- options(); str(op) #- save first; otherwise internal options() is used.
need.dev <- !exists(".Device") || is.null(.Device)
if(need.dev) postscript()
str(par()); if(need.dev) graphics.off()

ch <- letters[1:12]; is.na(ch) <- 3:5
str(ch) # character NA's

nchar(longch <- paste(rep(letters,100), collapse=""))
str(longch)
str(longch, nchar.max = 52)

lsf.str()#- how do the functions look like which I am using?
ls.str(mode = "list") #- what are the structured objects I have defined?
## which base functions have "file" in their name ?
lsf.str(pos = length(search()), pattern = "file")
\dontshow{
 ##-- Some "crazy" objects
 str(array(1:5, dim=20))
 str(factor(character(0)))
 str(as.data.frame(NULL))
}
}
\keyword{print}
\keyword{documentation}
\keyword{utilities}
\eof
\name{summaryRprof}
\alias{summaryRprof}
\title{Summarise Output of R Profiler}
\description{
Summarise the output of the \code{\link{Rprof}} function to show the
amount of time used by different \R functions.
}
\usage{
summaryRprof(filename = "Rprof.out", chunksize = 5000)
}
\arguments{
  \item{filename}{Name of a file produced by \code{Rprof()}}
  \item{chunksize}{Number of lines to read at a time}
}
\details{
This function is an alternative to \code{R CMD Rprof}. It provides the
convenience of an all-\R implementation but will be slower for large
files.

As the profiling output file could be larger than available memory, it
is read in blocks of \code{chunksize} lines. Increasing \code{chunksize}
will make the function run faster if sufficient memory is available.
}
\value{
  A list with components
  \item{by.self}{Timings sorted by \sQuote{self} time}
  \item{by.total}{Timings sorted by \sQuote{total} time}
  \item{sampling.time}{Total length of profiling run}
}


\seealso{
  The chapter on \dQuote{Tidying and profiling R code} in
  \dQuote{Writing \R Extensions} (see the \file{doc/manual} subdirectory
  of the \R source tree).

  \code{\link{Rprof}}
}
\examples{
\dontrun{
## Rprof() is not available on all platforms
Rprof(tmp <- tempfile())
example(glm)
Rprof()
summaryRprof(tmp)
unlink(tmp)
}
}
\keyword{utilities}
\eof
\name{update.packages}
\alias{update.packages}
\alias{installed.packages}
\alias{CRAN.packages}
\alias{old.packages}
\alias{download.packages}
\alias{install.packages}
\alias{contrib.url}
\title{Download Packages from CRAN}
\description{
  These functions can be used to automatically compare the version
  numbers of installed packages with the newest available version on
  CRAN and update outdated packages on the fly.
}
\usage{
update.packages(lib.loc = NULL, CRAN = getOption("CRAN"),
                contriburl = contrib.url(CRAN),
                method, instlib = NULL,
                ask=TRUE, available=NULL, destdir=NULL,
		installWithVers=FALSE)

installed.packages(lib.loc = NULL, priority = NULL)
CRAN.packages(CRAN = getOption("CRAN"), method,
              contriburl = contrib.url(CRAN))
old.packages(lib.loc = NULL, CRAN = getOption("CRAN"),
             contriburl = contrib.url(CRAN),
             method, available = NULL)

download.packages(pkgs, destdir, available = NULL,
                  CRAN = getOption("CRAN"),
                  contriburl = contrib.url(CRAN), method)
install.packages(pkgs, lib, CRAN = getOption("CRAN"),
                 contriburl = contrib.url(CRAN),
                 method, available = NULL, destdir = NULL,
		 installWithVers = FALSE)
}
\arguments{
  \item{lib.loc}{character vector describing the location of R
    library trees to search through (and update packages therein).}
  \item{CRAN}{character, the base URL of the CRAN mirror to use, i.e.,
    the URL of a CRAN root such as \code{"http://cran.r-project.org"}
    (the default) or its Statlib mirror,
    \code{"http://lib.stat.cmu.edu/R/CRAN"}.}
  \item{contriburl}{URL of the contrib section of CRAN. Use this argument
    only if your CRAN mirror is incomplete, e.g., because you burned
    only the contrib section on a CD.  Overrides argument \code{CRAN}.}
  \item{method}{Download method, see \code{\link{download.file}}.}
  \item{pkgs}{character vector of the short names of
    packages whose current versions should be downloaded from \code{CRAN}.}
  \item{destdir}{directory where downloaded packages are stored.}
  \item{priority}{character vector or \code{NULL} (default).  If
    non-null, used to select packages; \code{"high"} is equivalent to
    \code{c("base","recommended")}.  To select all packages without an
    assigned priority, use \code{priority = "NA"}.}
  \item{available}{list of packages available at CRAN as returned by
    \code{CRAN.packages}.}
  \item{lib,instlib}{character string giving the library directory where to
    install the packages.}
  \item{ask}{logical indicating to ask before packages are actually downloaded
    and installed.}
  \item{installWithVers}{If \code{TRUE}, will invoke the install the
    package such that it can be referenced by package version}
}
\details{
  \code{installed.packages} scans the \file{DESCRIPTION} files of each
  package found along \code{lib.loc} and returns a list of package
  names, library paths and version numbers.  \code{CRAN.packages}
  returns a similar list, but corresponding to packages currently
  available in the contrib section of CRAN, the comprehensive R archive
  network.  The current list of packages is downloaded over the internet
  (or copied from a local CRAN mirror). Both functions use
  \code{\link{read.dcf}} for parsing the description files.
  \code{old.packages} compares the two lists and reports installed
  packages that have newer versions on CRAN.

  \code{download.packages} takes a list of package names and a
  destination directory, downloads the newest versions of the package
  sources and saves them in \code{destdir}.  If the list of available
  packages is not given as argument, it is also directly obtained from
  CRAN.  If CRAN is local, i.e., the URL starts with \code{"file:"},
  then the packages are not downloaded but used directly.

  The main function of the bundle is \code{update.packages}.  First a
  list of all packages found in \code{lib.loc} is created and compared
  with the packages available on CRAN.  Outdated packages are reported
  and for each outdated package the user can specify if it should be
  automatically updated.
#ifdef unix
  If so, the package sources are downloaded from
  CRAN and installed in the respective library path (or \code{instlib}
  if specified) using the \R \code{\link{INSTALL}} mechanism.
#endif
#ifdef windows
  If so, the pre-compiled packages are
  downloaded from CRAN and installed in the respective library path (or
  \code{instlib} if specified).
#endif

  \code{install.packages} can be used to install new packages, it
  takes a vector of package names and a destination library, downloads
  the packages from CRAN and installs them. If the library
  is omitted it defaults to the first directory in \code{.libPaths()}, with
  a warning if there is more than one.
%  The first argument can be an unquoted name, as in
%  \code{install.packages(nlme)}.
#ifdef windows
  Argument \code{pkgs} can also be a character vector of file names of
  zip files if \code{CRAN=NULL}. The zip files are then unpacked directly.
#endif

  For \code{install.packages} and \code{update.packages}, \code{destdir}
  is the directory to which packages will be downloaded. If it is
  \code{NULL} (the default) a temporary directory is used, and the user
  will be given the option of deleting the temporary files once the
  packages are installed.  (They will always be deleted at the end of
  the \R session.)
}
\seealso{
  See \code{\link{download.file}} for how to handle proxies and
  other options to monitor file transfers.

#ifdef unix
  \code{\link{INSTALL}}, \code{\link{REMOVE}},
#endif
  \code{\link{library}}, \code{\link{.packages}}, \code{\link{read.dcf}}
}
\examples{
str(ip <- installed.packages(priority = "high"))
ip[, c(1,3:5)]
}
\keyword{utilities}
\eof
\name{url.show}
\alias{url.show}
\title{Display a text URL}
\description{
  Extension of \code{\link{file.show}} to display text files on a remote 
  server.
}
\usage{
url.show(url, title = url, file = tempfile(),
         delete.file = TRUE, method, \dots)
}
\arguments{
  \item{url}{The URL to read from.}
  \item{title}{Title for the browser.}
  \item{file}{File to copy to.}
  \item{delete.file}{Delete the file afterwards?}
  \item{method}{File transfer method: see \code{\link{download.file}}}
  \item{\dots}{Arguments to pass to \code{\link{file.show}}.}
}
\seealso{
  \code{\link{url}}, \code{\link{file.show}},\code{\link{download.file}}
}
\examples{
\dontrun{url.show("http://lib.stat.cmu.edu/datasets/csb/ch3a.txt")}
}
\keyword{file}
\keyword{misc}
\eof
\name{utils-deprecated}
%------ NOTE:  ../R/utils-deprecated.R   must be synchronized with this!
\alias{utils-deprecated}
\alias{package.contents}
\title{Deprecated Functions in Package utils}
\description{
  These functions are provided for compatibility with older versions of
  \R only, and may be defunct as soon as of the next release.
}
\usage{
package.contents(pkg, lib.loc = NULL)
}
\seealso{
  \code{\link{Deprecated}}, \code{\link{Defunct}}
}
\keyword{internal}
\eof
\name{vignette}
\alias{vignette}
\title{View or List Vignettes}
\description{
  View a specified vignette, or list the available ones.
}
\usage{
vignette(topic, package = NULL, lib.loc = NULL)
}
\arguments{
  \item{topic}{a character string giving the (base) name of the vignette
    to view.}
  \item{package}{a character vector with the names of packages to
    search through, or \code{NULL} in which case \emph{all} available
    packages in the library trees specified by \code{lib.loc} are
    searched.}
  \item{lib.loc}{a character vector of directory names of \R libraries,
    or \code{NULL}.  The default value of \code{NULL} corresponds to all
    libraries currently known.}
}
\details{
  Currently, only PDF versions of vignettes can be viewed.
#ifdef unix
  The program specified by the \code{pdfviewer} option is used for this.
#endif
  If several vignettes have PDF versions with base name identical to
  \code{topic}, the first one found is used for viewing.

  If no topics are given, the available vignettes are listed.  The
  corresponding information is returned in an object of class
  \code{"packageIQR"}.  The structure of this class is experimental.
}
\examples{
## List vignettes in all attached packages
vignette()
## List vignettes in all available packages
vignette(package = .packages(all.available = TRUE))
}
\keyword{documentation}
  
  
\eof
\name{BATCH}
\alias{BATCH}
\title{Batch Execution of R}
\description{Run \R non-interactively with input from \code{infile} and
  send output (stdout/stderr) to another file.}
\usage{
R CMD BATCH [options] infile [outfile]
}
\arguments{
  \item{infile}{the name of a file with \R code to be executed.}
  \item{options}{a list of \R command line options, e.g., for setting the
    amount of memory available and controlling the load/save process.
    If \code{infile} starts with a \samp{-}, use \samp{--} as the final
    option.  The default options are \samp{--restore --save --no-readline}.}
  \item{outfile}{the name of a file to which to write output.  If not
    given, the name used is that of \code{infile}, with a possible
    \file{.R} extension stripped, and \file{.Rout} appended.}
}
\details{
  Use \code{R CMD BATCH --help} to be reminded of the usage.

  By default, the input commands are printed along with the output.  To
  suppress this behavior, add \code{options(echo = FALSE)} at the
  beginning of \code{infile}.

  The \code{infile} can have end of line marked by LF or CRLF (but not
  just CR), and files with an incomplete last line (missing end of line
  (EOL) mark) are processed correctly.

  Using \code{R CMD BATCH} sets the GUI to \code{"none"}, so none of
  \code{\link[graphics]{x11}}, \code{\link[graphics]{jpeg}}
  and \code{\link[graphics]{png}} are available.
}
\note{
  Unlike \code{Splus BATCH}, this does not run the \R process in the
  background.  In most shells,
  \code{R CMD BATCH [options] infile [outfile] &}
  will do so.

  Report bugs to \email{r-bugs@r-project.org}.
}
\keyword{utilities}
\eof
\name{COMPILE}
\alias{COMPILE}
\title{Compile Files for Use with R}
\description{
  Compile given source files so that they can subsequently be collected
  into a shared library using \code{R CMD SHLIB} and be loaded into R
  using \code{dyn.load()}.
}
\usage{R CMD COMPILE [options] srcfiles}
\arguments{
  \item{srcfiles}{A list of the names of source files to be compiled.
    Currently, C, C++ and FORTRAN are supported; the corresponding files
    should have the extensions \file{.c}, \file{.cc} (or \file{.cpp} or
    \file{.C}), and \file{.f}, respectively.}
  \item{options}{A list of compile-relevant settings, such as special
    values for \code{CFLAGS} or \code{FFLAGS}, or for obtaining
    information about usage and version of the utility.}
}
\details{
  Note that Ratfor is not supported.  If you have Ratfor source code,
  you need to convert it to FORTRAN.  On many Solaris systems mixing
  Ratfor and FORTRAN code will work.
}
\seealso{
  \code{\link{SHLIB}},
  \code{\link{dyn.load}}
}
\keyword{utilities}
\eof
\name{INSTALL}
\alias{INSTALL}
\title{Install Add-on Packages}
\description{Utility for installing add-on packages.}
\usage{R CMD INSTALL [options] [-l lib] pkgs}
\arguments{
  \item{pkgs}{A list with the path names of the packages to be
    installed.}
  \item{lib}{the path name of the \R library tree to install to.}
  \item{options}{a space-separated list of options through which in
    particular the process for building the help files can be
    controlled.  Options should only be given once.
    Use \code{R CMD INSTALL --help} for the current list of options.
  }
}
\details{
  If used as \code{R CMD INSTALL pkgs} without explicitly specifying
  \code{lib}, packages are installed into the library tree rooted at the
  first directory given in the environment variable \env{R\_LIBS} if
  this is set and non-null, and to the default library tree (which is
  rooted at \file{\$R\_HOME/library}) otherwise.

  To install into the library tree \code{lib}, use
  \code{R CMD INSTALL -l lib pkgs}.

  Both \code{lib} and the elements of \code{pkgs} may be absolute or
  relative path names. \code{pkgs} can also contain name of package
  archive files of the form \file{pkg\_version.tar.gz} as obtained from
  CRAN, these are then extracted in a temporary directory.
  
  Some package sources contain a \file{configure} script that can be
  passed arguments or variables via the option \option{--configure-args}
  and \option{--configure-vars}, respectively, if necessary.  The latter
  is useful in particular if libraries or header files needed for the
  package are in non-system directories.  In this case, one can use the
  configure variables \code{LIBS} and \code{CPPFLAGS} to specify these
  locations (and set these via \option{--configure-vars}), see section
  \dQuote{Configuration variables} in \dQuote{R Installation and
    Administration} for more information.  (If these are used more than
  once on the command line, only the last instance is used.)  One can
  bypass the configure mechanism using the option \option{--no-configure}.
  
  If \option{--no-docs} is given, no help files are built.  Options
  \option{--no-text}, \option{--no-html}, and \option{--no-latex} suppress
  creating the text, HTML, and LaTeX versions, respectively.  The
  default is to build help files in all three versions.

  If the option \option{--save} is used, the installation procedure
  creates a binary image of the package code, which is then loaded when
  the package is attached, rather than evaluating the package source at
  that time.  Having a file \file{install.R} in the package directory
  makes this the default behavior for the package (option
  \option{--no-save} overrides).  You may need  \option{--save} if your
  package requires other packages to evaluate its own source.  If the
  file \file{install.R} is non-empty, it should contain R expressions to
  be executed when the package is attached, after loading the saved
  image.  Options to be passed to R when creating the save image can be
  specified via \option{--save=ARGS}.

  If the attempt to install the package fails, leftovers are removed.
  If the package was already installed, the old version is restored.

  Use \code{R CMD INSTALL --help} for more usage information.
}

\section{Packages using the methods package}{
  Packages that require the methods package, and that use functions such
  as \code{\link[methods]{setMethod}} or \code{\link[methods]{setClass}},
  should be installed by creating a binary image.

  The presence of a file named
  \file{install.R} in the package's main directory causes an image to
  be saved.  Note that the file is not in the \file{R} subdirectory: all
  the code in that subdirectory is used to construct the binary image.

  Normally, the file \file{install.R} will be empty; if it does contain
  R expressions
  these will be evaluated when the package is attached, e.g. by a call to
  the function \code{\link{library}}.  (Specifically, the source code
  evaluated for a package with a saved image consists of a suitable
  definition of \code{.First.lib} to ensure loading of the saved
  image, followed by the R code in file \file{install.R}, if any.)

}
\seealso{
  \code{\link{REMOVE}} and \code{\link{library}} for information on
  using several library trees;
  \code{\link{update.packages}} for automatic update of packages using
  the internet;
  the chapter on \dQuote{Creating R packages} in \dQuote{Writing \R
    Extensions} (see the \file{doc/manual} subdirectory of the \R source
  tree).
}
\keyword{utilities}
\eof
\name{LINK}
\alias{LINK}
\title{Create Executable Programs}
\description{
  Front-end for creating executable programs.
}
\usage{
R CMD LINK [options] linkcmd
}
\arguments{
  \item{linkcmd}{a list of commands to link together suitable object
    files (include library objects) to create the executable program.}
  \item{options}{further options to control the linking, or for
    obtaining information about usage and version.}
}
\details{
  The linker front-end is useful in particular when linking against the
  R shared library, in which case \code{linkcmd} must contain \code{-lR}
  but need not specify its library path.
  
  Currently only works if the C compiler is used for linking, and no C++
  code is used.

  Use \code{R CMD LINK --help} for more usage information.
}
\keyword{utilities}
\eof
\name{REMOVE}
\alias{REMOVE}
\title{Remove Add-on Packages}
\description{Utility for removing add-on packages.}
\usage{R CMD REMOVE [options] [-l lib] pkgs}
\arguments{
  \item{pkgs}{a list with the names of the packages to be removed.}
  \item{lib}{the path name of the \R library tree to remove from.  May
    be absolute or relative.}
  \item{options}{further options.}
}
\details{
  If used as \code{R CMD REMOVE pkgs} without explicitly specifying
  \code{lib}, packages are removed from the library tree rooted at the
  first directory given in \code{\$R\_LIBS} if this is set and non-null,
  and to the default library tree (which is rooted at
  \file{\$R\_HOME/library}) otherwise.

  To remove from the library tree \code{lib}, use
  \code{R CMD REMOVE -l lib pkgs}.

  Use \code{R CMD REMOVE --help} for more usage information.  
}
\seealso{
  \code{\link{INSTALL}}
}
\keyword{utilities}
\eof
\name{RHOME}
\alias{RHOME}
\alias{R_HOME}
\title{R Home Directory}
\description{
  Returns the location of the \R home directory, which is the root of
  the installed \R tree.
}
\usage{R RHOME}
\keyword{utilities}
\eof
\name{SHLIB}
\alias{SHLIB}
\title{Build Shared Library for Dynamic Loading}
\description{
  Compile given source files using \code{R CMD COMPILE}, and then link
  all specified object files into a shared library which can be loaded
  into R using \code{dyn.load} or \code{library.dynam}.
}
\usage{R CMD SHLIB [options] [-o libname] files}
\arguments{
  \item{files}{a list specifying the object files to be included in the
    shared library.  You can also include the name of source files, for
    which the object files are automagically made from their sources.}
  \item{libname}{the full name of the shared library to be built,
    including the extension (typically \file{.so} on Unix systems).  If
    not given, the name of the library is taken from the first file.}
  \item{options}{Further options to control the processing, or for
    obtaining information about usage and version of the utility.}
}
\seealso{
  \code{\link{COMPILE}}, \code{\link{dyn.load}}, \code{\link{library.dynam}}
}
\keyword{utilities}
\eof
\name{make.packages.html}
\alias{make.packages.html}
\title{Update HTML documentation files}
\usage{
make.packages.html(lib.loc = .libPaths())
}
\description{
  Functions to re-create the HTML documentation files to reflect all
  installed packages.
}
\arguments{
  \item{lib.loc}{character vector. List of libraries to be included.}
}
\details{
  This sets up the links from packages in libraries to the \file{.R}
  subdirectory of the per-session directory (see \code{\link{tempdir}})
  and then creates the \file{packages.html} and \file{index.txt} files
  to point to those links.

  If a package is available in more than one library tree, all the
  copies are linked, after the first with suffix \code{.1} etc.
}
\value{
  Logical, whether the function succeeded in recreating the files.
}
\seealso{
  \code{\link{help.start}}
}
\keyword{utilities}
\eof
\name{nsl}
\alias{nsl}
\title{Look up the IP Address by Hostname}
\description{
  Interface to \code{gethostbyname}.
}
\usage{
nsl(hostname)
}
\arguments{
 \item{hostname}{the name of the host.}
}
\value{
  The IP address, as a character string, or \code{NULL} if the call fails.
}
\note{
  This was included as a test of internet connectivity, to fail if
  the node running R is not connected.  It will also return \code{NULL}
  if BSD networking is not supported, including the header file
  \file{arpa/inet.h}.
}
\examples{
\dontrun{nsl("www.r-project.org")}
}
\keyword{utilities}
\eof
