 /*
    This file is part of libkcal.

    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/
#ifndef KCAL_RESOURCECACHED_H
#define KCAL_RESOURCECACHED_H

#include <qptrlist.h>
#include <qstring.h>
#include <qdatetime.h>
#include <kconfig.h>

#include "incidence.h"
#include "calendarlocal.h"

#include "resourcecalendar.h"

namespace KCal {

/**
  This class provides a calendar resource using a local CalendarLocal object to
  cache the calendar data.
*/
class ResourceCached : public ResourceCalendar
{
  public:
    ResourceCached( const KConfig * );
    virtual ~ResourceCached();

    /**
      Add event to calendar.
    */
    bool addEvent(Event *anEvent);
    /**
      Deletes an event from this calendar.
    */
    void deleteEvent(Event *);

    /**
      Retrieves an event on the basis of the unique string ID.
    */
    Event *event(const QString &UniqueStr);
    /**
      Return filtered list of all events in calendar.
    */
    Event::List events();
    /**
      Return unfiltered list of all events in calendar.
    */
    Event::List rawEvents();
    /**
      Builds and then returns a list of all events that match for the
      date specified. useful for dayView, etc. etc.
    */
    Event::List rawEventsForDate( const QDate &date, bool sorted = false );
    /**
      Get unfiltered events for date \a qdt.
    */
    Event::List rawEventsForDate( const QDateTime &qdt );
    /**
      Get unfiltered events in a range of dates. If inclusive is set to true,
      only events are returned, which are completely included in the range.
    */
    Event::List rawEvents( const QDate &start, const QDate &end,
                               bool inclusive = false );
  

    /**
      Add a todo to the todolist.
    */
    bool addTodo( Todo *todo );
    /**
      Remove a todo from the todolist.
    */
    void deleteTodo( Todo * );
    /**
      Searches todolist for an event with this unique string identifier,
      returns a pointer or null.
    */
    Todo *todo( const QString &uid );
    /**
      Return list of all todos.
    */
    Todo::List rawTodos();
    /**
      Returns list of todos due on the specified date.
    */
    Todo::List todos( const QDate &date );
    /**
      Add a Journal entry to calendar
    */
    virtual bool addJournal( Journal * );
    /**
      Remove a Journal from the calendar
    */
    virtual void deleteJournal( Journal * );
    /**
      Return Journal for given date.
    */
    virtual Journal *journal( const QDate & );
    /**
      Return Journal with given unique id.
    */
    virtual Journal *journal( const QString &uid );
    /**
      Return list of all Journals stored in calendar
    */
    Journal::List journals();

    /**
      Return all alarms, which ocur in the given time interval.
    */
    Alarm::List alarms( const QDateTime &from, const QDateTime &to );

    /**
      Return all alarms, which ocur before given date.
    */
    Alarm::List alarmsTo( const QDateTime &to );

    /**
      Set id of timezone, e.g. "Europe/Berlin"
    */
    void setTimeZoneId( const QString& tzid );
  
  protected:
    CalendarLocal mCalendar;

  private:
    class Private;
    Private *d;
};

}

#endif
