-module(erl_parse).
-define(THIS_MODULE, erl_parse).
-export([parse/1, parse_and_scan/1, format_error/1]).

-author('rv@cslab.ericsson.se').
-copyright('Copyright (c) 1996 Ericsson Telecommunications AB').

%% This is a hack to help the JAM get better code.
-ifdef(JAM).
-compile([{parse_transform,jam_yecc_pj},pj]).
-endif.

-export([parse_form/1,parse_exprs/1,parse_term/1]).
-export([normalise/1,abstract/1,tokens/1,tokens/2]).
-export([inop_prec/1,preop_prec/1,func_prec/0,max_prec/0]).

%% Entry points compatible to old erl_parse.
%% These really suck and are only here until Calle gets multiple
%% entry points working.

parse_form(Tokens) ->
    parse(Tokens).

parse_exprs(Tokens) ->
    case parse([{atom,0,f},{'(',0},{')',0},{'->',0}|Tokens]) of
	{ok,{function,Lf,f,0,[{clause,Lc,[],[],Exprs}]}} ->
	    {ok,Exprs};
	{error,E} -> {error,E}
    end.

parse_term(Tokens) ->
    case parse([{atom,0,f},{'(',0},{')',0},{'->',0}|Tokens]) of
	{ok,{function,Lf,f,0,[{clause,Lc,[],[],[Expr]}]}} ->
	    case catch normalise(Expr) of
		{'EXIT',R} ->
		    {error,{element(2, Expr),?MODULE,"bad term"}};
		Term -> {ok,Term}
	    end;
	{ok,{function,Lf,f,0,[{clause,Lc,[],[],[E1,E2|Es]}]}} ->
	    {error,{element(2, E2),?MODULE,"bad term"}};
	{error,E} -> {error,E}
    end.

%% build_attribute(AttrName, AttrValue) ->
%%	{attribute,Line,module,Module}
%%	{attribute,Line,export,Exports}
%%	{attribute,Line,import,Imports}
%%	{attribute,Line,record,{Name,Inits}}
%%	{attribute,Line,file,{Name,Line}}
%%	{attribute,Line,Name,Val}

build_attribute({atom,La,module}, Val) ->
    case Val of
	[{atom,Lm,Module}] -> {attribute,La,module,Module};
	Other -> return_error(La, "bad module declaration")
    end;
build_attribute({atom,La,export}, Val) ->
    case Val of
	[ExpList] ->
	    {attribute,La,export,farity_list(ExpList)};
	Other -> return_error(La, "bad export declaration")
    end;
build_attribute({atom,La,import}, Val) ->
    case Val of
	[{atom,Lm,Mod},ImpList] ->
	    {attribute,La,import,{Mod,farity_list(ImpList)}};
	Other -> return_error(La, "bad import declaration")
    end;
build_attribute({atom,La,record}, Val) ->
    case Val of
	[{atom,Ln,Record},RecTuple] ->
	    {attribute,La,record,{Record,record_tuple(RecTuple)}};
	Other -> return_error(La, "bad record declaration")
    end;
build_attribute({atom,La,file}, Val) ->
    case Val of
	[{string,Ln,Name},{integer,Ll,Line}] ->
	    {attribute,La,file,{Name,Line}};
	Other -> return_error(La, "bad file declaration")
    end;
build_attribute({atom,La,Attr}, Val) ->
    case Val of
	[Expr] ->
	    {attribute,La,Attr,term(Expr)};
	Other -> return_error(La, "bad attribute")
    end.

farity_list({cons,Lc,{op,Lo,'/',{atom,La,A},{integer,Li,I}},Tail}) ->
    [{A,I}|farity_list(Tail)];
farity_list({nil,Ln}) -> [];
farity_list(Other) ->
    return_error(element(2, Other), "bad function arity").

record_tuple({tuple,Lt,Fields}) ->
    record_fields(Fields);
record_tuple(Other) ->
    return_error(element(2, Other), "bad record declaration").

record_fields([{atom,La,A}|Fields]) ->
    [{record_field,La,{atom,La,A}}|record_fields(Fields)];
record_fields([{match,Lm,{atom,La,A},Expr}|Fields]) ->
    [{record_field,La,{atom,La,A},Expr}|record_fields(Fields)];
record_fields([Other|Fields]) ->
    return_error(element(2, Other), "bad record field");
record_fields([]) -> [].

term(Expr) ->
    case catch normalise(Expr) of
	{'EXIT',R} -> return_error(element(2, Expr), "bad attribute");
	Term -> Term
    end.

%% build_function([Clause]) -> {function,Line,Name,Arity,[Clause]}

build_function(Cs) ->
    Name = element(3, hd(Cs)),
    Arity = length(element(4, hd(Cs))),
    {function,element(2, hd(Cs)),Name,Arity,check_clauses(Cs, Name, Arity)}.

%% build_rule([Clause]) -> {rule,Line,Name,Arity,[Clause]'}

build_rule(Cs) ->
    Name = element(3, hd(Cs)),
    Arity = length(element(4, hd(Cs))),
    {rule,element(2, hd(Cs)),Name,Arity,check_clauses(Cs, Name, Arity)}.

%% build_lambda(Line, [Clause]) -> {'fun',Line,{clauses,[Clause]}}.

build_lambda(Line, Cs) ->
    Arity = length(element(4, hd(Cs))),
    {'fun',Line,{clauses,check_clauses(Cs, 'fun', Arity)}}.

check_clauses(Cs, Name, Arity) ->
     mapl(fun ({clause,L,N,As,G,B}) when N == Name, length(As) == Arity ->
		 {clause,L,As,G,B};
	     ({clause,L,N,As,G,B}) ->
		 return_error(L, "head mismatch") end, Cs).

%% mapl(F,List)
%% an alternative map which always maps from left to right
%% and makes it possible to interrupt the mapping with throw on
%% the first occurence from left as expected.
%% can be removed when the jam machine (and all other machines)
%% uses the standardized (Erlang 5.0) evaluation order (from left to right)
mapl(F, [H|T]) ->
	V = F(H),
	[V | mapl(F,T)];
mapl(_, []) ->
	[].

%% normalise(AbsTerm)
%% abstract(Term)
%%  Convert between the abstract form of a term and a term.

normalise({atom,_,A}) -> A;
normalise({integer,_,I}) -> I;
normalise({float,_,F}) -> F;
normalise({char,_,C}) -> C;
normalise({string,_,S}) -> S;
normalise({nil,_}) -> [];
normalise({cons,_,Head,Tail}) ->
    [normalise(Head)|normalise(Tail)];
normalise({tuple,_,Args}) ->
    list_to_tuple(normalise_list(Args));
%% Special case for unary +/-.
normalise({op,_,'+',{integer,_,I}}) -> I;
normalise({op,_,'+',{float,_,F}}) -> F;
normalise({op,_,'-',{integer,_,I}}) -> -I;
normalise({op,_,'-',{float,_,F}}) -> -F.

normalise_list([H|T]) ->
    [normalise(H)|normalise_list(T)];
normalise_list([]) ->
    [].

abstract(T) when atom(T) ->
    {atom,0,T};
abstract(T) when integer(T) ->
    {integer,0,T};
abstract(T) when float(T) ->
    {float,0,T};
abstract([]) ->
    {nil,0};
abstract([C|T]) when integer(C), 0 =< C, C < 256 ->
    abstract_string(T, [C]);
abstract([H|T]) ->
    {cons,0,abstract(H),abstract(T)};
abstract(Tuple) when tuple(Tuple) ->
    {tuple,0,abstract_list(tuple_to_list(Tuple))}.

abstract_string([C|T], String) when integer(C), 0 =< C, C < 256 ->
    abstract_string(T, [C|String]);
abstract_string([H|T], String) ->
    not_string(String, abstract([H|T]));
abstract_string([], String) ->
    {string, 0, lists:reverse(String)}.

not_string([C|T], Result) ->
    not_string(T, {cons, 0, {integer, 0, C}, Result});
not_string([], Result) ->
    Result.

abstract_list([H|T]) ->
    [abstract(H)|abstract_list(T)];
abstract_list([]) ->
    [].

%% tokens(AbsTerm) -> [Token]
%% tokens(AbsTerm, More) -> [Token]
%%  Generate a list of tokens representing the abstract term.

tokens(Abs) ->
    tokens(Abs, []).

tokens({atom,L,A}, More) -> [{atom,L,A}|More];
tokens({integer,L,N}, More) -> [{integer,L,N}|More];
tokens({float,L,F}, More) -> [{float,L,F}|More];
tokens({char,L,C}, More) -> [{char,L,C}|More];
tokens({var,L,V}, More) -> [{var,L,V}|More];
tokens({string,L,S}, More) -> [{string,L,S}|More];
tokens({nil,L}, More) -> [{'[',L},{']',L}|More];
tokens({cons,L,Head,Tail}, More) ->
    [{'[',L}|tokens(Head, tokens_tail(Tail, More))];
tokens({tuple,L,[]}, More) ->
    [{'{',L},{'}',L}|More];
tokens({tuple,L,[E|Es]}, More) ->
    [{'{',L}|tokens(E, tokens_tuple(Es, element(2, E), More))].

tokens_tail({cons,L,Head,Tail}, More) ->
    [{',',L}|tokens(Head, tokens_tail(Tail, More))];
tokens_tail({nil,L}, More) ->
    [{']',L}|More];
tokens_tail(Other, More) ->
    L = element(2, Other),
    [{'|',L}|tokens(Other, [{']',L}|More])].

tokens_tuple([E|Es], Line, More) ->
    [{',',Line}|tokens(E, tokens_tuple(Es, element(2, E), More))];
tokens_tuple([], Line, More) ->
    [{'}',Line}|More].

%% Give the relative precedences of operators.

inop_prec('=') -> {200,100,100};
inop_prec('!') -> {200,100,100};
inop_prec('==') -> {300,200,300};
inop_prec('/=') -> {300,200,300};
inop_prec('=<') -> {300,200,300};
inop_prec('<') -> {300,200,300};
inop_prec('>=') -> {300,200,300};
inop_prec('>') -> {300,200,300};
inop_prec('=:=') -> {300,200,300};
inop_prec('=/=') -> {300,200,300};
inop_prec('++') -> {300,300,400};
inop_prec('--') -> {300,300,400};
inop_prec('+') -> {400,400,500};
inop_prec('-') -> {400,400,500};
inop_prec('bor') -> {400,400,500};
inop_prec('bxor') -> {400,400,500};
inop_prec('bsl') -> {400,400,500};
inop_prec('bsr') -> {400,400,500};
inop_prec('or') -> {400,400,500};
inop_prec('xor') -> {400,400,500};
inop_prec('*') -> {500,500,600};
inop_prec('/') -> {500,500,600};
inop_prec('div') -> {500,500,600};
inop_prec('rem') -> {500,500,600};
inop_prec('band') -> {500,500,600};
inop_prec('and') -> {500,500,600};
inop_prec('#') -> {800,700,800};
inop_prec(':') -> {900,800,900}.

preop_prec('catch') -> {0,100};
preop_prec('+') -> {600,700};
preop_prec('-') -> {600,700};
preop_prec('bnot') -> {600,700};
preop_prec('not') -> {600,700};
preop_prec('#') -> {700,800}.

func_prec() -> {800,700}.

max_prec() -> 900.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The parser generator will insert appropriate declarations before this line.%

parse(Tokens) ->
    case catch yeccpars1(Tokens, false, 0, [], []) of
	error ->
	    Errorline =
		if Tokens == [] -> 0; true -> element(2, hd(Tokens)) end,
	    {error,
	     {Errorline, ?THIS_MODULE, "syntax error at or after this line."}};
	Other ->
	    Other
    end.

parse_and_scan({Mod, Fun, Args}) ->
    case apply(Mod, Fun, Args) of
	{eof, _} ->
	    {ok, eof};
	{error, Descriptor, _} ->
	    {error, Descriptor};
	{ok, Tokens, _} ->
	    yeccpars1(Tokens, {Mod, Fun, Args}, 0, [], [])
    end.

format_error(Message) ->
    case io_lib:deep_char_list(Message) of
	true ->
	    Message;
	_ ->
	    io_lib:write(Message)
    end.

% To be used in grammar files to throw an error message to the parser toplevel.
% Doesn't have to be exported!
return_error(Line, Message) ->
    throw({error, {Line, ?THIS_MODULE, Message}}).


% Don't change yeccpars1/6 too much, it is called recursively by yeccpars2/8!
yeccpars1([Token | Tokens], Tokenizer, State, States, Vstack) ->
    yeccpars2(State, element(1, Token), States, Vstack, Token, Tokens,
	      Tokenizer);
yeccpars1([], {M, F, A}, State, States, Vstack) ->
    case catch apply(M, F, A) of
        {eof, Endline} ->
            {error, {Endline, ?THIS_MODULE, "end_of_file"}};
        {error, Descriptor, Endline} ->
            {error, Descriptor};
        {'EXIT', Reason} ->
            {error, {0, ?THIS_MODULE, Reason}};
        {ok, Tokens, Endline} ->
	    case catch yeccpars1(Tokens, {M, F, A}, State, States, Vstack) of
		error ->
		    Errorline = element(2, hd(Tokens)),
		    {error, {Errorline, ?THIS_MODULE,
			     "syntax error at or after this line."}};
		Other ->
		    Other
	    end
    end;
yeccpars1([], false, State, States, Vstack) ->
    yeccpars2(State, '$end', States, Vstack, {'$end', 999999}, [], false).

% For internal use only.
yeccerror(Token) ->
    {error,
     {element(2, Token), ?THIS_MODULE,
      ["syntax error before: ", yecctoken2string(Token)]}}.

yecctoken2string({atom, _, A}) -> io_lib:write(A);
yecctoken2string({integer,_,N}) -> io_lib:write(N);
yecctoken2string({float,_,F}) -> io_lib:write(F);
yecctoken2string({char,_,C}) -> io_lib:write_char(C);
yecctoken2string({var,_,V}) -> io_lib:format('~s', [V]);
yecctoken2string({string,_,S}) -> io_lib:write_string(S);
yecctoken2string({reserved_symbol, _, A}) -> io_lib:format('~w', [A]);
yecctoken2string({Cat, _, Val}) -> io_lib:format('~w', [Val]);

yecctoken2string({'dot', _}) -> io_lib:format('~w', ['.']);
yecctoken2string({'$end', _}) ->
    [];
yecctoken2string({Other, _}) when atom(Other) ->
    io_lib:format('~w', [Other]);
yecctoken2string(Other) ->
    io_lib:write(Other).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


yeccpars2(0, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 2, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 1, [0 | __Ss], [__T | __Stack]);
yeccpars2(0, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(1, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 217, [1 | __Ss], [__T | __Stack]);
yeccpars2(1, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(2, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 16, [2 | __Ss], [__T | __Stack]);
yeccpars2(2, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(3, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 214, [3 | __Ss], [__T | __Stack]);
yeccpars2(3, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(4, '$end', _, __Stack, _, _, _) ->
 {ok, hd(__Stack)};
yeccpars2(4, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(5, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 213, [5 | __Ss], [__T | __Stack]);
yeccpars2(5, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(6, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 207, [6 | __Ss], [__T | __Stack]);
yeccpars2(6, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(function_clauses, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(7, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  build_function(__1),
 yeccpars2(yeccgoto(function, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(8, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 206, [8 | __Ss], [__T | __Stack]);
yeccpars2(8, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(9, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 13, [9 | __Ss], [__T | __Stack]);
yeccpars2(9, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(rule_clauses, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(10, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  build_rule(__1),
 yeccpars2(yeccgoto(rule, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(11, dot, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 12, [11 | __Ss], [__T | __Stack]);
yeccpars2(11, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(12, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(13, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 14, [13 | __Ss], [__T | __Stack]);
yeccpars2(13, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(14, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 16, [14 | __Ss], [__T | __Stack]);
yeccpars2(14, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(15, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(rule_clauses, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(16, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 203, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [16 | __Ss], [__T | __Stack]);
yeccpars2(16, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(17, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  element(1,__1),
 yeccpars2(yeccgoto(clause_args, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(18, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [18 | __Ss], [__T | __Stack]);
yeccpars2(18, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(19, __Cat, [18 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(19, ':-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 200, [19 | __Ss], [__T | __Stack]);
yeccpars2(19, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(20, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [20 | __Ss], [__T | __Stack]);
yeccpars2(20, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(21, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 196, [21 | __Ss], [__T | __Stack]);
yeccpars2(21, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(22, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [22 | __Ss], [__T | __Stack]);
yeccpars2(22, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(23, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(24, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(25, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 184, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [25 | __Ss], [__T | __Stack]);
yeccpars2(25, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(26, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(27, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(28, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [28 | __Ss], [__T | __Stack]);
yeccpars2(28, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(29, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(30, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [30 | __Ss], [__T | __Stack]);
yeccpars2(30, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(31, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(32, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [32 | __Ss], [__T | __Stack]);
yeccpars2(32, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(33, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 175, [33 | __Ss], [__T | __Stack]);
yeccpars2(33, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(exprs, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(34, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(35, '!', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 171, [35 | __Ss], [__T | __Stack]);
yeccpars2(35, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 172, [35 | __Ss], [__T | __Stack]);
yeccpars2(35, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr100, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(36, '=/=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 163, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '=:=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 164, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '>', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 167, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '>=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 168, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 162, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '=<', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 165, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '/=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 161, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, '==', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 166, [36 | __Ss], [__T | __Stack]);
yeccpars2(36, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr200, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(37, '--', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 150, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '++', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 148, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, 'xor', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 158, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, 'or', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 157, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, 'bsr', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 154, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, 'bsl', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 153, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, 'bxor', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 155, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, 'bor', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 152, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 149, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 147, [37 | __Ss], [__T | __Stack]);
yeccpars2(37, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr300, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(38, 'and', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [38 | __Ss], [__T | __Stack]);
yeccpars2(38, 'band', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [38 | __Ss], [__T | __Stack]);
yeccpars2(38, 'rem', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [38 | __Ss], [__T | __Stack]);
yeccpars2(38, 'div', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [38 | __Ss], [__T | __Stack]);
yeccpars2(38, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [38 | __Ss], [__T | __Stack]);
yeccpars2(38, '*', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [38 | __Ss], [__T | __Stack]);
yeccpars2(38, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr400, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(39, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr500, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(40, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr600, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(41, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 16, [41 | __Ss], [__T | __Stack]);
yeccpars2(41, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 122, [41 | __Ss], [__T | __Stack]);
yeccpars2(41, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 121, [41 | __Ss], [__T | __Stack]);
yeccpars2(41, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_700, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(42, ':', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 119, [42 | __Ss], [__T | __Stack]);
yeccpars2(42, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_800, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(43, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(guard, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(44, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(45, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 16, [45 | __Ss], [__T | __Stack]);
yeccpars2(45, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 109, [45 | __Ss], [__T | __Stack]);
yeccpars2(45, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(46, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(47, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(clause_guard, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(48, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [48 | __Ss], [__T | __Stack]);
yeccpars2(48, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(49, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(50, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(51, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(52, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(53, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(prefix_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(54, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [54 | __Ss], [__T | __Stack]);
yeccpars2(54, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(55, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 87, [55 | __Ss], [__T | __Stack]);
yeccpars2(55, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(56, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(57, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 69, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [57 | __Ss], [__T | __Stack]);
yeccpars2(57, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(58, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(59, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_700, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(60, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [60 | __Ss], [__T | __Stack]);
yeccpars2(60, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(strings, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(61, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(atomic, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(62, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(63, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(expr_max, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(64, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 66, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [64 | __Ss], [__T | __Stack]);
yeccpars2(64, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(65, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 67, [65 | __Ss], [__T | __Stack]);
yeccpars2(65, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(66, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {tuple,element(2,__1),[]},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(tuple, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(67, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {tuple,element(2,__1),__2},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(tuple, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(68, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {string,element(2,__1),element(3,__1) ++ element(3,__2)},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(strings, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(69, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [69 | __Ss], [__T | __Stack]);
yeccpars2(69, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(70, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 82, [70 | __Ss], [__T | __Stack]);
yeccpars2(70, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(cr_clauses, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(71, 'after', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 77, [71 | __Ss], [__T | __Stack]);
yeccpars2(71, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 78, [71 | __Ss], [__T | __Stack]);
yeccpars2(71, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(72, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [72 | __Ss], [__T | __Stack]);
yeccpars2(72, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(73, __Cat, [72 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(73, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [73 | __Ss], [__T | __Stack]);
yeccpars2(73, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(74, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [74 | __Ss], [__T | __Stack]);
yeccpars2(74, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(75, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {clause,element(2,__1),[__1],__2,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(cr_clause, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(76, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(clause_body, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(77, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [77 | __Ss], [__T | __Stack]);
yeccpars2(77, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(78, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'receive',element(2,__1),__2},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(receive_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(79, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [79 | __Ss], [__T | __Stack]);
yeccpars2(79, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(80, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 81, [80 | __Ss], [__T | __Stack]);
yeccpars2(80, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(81, __Cat, __Ss,  [__6,__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'receive',element(2,__1),__2,__4,__5},
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(receive_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(82, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [82 | __Ss], [__T | __Stack]);
yeccpars2(82, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(83, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(cr_clauses, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(84, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [84 | __Ss], [__T | __Stack]);
yeccpars2(84, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(85, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 86, [85 | __Ss], [__T | __Stack]);
yeccpars2(85, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(86, __Cat, __Ss,  [__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'receive',element(2,__1),[],__3,__4},
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(receive_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(87, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [87 | __Ss], [__T | __Stack]);
yeccpars2(87, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(88, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 89, [88 | __Ss], [__T | __Stack]);
yeccpars2(88, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(89, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'query',element(2,__1),__2},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(query_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(90, '||', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 91, [90 | __Ss], [__T | __Stack]);
yeccpars2(90, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(91, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [91 | __Ss], [__T | __Stack]);
yeccpars2(91, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(92, '<-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 98, [92 | __Ss], [__T | __Stack]);
yeccpars2(92, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(lc_expr, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(93, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 96, [93 | __Ss], [__T | __Stack]);
yeccpars2(93, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(lc_exprs, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(94, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 95, [94 | __Ss], [__T | __Stack]);
yeccpars2(94, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(95, __Cat, __Ss,  [__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {lc,element(2,__1),__2,__4},
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(list_comprehension, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(96, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [96 | __Ss], [__T | __Stack]);
yeccpars2(96, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(97, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(lc_exprs, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(98, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [98 | __Ss], [__T | __Stack]);
yeccpars2(98, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(99, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {generate,element(2,__2),__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(lc_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(100, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val = begin
   {Op,Pos} = __1, {op,Pos,Op,__2}
  end,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(expr600, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(101, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [101 | __Ss], [__T | __Stack]);
yeccpars2(101, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(102, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 105, [102 | __Ss], [__T | __Stack]);
yeccpars2(102, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(if_clauses, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(103, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 104, [103 | __Ss], [__T | __Stack]);
yeccpars2(103, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(104, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'if',element(2,__1),__2},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(if_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(105, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [105 | __Ss], [__T | __Stack]);
yeccpars2(105, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(106, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(if_clauses, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(107, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {clause,element(2,hd(__1)),[],__1,__2},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(if_clause, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(108, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [108 | __Ss], [__T | __Stack]);
yeccpars2(108, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(117, __Cat, [108 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(109, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 115, [109 | __Ss], [__T | __Stack]);
yeccpars2(109, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(110, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 113, [110 | __Ss], [__T | __Stack]);
yeccpars2(110, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(fun_clauses, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(111, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 112, [111 | __Ss], [__T | __Stack]);
yeccpars2(111, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(112, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  build_lambda(element(2,__1),__2),
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(113, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 16, [113 | __Ss], [__T | __Stack]);
yeccpars2(113, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(114, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_clauses, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(115, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 116, [115 | __Ss], [__T | __Stack]);
yeccpars2(115, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(116, __Cat, __Ss,  [__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'fun',element(2,__1),{function,element(3,__2),element(3,__4)}},
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(fun_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(117, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [117 | __Ss], [__T | __Stack]);
yeccpars2(117, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(118, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val = begin
   {Args,Pos} = __1, {clause,Pos,'fun',Args,__2,__3}
  end,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(fun_clause, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(119, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [119 | __Ss], [__T | __Stack]);
yeccpars2(119, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(120, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {remote,element(2,__2),__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_800, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(121, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 125, [121 | __Ss], [__T | __Stack]);
yeccpars2(121, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(122, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 124, [122 | __Ss], [__T | __Stack]);
yeccpars2(122, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(123, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {call,element(2,__1),__1,element(1,__2)},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(expr_700, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(124, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {record_field,element(2,__2),__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(125, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [125 | __Ss], [__T | __Stack]);
yeccpars2(125, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 126, [125 | __Ss], [__T | __Stack]);
yeccpars2(125, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(126, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 138, [126 | __Ss], [__T | __Stack]);
yeccpars2(126, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(127, __Cat, __Ss,  [__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {record,element(2,__2),__1,element(3,__3),__4},
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(128, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 132, [128 | __Ss], [__T | __Stack]);
yeccpars2(128, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(129, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 136, [129 | __Ss], [__T | __Stack]);
yeccpars2(129, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(130, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 134, [130 | __Ss], [__T | __Stack]);
yeccpars2(130, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(record_fields, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(131, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 133, [131 | __Ss], [__T | __Stack]);
yeccpars2(131, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(132, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [],
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(record_tuple, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(133, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_tuple, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(134, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 129, [134 | __Ss], [__T | __Stack]);
yeccpars2(134, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(135, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_fields, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(136, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [136 | __Ss], [__T | __Stack]);
yeccpars2(136, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(137, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {record_field,element(2,__1),__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_field, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(138, __Cat, __Ss,  [__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {record_field,element(2,__2),__1,element(3,__3),__5},
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(139, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(140, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(141, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(142, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(143, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(144, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [144 | __Ss], [__T | __Stack]);
yeccpars2(144, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(145, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(mult_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(146, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val = begin
   {Op,Pos} = __2, {op,Pos,Op,__1,__3}
  end,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr500, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(147, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(148, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(list_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(149, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(150, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(list_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(151, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [151 | __Ss], [__T | __Stack]);
yeccpars2(151, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(152, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(153, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(154, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(155, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(156, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [156 | __Ss], [__T | __Stack]);
yeccpars2(156, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(157, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(158, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(add_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(159, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val = begin
   {Op,Pos} = __2, {op,Pos,Op,__1,__3}
  end,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr300, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(160, 'and', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 141, [160 | __Ss], [__T | __Stack]);
yeccpars2(160, 'band', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 142, [160 | __Ss], [__T | __Stack]);
yeccpars2(160, 'rem', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 145, [160 | __Ss], [__T | __Stack]);
yeccpars2(160, 'div', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 143, [160 | __Ss], [__T | __Stack]);
yeccpars2(160, '/', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 140, [160 | __Ss], [__T | __Stack]);
yeccpars2(160, '*', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 139, [160 | __Ss], [__T | __Stack]);
yeccpars2(160, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val = begin
   {Op,Pos} = __2, {op,Pos,Op,__1,__3}
  end,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr400, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(161, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(162, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(163, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(164, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(165, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(166, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(167, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(168, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(comp_op, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(169, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [169 | __Ss], [__T | __Stack]);
yeccpars2(169, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(170, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val = begin
   {Op,Pos} = __2, {op,Pos,Op,__1,__3}
  end,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr200, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(171, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [171 | __Ss], [__T | __Stack]);
yeccpars2(171, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(172, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [172 | __Ss], [__T | __Stack]);
yeccpars2(172, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(173, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {match,element(2,__2),__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr100, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(174, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {op,element(2,__2),'!',__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr100, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(175, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [175 | __Ss], [__T | __Stack]);
yeccpars2(175, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(176, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(exprs, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(177, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'catch',element(2,__1),__2},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(178, 'of', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 179, [178 | __Ss], [__T | __Stack]);
yeccpars2(178, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(179, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [179 | __Ss], [__T | __Stack]);
yeccpars2(179, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(180, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 181, [180 | __Ss], [__T | __Stack]);
yeccpars2(180, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(181, __Cat, __Ss,  [__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {'case',element(2,__1),__2,__4},
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(case_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(182, 'end', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 183, [182 | __Ss], [__T | __Stack]);
yeccpars2(182, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(183, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {block,element(2,__1),__2},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_max, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(184, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {nil,element(2,__1)},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(list, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(185, '||', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 91, [185 | __Ss], [__T | __Stack]);
yeccpars2(185, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 186, [185 | __Ss], [__T | __Stack]);
yeccpars2(185, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 189, [185 | __Ss], [__T | __Stack]);
yeccpars2(185, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 187, [185 | __Ss], [__T | __Stack]);
yeccpars2(185, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(186, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [186 | __Ss], [__T | __Stack]);
yeccpars2(186, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(187, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {nil,element(2,__1)},
 yeccpars2(yeccgoto(tail, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(188, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {cons,element(2,__1),__2,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(list, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(189, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [189 | __Ss], [__T | __Stack]);
yeccpars2(189, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(190, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 191, [190 | __Ss], [__T | __Stack]);
yeccpars2(190, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(191, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(tail, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(192, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 186, [192 | __Ss], [__T | __Stack]);
yeccpars2(192, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 189, [192 | __Ss], [__T | __Stack]);
yeccpars2(192, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 187, [192 | __Ss], [__T | __Stack]);
yeccpars2(192, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(193, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {cons,element(2,__1),__2,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(tail, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(194, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 195, [194 | __Ss], [__T | __Stack]);
yeccpars2(194, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(195, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(expr_max, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(196, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 128, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, '.', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 197, [196 | __Ss], [__T | __Stack]);
yeccpars2(196, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(197, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 199, [197 | __Ss], [__T | __Stack]);
yeccpars2(197, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(198, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {record,element(2,__1),element(3,__2),__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(199, __Cat, __Ss,  [__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {record_index,element(2,__1),element(3,__2),__4},
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(record_expr, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(200, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [200 | __Ss], [__T | __Stack]);
yeccpars2(200, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(201, __Cat, __Ss,  [__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {clause,element(2,__1),element(3,__1),__2,__3,__4},
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(rule_clause, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(202, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(rule_body, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(203, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {[],element(2,__1)},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(argument_list, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(204, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 205, [204 | __Ss], [__T | __Stack]);
yeccpars2(204, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(205, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {__2,element(2,__1)},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(argument_list, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(206, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(207, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 208, [207 | __Ss], [__T | __Stack]);
yeccpars2(207, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(208, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 16, [208 | __Ss], [__T | __Stack]);
yeccpars2(208, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(209, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(function_clauses, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(210, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [210 | __Ss], [__T | __Stack]);
yeccpars2(210, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(211, __Cat, [210 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(211, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [211 | __Ss], [__T | __Stack]);
yeccpars2(211, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(212, __Cat, __Ss,  [__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {clause,element(2,__1),element(3,__1),__2,__3,__4},
 __Nss = lists:nthtail(3, __Ss),
 yeccpars2(yeccgoto(function_clause, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(213, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(214, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(form, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(215, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 20, [215 | __Ss], [__T | __Stack]);
yeccpars2(215, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(216, __Cat, [215 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(216, ':-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 200, [216 | __Ss], [__T | __Stack]);
yeccpars2(216, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 74, [216 | __Ss], [__T | __Stack]);
yeccpars2(216, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(217, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 219, [217 | __Ss], [__T | __Stack]);
yeccpars2(217, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 218, [217 | __Ss], [__T | __Stack]);
yeccpars2(217, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(218, 'not', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 53, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'bnot', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 29, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, '-', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 24, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, '+', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 23, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, string, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 60, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 26, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, float, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 44, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 50, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'query', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 55, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'fun', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 45, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'receive', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 57, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'case', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 30, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'if', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 48, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, '#', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 21, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 64, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 25, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'begin', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 28, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 22, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 63, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, 'catch', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 32, [218 | __Ss], [__T | __Stack]);
yeccpars2(218, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(219, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 264, [219 | __Ss], [__T | __Stack]);
yeccpars2(219, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(220, '->', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 221, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, '=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 222, [220 | __Ss], [__T | __Stack]);
yeccpars2(220, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(221, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [221 | __Ss], [__T | __Stack]);
yeccpars2(221, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(222, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [222 | __Ss], [__T | __Stack]);
yeccpars2(222, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(223, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 271, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [223 | __Ss], [__T | __Stack]);
yeccpars2(223, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(224, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [224 | __Ss], [__T | __Stack]);
yeccpars2(224, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [224 | __Ss], [__T | __Stack]);
yeccpars2(224, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [224 | __Ss], [__T | __Stack]);
yeccpars2(224, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [224 | __Ss], [__T | __Stack]);
yeccpars2(224, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [224 | __Ss], [__T | __Stack]);
yeccpars2(224, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(225, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 264, [225 | __Ss], [__T | __Stack]);
yeccpars2(225, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {atom,element(3,__1)},
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(226, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(ptypes, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(227, '|', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 261, [227 | __Ss], [__T | __Stack]);
yeccpars2(227, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {utype,__1,[]},
 yeccpars2(yeccgoto(utype, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(228, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 239, [228 | __Ss], [__T | __Stack]);
yeccpars2(228, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(238, __Cat, [228 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(229, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {tuple,__1},
 yeccpars2(yeccgoto(ptype, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(230, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {utype,[],[element(3,__1)]},
 yeccpars2(yeccgoto(utype, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(231, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 234, [231 | __Ss], [__T | __Stack]);
yeccpars2(231, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(232, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 236, [232 | __Ss], [__T | __Stack]);
yeccpars2(232, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(utypes, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(233, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 235, [233 | __Ss], [__T | __Stack]);
yeccpars2(233, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(234, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [],
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(utype_tuple, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(235, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype_tuple, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(236, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [236 | __Ss], [__T | __Stack]);
yeccpars2(236, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [236 | __Ss], [__T | __Stack]);
yeccpars2(236, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [236 | __Ss], [__T | __Stack]);
yeccpars2(236, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [236 | __Ss], [__T | __Stack]);
yeccpars2(236, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [236 | __Ss], [__T | __Stack]);
yeccpars2(236, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(237, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utypes, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(238, __Cat, __Ss,  [__6,__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  if element(3,__2) == type -> {type,element(2,__2),def,__3,__5,__6}; true -> return_error(element(2,__2),"illegal type declaration") end,
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(type_decl, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(239, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [239 | __Ss], [__T | __Stack]);
yeccpars2(239, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [239 | __Ss], [__T | __Stack]);
yeccpars2(239, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [239 | __Ss], [__T | __Stack]);
yeccpars2(239, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 243, [239 | __Ss], [__T | __Stack]);
yeccpars2(239, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [239 | __Ss], [__T | __Stack]);
yeccpars2(239, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(240, ';', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 259, [240 | __Ss], [__T | __Stack]);
yeccpars2(240, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(constraints, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(241, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(type_constraints, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(242, '<=', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 257, [242 | __Ss], [__T | __Stack]);
yeccpars2(242, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(243, '\\', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 244, [243 | __Ss], [__T | __Stack]);
yeccpars2(243, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {utype,[],[element(3,__1)]},
 yeccpars2(yeccgoto(utype, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(244, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 245, [244 | __Ss], [__T | __Stack]);
yeccpars2(244, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 249, [244 | __Ss], [__T | __Stack]);
yeccpars2(244, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 246, [244 | __Ss], [__T | __Stack]);
yeccpars2(244, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(245, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 256, [245 | __Ss], [__T | __Stack]);
yeccpars2(245, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(246, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 254, [246 | __Ss], [__T | __Stack]);
yeccpars2(246, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {atom,element(3,__1)},
 yeccpars2(yeccgoto(type_tag, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(247, ',', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 252, [247 | __Ss], [__T | __Stack]);
yeccpars2(247, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1],
 yeccpars2(yeccgoto(type_tags, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(248, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {vcon,element(3,__1),__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(constraint, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(249, integer, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 250, [249 | __Ss], [__T | __Stack]);
yeccpars2(249, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(250, '}', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 251, [250 | __Ss], [__T | __Stack]);
yeccpars2(250, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(251, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {tuple,element(3,__2)},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(type_tag, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(252, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 245, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 249, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 246, [252 | __Ss], [__T | __Stack]);
yeccpars2(252, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(253, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(type_tags, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(254, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 255, [254 | __Ss], [__T | __Stack]);
yeccpars2(254, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(255, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {tag,element(3,__1)},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(type_tag, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(256, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  list,
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(type_tag, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(257, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [257 | __Ss], [__T | __Stack]);
yeccpars2(257, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [257 | __Ss], [__T | __Stack]);
yeccpars2(257, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [257 | __Ss], [__T | __Stack]);
yeccpars2(257, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [257 | __Ss], [__T | __Stack]);
yeccpars2(257, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [257 | __Ss], [__T | __Stack]);
yeccpars2(257, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(258, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {tcon,__1,__3},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(constraint, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(259, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [259 | __Ss], [__T | __Stack]);
yeccpars2(259, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [259 | __Ss], [__T | __Stack]);
yeccpars2(259, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [259 | __Ss], [__T | __Stack]);
yeccpars2(259, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 243, [259 | __Ss], [__T | __Stack]);
yeccpars2(259, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [259 | __Ss], [__T | __Stack]);
yeccpars2(259, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(260, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [__1|__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(constraints, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(261, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [261 | __Ss], [__T | __Stack]);
yeccpars2(261, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [261 | __Ss], [__T | __Stack]);
yeccpars2(261, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 263, [261 | __Ss], [__T | __Stack]);
yeccpars2(261, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [261 | __Ss], [__T | __Stack]);
yeccpars2(261, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(262, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1 ++ [__3],
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(ptypes, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(263, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {utype,__1,[element(3,__3)]},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(264, '[', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 224, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, atom, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 225, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, '(', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 223, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, var, __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 230, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, '{', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 231, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 266, [264 | __Ss], [__T | __Stack]);
yeccpars2(264, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(265, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {type,element(3,__1),__2},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(266, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  [],
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(utype_list, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(267, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 268, [267 | __Ss], [__T | __Stack]);
yeccpars2(267, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(268, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype_list, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(269, ']', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 270, [269 | __Ss], [__T | __Stack]);
yeccpars2(269, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(270, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {list,__2},
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(ptype, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(271, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {utype,[],[]},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(utype, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(272, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 273, [272 | __Ss], [__T | __Stack]);
yeccpars2(272, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(273, __Cat, __Ss,  [__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __2,
 __Nss = lists:nthtail(2, __Ss),
 yeccpars2(yeccgoto(utype, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(274, 'when', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 239, [274 | __Ss], [__T | __Stack]);
yeccpars2(274, __Cat, __Ss,  __Stack, __T, __Ts, __Tzr) ->
 __Val =  [],
 yeccpars2(275, __Cat, [274 | __Ss], [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(275, __Cat, __Ss,  [__6,__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  if element(3,__2) == type -> {type,element(2,__2),sig,__3,__5,__6}; true -> return_error(element(2,__2),"illegal type declaration") end,
 __Nss = lists:nthtail(5, __Ss),
 yeccpars2(yeccgoto(type_decl, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(276, __Cat, __Ss,  [__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  {element(3,__1),__2},
 __Nss = lists:nthtail(1, __Ss),
 yeccpars2(yeccgoto(type_header, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(277, ')', __Ss, __Stack, __T, __Ts, __Tzr) ->
 yeccpars1(__Ts, __Tzr, 279, [277 | __Ss], [__T | __Stack]);
yeccpars2(277, _, _, _, __T, _, _) ->
 yeccerror(__T);
yeccpars2(278, __Cat, __Ss,  [__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  __1,
 yeccpars2(yeccgoto(attr_val, hd(__Ss)), __Cat, __Ss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(279, __Cat, __Ss,  [__5,__4,__3,__2,__1|__Stack], __T, __Ts, __Tzr) ->
 __Val =  build_attribute(__2,__4),
 __Nss = lists:nthtail(4, __Ss),
 yeccpars2(yeccgoto(attribute, hd(__Nss)), __Cat, __Nss, [__Val | __Stack], __T, __Ts, __Tzr);
yeccpars2(__Other, _, _, _, _, _, _) ->
 exit({parser, __Other, missing_state_in_action_table}).

yeccgoto(form, 0) ->
 4;
yeccgoto(attribute, 0) ->
 3;
yeccgoto(attr_val, 218) ->
 277;
yeccgoto(type_decl, 0) ->
 11;
yeccgoto(type_header, 217) ->
 220;
yeccgoto(type_constraints, 228) ->
 238;
yeccgoto(type_constraints, 274) ->
 275;
yeccgoto(constraints, 239) ->
 241;
yeccgoto(constraints, 259) ->
 260;
yeccgoto(constraint, 239) ->
 240;
yeccgoto(constraint, 259) ->
 240;
yeccgoto(utype_list, 225) ->
 265;
yeccgoto(utype_list, 219) ->
 276;
yeccgoto(utype_tuple, 222) ->
 229;
yeccgoto(utype_tuple, 231) ->
 229;
yeccgoto(utype_tuple, 236) ->
 229;
yeccgoto(utype_tuple, 239) ->
 229;
yeccgoto(utype_tuple, 257) ->
 229;
yeccgoto(utype_tuple, 259) ->
 229;
yeccgoto(utype_tuple, 261) ->
 229;
yeccgoto(utype_tuple, 264) ->
 229;
yeccgoto(utype_tuple, 224) ->
 229;
yeccgoto(utype_tuple, 223) ->
 229;
yeccgoto(utype_tuple, 221) ->
 229;
yeccgoto(utypes, 231) ->
 233;
yeccgoto(utypes, 236) ->
 237;
yeccgoto(utypes, 264) ->
 267;
yeccgoto(utype, 222) ->
 228;
yeccgoto(utype, 231) ->
 232;
yeccgoto(utype, 236) ->
 232;
yeccgoto(utype, 239) ->
 242;
yeccgoto(utype, 257) ->
 258;
yeccgoto(utype, 259) ->
 242;
yeccgoto(utype, 264) ->
 232;
yeccgoto(utype, 224) ->
 269;
yeccgoto(utype, 223) ->
 272;
yeccgoto(utype, 221) ->
 274;
yeccgoto(ptypes, 222) ->
 227;
yeccgoto(ptypes, 231) ->
 227;
yeccgoto(ptypes, 236) ->
 227;
yeccgoto(ptypes, 239) ->
 227;
yeccgoto(ptypes, 257) ->
 227;
yeccgoto(ptypes, 259) ->
 227;
yeccgoto(ptypes, 264) ->
 227;
yeccgoto(ptypes, 224) ->
 227;
yeccgoto(ptypes, 223) ->
 227;
yeccgoto(ptypes, 221) ->
 227;
yeccgoto(ptype, 222) ->
 226;
yeccgoto(ptype, 231) ->
 226;
yeccgoto(ptype, 236) ->
 226;
yeccgoto(ptype, 239) ->
 226;
yeccgoto(ptype, 257) ->
 226;
yeccgoto(ptype, 259) ->
 226;
yeccgoto(ptype, 261) ->
 262;
yeccgoto(ptype, 264) ->
 226;
yeccgoto(ptype, 224) ->
 226;
yeccgoto(ptype, 223) ->
 226;
yeccgoto(ptype, 221) ->
 226;
yeccgoto(type_tags, 244) ->
 248;
yeccgoto(type_tags, 252) ->
 253;
yeccgoto(type_tag, 244) ->
 247;
yeccgoto(type_tag, 252) ->
 247;
yeccgoto(function, 0) ->
 5;
yeccgoto(function_clauses, 0) ->
 7;
yeccgoto(function_clauses, 207) ->
 209;
yeccgoto(function_clause, 0) ->
 6;
yeccgoto(function_clause, 207) ->
 6;
yeccgoto(clause_args, 14) ->
 18;
yeccgoto(clause_args, 208) ->
 210;
yeccgoto(clause_args, 2) ->
 215;
yeccgoto(clause_guard, 18) ->
 19;
yeccgoto(clause_guard, 72) ->
 73;
yeccgoto(clause_guard, 108) ->
 117;
yeccgoto(clause_guard, 210) ->
 211;
yeccgoto(clause_guard, 215) ->
 216;
yeccgoto(clause_body, 73) ->
 75;
yeccgoto(clause_body, 79) ->
 80;
yeccgoto(clause_body, 84) ->
 85;
yeccgoto(clause_body, 101) ->
 107;
yeccgoto(clause_body, 117) ->
 118;
yeccgoto(clause_body, 211) ->
 212;
yeccgoto(clause_body, 216) ->
 212;
yeccgoto(expr, 20) ->
 33;
yeccgoto(expr, 64) ->
 33;
yeccgoto(expr, 57) ->
 72;
yeccgoto(expr, 74) ->
 33;
yeccgoto(expr, 77) ->
 79;
yeccgoto(expr, 82) ->
 72;
yeccgoto(expr, 69) ->
 84;
yeccgoto(expr, 87) ->
 90;
yeccgoto(expr, 91) ->
 92;
yeccgoto(expr, 96) ->
 92;
yeccgoto(expr, 98) ->
 99;
yeccgoto(expr, 48) ->
 33;
yeccgoto(expr, 105) ->
 33;
yeccgoto(expr, 136) ->
 137;
yeccgoto(expr, 175) ->
 33;
yeccgoto(expr, 32) ->
 177;
yeccgoto(expr, 30) ->
 178;
yeccgoto(expr, 179) ->
 72;
yeccgoto(expr, 28) ->
 33;
yeccgoto(expr, 25) ->
 185;
yeccgoto(expr, 189) ->
 190;
yeccgoto(expr, 186) ->
 192;
yeccgoto(expr, 22) ->
 194;
yeccgoto(expr, 200) ->
 92;
yeccgoto(expr, 16) ->
 33;
yeccgoto(expr, 218) ->
 33;
yeccgoto(expr100, 20) ->
 34;
yeccgoto(expr100, 64) ->
 34;
yeccgoto(expr100, 57) ->
 34;
yeccgoto(expr100, 74) ->
 34;
yeccgoto(expr100, 77) ->
 34;
yeccgoto(expr100, 82) ->
 34;
yeccgoto(expr100, 69) ->
 34;
yeccgoto(expr100, 87) ->
 34;
yeccgoto(expr100, 91) ->
 34;
yeccgoto(expr100, 96) ->
 34;
yeccgoto(expr100, 98) ->
 34;
yeccgoto(expr100, 48) ->
 34;
yeccgoto(expr100, 105) ->
 34;
yeccgoto(expr100, 136) ->
 34;
yeccgoto(expr100, 172) ->
 173;
yeccgoto(expr100, 171) ->
 174;
yeccgoto(expr100, 175) ->
 34;
yeccgoto(expr100, 32) ->
 34;
yeccgoto(expr100, 30) ->
 34;
yeccgoto(expr100, 179) ->
 34;
yeccgoto(expr100, 28) ->
 34;
yeccgoto(expr100, 25) ->
 34;
yeccgoto(expr100, 189) ->
 34;
yeccgoto(expr100, 186) ->
 34;
yeccgoto(expr100, 22) ->
 34;
yeccgoto(expr100, 200) ->
 34;
yeccgoto(expr100, 16) ->
 34;
yeccgoto(expr100, 218) ->
 34;
yeccgoto(expr200, 20) ->
 35;
yeccgoto(expr200, 64) ->
 35;
yeccgoto(expr200, 57) ->
 35;
yeccgoto(expr200, 74) ->
 35;
yeccgoto(expr200, 77) ->
 35;
yeccgoto(expr200, 82) ->
 35;
yeccgoto(expr200, 69) ->
 35;
yeccgoto(expr200, 87) ->
 35;
yeccgoto(expr200, 91) ->
 35;
yeccgoto(expr200, 96) ->
 35;
yeccgoto(expr200, 98) ->
 35;
yeccgoto(expr200, 48) ->
 35;
yeccgoto(expr200, 105) ->
 35;
yeccgoto(expr200, 136) ->
 35;
yeccgoto(expr200, 172) ->
 35;
yeccgoto(expr200, 171) ->
 35;
yeccgoto(expr200, 175) ->
 35;
yeccgoto(expr200, 32) ->
 35;
yeccgoto(expr200, 30) ->
 35;
yeccgoto(expr200, 179) ->
 35;
yeccgoto(expr200, 28) ->
 35;
yeccgoto(expr200, 25) ->
 35;
yeccgoto(expr200, 189) ->
 35;
yeccgoto(expr200, 186) ->
 35;
yeccgoto(expr200, 22) ->
 35;
yeccgoto(expr200, 200) ->
 35;
yeccgoto(expr200, 16) ->
 35;
yeccgoto(expr200, 218) ->
 35;
yeccgoto(expr300, 20) ->
 36;
yeccgoto(expr300, 64) ->
 36;
yeccgoto(expr300, 57) ->
 36;
yeccgoto(expr300, 74) ->
 36;
yeccgoto(expr300, 77) ->
 36;
yeccgoto(expr300, 82) ->
 36;
yeccgoto(expr300, 69) ->
 36;
yeccgoto(expr300, 87) ->
 36;
yeccgoto(expr300, 91) ->
 36;
yeccgoto(expr300, 96) ->
 36;
yeccgoto(expr300, 98) ->
 36;
yeccgoto(expr300, 48) ->
 36;
yeccgoto(expr300, 105) ->
 36;
yeccgoto(expr300, 136) ->
 36;
yeccgoto(expr300, 156) ->
 159;
yeccgoto(expr300, 169) ->
 170;
yeccgoto(expr300, 172) ->
 36;
yeccgoto(expr300, 171) ->
 36;
yeccgoto(expr300, 175) ->
 36;
yeccgoto(expr300, 32) ->
 36;
yeccgoto(expr300, 30) ->
 36;
yeccgoto(expr300, 179) ->
 36;
yeccgoto(expr300, 28) ->
 36;
yeccgoto(expr300, 25) ->
 36;
yeccgoto(expr300, 189) ->
 36;
yeccgoto(expr300, 186) ->
 36;
yeccgoto(expr300, 22) ->
 36;
yeccgoto(expr300, 200) ->
 36;
yeccgoto(expr300, 16) ->
 36;
yeccgoto(expr300, 218) ->
 36;
yeccgoto(expr400, 20) ->
 37;
yeccgoto(expr400, 64) ->
 37;
yeccgoto(expr400, 57) ->
 37;
yeccgoto(expr400, 74) ->
 37;
yeccgoto(expr400, 77) ->
 37;
yeccgoto(expr400, 82) ->
 37;
yeccgoto(expr400, 69) ->
 37;
yeccgoto(expr400, 87) ->
 37;
yeccgoto(expr400, 91) ->
 37;
yeccgoto(expr400, 96) ->
 37;
yeccgoto(expr400, 98) ->
 37;
yeccgoto(expr400, 48) ->
 37;
yeccgoto(expr400, 105) ->
 37;
yeccgoto(expr400, 136) ->
 37;
yeccgoto(expr400, 156) ->
 37;
yeccgoto(expr400, 169) ->
 37;
yeccgoto(expr400, 172) ->
 37;
yeccgoto(expr400, 171) ->
 37;
yeccgoto(expr400, 175) ->
 37;
yeccgoto(expr400, 32) ->
 37;
yeccgoto(expr400, 30) ->
 37;
yeccgoto(expr400, 179) ->
 37;
yeccgoto(expr400, 28) ->
 37;
yeccgoto(expr400, 25) ->
 37;
yeccgoto(expr400, 189) ->
 37;
yeccgoto(expr400, 186) ->
 37;
yeccgoto(expr400, 22) ->
 37;
yeccgoto(expr400, 200) ->
 37;
yeccgoto(expr400, 16) ->
 37;
yeccgoto(expr400, 218) ->
 37;
yeccgoto(expr500, 20) ->
 38;
yeccgoto(expr500, 64) ->
 38;
yeccgoto(expr500, 57) ->
 38;
yeccgoto(expr500, 74) ->
 38;
yeccgoto(expr500, 77) ->
 38;
yeccgoto(expr500, 82) ->
 38;
yeccgoto(expr500, 69) ->
 38;
yeccgoto(expr500, 87) ->
 38;
yeccgoto(expr500, 91) ->
 38;
yeccgoto(expr500, 96) ->
 38;
yeccgoto(expr500, 98) ->
 38;
yeccgoto(expr500, 48) ->
 38;
yeccgoto(expr500, 105) ->
 38;
yeccgoto(expr500, 136) ->
 38;
yeccgoto(expr500, 156) ->
 38;
yeccgoto(expr500, 151) ->
 160;
yeccgoto(expr500, 169) ->
 38;
yeccgoto(expr500, 172) ->
 38;
yeccgoto(expr500, 171) ->
 38;
yeccgoto(expr500, 175) ->
 38;
yeccgoto(expr500, 32) ->
 38;
yeccgoto(expr500, 30) ->
 38;
yeccgoto(expr500, 179) ->
 38;
yeccgoto(expr500, 28) ->
 38;
yeccgoto(expr500, 25) ->
 38;
yeccgoto(expr500, 189) ->
 38;
yeccgoto(expr500, 186) ->
 38;
yeccgoto(expr500, 22) ->
 38;
yeccgoto(expr500, 200) ->
 38;
yeccgoto(expr500, 16) ->
 38;
yeccgoto(expr500, 218) ->
 38;
yeccgoto(expr600, 20) ->
 39;
yeccgoto(expr600, 64) ->
 39;
yeccgoto(expr600, 57) ->
 39;
yeccgoto(expr600, 74) ->
 39;
yeccgoto(expr600, 77) ->
 39;
yeccgoto(expr600, 82) ->
 39;
yeccgoto(expr600, 69) ->
 39;
yeccgoto(expr600, 87) ->
 39;
yeccgoto(expr600, 91) ->
 39;
yeccgoto(expr600, 96) ->
 39;
yeccgoto(expr600, 98) ->
 39;
yeccgoto(expr600, 48) ->
 39;
yeccgoto(expr600, 105) ->
 39;
yeccgoto(expr600, 136) ->
 39;
yeccgoto(expr600, 144) ->
 146;
yeccgoto(expr600, 156) ->
 39;
yeccgoto(expr600, 151) ->
 39;
yeccgoto(expr600, 169) ->
 39;
yeccgoto(expr600, 172) ->
 39;
yeccgoto(expr600, 171) ->
 39;
yeccgoto(expr600, 175) ->
 39;
yeccgoto(expr600, 32) ->
 39;
yeccgoto(expr600, 30) ->
 39;
yeccgoto(expr600, 179) ->
 39;
yeccgoto(expr600, 28) ->
 39;
yeccgoto(expr600, 25) ->
 39;
yeccgoto(expr600, 189) ->
 39;
yeccgoto(expr600, 186) ->
 39;
yeccgoto(expr600, 22) ->
 39;
yeccgoto(expr600, 200) ->
 39;
yeccgoto(expr600, 16) ->
 39;
yeccgoto(expr600, 218) ->
 39;
yeccgoto(expr_700, 20) ->
 40;
yeccgoto(expr_700, 64) ->
 40;
yeccgoto(expr_700, 57) ->
 40;
yeccgoto(expr_700, 74) ->
 40;
yeccgoto(expr_700, 77) ->
 40;
yeccgoto(expr_700, 82) ->
 40;
yeccgoto(expr_700, 69) ->
 40;
yeccgoto(expr_700, 87) ->
 40;
yeccgoto(expr_700, 91) ->
 40;
yeccgoto(expr_700, 96) ->
 40;
yeccgoto(expr_700, 98) ->
 40;
yeccgoto(expr_700, 54) ->
 100;
yeccgoto(expr_700, 48) ->
 40;
yeccgoto(expr_700, 105) ->
 40;
yeccgoto(expr_700, 136) ->
 40;
yeccgoto(expr_700, 144) ->
 40;
yeccgoto(expr_700, 156) ->
 40;
yeccgoto(expr_700, 151) ->
 40;
yeccgoto(expr_700, 169) ->
 40;
yeccgoto(expr_700, 172) ->
 40;
yeccgoto(expr_700, 171) ->
 40;
yeccgoto(expr_700, 175) ->
 40;
yeccgoto(expr_700, 32) ->
 40;
yeccgoto(expr_700, 30) ->
 40;
yeccgoto(expr_700, 179) ->
 40;
yeccgoto(expr_700, 28) ->
 40;
yeccgoto(expr_700, 25) ->
 40;
yeccgoto(expr_700, 189) ->
 40;
yeccgoto(expr_700, 186) ->
 40;
yeccgoto(expr_700, 22) ->
 40;
yeccgoto(expr_700, 200) ->
 40;
yeccgoto(expr_700, 16) ->
 40;
yeccgoto(expr_700, 218) ->
 40;
yeccgoto(expr_800, 20) ->
 41;
yeccgoto(expr_800, 64) ->
 41;
yeccgoto(expr_800, 57) ->
 41;
yeccgoto(expr_800, 74) ->
 41;
yeccgoto(expr_800, 77) ->
 41;
yeccgoto(expr_800, 82) ->
 41;
yeccgoto(expr_800, 69) ->
 41;
yeccgoto(expr_800, 87) ->
 41;
yeccgoto(expr_800, 91) ->
 41;
yeccgoto(expr_800, 96) ->
 41;
yeccgoto(expr_800, 98) ->
 41;
yeccgoto(expr_800, 54) ->
 41;
yeccgoto(expr_800, 48) ->
 41;
yeccgoto(expr_800, 105) ->
 41;
yeccgoto(expr_800, 136) ->
 41;
yeccgoto(expr_800, 144) ->
 41;
yeccgoto(expr_800, 156) ->
 41;
yeccgoto(expr_800, 151) ->
 41;
yeccgoto(expr_800, 169) ->
 41;
yeccgoto(expr_800, 172) ->
 41;
yeccgoto(expr_800, 171) ->
 41;
yeccgoto(expr_800, 175) ->
 41;
yeccgoto(expr_800, 32) ->
 41;
yeccgoto(expr_800, 30) ->
 41;
yeccgoto(expr_800, 179) ->
 41;
yeccgoto(expr_800, 28) ->
 41;
yeccgoto(expr_800, 25) ->
 41;
yeccgoto(expr_800, 189) ->
 41;
yeccgoto(expr_800, 186) ->
 41;
yeccgoto(expr_800, 22) ->
 41;
yeccgoto(expr_800, 200) ->
 41;
yeccgoto(expr_800, 16) ->
 41;
yeccgoto(expr_800, 218) ->
 41;
yeccgoto(expr_max, 20) ->
 42;
yeccgoto(expr_max, 64) ->
 42;
yeccgoto(expr_max, 57) ->
 42;
yeccgoto(expr_max, 74) ->
 42;
yeccgoto(expr_max, 77) ->
 42;
yeccgoto(expr_max, 82) ->
 42;
yeccgoto(expr_max, 69) ->
 42;
yeccgoto(expr_max, 87) ->
 42;
yeccgoto(expr_max, 91) ->
 42;
yeccgoto(expr_max, 96) ->
 42;
yeccgoto(expr_max, 98) ->
 42;
yeccgoto(expr_max, 54) ->
 42;
yeccgoto(expr_max, 48) ->
 42;
yeccgoto(expr_max, 105) ->
 42;
yeccgoto(expr_max, 119) ->
 120;
yeccgoto(expr_max, 136) ->
 42;
yeccgoto(expr_max, 144) ->
 42;
yeccgoto(expr_max, 156) ->
 42;
yeccgoto(expr_max, 151) ->
 42;
yeccgoto(expr_max, 169) ->
 42;
yeccgoto(expr_max, 172) ->
 42;
yeccgoto(expr_max, 171) ->
 42;
yeccgoto(expr_max, 175) ->
 42;
yeccgoto(expr_max, 32) ->
 42;
yeccgoto(expr_max, 30) ->
 42;
yeccgoto(expr_max, 179) ->
 42;
yeccgoto(expr_max, 28) ->
 42;
yeccgoto(expr_max, 25) ->
 42;
yeccgoto(expr_max, 189) ->
 42;
yeccgoto(expr_max, 186) ->
 42;
yeccgoto(expr_max, 22) ->
 42;
yeccgoto(expr_max, 200) ->
 42;
yeccgoto(expr_max, 16) ->
 42;
yeccgoto(expr_max, 218) ->
 42;
yeccgoto(list, 20) ->
 51;
yeccgoto(list, 64) ->
 51;
yeccgoto(list, 57) ->
 51;
yeccgoto(list, 74) ->
 51;
yeccgoto(list, 77) ->
 51;
yeccgoto(list, 82) ->
 51;
yeccgoto(list, 69) ->
 51;
yeccgoto(list, 87) ->
 51;
yeccgoto(list, 91) ->
 51;
yeccgoto(list, 96) ->
 51;
yeccgoto(list, 98) ->
 51;
yeccgoto(list, 54) ->
 51;
yeccgoto(list, 48) ->
 51;
yeccgoto(list, 105) ->
 51;
yeccgoto(list, 119) ->
 51;
yeccgoto(list, 136) ->
 51;
yeccgoto(list, 144) ->
 51;
yeccgoto(list, 156) ->
 51;
yeccgoto(list, 151) ->
 51;
yeccgoto(list, 169) ->
 51;
yeccgoto(list, 172) ->
 51;
yeccgoto(list, 171) ->
 51;
yeccgoto(list, 175) ->
 51;
yeccgoto(list, 32) ->
 51;
yeccgoto(list, 30) ->
 51;
yeccgoto(list, 179) ->
 51;
yeccgoto(list, 28) ->
 51;
yeccgoto(list, 25) ->
 51;
yeccgoto(list, 189) ->
 51;
yeccgoto(list, 186) ->
 51;
yeccgoto(list, 22) ->
 51;
yeccgoto(list, 200) ->
 51;
yeccgoto(list, 16) ->
 51;
yeccgoto(list, 218) ->
 51;
yeccgoto(tail, 185) ->
 188;
yeccgoto(tail, 192) ->
 193;
yeccgoto(list_comprehension, 20) ->
 52;
yeccgoto(list_comprehension, 64) ->
 52;
yeccgoto(list_comprehension, 57) ->
 52;
yeccgoto(list_comprehension, 74) ->
 52;
yeccgoto(list_comprehension, 77) ->
 52;
yeccgoto(list_comprehension, 82) ->
 52;
yeccgoto(list_comprehension, 69) ->
 52;
yeccgoto(list_comprehension, 55) ->
 88;
yeccgoto(list_comprehension, 87) ->
 52;
yeccgoto(list_comprehension, 91) ->
 52;
yeccgoto(list_comprehension, 96) ->
 52;
yeccgoto(list_comprehension, 98) ->
 52;
yeccgoto(list_comprehension, 54) ->
 52;
yeccgoto(list_comprehension, 48) ->
 52;
yeccgoto(list_comprehension, 105) ->
 52;
yeccgoto(list_comprehension, 119) ->
 52;
yeccgoto(list_comprehension, 136) ->
 52;
yeccgoto(list_comprehension, 144) ->
 52;
yeccgoto(list_comprehension, 156) ->
 52;
yeccgoto(list_comprehension, 151) ->
 52;
yeccgoto(list_comprehension, 169) ->
 52;
yeccgoto(list_comprehension, 172) ->
 52;
yeccgoto(list_comprehension, 171) ->
 52;
yeccgoto(list_comprehension, 175) ->
 52;
yeccgoto(list_comprehension, 32) ->
 52;
yeccgoto(list_comprehension, 30) ->
 52;
yeccgoto(list_comprehension, 179) ->
 52;
yeccgoto(list_comprehension, 28) ->
 52;
yeccgoto(list_comprehension, 25) ->
 52;
yeccgoto(list_comprehension, 189) ->
 52;
yeccgoto(list_comprehension, 186) ->
 52;
yeccgoto(list_comprehension, 22) ->
 52;
yeccgoto(list_comprehension, 200) ->
 52;
yeccgoto(list_comprehension, 16) ->
 52;
yeccgoto(list_comprehension, 218) ->
 52;
yeccgoto(lc_expr, 91) ->
 93;
yeccgoto(lc_expr, 96) ->
 93;
yeccgoto(lc_expr, 200) ->
 93;
yeccgoto(lc_exprs, 91) ->
 94;
yeccgoto(lc_exprs, 96) ->
 97;
yeccgoto(lc_exprs, 200) ->
 202;
yeccgoto(tuple, 20) ->
 62;
yeccgoto(tuple, 64) ->
 62;
yeccgoto(tuple, 57) ->
 62;
yeccgoto(tuple, 74) ->
 62;
yeccgoto(tuple, 77) ->
 62;
yeccgoto(tuple, 82) ->
 62;
yeccgoto(tuple, 69) ->
 62;
yeccgoto(tuple, 87) ->
 62;
yeccgoto(tuple, 91) ->
 62;
yeccgoto(tuple, 96) ->
 62;
yeccgoto(tuple, 98) ->
 62;
yeccgoto(tuple, 54) ->
 62;
yeccgoto(tuple, 48) ->
 62;
yeccgoto(tuple, 105) ->
 62;
yeccgoto(tuple, 119) ->
 62;
yeccgoto(tuple, 136) ->
 62;
yeccgoto(tuple, 144) ->
 62;
yeccgoto(tuple, 156) ->
 62;
yeccgoto(tuple, 151) ->
 62;
yeccgoto(tuple, 169) ->
 62;
yeccgoto(tuple, 172) ->
 62;
yeccgoto(tuple, 171) ->
 62;
yeccgoto(tuple, 175) ->
 62;
yeccgoto(tuple, 32) ->
 62;
yeccgoto(tuple, 30) ->
 62;
yeccgoto(tuple, 179) ->
 62;
yeccgoto(tuple, 28) ->
 62;
yeccgoto(tuple, 25) ->
 62;
yeccgoto(tuple, 189) ->
 62;
yeccgoto(tuple, 186) ->
 62;
yeccgoto(tuple, 22) ->
 62;
yeccgoto(tuple, 200) ->
 62;
yeccgoto(tuple, 16) ->
 62;
yeccgoto(tuple, 218) ->
 62;
yeccgoto(record_expr, 20) ->
 59;
yeccgoto(record_expr, 64) ->
 59;
yeccgoto(record_expr, 57) ->
 59;
yeccgoto(record_expr, 74) ->
 59;
yeccgoto(record_expr, 77) ->
 59;
yeccgoto(record_expr, 82) ->
 59;
yeccgoto(record_expr, 69) ->
 59;
yeccgoto(record_expr, 87) ->
 59;
yeccgoto(record_expr, 91) ->
 59;
yeccgoto(record_expr, 96) ->
 59;
yeccgoto(record_expr, 98) ->
 59;
yeccgoto(record_expr, 54) ->
 59;
yeccgoto(record_expr, 48) ->
 59;
yeccgoto(record_expr, 105) ->
 59;
yeccgoto(record_expr, 136) ->
 59;
yeccgoto(record_expr, 144) ->
 59;
yeccgoto(record_expr, 156) ->
 59;
yeccgoto(record_expr, 151) ->
 59;
yeccgoto(record_expr, 169) ->
 59;
yeccgoto(record_expr, 172) ->
 59;
yeccgoto(record_expr, 171) ->
 59;
yeccgoto(record_expr, 175) ->
 59;
yeccgoto(record_expr, 32) ->
 59;
yeccgoto(record_expr, 30) ->
 59;
yeccgoto(record_expr, 179) ->
 59;
yeccgoto(record_expr, 28) ->
 59;
yeccgoto(record_expr, 25) ->
 59;
yeccgoto(record_expr, 189) ->
 59;
yeccgoto(record_expr, 186) ->
 59;
yeccgoto(record_expr, 22) ->
 59;
yeccgoto(record_expr, 200) ->
 59;
yeccgoto(record_expr, 16) ->
 59;
yeccgoto(record_expr, 218) ->
 59;
yeccgoto(record_tuple, 125) ->
 127;
yeccgoto(record_tuple, 196) ->
 198;
yeccgoto(record_field, 128) ->
 130;
yeccgoto(record_field, 134) ->
 130;
yeccgoto(record_fields, 128) ->
 131;
yeccgoto(record_fields, 134) ->
 135;
yeccgoto(if_expr, 20) ->
 49;
yeccgoto(if_expr, 64) ->
 49;
yeccgoto(if_expr, 57) ->
 49;
yeccgoto(if_expr, 74) ->
 49;
yeccgoto(if_expr, 77) ->
 49;
yeccgoto(if_expr, 82) ->
 49;
yeccgoto(if_expr, 69) ->
 49;
yeccgoto(if_expr, 87) ->
 49;
yeccgoto(if_expr, 91) ->
 49;
yeccgoto(if_expr, 96) ->
 49;
yeccgoto(if_expr, 98) ->
 49;
yeccgoto(if_expr, 54) ->
 49;
yeccgoto(if_expr, 48) ->
 49;
yeccgoto(if_expr, 105) ->
 49;
yeccgoto(if_expr, 119) ->
 49;
yeccgoto(if_expr, 136) ->
 49;
yeccgoto(if_expr, 144) ->
 49;
yeccgoto(if_expr, 156) ->
 49;
yeccgoto(if_expr, 151) ->
 49;
yeccgoto(if_expr, 169) ->
 49;
yeccgoto(if_expr, 172) ->
 49;
yeccgoto(if_expr, 171) ->
 49;
yeccgoto(if_expr, 175) ->
 49;
yeccgoto(if_expr, 32) ->
 49;
yeccgoto(if_expr, 30) ->
 49;
yeccgoto(if_expr, 179) ->
 49;
yeccgoto(if_expr, 28) ->
 49;
yeccgoto(if_expr, 25) ->
 49;
yeccgoto(if_expr, 189) ->
 49;
yeccgoto(if_expr, 186) ->
 49;
yeccgoto(if_expr, 22) ->
 49;
yeccgoto(if_expr, 200) ->
 49;
yeccgoto(if_expr, 16) ->
 49;
yeccgoto(if_expr, 218) ->
 49;
yeccgoto(if_clause, 48) ->
 102;
yeccgoto(if_clause, 105) ->
 102;
yeccgoto(if_clauses, 48) ->
 103;
yeccgoto(if_clauses, 105) ->
 106;
yeccgoto(case_expr, 20) ->
 31;
yeccgoto(case_expr, 64) ->
 31;
yeccgoto(case_expr, 57) ->
 31;
yeccgoto(case_expr, 74) ->
 31;
yeccgoto(case_expr, 77) ->
 31;
yeccgoto(case_expr, 82) ->
 31;
yeccgoto(case_expr, 69) ->
 31;
yeccgoto(case_expr, 87) ->
 31;
yeccgoto(case_expr, 91) ->
 31;
yeccgoto(case_expr, 96) ->
 31;
yeccgoto(case_expr, 98) ->
 31;
yeccgoto(case_expr, 54) ->
 31;
yeccgoto(case_expr, 48) ->
 31;
yeccgoto(case_expr, 105) ->
 31;
yeccgoto(case_expr, 119) ->
 31;
yeccgoto(case_expr, 136) ->
 31;
yeccgoto(case_expr, 144) ->
 31;
yeccgoto(case_expr, 156) ->
 31;
yeccgoto(case_expr, 151) ->
 31;
yeccgoto(case_expr, 169) ->
 31;
yeccgoto(case_expr, 172) ->
 31;
yeccgoto(case_expr, 171) ->
 31;
yeccgoto(case_expr, 175) ->
 31;
yeccgoto(case_expr, 32) ->
 31;
yeccgoto(case_expr, 30) ->
 31;
yeccgoto(case_expr, 179) ->
 31;
yeccgoto(case_expr, 28) ->
 31;
yeccgoto(case_expr, 25) ->
 31;
yeccgoto(case_expr, 189) ->
 31;
yeccgoto(case_expr, 186) ->
 31;
yeccgoto(case_expr, 22) ->
 31;
yeccgoto(case_expr, 200) ->
 31;
yeccgoto(case_expr, 16) ->
 31;
yeccgoto(case_expr, 218) ->
 31;
yeccgoto(cr_clause, 57) ->
 70;
yeccgoto(cr_clause, 82) ->
 70;
yeccgoto(cr_clause, 179) ->
 70;
yeccgoto(cr_clauses, 57) ->
 71;
yeccgoto(cr_clauses, 82) ->
 83;
yeccgoto(cr_clauses, 179) ->
 180;
yeccgoto(receive_expr, 20) ->
 58;
yeccgoto(receive_expr, 64) ->
 58;
yeccgoto(receive_expr, 57) ->
 58;
yeccgoto(receive_expr, 74) ->
 58;
yeccgoto(receive_expr, 77) ->
 58;
yeccgoto(receive_expr, 82) ->
 58;
yeccgoto(receive_expr, 69) ->
 58;
yeccgoto(receive_expr, 87) ->
 58;
yeccgoto(receive_expr, 91) ->
 58;
yeccgoto(receive_expr, 96) ->
 58;
yeccgoto(receive_expr, 98) ->
 58;
yeccgoto(receive_expr, 54) ->
 58;
yeccgoto(receive_expr, 48) ->
 58;
yeccgoto(receive_expr, 105) ->
 58;
yeccgoto(receive_expr, 119) ->
 58;
yeccgoto(receive_expr, 136) ->
 58;
yeccgoto(receive_expr, 144) ->
 58;
yeccgoto(receive_expr, 156) ->
 58;
yeccgoto(receive_expr, 151) ->
 58;
yeccgoto(receive_expr, 169) ->
 58;
yeccgoto(receive_expr, 172) ->
 58;
yeccgoto(receive_expr, 171) ->
 58;
yeccgoto(receive_expr, 175) ->
 58;
yeccgoto(receive_expr, 32) ->
 58;
yeccgoto(receive_expr, 30) ->
 58;
yeccgoto(receive_expr, 179) ->
 58;
yeccgoto(receive_expr, 28) ->
 58;
yeccgoto(receive_expr, 25) ->
 58;
yeccgoto(receive_expr, 189) ->
 58;
yeccgoto(receive_expr, 186) ->
 58;
yeccgoto(receive_expr, 22) ->
 58;
yeccgoto(receive_expr, 200) ->
 58;
yeccgoto(receive_expr, 16) ->
 58;
yeccgoto(receive_expr, 218) ->
 58;
yeccgoto(fun_expr, 20) ->
 46;
yeccgoto(fun_expr, 64) ->
 46;
yeccgoto(fun_expr, 57) ->
 46;
yeccgoto(fun_expr, 74) ->
 46;
yeccgoto(fun_expr, 77) ->
 46;
yeccgoto(fun_expr, 82) ->
 46;
yeccgoto(fun_expr, 69) ->
 46;
yeccgoto(fun_expr, 87) ->
 46;
yeccgoto(fun_expr, 91) ->
 46;
yeccgoto(fun_expr, 96) ->
 46;
yeccgoto(fun_expr, 98) ->
 46;
yeccgoto(fun_expr, 54) ->
 46;
yeccgoto(fun_expr, 48) ->
 46;
yeccgoto(fun_expr, 105) ->
 46;
yeccgoto(fun_expr, 119) ->
 46;
yeccgoto(fun_expr, 136) ->
 46;
yeccgoto(fun_expr, 144) ->
 46;
yeccgoto(fun_expr, 156) ->
 46;
yeccgoto(fun_expr, 151) ->
 46;
yeccgoto(fun_expr, 169) ->
 46;
yeccgoto(fun_expr, 172) ->
 46;
yeccgoto(fun_expr, 171) ->
 46;
yeccgoto(fun_expr, 175) ->
 46;
yeccgoto(fun_expr, 32) ->
 46;
yeccgoto(fun_expr, 30) ->
 46;
yeccgoto(fun_expr, 179) ->
 46;
yeccgoto(fun_expr, 28) ->
 46;
yeccgoto(fun_expr, 25) ->
 46;
yeccgoto(fun_expr, 189) ->
 46;
yeccgoto(fun_expr, 186) ->
 46;
yeccgoto(fun_expr, 22) ->
 46;
yeccgoto(fun_expr, 200) ->
 46;
yeccgoto(fun_expr, 16) ->
 46;
yeccgoto(fun_expr, 218) ->
 46;
yeccgoto(fun_clause, 45) ->
 110;
yeccgoto(fun_clause, 113) ->
 110;
yeccgoto(fun_clauses, 45) ->
 111;
yeccgoto(fun_clauses, 113) ->
 114;
yeccgoto(query_expr, 20) ->
 56;
yeccgoto(query_expr, 64) ->
 56;
yeccgoto(query_expr, 57) ->
 56;
yeccgoto(query_expr, 74) ->
 56;
yeccgoto(query_expr, 77) ->
 56;
yeccgoto(query_expr, 82) ->
 56;
yeccgoto(query_expr, 69) ->
 56;
yeccgoto(query_expr, 87) ->
 56;
yeccgoto(query_expr, 91) ->
 56;
yeccgoto(query_expr, 96) ->
 56;
yeccgoto(query_expr, 98) ->
 56;
yeccgoto(query_expr, 54) ->
 56;
yeccgoto(query_expr, 48) ->
 56;
yeccgoto(query_expr, 105) ->
 56;
yeccgoto(query_expr, 119) ->
 56;
yeccgoto(query_expr, 136) ->
 56;
yeccgoto(query_expr, 144) ->
 56;
yeccgoto(query_expr, 156) ->
 56;
yeccgoto(query_expr, 151) ->
 56;
yeccgoto(query_expr, 169) ->
 56;
yeccgoto(query_expr, 172) ->
 56;
yeccgoto(query_expr, 171) ->
 56;
yeccgoto(query_expr, 175) ->
 56;
yeccgoto(query_expr, 32) ->
 56;
yeccgoto(query_expr, 30) ->
 56;
yeccgoto(query_expr, 179) ->
 56;
yeccgoto(query_expr, 28) ->
 56;
yeccgoto(query_expr, 25) ->
 56;
yeccgoto(query_expr, 189) ->
 56;
yeccgoto(query_expr, 186) ->
 56;
yeccgoto(query_expr, 22) ->
 56;
yeccgoto(query_expr, 200) ->
 56;
yeccgoto(query_expr, 16) ->
 56;
yeccgoto(query_expr, 218) ->
 56;
yeccgoto(argument_list, 14) ->
 17;
yeccgoto(argument_list, 45) ->
 108;
yeccgoto(argument_list, 113) ->
 108;
yeccgoto(argument_list, 41) ->
 123;
yeccgoto(argument_list, 208) ->
 17;
yeccgoto(argument_list, 2) ->
 17;
yeccgoto(exprs, 20) ->
 43;
yeccgoto(exprs, 64) ->
 65;
yeccgoto(exprs, 74) ->
 76;
yeccgoto(exprs, 48) ->
 43;
yeccgoto(exprs, 105) ->
 43;
yeccgoto(exprs, 175) ->
 176;
yeccgoto(exprs, 28) ->
 182;
yeccgoto(exprs, 16) ->
 204;
yeccgoto(exprs, 218) ->
 278;
yeccgoto(guard, 20) ->
 47;
yeccgoto(guard, 48) ->
 101;
yeccgoto(guard, 105) ->
 101;
yeccgoto(atomic, 20) ->
 27;
yeccgoto(atomic, 64) ->
 27;
yeccgoto(atomic, 57) ->
 27;
yeccgoto(atomic, 74) ->
 27;
yeccgoto(atomic, 77) ->
 27;
yeccgoto(atomic, 82) ->
 27;
yeccgoto(atomic, 69) ->
 27;
yeccgoto(atomic, 87) ->
 27;
yeccgoto(atomic, 91) ->
 27;
yeccgoto(atomic, 96) ->
 27;
yeccgoto(atomic, 98) ->
 27;
yeccgoto(atomic, 54) ->
 27;
yeccgoto(atomic, 48) ->
 27;
yeccgoto(atomic, 105) ->
 27;
yeccgoto(atomic, 119) ->
 27;
yeccgoto(atomic, 136) ->
 27;
yeccgoto(atomic, 144) ->
 27;
yeccgoto(atomic, 156) ->
 27;
yeccgoto(atomic, 151) ->
 27;
yeccgoto(atomic, 169) ->
 27;
yeccgoto(atomic, 172) ->
 27;
yeccgoto(atomic, 171) ->
 27;
yeccgoto(atomic, 175) ->
 27;
yeccgoto(atomic, 32) ->
 27;
yeccgoto(atomic, 30) ->
 27;
yeccgoto(atomic, 179) ->
 27;
yeccgoto(atomic, 28) ->
 27;
yeccgoto(atomic, 25) ->
 27;
yeccgoto(atomic, 189) ->
 27;
yeccgoto(atomic, 186) ->
 27;
yeccgoto(atomic, 22) ->
 27;
yeccgoto(atomic, 200) ->
 27;
yeccgoto(atomic, 16) ->
 27;
yeccgoto(atomic, 218) ->
 27;
yeccgoto(strings, 20) ->
 61;
yeccgoto(strings, 64) ->
 61;
yeccgoto(strings, 60) ->
 68;
yeccgoto(strings, 57) ->
 61;
yeccgoto(strings, 74) ->
 61;
yeccgoto(strings, 77) ->
 61;
yeccgoto(strings, 82) ->
 61;
yeccgoto(strings, 69) ->
 61;
yeccgoto(strings, 87) ->
 61;
yeccgoto(strings, 91) ->
 61;
yeccgoto(strings, 96) ->
 61;
yeccgoto(strings, 98) ->
 61;
yeccgoto(strings, 54) ->
 61;
yeccgoto(strings, 48) ->
 61;
yeccgoto(strings, 105) ->
 61;
yeccgoto(strings, 119) ->
 61;
yeccgoto(strings, 136) ->
 61;
yeccgoto(strings, 144) ->
 61;
yeccgoto(strings, 156) ->
 61;
yeccgoto(strings, 151) ->
 61;
yeccgoto(strings, 169) ->
 61;
yeccgoto(strings, 172) ->
 61;
yeccgoto(strings, 171) ->
 61;
yeccgoto(strings, 175) ->
 61;
yeccgoto(strings, 32) ->
 61;
yeccgoto(strings, 30) ->
 61;
yeccgoto(strings, 179) ->
 61;
yeccgoto(strings, 28) ->
 61;
yeccgoto(strings, 25) ->
 61;
yeccgoto(strings, 189) ->
 61;
yeccgoto(strings, 186) ->
 61;
yeccgoto(strings, 22) ->
 61;
yeccgoto(strings, 200) ->
 61;
yeccgoto(strings, 16) ->
 61;
yeccgoto(strings, 218) ->
 61;
yeccgoto(prefix_op, 20) ->
 54;
yeccgoto(prefix_op, 64) ->
 54;
yeccgoto(prefix_op, 57) ->
 54;
yeccgoto(prefix_op, 74) ->
 54;
yeccgoto(prefix_op, 77) ->
 54;
yeccgoto(prefix_op, 82) ->
 54;
yeccgoto(prefix_op, 69) ->
 54;
yeccgoto(prefix_op, 87) ->
 54;
yeccgoto(prefix_op, 91) ->
 54;
yeccgoto(prefix_op, 96) ->
 54;
yeccgoto(prefix_op, 98) ->
 54;
yeccgoto(prefix_op, 48) ->
 54;
yeccgoto(prefix_op, 105) ->
 54;
yeccgoto(prefix_op, 136) ->
 54;
yeccgoto(prefix_op, 144) ->
 54;
yeccgoto(prefix_op, 156) ->
 54;
yeccgoto(prefix_op, 151) ->
 54;
yeccgoto(prefix_op, 169) ->
 54;
yeccgoto(prefix_op, 172) ->
 54;
yeccgoto(prefix_op, 171) ->
 54;
yeccgoto(prefix_op, 175) ->
 54;
yeccgoto(prefix_op, 32) ->
 54;
yeccgoto(prefix_op, 30) ->
 54;
yeccgoto(prefix_op, 179) ->
 54;
yeccgoto(prefix_op, 28) ->
 54;
yeccgoto(prefix_op, 25) ->
 54;
yeccgoto(prefix_op, 189) ->
 54;
yeccgoto(prefix_op, 186) ->
 54;
yeccgoto(prefix_op, 22) ->
 54;
yeccgoto(prefix_op, 200) ->
 54;
yeccgoto(prefix_op, 16) ->
 54;
yeccgoto(prefix_op, 218) ->
 54;
yeccgoto(mult_op, 38) ->
 144;
yeccgoto(mult_op, 160) ->
 144;
yeccgoto(add_op, 37) ->
 151;
yeccgoto(list_op, 37) ->
 156;
yeccgoto(comp_op, 36) ->
 169;
yeccgoto(rule, 0) ->
 8;
yeccgoto(rule_clauses, 0) ->
 10;
yeccgoto(rule_clauses, 13) ->
 15;
yeccgoto(rule_clause, 0) ->
 9;
yeccgoto(rule_clause, 13) ->
 9;
yeccgoto(rule_body, 19) ->
 201;
yeccgoto(rule_body, 216) ->
 201;
yeccgoto(__Symbol, __State) ->
 exit({__Symbol, __State, missing_in_goto_table}).


