import line_style
import legend
import axis
import pychart_util
import chart_object
import fill_style
import types
import math
import canvas
import area_doc

def CoordSystemType(s):
    if s == 'linear' or s == 'log' or s == 'category':
        return ""
    else:
        return s + """: Unsupported coordinate system. 
The value must be one of 'linear', 'log', or 'category'."""

_dummy_legend = legend.T()

# LEN: the length of the axis (in points)
# VAL: the value 
# RANGE: the min and max values of the axis
# SKIP: None or the region that should be compressed by a zigzag.
zigZagSize=30

def linear_coord_pos(len, val, range, skip):
    if skip == None:
	return len * (val - range[0]) / float(range[1]-range[0])

    sec1 = skip[0] - range[0]
    sec2 = range[1] - skip[1]
    len_low = (len - zigZagSize) * sec1 / (sec1+sec2)
    len_high = (len - zigZagSize) * sec2 / (sec1+sec2)

    if val < skip[0]:
	range_low = (range[0], skip[0])
	return linear_coord_pos(len_low, val, range_low, None)
    if val < skip[1]:
        # this part won't be drawn.
	return len_low + zigZagSize/2.0
    else:
	range_high = (skip[1], range[1])
	return len_low + zigZagSize + linear_coord_pos(len_high, val, range_high, None)

def log_coord_pos(len, val, range):
   xminl = math.log(range[0])
   xmaxl = math.log(range[1])
   if val <= 0:
       return 0
   vl = math.log(val)
   return len * (vl-xminl) / float(xmaxl-xminl)

def category_coord_pos(size, val, data, col):
    i = 0.5
    for v in data:
        if v[col] == val:
            return size * i / float(len(data))
        i = i + 1
    # the drawing area is clipped. So negative offset will make this plot
    # invisible.
    return canvas.invalid_coord;

_keys = {
    "loc" : (pychart_util.CoordType, 0, (0,0),
             "Bottom-left corner of the chart. "),
    "size" : (pychart_util.CoordType, 0, (120,110),
              """The size of the chart-drawing area, excluding axis labels,
              legends, tick marks, etc."""),
    "bg_style": (fill_style.T, 1, None, "Background fill-pattern."),
    "border_line_style": (line_style.T, 1, None, "Line style of the outer frameof the chart."),
    "x_coord_system":
    (CoordSystemType, 0, 'linear',
     """Either 'linear', 'log' or 'category'. "Linear" and "log" makes 
x coordinate scale linearly or logarithmically. "Category" enumerates the X
values through x_category_data and x_category_col attributes."""),
    "x_category_data":
    (types.ListType, 1, None,
     """Meaningful only when x_coord_system == 'category'.
     This attribute specifies the data-set from which the X values are extracted. See also x_category_col."""),
    "x_category_col":
    (types.IntType, 1, 0,
     "This attribute is meaningful only when x_coord_system == 'category'. This attribute selects the column of 'x_category_data' from which X values are computed."),
    "y_coord_system": (CoordSystemType, 0, 'linear',
                       "Set the Y coordinate scaling. See also x_coord_system."),
    "x_range": (pychart_util.CoordType, 1, None,
                "The minimum and maximum X values. "
                + " If either of them is None, its value is computed automatically. "),
    "y_range": (pychart_util.CoordType, 1, None,
               "The minimum and maximum Y values. "
                + " If either of them is None, its value is computed automatically. "),
    "y_category_data": (types.ListType, 1, None, "See x_category_data."),
    "y_category_col": (types.IntType, 1, 1, "See x_category_col."),
    "x_axis": (axis.X, 1, None, "The X axis. <<axis>>."),
    "y_axis": (axis.Y, 1, None, "The Y axis. <<axis>>."),
    "x_grid_style" : (line_style.T, 1, None,
                      "The style of X grid lines."),
    "y_grid_style" : (line_style.T, 1, line_style.gray70_dash3,
                      "The style of Y grid lines."),
    "x_grid_interval": (pychart_util.IntervalType, 1, None,
                        "The horizontal grid-line interval."),
    "y_grid_interval": (pychart_util.IntervalType, 1, None,
                        "The vertical grid-line interval."),
    "x_grid_over_plot": (types.IntType, 0, 0,
                      "If true, grid lines are drawn over plots. Otherwise, plots are drawn over grid lines."),
    "y_grid_over_plot": (types.IntType, 0, 0, "See x_grid_over_plot."),
    "_plots": (types.ListType, 1, pychart_util.new_list, "Used only intervally by pychart"),
    "legend": (legend.T, 1, _dummy_legend, "The legend of the chart."),
    }

def _log_tics(min, max, interval):
   "Generate the list of places for drawing tick marks."
   v = []
   x = min
   while x <= max:
      v.append(x)
      x = x * interval
   return v

def _linear_tics(min, max, interval):
   "Generate the list of places for drawing tick marks."   
   v = []
   x = min
   while x <= max:
      v.append(x)
      x = x + interval
   return v

class T(chart_object.T):
    keys = _keys
    __doc__ = area_doc.doc
    
    def x_pos(self, xval):
        "Return the x position on the canvas corresponding to the value XVAL."
        off = 0
        if self.x_coord_system == 'linear':
            off = linear_coord_pos(self.size[0], xval, self.x_range, None)
        elif self.x_coord_system == 'log':
            off = log_coord_pos(self.size[0], xval, self.x_range)
        else:
            off = category_coord_pos(self.size[0], xval,
                                     self.x_category_data, self.x_category_col)
        return self.loc[0] + off
    
    def y_pos(self, yval):
        "Return the y position on the canvas corresponding to the value YVAL."
        off = 0
        if self.y_coord_system == 'linear':
            off = linear_coord_pos(self.size[1], yval, self.y_range, None)
        elif self.y_coord_system == 'log':
            off = log_coord_pos(self.size[1], yval, self.y_range)
        else:
            off = category_coord_pos(self.size[1], yval,
                                     self.y_category_data, self.y_category_col)
        return self.loc[1] + off

    def __tic_points(self, coord, range, interval, data, d_col):
        if type(interval) == types.FunctionType:
            return apply(interval, range)

        min_x, max_x = range
      
        if coord == 'linear':
            return _linear_tics(min_x, max_x, interval)
        elif coord == 'log':
            return _log_tics(min_x, max_x, interval)
        else:
            return map(lambda pair, col=d_col: pair[col], data)
       
    def x_tic_points(self, interval):
        "Return the list of X values for which tick marks and grid lines are drawn."
        return self.__tic_points(self.x_coord_system, self.x_range, interval,
                                 self.x_category_data, self.x_category_col)
    def y_tic_points(self, interval):
        "Return the list of Y values for which tick marks and grid lines are drawn."
        return self.__tic_points(self.y_coord_system, self.y_range, interval,
                                 self.y_category_data, self.y_category_col)

    def __draw_x_grid_and_axis(self):
        if self.x_grid_style:
            for i in self.x_tic_points(self.x_grid_interval):
                x = self.x_pos(i)
                if x > self.loc[0]:
                    canvas.line(self.x_grid_style,
                                x, self.loc[1], x, self.loc[1]+self.size[1])
        if self.x_axis:
            self.x_axis.draw(self)
    def __draw_y_grid_and_axis(self):
        if self.y_grid_style:
            for i in self.y_tic_points(self.y_grid_interval):
                y = self.y_pos(i)
                if y > self.loc[1]:
                    canvas.line(self.y_grid_style,
                                self.loc[0], y,
                                self.loc[0]+self.size[0], y)
        if self.y_axis:
            self.y_axis.draw(self)

    def __get_data_range(self, r, which, coord_system, interval):
        if coord_system == 'category':
            # This info is unused for the category coord type.
            # So I just return a random value.
            return ((0,0), 1)

        r = r or (None, None)
        
        if len(self._plots) == 0:
            raise ValueError, "No chart to draw, and no data range specified.\n";
        dmin, dmax = 99999999, -99999999
 
        for plot in self._plots:
            this_min, this_max = plot.get_data_range(which)
            dmin = min(this_min, dmin)
            dmax = max(this_max, dmax)

        if interval and type(interval) == types.FunctionType:
            tics = apply(interval, dmin, dmax)
            dmin = tics[0]
            dmax = tics[len(tics)-1]
        elif coord_system == 'linear':
            if not interval:
                if dmax == dmin:
                    interval = 10
                else:
                    interval = 10 ** (float(int(math.log(dmax-dmin)/math.log(10))))
            dmin = min(dmin, pychart_util.round_down(dmin, interval))
            dmax = max(dmax, pychart_util.round_up(dmax, interval) + interval/2.0)
        elif coord_system == 'log':
            if not interval:
                interval = 10
            if dmin <= 0:
                # we can't have a negative value with a log scale.
                dmin = 1
            v = 1.0
            while v > dmin:
                v = v / interval
            dmin = v
            v = 1.0
            while v < dmax:
                v = v * interval
            dmax = v
        else:
            raise ValueError, "Unknown coord system: " + coord_system
        
        return ((r[0] or dmin, r[1] or dmax), interval)
    
    def draw(self):
        "Draw the charts."
        self.type_check()

        self.x_range, self.x_grid_interval = \
                      self.__get_data_range(self.x_range, 'X',
                                            self.x_coord_system,
                                            self.x_grid_interval)
            
        self.y_range, self.y_grid_interval = \
                      self.__get_data_range(self.y_range, 'Y',
                                            self.y_coord_system,
                                            self.y_grid_interval)
        
        canvas.rectangle(self.border_line_style, self.bg_style,
                         self.loc[0], self.loc[1],
                         self.loc[0] + self.size[0], self.loc[1] + self.size[1])

        if not self.x_grid_over_plot:
            self.__draw_x_grid_and_axis()

        if not self.y_grid_over_plot:
            self.__draw_y_grid_and_axis()
            
        canvas.clip(self.loc[0] - 10, self.loc[1] - 10,
                    self.loc[0] + self.size[0] + 10,
                    self.loc[1] + self.size[1] + 10)
        for plot in self._plots:
            plot.draw(self)
            
        canvas.endclip()
            
        if self.x_grid_over_plot:
            self.__draw_x_grid_and_axis()
        if self.y_grid_over_plot:
            self.__draw_y_grid_and_axis()

        if self.legend == _dummy_legend:
            self.legend = legend.T()
            
        if self.legend:
            legends = []
            for plot in self._plots:
                entry = plot.get_legend_entry()
                if entry == None:
                    pass
                elif type(entry) != types.ListType:
                    legends.append(entry)
                else:
                    for e in entry:
                        legends.append(e)
            self.legend.draw(self, legends)

    def add_plot(self, *plots):
        "Add PLOTS... to the area."
        for plot in plots:
            self._plots.append(plot)
