/* libspectrum.h: the library for dealing with ZX Spectrum emulator files
   Copyright (c) 2001-2002 Philip Kendall, Darren Salt

   $Id: libspectrum.h.in,v 1.58 2002/11/22 14:23:06 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: pak21-fuse@srcf.ucam.org
   Postal address: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

*/

/* NB: This file is autogenerated from libspectrum.h.in. Do not edit
   unless you know what you're doing */

#ifndef LIBSPECTRUM_LIBSPECTRUM_H
#define LIBSPECTRUM_LIBSPECTRUM_H

#ifdef __cplusplus
extern "C" {
#endif				/* #ifdef __cplusplus */

#include <stdarg.h>
#include <stdlib.h>

typedef unsigned  char libspectrum_byte;
typedef unsigned short libspectrum_word;
typedef unsigned   int libspectrum_dword;

#include <glib.h>

/*
 * General libspectrum routines
 */

/* Error handling */

/* The various errors which can occur */
typedef enum libspectrum_error {

  LIBSPECTRUM_ERROR_NONE = 0,

  LIBSPECTRUM_ERROR_WARNING,

  LIBSPECTRUM_ERROR_MEMORY,
  LIBSPECTRUM_ERROR_UNKNOWN,
  LIBSPECTRUM_ERROR_CORRUPT,
  LIBSPECTRUM_ERROR_SIGNATURE,
  LIBSPECTRUM_ERROR_SLT,	/* .slt data found at end of a .z80 file */

  LIBSPECTRUM_ERROR_LOGIC = -1,

} libspectrum_error;

typedef libspectrum_error
(*libspectrum_error_function_t)( libspectrum_error error,
				 const char *format, va_list ap );

extern libspectrum_error_function_t libspectrum_error_function;

libspectrum_error
libspectrum_default_error_function( libspectrum_error error,
				    const char *format, va_list ap );

/* Attempt to identify a given file */

/* Various types of file we might manage to identify */
typedef enum libspectrum_id_t {

  /* Unidentified file */
  LIBSPECTRUM_ID_UNKNOWN = 0,
  
  /* Input recording files */
  LIBSPECTRUM_ID_RECORDING_RZX,

  /* Snapshot files */
  LIBSPECTRUM_ID_SNAPSHOT_SNA,
  LIBSPECTRUM_ID_SNAPSHOT_Z80,

  /* Tape files */
  LIBSPECTRUM_ID_TAPE_TAP,
  LIBSPECTRUM_ID_TAPE_TZX,

} libspectrum_id_t;

libspectrum_error
libspectrum_identify_file( libspectrum_id_t *type, const char *filename,
                           const unsigned char *buffer, size_t length );

/* Different Spectrum variants and their capabilities */

/* The machine types we can handle */
typedef enum libspectrum_machine {

  LIBSPECTRUM_MACHINE_48,
  LIBSPECTRUM_MACHINE_TC2048,
  LIBSPECTRUM_MACHINE_128,
  LIBSPECTRUM_MACHINE_PLUS2,
  LIBSPECTRUM_MACHINE_PENT,
  LIBSPECTRUM_MACHINE_PLUS2A,
  LIBSPECTRUM_MACHINE_PLUS3,

  /* Used by libspectrum_tape_guess_hardware if we can't work out what
     hardware should be used */
  LIBSPECTRUM_MACHINE_UNKNOWN,

} libspectrum_machine;

const char* libspectrum_machine_name( libspectrum_machine type );

/* The various capabilities of the different machines */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_AY;		/* AY-3-8192 */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_128_MEMORY;
                                                  /* 128-style memory paging */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_MEMORY;
                                                   /* +3-style memory paging */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_DISK;
                                                      /* +3-style disk drive */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_MEMORY;
                                            /* TC20[46]8-style memory paging */
extern const int LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_VIDEO;
                                              /* TC20[46]8-style video modes */

/* Get the capabilities of a machine */
int libspectrum_machine_capabilities( libspectrum_machine type );

/*
 * Snap handling routines
 */

typedef enum libspectrum_slt_type {

  LIBSPECTRUM_SLT_TYPE_END = 0,
  LIBSPECTRUM_SLT_TYPE_LEVEL,
  LIBSPECTRUM_SLT_TYPE_INSTRUCTIONS,
  LIBSPECTRUM_SLT_TYPE_SCREEN,
  LIBSPECTRUM_SLT_TYPE_PICTURE,
  LIBSPECTRUM_SLT_TYPE_POKE,

} libspectrum_slt_type;

typedef struct libspectrum_snap {

  /* Which machine are we using here? */

  libspectrum_machine machine;

  /* Registers and the like */

  libspectrum_byte a , f ; libspectrum_word bc , de , hl ;
  libspectrum_byte a_, f_; libspectrum_word bc_, de_, hl_;

  libspectrum_word ix, iy; libspectrum_byte i, r;
  libspectrum_word sp, pc;

  libspectrum_byte iff1, iff2, im;

  /* RAM */

  libspectrum_byte *pages[8];

  /* Data from .slt files */

  libspectrum_byte *slt[256];	/* Level data */
  size_t slt_length[256];	/* Length of each level */

  libspectrum_byte *slt_screen;	/* Loading screen */
  int slt_screen_level;		/* The id of the loading screen. Not used
				   for anything AFAIK, but I'll copy it
				   around just in case */

  /* Peripheral status */

  libspectrum_byte out_ula; libspectrum_dword tstates;

  libspectrum_byte out_128_memoryport;

  libspectrum_byte out_ay_registerport, ay_registers[16];

  libspectrum_byte out_plus3_memoryport;

  /* Timex-specific bits */
  libspectrum_byte out_scld_hsr, out_scld_dec;

  /* Internal use only */

  int version;
  int compressed;

} libspectrum_snap;

libspectrum_error libspectrum_snap_alloc( libspectrum_snap **snap );
libspectrum_error libspectrum_snap_free( libspectrum_snap *snap );

/* Read in a snapshot, optionally guessing what type it is */
libspectrum_error
libspectrum_snap_read( libspectrum_snap *snap, const libspectrum_byte *buffer,
		       size_t length, libspectrum_id_t type,
		       const char *filename );

/* .sna specific routines */

libspectrum_error
libspectrum_sna_read( libspectrum_snap *snap,
	              const libspectrum_byte *buffer, size_t buffer_length );

/* .z80 specific routines */

libspectrum_error
libspectrum_z80_read( libspectrum_snap *snap,
	              const libspectrum_byte *buffer, size_t buffer_length );

libspectrum_error
libspectrum_z80_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_snap *snap );

/*
 * Tape handling routines
 */

/* The various types of block available
   The values here must be the same as used in the .tzx format */
typedef enum libspectrum_tape_type {
  LIBSPECTRUM_TAPE_BLOCK_ROM = 0x10,
  LIBSPECTRUM_TAPE_BLOCK_TURBO,
  LIBSPECTRUM_TAPE_BLOCK_PURE_TONE,
  LIBSPECTRUM_TAPE_BLOCK_PULSES,
  LIBSPECTRUM_TAPE_BLOCK_PURE_DATA,
  LIBSPECTRUM_TAPE_BLOCK_RAW_DATA,

  LIBSPECTRUM_TAPE_BLOCK_PAUSE = 0x20,
  LIBSPECTRUM_TAPE_BLOCK_GROUP_START,
  LIBSPECTRUM_TAPE_BLOCK_GROUP_END,
  LIBSPECTRUM_TAPE_BLOCK_JUMP,
  LIBSPECTRUM_TAPE_BLOCK_LOOP_START,
  LIBSPECTRUM_TAPE_BLOCK_LOOP_END,

  LIBSPECTRUM_TAPE_BLOCK_SELECT = 0x28,

  LIBSPECTRUM_TAPE_BLOCK_STOP48 = 0x2a,

  LIBSPECTRUM_TAPE_BLOCK_COMMENT = 0x30,
  LIBSPECTRUM_TAPE_BLOCK_MESSAGE,
  LIBSPECTRUM_TAPE_BLOCK_ARCHIVE_INFO,
  LIBSPECTRUM_TAPE_BLOCK_HARDWARE,

  LIBSPECTRUM_TAPE_BLOCK_CUSTOM = 0x35,

  LIBSPECTRUM_TAPE_BLOCK_CONCAT = 0x5a,
} libspectrum_tape_type;

/* Some flags */
extern const int LIBSPECTRUM_TAPE_FLAGS_BLOCK;	/* End of block */
extern const int LIBSPECTRUM_TAPE_FLAGS_STOP;	/* Stop tape */
extern const int LIBSPECTRUM_TAPE_FLAGS_STOP48;	/* Stop tape if in 48K mode */

/* The states which a block can be in */
typedef enum libspectrum_tape_state_type {

  LIBSPECTRUM_TAPE_STATE_PILOT,	/* Pilot pulses */
  LIBSPECTRUM_TAPE_STATE_SYNC1,	/* First sync pulse */
  LIBSPECTRUM_TAPE_STATE_SYNC2,	/* Second sync pulse */
  LIBSPECTRUM_TAPE_STATE_DATA1,	/* First edge of a data bit */
  LIBSPECTRUM_TAPE_STATE_DATA2,	/* Second edge of a data bit */
  LIBSPECTRUM_TAPE_STATE_PAUSE,	/* The pause at the end of a block */

} libspectrum_tape_state_type;

/* A standard ROM loading block */
typedef struct libspectrum_tape_rom_block {
  
  size_t length;		/* How long is this block */
  libspectrum_byte *data;	/* The actual data */
  libspectrum_dword pause;	/* Pause after block (milliseconds) */

  /* Private data */

  libspectrum_tape_state_type state;

  size_t edge_count;		/* Number of pilot pulses to go */

  size_t bytes_through_block;
  size_t bits_through_byte;	/* How far through the data are we? */

  libspectrum_byte current_byte; /* The current data byte; gets shifted out
				    as we read bits from it */
  libspectrum_dword bit_tstates; /* How long is an edge for the current bit */

} libspectrum_tape_rom_block;

/* A turbo loading block */
typedef struct libspectrum_tape_turbo_block {

  size_t length;		/* Length of data */
  size_t bits_in_last_byte;	/* How many bits are in the last byte? */
  libspectrum_byte *data;	/* The actual data */
  libspectrum_dword pause;	/* Pause after data (in ms) */

  libspectrum_dword pilot_length; /* Length of pilot pulse (in tstates) */
  size_t pilot_pulses;		/* Number of pilot pulses */

  libspectrum_dword sync1_length, sync2_length; /* Length of the sync pulses */
  libspectrum_dword bit0_length, bit1_length; /* Length of (re)set bits */

  /* Private data */

  libspectrum_tape_state_type state;

  size_t edge_count;		/* Number of pilot pulses to go */

  size_t bytes_through_block;
  size_t bits_through_byte;	/* How far through the data are we? */

  libspectrum_byte current_byte; /* The current data byte; gets shifted out
				    as we read bits from it */
  libspectrum_dword bit_tstates; /* How long is an edge for the current bit */

} libspectrum_tape_turbo_block;

/* A pure tone block */
typedef struct libspectrum_tape_pure_tone_block {

  libspectrum_dword length;
  size_t pulses;

  /* Private data */

  size_t edge_count;

} libspectrum_tape_pure_tone_block;

/* A list of pulses of different lengths */
typedef struct libspectrum_tape_pulses_block {

  size_t count;
  libspectrum_dword *lengths;

  /* Private data */

  size_t edge_count;

} libspectrum_tape_pulses_block;

/* A block of just of data */
typedef struct libspectrum_tape_pure_data_block {

  size_t length;		/* Length of data */
  size_t bits_in_last_byte;	/* How many bits are in the last byte? */
  libspectrum_byte *data;	/* The actual data */
  libspectrum_dword pause;	/* Pause after data (in ms) */

  libspectrum_dword bit0_length, bit1_length; /* Length of (re)set bits */

  /* Private data */

  libspectrum_tape_state_type state;

  size_t bytes_through_block;
  size_t bits_through_byte;	/* How far through the data are we? */

  libspectrum_byte current_byte; /* The current data byte; gets shifted out
				    as we read bits from it */
  libspectrum_dword bit_tstates; /* How long is an edge for the current bit */

} libspectrum_tape_pure_data_block;

/* A raw data block */
typedef struct libspectrum_tape_raw_data_block {

  size_t length;		/* Length of data */
  size_t bits_in_last_byte;	/* How many bits are in the last byte? */
  libspectrum_byte *data;	/* The actual data */
  libspectrum_dword pause;	/* Pause after data (in ms) */

  libspectrum_dword bit_length; /* Bit length. *Not* pulse length! */

  /* Private data */

  libspectrum_tape_state_type state;

  size_t bytes_through_block;
  size_t bits_through_byte;	/* How far through the data are we? */

  libspectrum_byte last_bit;	/* The last bit which was read */
  libspectrum_dword bit_tstates; /* How long is an edge for the current bit */

} libspectrum_tape_raw_data_block;

/* A pause block */
typedef struct libspectrum_tape_pause_block {

  libspectrum_dword length;

} libspectrum_tape_pause_block;

/* A group start block */
typedef struct libspectrum_tape_group_start_block {

  libspectrum_byte *name;

} libspectrum_tape_group_start_block;

/* No group end block needed as it contains no data */

/* A jump block */
typedef struct libspectrum_tape_jump_block {

  int offset;

} libspectrum_tape_jump_block;

/* A loop start block */
typedef struct libspectrum_tape_loop_start_block {

  int count;

} libspectrum_tape_loop_start_block;

/* No loop end block needed as it contains no data */

/* A select block */
typedef struct libspectrum_tape_select_block {

  /* Number of selections */
  size_t count;

  /* Offset of each selection, and a description of each */
  int *offsets;
  libspectrum_byte **descriptions;

} libspectrum_tape_select_block;

/* No `stop tape if in 48K mode' block as it contains no data */

/* A comment block */
typedef struct libspectrum_tape_comment_block {

  libspectrum_byte *text;

} libspectrum_tape_comment_block;

/* A message block */
typedef struct libspectrum_tape_message_block {

  int time;
  libspectrum_byte *text;

} libspectrum_tape_message_block;

/* An archive info block */
typedef struct libspectrum_tape_archive_info_block {

  /* Number of strings */
  size_t count;

  /* ID for each string */
  int *ids;

  /* Text of each string */
  libspectrum_byte **strings;

} libspectrum_tape_archive_info_block;

/* A hardware info block */
typedef struct libspectrum_tape_hardware_block {

  /* Number of entries */
  size_t count;

  /* For each entry, a type, an ID and a value */
  int *types, *ids, *values;

} libspectrum_tape_hardware_block;

/* A custom block */
typedef struct libspectrum_tape_custom_block {

  /* Description of this block */
  char description[17];

  /* And the data for it; currently, no attempt is made to interpret
     this data */
  size_t length; libspectrum_byte *data;

} libspectrum_tape_custom_block;

/* No block needed for concatenation block, as it isn't stored */

/* A generic tape block */
typedef struct libspectrum_tape_block {

  libspectrum_tape_type type;

  union {
    libspectrum_tape_rom_block rom;
    libspectrum_tape_turbo_block turbo;
    libspectrum_tape_pure_tone_block pure_tone;
    libspectrum_tape_pulses_block pulses;
    libspectrum_tape_pure_data_block pure_data;
    libspectrum_tape_raw_data_block raw_data;

    libspectrum_tape_pause_block pause;
    libspectrum_tape_group_start_block group_start;
    /* No group end block needed as it contains no data */
    libspectrum_tape_jump_block jump;
    libspectrum_tape_loop_start_block loop_start;
    /* No loop end block needed as it contains no data */

    libspectrum_tape_select_block select;

    /* No `stop tape if in 48K mode' block as it contains no data */

    libspectrum_tape_comment_block comment;
    libspectrum_tape_message_block message;
    libspectrum_tape_archive_info_block archive_info;
    libspectrum_tape_hardware_block hardware;

    libspectrum_tape_custom_block custom;
  } types;

} libspectrum_tape_block;

/* A linked list of tape blocks */
typedef struct libspectrum_tape {

  /* All the blocks */
  GSList* blocks;

  /* The current block */
  GSList* current_block;

  /* Private data */

  /* Where to return to after a loop, and how many iterations of the loop
     to do */
  GSList* loop_block;
  size_t loop_count;

} libspectrum_tape;

/* Routines to manipulate tape blocks */

libspectrum_error libspectrum_tape_alloc( libspectrum_tape **tape );
libspectrum_error libspectrum_tape_clear( libspectrum_tape *tape );
libspectrum_error libspectrum_tape_free( libspectrum_tape *tape );

/* Read in a tape file, optionally guessing what sort of file it is */
libspectrum_error
libspectrum_tape_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		       size_t length, libspectrum_id_t type,
		       const char *filename );

/* Does this tape structure actually contain a tape? */
int libspectrum_tape_present( const libspectrum_tape *tape );

libspectrum_error
libspectrum_tape_init_block( GSList *current_block );

libspectrum_error
libspectrum_tape_get_next_edge( libspectrum_dword *tstates, int *flags,
	                        libspectrum_tape *tape );

/* Get the position on the tape of the current block */
libspectrum_error libspectrum_tape_position( int *n, libspectrum_tape *tape );

/* Select the nth block on the tape */
libspectrum_error
libspectrum_tape_nth_block( libspectrum_tape *tape, int n );

libspectrum_error
libspectrum_tape_block_description( char *buffer, size_t length,
	                            libspectrum_tape_block *block );

/* Append a block to the current tape */
libspectrum_error
libspectrum_tape_append_block( libspectrum_tape *tape,
			       libspectrum_tape_block *block );

/*** Routines for .tap format files ***/

libspectrum_error
libspectrum_tap_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		      const size_t length );
libspectrum_error
libspectrum_tap_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_tape *tape );

/*** Routines for .tzx format files ***/

libspectrum_error
libspectrum_tzx_read( libspectrum_tape *tape, const libspectrum_byte *buffer,
		      const size_t length );

libspectrum_error
libspectrum_tzx_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_tape *tape );

typedef struct libspectrum_rzx libspectrum_rzx;

libspectrum_error libspectrum_rzx_alloc( libspectrum_rzx **rzx );
libspectrum_error libspectrum_rzx_free( libspectrum_rzx *rzx );

libspectrum_error libspectrum_rzx_store_frame( libspectrum_rzx *rzx,
					       size_t instructions, 
					       size_t count,
					       libspectrum_byte *in_bytes );

libspectrum_error libspectrum_rzx_start_playback( libspectrum_rzx *rzx );
libspectrum_error libspectrum_rzx_playback_frame( libspectrum_rzx *rzx,
						  int *finished );
libspectrum_error libspectrum_rzx_playback( libspectrum_rzx *rzx,
					    libspectrum_byte *byte );

/* Get and set the tstate counter */
size_t libspectrum_rzx_tstates( libspectrum_rzx *rzx );
size_t libspectrum_rzx_set_tstates( libspectrum_rzx *rzx, size_t tstates );

/* Get the current frame's instruction count */
size_t libspectrum_rzx_instructions( libspectrum_rzx *rzx );

libspectrum_error
libspectrum_rzx_read( libspectrum_rzx *rzx, libspectrum_snap **snap,
	              const libspectrum_byte *buffer, const size_t length );

libspectrum_error
libspectrum_rzx_write( libspectrum_byte **buffer, size_t *length,
		       libspectrum_rzx *rzx, libspectrum_snap *snap,
		       const char *program, libspectrum_word major,
		       libspectrum_word minor, int compress );

#ifdef __cplusplus
};
#endif				/* #ifdef __cplusplus */

#endif				/* #ifndef LIBSPECTRUM_LIBSPECTRUM_H */
