<?php
// $Horde: kronolith/lib/Driver/sql.php,v 1.31.2.4 2002/05/17 22:58:16 jan Exp $

/**
 * The Kronolith_Driver_sql:: class implements the Kronolith_Driver
 * API for a SQL backend.
 *
 * @author  Luc Saillard <luc.saillard@fr.alcove.com>
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.31.2.4 $
 * @since   Kronolith 0.3
 * @package kronolith
 */
class Kronolith_Driver_sql {

    /** Hash containing connection parameters. */
    var $params = array();

    /** Handle for the current database connection. */
    var $db;

    /** Boolean indicating whether or not we're connected to the SQL server. */
    var $connected = false;

    /** The calendar to open. */
    var $calendar;

    function Kronolith_Driver_sql($params)
    {
        $this->params = $params;
    }

    /**
     * Attempts to open a persistent connection to the SQL server.
     *
     * @return boolean True.
     */
    function _connect()
    {
        if (!$this->connected) {
            include_once 'DB.php';

            if (!is_array($this->params)) {
                Horde::fatal(new PEAR_Error(_("No configuration information specified for SQL Calendar.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['phptype'])) {
                Horde::fatal(new PEAR_Error(_("Required 'phptype' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['hostspec'])) {
                Horde::fatal(new PEAR_Error(_("Required 'hostspec' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['username'])) {
                Horde::fatal(new PEAR_Error(_("Required 'username' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['password'])) {
                Horde::fatal(new PEAR_Error(_("Required 'password' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['database'])) {
                Horde::fatal(new PEAR_Error(_("Required 'database' not specified in calendar configuration.")), __FILE__, __LINE__);
            }

            /* Connect to the SQL server using the supplied parameters. */
            $this->db = &DB::connect($this->params, true);
            if (DB::isError($this->db) || DB::isWarning($this->db)) {
                Horde::fatal($this->db, __FILE__, __LINE__);
            }

            /* Enable the "portability" option. */
            $this->db->setOption('optimize', 'portability');

            $this->connected = true;
        }

        return true;
    }

    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean true on success, false on failure.
     */
    function _disconnect()
    {
        if ($this->connected) {
            $this->connected = false;
            return $this->db->disconnect();
        }

        return true;
    }

    function open($calendar)
    {
        $this->calendar = $calendar;
        $this->_connect();
    }

    function listAlarms($date)
    {
        // TODO:
        $events = array();
        return is_array($events) ? $events : array();
    }

    function listEvents($startDate = null, $endDate = null)
    {
        $events = array();

        if (!isset($endDate)) {
            $endDate = Kronolith::dateObject(array('mday' => 31, 'month' => 12, 'year' => 9999));
        } else {
            list($endDate->mday, $endDate->month, $endDate->year) = explode('/', Date_Calc::nextDay($endDate->mday, $endDate->month, $endDate->year, '%d/%m/%Y'));
        }

        $etime = sprintf("'%04d-%02d-%02d 00:00:00.000'", $endDate->year, $endDate->month, $endDate->mday);
        if (isset($startDate)) {
            $stime = sprintf("'%04d-%02d-%02d 00:00:00.000'", $startDate->year, $startDate->month, $startDate->mday);
        }

        $q = 'SELECT DISTINCT e.event_id, e.event_recurtype FROM ' . $this->params['table'] . ' e' .
             ' WHERE e.calendar_id = ' . $this->db->quote($this->calendar) . ' AND ((';

        if (isset($stime)) {
            $q .= 'e.event_end > ' . $stime . ' AND ';
        }
        $q .= 'e.event_start < ' . $etime . ') OR (';
        if (isset($stime)) {
            $q .= 'e.event_recurenddate >= ' . $stime . ' AND ';
        }
        $q .= 'e.event_start <= ' . $etime .
              ' AND e.event_recurtype != ' . KRONOLITH_RECUR_NONE . '))';

        $qr = $this->db->query($q);

        if (!DB::isError($qr)) {
            $row = $qr->fetchRow(DB_FETCHMODE_ASSOC);
            while ($row && !DB::isError($row)) {
                if ($row['event_recurtype'] == KRONOLITH_RECUR_NONE) {
                    $events[] = $row['event_id'];
                } else {
                    $next = $this->nextRecurrence($row['event_id'], $startDate);
                    if ($next && Kronolith::compareDates($next, $endDate) < 0) {
                        $events[] = $row['event_id'];
                    }
                }

                $row = $qr->fetchRow(DB_FETCHMODE_ASSOC);
            }
        }

        return $events;
    }

    function getEvent($eventID)
    {
        $eventID = (int)$eventID;
        $row = &$this->db->getRow('SELECT event_id, event_description, event_location,' .
                                  ' event_keywords, event_title, event_category,' .
                                  ' event_recurtype, event_recurenddate, event_recurinterval,' .
                                  ' event_recurdays, event_start, event_end, event_alarm,' .
                                  ' event_modified, event_exceptions' .
                                  ' FROM ' . $this->params['table'] .
                                  ' WHERE event_id = ' . $eventID, DB_FETCHMODE_ASSOC);

        if (!DB::isError($row)) {
            return $row;
        }

        return false;
    }

    function getEventObject($eventID = null)
    {
        if (isset($eventID)) {
            return new Kronolith_Event_sql($this, $this->getEvent($eventID, true));
        } else {
            return new Kronolith_Event_sql($this);
        }
    }

    function saveEvent($event)
    {
        if (!is_null($event->getID())) {
            $query = 'UPDATE ' . $this->params['table'] . ' SET ';

            foreach ($event->properties as $key => $val) {
                $query .= " $key = " . $this->db->quote($val) . ',';
            }
            $query = substr($query, 0, -1);
            $query .= ' WHERE event_id = ' . (int)$event->getID();

            if (DB::isError($res = $this->db->query($query))) {
                return false;
            }

            return true;
        } else {
            $id = $this->db->nextId($this->params['table']);
            if (DB::isError($id)) {
                return $id;
            }

            $query = 'INSERT INTO '.$this->params['table'].' ';
            $cols_name = '(event_id,';
            $cols_values = 'values (' . $id . ',';

            foreach ($event->properties as $key => $val) {
                $cols_name .= " $key,";
                $cols_values .= $this->db->quote($val) . ',';
            }

            $cols_name .= ' calendar_id)';
            $cols_values .= $this->db->quote($this->calendar) . ')';

            if (DB::isError($res = $this->db->query($query . $cols_name . $cols_values))) {
                return false;
            }

            return true;
        }
    }

    function nextRecurrence($eventID, $afterDate, $weekstart = KRONOLITH_SUNDAY)
    {
        $event = $this->getEventObject($eventID);
        $afterDate = Kronolith::dateObject($afterDate);

        if (Kronolith::compareDates($event->start, $afterDate) > 0) {
            return $event->start;
        }

        $event->recurEnd->hour = 23;
        $event->recurEnd->min  = 59;
        $event->recurEnd->sec  = 59;

        switch ($event->recurType) {

        case KRONOLITH_RECUR_DAILY:
            $diff = Kronolith::dateDiff($event->start, $afterDate);
            $recur = (ceil($diff->mday / $event->recurInterval)) * $event->recurInterval;
            $next = $event->start;
            list($next->mday, $next->month, $next->year) = explode('/', Date_Calc::daysToDate(Date_Calc::dateToDays($next->mday, $next->month, $next->year) + $recur, '%e/%m/%Y'));
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0 &&
                Kronolith::compareDates($next, $afterDate) >= 0) {
                return $next;
            }
            break;

        case KRONOLITH_RECUR_WEEKLY:
            list($start_week->mday, $start_week->month, $start_week->year) = explode('/', Date_Calc::beginOfWeek($event->start->mday, $event->start->month, $event->start->year, '%e/%m/%Y'));
            $start_week->hour = $event->start->hour;
            $start_week->min = $event->start->min;
            $start_week->sec = $event->start->sec;
            list($after_week->mday, $after_week->month, $after_week->year) = explode('/', Date_Calc::beginOfWeek($afterDate->mday, $afterDate->month, $afterDate->year, '%e/%m/%Y'));
            $after_week_end = $after_week;
            $after_week_end->mday += 7;
            $after_week_end = Kronolith::correctDate($after_week_end);
            $diff = Kronolith::dateDiff($start_week, $after_week);
            $recur = $diff->mday + $diff->mday % ($event->recurInterval * 7);
            $next = $start_week;
            list($next->mday, $next->month, $next->year) = explode('/', Date_Calc::daysToDate(Date_Calc::dateToDays($next->mday, $next->month, $next->year) + $recur, '%e/%m/%Y'));
            while (Kronolith::compareDates($next, $afterDate) < 0 && Kronolith::compareDates($next, $after_week_end) < 0) {
                $next->mday++;
                $next = Kronolith::correctDate($next);
            }
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0) {
                if (Kronolith::compareDates($next, $after_week_end) >= 0) {
                    return $GLOBALS['calendar']->nextRecurrence($eventID, $after_week_end);
                }
                while (!$event->recurOnDay((int)pow(2, (int)Date_Calc::dayOfWeek($next->mday, $next->month, $next->year))) && Kronolith::compareDates($next, $after_week_end) < 0) {
                    $next->mday++;
                    $next = Kronolith::correctDate($next);
                }
                if (Kronolith::compareDates($next, $event->recurEnd) <= 0) {
                    if (Kronolith::compareDates($next, $after_week_end) >= 0) {
                        return $GLOBALS['calendar']->nextRecurrence($eventID, $after_week_end);
                    } else {
                        return Kronolith::dateObject($next);
                    }
                }
            }
            break;

        case KRONOLITH_RECUR_DAY_OF_MONTH:
            $diff = Kronolith::dateDiff($event->start, $afterDate);
            $recur = $diff->month + $diff->month % $event->recurInterval;
            $next = $event->start;
            $next->month += $recur;
            $next = Kronolith::correctDate($next);
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0 &&
                Kronolith::compareDates($next, $afterDate) >= 0) {
                return $next;
            }
            break;

        case KRONOLITH_RECUR_WEEK_OF_MONTH:
            break;

        case KRONOLITH_RECUR_YEARLY:
            $diff = Kronolith::dateDiff($event->start, $afterDate);
            $recur = $diff->year + $diff->year % $event->recurInterval;
            $next = $event->start;
            $next->year += $recur;
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0 &&
                Kronolith::compareDates($next, $afterDate) >= 0) {
                return $next;
            }
            break;

        }

        return false;
    }

    function deleteEvent($eventID)
    {
        $eventID = (int)$eventID;
        $query = 'DELETE FROM ' . $this->params['table'] . " WHERE event_id = $eventID";
        if (DB::isError($res = $this->db->query($query))) {
            return false;
        }

        return true;
    }

}

class Kronolith_Event_sql extends Kronolith_Event {

    var $properties = array();

    function fromDriver($SQLEvent)
    {
        list($this->start->year, $this->start->month, $this->start->mday, $this->start->hour, $this->start->min, $this->start->sec) = sscanf($SQLEvent['event_start'], '%04d-%02d-%02d %02d:%02d:%02d');
        list($this->end->year, $this->end->month, $this->end->mday, $this->end->hour, $this->end->min, $this->end->sec) = sscanf($SQLEvent['event_end'], '%04d-%02d-%02d %02d:%02d:%02d');

        $this->startTimestamp = mktime($this->start->hour, $this->start->min, $this->start->sec, $this->start->month, $this->start->mday, $this->start->year);
        $this->endTimestamp = mktime($this->end->hour, $this->end->min, $this->end->sec, $this->end->month, $this->end->mday, $this->end->year);

        $this->durMin = ($this->endTimestamp - $this->startTimestamp) / 60;

        if (isset($SQLEvent['event_recurenddate'])) {
            list($this->recurEnd->year, $this->recurEnd->month, $this->recurEnd->mday, $this->recurEnd->hour, $this->recurEnd->min, $this->recurEnd->sec) = sscanf($SQLEvent['event_recurenddate'], '%04d-%02d-%02d %02d:%02d:%02d');
            $this->recurEndTimestamp = mktime($this->recurEnd->hour, $this->recurEnd->min, $this->recurEnd->sec, $this->recurEnd->month, $this->recurEnd->mday, $this->recurEnd->year);
        }

        $this->title = $SQLEvent['event_title'];
        $this->eventID = $SQLEvent['event_id'];
        $this->recurType = (int)$SQLEvent['event_recurtype'];
        $this->recurInterval = (int)$SQLEvent['event_recurinterval'];

        if (isset($SQLEvent['event_category'])) {
            $this->category = $SQLEvent['event_category'];
        }
        if (isset($SQLEvent['event_location'])) {
            $this->location = $SQLEvent['event_location'];
        }
        if (isset($SQLEvent['event_keywords'])) {
            $this->keywords = explode(',', $SQLEvent['event_keywords']);
        }
        if (isset($SQLEvent['event_exceptions'])) {
            $this->exceptions = explode(',', $SQLEvent['event_exceptions']);
        }
        if (isset($SQLEvent['event_description'])) {
            $this->description = $SQLEvent['event_description'];
        }
        if (isset($SQLEvent['event_alarm'])) {
            $this->alarm = (int)$SQLEvent['event_alarm'];
        }
        if (isset($SQLEvent['event_recurdays'])) {
            $this->recurData = (int)$SQLEvent['event_recurdays'];
        }

        $this->initialized = true;
    }

    function toDriver()
    {
        // basic fields
        $this->properties['event_title'] = $this->getTitle();
        $this->properties['event_description'] = $this->getDescription();
        $this->properties['event_category'] = $this->getCategory();
        $this->properties['event_location'] = $this->getLocation();
        $this->properties['event_keywords'] = implode(',', $this->getKeywords());
        $this->properties['event_exceptions'] = implode(',', $this->getExceptions());
        $this->properties['event_modified'] = time();

        // event start
        $this->properties['event_start'] = date('Y-m-d H:i:s', $this->getStartTimestamp());

        // event end
        $this->properties['event_end'] = date('Y-m-d H:i:s', $this->getEndTimestamp());

        // alarm
        $this->properties['event_alarm'] = $this->getAlarm();

        // recurrence
        $recur_end = explode(':', date('Y:n:j', $this->getRecurEndTimestamp()));
        if ($recur_end[0] == 1970) {
            $recur_end[0] = 9999;
            $recur_end[1] = 12;
            $recur_end[2] = 31;
        }

        $recur = $this->getRecurType();
        $this->properties['event_recurtype'] = $recur;
        if ($recur != KRONOLITH_RECUR_NONE) {
            $this->properties['event_recurinterval'] = $this->getRecurInterval();
            $this->properties['event_recurenddate'] = sprintf('%04d%02d%02d', $recur_end[0],
                                                              $recur_end[1], $recur_end[2]);

            switch ($recur) {
            case KRONOLITH_RECUR_WEEKLY:
                $this->properties['event_recurdays'] = $this->getRecurOnDays();
                break;
            }
        }
    }

}
?>
