<?php
// $Horde: kronolith/lib/Driver/mcal.php,v 1.24.2.1 2002/04/23 17:59:53 chuck Exp $

/**
 * The Kronolith_Driver_mcal:: class implements the Kronolith_Driver
 * API for an MCAL backend.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.24.2.1 $
 * @since   Kronolith 0.1
 * @package kronolith
 */
class Kronolith_Driver_mcal {

    var $mcalUsername;
    var $mcalPassword;
    var $stream;

    function Kronolith_Driver_mcal($params)
    {
        if (empty($params['username']) || empty($params['password'])) {
            return false;
        }

        $this->mcalUsername = $params['username'];
        $this->mcalPassword = $params['password'];
    }

    function open($calendar)
    {
        $this->stream = @mcal_popen("{/mstore}<$calendar>", $this->mcalUsername, $this->mcalPassword);
    }

    function listEvents($startDate = null, $endDate = null)
    {
        $events = @mcal_list_events($this->stream, $startDate->year, $startDate->month, $startDate->mday, $endDate->year, $endDate->month, $endDate->mday);
        return is_array($events) ? $events : array();
    }

    function listAlarms($date)
    {
        $events = @mcal_list_alarms($this->stream, $date->year, $date->month, $date->mday, $date->hour, $date->min, $date->sec);
        return is_array($events) ? $events : array();
    }

    function getEvent($eventID)
    {
        return mcal_fetch_event($this->stream, $eventID);
    }

    function getEventObject($eventID = null)
    {
        if (isset($eventID)) {
            return new Kronolith_Event_mcal($this, $this->getEvent($eventID));
        } else {
            return new Kronolith_Event_mcal($this);
        }
    }

    function saveEvent($event)
    {
        if (!is_null($event->getID())) {
            return mcal_store_event($this->stream);
        } else {
            return mcal_append_event($this->stream);
        }
    }

    function nextRecurrence($eventID, $afterDate, $weekstart = KRONOLITH_SUNDAY)
    {
        return Kronolith::dateObject(mcal_next_recurrence($this->stream, $weekstart, $afterDate));
    }

    function deleteEvent($eventID)
    {
        return mcal_delete_event($this->stream, $eventID);
    }

    function parseMCALDate($dateObject)
    {
        if (count($dateObject) === 0) {
            return 0;
        }

        $year = isset($dateObject->year) ? $dateObject->year : 0;
        $month = isset($dateObject->month) ? $dateObject->month : 0;
        $day = isset($dateObject->mday) ? $dateObject->mday : 0;

        // Check for events with no recur_enddate
        if ($year == 9999 && $month == 12 && $day == 31) {
            return 0;
        }

        $hour = isset($dateObject->hour) ? $dateObject->hour : 0;
        $minute = isset($dateObject->min) ? $dateObject->min : 0;
        $second = isset($dateObject->sec) ? $dateObject->sec : 0;

        return mktime($hour, $minute, $second, $month, $day, $year);
    }

}

class Kronolith_Event_mcal extends Kronolith_Event {

    function toDriver()
    {
        // basic fields
        mcal_event_set_title($this->driver->stream, $this->getTitle());
        mcal_event_set_description($this->driver->stream, $this->getDescription());
        mcal_event_set_category($this->driver->stream, $this->getCategory());
        mcal_event_add_attribute($this->driver->stream, 'location', $this->getLocation());
        mcal_event_add_attribute($this->driver->stream, 'keywords', implode(',', $this->getKeywords()));
        mcal_event_add_attribute($this->driver->stream, 'exceptions', implode(',', $this->getExceptions()));
        mcal_event_add_attribute($this->driver->stream, 'modified', time());

        // event start
        $start = explode(':', date('Y:n:j:G:i', $this->getStartTimestamp()));
        mcal_event_set_start($this->driver->stream, $start[0], $start[1], $start[2], $start[3], $start[4]);

        // event end
        $end = explode(':', date('Y:n:j:G:i', $this->getEndTimestamp()));
        mcal_event_set_end($this->driver->stream, $end[0], $end[1], $end[2], $end[3], $end[4]);

        // alarm
        mcal_event_set_alarm($this->driver->stream, $this->getAlarm());

        // recurrence
        $recur_end = explode(':', date('Y:n:j', $this->getRecurEndTimestamp()));
        if ($recur_end[0] == 1969) {
            $recur_end[0] = 9999;
            $recur_end[1] = 12;
            $recur_end[2] = 31;
        }

        switch ($this->getRecurType()) {
        case KRONOLITH_RECUR_NONE:
            mcal_event_set_recur_none($this->driver->stream);
            break;

        case KRONOLITH_RECUR_DAILY:
            mcal_event_set_recur_daily($this->driver->stream,
                                       $recur_end[0],
                                       $recur_end[1],
                                       $recur_end[2],
                                       $this->getRecurInterval());
            break;

        case KRONOLITH_RECUR_WEEKLY:
            mcal_event_set_recur_weekly($this->driver->stream,
                                        $recur_end[0],
                                        $recur_end[1],
                                        $recur_end[2],
                                        $this->getRecurInterval(),
                                        $this->getRecurOnDays());
            break;

        case KRONOLITH_RECUR_DAY_OF_MONTH:
            mcal_event_set_recur_monthly_mday($this->driver->stream,
                                              $recur_end[0],
                                              $recur_end[1],
                                              $recur_end[2],
                                              $this->getRecurInterval());
            break;

        case KRONOLITH_RECUR_WEEK_OF_MONTH:
            mcal_event_set_recur_monthly_wday($this->driver->stream,
                                              $recur_end[0],
                                              $recur_end[1],
                                              $recur_end[2],
                                              $this->getRecurInterval());
            break;

        case KRONOLITH_RECUR_YEARLY:
            mcal_event_set_recur_yearly($this->driver->stream,
                                        $recur_end[0],
                                        $recur_end[1],
                                        $recur_end[2],
                                        $this->getRecurInterval());
            break;
        }
    }

    function fromDriver($mcalEvent)
    {
        $this->title = $mcalEvent->title;
        if (isset($mcalEvent->category)) {
            $this->category = $mcalEvent->category;
        }
        if (isset($mcalEvent->description)) {
            $this->description = $mcalEvent->description;
        }
        if (isset($mcalEvent->attrlist['location'])) {
            $this->location = $mcalEvent->attrlist['location'];
        }
        if (isset($mcalEvent->attrlist['keywords'])) {
            $this->keywords = explode(',', $mcalEvent->attrlist['keywords']);
        }
        if (isset($mcalEvent->attrlist['exceptions'])) {
            $this->exceptions = explode(',', $mcalEvent->attrlist['exceptions']);
        }
        $this->eventID = $mcalEvent->id;

        $this->startTimestamp = Kronolith_Driver_mcal::parseMCALDate($mcalEvent->start);
        $this->start = Kronolith::timestampToObject($this->startTimestamp);

        $this->endTimestamp = Kronolith_Driver_mcal::parseMCALDate($mcalEvent->end);
        $this->end = Kronolith::timestampToObject($this->endTimestamp);

        $this->durMin = ($this->endTimestamp - $this->startTimestamp) / 60;

        if (isset($mcalEvent->recur_enddate)) {
            $this->recurEndTimestamp = Kronolith_Driver_mcal::parseMCALDate($mcalEvent->recur_enddate);
            $this->recurEnd = $mcalEvent->recur_enddate;
        }

        $this->alarm = $mcalEvent->alarm;

        $this->recurType = $mcalEvent->recur_type;
        $this->recurInterval = $mcalEvent->recur_interval;
        if (isset($mcalEvent->recur_data)) {
            $this->recurData = $mcalEvent->recur_data;
        }

        $this->initialized = true;
    }

}
?>
