%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%%%----------------------------------------------------------------------
%%% File    : application_starter.erl
%%% Author  : Esko Vierumaki <esko@gimli>
%%%           
%%% Purpose : Starts applications in the phases defined in the .app file's
%%%           start_phases key. If the application includes other applications 
%%%           these are also started according to their mod and 
%%%           start_phases-keys in their .app file.
%%%
%%% Created : 10 Mar 1998 by Esko Vierumaki <esko@gimli>
%%%----------------------------------------------------------------------

-module(application_starter).
-author('esko@gimli').

-export([start/3]).

%%%=============================================================================
%%%=============================================================================
%%%=============================================================================
%%% start(Phases, Type, Applications) -> ok | {error, ErrorMessage}
%%%
%%% The applications are started by calling Module:start_phase(Phase, Type, Args) 
%%% where Module and is defined in the mod-key, Phase and Args are defined in
%%% the start_phases-key.
%%%=============================================================================
%%%=============================================================================
%%%=============================================================================
start([], Type, Apps) ->
    ok;
start([{P,A}|Phases], Type, Apps) ->
    case start_apps(P, Type, Apps) of
	{error, Error} ->
	    {error, Error};
	_ ->
	    start(Phases, Type, Apps)
    end.


%%%=============================================================================
%%% Start each application in the phase P. 
%%%=============================================================================
start_apps(P, Type, []) ->
    ok;
start_apps(P, Type, [App | Apps]) ->
    case catch run_start_phase(P, Type, App) of
	{error, Error} ->
	    {error, Error};
	_ ->
	    start_apps(P, Type, Apps)
    end.


%%%=============================================================================
%%% If application_starter is used recursively, start also all the included
%%% applications in the phase P. 
%%%=============================================================================
run_start_phase(P, Type, App) ->
    {ok,{Mod,Arg}} = application:get_key(App, mod),
    case Mod of
	application_starter ->
	    run_the_phase(P, Type, App),
	    {ok, IncApps} = application:get_key(App, included_applications),
	    start_apps(P, Type, IncApps);
	_ ->
	    run_the_phase(P, Type, App)
    end.
    


%%%=============================================================================
%%% Start the application only if the start phase is defined in the 
%%% start_phases-key. 
%%%=============================================================================
run_the_phase(P, Type, M) ->
    Start_phases = case application_controller:get_key(M, start_phases) of
		       {ok, undefined} ->
			   throw({error, {start_phases_undefined, M}});
		       {ok, Sp} ->
			   Sp
		   end,
    case lists:keysearch(P, 1, Start_phases) of
	false ->
	    ok;
	{value, {P, A}} -> 
	    case catch M:start_phase(P, Type, A) of
		ok ->
		    ok;
		{error, Reason} ->
		    throw({error, {Reason, {M, start_phase, [P, Type, A]}}});
		Other ->
		    throw({error, {bad_return_value, 
				   {{M, start_phase, [P, Type, A]}, Other}}})
	    end
    end.



