/* $OpenBSD: pci_6600.c,v 1.5 2001/08/17 22:26:58 mickey Exp $ */
/* $NetBSD: pci_6600.c,v 1.5 2000/06/06 00:50:15 thorpej Exp $ */

/*-
 * Copyright (c) 1999 by Ross Harvey.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Ross Harvey.
 * 4. The name of Ross Harvey may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ROSS HARVEY ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURP0SE
 * ARE DISCLAIMED.  IN NO EVENT SHALL ROSS HARVEY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/malloc.h>

#include <vm/vm.h>
#include <uvm/uvm.h>

#include <machine/autoconf.h>
#define _ALPHA_BUS_DMA_PRIVATE
#include <machine/bus.h>
#include <machine/rpb.h>

#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>
#include <dev/pci/pciidereg.h>
#include <dev/pci/pciidevar.h>

#include <alpha/pci/tsreg.h>
#include <alpha/pci/tsvar.h>
#include <alpha/pci/pci_6600.h>

#define pci_6600() { Generate ctags(1) key. }

#include "sio.h"
#if NSIO
#include <alpha/pci/siovar.h>
#endif

#define	PCI_NIRQ		64
#define	PCI_STRAY_MAX		5

/*
 * Some Tsunami models have a PCI device (the USB controller) with interrupts
 * tied to ISA IRQ lines.  The IRQ is encoded as:
 *
 *	line = 0xe0 | isa_irq;
 */
#define	DEC_6600_LINE_IS_ISA(line)	((line) >= 0xe0 && (line) <= 0xef)
#define	DEC_6600_LINE_ISA_IRQ(line)	((line) & 0x0f)

static char *irqtype = "6600 irq";
static struct tsp_config *sioprimary;

void dec_6600_intr_disestablish __P((void *, void *));
void *dec_6600_intr_establish __P((
    void *, pci_intr_handle_t, int, int (*func)(void *), void *, char *));
const char *dec_6600_intr_string __P((void *, pci_intr_handle_t));
int dec_6600_intr_line __P((void *, pci_intr_handle_t));
const struct evcnt *dec_6600_intr_evcnt __P((void *, pci_intr_handle_t));
int dec_6600_intr_map __P((void *, pcitag_t, int, int, pci_intr_handle_t *));
void *dec_6600_pciide_compat_intr_establish __P((void *, struct device *,
    struct pci_attach_args *, int, int (*)(void *), void *));
void  dec_6600_pciide_compat_intr_disestablish __P((void *, void *));

struct alpha_shared_intr *dec_6600_pci_intr;

void dec_6600_iointr __P((void *framep, unsigned long vec));
extern void dec_6600_intr_enable __P((int irq));
extern void dec_6600_intr_disable __P((int irq));

void
pci_6600_pickintr(pcp)
	struct tsp_config *pcp;
{
	bus_space_tag_t iot = pcp->pc_iot;
	pci_chipset_tag_t pc = &pcp->pc_pc;
#if 0
	char *cp;
#endif
	int i;

        pc->pc_intr_v = pcp;
        pc->pc_intr_map = dec_6600_intr_map;
        pc->pc_intr_string = dec_6600_intr_string;
        pc->pc_intr_line = dec_6600_intr_line;
#if 0
	pc->pc_intr_evcnt = dec_6600_intr_evcnt;
#endif
        pc->pc_intr_establish = dec_6600_intr_establish;
        pc->pc_intr_disestablish = dec_6600_intr_disestablish;
	pc->pc_pciide_compat_intr_establish = NULL;

	/*
	 * System-wide and Pchip-0-only logic...
	 */
	if (dec_6600_pci_intr == NULL) {
		sioprimary = pcp;
		pc->pc_pciide_compat_intr_establish =
		    dec_6600_pciide_compat_intr_establish;
		dec_6600_pci_intr = alpha_shared_intr_alloc(PCI_NIRQ);
		for (i = 0; i < PCI_NIRQ; i++) {
			alpha_shared_intr_set_maxstrays(dec_6600_pci_intr, i,
			    PCI_STRAY_MAX);
			alpha_shared_intr_set_private(dec_6600_pci_intr, i,
			    sioprimary);

#if 0
			cp = alpha_shared_intr_string(dec_6600_pci_intr);
			sprintf(cp, "irq %d", i);
			evcnt_attach_dynamic(alpha_shared_intr_evcnt(
			    dec_6600_pci_intr, 1), EVCNT_TYPE_INTR, NULL,
			    "dec_6600", cp);
#endif
		}
#if NSIO
		sio_intr_setup(pc, iot);
		dec_6600_intr_enable(55);	/* irq line for sio */
#endif
		set_iointr(dec_6600_iointr);
	}
}

int     
dec_6600_intr_map(acv, bustag, buspin, line, ihp)
        void *acv;
        pcitag_t bustag; 
        int buspin, line;
        pci_intr_handle_t *ihp;
{
	struct tsp_config *pcp = acv;
	pci_chipset_tag_t pc = &pcp->pc_pc;
	int bus, device, function;

	if (buspin == 0) {
		/* No IRQ used. */
		return 1;
	}
	if (buspin > 4) {
		printf("intr_map: bad interrupt pin %d\n", buspin);
		return 1;
	}

	alpha_pci_decompose_tag(pc, bustag, &bus, &device, &function);

	/*
	 * The console places the interrupt mapping in the "line" value.
	 * A value of (char)-1 indicates there is no mapping.
	 */
	if (line == 0xff) {
		printf("dec_6600_intr_map: no mapping for %d/%d/%d\n",
		    bus, device, function);
		return (1);
	}

#if NSIO == 0
	if (DEC_6600_LINE_IS_ISA(line)) {
		printf("dec_6600_intr_map: ISA IRQ %d for %d/%d/%d\n",
		    DEC_6600_LINE_ISA_IRQ(line), bus, device, function);
		return (1);
	}
#endif

	if (DEC_6600_LINE_IS_ISA(line) == 0 && line >= PCI_NIRQ)
		panic("dec_6600_intr_map: dec 6600 irq too large (%d)\n",
		    line);

	*ihp = line;
	return (0);
}

const char *
dec_6600_intr_string(acv, ih)
	void *acv;
	pci_intr_handle_t ih;
{

	static const char irqfmt[] = "dec 6600 irq %ld";
	static char irqstr[sizeof irqfmt];

#if NSIO
	if (DEC_6600_LINE_IS_ISA(ih))
		return (sio_intr_string(NULL /*XXX*/,
		    DEC_6600_LINE_ISA_IRQ(ih)));
#endif

	snprintf(irqstr, sizeof irqstr, irqfmt, ih);
	return (irqstr);
}

int
dec_6600_intr_line(acv, ih)
	void *acv;
	pci_intr_handle_t ih;
{

#if NSIO
	if (DEC_6600_LINE_IS_ISA(ih))
		return (sio_intr_line(NULL /*XXX*/,
		    DEC_6600_LINE_ISA_IRQ(ih)));
#endif

	return (ih);
}

#if 0
const struct evcnt *
dec_6600_intr_evcnt(acv, ih)
	void *acv;
	pci_intr_handle_t ih;
{

#if NSIO
	if (DEC_6600_LINE_IS_ISA(ih))
		return (sio_intr_evcnt(NULL /*XXX*/,
		    DEC_6600_LINE_ISA_IRQ(ih)));
#endif

	return (alpha_shared_intr_evcnt(dec_6600_pci_intr, ih));
}
#endif

void *
dec_6600_intr_establish(acv, ih, level, func, arg, name)
        void *acv, *arg;
        pci_intr_handle_t ih;
        int level;
        int (*func) __P((void *));
	char *name;
{
	void *cookie;

#if NSIO
	if (DEC_6600_LINE_IS_ISA(ih))
		return (sio_intr_establish(NULL /*XXX*/,
		    DEC_6600_LINE_ISA_IRQ(ih), IST_LEVEL, level, func, arg,
	            name));
#endif

	if (ih >= PCI_NIRQ)
		panic("dec_6600_intr_establish: bogus dec 6600 IRQ 0x%lx\n",
		    ih);

	cookie = alpha_shared_intr_establish(dec_6600_pci_intr, ih, IST_LEVEL,
	    level, func, arg, name);

	if (cookie != NULL && alpha_shared_intr_isactive(dec_6600_pci_intr, ih))
		dec_6600_intr_enable(ih);
	return (cookie);
}

void
dec_6600_intr_disestablish(acv, cookie)
        void *acv, *cookie;
{
	struct alpha_shared_intrhand *ih = cookie;
	unsigned int irq = ih->ih_num;
	int s;

#if NSIO
	/*
	 * We have to determine if this is an ISA IRQ or not!  We do this
	 * by checking to see if the intrhand points back to an intrhead
	 * that points to the sioprimary TSP.  If not, it's an ISA IRQ.
	 * Pretty disgusting, eh?
	 */
	if (ih->ih_intrhead->intr_private != sioprimary) {
		sio_intr_disestablish(NULL /*XXX*/, cookie);
		return;
	}
#endif
 
	s = splhigh();

	alpha_shared_intr_disestablish(dec_6600_pci_intr, cookie, irqtype);
	if (alpha_shared_intr_isactive(dec_6600_pci_intr, irq) == 0) {
		dec_6600_intr_disable(irq);
		alpha_shared_intr_set_dfltsharetype(dec_6600_pci_intr, irq,
		    IST_NONE);
	}
 
	splx(s);
}

void
dec_6600_iointr(framep, vec)
	void *framep;
	unsigned long vec;
{
	int irq; 

	if (vec >= 0x900) {
		irq = (vec - 0x900) >> 4;

		if (irq >= PCI_NIRQ)
			panic("iointr: irq %d is too high", irq);

		if (!alpha_shared_intr_dispatch(dec_6600_pci_intr, irq)) {
			alpha_shared_intr_stray(dec_6600_pci_intr, irq,
			    irqtype);
			if (ALPHA_SHARED_INTR_DISABLE(dec_6600_pci_intr, irq))
				dec_6600_intr_disable(irq);
		}
		return;
	}
#if NSIO
	if (vec >= 0x800) {
		sio_iointr(framep, vec);
		return;
	}
#endif
	panic("iointr: weird vec 0x%lx\n", vec);
}

void
dec_6600_intr_enable(irq)
	int irq;
{
	alpha_mb();
	STQP(TS_C_DIM0) |= 1UL << irq;
	alpha_mb();
}

void
dec_6600_intr_disable(irq)
	int irq;
{
	alpha_mb();
	STQP(TS_C_DIM0) &= ~(1UL << irq);
	alpha_mb();
}

void *
dec_6600_pciide_compat_intr_establish(v, dev, pa, chan, func, arg)
	void *v;
	struct device *dev;
	struct pci_attach_args *pa;
	int chan;
	int (*func) __P((void *));
	void *arg;
{
	pci_chipset_tag_t pc = pa->pa_pc;
	void *cookie = NULL;
	int bus, irq;

	alpha_pci_decompose_tag(pc, pa->pa_tag, &bus, NULL, NULL);

	/*
	 * If this isn't PCI bus #0 on the TSP that holds the PCI-ISA
	 * bridge, all bets are off.
	 */
	if (bus != 0 || pc->pc_intr_v != sioprimary)
		return (NULL);

	irq = PCIIDE_COMPAT_IRQ(chan);
#if NSIO
	cookie = sio_intr_establish(NULL /*XXX*/, irq, IST_EDGE, IPL_BIO,
	    func, arg, "dec 6600 irq");

	if (cookie == NULL)
		return (NULL);
#endif
	return (cookie);
}

void
dec_6600_pciide_compat_intr_disestablish(v, cookie)
	void *v;
	void *cookie;
{
	sio_intr_disestablish(NULL, cookie);
}
