# watcher.tcl --
#
# This file contains code which handles the watcher window.
#
#
#  TkRat software and its included text is Copyright 1996-1999 by
#  by Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# WatcherInit --
#
# Initializes some watcher variables
#
# Arguments:
# handler -	The folder handler to which this watcher is connected

proc WatcherInit {handler} {
    global folderUnseen
    upvar #0 watcherHd_$handler hd

    set hd(unseen) $folderUnseen($handler)
}


# WatcherCreate --
#
# Builds the watcher window
#
# Arguments:
# handler -	The folder handler to which this watcher is connected

proc WatcherCreate {handler} {
    global b idCnt option watcherWins vFolderName watcherFont
    upvar #0 watcherHd_$handler hd

    # Create toplevel
    set w .w[incr idCnt]
    set watcherWins($handler) $w
    set hd(watcher_w) $w
    set hd(watcher_list) $w.list
    set hd(watcher_size) ""
    toplevel $w -class TkRat
    wm title $w $option(watcher_name)
    wm protocol $w WM_DELETE_WINDOW "WatcherSleep $handler"

    # Populate window
    frame $w.info
    label $w.info.name -textvariable vFolderName($handler) \
	    -font $watcherFont -relief raised -bd 1 -anchor w
    label $w.info.size -textvariable watcherHd_${handler}(watcher_size) \
	    -font $watcherFont -width 11 -relief raised -bd 1
    pack $w.info.size -side right
    pack $w.info.name -fill x -expand 1
    pack $w.info -side top -fill x -expand 1
    scrollbar $w.scroll \
	      -relief raised \
	      -bd 1 \
	      -highlightthickness 0 \
	      -command "$w.list yview"
    listbox $hd(watcher_list) \
	    -yscroll "$w.scroll set" \
	    -relief raised \
	    -bd 1 \
	    -font $watcherFont \
	    -exportselection false \
	    -highlightthickness 0
    set b($hd(watcher_list) watcher
    Size $hd(watcher_list) watcher
    pack $w.scroll -side right -fill y
    pack $w.list -side left -expand 1 -fill both
    Place $w watcher

    foreach but {<1> <B1-Motion> <ButtonRelease-1> <Shift-1> <Control-1>
	       <B1-Leave> <B1-Enter> <space> <Select> <Control-Shift-space>
	       <Shift-Select> <Control-slash> <Control-backslash>} {
	bind $w.list $but {break}
    }
    bind $w.list <ButtonRelease-1>	"WatcherWakeMaster $handler"
    bind $w.info.name <ButtonRelease-1>	"WatcherWakeMaster $handler"
    bind $w.info.size <ButtonRelease-1>	"WatcherWakeMaster $handler"
    bind $w.list <ButtonRelease-3>	"WatcherSleep $handler"
    bind $w.info.name <ButtonRelease-3>	"WatcherSleep $handler"
    bind $w.info.size <ButtonRelease-3>	"WatcherSleep $handler"
    bind $w.info.name <Destroy>		"unset watcherWins($handler) \
					       watcherHd_$handler"
    wm withdraw $w
}


# WatcherSleep --
#
# Unmaps the watcher window if it was mapped. This should be called
# whenever the folder window is unmapped.
#
# Arguments:
# handler -	The folder handler to which this watcher is connected

proc WatcherSleep {handler} {
    upvar #0 watcherHd_$handler hd
    if {[info exists hd(watcher_w)] && [winfo ismapped $hd(watcher_w)]} {
	wm withdraw $hd(watcher_w)
	regsub {[0-9]+x[0-9]+} [wm geom $hd(watcher_w)] {} hd(watcher_geom)
    }
}


# WatcherTrig --
#
# Called when the number of messages in the folder has changed
#
# Arguments:
# name1, name2, op - Trace arguments

proc WatcherTrig {name1 name2 op} {
    global option t folderUnseen vFolderWatch folderWindowList watcherWins

    if {"u" == $op} {
	foreach fhd [array names folderWindowList] {
	    if {"$name2" == $folderWindowList($fhd)} {
		FolderWindowClear $fhd
	    }
	}
	return
    }

    if {!$vFolderWatch($name2)} {
	return
    }
    upvar #0 watcherHd_$name2 hd

    if ![info exists watcherWins($name2)] {
	WatcherCreate $name2
    }

    # Check for new messages
    if {$hd(unseen) < $folderUnseen($name2)} {
	if {1 == [llength [info commands RatUP_Bell]]} {
	    if [catch {RatUP_Bell} text] {
		Popup "$t(bell_cmd_failed): $text"
	    }
	} else {
	    for {set i 0} {$i < $option(watcher_bell)} {incr i} {
		after 200
		bell
	    }
	}
	set popup $option(watcher_enable)
	foreach fhd [array names folderWindowList] {
	    if {"$name2" == $folderWindowList($fhd)} {
		upvar #0 $fhd fh

		Sync $fhd update
		if [winfo ismapped $fh(toplevel)] {
		    set popup 0
		}
	    }
	}
	if $popup {
	    WatcherPopup $name2
	}
    }
    set hd(unseen) $folderUnseen($name2)
}


# WatcherPopup --
#
# New mail has arrived so we need to populate and popup the watcher window.
#
# Arguments:
# handler -	The folder handler to which this watcher is connected

proc WatcherPopup {handler} {
    global option
    upvar #0 watcherHd_$handler hd

    # Some window managers (at least I think its them) like to move
    # the watcher while it is iconified. This tries to fix that.
    if [info exists hd(watcher_geom)] {
	wm geometry $hd(watcher_w) $hd(watcher_geom)
    }

    # Populate listbox
    $hd(watcher_list) delete 0 end
    switch $option(watcher_show) {
    new {
	    set fullList [$handler list $option(watcher_format)]
	    set i 0
	    foreach elem [$handler list %S] {
		if [regexp N $elem] {
		    $hd(watcher_list) insert end [lindex $fullList $i]
		}
		incr i
	    }
	}
    default {
	    eval "$fh(watcher_list) insert 0 \
		    [$handler list $option(watcher_format)]"
	}
    }
    set lines [$hd(watcher_list) size]
    set height $option(watcher_max_height)
    if {$lines > $height} {
	set lines $height
	if { -1 == [lsearch -exact \
		[pack slaves $hd(watcher_w)] $hd(watcher_w).scroll]} {
	    pack $hd(watcher_w).scroll -side right -fill y
	}
    } elseif { -1 != [lsearch -exact \
		[pack slaves $hd(watcher_w)] $hd(watcher_w).scroll]} {
	pack forget $hd(watcher_w).scroll
    }
    if !$lines {
	set lines 1
    }
    $hd(watcher_list) configure -height $lines
    $hd(watcher_list) see [expr [$hd(watcher_list) size]-1]
    set info [$handler info]
    set hd(watcher_size) "[lindex $info 1]/[RatMangleNumber [lindex $info 2]]"

    wm deiconify $hd(watcher_w)
}


# WatcherWakeMaster --
#
# The user wants us to wake the master up, so do that.
#
# Arguments:
# handler -	The folder handler to which this watcher is connected

proc WatcherWakeMaster {handler} {
    global folderWindowList

    foreach fhd [array names folderWindowList] {
	if {"$handler" == $folderWindowList($fhd)} {
	    upvar #0 $fhd fh
	    FolderSelectUnread $fhd
	    wm deiconify $fh(toplevel)
	    return
	}
    }
    WatcherSleep $handler

    # This must be a monitored folder, so find it and create a new folder
    global vFolderMonitorID vFolderDef
    NewFolderWin $vFolderDef($vFolderMonitorID($handler))
}
