/*	$NetBSD: $ */

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/kernel.h>

#include <uvm/uvm_extern.h>

#include <machine/pio.h>

#include <dev/ofw/openfirm.h>

#include <machine/autoconf.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include <dev/usb/usbdivar.h>
#include <dev/usb/usb_mem.h>

#include <dev/usb/ohcireg.h>
#include <dev/usb/ohcivar.h>

#include <arch/powerpc/mpc5200/mpc5200reg.h>
#include <arch/powerpc/mpc5200/mpc5200var.h>

static int mpc5200_ohci_match(device_t, cfdata_t, void *);
static void mpc5200_ohci_attach(device_t, device_t, void *);

struct mpc5200_ohci_softc {
	ohci_softc_t	sc;

	void *sc_ih;
	int sc_irq;
};

CFATTACH_DECL_NEW(ohci_mpc, sizeof(struct mpc5200_ohci_softc),
	mpc5200_ohci_match, mpc5200_ohci_attach, NULL, NULL);

static int
mpc5200_ohci_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;
	char name[32];

	if (strcmp(ca->ca_name, "usb") != 0)
		return 0;

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	aprint_normal("ohci_match name: %s\n", name);
#if 0
	if (strcmp(name, "mpc5200-ohci") != 0)
		return 0;
#endif

	return 1;
}

static void
mpc5200_ohci_attach(device_t parent, device_t self, void *aux)
{
	struct mpc5200_ohci_softc *sc = device_private(self);
	struct confargs *ca = aux;
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;
	uint32_t interrupts[3];
	char name[32];
	usbd_status r;

	sc->sc.sc_dev = self;
	sc->sc.sc_bus.hci_private = sc;
	sc->sc.sc_size = 0;
	sc->sc.iot = ca->ca_tag;
	sc->sc.sc_bus.dmatag = &bestcomm_bus_dma_tag;
	sc->sc.sc_endian = OHCI_HOST_ENDIAN;

	aprint_normal("\n");
	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range))
	    != sizeof(range))
		return;

	sc->sc.sc_size = rp->size;
	if (bus_space_map(sc->sc.iot, rp->addr, rp->size, 0, &sc->sc.ioh) != 0) {
		aprint_normal_dev(self, "map failed\n");
		return;
	}
	if (OF_getprop(ca->ca_node, "interrupts", interrupts, sizeof(interrupts))
	    != sizeof(interrupts))
		return;

	sc->sc_irq = (interrupts[0] * 32) + interrupts[1];
	sc->sc_ih = intr_establish(sc->sc_irq, IST_EDGE, IPL_USB,
				   ohci_intr, &sc->sc);

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	strlcpy(sc->sc.sc_vendor, name, sizeof(sc->sc.sc_vendor));

	r = ohci_init(&sc->sc);
	if (r != USBD_NORMAL_COMPLETION) {
		aprint_error_dev(sc->sc.sc_dev, "init failed, error=%d\n", r);
		goto free_intr;
	}		
	sc->sc.sc_child = config_found(self, &sc->sc.sc_bus, usbctlprint);
	return;

free_intr:
	intr_disestablish(sc->sc_ih);
}
