/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Alistair Crooks (agc@NetBSD.org)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/stat.h>

#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/*
 * This code implements the Gutmann overwrite method for scrubbing disks.
 * For more information on this method, please see:
 *
 *	http://en.wikipedia.org/wiki/Gutmann_method
 *
 * The method is designed to obliterate all traces of information on
 * a disk, whatever the disk encoding method or technology. This is
 * almost certainly overkill.
 */

#define MB(x)	(1024 * 1024 * (x))

#ifndef MIN
#define MIN(a, b)	((a) < (b) ? (a) : (b))
#endif

enum {
	DEFAULT_NUM_PASSES = 35
};

/*
 * This struct contains three const integers which define the sequence
 * to be written
 */
typedef struct pass_t {
	const uint32_t	pat0;
	const uint32_t	pat1;
	const uint32_t	pat2;
} pass_t;

/* these are the passes from the Gutmann method */
static pass_t	passes[35] = {
 	{	'r', 'a', 'n' 	},
 	{	'r', 'a', 'n'	},
 	{	'r', 'a', 'n'	},
 	{	'r', 'a', 'n'	},
 	{	0x55555555, 0x55555555, 0x55555555	},
 	{	0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa	},
 	{	0x92492492, 0x49249249, 0x24924924	},
 	{	0x49249249, 0x24924924, 0x92492492	},
 	{	0x24924924, 0x92492492, 0x49249249	},
 	{	0x0, 0x0, 0x0	},
 	{	0x11111111, 0x11111111, 0x11111111	},
 	{	0x22222222, 0x22222222, 0x22222222	},
 	{	0x33333333, 0x33333333, 0x33333333	},
 	{	0x44444444, 0x44444444, 0x44444444	},
 	{	0x55555555, 0x55555555, 0x55555555	},
 	{	0x66666666, 0x66666666, 0x66666666	},
 	{	0x77777777, 0x77777777, 0x77777777	},
 	{	0x88888888, 0x88888888, 0x88888888	},
 	{	0x99999999, 0x99999999, 0x99999999	},
 	{	0xaaaaaaaa, 0xaaaaaaaa, 0xaaaaaaaa	},
 	{	0xbbbbbbbb, 0xbbbbbbbb, 0xbbbbbbbb	},
 	{	0xcccccccc, 0xcccccccc, 0xcccccccc	},
 	{	0xdddddddd, 0xdddddddd, 0xdddddddd	},
 	{	0xeeeeeeee, 0xeeeeeeee, 0xeeeeeeee	},
 	{	0xffffffff, 0xffffffff, 0xffffffff	},
 	{	0x92492492, 0x49249249, 0x24924924	},
 	{	0x49249249, 0x24924924, 0x92492492	},
 	{	0x24924924, 0x92492492, 0x49249249	},
 	{	0x6db6db6d, 0xb6db6db6, 0xdb6db6db	},
 	{	0xb6db6db6, 0xdb6db6db, 0x6db6db6d	},
 	{	0xdb6db6db, 0x6db6db6d, 0xb6db6db6	},
 	{	'r', 'a', 'n'	},
 	{	'r', 'a', 'n'	},
 	{	'r', 'a', 'n'	},
 	{	'r', 'a', 'n'	},
};

#define USE_RANDOM(p0, p1, p2)	((p0) == 'r' && (p1) == 'a' && (p2) == 'n')

/*
 * We do the check for pass > passc in this function because it makes
 * main() cleaner.  The function opens "path", and writes a pattern into
 * it.  The three 32-bit integers represent the continuing series of the
 * pattern.  If they are 'r', 'a' and 'n', then we generate a random
 * pattern instead of a fixed pattern.
 */
static int
scrubber(const int passnum, const char *path, pass_t *pass, int verbose)
{
	struct stat	st;
	int64_t		i;
	int32_t		randval;
	char		buf[MB(1)];
	int		ret;
	int		fd;
	int		cc;

	if (verbose > 0) {
		printf("Gutmann pass %2.d, ", passnum);
		(void) fflush(stdout);
	}
	if ((fd = open(path, O_WRONLY, 0666)) < 0) {
		(void) fprintf(stderr, "can't open \"%s\" for writing", path);
		return 0;
	}
	if (USE_RANDOM(pass->pat0, pass->pat1, pass->pat2)) {
		if (verbose > 0) {
			printf("randomising buffer, ");
			(void) fflush(stdout);
		}
		for (i = 0 ; i < sizeof(buf) ; i += sizeof(randval)) {
			randval = random();
			(void) memcpy(&buf[i], &randval, sizeof(randval));
		}
	} else {
		if (verbose > 0) {
			printf("%08x%08x%08x, ", pass->pat0, pass->pat1, pass->pat2);
			(void) fflush(stdout);
		}
		(void) memcpy(buf, &pass->pat0, sizeof(uint32_t));
		(void) memcpy(&buf[sizeof(uint32_t)], &pass->pat1, sizeof(uint32_t));
		(void) memcpy(&buf[sizeof(uint32_t) * 2], &pass->pat2, sizeof(uint32_t));
		for (i = sizeof(uint32_t) * 3 ; i < sizeof(buf) ; i += sizeof(uint32_t) * 3) {
			(void) memcpy(&buf[i], buf, sizeof(uint32_t) * 3);
		}
	}
	if (fstat(fd, &st) < 0) {
		(void) fprintf(stderr, "can't stat %s to find its size\n", path);
		(void) close(fd);
		return 0;
	}
	if (verbose > 0) {
		printf("writing to \"%s\"\n", path);
		(void) fflush(stdout);
	}
	for (ret = 1, i = 0 ; i < st.st_size ; i += sizeof(buf)) {
		cc = MIN(st.st_size - i, sizeof(buf));
		if (write(fd, buf, cc) != cc) {
			(void) fprintf(stderr,
				"short write to \"%s\", pass %d, pattern %08x%08x%08x\n",
				path, passnum, pass->pat0, pass->pat1, pass->pat2);
			ret = 0;
		}
		if (fsync(fd) < 0) {
			(void) fprintf(stderr,
				"bad fsync to \"%s\", pass %d, pattern %08x%08x%08x\n",
				path, passnum, pass->pat0, pass->pat1, pass->pat2);
		}
	}
	(void) close(fd);
	return ret;
}

int
main(int argc, char **argv)
{
	int	verbose;
	int	passc;
	int	arg;
	int	ex;
	int	i;

	verbose = 0;
	passc = DEFAULT_NUM_PASSES;
	while ((i = getopt(argc, argv, "n:v")) != -1) {
		switch(i) {
		case 'n':
			if ((passc = atoi(optarg)) < 1) {
				passc = DEFAULT_NUM_PASSES;
			}
			break;
		case 'v':
			verbose += 1;
			break;
		default:
			(void) fprintf(stderr, "unknown option \"%c\"", i);
			break;
		}
	}
	for (ex = EXIT_SUCCESS, arg = optind ; arg < argc ; arg++) {
		for (i = 0 ; i < passc ; i++) {
			if (!scrubber(i + 1, argv[arg], &passes[i], verbose)) {
				(void) fprintf(stderr,
					"can't complete pass %d on %s\n",
					i + 1,
					argv[arg]);
				ex = EXIT_FAILURE;
			}
		}
	}
	exit(ex);
}
