# SPDX-License-Identifier: GPL-2.0-only
#
# Fish shell completion for StGit (stg)
#
# To use, copy this file to one of the paths in $fish_complete_path, e.g.:
#
#   ~/.config/fish/completions
#
# This file is autogenerated.

function __fish_stg
    set -l cd_args (__fish_stg_get_cd_args)
    eval command stg $cd_args $argv
end

function __fish_stg_git
    set -l cd_args (__fish_stg_get_cd_args)
    eval command git $cd_args $argv
end

function __fish_stg_get_cd_args
    set -l cmd (commandline -opc)
    # Erase 'stg' from command line
    set -e cmd[1]
    if argparse -s C=+ -- $cmd 2>/dev/null
        for path in $_flag_C
            echo -- -C
            echo -- $path
        end
    end
end

function __fish_stg_needs_command
    set -l cmd (commandline -opc)
    set -e cmd[1]
    argparse -s C=+ color= h/help version -- $cmd 2>/dev/null
    or return 0
    set -q _flag_version; and return 1
    if set -q argv[1]
        echo $argv[1]
        return 1
    end
    return 0
end

function __fish_stg_is_alias
    set -l alias_name (__fish_stg_needs_command)
    test -z "$alias_name"
    and return 1
    contains -- $alias_name (__fish_stg completion list aliases)
    and return 0
end

function __fish_stg_complete_alias
    set -l cmd (commandline -opc) (commandline -ct)
    set -l alias_name (__fish_stg_needs_command)
    set -e cmd[1]  # Erase 'stg'
    while test "$cmd[1]" != "$alias_name"
        set -e cmd[1]
    end
    set -e cmd[1]
    __fish_stg completion list aliases --style=fish --show-expansion | \
    while read -l -d \t aname expansion
        if test "$alias_name" = "$aname"
            if test (string sub --length=4 $expansion) = '!git'
                set -l cd_args (__fish_stg_get_cd_args)
                set -l expansion (string sub --start=5 $expansion)
                complete -C "git $cd_args $expansion $cmd"
            else if test (string sub --length=1 $expansion) = '!'
                set -l expansion (string sub --start=2 $expansion)
                complete -C "$expansion $cmd"
            else
                set -l cd_args (__fish_stg_get_cd_args)
                complete -C "stg $cd_args $expansion $cmd"
            end
            break
        end
    end
end

function __fish_stg_all_branches
    __fish_stg_git for-each-ref --format='%\(refname\)' \
        refs/heads/ refs/remotes/ 2>/dev/null \
        | string replace -r '^refs/heads/(.*)$' '$1\tLocal Branch' \
        | string replace -r '^refs/remotes/(.*)$' '$1\tRemote Branch'
end

function __fish_stg_stg_branches
    __fish_stg branch --list 2>/dev/null \
        | string match -r ". s.\t\S+" \
        | string replace -r ". s.\t" ""
end

function __fish_stg_patches
    set -l index 1
    __fish_stg series --color=never --no-description $argv 2>/dev/null | \
    while read -d ' ' sigil patch
        echo -e "$patch\t$index $sigil"
        set index (math $index + 1)
    end
end

function __fish_stg_tags
    __fish_stg_git tag --sort=-creatordate 2>/dev/null
end

function __fish_stg_commit
    __fish_stg_all_branches __fish_stg_tags
end

function __fish_stg_conflicting_files
    __fish_stg_git ls-files --unmerged \
        | string replace -rf '^.*\t(.*)$' '$1' \
        | sort -u
end

function __fish_stg_dirty_files
    __fish_stg_git diff-index --name-only HEAD 2>/dev/null
end

function __fish_stg_unknown_files
    __fish_stg_git ls-files --others --exclude-standard 2>/dev/null
end

function __fish_stg_known_files
    __fish_stg_git ls-files 2>/dev/null
end

function __fish_stg_mail_aliases
    __fish_stg_git config --name-only --get-regexp "^mail\.alias\." \
    | cut -d. -f 3
end

function __fish_stg_git_diff_opts
    __fish_stg_git diff-tree --git-completion-helper \
    | string split ' ' \
    | string match --regex '\--.+'
end

function __fish_stg_git_format_patch_opts
    __fish_stg_git format-patch --git-completion-helper \
    | string split ' ' \
    | string match --regex '\--.+'
end

function __fish_stg_git_send_email_opts
    __fish_stg_git send-email --git-completion-helper \
    | string split ' ' \
    | string match --regex '\--.+'
end

complete -c stg -n __fish_stg_needs_command -xa '(stg completion list commands-and-aliases --style=fish)'
complete -c stg -n __fish_stg_needs_command -s C -xa '(__fish_complete_directories)' -d 'Run as if started in this directory'
complete -c stg -n __fish_stg_needs_command -l color -a 'auto always ansi never' -d 'When to colorize output'
complete -c stg -n __fish_stg_needs_command -l version -d 'Print version information'
complete -c stg -n __fish_stg_needs_command -s h -l help -d 'Print help information'

complete -c stg -n __fish_stg_is_alias -xa '(__fish_stg_complete_alias)'

### branch
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -xa '(__fish_stg_all_branches)' -d 'Branch to switch to'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d'      -l merge -d 'Merge work tree changes into the other branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s l -l list -d 'List branches in this repository'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s c -l create -d 'Create and switch to a new branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s C -l clone -d 'Clone the contents of the current branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s r -l rename -d 'Rename an existing branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s p -l protect -d 'Prevent StGit from modifying a branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s u -l unprotect -d 'Allow StGit to modify a previously protected branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s D -l delete -d 'Delete a branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -l cleanup -d 'Remove StGit patch stack from branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list -l --create -c --clone -C --rename -r --protect -p --unprotect -u --delete -D --cleanup --describe --description -d' -s d -l describe -d 'Set the branch description'

### --list
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --list -l'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --list -l' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --list -l' -f

### --create
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --create -c' -x -d 'New branch name'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --create -c' -xa '(__fish_stg_commit)' -d 'Base commit for new branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --create -c'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --create -c' -s h -l help -d 'Print help (see more with \'--help\')'

### --clone
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --clone -C' -x -d 'New branch name'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --clone -C'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --clone -C' -s h -l help -d 'Print help (see more with \'--help\')'

### --rename
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --rename -r' -xa '(__fish_stg_all_branches)' -d 'Optional name of branch to rename and new branch name'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --rename -r'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --rename -r' -s h -l help -d 'Print help (see more with \'--help\')'

### --protect
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --protect -p' -xa '(__fish_stg_stg_branches)' -d 'Branch to protect'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --protect -p'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --protect -p' -s h -l help -d 'Print help (see more with \'--help\')'

### --unprotect
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --unprotect -u' -xa '(__fish_stg_stg_branches)' -d 'Branch to unprotect'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --unprotect -u'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --unprotect -u' -s h -l help -d 'Print help (see more with \'--help\')'

### --delete
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --delete -D' -xa '(__fish_stg_all_branches)' -d 'Branch to delete'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --delete -D'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --delete -D'      -l force -d 'Force deletion even if branch has patches'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --delete -D' -s h -l help -d 'Print help (see more with \'--help\')'

### --cleanup
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --cleanup' -xa '(__fish_stg_stg_branches)' -d 'Branch to clean up'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --cleanup'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --cleanup'      -l force -d 'Force clean up even if branch has patches'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --cleanup' -s h -l help -d 'Print help (see more with \'--help\')'

### --describe
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --describe --description -d' -x -d 'Description string for branch'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --describe --description -d' -xa '(__fish_stg_all_branches)' -d 'Branch to describe'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --describe --description -d'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --describe --description -d' -s h -l help -d 'Print help (see more with \'--help\')'

### clean
complete -c stg -n '__fish_seen_subcommand_from clean'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from clean' -s A -l applied -d 'Delete empty applied patches'
complete -c stg -n '__fish_seen_subcommand_from clean' -s U -l unapplied -d 'Delete empty unapplied patches'
complete -c stg -n '__fish_seen_subcommand_from clean' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from clean' -f

### commit
complete -c stg -n '__fish_seen_subcommand_from commit' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to commit'
complete -c stg -n '__fish_seen_subcommand_from commit' -s n -l number -x -d 'Commit the specified number of applied patches'
complete -c stg -n '__fish_seen_subcommand_from commit'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from commit' -s a -l all -d 'Commit all applied patches'
complete -c stg -n '__fish_seen_subcommand_from commit'      -l allow-empty -d 'Allow empty patches to be committed'
complete -c stg -n '__fish_seen_subcommand_from commit' -s h -l help -d 'Print help (see more with \'--help\')'

### completion
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -s o -l output -rF -d 'Output to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'bash' -d 'Generate bash completion script'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'fish' -d 'Generate fish shell completion script'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'zsh' -d 'Generate zsh completion script'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'list' -d 'List StGit command information'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'man' -d 'Generate asciidoc man pages'
complete -c stg -n '__fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### bash
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from bash' -s o -l output -rF -d 'Output completion script to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from bash'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from bash' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from bash' -f

### fish
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from fish' -s o -l output -rF -d 'Output completion script to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from fish'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from fish' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from fish' -f

### zsh
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from zsh' -s o -l output -rF -d 'Output completion script to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from zsh'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from zsh' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from zsh' -f

### list
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help'      -l style -xa 'name-only asciidoc fish zsh' -d 'Choose output format style'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -s o -l output -rF -d 'Output to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'aliases' -d 'List aliases'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'commands' -d 'List StGit commands'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'commands-and-aliases' -d 'List StGit commands and aliases'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### aliases
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases'      -l style -xa 'name-only asciidoc fish zsh' -d 'Choose output format style'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases' -s o -l output -rF -d 'Output to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases'      -l show-expansion -d 'Show alias expansion'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases' -f

### commands
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands'      -l style -xa 'name-only asciidoc fish zsh' -d 'Choose output format style'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands' -s o -l output -rF -d 'Output to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands' -f

### commands-and-aliases
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases'      -l style -xa 'name-only asciidoc fish zsh' -d 'Choose output format style'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases' -s o -l output -rF -d 'Output to <path>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases' -f

### help
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'aliases' -d 'List aliases'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'commands' -d 'List StGit commands'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'commands-and-aliases' -d 'List StGit commands and aliases'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### aliases
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from aliases' -f

### commands
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from commands' -f

### commands-and-aliases
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from commands-and-aliases' -f

### help
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from help' -f

### man
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from man' -s o -l output -xa '(__fish_complete_directories)' -d 'Output man pages to <dir>'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from man'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from man' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from man' -f

### help
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'bash' -d 'Generate bash completion script'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'fish' -d 'Generate fish shell completion script'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'zsh' -d 'Generate zsh completion script'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'list' -d 'List StGit command information'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'man' -d 'Generate asciidoc man pages'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from bash fish zsh list man help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### bash
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from bash' -f

### fish
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from fish' -f

### zsh
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from zsh' -f

### list
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases' -xa 'aliases' -d 'List aliases'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases' -xa 'commands' -d 'List StGit commands'
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases' -xa 'commands-and-aliases' -d 'List StGit commands and aliases'

### aliases
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases' -f

### commands
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands' -f

### commands-and-aliases
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases' -f

### man
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from man' -f

### help
complete -c stg -n '__fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from help' -f

### delete
complete -c stg -n '__fish_seen_subcommand_from delete' -kxa '(__fish_stg_patches --all)' -d 'Patches to delete'
complete -c stg -n '__fish_seen_subcommand_from delete' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from delete'      -l conflicts -xa 'allow disallow' -d '"allow" or "disallow" pushing a patch with conflicts'
complete -c stg -n '__fish_seen_subcommand_from delete'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from delete' -s a -l all -d 'Delete all patches'
complete -c stg -n '__fish_seen_subcommand_from delete' -s A -l applied -d 'Delete the applied patches'
complete -c stg -n '__fish_seen_subcommand_from delete' -s U -l unapplied -d 'Delete the unapplied patches'
complete -c stg -n '__fish_seen_subcommand_from delete' -s H -l hidden -d 'Delete the hidden patches'
complete -c stg -n '__fish_seen_subcommand_from delete' -s t -l top -d 'Delete topmost patch'
complete -c stg -n '__fish_seen_subcommand_from delete'      -l spill -d 'Spill patch contents to worktree and index'
complete -c stg -n '__fish_seen_subcommand_from delete' -s h -l help -d 'Print help (see more with \'--help\')'

### diff
complete -c stg -n '__fish_seen_subcommand_from diff' -rF -d 'Limit diff to files matching path(s)'
complete -c stg -n '__fish_seen_subcommand_from diff' -s r -l range -x -d 'Show the diff between specified revisions'
complete -c stg -n '__fish_seen_subcommand_from diff' -s O -l diff-opt -xa '(__fish_stg_git_diff_opts)' -d 'Pass additional <option> to `git diff`'
complete -c stg -n '__fish_seen_subcommand_from diff'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from diff' -s s -l stat -d 'Show the stat instead of the diff'
complete -c stg -n '__fish_seen_subcommand_from diff' -s h -l help -d 'Print help (see more with \'--help\')'

### edit
complete -c stg -n '__fish_seen_subcommand_from edit' -kxa '(__fish_stg_patches -A -U)' -d 'Patch to edit'
complete -c stg -n '__fish_seen_subcommand_from edit' -s m -l message -x -d 'Use message for patch'
complete -c stg -n '__fish_seen_subcommand_from edit' -s f -l file -rF -d 'Get message from file'
complete -c stg -n '__fish_seen_subcommand_from edit' -s s -l signoff -x -d 'Add Signed-off-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l ack -x -d 'Add Acked-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l review -x -d 'Add Reviewed-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l author -x -d 'Set the author "name <email>"'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l authname -x -d 'Set the author name'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l authemail -x -d 'Set the author email'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l authdate -x -d 'Set the author date'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l save-template -rF -d 'Save the patch description to FILE and exit'
complete -c stg -n '__fish_seen_subcommand_from edit' -s t -l set-tree -kxa '(__fish_stg_patches --all)' -d 'Set patch\'s tree to treeish'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from edit' -s e -l edit -d 'Invoke editor for patch description'
complete -c stg -n '__fish_seen_subcommand_from edit' -s d -l diff -d 'Show diff when editing patch description'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l no-verify -d 'Disable commit-msg hook'
complete -c stg -n '__fish_seen_subcommand_from edit'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from edit' -s h -l help -d 'Print help (see more with \'--help\')'

### email
complete -c stg -n '__fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send help'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send help' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send help' -xa 'format' -d 'Format patches as email files'
complete -c stg -n '__fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send help' -xa 'send' -d 'Send patches as emails'
complete -c stg -n '__fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### format
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to format'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s G -l git-opt -xa '(__fish_stg_git_format_patch_opts)' -d 'Pass additional <option> to `git format-patch`'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s o -l output-directory -xa '(__fish_complete_directories)' -d 'Store output files in <dir>'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l start-number -x -d 'Start numbering at <n> instead of 1'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s v -l reroll-count -x -d 'Mark the series as the <n>th reroll'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l subject-prefix -x -d 'Use [<prefix>] instead of [PATCH]'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l suffix -x -d 'Use <suffix> instead of \'.patch\''
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l to -x -d 'Specify a To: address for each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l cc -x -d 'Specify a Cc: address for each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l in-reply-to -x -d 'Make first mail a reply to <message-id>'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l add-header -x -d 'Add an arbitrary email header'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l thread -xa 'shallow deep ' -d 'Enable message threading, styles: shallow or deep'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l signature -x -d 'Add a signature to each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l signature-file -rF -d 'Add a signature from a file'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l base -x -d 'Add prerequisite tree info to the patch series'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l interdiff -x -d 'Show changes against <rev> in cover letter'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l range-diff -x -d 'Show changes against <refspec> in cover letter'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l creation-factor -x -d 'Percentage by which creation is weighed'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s a -l all -d 'Format all applied patches'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l cover-letter -d 'Generate a cover letter'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s n -l numbered -d 'Use [PATCH n/m] even with a single patch'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s N -l no-numbered -d 'Use [PATCH] even with multiple patches'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l rfc -d 'Use [RFC PATCH] instead of [PATCH]'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l quiet -d 'Do not print the names of the generated files'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s s -l signoff -d 'Add a Signed-off-by trailer'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l numbered-files -d 'Use simple number sequence for output file names'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s k -l keep-subject -d 'Do not strip/add [PATCH]'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l no-binary -d 'Do not output binary diffs'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l zero-commit -d 'Output all-zero hash in From header'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l no-to -d 'Discard all To: headers added so far'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l no-cc -d 'Discard all Cc: addresses added so far'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l attach -d 'Attach the patch'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l inline -d 'Inline the patch'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l no-thread -d 'Disable message threading'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l no-signature -d 'Do not add a signature to each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format'      -l progress -d 'Show progress while generating patches'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -s h -l help -d 'Print help (see more with \'--help\')'

### send
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -x -d 'Patch files, directory of patch files, or patch names to send'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s G -l git-opt -xa '(__fish_stg_git_send_email_opts)' -d 'Pass additional <option> to `git send-email`'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l from -x -d 'Specify the "From:" address for each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l to -x -d 'Specify the "To:" address for each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l cc -x -d 'Specify a "Cc:" address for each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l bcc -x -d 'Specify a "Bcc:" address for each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l subject -x -d 'Specify email "Subject:"'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l reply-to -x -d 'Specify the "Reply-To:" address'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l in-reply-to -x -d 'Specify the "In-Reply-To:" identifier'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l identity -x -d 'Use the sendmail.<id> options'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l confirm -xa 'always never cc compose auto' -d 'Confirm recipients before sending'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l start-number -x -d 'Start numbering at <n> instead of 1'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s v -l reroll-count -x -d 'Mark the series as the <n>th reroll'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l subject-prefix -x -d 'Use [<prefix>] instead of [PATCH]'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s a -l all -d 'Send all applied patches'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l compose -d 'Open an editor for introduction'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l annotate -d 'Review each patch that will be sent in an editor'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l no-thread -d 'Do not add In-Reply-To and Reference headers to each email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l quiet -d 'Output one line of info per email'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l dry-run -d 'Do not actually send the emails'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l dump-aliases -d 'Dump configured aliases and exit'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s n -l numbered -d 'Use [PATCH n/m] even with a single patch'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s N -l no-numbered -d 'Use [PATCH] even with multiple patches'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send'      -l rfc -d 'Use [RFC PATCH] instead of [PATCH]'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -s h -l help -d 'Print help (see more with \'--help\')'

### help
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from format send help' -xa 'format' -d 'Format patches as email files'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from format send help' -xa 'send' -d 'Send patches as emails'
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from format send help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### format
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from format' -f

### send
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from send' -f

### help
complete -c stg -n '__fish_seen_subcommand_from email ; and __fish_seen_subcommand_from help ; and __fish_seen_subcommand_from help' -f

### export
complete -c stg -n '__fish_seen_subcommand_from export' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to export'
complete -c stg -n '__fish_seen_subcommand_from export' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from export' -s d -l dir -xa '(__fish_complete_directories)' -d 'Export patches to <dir> instead of the default'
complete -c stg -n '__fish_seen_subcommand_from export' -s e -l extension -x -d 'Suffix patch file names with ".<ext>"'
complete -c stg -n '__fish_seen_subcommand_from export' -s t -l template -rF -d 'Use <file> as template'
complete -c stg -n '__fish_seen_subcommand_from export' -s O -l diff-opt -xa '(__fish_stg_git_diff_opts)' -d 'Pass additional <option> to `git diff`'
complete -c stg -n '__fish_seen_subcommand_from export'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from export' -s p -l patch -d 'Suffix patch file names with ".patch"'
complete -c stg -n '__fish_seen_subcommand_from export' -s n -l numbered -d 'Prefix patch file names with order numbers.'
complete -c stg -n '__fish_seen_subcommand_from export' -s s -l stdout -d 'Export to stdout instead of directory'
complete -c stg -n '__fish_seen_subcommand_from export' -s h -l help -d 'Print help (see more with \'--help\')'

### files
complete -c stg -n '__fish_seen_subcommand_from files' -kxa '(__fish_stg_patches --all)' -d 'StGit revision'
complete -c stg -n '__fish_seen_subcommand_from files'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from files' -s s -l stat -d 'Show patch\'s diffstat'
complete -c stg -n '__fish_seen_subcommand_from files'      -l bare -d 'Print bare file names'
complete -c stg -n '__fish_seen_subcommand_from files' -s h -l help -d 'Print help (see more with \'--help\')'

### float
complete -c stg -n '__fish_seen_subcommand_from float' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to float'
complete -c stg -n '__fish_seen_subcommand_from float' -s S -l series -rF -d 'Rearrange according to a series <file>'
complete -c stg -n '__fish_seen_subcommand_from float'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from float'      -l noapply -d 'Reorder patches without reapplying any patches'
complete -c stg -n '__fish_seen_subcommand_from float' -s k -l keep -d 'Keep the local changes'
complete -c stg -n '__fish_seen_subcommand_from float'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from float' -s h -l help -d 'Print help (see more with \'--help\')'

### fold
complete -c stg -n '__fish_seen_subcommand_from fold' -rF -d 'GNU diff file'
complete -c stg -n '__fish_seen_subcommand_from fold' -s b -l base -x -d 'Use <committish> instead of HEAD when applying the patch'
complete -c stg -n '__fish_seen_subcommand_from fold' -s p -l strip -x -d 'Remove <n> leading components from diff paths (default 1)'
complete -c stg -n '__fish_seen_subcommand_from fold' -s C -x -d 'Ensure <n> lines of matching context for each change'
complete -c stg -n '__fish_seen_subcommand_from fold'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from fold' -s t -l threeway -d 'Perform a three-way merge with the current patch'
complete -c stg -n '__fish_seen_subcommand_from fold'      -l reject -d 'Leave rejected hunks in ".rej" files'
complete -c stg -n '__fish_seen_subcommand_from fold' -s h -l help -d 'Print help (see more with \'--help\')'

### goto
complete -c stg -n '__fish_seen_subcommand_from goto' -kxa '(__fish_stg_patches -A -U)' -d 'Patch to go to'
complete -c stg -n '__fish_seen_subcommand_from goto'      -l conflicts -xa 'allow disallow' -d '"allow" or "disallow" pushing a patch with conflicts'
complete -c stg -n '__fish_seen_subcommand_from goto'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from goto' -s k -l keep -d 'Keep the local changes'
complete -c stg -n '__fish_seen_subcommand_from goto' -s m -l merged -d 'Check for patches merged upstream'
complete -c stg -n '__fish_seen_subcommand_from goto'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from goto' -s h -l help -d 'Print help (see more with \'--help\')'

### hide
complete -c stg -n '__fish_seen_subcommand_from hide' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to hide'
complete -c stg -n '__fish_seen_subcommand_from hide' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from hide'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from hide' -s h -l help -d 'Print help (see more with \'--help\')'

### id
complete -c stg -n '__fish_seen_subcommand_from id' -kxa '(__fish_stg_patches --all)' -d 'StGit revision'
complete -c stg -n '__fish_seen_subcommand_from id' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from id'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from id' -s h -l help -d 'Print help (see more with \'--help\')'

### import
complete -c stg -n '__fish_seen_subcommand_from import' -rF -d 'Source of patches to import'
complete -c stg -n '__fish_seen_subcommand_from import' -s n -l name -x -d 'Use <name> as the patch name'
complete -c stg -n '__fish_seen_subcommand_from import' -s p -l strip -x -d 'Remove <n> leading components from diff paths (default 1)'
complete -c stg -n '__fish_seen_subcommand_from import'      -l directory -xa '(__fish_complete_directories)' -d 'Prepend <root> to all filenames'
complete -c stg -n '__fish_seen_subcommand_from import' -s C -x -d 'Ensure <n> lines of matching context for each change'
complete -c stg -n '__fish_seen_subcommand_from import' -s b -l base -x -d 'Use <committish> instead of HEAD for file importing'
complete -c stg -n '__fish_seen_subcommand_from import' -s s -l signoff -x -d 'Add Signed-off-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from import'      -l ack -x -d 'Add Acked-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from import'      -l review -x -d 'Add Reviewed-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from import'      -l author -x -d 'Set the author "name <email>"'
complete -c stg -n '__fish_seen_subcommand_from import'      -l authname -x -d 'Set the author name'
complete -c stg -n '__fish_seen_subcommand_from import'      -l authemail -x -d 'Set the author email'
complete -c stg -n '__fish_seen_subcommand_from import'      -l authdate -x -d 'Set the author date'
complete -c stg -n '__fish_seen_subcommand_from import'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from import' -s m -l mail -d 'Import patch from an email file'
complete -c stg -n '__fish_seen_subcommand_from import' -s M -l mbox -d 'Import patch series from an mbox file'
complete -c stg -n '__fish_seen_subcommand_from import' -s S -l series -d 'Import patch series'
complete -c stg -n '__fish_seen_subcommand_from import' -s u -l url -d 'Retrieve source from a url instead of local file'
complete -c stg -n '__fish_seen_subcommand_from import' -s t -l stripname -d 'Strip number and extension from patch name'
complete -c stg -n '__fish_seen_subcommand_from import' -s 3 -l 3way -d 'Attempt three-way merge'
complete -c stg -n '__fish_seen_subcommand_from import' -s i -l ignore -d 'Ignore the applied patches in the series'
complete -c stg -n '__fish_seen_subcommand_from import'      -l replace -d 'Replace the unapplied patches in the series'
complete -c stg -n '__fish_seen_subcommand_from import'      -l reject -d 'Leave rejected hunks in ".rej" files'
complete -c stg -n '__fish_seen_subcommand_from import'      -l keep-cr -d 'Do not remove "\\r" from email lines ending with "\\r\\n"'
complete -c stg -n '__fish_seen_subcommand_from import'      -l message-id -d 'Create Message-ID trailer from Message-ID header'
complete -c stg -n '__fish_seen_subcommand_from import' -s e -l edit -d 'Invoke editor for patch description'
complete -c stg -n '__fish_seen_subcommand_from import' -s d -l diff -d 'Show diff when editing patch description'
complete -c stg -n '__fish_seen_subcommand_from import'      -l no-verify -d 'Disable commit-msg hook'
complete -c stg -n '__fish_seen_subcommand_from import'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from import' -s h -l help -d 'Print help (see more with \'--help\')'

### init
complete -c stg -n '__fish_seen_subcommand_from init' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from init'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from init' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from init' -f

### log
complete -c stg -n '__fish_seen_subcommand_from log' -kxa '(__fish_stg_patches --all)' -d 'Only show history for these patches'
complete -c stg -n '__fish_seen_subcommand_from log' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from log' -s n -l number -x -d 'Limit output to <n> commits'
complete -c stg -n '__fish_seen_subcommand_from log'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from log' -s d -l diff -d 'Show stack state diffs'
complete -c stg -n '__fish_seen_subcommand_from log' -s f -l full -d 'Show using full commit log format'
complete -c stg -n '__fish_seen_subcommand_from log' -s g -l graphical -d 'Run gitk instead of printing to stdout'
complete -c stg -n '__fish_seen_subcommand_from log'      -l clear -d 'Clear the stack history'
complete -c stg -n '__fish_seen_subcommand_from log' -s h -l help -d 'Print help (see more with \'--help\')'

### name
complete -c stg -n '__fish_seen_subcommand_from name' -kxa '(__fish_stg_patches --all)' -d 'StGit revision'
complete -c stg -n '__fish_seen_subcommand_from name' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from name'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from name'      -l showbranch -d 'Display the branch name with the patch'
complete -c stg -n '__fish_seen_subcommand_from name' -s h -l help -d 'Print help (see more with \'--help\')'

### new
complete -c stg -n '__fish_seen_subcommand_from new' -x -d 'Name for new patch'
complete -c stg -n '__fish_seen_subcommand_from new' -rF -d 'Refresh files matching path(s)'
complete -c stg -n '__fish_seen_subcommand_from new' -s n -l name -x -d 'Name for new patch'
complete -c stg -n '__fish_seen_subcommand_from new' -s m -l message -x -d 'Use message for patch'
complete -c stg -n '__fish_seen_subcommand_from new' -s f -l file -rF -d 'Get message from file'
complete -c stg -n '__fish_seen_subcommand_from new' -s s -l signoff -x -d 'Add Signed-off-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from new'      -l ack -x -d 'Add Acked-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from new'      -l review -x -d 'Add Reviewed-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from new'      -l author -x -d 'Set the author "name <email>"'
complete -c stg -n '__fish_seen_subcommand_from new'      -l authname -x -d 'Set the author name'
complete -c stg -n '__fish_seen_subcommand_from new'      -l authemail -x -d 'Set the author email'
complete -c stg -n '__fish_seen_subcommand_from new'      -l authdate -x -d 'Set the author date'
complete -c stg -n '__fish_seen_subcommand_from new'      -l save-template -rF -d 'Save the patch description to FILE and exit'
complete -c stg -n '__fish_seen_subcommand_from new'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from new' -s r -l refresh -d 'Refresh new patch with changes from work tree or index'
complete -c stg -n '__fish_seen_subcommand_from new' -s i -l index -d 'Refresh from index instead of work tree'
complete -c stg -n '__fish_seen_subcommand_from new' -s F -l force -d 'Force refresh with staged and unstaged changes'
complete -c stg -n '__fish_seen_subcommand_from new'      -l submodules -d 'Include submodules in patch content'
complete -c stg -n '__fish_seen_subcommand_from new'      -l no-submodules -d 'Exclude submodules in patch content'
complete -c stg -n '__fish_seen_subcommand_from new' -s e -l edit -d 'Invoke editor for patch description'
complete -c stg -n '__fish_seen_subcommand_from new' -s d -l diff -d 'Show diff when editing patch description'
complete -c stg -n '__fish_seen_subcommand_from new'      -l no-verify -d 'Disable commit-msg hook'
complete -c stg -n '__fish_seen_subcommand_from new'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from new' -s h -l help -d 'Print help (see more with \'--help\')'

### next
complete -c stg -n '__fish_seen_subcommand_from next' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from next'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from next' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from next' -f

### patches
complete -c stg -n '__fish_seen_subcommand_from patches' -rF -d 'Show patches that modify these paths'
complete -c stg -n '__fish_seen_subcommand_from patches' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from patches' -s O -l diff-opt -xa '(__fish_stg_git_diff_opts)' -d 'Pass additional <option> to `git diff`'
complete -c stg -n '__fish_seen_subcommand_from patches'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from patches' -s d -l diff -d 'Show the diff for the given paths'
complete -c stg -n '__fish_seen_subcommand_from patches' -s h -l help -d 'Print help (see more with \'--help\')'

### pick
complete -c stg -n '__fish_seen_subcommand_from pick' -kxa '(__fish_stg_patches --all)' -d 'Patch name or committish to import'
complete -c stg -n '__fish_seen_subcommand_from pick' -s B -l ref-branch -xa '(__fish_stg_stg_branches)' -d 'Pick patches from <branch>'
complete -c stg -n '__fish_seen_subcommand_from pick' -s n -l name -x -d 'Use <name> for the patch name'
complete -c stg -n '__fish_seen_subcommand_from pick' -s p -l parent -x -d 'Use <committish> as parent'
complete -c stg -n '__fish_seen_subcommand_from pick' -s f -l file -rF -d 'Only fold the given file (may be used multiple times)'
complete -c stg -n '__fish_seen_subcommand_from pick'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from pick' -s r -l revert -d 'Revert the given commit object'
complete -c stg -n '__fish_seen_subcommand_from pick' -s x -l expose -d 'Append the imported commit id to the patch log'
complete -c stg -n '__fish_seen_subcommand_from pick'      -l noapply -d 'Keep the imported patch unapplied'
complete -c stg -n '__fish_seen_subcommand_from pick'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from pick'      -l fold -d 'Fold the commit object into the current patch'
complete -c stg -n '__fish_seen_subcommand_from pick'      -l update -d 'Like fold but only update the current patch\'s files'
complete -c stg -n '__fish_seen_subcommand_from pick' -s h -l help -d 'Print help (see more with \'--help\')'

### pop
complete -c stg -n '__fish_seen_subcommand_from pop' -kxa '(__fish_stg_patches --applied)' -d 'Patches to pop'
complete -c stg -n '__fish_seen_subcommand_from pop' -s n -l number -x -d 'Pop specified <number> of patches'
complete -c stg -n '__fish_seen_subcommand_from pop'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from pop' -s a -l all -d 'Pop all applied patches'
complete -c stg -n '__fish_seen_subcommand_from pop' -s s -l spill -d 'Keep patches\' modifications in index and worktree after popping'
complete -c stg -n '__fish_seen_subcommand_from pop' -s k -l keep -d 'Keep the local changes'
complete -c stg -n '__fish_seen_subcommand_from pop' -s h -l help -d 'Print help (see more with \'--help\')'

### prev
complete -c stg -n '__fish_seen_subcommand_from prev' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from prev'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from prev' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from prev' -f

### pull
complete -c stg -n '__fish_seen_subcommand_from pull' -x -d 'Repository to pull from'
complete -c stg -n '__fish_seen_subcommand_from pull'      -l conflicts -xa 'allow disallow' -d '"allow" or "disallow" pushing a patch with conflicts'
complete -c stg -n '__fish_seen_subcommand_from pull'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from pull' -s n -l nopush -d 'Do not push back patches after pulling'
complete -c stg -n '__fish_seen_subcommand_from pull' -s m -l merged -d 'Check for patches merged upstream'
complete -c stg -n '__fish_seen_subcommand_from pull' -s h -l help -d 'Print help (see more with \'--help\')'

### push
complete -c stg -n '__fish_seen_subcommand_from push' -kxa '(__fish_stg_patches --unapplied)' -d 'Patches to push'
complete -c stg -n '__fish_seen_subcommand_from push' -s n -l number -x -d 'Push specified number of patches'
complete -c stg -n '__fish_seen_subcommand_from push'      -l conflicts -xa 'allow disallow' -d '"allow" or "disallow" pushing a patch with conflicts'
complete -c stg -n '__fish_seen_subcommand_from push'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from push' -s a -l all -d 'Push all unapplied patches'
complete -c stg -n '__fish_seen_subcommand_from push'      -l reverse -d 'Push the patches in reverse order'
complete -c stg -n '__fish_seen_subcommand_from push'      -l noapply -d 'Reorder patches by pushing without applying'
complete -c stg -n '__fish_seen_subcommand_from push'      -l set-tree -d 'Push patches keeping their original trees'
complete -c stg -n '__fish_seen_subcommand_from push' -s k -l keep -d 'Keep the local changes'
complete -c stg -n '__fish_seen_subcommand_from push' -s m -l merged -d 'Check for patches merged upstream'
complete -c stg -n '__fish_seen_subcommand_from push'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from push' -s h -l help -d 'Print help (see more with \'--help\')'

### rebase
complete -c stg -n '__fish_seen_subcommand_from rebase' -xa '(__fish_stg_commit)' -d 'New base commit for the stack'
complete -c stg -n '__fish_seen_subcommand_from rebase'      -l conflicts -xa 'allow disallow' -d '"allow" or "disallow" pushing a patch with conflicts'
complete -c stg -n '__fish_seen_subcommand_from rebase'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from rebase' -s i -l interactive -d 'Interactively manipulate patches in editor'
complete -c stg -n '__fish_seen_subcommand_from rebase' -s n -l nopush -d 'Do not push back patches after rebasing'
complete -c stg -n '__fish_seen_subcommand_from rebase' -s m -l merged -d 'Check for patches merged upstream'
complete -c stg -n '__fish_seen_subcommand_from rebase'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from rebase'      -l autostash -d 'Stash changes before the rebase and reapply them after'
complete -c stg -n '__fish_seen_subcommand_from rebase' -s h -l help -d 'Print help (see more with \'--help\')'

### redo
complete -c stg -n '__fish_seen_subcommand_from redo' -s n -l number -x -d 'Undo the last <n> undos'
complete -c stg -n '__fish_seen_subcommand_from redo'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from redo'      -l hard -d 'Discard changes in the index and worktree'
complete -c stg -n '__fish_seen_subcommand_from redo' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from redo' -f

### refresh
complete -c stg -n '__fish_seen_subcommand_from refresh' -rF -d 'Only refresh files matching path'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l conflicts -xa 'allow disallow' -d '"allow" or "disallow" pushing a patch with conflicts'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s p -l patch -kxa '(__fish_stg_patches -A -U)' -d 'Refresh (applied) <patch> instead of the top patch'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s a -l annotate -x -d 'Annotate the patch log entry with <note>'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s m -l message -x -d 'Use message for patch'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s f -l file -rF -d 'Get message from file'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s s -l signoff -x -d 'Add Signed-off-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l ack -x -d 'Add Acked-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l review -x -d 'Add Reviewed-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l author -x -d 'Set the author "name <email>"'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l authname -x -d 'Set the author name'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l authemail -x -d 'Set the author email'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l authdate -x -d 'Set the author date'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s u -l update -d 'Only update the current patch files'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s i -l index -d 'Refresh from index instead of worktree'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s F -l force -d 'Force refresh even if index is dirty'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l submodules -d 'Include submodules in patch content'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l no-submodules -d 'Exclude submodules in patch content'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s e -l edit -d 'Invoke editor for patch description'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s d -l diff -d 'Show diff when editing patch description'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l no-verify -d 'Disable commit-msg hook'
complete -c stg -n '__fish_seen_subcommand_from refresh'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from refresh' -s h -l help -d 'Print help (see more with \'--help\')'

### rename
complete -c stg -n '__fish_seen_subcommand_from rename' -x -d 'Optional old patch and the new patch name'
complete -c stg -n '__fish_seen_subcommand_from rename' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from rename'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from rename' -s h -l help -d 'Print help (see more with \'--help\')'

### repair
complete -c stg -n '__fish_seen_subcommand_from repair'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from repair' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from repair' -f

### reset
complete -c stg -n '__fish_seen_subcommand_from reset' -xa '(__fish_stg_commit)' -d 'Stack state committish'
complete -c stg -n '__fish_seen_subcommand_from reset' -kxa '(__fish_stg_patches --all)' -d 'Only reset these patches'
complete -c stg -n '__fish_seen_subcommand_from reset'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from reset'      -l hard -d 'Discard changes in the index and worktree'
complete -c stg -n '__fish_seen_subcommand_from reset' -s h -l help -d 'Print help (see more with \'--help\')'

### series
complete -c stg -n '__fish_seen_subcommand_from series' -kxa '(__fish_stg_patches --all)' -d 'Patches to display'
complete -c stg -n '__fish_seen_subcommand_from series' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from series' -s s -l short -x -d 'Select <n> patches around the topmost patch only'
complete -c stg -n '__fish_seen_subcommand_from series' -s m -l missing -x -d 'Select patches in <branch> not present in current branch'
complete -c stg -n '__fish_seen_subcommand_from series' -s i -l commit-id -x -d 'Display the commit id for each patch'
complete -c stg -n '__fish_seen_subcommand_from series'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from series' -s a -l all -d 'Select all patches, including hidden patches'
complete -c stg -n '__fish_seen_subcommand_from series' -s A -l applied -d 'Select the applied patches only'
complete -c stg -n '__fish_seen_subcommand_from series' -s U -l unapplied -d 'Select the unapplied patches only'
complete -c stg -n '__fish_seen_subcommand_from series' -s H -l hidden -d 'Select the hidden patches only'
complete -c stg -n '__fish_seen_subcommand_from series'      -l author -d 'Display author name for each patch'
complete -c stg -n '__fish_seen_subcommand_from series' -s c -l count -d 'Display the number of selected patches and exit'
complete -c stg -n '__fish_seen_subcommand_from series' -s d -l description -d 'Display short description for each patch'
complete -c stg -n '__fish_seen_subcommand_from series' -s e -l empty -d 'Display whether patches are empty'
complete -c stg -n '__fish_seen_subcommand_from series' -s P -l no-prefix -d 'Do not display the patch status prefix'
complete -c stg -n '__fish_seen_subcommand_from series' -s I -l indices -d 'Display absolute patch indicies'
complete -c stg -n '__fish_seen_subcommand_from series' -s O -l offsets -d 'Display relative offsets from topmost patch'
complete -c stg -n '__fish_seen_subcommand_from series' -s r -l reverse -d 'Display patches in reverse order'
complete -c stg -n '__fish_seen_subcommand_from series'      -l showbranch -d 'Display the branch name with the listed patches'
complete -c stg -n '__fish_seen_subcommand_from series' -s h -l help -d 'Print help (see more with \'--help\')'

### show
complete -c stg -n '__fish_seen_subcommand_from show' -kxa '(__fish_stg_patches --all)' -d 'Patches or revisions to show'
complete -c stg -n '__fish_seen_subcommand_from show' -rF -d 'Limit diff to files matching path'
complete -c stg -n '__fish_seen_subcommand_from show' -s p -l patch -kxa '(__fish_stg_patches -A -U)' -d 'Patch or revision to show'
complete -c stg -n '__fish_seen_subcommand_from show' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from show' -s O -l diff-opt -xa '(__fish_stg_git_diff_opts)' -d 'Pass additional <option> to `git diff`'
complete -c stg -n '__fish_seen_subcommand_from show'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from show' -s s -l stat -d 'Show a diffstat summary instead of the full diff'
complete -c stg -n '__fish_seen_subcommand_from show' -s A -l applied -d 'Show the applied patches'
complete -c stg -n '__fish_seen_subcommand_from show' -s U -l unapplied -d 'Show the unapplied patches'
complete -c stg -n '__fish_seen_subcommand_from show' -s H -l hidden -d 'Show the hidden patches'
complete -c stg -n '__fish_seen_subcommand_from show' -s h -l help -d 'Print help (see more with \'--help\')'

### sink
complete -c stg -n '__fish_seen_subcommand_from sink' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to sink'
complete -c stg -n '__fish_seen_subcommand_from sink' -s t -l below -l to -x -d 'Sink patches below <target> patch'
complete -c stg -n '__fish_seen_subcommand_from sink' -s T -l above -x -d 'Sink patches above <target> patch'
complete -c stg -n '__fish_seen_subcommand_from sink'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from sink' -s n -l nopush -d 'Do not push any patches back after sinking'
complete -c stg -n '__fish_seen_subcommand_from sink' -s k -l keep -d 'Keep the local changes'
complete -c stg -n '__fish_seen_subcommand_from sink'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from sink' -s h -l help -d 'Print help (see more with \'--help\')'

### spill
complete -c stg -n '__fish_seen_subcommand_from spill' -rF -d 'Only spill files matching path'
complete -c stg -n '__fish_seen_subcommand_from spill' -s a -l annotate -x -d 'Annotate the patch log entry with note'
complete -c stg -n '__fish_seen_subcommand_from spill'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from spill' -s r -l reset -d 'Also reset the index'
complete -c stg -n '__fish_seen_subcommand_from spill'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from spill' -s h -l help -d 'Print help (see more with \'--help\')'

### squash
complete -c stg -n '__fish_seen_subcommand_from squash' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to squash'
complete -c stg -n '__fish_seen_subcommand_from squash' -s n -l name -x -d 'Use <name> for the squashed patch'
complete -c stg -n '__fish_seen_subcommand_from squash' -s m -l message -x -d 'Use message for patch'
complete -c stg -n '__fish_seen_subcommand_from squash' -s f -l file -rF -d 'Get message from file'
complete -c stg -n '__fish_seen_subcommand_from squash' -s s -l signoff -x -d 'Add Signed-off-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l ack -x -d 'Add Acked-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l review -x -d 'Add Reviewed-by message trailer'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l author -x -d 'Set the author "name <email>"'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l authname -x -d 'Set the author name'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l authemail -x -d 'Set the author email'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l authdate -x -d 'Set the author date'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l save-template -rF -d 'Save the patch description to FILE and exit'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from squash' -s e -l edit -d 'Invoke editor for patch description'
complete -c stg -n '__fish_seen_subcommand_from squash' -s d -l diff -d 'Show diff when editing patch description'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l no-verify -d 'Disable commit-msg hook'
complete -c stg -n '__fish_seen_subcommand_from squash'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from squash' -s h -l help -d 'Print help (see more with \'--help\')'

### sync
complete -c stg -n '__fish_seen_subcommand_from sync' -kxa '(__fish_stg_patches -A -U)' -d 'Patches to synchronize'
complete -c stg -n '__fish_seen_subcommand_from sync' -s B -l ref-branch -xa '(__fish_stg_stg_branches)' -d 'Synchronize patches with <branch>'
complete -c stg -n '__fish_seen_subcommand_from sync' -s S -l series -rF -d 'Synchronize patches with <series>'
complete -c stg -n '__fish_seen_subcommand_from sync'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from sync' -s a -l all -d 'Synchronize all applied patches'
complete -c stg -n '__fish_seen_subcommand_from sync'      -l committer-date-is-author-date -d 'Use author date as committer date'
complete -c stg -n '__fish_seen_subcommand_from sync' -s h -l help -d 'Print help (see more with \'--help\')'

### top
complete -c stg -n '__fish_seen_subcommand_from top' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from top'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from top' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from top' -f

### uncommit
complete -c stg -n '__fish_seen_subcommand_from uncommit' -x -d 'Patch names for the uncommitted commits'
complete -c stg -n '__fish_seen_subcommand_from uncommit' -s n -l number -x -d 'Uncommit the specified number of commits'
complete -c stg -n '__fish_seen_subcommand_from uncommit' -s t -l to -x -d 'Uncommit to the specified committish'
complete -c stg -n '__fish_seen_subcommand_from uncommit'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from uncommit' -s x -l exclusive -d 'Exclude the commit specified by the \'--to\' option'
complete -c stg -n '__fish_seen_subcommand_from uncommit' -s h -l help -d 'Print help (see more with \'--help\')'

### undo
complete -c stg -n '__fish_seen_subcommand_from undo' -s n -l number -x -d 'Undo the last <n> commands'
complete -c stg -n '__fish_seen_subcommand_from undo'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from undo'      -l hard -d 'Discard changes in the index and worktree'
complete -c stg -n '__fish_seen_subcommand_from undo' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from undo' -f

### unhide
complete -c stg -n '__fish_seen_subcommand_from unhide' -kxa '(__fish_stg_patches --hidden)' -d 'Patches to unhide'
complete -c stg -n '__fish_seen_subcommand_from unhide' -s b -l branch -xa '(__fish_stg_stg_branches)' -d 'Use <branch> instead of current branch'
complete -c stg -n '__fish_seen_subcommand_from unhide'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from unhide' -s h -l help -d 'Print help (see more with \'--help\')'

### version
complete -c stg -n '__fish_seen_subcommand_from version'      -l color -xa 'auto always ansi never' -d 'When to colorize output: auto, always, ansi, never'
complete -c stg -n '__fish_seen_subcommand_from version' -s s -l short -d 'Show abbreviated version information'
complete -c stg -n '__fish_seen_subcommand_from version' -s h -l help -d 'Print help (see more with \'--help\')'
complete -c stg -n '__fish_seen_subcommand_from version' -f

### help
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'branch' -d 'Branch operations: switch, list, create, rename, delete, ...'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'clean' -d 'Delete empty patches from the series'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'commit' -d 'Finalize patches to the stack base'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'completion' -d 'Support for shell completions'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'delete' -d 'Delete patches'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'diff' -d 'Show a diff'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'edit' -d 'Edit a patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'email' -d 'Format and send patches as email'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'export' -d 'Export patches to a directory'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'files' -d 'Show files modified by a patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'float' -d 'Push patches to the top, even if applied'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'fold' -d 'Fold diff file into the current patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'goto' -d 'Go to patch by pushing or popping as necessary'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'hide' -d 'Hide patches in the series'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'id' -d 'Print git hash of a StGit revision'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'import' -d 'Import patches to stack'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'init' -d 'Initialize a StGit stack on a branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'log' -d 'Display or optionally clear the stack changelog'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'name' -d 'Print patch name of a StGit revision'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'new' -d 'Create a new patch at top of the stack'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'next' -d 'Print the name of the next patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'patches' -d 'Show patches that modify files'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'pick' -d 'Import a patch from another branch or a commit object'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'pop' -d 'Pop (unapply) one or more applied patches'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'prev' -d 'Print the name of the previous patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'pull' -d 'Pull changes from a remote repository'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'push' -d 'Push (apply) one or more unapplied patches'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'rebase' -d 'Move the stack base to another point in history'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'redo' -d 'Undo the last undo operation'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'refresh' -d 'Incorporate worktree changes into current patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'rename' -d 'Rename a patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'repair' -d 'Repair stack after branch is modified with git commands'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'reset' -d 'Reset the patch stack to an earlier state'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'series' -d 'Display the patch series'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'show' -d 'Show patch commits'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'sink' -d 'Move patches deeper in the stack'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'spill' -d 'Spill changes from the topmost patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'squash' -d 'Squash two or more patches into one'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'sync' -d 'Synchronize patches with a branch or a series'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'top' -d 'Print the name of the top patch'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'uncommit' -d 'Convert regular Git commits into StGit patches'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'undo' -d 'Undo the last command'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'unhide' -d 'Unhide hidden patches'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'version' -d 'Print version information and exit'
complete -c stg -n '__fish_seen_subcommand_from help ; and not __fish_seen_subcommand_from branch clean commit completion delete diff edit email export files float fold goto hide id import init log name new next patches pick pop prev pull push rebase redo refresh rename repair reset series show sink spill squash sync top uncommit undo unhide version help' -xa 'help' -d 'Print this message or the help of the given subcommand(s)'

### branch
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l list -d 'List branches in this repository'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l create -d 'Create and switch to a new branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l clone -d 'Clone the contents of the current branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l rename -d 'Rename an existing branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l protect -d 'Prevent StGit from modifying a branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l unprotect -d 'Allow StGit to modify a previously protected branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l delete -d 'Delete a branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l cleanup -d 'Remove StGit patch stack from branch'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and not __fish_seen_subcommand_from --list --create --clone --rename --protect --unprotect --delete --cleanup --describe' -l describe -d 'Set the branch description'

### --list
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --list' -f

### --create
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --create' -f

### --clone
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --clone' -f

### --rename
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --rename' -f

### --protect
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --protect' -f

### --unprotect
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --unprotect' -f

### --delete
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --delete' -f

### --cleanup
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --cleanup' -f

### --describe
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from branch ; and __fish_seen_subcommand_from --describe' -f

### clean
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from clean' -f

### commit
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from commit' -f

### completion
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man' -xa 'bash' -d 'Generate bash completion script'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man' -xa 'fish' -d 'Generate fish shell completion script'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man' -xa 'zsh' -d 'Generate zsh completion script'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man' -xa 'list' -d 'List StGit command information'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and not __fish_seen_subcommand_from bash fish zsh list man' -xa 'man' -d 'Generate asciidoc man pages'

### bash
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from bash' -f

### fish
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from fish' -f

### zsh
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from zsh' -f

### list
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases' -xa 'aliases' -d 'List aliases'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases' -xa 'commands' -d 'List StGit commands'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and not __fish_seen_subcommand_from aliases commands commands-and-aliases' -xa 'commands-and-aliases' -d 'List StGit commands and aliases'

### aliases
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from aliases' -f

### commands
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands' -f

### commands-and-aliases
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from list ; and __fish_seen_subcommand_from commands-and-aliases' -f

### man
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from completion ; and __fish_seen_subcommand_from man' -f

### delete
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from delete' -f

### diff
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from diff' -f

### edit
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from edit' -f

### email
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send' -xa 'format' -d 'Format patches as email files'
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from email ; and not __fish_seen_subcommand_from format send' -xa 'send' -d 'Send patches as emails'

### format
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from email ; and __fish_seen_subcommand_from format' -f

### send
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from email ; and __fish_seen_subcommand_from send' -f

### export
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from export' -f

### files
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from files' -f

### float
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from float' -f

### fold
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from fold' -f

### goto
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from goto' -f

### hide
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from hide' -f

### id
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from id' -f

### import
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from import' -f

### init
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from init' -f

### log
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from log' -f

### name
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from name' -f

### new
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from new' -f

### next
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from next' -f

### patches
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from patches' -f

### pick
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from pick' -f

### pop
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from pop' -f

### prev
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from prev' -f

### pull
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from pull' -f

### push
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from push' -f

### rebase
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from rebase' -f

### redo
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from redo' -f

### refresh
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from refresh' -f

### rename
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from rename' -f

### repair
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from repair' -f

### reset
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from reset' -f

### series
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from series' -f

### show
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from show' -f

### sink
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from sink' -f

### spill
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from spill' -f

### squash
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from squash' -f

### sync
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from sync' -f

### top
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from top' -f

### uncommit
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from uncommit' -f

### undo
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from undo' -f

### unhide
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from unhide' -f

### version
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from version' -f

### help
complete -c stg -n '__fish_seen_subcommand_from help ; and __fish_seen_subcommand_from help' -f
