//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cWorld.hpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================
#ifndef cWorld_hpp
#define cWorld_hpp
//==============================================================================
// Includes
#include <list>
#include <vector>
#include <string>
#include "cMap.hpp"
#include "Types.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace std;
namespace ShootingStar {
//------------------------------------------------------------------------------
// Forward declarations
class cObject;
class cRenderable;
class cCollidable;
class cPositionable;
//==============================================================================
	
//==============================================================================
//! Game world (map and objects) singleton
//------------------------------------------------------------------------------
class cWorld
{
	// Constructor & Destructor
	private:
		//! Constructor
		cWorld (void);
	public:
		//! Destructor
		~cWorld (void);
	
	// Public methods
	public:
		//! Return singleton
		static cWorld &GetInstance (void);
	
		//! Load map from file
		void LoadMap (string filename);
	
		//! Spawn object
		void SpawnObject (cObject *pObject);
		//! Remove object
		void RemoveObject (cObject *pObject);
		//! Remove all objects
		void RemoveAllObjects (void);
		//! Return object with matching ID
		cObject *GetObject (ObjectID ID);
	
		//! Render the world
		void Render (Uint32 deltaTime);
		//! Update the world
		void Update (Uint32 deltaTime);
	
		//! Select random position & rotation for positionable object
		void RandomPlace (cPositionable *pObject, float range=0.0f);
		bool CollidesWithWalls (cVector2f begin, cVector2f end);
	
		//! Get map size
		void GetMapSize (int &width, int &height) const
		{
			width = mMap.GetWidth ();
			height = mMap.GetHeight ();
		};
		int GetObjectCount (void) const { return mObjectList.size (); };

	// Private methods
	private:
		void CollisionDetection (Uint32 deltaTime);
		void MoveObject (cCollidable *pObject, list<cCollidable*>::iterator temp, Uint32 deltaTime);
		/*void DebugRenderWalls (void);
		void BuildSectors (void);*/
	
	// Member variables
	private:
		list<cObject*> mObjectList;	//!< List of objects in the world
		list<cRenderable*> mRenderList;		//!< List of renderable objects
		list<cCollidable*> mCollisionList;	//!< List of collidable objects
		cMap mMap;
	
		// --- Collision detection ---
		vector<tWall> mWallList;
		/*int mSectorSize;	Stuff for more advanced collision detection
		int mHSectors;
		int mVSectors;
		vector<cVector2f> mPoints;
		vector<vector <tWall> > mSectors;*/
};
//==============================================================================

//==============================================================================
}		// End of the ShootingStar namespace
#endif // cWorld_hpp
//------------------------------------------------------------------------------
// EOF
//==============================================================================
