//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cAnimationHandle.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cAnimationHandle.hpp"

#include <GL/gl.h>
#include <SDL.h>
#include "Debug.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================

//==============================================================================
//! Constructor
//------------------------------------------------------------------------------
cAnimationHandle::cAnimationHandle (void):
mAnimationBase (0),
mCurrentFrame (0),
mNumberOfFrames (0),
mFrameDelay (100),
mLastFrame (0),
mLastUpdate (0),
mAlpha (1.0f)
{
	// Empty
};
//==============================================================================

//==============================================================================
//! Destructor
//------------------------------------------------------------------------------
cAnimationHandle::~cAnimationHandle (void)
{
	// Empty
};
//==============================================================================

//==============================================================================
//! Update
//------------------------------------------------------------------------------
void 
cAnimationHandle::Update (Uint32 deltaTime)
{
	if ( mLastUpdate == 0 )
	{
		mLastUpdate = SDL_GetTicks () - deltaTime;
		mLastFrame = mLastUpdate;
	}
	
	Uint32 now = mLastUpdate + deltaTime;
	
	if ( now > mLastFrame + mFrameDelay )
	{
		Uint16 deltaFrames = (now - mLastFrame) / mFrameDelay;
		mCurrentFrame += deltaFrames;
		mLastFrame += deltaFrames * mFrameDelay;
		
		if ( mCurrentFrame >= mNumberOfFrames )
			mCurrentFrame %= mNumberOfFrames;
	}
	mLastUpdate = now;
}
//==============================================================================

//==============================================================================
//! Render
//------------------------------------------------------------------------------
void 
cAnimationHandle::Render (void)
{
	dbg::assertion (dbg::error, DBG_ASSERTION (mAnimationBase != 0));
	dbg::assertion (dbg::error, DBG_ASSERTION (mCurrentFrame < mNumberOfFrames));
	
	glPushAttrib (GL_CURRENT_BIT);

	glColor4f (1.0f, 1.0f, 1.0f, mAlpha);
	glCallList (mAnimationBase + mCurrentFrame);

	glPopAttrib ();
}
//==============================================================================

//==============================================================================
// EOF
//==============================================================================
