using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Linq;
using System.Threading;
using IntervalTree;
using Microsoft.Extensions.Logging;
using OmniSharp.Extensions.LanguageServer.Protocol;
using OmniSharp.Extensions.LanguageServer.Protocol.Models;
using Range = OmniSharp.Extensions.LanguageServer.Protocol.Models.Range;

namespace Microsoft.Dafny.LanguageServer.Workspace;

public class SymbolTable {

  public static SymbolTable CreateFrom(ILogger logger, Program program, CancellationToken cancellationToken) {
    var visited = program.Visit(a => true, b => { });

    var references = visited.OfType<IHasReferences>()
      .SelectMany(r => {
        if (AutoGeneratedOrigin.Is(r.Origin)) {
          return [];
        }
        var usages = r.GetReferences();
        if (usages == null) {
          logger.LogError($"Value of type {r.GetType()} returned a null for ${nameof(IHasReferences.GetReferences)}");
          usages = Array.Empty<Reference>();
        }
        return usages.Where(d => d.Referred != null);
      }).ToList();

    var definitions = visited.OfType<IHasNavigationToken>();

    return new SymbolTable(references, definitions.ToList());
  }

  public static SymbolTable Empty() {
    return new SymbolTable();
  }

  private SymbolTable() {
    NodeToReferences = ImmutableDictionary<TokenRange, ISet<TokenRange>>.Empty;
    ReferenceToNode = ImmutableDictionary<TokenRange, IHasNavigationToken>.Empty;
    Symbols = ImmutableList<ISymbol>.Empty;
    Declarations = ImmutableDictionary<TokenRange, ISymbol>.Empty;
  }

  private SymbolTable(IReadOnlyList<Reference> references, IReadOnlyList<IHasNavigationToken> hasNavigationTokens) {
    var safeReferences = references.Where(k => k.Referer.Uri != null).ToImmutableList();

    var referencesAsKeyValuePairs = safeReferences.Select(k => KeyValuePair.Create(k.Referer, k.Referred));
    ReferenceToNode = referencesAsKeyValuePairs.DistinctBy(pair => pair.Key).ToImmutableDictionary();

    NodeToReferences = safeReferences.GroupBy(u => u.Referred.NavigationRange).ToImmutableDictionary(
      g => g.Key,
      g => (ISet<TokenRange>)g.Select(k => k.Referer).ToHashSet());

    Declarations = hasNavigationTokens.OfType<ISymbol>().Where(h => !AutoGeneratedOrigin.Is(h.Origin)
        && (h is not AliasModuleDecl alias /* see AliasModuleDecl.NavigationToken for explanation */ || alias.HasAlias)).
      DistinctBy(h => h.NavigationRange).
      ToImmutableDictionary(h => h.NavigationRange, h => h);

    var relevantDafnySymbolKinds = new HashSet<SymbolKind> {
      SymbolKind.Function,
      SymbolKind.Class,
      SymbolKind.Enum,
      SymbolKind.Method,
      SymbolKind.EnumMember,
      SymbolKind.Struct,
      SymbolKind.Interface,
      SymbolKind.Namespace,
    };
    // Since these definitions are checked for whether they
    // contain substrings when answering workspace/resolve queries,
    // performance can be improved by storing their names in a
    // data structure that makes this operation cheaper, such as
    // a suffix tree.
    Symbols = hasNavigationTokens.OfType<ISymbol>().
      Where(symbol => symbol.Kind.HasValue && relevantDafnySymbolKinds.Contains(symbol.Kind.Value)).
      ToImmutableList();

    var symbols = hasNavigationTokens.Where(s => !AutoGeneratedOrigin.Is(s.Origin))
      .Select(h => h.NavigationRange)
      .Concat(safeReferences.Select(r => r.Referer))
      .Concat(safeReferences.Select(r => r.Referred.NavigationRange))
      //.Where(s => !AutoGeneratedOrigin.Is(s))
      .Distinct();
    var symbolsByFile = symbols.GroupBy(s => s.Uri);
    foreach (var symbolsForFile in symbolsByFile) {
      if (symbolsForFile.Key == null) {
        continue;
      }
      var nodePositionsForFile = new IntervalTree<Position, TokenRange>();
      navigationRanges.Add(symbolsForFile.Key, nodePositionsForFile);
      foreach (var symbolForFile in symbolsForFile) {
        var range = symbolForFile.ToLspRange();
        nodePositionsForFile.Add(range.Start, range.End, symbolForFile);
      }
    }
  }

  private readonly Dictionary<Uri, IIntervalTree<Position, TokenRange>> navigationRanges = new();

  /// <summary>
  /// Maps each symbol declaration to itself, and each symbol usage to the symbol's declaration.
  /// </summary>
  public ImmutableDictionary<TokenRange, IHasNavigationToken> ReferenceToNode { get; }
  public ImmutableDictionary<TokenRange, ISymbol> Declarations { get; }

  /// <summary>
  /// Maps each symbol declaration to usages of the symbol, not including the declaration itself.
  /// </summary>
  public ImmutableDictionary<TokenRange, ISet<TokenRange>> NodeToReferences { get; }

  /// <summary>
  ///  A list of all definitions, such as methods, classes, functions, etc., used for workspace-wide symbol
  /// lookup.
  /// </summary>
  public ImmutableList<ISymbol> Symbols { get; }

  public IEnumerable<Location> GetReferences(Uri uri, Position position) {
    if (navigationRanges.TryGetValue(uri, out var rangesForFile)) {
      return rangesForFile.Query(position).
        SelectMany(node => NodeToReferences.GetOrDefault(node, () => (ISet<TokenRange>)new HashSet<TokenRange>())).
        Select(u => new Location { Uri = u.Uri, Range = u.ToLspRange() }).Distinct();
    }
    return [];
  }

  public Location? GetDeclaration(Uri uri, Position position) {
    var node = GetDeclarationNode(uri, position);
    return node == null ? null : NodeToLocation(node);
  }

  internal static Location? NodeToLocation(IHasNavigationToken node) {
    var origin = node.NavigationRange;
    if (origin.Uri == null) {
      return null;
    }
    return new Location {
      Uri = DocumentUri.From(origin.Uri),
      Range = origin.ToLspRange()
    };
  }

  public IHasNavigationToken? GetDeclarationNode(Uri uri, Position position) {
    if (!navigationRanges.TryGetValue(uri, out var forFile)) {
      return null;
    }

    var character = new Range(position, new Position(position.Line, position.Character + 1));
    var options = forFile.Query(position)
      .Where(n => n.ToLspRange().Contains(character)).OrderByDescending(o => o.Length);
    return options
      .Select(node => Declarations.TryGetValue(node, out var declaration)
        ? declaration : ReferenceToNode.GetValueOrDefault(node))
      .FirstOrDefault(x => x != null);
  }
}
