package com.fasterxml.jackson.dataformat.javaprop;

import java.io.*;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Map;

import org.junit.jupiter.api.Test;

import com.fasterxml.jackson.core.*;
import com.fasterxml.jackson.core.exc.StreamReadException;
import com.fasterxml.jackson.core.io.SerializedString;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.javaprop.io.JPropWriteContext;

import static org.junit.jupiter.api.Assertions.*;

public class SimpleStreamingTest extends ModuleTestBase
{
    private final ObjectMapper MAPPER = newPropertiesMapper();

    private final JavaPropsFactory F = new JavaPropsFactory();

    @Test
    public void testParsing() throws Exception
    {
        JsonParser p = F.createParser("foo = bar");
        Object src = p.getInputSource();
        assertTrue(src instanceof Reader);
        _verifyGetNumberTypeFail(p, "null");
        assertToken(JsonToken.START_OBJECT, p.nextToken());
        assertNull(p.getEmbeddedObject());
        assertNotNull(p.currentLocation()); // N/A
        assertNotNull(p.currentTokenLocation()); // N/A
        _verifyGetNumberTypeFail(p, "START_OBJECT");
        assertToken(JsonToken.FIELD_NAME, p.nextToken());
        assertEquals("foo", p.currentName());
        assertEquals("foo", p.getText());
        assertEquals("foo", p.getValueAsString());
        assertEquals("foo", p.getValueAsString("x"));
        assertToken(JsonToken.VALUE_STRING, p.nextToken());
        StringWriter sw = new StringWriter();
        assertEquals(3, p.getText(sw));
        assertEquals("bar", sw.toString());
        try {
            // try just one arbitrary one; all should fail similarly
            p.getLongValue();
            fail("Should not pass");
        } catch (StreamReadException e) {
            _verifyNonNumberTypeException(e, "VALUE_STRING");
        }

        p.close();
        assertTrue(p.isClosed());
        _verifyGetNumberTypeFail(p, "null");

        // one more thing, verify handling of non-binary
        p = F.createParser("foo = bar");
        assertToken(JsonToken.START_OBJECT, p.nextToken());
        assertToken(JsonToken.FIELD_NAME, p.nextToken());
        try {
            p.getBinaryValue();
            fail("Should not pass");
        } catch (JsonProcessingException e) {
            verifyException(e, "can not access as binary");
        }
        try {
            p.getDoubleValue();
            fail("Should not pass");
        } catch (JsonProcessingException e) {
            _verifyNonNumberTypeException(e, "FIELD_NAME");
        }
        assertToken(JsonToken.VALUE_STRING, p.nextToken());
        assertToken(JsonToken.END_OBJECT, p.nextToken());
        assertFalse(p.isClosed());
        assertNull(p.nextToken());
        p.close();
    }

    @Test
    public void testStreamingGeneration() throws Exception
    {
        StringWriter strw = new StringWriter();
        JsonGenerator gen = F.createGenerator(strw);

        Object target = gen.getOutputTarget();
        assertTrue(target instanceof Writer);
        
        gen.writeStartObject();
        gen.writeBooleanField("flagTrue", true);
        gen.writeBooleanField("flagFalse", false);
        gen.writeNullField("null");
        gen.writeNumberField("long", 10L);
        gen.writeNumberField("int", 10);
        gen.writeNumberField("double", 0.25);
        gen.writeNumberField("float", 0.5f);
        gen.writeNumberField("decimal", BigDecimal.valueOf(0.125));
        gen.writeFieldName(new SerializedString("bigInt"));
        gen.writeNumber(BigInteger.valueOf(123));
        gen.writeFieldName("numString");
        gen.writeNumber("123.0");
        gen.writeFieldName("charString");
        gen.writeString(new char[] { 'a', 'b', 'c' }, 1, 2);

        gen.writeFieldName("arr");
        gen.writeStartArray();

        JsonStreamContext ctxt = gen.getOutputContext();
        String path = ctxt.toString();
        assertTrue(ctxt instanceof JPropWriteContext);
        // Note: this context gives full path, unlike many others
        assertEquals("/arr/0", path);
        
        gen.writeEndArray();

        gen.writeEndObject();
        assertFalse(gen.isClosed());
        gen.flush();
        gen.close();

        String props = strw.toString();

        // Plus read back for fun
        Map<?,?> stuff = MAPPER.readValue(props, Map.class);
        assertEquals(11, stuff.size());
        assertEquals("10", stuff.get("long"));
    }

    @Test
    public void testStreamingGenerationRaw() throws Exception
    {
        StringWriter strw = new StringWriter();
        JsonGenerator gen = F.createGenerator(strw);

        String COMMENT = "# comment!\n";
        gen.writeRaw(COMMENT);
        gen.writeRaw(new SerializedString(COMMENT));
        gen.writeRaw(COMMENT, 0, COMMENT.length());
        gen.writeRaw('#');
        gen.writeRaw('\n');

        gen.writeStartObject();
        gen.writeBooleanField("enabled", true);
        gen.writeEndObject();
        
        gen.close();

        assertEquals(COMMENT + COMMENT + COMMENT
                + "#\nenabled=true\n", strw.toString());

        // Plus read back for fun
        Map<?,?> stuff = MAPPER.readValue(strw.toString(), Map.class);
        assertEquals(1, stuff.size());
        assertEquals("true", stuff.get("enabled"));
    }        

    @Test
    public void testStreamingLongRaw() throws Exception
    {
        StringWriter strw = new StringWriter();
        JsonGenerator gen = F.createGenerator(strw);

        StringBuilder sb = new StringBuilder();
        sb.append("# ");
        for (int i = 0; i < 12000; ++i) {
            sb.append('a');
        }
        gen.writeRaw(sb.toString());
        gen.close();

        assertEquals(sb.toString(), strw.toString());
    }

    // In Jackson 2.x, non-Number token should throw exception for parser.getNumberType()
    private void _verifyGetNumberTypeFail(JsonParser p, String token) throws Exception
    {
        try {
            p.getNumberType();
            fail("Should not pass");
        } catch (StreamReadException e) {
            _verifyNonNumberTypeException(e, token);
        }
    }

    private void _verifyNonNumberTypeException(Exception e, String token) throws Exception
    {
        verifyException(e, "Current token ("+token+") not numeric, can not use numeric");
    }
}
