/*  rspoint.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "rspoint.h"
#include "sector.h"
#include "functions.h"
#include "yamlutils.h"
#include "kinematics.h"
#include "files.h"
#include "int.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

// register type
namespace {
YAMLProxy<RSGenericSelection> dummy1;
YAMLProxy<RSFiniteGenericSelection> dummy2;
YAMLProxy<IdentityGenericSelection> dummy3;
YAMLProxy<ReductionGenericOptions> dummy4;
}

RSPoint::RSPoint() :
	r_(0), s_(0) {
}

RSPoint::RSPoint(int r, int s) :
	r_(r), s_(s) {
}

int RSPoint::r() const {
	return r_;
}

int RSPoint::s() const {
	return s_;
}

void RSGenericSelection::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << Flow;
	os << BeginMap;
	os << Key << "r" << Value //
			<< BeginSeq << r_min_ << r_max_ << EndSeq;
	os << Key << "s" << Value //
			<< BeginSeq << s_min_ << s_max_ << EndSeq;
	os << EndMap;
}

void RSGenericSelection::read(const YAML::Node& n) {
	verify_yaml_spec(n);
	GiNaC::lst symbols;
	symbols.append(t_symbol_);
	Reduze::read(n["r"][0], r_min_, symbols);
	Reduze::read(n["r"][1], r_max_, symbols);
	Reduze::read(n["s"][0], s_min_, symbols);
	Reduze::read(n["s"][1], s_max_, symbols);
}

bool RSGenericSelection::is_finite() const {
	return r_min(0) >= 0 && r_max(0) >= 0 //
			&& s_min(0) >= 0 && r_max(0) >= 0;
}
bool RSGenericSelection::is_empty() const {
	return r_min_ == 0 && r_max_ == 0 && s_min_ == 0 && s_max_ == 0;
}

int RSGenericSelection::get_explicite_value(const GiNaC::ex& e, int t) const {
	using namespace GiNaC;
	exmap subst;
	subst[t_symbol_] = t;
	ex es = e.subs(subst);
	if (!is_a<numeric> (es)) {
		YAML::Emitter ye;
		print(ye);
		ABORT("Invalid generic rs range specification: " << ye.c_str());
	}
	return ex_to<numeric> (es).to_int();
}

int RSGenericSelection::r_min(int t) const {
	return get_explicite_value(r_min_, t);
}

int RSGenericSelection::r_max(int t) const {
	return get_explicite_value(r_max_, t);
}

int RSGenericSelection::s_min(int t) const {
	return get_explicite_value(s_min_, t);
}

int RSGenericSelection::s_max(int t) const {
	return get_explicite_value(s_max_, t);
}

pair<ex, ex> RSGenericSelection::r_bounds() const {
	return make_pair(r_min_, r_max_);
}

pair<ex, ex> RSGenericSelection::s_bounds() const {
	return make_pair(s_min_, s_max_);
}

void RSGenericSelection::set_r_bounds(const GiNaC::ex& rmin,
		const GiNaC::ex& rmax) {
	r_min_ = rmin;
	r_max_ = rmax;
}

void RSGenericSelection::set_s_bounds(const GiNaC::ex& smin,
		const GiNaC::ex& smax) {
	s_min_ = smin;
	s_max_ = smax;
}

bool RSGenericSelection::contains(int t, const INT& i) const {
	int r = i.r();
	int s = i.s();
	if (r_min(t) >= 0 && r < r_min(t))
		return false;
	if (r_max(t) >= 0 && r > r_max(t))
		return false;
	if (s_min(t) >= 0 && s < s_min(t))
		return false;
	if (s_max(t) >= 0 && s > s_max(t))
		return false;
	return true;
}

const RSGenericSelection RSGenericSelection::full_range;

// RSFiniteGenericSelection

RSFiniteGenericSelection::RSFiniteGenericSelection(const RSGenericSelection& r) :
	RSGenericSelection(r) {
	init();
}

void RSFiniteGenericSelection::init() {
	if (!is_finite()) {
		YAML::Emitter ye;
		print(ye);
		throw std::runtime_error(string("Range is not finite:\n") + ye.c_str());
	}
}

void RSFiniteGenericSelection::read(const YAML::Node& n) {
	RSGenericSelection::read(n);
	init();
}

void RSFiniteGenericSelection::find_points(int t, std::set<RSPoint>& points) const {
	for (int r = r_min(t); r <= r_max(t); ++r)
		for (int s = s_min(t); s <= s_max(t); ++s)
			points.insert(RSPoint(r, s));
}

void RSFiniteGenericSelection::set_r_bounds(const GiNaC::ex& rmin,
		const GiNaC::ex& rmax) {
	pair<ex, ex> old = r_bounds();
	RSGenericSelection::set_r_bounds(rmin, rmax);
	if (!is_finite()) {
		RSGenericSelection::set_r_bounds(old.first, old.second);
		throw runtime_error("infinite r bounds for RSFiniteGenericSelection");
	}
}

void RSFiniteGenericSelection::set_s_bounds(const GiNaC::ex& smin,
		const GiNaC::ex& smax) {
	pair<ex, ex> old = s_bounds();
	RSGenericSelection::set_s_bounds(smin, smax);
	if (!is_finite()) {
		RSGenericSelection::set_s_bounds(old.first, old.second);
		throw runtime_error("infinite s bounds for RSFiniteGenericSelection");
	}
}

void RSFiniteGenericSelection::find_upper_border(const std::list<
		RSFiniteGenericSelection>& ranges, int t, std::set<RSPoint>& border) {
	std::set<RSPoint> all;
	std::list<RSFiniteGenericSelection>::const_iterator r;
	for (r = ranges.begin(); r != ranges.end(); ++r)
		r->find_points(t, all);
	border.clear();
	LOGX("Checking " << all.size() << " rs-points for border");
	std::set<RSPoint>::const_iterator p;
	for (p = all.begin(); p != all.end(); ++p) {
		int r = p->r();
		int s = p->s();
		if (/**/all.find(RSPoint(r, s + 1)) == all.end() || //
				all.find(RSPoint(r + 1, s + 1)) == all.end() || //
				all.find(RSPoint(r + 1, s)) == all.end()) {
			LOGX("  on border: " << r << " " << s);
			border.insert(*p);
		}
	}
}

long int RSFiniteGenericSelection::num_integrals(int n, int t) const {
	int rmin = r_min(t);
	int rmax = r_max(t);
	int smin = s_min(t);
	int smax = s_max(t);
	long int res = 0;
	for (int r = rmin; r <= rmax; ++r)
		for (int s = smin; s <= smax; ++s)
			res += Sector::get_num_integrals_static(n, t, r, s);
	return res;
}

template<class R>
bool rsranges_contain(const list<R>& l, int t, const INT& i) {
	for (typename list<R>::const_iterator r = l.begin(); r != l.end(); ++r)
		if (r->contains(t, i))
			return true;
	return false;
}

template
bool rsranges_contain<RSGenericSelection> (const list<RSGenericSelection>&,
		int, const INT&);

template
bool rsranges_contain<RSFiniteGenericSelection> (const list<
		RSFiniteGenericSelection>&, int, const INT&);

// IdentityGenericSelection

void IdentityGenericSelection::read(const YAML::Node& n) {
	verify_yaml_spec(n);
	if (n.FindValue("ibp"))
		n["ibp"] >> identities_["ibp"];
	if (n.FindValue("ibp_dim"))
		n["ibp_dim"] >> identities_["ibp_dim"];
	if (n.FindValue("ibp_dim_free"))
		n["ibp_dim_free"] >> identities_["ibp_dim_free"];
	if (n.FindValue("ibp_lee"))
		n["ibp_lee"] >> identities_["ibp_lee"];
	if (n.FindValue("lorentz"))
		n["lorentz"] >> identities_["lorentz"];
	if (n.FindValue("sector_relations"))
		n["sector_relations"] >> identities_["sector_relations"];
	if (n.FindValue("sector_symmetries"))
		n["sector_symmetries"] >> identities_["sector_symmetries"];
}

void IdentityGenericSelection::print(YAML::Emitter& os) const {
	using namespace YAML;
	map<string, list<RSFiniteGenericSelection> >::const_iterator id;
	os << BeginMap;
	for (id = identities_.begin(); id != identities_.end(); ++id)
		os << Key << id->first << Value << id->second;
	os << EndMap;
}

bool IdentityGenericSelection::empty() const {
	std::map<std::string, std::list<RSFiniteGenericSelection> >::const_iterator
			r;
	for (r = identities_.begin(); r != identities_.end(); ++r) {
		if (!r->second.empty())
			return false;
	}
	return true;
}

void IdentityGenericSelection::init() {
	list<RSFiniteGenericSelection> r;
	identities_["ibp"] = r;
	identities_["ibp_dim"] = r;
	identities_["ibp_dim_free"] = r;
	identities_["ibp_lee"] = r;
	identities_["lorentz"] = r;
	identities_["sector_relations"] = r;
	identities_["sector_symmetries"] = r;
}

void ReductionGenericOptions::read(const YAML::Node& n) {
	verify_yaml_spec(n);
	if (n.FindValue("requested_solutions"))
		n["requested_solutions"] >> requested_solutions_;
	if (n.FindValue("set_subsectors_to_zero"))
		n["set_subsectors_to_zero"] >> set_subsectors_to_zero_;
	if (n.FindValue("map_on_coeff")) {
		lst l = Files::instance()->kinematics()->kinematic_invariants_and_dimension();
		Reduze::read(n["map_on_coeff"], map_on_coeff_, l);
	}
	if (n.FindValue("discard_new_subsector_identities"))
		n["discard_new_subsector_identities"]
				>> discard_new_subsector_identities_;
}

void ReductionGenericOptions::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;
	os << Key << "requested_solutions" << Value << requested_solutions_;
	os << Key << "set_subsectors_to_zero" << Value << set_subsectors_to_zero_;
	os << Key << "map_on_coeff" << Value << Flow << map_on_coeff_;
	os << Key << "discard_new_subsector_identities" << Value
			<< discard_new_subsector_identities_;
	os << EndMap;
}

} // namespace Reduze
