/*  feynmanrules.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_FEYNMANRULES_H_
#define REDUZE_FEYNMANRULES_H_

#include "yamlconfigurable.h"

namespace Reduze {

// wrapper functions to read the user defined Feynman rules

// color
DECLARE_FUNCTION_2P (delta_a)
DECLARE_FUNCTION_2P (delta_f)
DECLARE_FUNCTION_3P (ColT)
DECLARE_FUNCTION_3P (ColF3)
DECLARE_FUNCTION_4P (ColF4)

// external legs
DECLARE_FUNCTION_3P (U)
DECLARE_FUNCTION_3P (Ubar)
DECLARE_FUNCTION_3P (V)
DECLARE_FUNCTION_3P (Vbar)

DECLARE_FUNCTION_5P (Pol_gluon)
DECLARE_FUNCTION_5P (PolC_gluon)
DECLARE_FUNCTION_3P (Pol_ghost)
DECLARE_FUNCTION_3P (PolC_ghost)
DECLARE_FUNCTION_3P (Pol_photon)
DECLARE_FUNCTION_3P (PolC_photon)

// minkowski
DECLARE_FUNCTION_2P (metric_l)
DECLARE_FUNCTION_4P (metric_l_diff)

/// indexed four vector
DECLARE_FUNCTION_2P (Mom)

// gamma matrices
DECLARE_FUNCTION_1P (GaOne)
DECLARE_FUNCTION_1P (Ga)
DECLARE_FUNCTION_1P (GaS)

class Spinor;

/// flags
class IndexFlags {
public:
	enum {
		lorentz = 1, color_a, color_f, undef
	};
};

/// stores the user defined Feynman rules and manages the dummy symbols
class FeynmanRules: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("feynmanrules");
		s.set_short_description("" //
						"Definition of the Feynman rules.");
		s.set_long_description("" //
						"Definition of the Feynamn rules for external legs,"
						" propagators, vertices and the template parameters"
						" (masses, momenta, indices) which are used in the Feynman"
						" rules and later replaced by their actual value when they"
						" are inserted into a diagram. The coupling constants and"
						" su(N) dimensions (Nc, Na) should be defined in the"
						" file \"global.yaml\", see the online help for "
						" \"global_symbols\". The Feynamn rules must be"
						" given as a map of a keyword and a value. The value may"
						" contain any expression which is known to GiNaC as well as"
						" severasyntax error while parsing a LinearCombinationl wrapper functions defined by Reduze. For a list of"
						" the possible functions we refer to the QED/QCD-Feynman"
						" rules file which is included in the source code package"
						" of Reduze. The Feynman rules are expressions for an ordered"
						" list of fields which we number from 1 to n. Any template"
						" parameter, which has been defined properly as a prefix,"
						" refers to the i-th field if one appends the integer suffix"
						" 'i' to it.");
		s.add_option("coupling_constants", false, "sequence of strings", "" //
						"For backward compatibility: definition of the coupling"
						" constant symbols. If defined here they will be used"
						" instead of the values defined in the file \"global.yaml\"."
						" See the online help for \"global_symbols\".");
		s.add_option("su_n_dimensions", false, "2-element map", "" //
						"For backward compatibility: definition of the dimensions"
						" of the fundamental and adjoint representation of su(N)."
						" If defined here they will be used instead of the values defined"
						" in the file \"global.yaml\". See the online help for"
						" \"global_symbols\".");
		s.add_option("momentum_prefix", false, "string", "" //
						"Momentum prefix (lower case letters, no numbers).");
		s.add_option("mass_prefix", false, "string", "" //
						"Mass prefix (lower case letters, no numbers).");
		s.add_option("polarization_orthogonal_vector_prefix", false, "string",
				"Orthogonal polarization vector prefix (lower case letters, no numbers).");
		s.add_option("index_prefixes", false, "map", "" //
						"Prefixes of indices (lorentz, color)  (lower case letters, no numbers).");
		s.add_option("max_vertex_degree", false, "integer", "" //
						"Maximal number of fields at a vertex."
						" Used to generate template parameters from the prefixes."
						" Must be greater than 2.");

		s.add_option("external_legs", false, "map", "" //
						"Definition of the rules for the incoming"
						" and outgoing legs.");
		s.add_option("propagators", true, "map", "" //
						"Definition of the rules for the propagators.");
		s.add_option("vertices", true, "map", "" //
						"Definition of the rules for the vertices.");
		s.add_option("polarization_sums", false, "sequence", "" //
						"Definition of the rules for the polarization sums.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	FeynmanRules();
	virtual ~FeynmanRules();

	virtual void read(const YAML::Node& n);
	virtual void print(YAML::Emitter& os) const;

	/// returns the user defined Feynman rule to the identifier 'id'
	const GiNaC::ex& rule(const std::string& id) const;
	/// get a number for the id
	int num_by_id(const std::string& id) const;
	/// get the polarization sums
	GiNaC::ex
	get_polarization_sum(const GiNaC::ex& g, const GiNaC::ex& gc) const;
	/// get the dummy symbol of momenta nr. 'i'
	GiNaC::symbol mom(int i) const;
	/// get the dummy symbol of mass nr. 'i'
	GiNaC::symbol mass(int i) const;
	/// get the dummy symbol of vector orthogonal to polarization for nr. 'i'
	GiNaC::symbol polarization_orthogonal_vector(int i) const;
	/// get the dummy symbol of index 's''i'
	GiNaC::symbol index(const std::string& s, int i) const;
	/// get the index prefixes
	const std::map<int, std::string>& index_prefixes() const;
	/// get the maximum number of fields at a vertex
	int max_vertex_degree() const;

	/// insert a new Feynman rule (string) with id 'id' and FeynmanRulesType 'type',
	// the symbols must have been set before
	void insert_rule(const std::string& id, int type, const std::string& rule);
	/// insert a new polarization sum for external leg functions 'pol' * 'polc'
	// the symbols must have been set before
	void insert_polarization_sum(const std::string& pol,
			const std::string& polc, const std::string& val);

private:
	/// Feynman rules by the id
	std::map<std::string, GiNaC::ex> rules_;
	/// momentum prefix
	std::string momentum_prefix_;
	/// mass prefix
	std::string mass_prefix_;
	/// polarization orthogonal vector prefix
	std::string polarization_orthogonal_vector_prefix_;
	/// index prefixes by type
	std::map<int, std::string> index_prefixes_;
	/// max number of fields at a vertex, used for generating symbols
	int max_vertex_degree_;
	/// a number for each idenitfier
	std::map<std::string, int> num_by_id_;

private:

	// polarization sums

	/// the parsed polarization sums
	struct parsed_ps {
		parsed_ps() :
				pol_(0), polc_(0), val_(0) {
		}
		parsed_ps(const GiNaC::ex& p, const GiNaC::ex& pc, const GiNaC::ex& v) :
				pol_(p), polc_(pc), val_(v) {
		}
		GiNaC::ex pol_, polc_, val_;
		GiNaC::ex mass() const;
		GiNaC::ex raw_momentum() const;
		GiNaC::ex raw_ortho() const; // zero for non-gluons
	};
	/// the unparsed polarization sums
	struct unparsed_ps {
		std::string pol_, polc_, val_;
	};
	/// unparsed polarization sums (for printing)
	std::map<std::string, unparsed_ps> unparsed_polarization_sums_;
	/// the polarization sums indexed by user defined keyword
	std::map<std::string, parsed_ps> polarization_sums_;
	/// symbols for the polarization sum
	// contains mass, momentum, ortho, the index prefixes and the indices with ending 1 and 2
	GiNaC::lst ps_symbols_;
	void insert_ps(const std::string& type, const parsed_ps& ps);
	const parsed_ps& ps(const std::string& type) const;

	/// list containing the symbols for the Feynman rules (indices, momenta, masses, ortho)
	// contains the symbols created by the prefix and and integer suffix
	GiNaC::lst symbols_;
	/// map of the symbols by name (indices, momenta, masses, ortho)
	std::map<std::string, GiNaC::symbol> symbols_by_name_;

	/// initializes the template symbols
	void init_symbols();
	/// insert the Feynman rule (expression)
	void insert_rule(const std::string& id, int type, const GiNaC::ex& rule);
	/// access to the local symbol with name 's'
	GiNaC::symbol local_symbol(const std::string& s) const;
	/// check valid external leg functions
	void assert_allowed_external_function(const GiNaC::ex& e) const;

	struct FeynmanRulesType {
		enum {
			external, propagator, vertex, undef
		};
	};
	/// type (external, propagator, vertex) of Feynman rule by the id
	std::map<std::string, int> rules_type_;
	/// the unparsed rules by the id (needed for printing)
	std::map<std::string, std::string> unparsed_rules_;
};

inline int FeynmanRules::max_vertex_degree() const {
	return max_vertex_degree_;
}
inline const std::map<int, std::string>& FeynmanRules::index_prefixes() const {
	return index_prefixes_;
}

inline void operator>>(const YAML::Node& n, FeynmanRules& r) {
	r.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye, const FeynmanRules& r) {
	r.print(ye);
	return ye;
}

} // namespace Reduze

#endif /* REDUZE_FEYNMANRULES_H_ */
