// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { MimeData } from '@lumino/coreutils';
/**
 * The clipboard interface.
 */
export var Clipboard;
(function (Clipboard) {
    /**
     * Get the application clipboard instance.
     */
    function getInstance() {
        return Private.instance;
    }
    Clipboard.getInstance = getInstance;
    /**
     * Set the application clipboard instance.
     */
    function setInstance(value) {
        Private.instance = value;
    }
    Clipboard.setInstance = setInstance;
    /**
     * Copy text to the system clipboard.
     *
     * #### Notes
     * This can only be called in response to a user input event.
     */
    function copyToSystem(clipboardData) {
        const node = document.body;
        const handler = (event) => {
            const data = event.clipboardData || window.clipboardData;
            if (typeof clipboardData === 'string') {
                data.setData('text', clipboardData);
            }
            else {
                clipboardData.types().map((mimeType) => {
                    data.setData(mimeType, clipboardData.getData(mimeType));
                });
            }
            event.preventDefault();
            node.removeEventListener('copy', handler);
        };
        node.addEventListener('copy', handler);
        generateEvent(node);
    }
    Clipboard.copyToSystem = copyToSystem;
    /**
     * Generate a clipboard event on a node.
     *
     * @param node - The element on which to generate the event.
     *
     * @param type - The type of event to generate.
     *   `'paste'` events cannot be programmatically generated.
     *
     * #### Notes
     * This can only be called in response to a user input event.
     */
    function generateEvent(node, type = 'copy') {
        // http://stackoverflow.com/a/5210367
        // Identify selected text.
        let sel = window.getSelection();
        // Save the current selection.
        const savedRanges = [];
        for (let i = 0, len = (sel === null || sel === void 0 ? void 0 : sel.rangeCount) || 0; i < len; ++i) {
            savedRanges[i] = sel.getRangeAt(i).cloneRange();
        }
        // Select the node content.
        const range = document.createRange();
        range.selectNodeContents(node);
        if (sel) {
            sel.removeAllRanges();
            sel.addRange(range);
        }
        // Execute the command.
        document.execCommand(type);
        // Restore the previous selection.
        sel = window.getSelection();
        if (sel) {
            sel.removeAllRanges();
            for (let i = 0, len = savedRanges.length; i < len; ++i) {
                sel.addRange(savedRanges[i]);
            }
        }
    }
    Clipboard.generateEvent = generateEvent;
})(Clipboard || (Clipboard = {}));
/**
 * The namespace for module private data.
 */
var Private;
(function (Private) {
    /**
     * The application clipboard instance.
     */
    Private.instance = new MimeData();
})(Private || (Private = {}));
//# sourceMappingURL=clipboard.js.map