# $Id: cmsa.awk,v 1.5 1992/03/06 00:17:13 nesheim Exp $
#############################################################################
# 	  Copyright (c) 1991 Thinking Machines Corporation, Inc.,	     
#		of Cambridge, Mass.   All rights reserved.		     
#									     
#  This notice is intended as a precaution against inadvertent publication   
#  and does not constitute an admission or acknowledgement that publication  
#  has occurred or constitute a waiver of confidentiality.		     
#									     
#  Connection Machine software is the proprietary and confidential property  
#  of Thinking Machines Corporation.					     
#############################################################################
#
# Generate a CM2 Accounting summary
#
# This `awk' script post processes the output of the cmsa command,
# which is a preprocessor for CM accounting records.
#
# cmsa writes out a line for each `job' it finds in the CM accounting file, in
# the format we specify on the command line, using ascii strings and
# keywords flagged by the % character.
#    
# This sample script is written to be able to handle the output of cmsa
# when run on multiple accounting files, potentially from more than one
# front-end, and will produce a suitable summary of the information
# gathered.
#
# The system administrator is encouraged to modify this script as she sees
# fit, to perform site specific accounting or resource monitoring
# functions.
#
# To produce a report, run the following pipeline:
#
# cmsa %id.pid %id.euid %id.egid %cm.interface %cm.uccs %cm.flags \
#    %cm.connected %cm.ucc_busy_ticks \
#    %io.p_kb_read %io.p_kb_written %io.s_kb_read %io.s_kb_written \
#    %ts.memory_integral  %cm.cm_name %cm.memory_size %cm.processors \
#    %cm.rusage.ru_utime %cm.rusage.ru_stime \
#    | awk -f summary.awk    
#
#
##########################################################################
#
# cmsa as run per the above command line will produce the following input
# fields for AWK to process:
#
# Input fields:
#    1       2        3          4           5        6       
# %id.pid %id.euid %id.egid %cm.interface %cm.uccs %cm.flags \
#	  7		 8
#   %cm.connected %cm.ucc_busy_ticks \
#           9		  10              11             12
#   %io.p_kb_read %io.p_kb_written %io.s_kb_read %io.s_kb_written \
#	     13		    14	         15	        16
#   %ts.memory_integral %cm.cm_name %cm.memory_size %cm.processors \
#            17               18
#   %cm.rusage.ru_utime %cm.rusage.ru_stime
#
#
##########################################################################
#
BEGIN {
      Jobs=0;
      MemorySize = 256;
      NInterfaces = 4;
      First = 1000000000;
      Last=0;
      #
      # Set this to 1 if you want a summary of all jobs by interface, rather
      # than just by CM.
      SummaryByInterface=0;
      }

##########################################################################
# Deal with the header summary stuff
##########################################################################
#
/^CM Accounting summary:/{
    Nrec = $4;
    Njobs = $6;
    next;
}
/^First:/{
    if (First > $2) {
	First=$2;
	FirstString=$3 " " $4 " " $5 " " $6 ;
    }
    next;
    }
/^Last:/{
    if (Last < $2) {
	Last=$2;
	LastString=$3 " " $4 " " $5 " " $6 ;
    }
    next;
    }
/^Oldest:/{
    if (Oldest < $2) {
	Oldest=$2;
	OldestString=$3 " " $4 " " $5 " " $6 ;
    }
    next;
    }
/^Highest numbered interface:/{
    NInterfaces = $4 + 1;
    next;
}


##########################################################################
#
# Here's the real work.  Deal with the job records
#
##########################################################################
{
    if (UserName[$2] != $2) {
	UserName[$2] = $2;
	Uids[NUids++] = $2;
    }
    if (GroupName[$3] != $3) {
	GroupName[$3] = $3;
	Gids[NGids++] = $3;
    }
    
    if (CMName[$14] != $14) {
	CMName[$14] = $14;
	AllCMNames[NCMs++] = $14;
    }

    for (i = 0; i < NCMs; i++) {
	if (AllCMNames[i] == $14){
	    CMNameIndex=i;
	    break;
	}
    }

	
    if ($5 == 0) Nseq = 0;
    else if ($5 == 1 || $5 == 2 || $5 == 4 || $5 == 8) Nseq = 1;
    else if ($5 == 3 || $5 == 12) Nseq = 2;
    else Nseq = 4;

    if ($6 == 0) {
# 	Exclusive mode job
#	accumulate connect time by "sequencers * seconds"	
#
	ExclConnectByUser[$2] += $7 * Nseq;
	ExclConnectByGroup[$3] += $7 * Nseq;
	ix = $5 + ($4 * 16)
	ExclConnectByIf[ix] += $7;
	ExclJobsByIf[ix]++;
	ix = $5 + (CMNameIndex * 16);
	ExclConnectByCM[ix] += $7;
	ExclJobsByCM[ix] ++;

	ExclJobsByUser[$2]++;
	ExclJobsByGroup[$2]++;
	NExclJobs++;
    } else {
#
# 	Timesharing job
#
	TSConnectByUser[$2] += $7 * Nseq;
	TSConnectByGroup[$3] += $7 * Nseq;
	ix = $5 + ($4 * 16)
	TSConnectByIf[ix] += $7;
	TSJobsByIf[ix] ++;

	ix = $5 + (CMNameIndex * 16);
	TSConnectByCM[ix] += $7;
	TSJobsByCM[ix] ++;

	TSJobsByUser[$2]++;
	TSJobsByGroup[$2]++;

	TSMemByUser[$2] += $13;
	TSMemByGroup[$3] += $13;
	NTSJobs++;
    }

    PIOByUser[$2] += ($9 + $10);
    PIOByGroup[$3] += ($9 + $10);
    SIOByUser[$2] += ( $11 + $12);
    SIOByGroup[$3] += ( $11 + $12);
    FEByUser[$2] += ($17 + $18);
    FEByGroup[$3] += ($17 + $18);
}
##########################################################################
#
# Now print out a summary
#
##########################################################################
END { 
    Epoch = Last - First;

    seqIx[0] = 1; seqIx[1] = 2; seqIx[2] = 4; seqIx[3] = 8;
    seqIx[4] = 3; seqIx[5] = 12; seqIx[6] = 15;

    printf("%d accounting records processed\n", NTSJobs + NExclJobs);
    printf("The oldest process started %s.\n", OldestString);
    printf("The first attach recorded was %s.\n", FirstString);
    printf("The last detach recorded was %s.\n", LastString);

    if (SummaryByInterface) {
	printf("\nCM usage statistics %s through %s.\n\n", FirstString, LastString);
	if (NTSJobs > 0) {
	    printf("Timesharing jobs for each interface and sequencer\n");
	    printf("%d timeshared jobs run\n", NTSJobs);
	    printf("\n"); 
	    printf(" Seq.      0        1         2         3        0-1       2-3      0-3\n");
	    for (i = 0; i < NInterfaces; i++) {
		printf("       ---------+---------+---------+---------+---------+---------+---------\n");
		printf("I/F %d: ", i);
		for (j = 0; j < 7; j++) {
		    ix = seqIx[j] + (i * 16);
		    if (TSJobsByIf[ix] > 0)
			printf("%3d:%02d:%02d ", \
			       TSConnectByIf[ix]/3600, \
			       (TSConnectByIf[ix]%3600)/60, \
				   TSConnectByIf[ix]%60);
		    else
			printf("          ");
	    }
	    
	    printf("\n       ");
	    for (j = 0; j < 7; j++) {
		ix = seqIx[j] + (i * 16);
		if (TSJobsByIf[ix] > 0)
		    printf("%9d ", TSJobsByIf[ix]);
		else
		    printf("          ");
	    }
	    
	    printf("\n");
	    }
	    printf("       ---------+---------+---------+---------+---------+---------+---------\n");
	    printf("\n\n"); 
	}
	if (NExclJobs > 0) {
	    printf("Exclusive mode jobs for each interface and sequencer\n");
	    printf("%d exclusive mode jobs run\n", NExclJobs);
	    printf("\n"); 
	    printf(" Seq.      0        1         2         3        0-1       2-3      0-3\n");
	    for (i = 0; i < NInterfaces; i++) {
		printf("       ---------+---------+---------+---------+---------+---------+---------\n");
		printf("I/F %d: ", i);
		for (j = 0; j < 7; j++) {
		    ix = seqIx[j] + (i * 16);
		    if (ExclJobsByIf[ix] > 0)
			printf("%3d:%02d:%02d ", \
			       ExclConnectByIf[ix]/3600, \
			       (ExclConnectByIf[ix]%3600)/60, \
				   ExclConnectByIf[ix]%60);
		    else
			printf("          ");
		}
		printf("\n");
		
		printf("       ");
		for (j = 0; j < 7; j++) {
		    ix = seqIx[j] + (i * 16);
		    if (ExclJobsByIf[ix] > 0)
			printf("%9d ", ExclJobsByIf[ix]);
		    else
			printf("          ");
		}
	    
		printf("\n");
	    }
	    printf("       ---------+---------+---------+---------+---------+---------+---------\n");
	    printf("\n\n"); 
	}
    }

    for (i = 0; i < NCMs; i++) {
	name = AllCMNames[i];
	printf("\n\"%s\" usage summary for %d hour period \n\t%s through %s\n", \
	       name, Epoch/3600, FirstString, LastString);
	printf("  Sequencer        0         1         2         3     \n");
	printf("               ---------+---------+---------+---------+\n");
	printf("Exclusive:     ", i);
	for (j = 0; j < 4; j++) {
	    ix = seqIx[j] + (i * 16);
	    # Sum is usage for this sequencer, plus usage for whole
	    # machine plus usage for two sequencer combination including
	    # this sequencer.
 	    sum = ExclConnectByCM[ix] + ExclConnectByCM[15 + (i * 16)];
 	    if (j == 0 || j == 1)
 		sum += ExclConnectByCM[3 + (i * 16)];
 	    if (j == 2 || j == 3)
 		sum += ExclConnectByCM[12 + (i * 16)];

	    if (sum > 0)
		    printf("%3d:%02d:%02d ", \
			   sum/3600, \
			   (sum%3600)/60, \
			   sum%60);
	    else
		printf("          ");
	    ExclTot[j] = sum;
	}
	printf("\nTimeshared:    ", i);
	for (j = 0; j < 4; j++) {
	    ix = seqIx[j] + (i * 16);
	    sum = TSConnectByCM[ix] + TSConnectByCM[15 + (i * 16)];
 	    if (j == 0 || j == 1)
 		sum += TSConnectByCM[3 + (i * 16)];
 	    if (j == 2 || j == 3)
 		sum += TSConnectByCM[12 + (i * 16)];

	    if (sum > 0)
		    printf("%3d:%02d:%02d ", \
			   sum/3600, \
			   (sum%3600)/60, \
			   sum%60);
	    else
		printf("          ");
	    TSTot[j] = sum;
	}
	printf("\n               ---------+---------+---------+---------+\n");
	printf("Total:         ", i);
	for (j = 0; j < 4; j++) {
	    ix = seqIx[j] + (i * 16);
	    sum = ExclTot[j] + TSTot[j];
	    if (sum > 0)
		    printf("%3d:%02d:%02d ", \
			   sum/3600, \
			   (sum%3600)/60, \
			   sum%60);
	    else
		printf("          ");
	}
	printf("\n");
	printf("Usage:         ");
	for (j = 0; j < 4; j++) {
	    ix = seqIx[j] + (i * 16);
	    sum = ExclTot[j] + TSTot[j];
	    printf("%8.0f%% ", (sum / Epoch)*100.0);
	}
	printf("\n");
	
    }


    printf("\n\n");
    printf("Usage summary by user and group id.\n");
    printf(" Connect times in Sequencers*Seconds,\n");
    printf(" I/O in kilobytes\n\n");
    printf("User        Exclusive  Timeshared  FE time   TS mem.  IO (par.)  IO (ser.)\n");
    printf("--------    ---------  ----------  --------  -------  ---------  ---------\n");
    for(i = 0; i < NUids; i++) {
	name = Uids[i]
        printf("%-10.10s %10.2f  %10.2f %9.2f %8.2e  %9.2e  %9.2e\n", \
	       name, ExclConnectByUser[name], TSConnectByUser[name], \
	       FEByUser[name], TSMemByUser[name], PIOByUser[name], SIOByUser[name]);
    }

    printf("\n\n");
    printf("Group       Exclusive  Timeshared  FE time   TS mem.  IO (par.)  IO (ser.)\n");
    printf("--------    ---------  ----------  --------  -------  ---------  ---------\n");
    for(i = 0; i < NGids; i++) {
	name = Gids[i]
        printf("%-10.10s %10.2f  %10.2f %9.2f %8.2e  %9.2e  %9.2e\n", \
	       name, ExclConnectByGroup[name], TSConnectByGroup[name], \
	       FEByGroup[name], TSMemByGroup[name], PIOByGroup[name], SIOByGroup[name]);
    }

}






