;;; -*- Base: 10; Syntax: Common-Lisp; Package: cmi; Patch-File: Yes; -*- 

;;;
;;; Some misc diag patches.
;;; 12/4/91 10:59:40 nesheim
;;;

(in-package 'cmi)
;;;
;;; manufacturing-system-test hangs in sleep 
;;; (inside wait-for-power-to-stabilize) on Lucid systems.  
;;; I don't understand what's happening, but this gets around it.
;;;
#+lucid
(defun wait-for-power-to-stabilize (&optional (n-seconds 1)) 
  (loop for i below (* n-seconds 3500000)))

;;;
;;; Call set-rev-3-nexus-clock with crystal number
;;; instead of mhz speed.  Works on both cm2 and cm200 correctly.
;;;
(defun system-test-set-frequency-and-power (old-info-string pass-count cycle-frequency cycle-power)
  (declare (special *info-string* *diag-errors*))
  (when cycle-frequency
    (ignore-errors
      (progn
	(set-rev-3-nexus-clock (mod (1- pass-count) 3))))	;; 1, 2, 0
	;; In case there's a glitch
	(initialize-febi-and-ucc))

  (when cycle-power
    (ignore-errors
      (progn
	;; Cycle power if requested.
	(if cycle-frequency
	    (when (= 1 (mod pass-count 3))
	      (case (mod (floor pass-count 3) 3)
		(2 (normal-operation))
		(0 (low-margin-operation))
		(1 (high-margin-operation))))
	    (case (mod pass-count 3)
	      (0 (normal-operation))
	      (1 (low-margin-operation))
	      (2 (high-margin-operation))))
	;; In case there's a glitch
	(my-cm-cold-boot))))

  (setq *info-string* (format nil "~a~a~a~aPass ~d "
			      old-info-string
			      (if *diag-errors*
				  (format nil "~d. ERRORS " (total-diag-errors))
				  "")
			      (format nil "CLOCK-~d " (ldb *clock-control* (read-nexus-register-3)))
			      (case *diag-current-power-status* 
				(:normal "NRML ")
				(:low "LOW  ")
				(:high "HIGH "))
			      pass-count)))

(defmacro do-for-each-frequency (&body body)
  `(loop for frequency below 3
	 do
      (progn
	(set-rev-3-nexus-clock frequency)
	(progn ,.body))))

(defun Manufacturing-System-Test
       (&key
	(n-passes     nil)		;; If NUMBERP, then do N-PASSES
	(loop-forever (not (integerp n-passes)))
	(boot-p t)
	(cycle-frequency            t)
	(cycle-power                nil)
	(abort-after-n-total-errors 300)  ;; an integer or nil
	(log-enabled t)
	(stop-on-error nil)
	(print-all-errors t)
	(ucc-tests t)
	(cs-tests nil)
	(memory-test t)
	(memory-test-serial nil)
	(chip-test t)
	(cube-test t)
	(news-test t)
	(error-test t)
	(matrix-rug-test t)
	(sprint-chip-test t)	;; NIL ==> t
	(fpu-chip-test t)
	(float-random-test t)
	(router-component-test t)
	(router-test t)
	(router-verifiers t)
	(float-verifiers t)
	(io-tests t)
	(error-cutoff 49.)		;; Fence post error (want 50.)
	(router-tests-error-cutoff 16.)
	)

  ;; NOTE: This assumes that microcode is loaded before calling SYSTEM-TEST.
  ;;(test-power-on)				;; Make sure we're connected to UCC
  ;; This may be required in the future.
  ;;(test-init)					;; Make sure it' sane.
  (clear-diag-errors)
  (initialize-febi-and-ucc)
  (clear-chip-histogram)			;; Clear out any NRT errors
  (clear-all-diag-test-statistics)		;; Clear out the stats.
  ;; Setup for the SPRINT tests.
  (setq *sprint-diag-environment* *sprint-hardware-environment*)

  ;; Instead of doing this, do a fast COLD-BOOT later.
  ;;(cm-init)

  (let ((*current-blink-led-mode* :diagnostics)	;; BOOT Constant
	(*delivery-error-on-timeout-p* nil)		;; BOOT Constant
	(*cm-error-running-diags* t)
	(*use-the-old-error-mechanism* nil)
	(old-info-string *info-string*)
	(*info-string* *info-string*)
	(*diag-pass-count* 0)
	(*diag-log-enabled* log-enabled)
	(*diag-halt-on-error*  stop-on-error)
	(*diag-stop-on-error*  nil)
	(*diag-break-on-error* stop-on-error)
	(*diag-loop-on-error* nil)
	(*diag-print-all-errors* print-all-errors)
	(*diag-error-cutoff* error-cutoff)
	(*router-tester-max-error-messages* router-tests-error-cutoff)
	(start-chip *physical-number-of-chips-offset*)
	(nchips *physical-number-of-chips-limit*)
	(nboards (floor *physical-number-of-chips-limit* *chips-per-board*))
	(time1   (get-universal-time))
	(time2   (get-universal-time))
	(log-start-and-stop log-enabled)
	(result nil)
	(passed nil))
    (declare (special *current-blink-led-mode* *delivery-error-on-timeout-p*
		      *router-tester-max-error-messages* *info-string*
		      *diag-pass-count* *cm-error-running-diags*
		      *use-the-old-error-mechanism*))

    ;; Instead of a CM-INIT, do a fast COLD-BOOT here.
    ;; If this is the first COLD-BOOT, then the START-CHIP and NCHIPS args
    ;; are incorrect. Unless the caller specified them, change to the correct
    ;; values after the COLD-BOOT.
    (when boot-p
      (my-cm-cold-boot 1)
      (setq start-chip *physical-number-of-chips-offset*)
      (setq nchips *physical-number-of-chips-limit*)
      (setq nboards (floor *physical-number-of-chips-limit* *chips-per-board*)))

    (multiple-value-bind (bogus-number-of-boards bogus-start-chip name)	;; START-CHIP is WRONG for anything less than pint.
	(diag-size-of-machine *size-of-machine*)
      (declare (ignore bogus-number-of-boards bogus-start-chip))
      (setq *diag-board-being-tested* name))

    (when (null *sprint-chip-p*)
      (setq *sprint-chip-p* :sprint))
    (setq old-info-string
	  (format nil "~a~a~s "
		  old-info-string
		  (if (= nboards 1)
		      "Board "
		      (format nil "~dK " (floor nboards 2)))
		  *sprint-chip-p*))

    (with-lispm-setup-for-diagnostics
      (unwind-protect
	  (prog ()
		(setq passed nil)
		(setq time1 (get-universal-time))
		(when log-start-and-stop
		  (DiagLogMessage "~%Starting System-Test of ~a~@
                                   Machine Size: ~a~@
                                   Front    End: ~a~@
                                   Microcode Version: ~a~@
                                   SPRINT Configuration: ~a"
				  (cm-descriptor-name *hardware-configuration*)
				  *size-of-machine*	;; ***
				  (machine-instance)
				  (microcode-version-number *microcode-release-version*)
				  *sprint-chip-p*))
	     loop
		;; Moved from inside (WHEN ...
		(incf *diag-pass-count*)		;; Keeps serial memory test from running when not-LOOP-FOREVER.

		;;(setq *info-string* (format nil "~aPass ~d " old-info-string *diag-pass-count*))
		(system-test-set-frequency-and-power old-info-string *diag-pass-count* cycle-frequency cycle-power)
		  
		(when loop-forever
		  (format t "~%;[~a] Pass ~d ~:[~;(errors being logged to editor buffer)~]"
			  (print-universal-time (get-universal-time) nil)
			  *diag-pass-count* (eq *diag-log-enabled* :editor)))
		(when log-enabled
		  (DiagLogMessage ";;;[~a] ~a"
				  (print-universal-time (get-universal-time) nil)
				  *info-string*))
;		(when log-enabled
;		  (DiagLogMessage ";[~a] Pass ~d" (print-universal-time (get-universal-time) nil) *diag-pass-count*))
		  
		(if (okay-to-run-test-p ucc-tests *diag-pass-count*)
		    (enable-diag-test 'test-uc)
		    (disable-diag-test 'test-uc))
		  
		;; This replaces TEST-WRITE-READ-PARALLEL
		(if (okay-to-run-test-p memory-test *diag-pass-count*)
		    (enable-diag-test 'test-cm-memory)
		    (disable-diag-test 'test-cm-memory))

		(if (okay-to-run-test-p memory-test-serial *diag-pass-count*)
		    (enable-diag-test 'memory-test-serial)
		    (disable-diag-test 'memory-test-serial))

		;; Replace the next group with a single DEFUN
		(if (okay-to-run-test-p chip-test *diag-pass-count*)
		    (enable-diag-test 'system-chip-tests)
		    (disable-diag-test 'system-chip-tests))

		;; Error tests.
		(if (okay-to-run-test-p error-test *diag-pass-count*)
		    (enable-diag-test 'system-error-test)
		    (disable-diag-test 'system-error-test))
		  
		;; MATRIX Board tests.
		(if (okay-to-run-test-p matrix-rug-test *diag-pass-count*)
		    (enable-diag-test 'matrix-rug-tester)
		    (disable-diag-test 'matrix-rug-tester))

		;; Cube tests.
		(if (okay-to-run-test-p cube-test *diag-pass-count*)
		    (enable-diag-test 'system-cube-tests)
		    (disable-diag-test 'system-cube-tests))

		;; News tests.
		(if (okay-to-run-test-p news-test *diag-pass-count*)
		    (enable-diag-test 'system-news-tests)
		    (disable-diag-test 'system-news-tests))
		  
		;; Router Component tests.
		  
		(if (okay-to-run-test-p router-component-test *diag-pass-count*)
		    (enable-diag-test 'test-router-components)
		    (disable-diag-test 'test-router-components))
		  
		;; SPRINT and FPU tests.
		  
		(if (okay-to-run-test-p sprint-chip-test *diag-pass-count*)
		    (enable-diag-test 'sprint-chip-test)
		    (disable-diag-test 'sprint-chip-test))

		(if (and (or (eql *sprint-chip-p* :wtl3132)
			     (eql *sprint-chip-p* :wtl3164))
			 (okay-to-run-test-p fpu-chip-test *diag-pass-count*))
		    (progn
		      (if (eql *sprint-chip-p* :wtl3132)
			  (enable-diag-test 'wtl3132-test)
			  (disable-diag-test 'wtl3132-test))
		      (if (eql *sprint-chip-p* :wtl3164)
			  (enable-diag-test 'wtl3164-test)
			  (disable-diag-test 'wtl3164-test)))
		    (progn
		      (disable-diag-test 'wtl3132-test)
		      (disable-diag-test 'wtl3164-test)))

		(if (and (or (eql *sprint-chip-p* :wtl3132)
			     (eql *sprint-chip-p* :wtl3164))
			 (okay-to-run-test-p float-random-test *diag-pass-count*))
		    (progn
		      (if (eql *sprint-chip-p* :wtl3132)
			  (enable-diag-test 'wtl3132-random-tests)
			  (disable-diag-test 'wtl3132-random-tests))
		      (if (eql *sprint-chip-p* :wtl3164)
			  (enable-diag-test 'wtl3164-random-tests)
			  (disable-diag-test 'wtl3164-random-tests)))
		    (progn
		      (disable-diag-test 'wtl3132-random-tests)
		      (disable-diag-test 'wtl3164-random-tests)))

		;; Full system Router tests.
		  
		;; In case the router ever decides to use sprint, run the
		;; router tests after SPRINT and WEITEK
		(if (okay-to-run-test-p router-test *diag-pass-count*)
		    (enable-diag-test 'new-router-tester)
		    (disable-diag-test 'new-router-tester))

		;; I/O tests (when available).
		(if (okay-to-run-test-p io-tests *diag-pass-count*)
		    (enable-diag-test 'test-cm-io)
		    (disable-diag-test 'test-cm-io))

		;; Router Verifiers
		(if (and (fboundp 'run-router-verifiers)	;; Make sure they're loaded
			 (okay-to-run-test-p router-verifiers *diag-pass-count*))
		    (enable-diag-test 'run-router-verifiers)
		    (disable-diag-test 'run-router-verifiers))

		;; Float Verifiers
		(if (and (or (eql *sprint-chip-p* :wtl3132)
			     (eql *sprint-chip-p* :wtl3164))
			 (fboundp 'float-torture-test)	;; Make sure they're loaded
			 (okay-to-run-test-p float-verifiers *diag-pass-count*))
		    (enable-diag-test 'float-torture-test)
		    (disable-diag-test 'float-torture-test))

		(let ((*cs-tests* (not cs-tests))
		      (*vp-list* (if (eql *sprint-chip-p* :wtl3164)
				     (list 2 4 8)
				     (list 1 2 4)))
		      (*use-double* (if (eql *sprint-chip-p* :wtl3164) t nil))
		      (*start-chip* start-chip)
		      (*nchips* nchips)
		      (*nboards* nboards))
		  (declare (special *cs-tests* *vp-list* *use-double* *start-chip* *nchips* *nboards*))
		  (setq result (run-diag-test-1 *manufacturing-system-tests*)))

		;; Set variable to indicate finished this pass of the test.
		(setq passed t)
		  
		;; If N-PASSES is a number, then keep looping for N-PASSES
		;; If LOOP-FOREVER, then loop.
		(when (and (or loop-forever
			       (and (integerp n-passes)
				    (< *diag-pass-count* n-passes)))
			   (or (not abort-after-n-total-errors)
			       (< (total-diag-errors) abort-after-n-total-errors)))
		  (go loop))
		)
	  
	(setq time2 (get-universal-time))

	(when cycle-frequency
	  (let* 
	      ((cm200-p (memq :cm200 (cm-descriptor-property-list *hardware-configuration*)))
	       (clock-field (cm-descriptor-clock-speed *hardware-configuration*))
	       (clock-number
		(if cm200-p
		    (selectq clock-field
			     ((:8-mhz :8.0-mhz) 0)
			     ((:10-mhz :10.0-mhz) 1)
			     (:10.5-mhz 2)
			     (:external 3))
		    (selectq clock-field
			     ((:4-MHZ :7-mhz :7.0-MHZ) 0);; To be compatable with REV-2
			     ((:8.0-MHZ :8-mhz)  1)
			     (:8.5-MHZ  2)
			     (:EXTERNAL 3)
			     ))))

	    (format t "~&Resetting ~a NEXUS Clock to ~a (CLOCK-~d) ... " 
		    (if cm200-p "CM200" "CM2") clock-field clock-number)
	    (with-febi-errors-suppressed 
	      (set-rev-3-nexus-clock clock-field))
	    (format t "Done.")))
	(when cycle-power
	  (format t "~&Reseting POWER to NORMAL ... ")
	  (normal-operation)
	  (format t "Done."))

	(when log-start-and-stop
	  (DiagLogMessage "End of System-Test. Total run time: ~a"
			  #+symbolics (time:seconds-to-interval-string (- time2 time1))
			  #-symbolics (- time2 time1))
	  (when (eq *diag-log-enabled* :editor)
	    (format t "~%~%*** NOTE: Errors have been logged into an editor buffer. ***"))
	  (error-summary *diag-pass-count* (- time2 time1) passed)
	  )

	(print-diag-errors)))
    (null result)))

(cmi::increment-patch-level 14)
