;;; -*- Base: 10; Syntax: Common-Lisp; Package: cmi; Patch-File: Yes; -*- 

;;;
;;; Nexus clock handling, works only for *rev3*
;;; 11/18/91 14:57:54 nesheim
;;;

(in-package 'cmi)

(defun set-rev-3-nexus-clock (value-or-keyword)
  (if (numberp value-or-keyword)
      (write-nexus-register-3 (dpb value-or-keyword *clock-control* #x00007FFF))
      (let* ((cm200-p (memq :cm200 (cm-descriptor-property-list *hardware-configuration*)))
	     (clock-field
	      (if cm200-p
		  (selectq value-or-keyword
			   ((:8-mhz :8.0-mhz) 0)
			   ((:10-mhz :10.0-mhz) 1)
			   (:10.5-mhz 2)
			   (:external 3)
			   (otherwise 
			     (error "~s is not a valid value for the CM200 clock speed" 
				    value-or-keyword)))
		  (selectq value-or-keyword
			   ((:4-MHZ :7-mhz :7.0-MHZ) 0);; To be compatable with REV-2
			   ((:8.0-MHZ :8-mhz)  1)
			   (:8.5-MHZ  2)
			   (:EXTERNAL 3)
			   (otherwise 0)))))
	(write-nexus-register-3 (dpb clock-field *clock-control* #x00007FFF)))))

(defun get-rev-3-nexus-clock (&OPTIONAL (reg-3 (read-nexus-register-3)) (crystal-p nil))
  (nth (ldb *clock-control* reg-3)
       (if (and (not crystal-p)
		(and (boundp '*hardware-configuration*)
		     *hardware-configuration*))
	   (if (memq :cm200 (cm-descriptor-property-list *hardware-configuration*))
	       '(:8-Mhz :10-Mhz :10.5-Mhz :external)
	       '(:7-Mhz :8-Mhz :8.5-Mhz :external))
	   '(:crystal-0 :crystal-1 :crystal-2 :external))))


(defun get-nexus-clock (&OPTIONAL raw-status)
  (declare (ignore raw-status))
  (get-rev-3-nexus-clock))

;;;
;;; I hate back compatability.
;;; We should really find all the places that call this and 
;;; make them *not* do it anymore.
;;;
(defun set-nexus-clock (&optional clock)
  (declare (ignore clock)))


(defun print-nexus-status-register 
    (&OPTIONAL
     (stream t)
     (raw-status (with-febi-errors-suppressed (READ-NEXUS-STATUS)))
     (reg-3 (with-febi-errors-suppressed
		    (READ-NEXUS-REGISTER-3)))
     (print-return-before-list nil))

  (format stream "~%Nexus Status for Front End ~a (Configured Front End Number ~d): (#x~8,'0x)~
                      ~%  Diagnostic Mode is ~:[Off~;On~], NEXUS Error is ~:[Off~;On~]~
                      ~%  Clock Rate ~a, Front End ID ~d, Access Mode ~d, Access ID ~d ~
                      ~%  Read ports (For each Front End, which SEQUENCER is read): ~a~a~
                      ~%  Write ports (For each SEQUENCER, which Front End writes): ~a~a"
	  (if (and (boundp '*hardware-configuration*)
		   *hardware-configuration*)
	      (nth (cm-descriptor-nexus-host-port *hardware-configuration*)
		   (cm-descriptor-host-list *hardware-configuration*))
	      "Unknown")
	  (if (and (boundp '*hardware-configuration*)
		   *hardware-configuration*)
	      (cm-descriptor-nexus-host-port *hardware-configuration*)
	      "Unknown")
	  (ldb (byte 32 0) raw-status)
	  (ldb-test *diagnostic-mode* raw-status);; Diagnostics Mode
	  (ldb-test *channel-error* raw-status);; NEXUS Error
	  (get-rev-3-nexus-clock reg-3 t)
	  (ldb *host-id* raw-status);; Host
	  (ldb *register-subfield* raw-status);; Access Mode
	  (ldb *host-subfield* raw-status);; Access ID
	  (if print-return-before-list #.(format nil "~%             ") "");; Just to keep the formatted output short.
	  (get-nexus-host-ports-complete raw-status)
	  (if print-return-before-list #.(format nil "~%              ") "");; Just to keep the formatted output short.
	  (get-nexus-ucc-ports raw-status))
  )

(defun CM:Power-Up (&key (slowly nil))
  "Sets up the nexus with an initial state.  This should only be done on power up 
   because other users of the CM will be interrupted if this occurs"
  
  (when (and (not (febi-attached-p))
             (not *Selected-CM*)
             (not *Selected-Interface*)
             (> (length *Possible-CMs*) 1))
    (select-cm :ask))
  
  
  (update-configuration :get-current-ucc-ports nil :LOAD-CONFIGURATION-FILE :IF-NOT-CURRENT)
  
  (let* ((host-list (delete-duplicates (mapcar #'(lambda (h)
                                                   (if (atom h) h (car h)))
                                               (cm-descriptor-host-list
                                                 *hardware-configuration*))))
         (n-hosts (loop for h in host-list
			when h
			sum 1))
         (febi-was-already-attached-p (febi-attached-p)))
    (when (or (<= n-hosts 1)
              (yes-or-no-p "This will detach anyone else using Connection Machine ~a.~@
	      Do you really want to do this? "
			   (cm-descriptor-name *hardware-configuration*)))
      (unwind-protect
	   (progn
	     (when (not febi-was-already-attached-p)
	       (attach-febi))
	     (initialize-febi :enable-error-traps nil)
	     #+symbolics
	     (PROCESS-RUN-FUNCTION
	       'NOTIFY-USERS-OF-POWER-UP
	       #'(LAMBDA ()
		   (dolist (host host-list)
		     (when (and host (not (eq (si:parse-host host) si:local-host)))
		       (ignore-errors
			 (chaos:notify
			   host
			   (format nil
				   "~a is powering up the Connection Machine.~@
                                   You are not attached to a CM.~@
                                   You will have to do another CM:ATTACH"
				   (machine-instance))))))))

	    ;; Make sure that the FEBI has errors disabled.
	     (initialize-febi :enable-error-traps nil)

	    ;; Clear the nexus status register
	     (write-nexus-status (if *rev-3-nexus* #x40000000 0))

	    ;; Initilize the REV 2 NEXUS shadow registers.
	     (when *rev-3-nexus*
	       (write-nexus-status 0)
	       (write-nexus-register-1 #x00000000)	;; Clear general register.
	       (write-nexus-register-2 #xC0000000)	;; NVRAM in power-down state.
	       (write-nexus-register-3 #x00007FFF)	;; Clear and disable all errors.
	      ;; Set the NEXUS clock from the configuration file.
	       (let* ((cm200-p (memq :cm200 
				     (cm-descriptor-property-list *hardware-configuration*)))
		      (requested-speed (cm-descriptor-clock-speed *hardware-configuration*))
		      (clock-field
		       (if cm200-p
			   (selectq requested-speed
				    (:8-mhz 0)
				    (:10-mhz 1)
				    (:10.5-mhz 2)
				    (:external 3)
				    (otherwise 
				      (progn
					(warn "~s is not a valid value for the CM200 clock speed" 
					     requested-speed)
					0)))
			   (selectq requested-speed
				    ((:4-MHZ :7-mhz :7.0-MHZ) 0);; To be compatable with REV-2
				    ((:8.0-MHZ :8-mhz)  1)
				    (:8.5-MHZ  2)
				    (:EXTERNAL 3)
				    (otherwise 
				      (progn
					(warn "~s is not a valid value for the CM200 clock speed" 
					     requested-speed)
					0))))))
		 (write-nexus-register-3 (dpb clock-field *clock-control* #x00007FFF)))
	       )

	     (if slowly
		 (initialize-nexus-slowly)
		 (initialize-nexus)))
	;;dont turn on errors on the hbi until we are booted.
	;; Cleanup:
        (when (not febi-was-already-attached-p)
          (detach-febi)))))
  (make-accounting-log-entry :CM_HARDWARE "Power-up; interface ~d" *Attached-interface*)
  T)



(cmi::increment-patch-level 10)
