/*	gw.c	1.1	11/11/84	*/

/*
 * Applebus / ethernet gateway.
 *
 * (C) 1984, Stanford Univ. SUMEX project.
 * May be used but not sold without permission.
 *
 * history
 * 11/11/84	Croft	Created.
 */

#include "gw.h"
#include "pbuf.h"
#include "atalk.h"
#include "ether.h"
#include "inet.h"


/*
 * Interface configuration.
 */

int	iloutput(),ilmatch(),aboutput(),abmatch();

struct ifnet ifil = {
	"il", 0, 			/* intel/interlan, unit 0 */
	0x242c0040, 0x800,		/* 36.44.0.64, type 0x800 */
	6, 1,				/* ether addr is 6 bytes, format 1*/
	iloutput, ilmatch
};

struct ifnet ifab = {
	"ab", 0,			/* applebus, unit 0 */
	0x241000fe, 0x800,		/* 36.16.0.254, type 0x800 */
	1, 3,				/* ab addr is 1 byte, format 3*/
	aboutput, abmatch
};


iaddr_t	iproutedef = 0x242c0020;	/* default route, 36.44.00.32 */

short	ipsubmask = 0xFF;		/* mask for subnet (0 if no subnets) */
short	ipsubshift = 16;		/* shift to subnet field */

iaddr_t	ipnet;				/* major IP net number */
short	ipid;				/* next IP id number */



/*
 * DDP to IP address mapping table.  At Stanford, this table is empty
 * because we use subnets for address conversion.  
 *
 * If you are not using subnets, this table is used by the routines
 * 'ddp2ip' and 'ip2ddp' to map from a ddp net/node number to an IP
 * address and vice versa.  (Even if you are using subnets, this table
 * is still checked, so you could for example, have a line that turns
 * a dnet/dnode into an arpanet address).
 *
 * The table is searched mainly by 'ddp2ip' in the code that wraps
 * DDPs in IP/UDP headers.  The table has three fields, DDP net #, 
 * DDP node #, and IP address (net # or complete address).  If
 * the 'dnode' field is '-1' (an impossible value), then the table
 * entry maps an entire DDP net, to an IP net.  In that case the 'inet'
 * field must consist of only an IP net number.  The ddp2ip(dnet,dnode)
 * call will find that IP net number, and OR into it the 'dnode' argument.
 *
 * If the 'dnode' table field is not negative, then the dnet/dnode pair
 * indicates a single applebus host, and 'inet' must be a complete
 * IP net/host address.
 */
struct ipmap {
	short	dnet;			/* ddp net number */
	short	dnode;			/* ddp node number */
	iaddr_t	inet;			/* ip net number or net/host */
} *ip2ddp(), ipmap[] = {
	-1, 0, 0			/* end of table */
};
/*
 * Here is a sample ipmap table that maps all of DDP net 14 and 15 to
 * IP nets 126 and 127.  It also maps ONE DDP address (net 13, node 1)
 * into ONE IP address (bbn-unix, 10.0.0.82, an address on the ARPANET
 * backbone).  
 * NOTE:  If you don't have subnets, you MUST have a line for the
 * DDP net number / IP net number that corresponds to your applebus
 * interface;  that is how the startup code finds its DDP net number.
 *
 *	14, -1,		0x7E000000,
 *	15, -1,		0x7F000000,
 *	13, 1,		0x0A000052,
 */

	

/*
 * Main event loop.
 */
main()
{
	register struct pbuf *p;
	register struct ifnet *ifp;
	struct ipmap *im;
	int s;

	printf("applebus/ethernet gateway\n");
	asminit();			/* initialize vectors, etc. */
	p_init();			/* init buffer pool */
	ifp = &ifab;
	ifp->if_haddr[0] = abusinit(0);	/* init applebus unit 0 */
	ifp->if_addrnet = ipnetpart(ifp->if_addr);
	if ((im = ip2ddp(ifp->if_addr)) == 0)
		panic("can't find DDP net#");
	ifp->if_dnet = im->dnet;
	ifp->if_dnode = im->dnode;

#ifdef dartarp
	{ register AddrBlock *ab = (AddrBlock *)&ifp->if_haddr[0];
	ab->node = ifp->if_haddr[0];
	ab->skt = ddpIPSkt;
	ab->net = ifp->if_dnet;
	ifp->if_haddrlen = 4; }
#endif
	
	ilinit(0);			/* init intel/interlan unit 0 */
	ifp = &ifil;
	ipnet = ifp->if_addrnet = ipnetpart(ifp->if_addr);

	spl0();

	for (;;) {
		if (pq.pq_head == 0) {	/* no packets on main queue */
			back();		/* perform backround processing */
			continue;
		}
		s = splimp();		/* lock interrupts */
		p = p_deq(&pq);		/* dequeue packet */
		splx(s);
		if (p->p_len < 0) {	/* if error packet */
			p_free(p);
			continue;
		}
		switch (p->p_type) {	/* switch on packet type */
		case PT_ABUS:
			abreceive(p);	/* receive applebus packet */
			break;

		case PT_ENET:
			ilreceive(p);	/* receive ethernet packet */
			break;

		default:
			panic("bad packet type");
		}
	}
}


/*
 * Backround processing.  Call once per second timeout routines.
 * Process console commands.
 */
back()
{
	static int oldclock;
	register i;

	if ((i = msclock - oldclock) < 0)
		i = -i;
	if (i > 1000) {		/* one second */
		oldclock = msclock;
		arptimer();		/* timeout arp cache */
		rtmptimer();		/* so Macs can find us */
		/* igptimer();*/	/* so internet can find us */
	}
	if (!linereadyrx(0))
		return;
	switch (lineget(0)) {
	case '\003':	/* ^C, call ddt */
		asm(".word 0x4e4e");
		break;
	case 's':	/* 's', print stats */
		printf("\n---\n");
		ilprintstats(0);
		printf("---\n");
		abprintstats(0);
		break;
	}
}


/*
 * Broadcast an RTMP periodically so that Macs on our segment know
 * their net# and address of this 'bridge'.
 * (Currently uses only the 1st applebus interface, ifab[0];  should
 * step thru all interfaces).
 */
rtmptimer()
{
	static	int mydelay;
	register struct pbuf *p;
	struct ShortDDP sddp;
	register struct RTMP *r;
	register struct ifnet *ifp = &ifab;
	extern u_char broadcastaddr[];

	if (++mydelay < 30)	/* only run every 30 seconds */
		return;
	mydelay = 0;
	if ((p = p_get(PT_DATA)) == 0)
		return;		/* no buffer */
	sddp.length = ddpSSize + rtmpSize;
	p->p_len = sddp.length + lapSize;
	sddp.dstSkt = sddp.srcSkt = rtmpSkt;
	sddp.type = ddpRTMP;
	bcopy((caddr_t)&sddp, p->p_off + lapSize, ddpSSize);
	r = (struct RTMP *)(p->p_off + lapSize + ddpSSize);
	r->net = ifab.if_dnet;
	r->idLen = 8;
#ifdef dartarp
	r->id = ifab.if_haddr[2];
#else
	r->id = ifab.if_haddr[0];
#endif	
	(*ifp->if_output)(ifp, p, AF_SDDP, broadcastaddr);
}	


/*
 * Some subroutines.
 */


/*
 * The functions of the "abreceive" and "ilreceive" routines below might 
 * normally belong in the input interrupt sections of their respective
 * drivers.  But since they do some protocol translation as well, we've
 * stuck them here.  (Another consideration is that the LAP input interrupt
 * code is very time critical).
 */

/*
 * Receive next packet from applebus.
 */
abreceive(p)
	register struct pbuf *p;
{
	struct LAP *lp;
	struct DDP ddp;
	struct ShortDDP ddps;
	int i;
	register struct udp *up;
	register struct ip *ip;

	if (dbsw & DB_ABR)
		p_print("ABR", p);
	lp = (struct LAP *)p->p_off;
	p->p_off += lapSize;
	p->p_len -= lapSize;
	/*
	 * Below we copy the DDP header into a local structure to
	 * get word alignment.  Since lapSize and ddpSize are both odd,
	 * after handling the headers the packet is again on an even
	 * boundary (odd+odd=even).
	 */
	switch (lp->type) {
	case lapShortDDP:
		if ((p->p_len -= ddpSSize) < 0)
			goto drop;
		bzero((caddr_t)&ddp, sizeof ddp);
		bcopy(p->p_off, (caddr_t)&ddps, ddpSSize);
		p->p_off += ddpSSize;
		ddp.length = ddps.length;
		ddp.dstSkt = ddps.dstSkt;
		ddp.srcSkt = ddps.srcSkt;
		ddp.type = ddps.type;
		break;

	case lapDDP:
		if ((p->p_len -= ddpSize) < 0)
			goto drop;
		bcopy(p->p_off, (caddr_t)&ddp, ddpSize);
		p->p_off += ddpSize;
		break;

	/*
	 * Else no handler for this lap type, drop it.  Note: in the future
	 * abus IP and ARP packets may a LAP type, instead of the DDP types
	 * they use now.  See the comments on "aboutput" (applebus output
	 * encapsulation routine) below for some further explanation.
	 */
	default:
		goto drop;
	}

	switch (ddp.type) {
	case ddpARP:
		arpinput(p);	/* pass to ARP input */
		return;

	case ddpIP:
		routeip(p);		/* route this IP */
		return;
	}
	/*
	 * Not an IP or ARP, let's encapsulate this DDP in an IP.
	 */
	if (lp->type != lapDDP || ddp.dstNet == 0)
		goto drop;	/* must be a long DDP to encapsulate */
	i = lapSize + ddpSize + sizeof *ip + sizeof *up;
	p->p_off -= i;
	p->p_len += i;
	bzero(p->p_off, sizeof *ip + sizeof *up);
	ip = (struct ip *)p->p_off;
	up = (struct udp *)(p->p_off + sizeof *ip);
	ip->ip_v = IPVERSION;
	ip->ip_hl = (sizeof *ip >> 2);
	ip->ip_len = p->p_len;
	ip->ip_id = ipid++;
	ip->ip_ttl = IPFRAGTTL;
	ip->ip_p = IPPROTO_UDP;
	if ((ip->ip_src = ddp2ip(ddp.srcNet, ddp.srcNode)) == 0)
		goto drop;
	if ((ip->ip_dst = ddp2ip(ddp.dstNet, ddp.dstNode)) == 0)
		goto drop;
	ip->ip_sum = in_cksum((caddr_t)ip, sizeof *ip);
	up->src = ddp2ipskt(ddp.srcSkt);
	up->dst = ddp2ipskt(ddp.dstSkt);
	up->length = p->p_len - sizeof *ip;
	routeip(p);
	return;
drop:
	if (dbsw & DB_DROPS)
		printf("ABdrop\n");
	p_free(p);
	stats.dropabin++;
	return;
}


/*
 * Receive next packet from intel/interlan ethernet.
 */
ilreceive(p)
	register struct pbuf *p;
{
	struct ether_header *ehp;
	register struct ip *ip;
	register struct udp *up;

	if (dbsw & DB_ILR)
		p_print("ILR", p);
	ehp = (struct ether_header *)p->p_off;
	p->p_off += sizeof *ehp;
	p->p_len -= sizeof *ehp;
	switch (ntohs(ehp->ether_type)) {
	case ETHERTYPE_ARPTYPE:
		arpinput(p);	/* pass it to ARP */
		return;

	case ETHERTYPE_IPTYPE:
		break;
	
	default:
		goto drop;		/* can't handle it */
	}
	/*
	 * If the proto type is IP/UDP and the port number is in the
	 * magic range, remove the encapsulation from the DDP datagram.
	 */
	ip = (struct ip *)p->p_off;
	if (p->p_len < sizeof *ip)
		goto drop;
	if (ip->ip_p != IPPROTO_UDP)
		goto toip;
	up = (struct udp *)(p->p_off + sizeof *ip);
	if (p->p_len < (sizeof *ip + sizeof *up))
		goto drop;
	if (up->dst != ddpNWKSUnix)
		goto toip;	/* if not the magic socket number */
	/*
	 * Decapsulate.
	 */
	p->p_off += (sizeof *ip + sizeof *up);
	p->p_len -= (sizeof *ip + sizeof *up);
	routeddp(p);		/* route as a DDP */
	return;
toip:
	routeip(p);
	return;
drop:
	if (dbsw & DB_DROPS)
		printf("ILdrop\n");
	stats.dropilin++;
	p_free(p);
	return;
}


/*
 * Called from arpinput to determine if an address is in our range (and
 * we should respond/forward).  These current match routines are
 * simple minded and assume only one abus and enet interface per gateway.
 *
 * These match routines return 0 if no match and 1 (true) if a match occurs.
 * A 'true' return value causes the arpinput code to return a 'fake ARP'
 * to the requester:  i.e. the requester will think that the real destination
 * host responded to the ARP, whereas it was really this gateway 'faking'
 * a response.
 */

/*
 * Match an applebus ARP.  If the subnet of the target is not my own
 * subnet, return true (match).  This has the effect of forwarding all
 * traffic for 'unknown' subnets, to the backbone ethernet.
 */
abmatch(ifp, target)
	register struct ifnet *ifp;
	iaddr_t target;
{
	if (ipsubmask == 0)
		return (0);		/* subnets not used */
	if (ipsub(target) != ipsub(ifp->if_addr))
		return (1);
	else
		return (0);
}

/*
 * Match an ethernet ARP.  Only if the subnet of the target is
 * equal to the subnet of [one of] the applebus interface(s) [or a
 * reachable applebus segment], return true.  This assumes smarter
 * gateways on the ethernet will respond to ARPs for the rest of
 * campus.
 */
ilmatch(ifp, target)
	register struct ifnet *ifp;
	iaddr_t target;
{
	if (ipsubmask == 0)
		return (0);		/* subnets not used */
	if (ipsub(target) == ipsub(ifab.if_addr))
		return (1);
	else
		return (0);
}


/*
 * Route an IP packet.  This code currently assumes only one abus and
 * ether interface per gateway;  instead it should scan a list of interfaces.
 * If a routing table existed within the gateway, it would determine
 * which interface pointer (ifp) and destination address (dst below)
 * are passed to ifp->if_output.
 */
routeip(p)
	struct pbuf *p;
{
	register struct ifnet *ifp = &ifil;	/* default output to ether */
	register struct ip *ip = (struct ip *)p->p_off;
	int dst = ip->ip_dst;
	register iaddr_t dstnet = ipnetpart(dst);

	/*
	 * If for me, drop it, since we currently don't have any servers
	 * within the gateway.
	 */
	if (dst == ifil.if_addr || dst == ifab.if_addr)
		goto drop;

	if (ipsubmask) {	/* if we have subnets */
		int dstsub = ipsub(dst);
		if (dstnet != ipnet)	{ /* not to our major net */
			dst = iproutedef;
			goto out;	/* forward to ether */
		}
		if (dstsub == ipsub(ip->ip_src)) /* srcsubnet = dstsubnet */
			goto drop;
		if (dstsub == ifab.if_dnet)
			ifp = &ifab;	/* forward to abus, else ether */
	} else {		/* no subnets */
		if (dstnet == ipnetpart(ip->ip_src))	/* srcnet = dstnet */
			goto drop;
		if (dstnet == ifab.if_addrnet)
			ifp = &ifab;
		else if (dstnet == ifil.if_addrnet)
			ifp = &ifil;
		else 			/* not for us */
			dst = iproutedef;
	}
out:
	(*ifp->if_output)(ifp, p, AF_IP, &dst);
	return;
drop:
	if (dbsw & DB_DROPS)
		printf("ROUTEIPdrop\n");
	p_free(p);
	stats.droprouteip++;
	return;
}


/*
 * Route a DDP packet.  The same comments apply to this router as to
 * the IP router.  It certainly isn't very general yet.
 */
routeddp(p)
	register struct pbuf *p;
{
	register struct ifnet *ifp = &ifab;	/* default to applebus */
	struct DDP ddp;
	u_char dst;

	if (p->p_len < (ddpSize + lapSize))
		goto drop;
	bcopy(p->p_off+lapSize, (caddr_t)&ddp, ddpSize);
	/*
	 * Currently simple minded.  If the net doesnt match our abus
	 * interface, then drop it.  See README/TODO for better ideas.
	 */
	if (ddp.srcNet == ddp.dstNet)
		goto drop;	/* avoid broadcast loops */
	if (ddp.dstNet != ifp->if_dnet)
		goto drop;
	if (ddp.dstNode == ifp->if_dnode)
		goto drop;	/* no DDP services in gw (yet) */
	dst = ddp.dstNode;	/* should obtain from routing table */
	(*ifp->if_output)(ifp, p, AF_DDP, &dst);
	return;
drop:
	if (dbsw & DB_DROPS)
		printf("ROUTEDDPdrop\n");
	stats.droprouteddp++;
	p_free(p);
}

	
/*
 * Convert a DDP net/node number to an IP address.
 * See the comments on the struct ipmap table.
 */
iaddr_t
ddp2ip(dnet, dnode)
	register dnet, dnode;
{
	register struct ipmap *im;

	for (im = &ipmap[0] ; im->dnet > 0 ; im++ ) {
		if (im->dnet != dnet)
			continue;
		if (im->dnode < 0)
			return (im->inet | dnode);
		if (im->dnode == dnode)
			return (im->inet);
	}
	/* no match from table */
	if (ipsubmask)
		return (((dnet & ipsubmask) << ipsubshift) | ipnet | dnode);
	return (0);
}


/*
 * Convert an IP address to a DDP net/node.
 * This is currently only called once from main at abusinit time.
 */
struct ipmap *
ip2ddp(ia)
	iaddr_t	ia;
{
	register struct ipmap *im;
	static struct ipmap ims;
	register iaddr_t net = ipnetpart(ia);

	for (im = &ipmap[0] ; im->dnet > 0 ; im++ ) {
		if (ipnetpart(im->inet) == net)
			return (im);
	}
	/* no match */
	if (ipsubmask) {
		ims.dnet = ((ia >> ipsubshift) & ipsubmask);
		ims.dnode = (ia & 0xFF);
		return (&ims);
	}
	return (0);
}


/*
 * Applebus output routine.
 * Encapsulate a packet of type af for the local net.
 * 
 * Look at a normal net output routine (such as ethernet output, iloutput),
 * for the more general case of an output routine which can handle 
 * multiple units (using the ifp->if_unit field) with interrupt driven
 * output (and output queues).
 *
 * Also, normally an output routine prepends only the link 
 * level header (e.g. LAP).  However this routine is a bit unusual:
 *
 *   Since the LAP and DDP header sizes are both odd, DDP packets are
 *   usually passed about with both headers on the front, so that the
 *   data will always be on an even boundary.
 *
 *   Since Dartmouth chose to put IPs and ARPs inside DDPs (instead of
 *   LAPs, as all other IP hardware types have done),
 *   we prefix a LAP/DDP header onto IPs and ARPs.  In the future this
 *   certainly could (should?) change to bring applebus IP more into
 *   line with official IP policy.
 */
aboutput(ifp, p, af, dst)
	register struct ifnet *ifp;
	register struct pbuf *p;
	caddr_t dst;
{
	int type, s;
	u_char adst;
	iaddr_t idst;
	u_char tdst[4];
	struct LAP l;
	struct DDP d;
	WDS wds[2];		/* write data structure for abuswrite */

	if (dbsw & DB_ABO)
		printf("ABO\n");
	switch (af) {

	case AF_DDP:	/* DDP: already has LAP header */
		l.dst = *dst;
		l.type = lapDDP;
		bcopy((caddr_t)&l, p->p_off, lapSize);
		break;
		
	case AF_SDDP:	/* ShortDDP: already has LAP header */
		l.dst = *dst;
		l.type = lapShortDDP;
		bcopy((caddr_t)&l, p->p_off, lapSize);
		break;
		
#ifndef dartarp
	case AF_ARP:
		adst = *dst;
		type = lapARP;
		goto lapit;

	case AF_IP:
		idst = *(iaddr_t *)dst;
		if (!arpresolve(ifp, p, &idst, &adst))
			return (0);	/* if not yet resolved */
		type = lapIP;
	lapit:
		l.dst = adst;		/* src will be set on output */
		l.type = type;
		p->p_off -= (lapSize);
		p->p_len += (lapSize);
		bcopy((caddr_t)&l, p->p_off, lapSize);
		break;

#else
	case AF_ARP:
		adst = dst[2];
		type = ddpARP;		/* why not lapARP? */
		goto lapddp;

	case AF_IP:
		idst = *(iaddr_t *)dst;
		if (!arpresolve(ifp, p, &idst, &tdst[0]))
			return (0);	/* if not yet resolved */
		adst = tdst[2];
		type = ddpIP;		/* why not lapIP? */
	lapddp:
		l.dst = adst;		/* src will be set on output */
		l.type = lapDDP;
		d.length = p->p_len + ddpSize;
		d.checksum = 0;
		d.dstNet = d.srcNet = ifp->if_dnet;
		d.srcNode = ifp->if_haddr[2];
		d.dstNode = adst;
		d.srcSkt = d.dstSkt = ddpIPSkt;
		d.type = type;
		p->p_off -= (lapSize + ddpSize);
		p->p_len += (lapSize + ddpSize);
		bcopy((caddr_t)&l, p->p_off, lapSize);
		bcopy((caddr_t)&d, p->p_off+lapSize, ddpSize);
		break;
#endif !dartarp

	default:
		panic("ab%d: can't handle af%d\n", ifp->if_unit, af);
	}

	wds[1].size = 0;	/* terminate list */
	wds[0].size = p->p_len;
	wds[0].ptr = p->p_off;
	if (dbsw & DB_ABO)
		p_print("ABO", p);
	abuswrite(&wds[0]);	
	p_free(p);
	return (0);
}


struct abstats {
	int	interrupts;
	int	ipackets;
	int	opackets;
	int	crc;
	int	ovr;
	int	iund;
	int	xx;
	int	yy;
	int	bad;
	int	coll;
	int	defer;
	int	idleto;
	int	zz;
	int	nodata;
	int	ound;
	int	badddp;
	int	spur;
};


/*
 * Print applebus stats.
 */
abprintstats(unit)
{
	extern struct abstats abstats;
	register struct abstats *s = &abstats;

	printf("ab status unit %d\n", unit);
	printf("ints %d, in %d, out %d, crc %d, ovr %d\n",
	  s->interrupts, s->ipackets, s->opackets, s->crc, s->ovr);
	printf("iund %d, bad %d, coll %d\n", s->iund, s->bad, s->coll);
	printf("defer %d, idleto %d, nodata %d, ound %d, badddp %d, spur %d\n",
	  s->defer, s->idleto, s->nodata, s->ound, s->badddp, s->spur);
}


/*
 * Panic 'halt'.  Don't really halt, that would defeat memory refresh.
 */
panic(s,a,b,c)
{
	splimp();
	printf("PANIC! \007\007");
	printf(s,a,b,c);
	printf("\n");
	for (;;) { }	/* just wait for operator 'break' key */
}


/*
 * rand: generates a pseudo-random number in the range 0 to 2**31-1 
 */

static	long	randx = 1;

rand()
{
	return(((randx = randx*1103515245 + 12345)>>16) & 0x7FFFFFFF);
}
