/* tree.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	This module implements operations on expression trees.  In
 *	particular, it handles storage management for these trees.
 *
 *	The trees have the same structure (roughly) as those in
 *	pass 1.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: tree.c,v 5.5 89/05/12 12:53:14 pcc Rel-3_0 $";
/* static char ID[] = "@(#)tree.c	15.4	of 86/11/27"; */
#endif

/*
 *	Imported Objects
 */

# include <pcc.h>
# include <assert.h>
# include <tree.h>
# include <dagsymbol.h>
# include <bool.h>
# include <storage.h>
# include <opt.h>
# include <ops.h>
# include <erroro.h>
# include <arith.h>

#define Threshold	100

static NodeList  FreeList = NULL;	/* List of available nodes */
static NodeList  InUse = NULL;		/* List of nodes that are busy */
static int	 GarbageNodes = 0;	/* Freed nodes on InUse list */

NodeList DebugNode = NULL;		/* For sdb use */

static void
AddList(list, e)
	NodeList *list;
	NodeList e;
{
	e->next = *list;
	*list = e;
}

static void
CollectGarbage()
{
	NodeList p, next, new_in;

	new_in = NULL;
	for (p = InUse; p != NULL; p = next) {
		next = p->next;
		if (p->n.in.op == FREE)
			AddList(&FreeList, p);
		else
			AddList(&new_in, p);
		p = next;
	}
	InUse = new_in;
	GarbageNodes = 0;
}

TreeNode
TreeAllocate()
{
	NodeList p, c;
	int i;

	if (FreeList == NULL && GarbageNodes > Threshold)
		CollectGarbage();

	if (FreeList == NULL) {		/* Still none available */
		c = GetArray(s_Trees, TreeChunk, TNode);
		CheckStorage(c, "storage for tree nodes", 0);

		/* Turn array into linked list (backwards!) */

		p = NULL;
		for (i=0; i < TreeChunk; ++i) {
			c->next = p;
			p = c++;
		}

		/*
		 * Now p points to the head of the free list (the
		 * last entry in the chunk).
		 */
		
	} else
		p = FreeList;

	FreeList = p->next;

	/*
	 *	p is the tree node to be used
	 */

	AddList(&InUse, p);

	assert(p != DebugNode);
	return &(p->n);
}

void
TreeFree(t)
	TreeNode t;
{
	assert(t != NULL);
	assert(t->in.op != FREE);
	t->in.op = FREE;
	++GarbageNodes;
}

void
TreeInit()
{
	NodeList p, next;

	for (p=InUse; p != NULL; p=next) {
		assert(p->n.in.op == FREE);
		next = p->next;
		AddList(&FreeList, p);
	}
	InUse = NULL;
	GarbageNodes = 0;
}

TreeNode
CopyTree(t)		/* This makes a new copy of a tree */
	TreeNode t;
{
	TreeNode new_t;

	new_t = TreeAllocate();
	*new_t = *t;
	if (optype(t->in.op) != LTYPE)
		 new_t->in.left = CopyTree(t->in.left);
	if (optype(t->in.op) == BITYPE)
		 new_t->in.right = CopyTree(t->in.right);
	return new_t;
}


#ifdef SDBDEBUG
void
DBprint(p)		/* for use when debugging from SDB */
	register NODE *p;
{
	fwalk(p, eprint, 0);
}
#endif /* SDBDEBUG */

int				/* Not really, but expected by fwalk */
eprint( p, down, a, b )
	TreeNode p;
	int *a, *b;
{

	*a = *b = down+1;
	while( down >= 2 ){
		printf( "\t" );
		down -= 2;
		}
	if( down-- ) printf( "    " );


	printf( "%x) %s ", p, opst[p->in.op] );
	switch( p->in.op ) { /* special cases */

	case REG:
	case TREG:
		printf( "register %d", p->tn.rval );
		break;

	case FCON:
		printf("%.20le", p->fpn.dval);
		break;

	case ICON:
	case LABCON:
#ifndef MPX
	case TCON:	/* (MEY) add TCON and HCON for Alpha C */
	case HCON:	
	case CADDR:
#endif
	case LADDR:
	case LNAME:
	case PADDR:
	case PNAME:
	case TNAME:
	case LTEMP:
	case STLABEL:
	case STADDR:
	case STATNAME:
	case ADDR:
	case NAME:
		if (p->in.identifier == NULL || *(p->in.identifier) == '\0') {
			printf(CONFMT, p->tn.lval);
		} else
		if (p->tn.lval == 0) {
			printf("%s", p->in.identifier);
		} else {
			printf("%s+", p->in.identifier);
			printf(CONFMT, p->tn.lval);
		}
		printf(" %d ", p->tn.rval);
		break;

	case OREG:
		printf("rval %x", p->tn.rval);
		break;

	case STCALL:
	case UNARY STCALL:
	case STARG:
	case STASG:
		printf( "size=%d", p->stn.stsize );
		printf( "align=%d", p->stn.stalign );
		break;
		}

	printf( ", " );
	tprint( p->in.type );
	printf( "\n" );
	return 0;			/* for fwalk's benefit */
	}

/******************************************************************************
 ******************************************************************************

 * Tree pre- and post-processing

 ******************************************************************************
 *****************************************************************************/


static TreeNode FixOffset();

/*
 *	Pre-process the tree.  Makes some type changes, and looks
 *	for & Local
 */

TreeNode
PreTree(t)
	TreeNode t;
{
	TreeNode p;
	Identifier id;

#ifdef MPX
/* #ifdef FORT */
	TreeNode q;
	TWORD ty;
#endif

	switch(optype(t->in.op))
	{
		case BITYPE:
			t->in.right = PreTree(t->in.right);
			t->in.left  = PreTree(t->in.left);
			break;

		case UTYPE:
			t->in.left  = PreTree(t->in.left);
			break;

		case LTYPE:
			break;

		default:
			InternalFault("Bad optype for op %d", t->in.op);
			break;
	}

	switch(t->in.op)
	{
		case ICON:
			t = FixType(t);		/* ensure correct lit type */
			break;

		case UNARY CALL:	/* Procedure calls */
		case UNARY FORTCALL:
		case CALL:
		case FORTCALL:
		case STCALL:
		case UNARY STCALL:

			/*	Check if this is a call of a bad function */

#ifndef MPX
			if (BadFcn < 0 && t->in.left->in.op == CADDR) {
				{
					register int ix = 0;
					register TreeNode lt;

					lt = t->in.left;
					while (BadFcnNames[ix] != NULL &&
						strcmp(BadFcnNames[ix], lt->tn.identifier) != 0)
						++ix;

					if( BadFcnNames[ix] != NULL )
						BadFcn = ix;
				}
			}
#endif
			break;

			/* For FORTRAN, turn OREG's into UNARY MUL, PLUS, REG,
			 * ICON.  This should disappear some day.
			 * For C, OREGs not allowed
			 */
		case OREG:
#ifdef MPX
/*XXX*/			break;
#endif
#ifdef FORT
			ty = t->in.type;
			if( t->tn.lval == 0 )
			{
				t->in.op = REG;
				t->tn.lval = t->tn.rval; /* id cookie */
				t = FixType(t);
			}
			else
			{
				p = TreeAllocate();
				p->in.op = REG;
				p->in.type = INCREF(ty);
				p->in.identifier = NULL;
				p->tn.rval = t->tn.rval;
				p->tn.lval = p->tn.rval;	/* id cookie */
				p = FixType(p);

				q = TreeAllocate();
				q->in.op = ICON;
				q->in.type = INT;
				q->in.identifier = NULL;
				q->tn.lval = t->tn.lval;
				q->tn.rval = 0;

				t->in.op = PLUS;
				t->in.left = p;
				t->in.right = q;
			}

			p = TreeAllocate();
			p->in.op = UNARY MUL;
			p->in.type = ty;
			p->in.identifier = NULL;
			p->in.left = t;
			p->tn.rval = 0;

			t->in.type = INCREF(ty);

			return p;
#else
#ifndef MPX
			ExternalFault("OREG generated by pass 1");
			break;
#endif
#endif

		case CBRANCH:
			assert(t->in.right->in.op == ICON);
			t->in.right->in.op = STLABEL;
			break;

		case LNAME:
		case PNAME:
		case STATNAME:
			id = IdLookUp(t->in.op, (CONSZ) t->tn.rval, ANYTYPE);
			return(FixOffset(t,id));

		case REG:		/* Check that the "cookie" is ok */
			assert(IsRegVar(t->tn.rval) || t->tn.rval==t->tn.lval);
			return(FixType(t));

		case NAME:
			if( t->tn.rval == 0 )
			{
				/* a NAME node with no name.  This can happen
				 * in constructs like
				 *	*((int *)4) = 7;
				 * The best thing to do here is turn things
				 * into UNARY MUL, ICON
				 */

				p = TreeAllocate();
				*p = *t;

				t->in.op = ICON;
				t->in.type = INCREF(p->in.type);

				p->in.op = UNARY MUL;
				p->in.right = NIL;
				p->in.left = t;

				return(p);
			}
			id = IdLookUp(t->in.op, (CONSZ) t->tn.rval, ANYTYPE);

#ifdef FORT

			if( id == NoId )
			{
				/* Fortran pass1 will make up NAMEs for things
				 * to pass to i/o functions and the like at the
				 * drop of a hat.  (C pass1 is equally guilty.)
				 * To preserve everyone's sanity, we enter them
				 * in the symbol table with negative id
				 * cookies.
				 */

				assert( t->in.identifier[0] == '.' && t->tn.rval < 0 );
				fprintf(stderr, "Symbol %s undefined\n", t->in.identifier);/*DEBUG*/
				id = Pass1Symbol(NAME, (CONSZ) t->tn.rval,
					t->in.type, t->in.identifier);
			}
#endif

			assert(id != NoId);
			return(FixOffset(t,id));

		case ADDR:
			id = IdLookUp(NAME, (CONSZ) t->tn.rval, ANYTYPE);

#ifdef FORT
			if( id == NoId )
			{
				/* Fortran pass1 will make up ADDRs for things
				 * to pass to i/o functions and the like at the
				 * drop of a hat.  (C pass1 seems less guilty.)
				 * To preserve everyone's sanity, we enter them
				 * in the symbol table with negative id
				 * cookies.
				 */

				assert( t->in.identifier[0] == '.' && t->tn.rval < 0 );
				fprintf(stderr, "Symbol %s undefined\n", t->in.identifier);/*DEBUG*/
				id = Pass1Symbol(NAME, (CONSZ) t->tn.rval,
					DECREF(t->in.type), t->in.identifier);
			}
#endif

			assert(id != NoId);
			return(FixOffset(t,id));

		case STADDR:
		case LADDR:
		case PADDR:
			id = IdLookUp(ToNAME(t->in.op), (CONSZ)  t->tn.rval, ANYTYPE);
			if( t->in.op == PADDR )
				ParamAddressed();
			else
				SetAddressed(id);

			return(FixOffset(t,id));

		case UNARY AND:
			p = t->in.left;
			if (p->in.op == LNAME || p->in.op == PNAME ||
			    p->in.op == STATNAME || p->in.op == NAME) {

				/* Note that we've addressed a local, and
				 * replace everything with LADDR or PADDR
				 */

				id = IdLookUp(p->in.op, (CONSZ) p->tn.rval, ANYTYPE);
				if (p->in.op == PNAME)
					ParamAddressed();
				else
				if (p->in.op != NAME)
					SetAddressed(id);

				p->in.op = ToADDR(p->in.op);
				p->tn.type = INCREF(p->tn.type);

				/* check that type of leaf matches type on
				 * UNARY AND
				 */

				if( p->tn.type == t->in.type )
				{
					TreeFree(t);
					return(p);
				}
				t->in.op = OCONVLEAF;
			}
			else
			if( p->in.op == UNARY MUL )
			{
				/* eat UNARY AND, UNARY MUL */
				TreeFree(t);
				t = p->in.left;
				TreeFree(p);
			}
			break;
	}
	return(t);
}

/*
 *	Under some circumstances, on some machines, it is sensible to
 *	treat a NAME as * ADDR.  Although the decision is machine dependent
 *	the actual transformation is done below.  Note that this MODIFIES
 *	the existing tree t, it does not return a new one.  Beware of
 *	copies of the information in the NAME node, they may no
 *	longer be valid.
 */

void
RewriteNAME(t)
	TreeNode t;
{
	TreeNode p;
#ifdef FORT
	int new_cookie;
#endif /* FORT */

	/* PreTree handled NONAME NAMES */

	if( t->in.op == NAME && t->tn.rval != 0 )
	{
		p = TreeAllocate();
		*p = *t;
#ifdef FORT
		if( (new_cookie = InCommonBlock(IdLookUp(p->in.op, (CONSZ) p->tn.rval, ANYTYPE))) > 0 )
			p->tn.rval = new_cookie;
#endif /* FORT */
		p->in.op = ToADDR(t->in.op);
		p->in.type = INCREF(p->in.type);

		t->in.op = UNARY MUL;
		t->in.identifier = NULL;
		t->tn.rval = 0;
		t->in.left = p;
	}
}

void
DoOffset( t, offset )
	TreeNode t;
	CONSZ offset;
{
	TreeNode p;
	TreeNode q;

	p = TreeAllocate();
	*p = *t;
	p->in.op = ICON;
	p->in.type = INT;
	p->in.identifier = NULL;
	p->tn.rval = 0;

	p->tn.lval = t->tn.lval - offset;
	t->tn.lval = offset;

	q = TreeAllocate();
	*q = *t;

	t->in.op = PLUS;
	t->in.left = FixType(q);
	t->in.right = p;
	t->in.identifier = NULL;
}

/*
 * It is imperative that NAME, ADDR, LNAME, LADDR,.... nodes all have a
 * consistent offset when they appear in the Dag.  If necessary, this routine
 * fakes up a tree that looks like (PLUS, ADDR, ICON) to ensure that the offset
 * is always consistent.
 */

static TreeNode
FixOffset(t, id)
	TreeNode t;
	Identifier id;
{
	TreeNode p, q, tref;
	int symop;

	assert(id != NoId);
	symop = IdOp(id);

	tref = IdTree(id);
	if( tref == NULL )
	{
		/* No symbol table tree yet.  Make one */

		tref = TreeAllocate();
		*tref = *t;
		if( symop != t->in.op )
		{
			/* [*]ADDR node.  Enter as [*]NAME, with the correct
			 * type. */
			tref->in.op = symop;
			tref->in.type = SymType(id);
		}
		SetIdTree(id, tref);
		TreeFree(tref);
		tref = IdTree(id);
		if( tref->in.op == NAME || tref->in.op == STATNAME )
			tref->tn.lval = 0;
	}

#ifdef FORT
	if( t->tn.lval != tref->tn.lval && (t->in.op != NAME || ISARY(GetActualType(id))) )
#else
	if( t->tn.lval != tref->tn.lval )
#endif /* FORT */
	{
		/* A node with a funny offset. If this is any NAME class, turn
		 * it into UNARY MUL, [*]ADDR and let the ADDR code fix it.
		 * (It is NAME class if the op in the tree node is the same as
		 * the op in the symbol table.  Otherwise, its an address.)
		 * If ADDR class, change to PLUS, <thing>, ICON with the
		 * offset in the ICON
		 */

		if( symop == t->in.op )
		{
			p = TreeAllocate();
			*p = *t;
			t->in.op = ToADDR(t->in.op);
			t->in.type = INCREF(p->in.type);

			p->in.op = UNARY MUL;
			p->in.identifier = NULL;
			p->tn.rval = 0;
			p->in.left = PreTree(t);

			return(p);
		}

		DoOffset( t, tref->tn.lval );
		return( t ); 
	}
	return(FixType(t));
}

/*
 * Undo painted type conversions on certain types of leaves.
 */

TreeNode
FixType(t)
	TreeNode t;
{
	Identifier id;
	TWORD ty;
	TreeNode p;

	switch(t->in.op)
	{
	case NAME:
	case LNAME:
	case PNAME:
	case STATNAME:
		id = IdLookUp(t->in.op, (CONSZ) t->tn.rval, ANYTYPE);
		assert(id != NoId);
		ty = SymType(id);

		if( !CanBeLNAME(ty) )
		{
			/* [*]NAME node with an offset or a reference to
			 * the first element of a structure or an
			 * array.   Change it to UMUL, ADDR and let
			 * the (ST)ADDR code take care of removing any
			 * offset
			 */

			p = TreeAllocate();
			*p = *t;
			t->in.op = ToADDR(t->in.op);
			t->in.type = INCREF(p->in.type);

			p->in.op = UNARY MUL;
			p->tn.rval = 0;
			p->in.left = PreTree(t);
			p->in.identifier = NULL;

			return(p);
		}
		break;

	case ADDR:
	case LADDR:
	case PADDR:
	case STADDR:
		id = IdLookUp(ToNAME(t->in.op), (CONSZ) t->tn.rval, ANYTYPE);
		assert(id != NoId);
		ty = INCREF(SymType(id));
		break;

	case REG:
		id = IdLookUp(t->in.op, t->tn.lval, ANYTYPE);
		assert(id != NoId);
		ty = SymType(id);
		break;

	case ICON:
		/*
		 *	If the ICON's type is non-standard, then
		 *	make a standard ICON and insert a conversion
		 */

		ty = IconBaseType(t->in.type);		/* type we want */
		break;

	default:
		InternalFault("FixType called for op %d", t->in.op);
		break;
	}

	if( ty != t->in.type )
	{
		p = TreeAllocate();
		p->in.op = OCONVLEAF;
		p->in.type = t->in.type;
		p->in.left = t;
		p->tn.rval = 0;
		t->in.type = ty;
		return(p);
	}

	return(t);
}

/*
 *	This converts a node of one type to another type by inserting a
 *	conversion, if necessary.
 *	At present, no attempt is made to handle pointer types.
 */

static TreeNode
Convert(t, typ)
	TreeNode t;
	TWORD typ;	/* Type to convert node to */
{
	TreeNode t1;

	if( DoCast(t, typ) )	/* If cast is unnecessary */
		return t;
	t1 = TreeAllocate();
	t1->in.op = SCONV;	/* No pointer conversion can be necessary */
	t1->in.type = typ;
	t1->in.left = t;
	t1->in.identifier = NULL;
	t1->tn.rval = 0;
	return t1;
}

/*
 *	This determines the appropriate type for arithmetic when the
 *	operands coming in are of different types.
 */

static TWORD
OperationType(tyl, tyr)
	TWORD tyl, tyr;	/* Left and right types respectively */
{
	TWORD typ;	/* Build up type here */

#ifdef HAS_SINGLE_ARITHMETIC
	if(tyl == FLOAT && tyr == FLOAT)
		typ = FLOAT;
	else
#endif
	if(tyl == DOUBLE || tyl == FLOAT || tyr == DOUBLE || tyr == FLOAT)
		typ = DOUBLE;
	else {
		/* First, get the appropriate size */
		if(tyl == LONG || tyl == ULONG || tyr == LONG || tyr == ULONG)
			typ = LONG;
		else
			typ = INT;	/* Arithmetic done in ints */
		/* Now, make unsigned if necessary */
		if(ISUNSIGNED(tyl) || ISUNSIGNED(tyr))
			typ = ENUNSIGN(typ);
	}
	return typ;
}

/*
 *	This function ensures that the operands of a binary arithmetic
 *	operator are compatable.
 */

static TreeNode
MatchTypes(t)
	TreeNode t;
{
	TWORD ty = t->in.type;
	TWORD tyl = t->in.left->in.type;
	TWORD tyr = t->in.right->in.type;
	TWORD newtyp;

	if(tyl == tyr)
		return t;	/* Matched already. */

	if(BTYPE(tyl) != tyl || BTYPE(tyr) != tyr)
		return t;	/* If either is a pointer, checking for
				   validity has already occurred.  Pass2 will
				   handle it */

	newtyp = OperationType(tyl, tyr); /* Determine the correct type for
					     operation */
 	if(tyl != newtyp)
		t->in.left = Convert(t->in.left, newtyp);
	if(tyr != newtyp)
		t->in.right = Convert(t->in.right, newtyp);
	if(ty != newtyp) {
		t->in.type = newtyp;
		t = Convert(t, ty);
	}
	return t;
}

TreeNode
FinalOpt(t)
	TreeNode t;
{
	TreeNode ltree;
	TreeNode rtree;
	int ty;

	/* First, ensure that the types of arithmetic operations like
	 * + and - are properly matched.  They can get unmatched if
	 * pass 1 leaves the types untouched in (e.g.) i *= 0.5 and
	 * we do not recover the assignment operator.
	 */

	switch(t->in.op)
	{
	case PLUS:
	case MINUS:
#ifdef GOULD_NP1
	case INV:
#endif /* GOULD_NP1 */
	case DIV:
	case MOD:
	case MUL:
		/* Check for type mismatch */
		t = MatchTypes(t);
		break;
	}

	ty = optype(t->in.op);
	switch( ty )
	{
	default:
		InternalFault("Bad optype for op %d", t->in.op);
		break;

	case LTYPE:
		break;

	case BITYPE:
		t->in.right = rtree = FinalOpt(t->in.right);
	case UTYPE:
		t->in.left = ltree = FinalOpt(t->in.left);
		break;
	}

	if( asgop(t->in.op) )
	{
		/* If the right subtree is an OCONVTREE, and its type matches
		 * the type of the assignment, the conversion is unnecessary.
		 * NOTE: This will still generate unnecessary conversions
		 * among pointer types, and between int/long.  A better
		 * (machine dependent) version of this test should be written
		 * someday.
		 */

		if( rtree->in.op == OCONVTREE && t->in.type == rtree->in.type )
		{
			t->in.right = rtree->in.left;
			TreeFree(rtree);
		}
	}
	else
	switch(t->in.op)
	{
	default:
		if( ty != LTYPE && conval( ltree, t->in.op, t->in.type, rtree) )
		{
			if( ty == BITYPE )
				TreeFree(rtree);
			TreeFree(t);
			return(ltree);
		}
		break;

	case OCONVLEAF:
		if (optype(t->in.op) == LTYPE) {	/* normal case */
			ltree->in.type = t->in.type;
			TreeFree(t);
			return(ltree);
		} else {
			/*
			 *	OCONVLEAF with a non-leaf underneath.
			 *	We have replaced a leaf with an expression.
			 *	Set up an SCONV or PCONV, and then try to do
			 *	the cast.  All PCONV's should be painted down
			 *	eventually, but we start them off as PCONV's
			 *	anyway so that we can keep the rules straight.
			 */

			t->in.op = ((t->in.type &TMASK)==0) ? SCONV : PCONV;
			return(FinalOpt(t));
		}

	case OCONVTREE:
		t->in.op = ((t->in.type &TMASK)==0) ? SCONV : PCONV;
		return(FinalOpt(t));
		
	case SCONV:
	case PCONV:
			/* Try to get rid of cast */
		if( DoCast(ltree,t->in.type) )
		{
			TreeFree(t);
			return(FinalOpt(ltree)); /* in case we modified another
						  * conversion */
		}
		break;

	case LEAFNOP:
		InternalFault("LEAFNOP passed to FinalOpt");
		break;

	case UNARYNOP:
		TreeFree(t);
		return(ltree);

	case UNARY MUL:
		/******************************************************/
		/* Old original code:				      */
		/* 						      */
		/* if( ltree->in.op == ADDR || ltree->in.op == ICON ) */
		/* 						      */
		/* This code used to cause the indirection to be      */
		/* removed for all ADDR nodes and all ICON nodes.     */
		/* The UMUL ADDR or UMUL ICON was rewritten as a NAME */
		/* without any indirection; in this case our usage    */
		/* of indirection with string constants (that are in  */
		/* in far address space) could not be addressed since */
		/* the address literal was required.  This new code   */
		/* only allows the revision for ICONs unless they are */
		/* char pointers.  The removal for ADDR nodes from    */
		/* being rewritten allows the node associated with the*/
		/* &_iob[n] structure to remain as an ADDR node and   */
		/* is then appropriately addressed by a literal.      */
		/*						      */
		/* SPR# 87001506 and SPR# 87001948 addressed for PCO  */
		/*						      */
		/******************************************************/

		if( (ltree->in.op == ICON) &&
		   !( ltree->in.type == PTR+CHAR && ltree->tn.rval < 0) )
		{
			/* NOTE: this works for ADDRs because they can only
			 * refer to NAMEs.  If they ever can refer to
			 * LNAMEs or PNAMEs, this code must be smarter
			 */
			ltree->in.op = NAME;
			ltree->in.type = t->in.type;
			TreeFree(t);
			return(ltree);
		}
		else
		if( ltree->in.op == LADDR || ltree->in.op == PADDR ||
		    ltree->in.op == STADDR )
		{
			ltree->in.op = ToNAME(ltree->in.op);
			ltree->in.type = t->in.type;
			TreeFree(t);
			return(ltree);
		}
		break;

	case UNARY AND:
		if( ltree->in.op == NAME )
		{
			ltree->in.op = ADDR;
			ltree->in.type = t->in.type;
			TreeFree(t);
			return(ltree);
		}
		break;
	}

	return(t);
}


/*
 *	Is an automatic variable of type t OK for LNAME or PNAME?
 *	That is, is it an "atomic" type?
 */

Boolean
CanBeLNAME(t)
	TWORD t;
{
	if (ISPTR(t))
/***/		return True;
	else
	switch(t) {			/* somewhat naughty */
		case INT: case UNSIGNED:
		case LONG: case ULONG:
		case CHAR: case UCHAR:
		case SHORT: case USHORT:
		case FLOAT: case DOUBLE:
		case ENUMTY:
/***/			return True;

		default:
/***/			return False;
	}
}
