/* loop.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 * Loop invariant removal.  See Aho and Ullman Section 13.4
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: loop.c,v 5.5 89/05/12 12:51:42 pcc Rel-3_0 $";
/* static char ID[] = "@(#)loop.c	15.5	of 86/11/13"; */
#endif

/*
 * Import
 */

#include <assert.h>
#include <blocks.h>
#include <cost.h>
#include <dag.h>
#include <daghash.h>
#include <dagsymbol.h>
#include <duchain.h>
#include <erroro.h>
#include <flow.h>
#include <instruct.h>
#include <labels.h>
#include <livedead.h>
#include <longset.h>
#include <loop.h>
#include <ops.h>
#include <pcc.h>
#include <prune.h>
#include <refcount.h>
#include <storage.h>
#include <target.h>
#include <temp.h>
#include <udchain.h>

extern int GoodGlobals;

/*
 * Export
 */

int NoLoopInvariants = 0;	/* if on, do not do loop invariant removal */
int CautiousInvariants = 0;	/* if on, do only cautious loop inv. rem. */
int ldebug = 0;			/* if on, print debug info */

/*
 * Private
 */


typedef struct BTE {
	BasicBlock block;	/* pointer to block */

	} BTE_type, *BlockTableEntry;

static BlockTableEntry BTable = NULL;
static int BTSize;

static DAG_Node first;			/* first node of pre-header DAG */
static DAG_Node last;			/* last node of pre-header DAG */

static int ninvar;			/* count of number of invariant nodes
					 * found
					 */

static DAG_Node	*save_invar = NULL;	/* a work array */
static int save_size;			/* size of same */
#define SAVE_SLOP	20		/* In protecting itself from type puns,
					 * the optimizer may occasionally
					 * allocate more nodes in a pre-header
					 * than the number that were originally
					 * marked invariant.  This provides a
					 * bit of breathing room
					 */

static LongSet UDtmpset;		/* work set for UD info */

	/* Is DAG node d eligible to be lifted out of the loop, given that it
	 * is in block b?
	 */
#define MayLift(b,d)	((((b)->eligible && (d)->in_cond == NotConditional) || \
			(!CautiousInvariants && NoErrors(d))) \
			&& (!ConditionalOp((d)->op)))
	/*
	 *	Temporarily, we don't lift conditional operators because
	 *	UpdateActivity can't handle it.  (The zone numbers are
	 *	wrong).
	 */

static void	SetUpDAGNodes();
static void	CleanUpDAGNodes();
static void	ProcessLoop();
static Boolean	PropagateInvariants();
static void	MoveInvariants();
static DAG_Node	NewNode();
static DAG_Node	CopyLeaf();
static DAG_Node	CopyUnary();
static void	LiftAssigns();
static void	MoveAttached();
static Boolean	IdInvariant();
static void	TryPreHeader();
static Boolean	Try2();
static void	UsePreHeader();
static int	PreHdrBlock();
static void	UDUpdate();
static void	FixBranch();
static BasicBlock AddPreHeader();
static void	AddLTempList();
static Boolean	ReachedByOther();
static void	DFN_Fix();
static void	DOM_Fix();
static void	InitSave();

/* Loop invariant removal.  Also finds loops for use in deciding what things
 * go in registers.
 */
void
LoopInvariants()
{
	int i;
	BasicBlock b;
	LongSet newdefs, etemp;

	AdjustFlowGraph();
	FindLoops(False);
	PruneDags();

	if( NoLoopInvariants || nloops == 0 )
/**/		return;

	if( BTable != NULL ) {
		free(BTable);
		DecreaseSpace(s_BlckTable, BTSize * sizeof(BTE_type));
	}
	BTable = GetArray(s_BlckTable, NumFlowNodes + nloops, BTE_type);
	CheckStorage(BTable, "storage for loop block table", 0);
	BTSize = NumFlowNodes + nloops;

	UDtmpset = CreateSet(UDSetSize);
	newdefs = CreateSet(UDSetSize);
	etemp = CreateSet(NumFlowNodes);

	InitRefs();
	for( b = FirstBlock; b != NULL; b = b->next )
		SetUpDAGNodes(b);

	/* process loops FROM THE OUTSIDE IN */

	for( i = nloops; --i >= 0; )
	{
		ProcessLoop(LoopPtrs[i], newdefs, etemp);
	}

	PruneDags();
	for( b = FirstBlock; b != NULL; b = b->next )
		CleanUpDAGNodes(b);

	LDDataFlow();

	DestroySet(newdefs);
	DestroySet(UDtmpset);
	DestroySet(etemp);
}

static void
SetUpDAGNodes(b)
	BasicBlock b;
{
	DAG_Node d;
	Boolean known_stores, unknown_stores;
	AttachedID aid;
	int s;

	/* Initialization for DAG nodes in a block.  Propagate UD information
	 * through the entire block.  Determine if there is anything in this
	 * block that might set something that we fetch.
	 */

	/* if globals are evil, act as if every block stores */
	known_stores = unknown_stores = !GoodGlobals;

	if( b->reachable )
	{
		UDSecondWalk(b,True);
		for( d = b->Dag; d != NULL; d = d->next )
		{
			/* If fetches already bad news, don't bother checking
			 * NB: unknown_stores implies known_stores
			 */
			if( !known_stores )
			{
				if(callop(d->op) ||
				     ( d->op == UNARY MUL && !d->is_fetch))
				{
					known_stores = True;
					if( d->indirect == NoId )
						unknown_stores = True;
				}

				if( d->delayed != NULL )
				{
					s = IdOp(d->leaf_id);
					if( s == NAME ||
					  ((s == PNAME || s == LNAME || s == STATNAME) && WasAddressed(d->leaf_id)))
						known_stores = True;
				}
			}
			if( !known_stores )
			{
				for( aid = d->attached; aid != NULL; aid = aid->next )
				{
					s = IdOp(aid->id);
					if( s == NAME ||
					  ((s == PNAME || s == LNAME || s == STATNAME) && WasAddressed(aid->id)))
						known_stores = True;
				}
			}
		}
	}
	b->unknown_indir_stores = unknown_stores;
	b->known_indir_stores = known_stores;
}

/*
 * Cleanup - go around and remove reaching definitions from all nodes
 */

static void
CleanUpDAGNodes(b)
	BasicBlock b;
{
	DAG_Node d;

	for( d = b->Dag; d != NULL; d = d->next )
	if( d->In != NULL )
	{
		DestroySet(d->In);
		d->In = NULL;
	}
}

/*
 * This is the routine that does the real work of moving loop invariants.
 * It is called once for each loop.
 */

static void
ProcessLoop(e, newdefs, etemp)
	LTabEntry e;		/* The loop to be processed */
	LongSet newdefs;	/* A working set, used to keep track of the
				 * new definitions created for the pre-header.
				 */
	LongSet etemp;		/* A working set uset to determine which
				 * blocks dominate all exits of a loop
				 */
{
	int i, nblocks;
	BasicBlock b, LoopHeader, PreHeader;
	Boolean changed;

	if( ldebug )
	{
		printf("Loop invariance calculations for loop ");
		DumpSet(e->loop);
		printf("\n");
	}

	/* Start out by marking all the blocks as eligible/not eligible, exit
	 * etc.  Then run around marking DAG nodes as invariant until no more
	 * invariants can be found.
	 */

	nblocks = MarkBlocks(e, etemp);

	ninvar = 0;
	first = NULL;
	last = NULL;
	do
	{
		changed = False;
		for( i = 0; i < nblocks; i++ )
			changed = PropagateInvariants(e, BTable[i].block,
					changed);
	} while( changed );

	if( ninvar == 0 )
	{
		if( ldebug > 1 )
			printf("No nodes marked invariant\n");
/**/		return;
	}

	if( ldebug > 1 )
	{
		printf("Exits:");
		DumpSet(e->exits);

		printf("\nEligible:");
		DumpSet(etemp);
		printf("\n");

		if( ldebug > 5 )
		{
			for( i = 0; i < nblocks; i++ )
			{
				b = BTable[i].block;
				printf("Marked dag for block # %d:\n", b->blocknum);
				PrintGraph(b->Dag);
			}
		}
	}

	/* Create a block to be the pre-header */

	e->prehead = PreHeader = AddPreHeader(e->head, e->loop);

	/* Do some initializations, then call MoveInvariants() to move nodes
	 * into the pre-header.  Propagate UD information through the pre-
	 * header and update UD information for all other blocks in the loop.
	 *
	 * The initial value used for ud.In is a conservative lie; what we
	 * really should do is take the intersection over all predecessors
	 * of the pre-header
	 */

	NullSet(newdefs);
	InitSave(ninvar);
	MoveInvariants(e, first, newdefs);
	
	LoopHeader = FlowGraph[e->head].block;
	CopySet(PreHeader->ud.In, LoopHeader->ud.In);
	UDSecondWalk(PreHeader,True);
	LDWalk(PreHeader);

	for( i = 0; i < nblocks; i++ )
		UDUpdate(BTable[i].block, newdefs);


	if( ldebug > 2 )
	{
		printf("pre-header DAG:\n");
		PrintGraph(FirstNodeCreated);

		if( ldebug > 4 )
		{
			for( i = 0; i < nblocks; i++ )
			{
				b = BTable[i].block;
				printf("Revised DAG for block # %d\n", b->blocknum);
				PrintGraph(b->Dag);
			}
		}
	}
}

/*
 * Update loop block b with the definitions created in the pre-header.
 * Defs enter block, leave it, and are neither killed nor generated in the
 * block.
 * This probably belongs in udchain.c
 */

static void
UDUpdate(b, newdefs)
	BasicBlock b;		/* The block to update */
	LongSet newdefs;	/* The set of definitions created in the pre-
				 * header */
{
	DAG_Node d;

	Union(b->ud.In,  b->ud.In,  newdefs);
	Union(b->ud.Out, b->ud.Out, newdefs);
	Difference(b->ud.Kill, b->ud.Kill, newdefs);
	Difference(b->ud.Gen,  b->ud.Gen,  newdefs);

	for( d = b->Dag; d != NULL; d = d->next )
	if( d->In != NULL )
		Union(d->In, d->In, newdefs);
}

/* Create a block to be the pre-header.  This is a lot of work.  We have to
 * do the following:
 *	1) Create a new basic block, consisting of a new label and a branch
 *	   to the pre-header.  (This branch is usually unnecessary, but it
 *	   is too much trouble to figure this out.  Leave it for the
 *	   peepholer).
 *	2) Extend the flow graph to include the new pre-header.
 *	3) Initialize UD and DU information for the new block.
 *	4) Add the preheader to all loops containing the current loop - except
 *	   for the current loop itself.  If any such loop has the same header
 *	   as the current loop, the preheader of the current loop will be
 *	   the new pre-header for this loop.
 *	5) Find all predecessors of the header, and if they are outside the
 *	   loop, change them to point to the pre-header.
 *	6) While we are doing (5), look for a spot for the pre-header on the
 *	   list of basic blocks.  (This list determines the order in which the
 *	   blocks are output.)
 *	7) Clean up loose ends in the flow graph by adding the edge from the
 *	   pre-header to the header.
 *	8) Fix up dominator sets and DFN numbers.
 *	   (For now, recompute them).
 */

static BasicBlock
AddPreHeader(HIndex, loop)
	FlowIndex HIndex;
	LongSet loop;
{
	int label;
	BasicBlock b;
	BasicBlock LoopHeader, PreHeader;
	FlowIndex PHIndex;
	PredNode p, pnext;
	int i;
	LTabEntry lp;
	

	LoopHeader = FlowGraph[HIndex].block;
	PreHeader = CreateBlock();
	label = PreHdrBlock(PreHeader, LoopHeader);

	PHIndex = ExtendGraph(PreHeader);
	assert(LoopHeader->reachable);
	PreHeader->reachable = True;
	NumReachableNodes++;
	FlowGraph[PHIndex].nesting = FlowGraph[HIndex].nesting - 1;
	FlowGraph[PHIndex].loopweight = LoopWeight(FlowGraph[PHIndex].nesting);

	UDNewBlock(PreHeader);
	DUNewBlock(PreHeader);
	LDNewBlock(PreHeader);

	/* Add preheader to all loops containing the header, and
	 * fix up the headers of all containing loops that share the
	 * same header */

	for( i = 0; i < nloops; i++ )
	{
		lp = LoopPtrs[i];
		if( lp->loop != loop && Subset(loop,lp->loop))
		{
			Insert( (int) PHIndex, lp->loop);
			if( lp->head == HIndex )
				lp->head = PHIndex;
		}
	}

	b = NULL;	/* Will point to possible place to put pre-header */
	for( p = FlowGraph[HIndex].preds; p != NULL; p = pnext )
	{
		pnext = p->next;	/* FixBranch will remove p from list */
		if( !IsElement((int) p->p, loop) )
		{
			FixBranch(p->p, HIndex, PreHeader, label);
			if( b == NULL && PreHeader->next == NULL )
			{
				/* PreHeader->next still NULL meaning we have
				 * not found a place to link it in.  See if we
				 * can put it after this block.  If we can't,
				 * wipe b out.
				 */
				b = FlowGraph[p->p].block;
				if( !IsUBranch(b->code.last) &&
				    !IsSwitch(b->code.last) )
					b = NULL;
			}
		}
	}

	/* See if we succeeded in linking the pre-header in.  If not,
	 * find a place for it.
	 */

	if( PreHeader->next == NULL )
	{
		if( b == NULL )
		{
			/* Could not locate pre-header after any of its
			 * predecessors (meaning they all get to it
			 * via CBranches!).  Try to put it ANYWHERE!
			 */
			for( b = FirstBlock; b != NULL; b = b->next )
			{
				if( b->code.last != NULL && 
					(IsUBranch(b->code.last) ||
					 IsSwitch(b->code.last) ||
					 IsReturn(b->code.last) ))
			/**/	break;
			}
		}

		/* If we havent found a place by now, we are in BIG trouble
		 */
		assert(b != NULL);
		LinkBlock(PreHeader,b);
	}

	AddPred(HIndex, PHIndex);
	FlowGraph[PHIndex].u.exits[0] = HIndex;
	FlowGraph[PHIndex].nexits = 1;

	DFN_Fix(PHIndex, HIndex);
	DOM_Fix(PHIndex, HIndex);

	return PreHeader;
}

/* Provide DFN for pre-header. Idea here is to insert the pre-header in
 * the DFN array just ahead of the header
 */

static void
DFN_Fix(preheader, header)
	FlowIndex preheader;
	FlowIndex header;
{
	int i;
	int DFN_Header;


	DFN_Header = FlowGraph[header].DFN;
	for( i = NumReachableNodes-1; i > DFN_Header; i-- )
	{
		DFN[i] = DFN[i-1];
		FlowGraph[DFN[i]].DFN = i;
	}
	DFN[i] = preheader;
	FlowGraph[preheader].DFN = i;
}

/* Fix up dominator sets.  Main idea here is that preheader dominates
 * everything the header dominates (including the header itself) and
 * is dominated by everything that dominates the header EXCEPT for
 * the header itself.
 *
 * Likely this for() loop can start at FlowGraph[preheader].DFN
 */

static void
DOM_Fix(preheader, header)
	FlowIndex preheader;
	FlowIndex header;
{
	LongSet dm;
	int i;

	CopySet(FlowGraph[preheader].dom,FlowGraph[header].dom);
	for(i = 0; i < NumReachableNodes; i++ )
	{
		dm = FlowGraph[DFN[i]].dom;
		if(IsElement((int)header, dm))
			Insert((int)preheader, dm);
	}
	DelMember((int)header, FlowGraph[preheader].dom);
}

/*
 * Block "n" contains a branch to the pre-header, block "nto".  Block "n"
 * is also not in the current loop.  Change the branch into a branch to
 * the preheader.
 */
static void
FixBranch(n, nto, preheader, label)
	FlowIndex n;		/* block containing the branch */
	FlowIndex nto;		/* branch was to this block */
	BasicBlock preheader;	/* the pre-header */
	int label;		/* statement number of same */
{
	BasicBlock b;
	BasicBlock header;
	BasicBlock sb;
	Instruction c;
	Instruction l;
	DAG_Node d, right;
	TreeNode t;
	FlowIndex sn;
	int i;

	b = FlowGraph[n].block;
	header = FlowGraph[nto].block;

	/* If the pre-header has not yet been linked in, and if this block
	 * falls through to the header, link the pre-header in here
	 */
	if( preheader->next == NULL && b->next == header )
			LinkBlock(preheader,b);

	for( i = 0; i < FlowGraph[n].nexits; i++ )
	{
		sn = Successor(n, i);
		if( sn == nto )
		{
			DelPred(nto, n);
			AddPred(preheader->FGindex, n);
		}
	}

	l = b->code.last;
	if( IsSwitch(l) )
	{
		for( i = 0; i < l->u.sw.ncases; i++ )
		{
			sb = LookUpLabel((LabelType) l->u.sw.swtable[i].slab);
			if( sb == header )
			{
				l->u.sw.swtable[i].slab = label;
				FlowGraph[n].u.exit_list[i] = preheader->FGindex;
			}
		}
		sb = LookUpLabel(l->u.sw.defaultlabel);
		if( sb == header )
		{
			i = l->u.sw.ncases;
			l->u.sw.swtable[i].slab = label;
			FlowGraph[n].u.exit_list[i] = preheader->FGindex;
		}
	}
#ifdef FORT
	else
	if( IsArithIf(l) )
	{
		for( i = 0; i < NormNumExits; ++i)
		{
			sb = LookUpLabel(l->u.aif.labs[i]);
			if( sb == header )
			{
				l->u.aif.labs[i] = label;
				FlowGraph[n].u.exit_list[i] = preheader->FGindex;
			}
		}
	}
#endif /* FORT */
	else
	if( IsUBranch(l) )
	{
		l->u.lb.lab_num = label;
		FlowGraph[n].u.exits[0] = preheader->FGindex;
	}
	else
	if( IsCBranch(l) )
	{
		/* Find the CBranch and change it.  Have to do this in 2
		 * places:  The original set of instructions, so that we
		 * can build the flow graph correctly, and the DAG, so that
		 * the new generated code will be correct.
		 */

		if(Successor(n,1) == nto )
		{
			for( d = b->Dag; d != NULL; d = d->next )
				if( d->op == CBRANCH )
			/**/		break;

			assert(d != NULL );
			right = d->u.in.right;
			right->u.tn.lval = label;

			t = b->code.last->u.ex.root;
			assert(t->in.op == CBRANCH);
			t->in.right->tn.lval = label;
			FlowGraph[n].u.exits[1] = preheader->FGindex;
		}
		else
		{
			/* The only block allowed to fall through to the
			 * pre-header is the block that is its immediate
			 * predecessor in the original order.
			 */
			assert(b->next == preheader);
			FlowGraph[n].u.exits[0] = preheader->FGindex;
		}
	}
	else
	{
		FlowGraph[n].u.exits[0] = preheader->FGindex;
		if( b->next != preheader )
		{
			c = AddInstruction(UBranch, &b->code);
			c->u.lb.lab_num = label;
		}
	}
}

/*
 * Generate a new label.  Start where pass2 leaves off.
 */

static int LabelNum = LAST_P2_LABEL;

static int
NewLabel()
{
	return(++LabelNum);
}

/*
 * Make up a per-header block that consists of a label and a branch to the
 * loop header
 */
static int
PreHdrBlock(preheader, header)
	BasicBlock preheader;		/* pre-header, already allocated */
	BasicBlock header;		/* loop header */
{
	Instruction c1;			/* pre-header label */
	Instruction c2;			/* pre-header branch to the header */
	Instruction lab;		/* label on the header */

	lab = header->code.first;
	assert( InstType(lab) == Label );

	c1 = AddInstruction(Label, &preheader->code);
	c1->u.lb.lab_num = NewLabel();
	Associate(c1->u.lb.lab_num, preheader);

	c2 = AddInstruction(UBranch, &preheader->code);
	c2->u.ub.target = lab->u.lb.lab_num;

	return c1->u.lb.lab_num;
}

/* This routine computes various pieces of information on a loop and the blocks
 * in it:
 * 1) Are there any indirect pointer stores in the loop?  (A call counts as an
 *    indirect block.
 * 2) For each block in the loop:
 *    a) Is the block an exit from the loop?
 *    b) Does the block dominate all exits from the loop?
 *
 * In addition, various bits of initialization are done.
 */

static int
MarkBlocks(e, etemp)
	LTabEntry e;
	LongSet etemp;		/* temporary set used in determination of
				 * blocks that dominate all loop exits.
				 * Debugging code prints this set out as
				 * the set of eligible blocks
				 */
{
	int i, nblocks;
	FlowIndex n;
	BasicBlock b;
	DAG_Node d;

	CopySet(etemp, e->loop);
	nblocks = 0;
	e->unknown_indir_stores = False;
	e->known_indir_stores = False;

	/* Find and mark blocks from which control might exit the loop.
	 * Do other bits of initialization.
	 */
	for( i = 0; i < NumReachableNodes; i++ )
	{
		n = DFN[i];
		b = FlowGraph[n].block;
		if( IsElement( (int)n, e->loop ) )
		{
			/* An unsophisticated test to determine if this loop
			 * might do indirect stores.  Someday we will have to
			 * be smarter.
			 */

			e->unknown_indir_stores |= b->unknown_indir_stores;
			e->known_indir_stores |= b->known_indir_stores;

			BTable[nblocks].block = b;
			b->in_loop = True;
			b->eligible = False;
			BlockCost(b);
			b->exit = IsElement((int)n, e->exits);

			/* If this block is an exit, update the set of blocks
			 * dominating all exits.  Dom(n) is the set of blocks
			 * dominating this one, including the block itself.
			 */
			if( b->exit )
				Intersection(etemp, etemp, Dom(n));

			for( d = b->Dag; d != NULL; d = d->next )
			{
				d->invariant = False;
				d->never_invariant = False;
			}

			nblocks++;
		}
		else
		{
			b->in_loop = False;
		}
	}

	/* Now mark blocks that dominate all exits */

	for( i = FirstElement(etemp); i != NoElement; i = NextElement(i,etemp))
		FlowGraph[i].block->eligible = True;

	return nblocks;
}

/*
 * Move invariants into the pre-header of a loop.
 * General strategy: Move all computations and assignments from blocks
 * that dominate all loop exits.  From other blocks, move only computations
 * that cannot cause errors, and do this only if we have not been told
 * to be conservative.
 */

static void
MoveInvariants(e, d, newdefs)
	LTabEntry e;		/* Information on this loop */
	DAG_Node d;		/* chain of DAG nodes found loop invariant */
	LongSet newdefs;	/* set of definitions created in the pre-header
				 */
{
	BasicBlock b;
	DAG_Node dnext, dnew;
	int nmoved, i;

	nmoved = 0;
	InitDAG();
	ResetTable();
	InitHash();
	for( ; d != NULL; d = dnext )
	{
		dnext = d->chain.loop_invar;
		b = FlowGraph[d->FGindex].block;
		if( ldebug > 5 )
		{
			printf("Node %d (%x, block # %d) marked invariant, ",
					d->order, d, b->blocknum);
			printf("%seligible\n",
					b->eligible ? "" : "not ");
		}

		if( MayLift(b,d) )
		{
			d->chain.loop_invar = dnew = NewNode(d,e);
			if( dnew != NULL )
			{
				/* A new DAG node has been created.  Move
				 * attached identifiers only if block
				 * dominates all exits.  Remember this
				 * node
				 */
				if( b->eligible && d->in_cond == NotConditional )
					LiftAssigns(d, dnew, newdefs);
				save_invar[nmoved++] = d;
			}
		}
		else
			d->chain.loop_invar = NULL;
	}

	/* Now that we have created the DAG for the pre-header, run around
	 * modifying the DAGs within the loop to use it.  This takes several
	 * passes over things.  First, we run over the pre-header initializing
	 * all the loop_invar links to null.  (We cannot do this as the DAG is
	 * built because this field is used for hash links.)  Second, we run
	 * down the list of loop-invariant nodes cloned in the preheader, and
	 * attach each node in the list to the appropriate node in the
	 * preheader.  Third, we run over the pre-header again, making a list
	 * of preheader nodes in reverse order.  Finally, we run over this
	 * list, modifying DAGs within the loop where appropriate.
	 */

	e->prehead->Dag = dnew = FirstNodeCreated;
	InitFGindex(e->prehead);

	for(; dnew != NULL ; dnew = dnew->next )
		dnew->chain.loop_invar = NULL;

	while( nmoved > 0 )
	{	
		d = save_invar[--nmoved];
		dnew = d->chain.loop_invar;
		d->chain.loop_invar = dnew->chain.loop_invar;
		dnew->chain.loop_invar = d;
	}

	BlockCost(e->prehead);
	BlockRefs(e->prehead);

	/* Need to run over preheader in reverse order */

	i = 0;
	for( dnew = e->prehead->Dag; dnew != NULL; dnew = dnew->next )
	{
		if( i >= save_size )
			InitSave(i+1);
		save_invar[i++] = dnew;
	}

	while( i > 0 )
		TryPreHeader(e, save_invar[--i]);
}

/*
 * For each DAG node "dnew" in the pre-header, change all nodes in the
 * loop to point to it (where appropriate), provided it is cheaper to do
 * this.
 */

static void
TryPreHeader(e, dnew)
	LTabEntry e;		/* Information on this loop */
	DAG_Node dnew;		/* The node in the preheader */
{
	Operator ns;
	Boolean MovedIt;	/* Did we actually move anything? */
	int ref;

	assert(dnew != NULL );

	/* If the Pre-header contains expression lifted out of some containing
	 * loop then decisions on whether or not to use the carrier have
	 * already been made.
	 */

	if( dnew->prev_valid_carrier )
	{
/**/		return;
	}
	else
	if( dnew->carrier == NoId )
	{
		MovedIt = Try2(e, dnew, LTEMP);
		if( !MovedIt )
			MovedIt = Try2(e, dnew, TREG);
	}
	else
	{
		ns = IdOp(dnew->carrier);
		MovedIt = Try2(e, dnew, ns);
		if( !MovedIt && ns == LTEMP )
			MovedIt = Try2(e, dnew, TREG);
	}

	/* If this node no longer does anything, try to prune it away
	 * and adjust reference counts for this block.
	 */
	if( !MovedIt )
	{
		PruneTree(dnew, NotConditional, !RecursivePrune);
		AdjustRefs(dnew);
	}
}

static Boolean
Try2(e, dnew, ns)
	LTabEntry e;	/* Information on this loop */
	DAG_Node dnew;	/* DAG node in preheader */
	Operator ns;	/* name space of carrier on dnew */
{
	DAG_Node d;
	CostType cost_after, cost_now, refcost;
	int d_weight, dnew_weight;
	int w_refs;		/* weighted reference count */

	if( dnew->op == LNAME || dnew->op == PNAME )
	{
		/* Do not put transparent temps on things that are themselves
		 * eligible for registers.
		 */
		return False;
	}

	dnew_weight = FlowGraph[dnew->FGindex].loopweight;

	cost_now = 0;
	cost_after = 0;

	/*
	 * If there is no carrier on the DAG node now, we will have to add
	 * one.  This costs.  If there is a carrier (perhaps we have lifted
	 * an assignment), then we would have to do the store anyway, and
	 * the cost of the evaluation and store should not be included.
	 * Further, if there are references to the preheader node in the
	 * preheader, the computation will have to be done anyway, so
	 * do not charge its cost.
	 */

	if( dnew->carrier == NoId )
	{
		cost_after += StoreCost(ns,dnew->type) * dnew_weight;
		if( dnew->refs == 0 )
			cost_after += dnew->cost * dnew_weight;
	}

	/*
	 * Now run down the list of all DAG nodes in the loop that might
	 * use this preheader node.  For each of these nodes, compute the
	 * current cost of doing the computation and the cost if we move
	 * the computation to the pre-header.
	 */

	for( d = dnew->chain.loop_invar; d != NULL; d = d->chain.loop_invar )
	{
		assert(d->FGindex != NullFlow);
		d_weight = FlowGraph[d->FGindex].loopweight;

		/* Handle special case where carrier in loop is guaranteed
		 * to be a register
		 */

		/*	we evaluate the node refs times, and once to
		 *	store into attached ids, if any.
		 */

		w_refs = d_weight * (d->refs + ((d->attached==NULL) ? 0 : 1));
		if( d->carrier != NoId && IdOp(d->carrier) == REG )
			cost_after += RefCost(REG,d->type) * w_refs;
		else
			cost_after += RefCost(ns,d->type) * w_refs;

		if( d->carrier == NoId || IsTransparent(d->carrier) )
		{
			/* Expression gets re-evaluated each & every time.
			 * Work out cost of doing computation in the
			 * preheader and stashing result in an LTEMP (or
			 * permanent carrier if there was one).
			 */

			cost_now += d->cost * d->refs * d_weight;
		}
		else
		{
			/* Expression gets re-evaluated once through the loop
			 * and then carrier is used.  The stores into the
			 * carrier are fixed overhead and do not count.
			 * If it is cheaper to recompute than to reference the
			 * carrier, assume that the code generator will do this.
			 */
	
			refcost = IdRefCost(d->carrier);
			if( refcost > d->cost )
				refcost = d->cost;
			cost_now += (d->cost + refcost * d->refs) * d_weight;
		}
	}

	if( cost_now > cost_after )
	{
		UsePreHeader(e, dnew);
		return True;
	}
	else
		return False;
}

/*
 * Using the pre-header node is cheaper than using the original DAG node.
 * Arrange for something to carry its value, and record the new definition.
 */
static void
UsePreHeader(e, dnew)
	LTabEntry e;		/* Information on this loop */
	DAG_Node dnew;		/* the preheader node */
{
	DAG_Node d;
	Identifier id;
	int r;

	if( (id = dnew->carrier) == NoId )
	{
		/*
		 * Have no way to get at value - create one.
		 * Expand the type if needed.
		 * Allocate a temp, enter it in the table
		 * and update the UD and DU information for
		 * it. 
		 */
		id = GetLTemp(TempType(dnew->type));
		AddLTempList(e, id);
		(void)AttachID(id, dnew);
		dnew->carrier = id;
		SetNodeID(id, dnew);
		AddTempDef(id, dnew);

		/* Set things up so that this temp can be removed
		 * later if it does not go in a register.
		 */

		MarkTransparent(id);

		if( ldebug > 3 )
			printf("Attaching temp to %x\n", dnew);
	}

	for( d = dnew->chain.loop_invar; d != NULL; d = d->chain.loop_invar)
	{
		if(d->refs == 0 && d->attached == NULL && d->delay_count == 0)
		{
	/**/		continue;
		}

		/* Modify the carrier on this node and update DU information
		 * and reference counts.
		 *
		 * If the carrier is a register, this code replaces it
		 * with a loop temp.  If the loop temp does not go into a
		 * register, and if the expression is simple, the loop temp
		 * will be removed. At this point we have lost all knowledge
		 * of the previous (REG) carrier.  This is poor, and should
		 * be fixed.
		 */

		d->carrier = id;
		d->prev_valid_carrier = True;
		if( IdOp(id) != LTEMP )
			Gen1Use(id,d,FlowGraph[d->FGindex].block);
		AdjustRefs(d);
	}
}

/*
 * Create a new pre-header DAG node corresponding to DAG node d.  
 * Do not create the node if there are children for which we have not
 * created nodes.
 */

static DAG_Node
NewNode(d,e)
	DAG_Node d;		/* DAG node to clone */
	LTabEntry e;		/* Information on this loop */
{
	DAG_Node left;
	DAG_Node right;
	DAG_Node dnew;
	Operator op;

	switch(optype(d->op))
	{
	case LTYPE:
		switch(d->op)
		{
		case TNAME:
		case TREG:
			/* These can never be lifted - don't even bother
			 * asking
			 */
			return(NULL);

		case LTEMP:
			/* These can always be lifted, and have a unique
			 * definition
			 */
			dnew = CopyLeaf(d->leaf_id, d->op, d->type, e->prehead);
			break;

		case ICON:
#ifndef MPX
		case TCON:	/* (MEY) (22-Jan-87) add for Alpha C */
		case HCON:
		case CADDR:
#endif
		case LABCON:
		case ADDR:
		case LADDR:
		case PADDR:
		case STADDR:
			dnew = CopyLeaf(d->leaf_id, d->op, d->type, e->prehead);
			break;

		case NAME:
		case LNAME:
		case PNAME:
		case STATNAME:
		case REG:
			/* The call to IdInvariant() here fixes the
			 * problem exemplified by the following code:
			 *
			 *	for(...) {
			 *	t = x + y;
			 *	...
			 *	x = something loop invariant;
			 *	...
			 *	s = x - y;
			 *	}
			 *
			 * If this is the only definition of x inside
			 * the loop, it will be marked loop invariant,
			 * and so will the second use of x.  When it
			 * comes time to build the pre-header, the
			 * assignment to x will not be lifted because
			 * the definition of x is not the only one
			 * reaching the first use.  HOWEVER, the
			 * second use will be (incorrectly) lifted,
			 * having been marked loop invariant.  Thus,
			 * we must insist on strict invariance, and
			 * check it here.  If there was a single def
			 * inside the loop and it was moved outside the
			 * loop, the UD information wil have been
			 * updated to reflect this
			 */

			if( !IdInvariant(d->leaf_id, d, True) )
/**/				return(NULL);
			dnew = CopyLeaf(d->leaf_id, d->op, d->type, e->prehead);
			break;

		case FCON:
			dnew = FconDAG( d->op, d->type, d->u.fpn.dval, False );
			break;

		case STLABEL:
			dnew = CreateNode(d->op, d->type, False);
			dnew->u.tn.lval = d->u.tn.lval;
			dnew->u.tn.rval = d->u.tn.rval;
			dnew->hash = dnew->order;
			break;

		default:
			InternalFault("loop.c: invalid leaf op: %d", d->op);
		}
		break;

	case UTYPE:
		left = d->u.in.left->chain.loop_invar;
		if( left == NULL )
/**/			return(NULL);
		dnew = CopyUnary(d->op, d->type, left, d->u.tn.rval);
		break;

	case BITYPE:
		right = d->u.in.right->chain.loop_invar;
		if( right == NULL )
/**/			return(NULL);
		left = d->u.in.left->chain.loop_invar;
		if( left == NULL )
/**/			return(NULL);

		dnew = BinaryHashLook(d->op, d->type, left, right);
		if( dnew == NULL )
		{
			dnew = CreateNode(d->op, d->type, False);
			dnew->u.in.right = right; ++right->in_degree;
			dnew->u.in.left = left; ++left->in_degree;
			dnew->hash = dnew->order;
			EnterHash(dnew);
		}
		break;

	default:
		InternalFault("loop.c: Bad optype for op %d", d->op);
	}


	/*
	 *	Handle special case ST* operators
	 */

	op = dnew->op;
	if ((op == STASG || op == STARG || op == STCALL ||
		     op == UNARY STCALL)
		&& (dnew->new_tree == NULL)) {
		assert(d->new_tree != NULL);
		dnew->new_tree = TreeAllocate();
		*(dnew->new_tree) = *(d->new_tree);
	}

	if( d->prev_valid_carrier )
	{
		dnew->carrier = d->carrier;
		dnew->prev_valid_carrier = True;
	}

	dnew->indirect = d->indirect;
	dnew->line_number = d->line_number;
	dnew->file_name = d->file_name;
	dnew->FGindex = e->prehead->FGindex;

	return dnew;
}

static DAG_Node
CopyLeaf(id, op, type, b)
	Identifier id;
	Operator op;
	TWORD type;
	BasicBlock b;
{
	DAG_Node dnew;
	TWORD tnew;
	DAG_Node left;
	TreeNode t;

	dnew = NodeID(id);
	if( dnew == NULL )
	{
		/* If this was an ICON created by constant folding, it
		 * may have a strange type.  We must enter it in the
		 * pre-header with a consistent type so that others
		 * can find it.
		 */
		if( op == ICON && (tnew = IconBaseType(type)) != type )
		{
			t = IdTree(id);
			assert(t != NULL);
			id = EnterSymbol(ICON, t->tn.lval, tnew, (char *)NULL, True);
			if( IdTree(id) == NULL )
			{
				/* Brand new ICON that we haven't seen before.
				 * Need to save tree rep. for it in symbol
				 * table */

				t = CopyTree(t);
				t->in.type = tnew;
				SetIdTree(id, t);
				TreeFree(t);
			}
			left = CopyLeaf(id, ICON, tnew, b);
			dnew = CopyUnary(OCONVLEAF, type, left, 0);
		}
		else
		{
			dnew = CreateNode(op, type, False);
			SetNodeID(id, dnew);
			dnew->leaf_id = id;
			dnew->hash = dnew->order;
			SetLeafRef(id, dnew);

			/* For non-constants, UD chains must note that they
			 * might be referenced here */

			switch( op )
			{
			case REG:
			case NAME:
			case LNAME:
			case PNAME:
			case STATNAME:
				Gen1Use(id,dnew,b);
				break;
			}
		}
	}
	return dnew;
}

static DAG_Node
CopyUnary(op, type, left, right)
	Operator op;
	TWORD type;
	DAG_Node left;
	int right;
{
	DAG_Node dnew;

	dnew = UnaryHashLook(op, type, left, right);
	if( dnew == NULL )
	{
		dnew = CreateNode(op, type, False);
		dnew->u.tn.rval = right;
		dnew->u.in.left = left; ++left->in_degree;
		dnew->hash = dnew->order;
		EnterHash(dnew);
	}

	return dnew;
}

/*
 * Transfer attached identifiers from DAG node "d" to DAG node "dnew" in the
 * pre-header.  An attached identifier can be moved if
 * (i) there are no other definitions of the identifier inside the loop
 * and
 * (ii) all uses of this identifier in the loop are reachable only by this
 *      definition
 */
static void
LiftAssigns(d, dnew, newdefs)
	DAG_Node d;		/* the original DAG node */
	DAG_Node dnew;		/* the new DAG node */
	LongSet newdefs;	/* the set of all new definitions created
				 * by loop invariant motion
				 */
{
	AttachedID aid, anext;

	for( aid = d->attached; aid != NULL; aid = anext )
	{
		anext = aid->next;

		/* Don't bother looking at temps - info may be missing.
		 * If globals are evil, don't bother looking at NAMEs
		 */

		switch(IdOp(aid->id))
		{
		case NAME:
			if( GoodGlobals )
				MoveAttached(aid->id, d, dnew, newdefs);
			break;
		case LNAME:
		case PNAME:
		case STATNAME:
		case REG:
			MoveAttached(aid->id, d, dnew, newdefs);
			break;
		}
	}
}

/* Try to move the assignment to a single identifier */

static void
MoveAttached(id, d, dnew, newdefs)
	Identifier id;		/* The identifier we are trying to move */
	DAG_Node d;		/* the original DAG node */
	DAG_Node dnew;		/* the new DAG node */
	LongSet newdefs;	/* the set of all new definitions created
				 * by loop invariant motion
				 */
{
	AttachedID aid;
	DAG_Node dagp;
	BasicBlock b;
	LongSet sdefs, suses;
	int i;
	int thisdef;

	sdefs = GetDefs(id);
	suses = GetUses(id);
	assert(sdefs != NULL && suses != NULL);
	thisdef = NoElement;

	/* look for other definitions inside the loop */

	for( i = FirstElement(sdefs); i != NoElement; i = NextElement(i,sdefs) )
	{
		/* If dagp->FGindex is NullFlow, this def is the
		 * "Undefined" definition, or is in an unreachable
		 * block.  In any case, it is safe to ignore it
		 */

		dagp = UDDagPtrs[i];
		if( dagp->FGindex != NullFlow )
		{
			b = FlowGraph[dagp->FGindex].block;
			if( b->in_loop )
			{
				if( dagp == d )
					thisdef = i;
				else
	/**/				break;
			}
		}
	}

	if( i != NoElement )
	{
		if( ldebug > 5 )
			printf("No move for id %d - multiple defs in loop\n", id);
/**/		return;
	}
	assert(thisdef != NoElement);

	/* now look at all uses of this identifier inside the loop */
	/* NOTE: some of the uses may have been stomped on by constant
	 * folding
	 */

	for( i = FirstElement(suses); i != NoElement; i = NextElement(i,suses) )
	{
		dagp = DUDagPtrs[i];
		if( dagp->FGindex != NullFlow )
		{
			b = FlowGraph[dagp->FGindex].block;
			if( b->in_loop && ReachedByOther(dagp, thisdef, sdefs))
	/**/			break;
		}
	}

	if( i != NoElement )
	{
		if( ldebug > 5 )
			printf("No move for id %d - multiple reaching defs at node %x\n", id, dagp);
/**/		return;
	}

	/*
	 * Finally ensure that there is no other variable with the
	 * same offset.
	 */

	if( !ForceUnique(id))
	{
		if( ldebug > 5 )
			printf("No move for id %d - aliased\n", id);
/**/		return;
	}

	/* We've got a winner.  Attach the id to the node in the pre-
	 * header and remove it from the node in the loop.  Set the
	 * carrier in the loop to the carrier in the preheader, and
	 * note that this carrier is defined elsewhere.  (This includes
	 * fixing up ref counts on children IF needed.)  Change the
	 * UD pointer to reflect the change and remember that this
	 * was done so that we can update the UD information later.
	 * Update live-dead information about this variable.
	 * Also update symbol table information, and check that there
	 * is not already a leaf (or assignment!) for this id in the header.
	 * The check is defensive; the assertion should "never" fire
	 */

	b = FlowGraph[d->FGindex].block;

	DelMember((int)id, b->ld.Def);
	DelMember((int)id, b->ld.PDef);

	aid = AttachID(id, dnew);
	DeleteID(id, d, True);
	if( dnew->carrier == NoId || IdOp(id) == REG )
		dnew->carrier = id;

	if( d->carrier != dnew->carrier )
	{
		switch(IdOp(id))
		{
		case REG:
		case NAME:
		case LNAME:
		case PNAME:
		case STATNAME:
			/* Update DU and LD information about carriers if
			 * necessary */
			Gen1Use(id,d,b);
			DelMember((int)d->carrier, b->ld.Use);
			Insert((int)id, b->ld.Use);
		}
		d->carrier = dnew->carrier;
	}
	d->prev_valid_carrier = True;

	assert(NodeID(id)==NULL);
	SetNodeID(id,dnew);

	UDDagPtrs[thisdef] = dnew;
	Insert(thisdef, newdefs);
	aid->UDIndex = thisdef;

	AdjustRefs(d);

	if( ldebug > 3 )
		printf("Move attached id %d from node %x to node %x\n",
			aid->id, d, dnew);
}

/*
 * Is the use of an identifier at dag node dagp reached by anything other
 * than def "thisdef".  "sdefs" is the set of all defs for this identifier
 */
static Boolean
ReachedByOther(dagp, thisdef, sdefs)
	DAG_Node dagp;
	int thisdef;
	LongSet sdefs;
{
	
	switch(dagp->op)
	{
	case UNARY MUL:
		assert(dagp->is_fetch); /*store is not a use of an identifier*/

		/* FALL THROUGH */
	case UNARY CALL:
	case UNARY FORTCALL:
	case UNARY STCALL:
	case CALL:
	case FORTCALL:
	case STCALL:
	case REG:
	case NAME:
	case LNAME:
	case PNAME:
	case STATNAME:
		assert(dagp->In != NULL);
		Intersection(UDtmpset, sdefs, dagp->In);
		DelMember(thisdef, UDtmpset);
		return Cardinality(UDtmpset) != 0;

	case TNAME:
	case TREG:
	case LTEMP:
		/* should never be called for TNAME, TREG or LTEMP */
		InternalFault("ReachedByOther called on op %d", dagp->op);
		/* NOTREACHED */

	case LEAFNOP:
		/* A use of an identifier that was canned */
		return False;

	default:
		/* Something that could not ordinarily be a use of an
		 * identifier.  It could have been put here by constant
		 * folding.  If it is not a literal, there is something funny
		 * going on.
		 */
		if( !Literal(dagp->op) )
			InternalFault("ReachedByOther called on op %d", dagp->op);
		return False;
	}
	/* NOTREACHED */
}

/*
 * Propagate invariants through the flow graph
 */
static Boolean
PropagateInvariants(e, b, changed)
	LTabEntry e;
	BasicBlock b;
	Boolean changed;
{
	Boolean new_inv;
	Boolean never_inv;
	DAG_Node d;

	for( d = b->Dag; d != NULL; d = d->next )
	if( !d->invariant && !d->never_invariant && !callop(d->op) )
	{
		never_inv = False;
		new_inv = False;
		switch(d->op)
		{
		case ICON:
#ifndef MPX
		case TCON:	/* (MEY) (22-Jan-87) add for Alpha C */
		case HCON:
		case CADDR:
#endif
		case LABCON:
		case FCON:
		case ADDR:
		case LADDR:
		case PADDR:
		case STADDR:
		case LTEMP:		/* Because loops are processed outside
					 * in, any LTEMPs we see must be loop
					 * invariant */
			new_inv = True;
			break;

		case STASG:
		case STARG:		/* STARGs must appear in calls, and
					 * hence are never loop invariant */
		case LEAFNOP:
		case CM:
		case CBRANCH:
		case STLABEL:
		case FORCE:
		case COLON:		/* FOR NOW */
		case TNAME:
		case TREG:
			never_inv = True;
			break;

		case NAME:
		case LNAME:
		case PNAME:
		case STATNAME:
		case REG:
			new_inv = IdInvariant(d->leaf_id, d, False);
			break;

		case UNARY MUL:
			if( d->is_fetch && !e->unknown_indir_stores &&
				(!e->known_indir_stores ||
				 (d->indirect != NoId && IdInvariant(d->indirect, d, True))) )
			{
				new_inv = d->u.in.left->invariant;
				never_inv = d->u.in.left->never_invariant;
			}
			else
				never_inv = True;
			break;

		default:
			switch(optype(d->op))
			{
			case BITYPE:
				new_inv = d->u.in.left->invariant &&
						d->u.in.right->invariant;
				never_inv = d->u.in.left->never_invariant ||
						d->u.in.right->never_invariant;
				break;

			case UTYPE:
				new_inv = d->u.in.left->invariant;
				never_inv = d->u.in.left->never_invariant;
				break;

			case LTYPE:
#ifndef MPX
    			break;
#else
				InternalFault("loop.c: Unrecognized leaf in PropagateInvariants, op=%d", d->op);
#endif

			default:
				InternalFault("loop.c: Invalid optype, op=%d", d->op);
			}
			break;
		}

		if( new_inv )
		{
			/* We have just marked this node as invariant.
			 * Add it to the list of invariant nodes
			 */

			d->invariant = True;
			changed = True;
			ninvar++;
			d->chain.loop_invar = NULL;
			if( first == NULL )
				first = d;
			else
				last->chain.loop_invar = d;
			last = d;
		}
		else
		if( never_inv )
		{
			d->never_invariant = True;
		}
	}

	return changed;
}

/*
 * Determine if the identifier "id"  is strictly loop-invariant or
 * loop-invariant at node "d".  (d is either a leaf whose leaf_id is id
 * or a UNARY MUL fetch).
 * It is loop-invariant if
 * (i) all definitions reaching it are outside the loop
 * or
 * (ii) it is reached by exactly one definition and that definition is
 *      marked loop invariant and (as far as we can tell) will be moved
 *	out of the loop.
 * It is strictly loop-invariant only if (i) is true.
 *
 * If there are two or more reaching definitions and at least one is inside
 * the loop, we mark the node so that we never have to come back here again
 * in this loop
 */

static Boolean
IdInvariant(id, d, strict)
	Identifier id;
	DAG_Node d;
	Boolean strict;
{
	LongSet sdefs;
	DAG_Node defp;
	int i, numdefs;
	BasicBlock b;
	int space;

	/* If globals may change under our feet, they can never be loop
	 * invariant
	 */

	if( !GoodGlobals )
	{
		space = IdOp(id);
		if( space == NAME || ((space == LNAME || space == PNAME || space == STATNAME) &&
					WasAddressed(id)))
		{
			d->never_invariant = True;
/**/			return False;
		}
	}
	sdefs = GetDefs(id);
	assert(sdefs != NULL && d->In != NULL);
	Intersection(UDtmpset, sdefs, d->In); /* this Should be done statically */
	numdefs = Cardinality(UDtmpset);

	switch(numdefs)
	{
	case 0:
/**/		return True;

	case 1:
		/* If this definition is unreachable (or the "Undefined"
		 * definition that each identifier has initially), then
		 * it is certainly invariant
		 */
		defp = UDDagPtrs[FirstElement(UDtmpset)];
		if( defp->FGindex == NullFlow )
/**/			return True;

		/* If this definition is outside the loop, then d is loop
		 * invariant.  If it is inside the loop, then it is not
		 * strictly invariant.  However it is invariant if the
		 * definition is inside the loop, marked invariant, and
		 * eligible for motion outside the loop.
		 */
		b = FlowGraph[defp->FGindex].block;
/**/		return !b->in_loop || (!strict && defp->invariant &&
					MayLift(b,defp));

	default:
		for( i = FirstElement(UDtmpset); i != NoElement; i = NextElement(i,UDtmpset) )
		{
			defp = UDDagPtrs[i];
			if( defp->FGindex != NullFlow )
			{
				b = FlowGraph[defp->FGindex].block;
				if( b->in_loop )
				{
					d->never_invariant = True;
/**/					return False;
				}
			}
		}
		return True;
	}
	/*NOTREACHED*/

}






/*
 * Manipulation of loop temporary lists
 */

static LTempList LTFree = NULL;		/* Loop temp free list */

/*
 * Add identifier "id" to list of loop temporaries for loop "e"
 */

static void
AddLTempList(e, id)
	LTabEntry e;
	Identifier id;
{
	LTempList l, lchunk;
	int i;

	if( LTFree == NULL )
	{
		lchunk = GetArray(s_Templist, LTEChunk, LTmE_type);
		CheckStorage(lchunk, "storage for loop temp data", 0);

		l = NULL;
		for( i = 0; i < LTEChunk; i++ )
		{
			lchunk->next = l;
			l = lchunk++;
		}
	}
	else
		l = LTFree;

	LTFree = l->next;
	l->id = id;
	l->next = e->ltemps;
	e->ltemps = l;
}

/*
 * Free up a list of loop temporaries
 */

void
FreeLTempList(l)
	LTempList l;
{
	LTempList lend;

	if( l != NULL )
	{
		/* Find the end of the list and insert the whole mess on the
		 * start of the free list
		 */

		lend = l;
		while( lend->next != NULL )
			lend = lend->next;
		lend->next = LTFree;
		LTFree = l;
	}
}

/* Initialize space for work array */

static void
InitSave(n)
	int n;		/* size of array requested. */
{

	n += SAVE_SLOP;		/* allow a bit of breathing room */
	if( save_invar == NULL )
	{
		save_invar = GetArray(s_Invariants, n, DAG_Node);
		CheckStorage(save_invar, "storage for loop invariant work array.  New size %d", n);
	}
	else
	if( n > save_size )
	{
		save_invar = Reallocate(s_Invariants, save_invar, save_size,
					n, DAG_Node);
		CheckStorage(save_invar, "storage for loop invariant work array.  new size %d", n);
	}
	save_size = n;
}
