/* gcomsubex.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: gcomsubex.c,v 5.5 89/05/12 12:50:43 pcc Rel-3_0 $";
/* static char ID[] = "@(#)gcomsubex.c	15.6	of 86/12/04"; */
#endif

/*
 *	Global Common Subexpressions
 *	See Aho & Ullman, Principles of Compiler Design, section 14.2
 *
 */

# include <assert.h>
# include <availexpr.h>
# include <blocks.h>
# include <bool.h>
# include <cost.h>
# include <dag.h>
# include <daghash.h>
# include <dagsymbol.h>
# include <erroro.h>
# include <flow.h>
# include <gcomsubex.h>
# include <identifier.h>
# include <livedead.h>
# include <longset.h>
# include <option.h>
# include <pcc.h>
# include <refcount.h>
# include <storage.h>
# include <temp.h>

/*
 * Export
 */

int GlComdebug;
int NoGlobalCommon;

/*
 * Private
 */

static Boolean workdone;	/* did global common subexprs do anything? */

/*
 * Forward
 */

static void	CommonExpression();
static DAG_Node	FindDefPoint();
static CostType	GComBenefit();
static void	MakeGCSEHappen();
static void	CommonCleanup();

void
GlobalCommon()
{
	int i, j;
	BasicBlock b;
	AvailIndex *save_exprs;
	DAG_Node *defpoints;
	FlowIndex *flowstack;
	int save_index;

	if( NoGlobalCommon > 0 )
		return;

	workdone = False;
	save_exprs = GetArray(s_GlComm, NumAvailableExpressions, AvailIndex);
	CheckStorage(save_exprs, "storage to save %d available expressions", NumAvailableExpressions);
	
	flowstack = GetArray(s_GlComm, NumFlowNodes, FlowIndex);
	CheckStorage(flowstack, "storage for flowstack for global common subexpressions", 0);

	defpoints = GetArray(s_GlComm, NumFlowNodes, DAG_Node);
	CheckStorage(defpoints, "storage to save %d definition points", NumFlowNodes);

	ComputeCosts();
	InitRefs();

	for( i = 0; i < NumReachableNodes; i++ )
	{
		/* For each block, compute the set of possible global common
		 * subexpressions as the intersection of av.In and av.Comp.
		 * Be conservative and exclude anything that was killed
		 * in the block.
		 */

		b = FlowGraph[DFN[i]].block;
		b->gcse.Pos = CreateSet(NumAvailableExpressions);
		Intersection(b->gcse.Pos, b->av.In, b->av.Comp);
		Difference(b->gcse.Pos, b->gcse.Pos, b->av.Kill);
	}

	for( i = NumReachableNodes - 1; i >= 0; i-- )
	{
		save_index = 0;
		b = FlowGraph[DFN[i]].block;

		/* To catch ICONs & such, start this search earlier.  (Needs
		 * other mods too).
		 */

		for( j = NextElement((int)AELastIdentifier,b->gcse.Pos);
		     j != NoElement;
		     j = NextElement(j,b->gcse.Pos) )
		{
			save_exprs[save_index++] = (AvailIndex) j;
		}

		/* Now run through potential common subexpressions in reverse
		 * order
		 */

		while( save_index > 0 )
		{
			if( GlComdebug > 1 )
				printf("Block %d potential common subexpr %d\n",
					b->blocknum, save_exprs[save_index-1]);

			CommonExpression(save_exprs[--save_index], b->FGindex,
					flowstack, defpoints);
		}
	}

	for( i = 0; i < NumReachableNodes; i++ )
	{
		b = FlowGraph[DFN[i]].block;
		DestroySet(b->gcse.Pos);
	}

	DecreaseSpace(s_GlComm, NumAvailableExpressions * sizeof(AvailIndex));
	free((char *)save_exprs);
	DecreaseSpace(s_GlComm, NumFlowNodes * sizeof(FlowIndex));
	free((char *)flowstack);
	DecreaseSpace(s_GlComm, NumFlowNodes * sizeof(DAG_Node));
	free((char *)defpoints);

	/* If we did anything, need to update live-dead info */

	if( workdone )
		LDDataFlow();
}

/*
 * "Aindex" represents an available expression in some block.  It is
 * generated in the block, and also appears in In for this block, and so
 * is a potential global common subexpression.  Determine if turning it
 * into a gce. is worthwhile, and if it is, do it.
 *
 * The suggestion in Aho and Ullman that reaching definitions be used in
 * this determination does not apply when expressions can be anonymous, and
 * in any case is just plain WRONG.  Aho, Sethi and Ullman get it right.
 */

static void
CommonExpression(Aindex, foundhere, flowstack, defpoints)
	AvailIndex Aindex;	/* available expression index of potential
				 * global common subexpr */
	FlowIndex foundhere;
	FlowIndex *flowstack;	/* scratch array to use as stack */
	DAG_Node *defpoints;	/* scratch array to keep track of def points */
{
	PredNode p;
	int i, j;
	FlowIndex n, nprime;
	BasicBlock b, bprime;
	DAG_Node d;
	Identifier id;		/* Temporary created to hold value */
	TWORD ty;		/* Type of temporary */
	CostType gcse_benefit;	/* Benefit of storing this gcse & using the
				 * stored value later */

	/* First phase - find all blocks that generate this available
	 * expression such that either
	 * 1) the expression reaches "thisnode"
	 * 2) the expression reaches another node generating the available
	 *	expression, and that node satisfies (1) or (2).
	 *
	 * We use the following algorithm for this:
	 *
	 *	Push block for thisnode on stack, and mark it visited
	 *	While the stack is not empty
	 *		Pop a block off the stack.
	 *		If the available expression is in In for this block
	 *		 and not in Kill
	 *			Add to the stack all predecessors of this
	 *			block that have not already been visited.
	 *			Mark all such predecessors as visited.
	 *
	 * The code below also does some of bookkeeping in blocks that
	 * compute the expression.
	 */

	for( i = 0; i < NumReachableNodes; i++ )
	{
		n = DFN[i];
		FlowGraph[n].visited = False;
		defpoints[n] = NULL;
	}

	i = 0;
	FlowGraph[foundhere].visited = True;
	flowstack[i++] = foundhere;
	if( GlComdebug > 3 )
		printf("first node stacked: %d\n", foundhere);

	while( i > 0 )
	{
		n = flowstack[--i];
		b = FlowGraph[n].block;

		if( IsElement((int)Aindex, b->av.In) &&
		   !IsElement((int)Aindex, b->av.Kill) )
		{
			for( p = FlowGraph[n].preds; p != NULL; p = p->next )
			    if( !FlowGraph[p->p].visited &&
				FlowGraph[p->p].block->reachable )
			    {
				if( GlComdebug > 3 )
					printf("stack: %d\n", p->p);
				FlowGraph[p->p].visited = True;
				flowstack[i++] = p->p;
			    }

		}
		else
		{
			/* This block will be a defintion point for the
			 * temporary that we will add to carry the value of
			 * the global common subexpression.  Figure out which
			 * DAG node constitutes the definition.
			 * There are some circumstances under which we cannot
			 * attach the temporary.  If this happens, give up.
			 */

			defpoints[n] = d = FindDefPoint(Aindex, b);
			if( d == NULL )
			{
				CommonCleanup(Aindex);
/**/				return;
			}
			ty = d->type;
		}

		if( IsElement((int)Aindex, b->av.Out) )
		{
			for( j = 0; j < NumExits(n); j++ )
			{
				nprime = Successor(n,j);
				bprime = FlowGraph[nprime].block;
				if( bprime->reachable &&
				   !FlowGraph[nprime].visited &&
				    IsElement((int)Aindex, bprime->av.In) &&
				   !IsElement((int)Aindex, bprime->av.Kill) )
				{
					if( GlComdebug > 3 )
						printf("stack: %d\n", nprime);
					FlowGraph[nprime].visited = True;
					flowstack[i++] = nprime;
				}
			}
		}
	}


	/* If this will cost too much, return. Try as an LNAME
	 * first, then as a REG
	 */


	ty = TempType(ty);
	gcse_benefit = GComBenefit(Aindex, LNAME, defpoints);

	if( gcse_benefit > (RefCost(LNAME,ty) + StoreCost(LNAME,ty)) )
	{
		id = GetTemp(LNAME, ty, 'v');
	}
	else
	if( gcse_benefit > 0  ||  GComBenefit(Aindex, REG, defpoints) > 0 )
	{
		id = GetTemp(LNAME, ty, 'v');
		MarkTransparent(id);
	}
	else
	{
		CommonCleanup(Aindex);
/**/		return;
	}

	if( GlComdebug > 0 )
		printf("Create temp: id %d for A.Ex. %d\n", id, Aindex);

	MakeGCSEHappen(Aindex, id, defpoints);
	CommonCleanup(Aindex);

	workdone = True;
}

static DAG_Node
FindDefPoint(Aindex, b)
	AvailIndex Aindex;
	BasicBlock b;
{
	DAG_Node ddef, d;

	assert(IsElement((int)Aindex, b->av.Gen)); /* must be gen. somewhere */

	ddef = NULL;

	if( IsElement((int)Aindex, b->av.Kill) )
	{
		/* The expression is killed in this block, and is generated
		 * later.  In the absence of more information, the last
		 * point at which the expression is computed (unconditionally)
		 * is deemed to be the definition point.
		 */

		for( d = AEChain[Aindex]; d != NULL; d = d->av_expr_link )
		{
			if( d->FGindex == b->FGindex &&		/* comp. here*/
			    d->in_cond == NotConditional &&	/* uncondit. */
			    (ddef == NULL || ddef->order < d->order) )
			{
				ddef = d;
			}
		}
	}
	else
	{
		/* The expression is not killed in this block.  The first
		 * point at which the expression is computed (unconditionally)
		 * is deemed to be the definition point
		 */

		for( d = AEChain[Aindex]; d != NULL; d = d->av_expr_link )
		{
			if( d->FGindex == b->FGindex &&		/* comp. here*/
			    d->in_cond == NotConditional &&	/* uncondit. */
			    (ddef == NULL || ddef->order > d->order) )
			{
				ddef = d;
			}
		}
	}

	assert(ddef != NULL);

	/* This is bit of a hack.  If the definition point already has
	 * a temp attached, give up.  This avoids attaching more than one
	 * temp to the node.
	 *
	 * To do this right, should replace the loop temp everywhere with
	 * the global common subexpression temp.
	 */

	if( ddef->carrier != NoId )
		switch(IdOp(ddef->carrier))
		{
		case LTEMP:
		case TNAME:
		case TREG:
			if( GlComdebug > 1 )
				printf("Abandon %d: temp carrier on %x\n",
					Aindex, ddef);
/**/			return NULL;
		}

	return ddef;
}

static CostType
GComBenefit(Aindex, ns, defpoints)
	AvailIndex Aindex;
	Operator ns;
	DAG_Node *defpoints;
{
	DAG_Node d, ddef;
	CostType refcost;
	int d_weight;
	int w_refs;		/* weighted reference count */
	BasicBlock b;
	CostType cost_now, cost_after;

	cost_now = cost_after = 0;

	for( d = AEChain[Aindex]; d != NULL; d = d->av_expr_link )
	    if( FlowGraph[d->FGindex].visited )
	    {
		ddef = defpoints[d->FGindex];
		b = FlowGraph[d->FGindex].block;

		/* If this block is a definition point, and this node appears
		 * before the real definition point, forget it.
		 * If this block is not a definition point and the expression
		 * is killed in this block, forget it also.
		 */

		if( ddef != NULL )
		{
			if( d->order < ddef->order )
	/**/			continue;
		}
		else
		{
			if( IsElement((int)Aindex, b->av.Kill) )
	/**/			continue;
		}

		/* If this expression is no longer computed here, give up
		 * (likely was taken care of as GCSE earlier on).
		 */
		if( !IsElement((int)Aindex, b->av.Comp) )
	/**/		continue;

		d_weight = FlowGraph[d->FGindex].loopweight;


		if( ddef == d )
		{
			/* A definition.  Need to evaluate the expression
			 * and store it (in our temp), then reference our
			 * temp as the carrier
			 */

			assert(!d->prev_valid_carrier);

			cost_after += (d->cost + StoreCost(ns, d->type))
					* d_weight;
			w_refs = d->refs * d_weight;
		}
		else
		{
			/* Avoid something with a previously valid carrier */

			if( d->prev_valid_carrier )
	/**/			continue;

			/*	we evaluate the node refs times, and once to
			 *	store into attached ids, if any.
			 */

			w_refs = (d->refs + ((d->attached==NULL) ? 0 : 1))
				 * d_weight;
		}

		cost_after += RefCost(ns, d->type) * w_refs;


		if( d->carrier == NoId )
		{
			/* Expression gets re-evaluated each & every time.
			 * Work out cost of doing computation
			 * and stashing result in a temp (or
			 * permanent carrier if there was one).
			 */

			cost_now += d->cost * d->refs * d_weight;
		}
		else
		{
			/* Expression gets evaluated once
			 * and then carrier is used.  The stores into the
			 * carrier are fixed overhead and do not count.
			 * If it is cheaper to recompute than to reference the
			 * carrier, assume that the code generator will do this.
			 * For transparent carriers, assume they go into
			 * registers.
			 */
	
			refcost = IsTransparent(d->carrier)
					? RefCost(TREG, SymType(d->carrier))
					: IdRefCost(d->carrier);
			if( refcost > d->cost )
				refcost = d->cost;
			cost_now += (d->cost + refcost * d->refs) * d_weight;
		}
	    }

	if( GlComdebug > 1 )
	{
		if( GlComdebug > 2 || cost_now <= cost_after )
			printf("Available expr. %d: cost now %d after %d\n",
				Aindex, cost_now, cost_after);
	}

	return cost_now - cost_after;
}

/*
 * Now that we know that making the available expression "Aindex" into a
 * global common subexpression is worthwhile, do it.  The idea here is to
 * examine every instance of the expression.  On definitions, add an
 * assignment to the temp "id".  On uses, set the carrier and note that it
 * is defined elsewhere.
 */

static void
MakeGCSEHappen(Aindex, id, defpoints)
	AvailIndex Aindex;	/* AvailIndex of the global common subexpr */
	Identifier id;		/* The id to hold the value */
	DAG_Node *defpoints;	/* Array of definition points */
{
	DAG_Node d, ddef;
	BasicBlock b;
	AttachedID aid;

	for( d = AEChain[Aindex]; d != NULL; d = d->av_expr_link)
	    if( FlowGraph[d->FGindex].visited )
	    {
		ddef = defpoints[d->FGindex];
		b = FlowGraph[d->FGindex].block;

		/* If this block is a definition point, and this node appears
		 * before the real definition point, forget it.
		 * If this block is not a definition point and the expression
		 * is killed in this block, forget it also.
		 */

		if( ddef != NULL )
		{
			if( d->order < ddef->order )
	/**/			continue;
		}
		else
		{
			if( IsElement((int)Aindex, b->av.Kill) )
	/**/			continue;
		}

		/* If this expression is no longer computed here, give up
		 * (likely was taken care of as GCSE earlier on).
		 */

		if( d->prev_valid_carrier )
	/**/		continue;

		/*
		 * FINALLY, do it!
		 */

		if( d != ddef )
		{
			/* A use.  This block no longer generates or computes
			 * this expression, but uses the temp instead.
			 */

			d->prev_valid_carrier = True;
			d->carrier = id;

			Insert((int)id, b->ld.Use);
			AdjustRefs(d);

			if( GlComdebug > 0 )
				printf("Modify node %x (use)\n", d);
		}
		else
		{
			/* A definition.  Attach the temp and set the carrier.
			 * Update live-dead information too
			 */

			aid = AttachID(id, d);
			aid->valid_carrier = True;
			d->carrier = id;
#if 1
	/* Restore when GSX multiple defs problem solved */
			AddTempDef(id, d);
#endif

			Insert((int)id, b->ld.Def);
			Insert((int)id, b->ld.PDef);

			if( GlComdebug > 0 )
				printf("Modify node %x (def)\n", d);
		}

		/* If there is no definition in this block, say we no longer
		 * compute the expression
		 */
		if( ddef == NULL )
		{
			DelMember((int)Aindex, b->av.Gen);
			DelMember((int)Aindex, b->av.Comp);
		}
	    }
}

/*
 * Clean up after doing a global common subexpression - make sure that
 * we don't try this one again.
 */

static void
CommonCleanup(Aindex)
	AvailIndex Aindex;
{
	int i;

	for( i = 0; i < NumReachableNodes; i++ )
	{
		if(FlowGraph[DFN[i]].visited)
			DelMember((int)Aindex,
				FlowGraph[DFN[i]].block->gcse.Pos);
	}
}
