/********** ctime.c: do time conversion in Sys V style **********
 *
 * Note: time zone info comes from the TZ environment variable which has a
 *       string like "PST8PDT" where the two alpha strings give names for
 *       normal and daylight savings time (if second is absent, no daylight
 *       savings conversions will be performed; if present, the calculation
 *       will be performed with the U.S. algorithm at the appropriate part
 *       of the year).  The number is "hours West of Greenwich".  It can be
 *       negative to give East.
 *
 */

#include <time.h>

#define YES     1
#define NO      0

#define EPOCHWD         4               /* day of week (thu) for 70/1/1 */
#define EPOCHYR         70              /* epoch starts in 1970 */

long timezone = 8*60*60;		/* seconds West of Greenwich */
int  daylight = 1;			/* 1 = do conversion, if applicable */
char *tzname[2] = {"MST", "MDT"};	/* names for normal & dst zones */
struct tm *gmtime(), *localtime();
static char *tzi2a();
extern char *strncpy();
static char *tzp;               /* for scanning TZ string */
static int months[12] = {31,28,31,30,31,30,31,31,30,31,30,31};
static char dayname[] = "SunMonTueWedThuFriSat";
static char mnthname[] = "JanFebMarAprMayJunJulAugSepOctNovDec";
static isdst();
static lastsun();

/*
 * asctime: return printable time given a broken out time structure.  Format is
 *
 *      Sun Sep 16 01:03:52 1973\n\0    (exactly 26 characters)
 */

char *
asctime(ptm) const struct tm *ptm; {
	char *p;
	static char buf[26];

	p = strncpy(buf, dayname + ptm->tm_wday * 3, 3);
	*p++ = ' ';
	p = strncpy(p, mnthname + ptm->tm_mon * 3, 3);
	*p++ = ' ';
	p = tzi2a(p, ptm->tm_mday, ' ', ' ' );
	p = tzi2a(p, ptm->tm_hour, '0', ':' );
	p = tzi2a(p, ptm->tm_min, '0', ':' );
	p = tzi2a(p, ptm->tm_sec, '0', ' ' );
	p = strncpy(p, (ptm->tm_year < 100) ? "19" : "20", 2);
	p = tzi2a(p, ptm->tm_year, '0', '\n' );
	return buf;
}


/*
 * ctime: return printable form of local time given a gmt clock
 */

char *
ctime(clock) const long *clock; { return asctime(localtime(clock)); }

/*
 * gmtime: return pointer to broken up time structure given a pointer to
 *         time of day (seconds since 70/01/01 00:00).  Time of day may be
 *         up to one year negative (allows handling time = 0 when West of
 *         Greenwich).
 */

struct tm *
gmtime(clock) const long *clock; {
	static struct tm tm;
	long day, tod, clk;
	int hrs, mins, secs;
	int weekday, year, yearday, month, monthday;
	int yearsize;

	clk = *clock + 31536000L;	/* seconds in a year */
	day = (clk / 86400L);
	tod = clk % 86400L;
	weekday = (day + EPOCHWD - 1) % 7;
	if (weekday < 0) weekday = -weekday;
	secs = tod%60;
	tod /= 60;
	mins = tod%60;
	hrs = tod/60;
	for (yearday = day, year = EPOCHYR - 1;;) {
		yearsize = (year % 4) ? 365 : 366;
		if (yearday < yearsize) break;
		year++;
		yearday -= yearsize;
	}
	if (yearsize == 366) months[1] = 29;
	for (monthday = yearday, month = 0;;) {
		if (monthday < months[month]) break;
		monthday -= months[month];
		month++;
	}
	tm.tm_sec = secs;
	tm.tm_min = mins;
	tm.tm_hour = hrs;
	tm.tm_mday = monthday + 1;
	tm.tm_mon = month;
	tm.tm_year = year;
	tm.tm_wday = weekday;
	tm.tm_yday = yearday;
	tm.tm_isdst = 0;
	months[1] = 28;
	return &tm;
}

/*
 * localtime: returns local time in broken out structure, given a gmt time
 */

struct tm *
localtime(clock) const long *clock; {
	struct tm *tm;
	long local;
	int n, s;
	int c;

	if ((tzp = getenv("TZ")) && *tzp){  /* if TZ exists & is not empty!! */

		/* make sure timezone info is properly set up from the TZ
		   env variable */

		strncpy(tzname[0],tzp,3);
		if ((s = *tzp) == '-' || s == '+') tzp++;
		n = 0;
		while ((c = *tzp++) >= '0' && c <= '9') n = n * 10 + (c - '0');
		tzp--;
		n = (s == '-') ? -n : n;
		timezone = (long) n * 3600L;
		strncpy(tzname[1],tzp,3);
		daylight = *tzname[1] != 0;
	}
	local = *clock - timezone;
	tm = gmtime(&local);
	if (isdst(tm)) {
		local += 3600L;
		tm = gmtime(&local);
		tm->tm_isdst = 1;
	}
	return tm;
}

/********** local routines **********/
/*
 * isdst: return true if daylight savings time is in effect
 */

static
isdst(tm) struct tm *tm; {
	int start, end;

	if (!daylight) return NO;

	/*** algorithm for U.S. until congress changes its mind ***/

	start = 118;            /* last sunday in april */
	end = 300;              /* last sunday in october */
	start = lastsun(start,tm);
	end = lastsun(end,tm);
	if (tm->tm_yday < start ||
		(tm->tm_yday == start && tm->tm_hour < 2))
			return NO;
	if (tm->tm_yday > end ||
		(tm->tm_yday == end && tm->tm_hour >= 1))
			return NO;
	return YES;
}

/*
 * lastsun: given a day number (starting at zero) and a time structure for
 *          some day in the current year, return the day number of the
 *          nearest sunday not greater that the day
 */

static
lastsun(day,tm) struct tm *tm; {
	int adjust, dow;

	if( (tm->tm_year % 4) == 0) day++;  /* bump if in leap year */
	adjust = (7 + (tm->tm_wday - (tm->tm_yday % 7))) % 7;
	dow = (day + adjust) % 7;
	if (dow != 0) day -= dow;
	return day;
}

/*
 * tzi2a: convert up to 2-digits to ascii & put in string dest.  return updated
 *        dest.  fill is character to use as first if value < 10.
 *	  term is the character to be placed after the field
 */

static
char *
tzi2a(dest,value,fill,term) register char *dest; {

	dest[0] = fill;
	dest[1] = value % 10 + '0';
	dest[2] = term;
	dest[3] = 0;
	if( value >= 10 ) dest[0] = (value / 10) % 10 + '0';
	return dest + 3;
}
