/* $Id: macport.cpp,v 1.6 93/08/12 13:57:43 tal Exp Locker: tal $ */

/* Copyright (C) MICA Inc. and Tal Korin 1991 */
/* All rights reserved */

/* macport.cpp - implementation of class MacSerialPort */

/* FOR_ISC */

#ifdef OSMAC

#include "macport.hpp"

static char iBuf[4096], oBuf[4096];

/* public constructor */
MacSerialPort::MacSerialPort (short i_portNumber)
{
	SetPortNumber (i_portNumber);
	setOutputQueue ();
}

/* public destructor */
MacSerialPort::~MacSerialPort ()
{
}

/* virtual public method to open the Mac port */
short MacSerialPort::Open (enum OpenMode i_mode)
{
	SerShk handshake;

openMode = i_mode;

	switch (portNumber) {
	case sPortA:
		refnum.in = MODEM_READ;
		refnum.out = MODEM_WRITE;
		strcpy ((char*) drname.in, "\004.AIn");
		strcpy ((char*) drname.out, "\005.AOut");
		break;
	case sPortB:
		refnum.in = PRINTER_READ;
		refnum.out = PRINTER_WRITE;
		strcpy ((char*) drname.in, "\004.BIn");
		strcpy ((char*) drname.out, "\005.BOut");
		break;
	}
#ifdef DO_RAM_OPEN
	if (RAMSDOpen(portNumber) != noErr) {
		openMode = NO_OPEN;
		return -1;
	}
#else
	if (openMode == IO_PORT_WRITE_ONLY || openMode == IO_PORT_READ_WRITE) {
		if (OpenDriver (drname.out, &refnum.out) != noErr) {
			openMode = NO_OPEN;
			return -1;
		}
	}
	if (openMode == IO_PORT_READ_ONLY || openMode == IO_PORT_READ_WRITE) {
		if (OpenDriver (drname.in, &refnum.in) != noErr) {
			openMode = NO_OPEN;
			return -1;
		}
	}

#endif

	if (openMode != NO_OPEN) {
		if (openMode == IO_PORT_WRITE_ONLY || openMode == IO_PORT_READ_WRITE) {
			/* not all devices assert CTS; better to ignore it */
			memset((char *)&handshake, 0, sizeof(handshake));
			SerHShake(refnum.out, &handshake);
		}
		if (openMode == IO_PORT_WRITE_ONLY || openMode == IO_PORT_READ_WRITE) {
			SerSetBuf(refnum.out , oBuf, sizeof(oBuf));
			KillIO (refnum.out);
		}
		if (openMode == IO_PORT_READ_ONLY || openMode == IO_PORT_READ_WRITE) {
			SerSetBuf(refnum.in , iBuf, sizeof(iBuf));
			KillIO (refnum.in);
		}

		return setCommParams();
	}
}

/* virtual public method to close the port */
short MacSerialPort::Close ()
{
	if (openMode != NO_OPEN) {
		if (openMode == IO_PORT_WRITE_ONLY || openMode == IO_PORT_READ_WRITE) {
			KillIO (refnum.out);
			SerSetBuf(refnum.out , oBuf, 0);
		}
		if (openMode == IO_PORT_READ_ONLY || openMode == IO_PORT_READ_WRITE) {
			KillIO (refnum.in);
			SerSetBuf(refnum.in , iBuf, 0);
		}
#ifdef DO_RAM_OPEN
		RAMSDClose(portNumber);
#endif
	}

	return PhysicalSerialPort::Close ();
}

/* virtual public method to write to the port.  Replies the number of bytes
	actually written or -n if there is an error. */
short MacSerialPort::Write (char *i_buf, short i_len)
{
	long count = i_len;

	if (openMode != IO_PORT_WRITE_ONLY && openMode != IO_PORT_READ_WRITE)
		return -1;
	CHECK_QUEUED_OUTPUT (i_buf, i_len);

	if (FSWrite(refnum.out, &count, i_buf) != noErr) {
		return -1;
	}
	logOutput (i_buf, (short)count);
	return count;
}


/* virtual public method to read a max number of bytes.  Replies the number of
	bytes actually read or -n if there is an error */
short MacSerialPort::Read (char *o_buf, short i_maxLen)
{
	long avail;

	if (openMode != IO_PORT_READ_ONLY && openMode != IO_PORT_READ_WRITE)
		return -1;

	if (SerGetBuf(refnum.in, &avail) != noErr)
		return -1;
	if (avail == 0)
		return 0;	// no data in driver input buffer, read would block.
	avail = min(avail, i_maxLen);
	if (FSRead(refnum.in, &avail, o_buf) != noErr)
		return -1;
	logInput (o_buf, (short)avail);
	return avail;
}

/* virtual public method to set the baud rate (300-57600 baud) */
short MacSerialPort::SetBaud (unsigned short i_baudRate)
{
	if (i_baudRate < 300)
		baudRate = 300;
	else if (i_baudRate > 19200)
		baudRate = 57600;
	else
		baudRate = i_baudRate;

	return setCommParams();
}


/* virtual public method to set number of data bits (7-8) */
short MacSerialPort::SetDataBits (char i_dataBits)
{
	switch (i_dataBits) {
	case 7:
	case 8:
		nDataBits = i_dataBits;
		return setCommParams();
	}
	return -1;
}

/* virtual public method to set number of stop bits (1-2) */
short MacSerialPort::SetStopBits (char i_stopBits)
{
	switch (i_stopBits) {
	case 1:
	case 2:
		nStopBits = i_stopBits;
		return setCommParams();
	}
	return -1;
}

/* virtual public method to set the parity type ('E','O','N') */
short MacSerialPort::SetParity (char i_parity)
{
	switch (i_parity) {
	case 'E':
	case 'O':
	case 'N':
		parity = i_parity;
		return setCommParams();
	}
	return -1;
}

/* protected method to set serial driver parameters */
short MacSerialPort::setCommParams ()
{
	int serConfig = 0;

	if (openMode == NO_OPEN)
		return 0;

	switch (baudRate) {
	case 300:
		serConfig += baud300;
		break;
	case 600:
		serConfig += baud600;
		break;
	case 1200:
		serConfig += baud1200;
		break;
	case 2400:
		serConfig += baud2400;
		break;
	case 4800:
		serConfig += baud4800;
		break;
	case 9600:
		serConfig += baud9600;
		break;
	case 19200:
		serConfig += baud19200;
		break;
	case 57600:
		serConfig += baud57600;
		break;
	}
	switch (nDataBits) {
	case 7:
		serConfig += data7;
		break;
	case 8:
		serConfig += data8;
		break;
	}
	switch (nStopBits) {
	case 1:
		serConfig += stop10;
		break;
	case 2:
		serConfig += stop20;
		break;
	}
	switch (parity) {
	case 'E':
		serConfig += evenParity;
		break;
	case 'O':
		serConfig += oddParity;
		break;
	case 'N':
		serConfig += noParity;
		break;
	}

	/* Set both input and output drivers */
	if (SerReset(refnum.in, serConfig) == noErr && SerReset(refnum.out, serConfig) == noErr)
		return 0;

	return -1;
}


/* virtual public method to create a deep copy */
PhysicalSerialPortHANDLE MacSerialPort::DeepCopy ()
{
	MacSerialPortHANDLE newPort;

	newPort = NEW (MacSerialPort (ReplyPortNumber ()));
	newPort->SetParity (parity);
	newPort->SetDataBits (nDataBits);
	newPort->SetStopBits (nStopBits);
	newPort->SetBaud (baudRate);
	return newPort;
}


/* virtual public method to reply the port number */
short MacSerialPort::ReplyPortNumber ()
{
	return (portNumber == sPortA)? 0 : 1;
}

/* virtual public method to set the port number */
void MacSerialPort::SetPortNumber (short i_number)
{
	portNumber = (i_number == 1)? sPortB : sPortA;
}
#endif
