/* MPXIO.C - MPX unblocked no-wait I/O routines */
/* This module provides Logical I/O services for writing MPX-32 files */
/*  Three routines are exported from this module:              */

/*      mpxopen - accepts a pathname and an access-mode argument.
            Assigns an (internally generated) LFC to the file and
            opens it for the specified access.  Returns a pointer
            to be used for all subsequent accesses to the file.     */

/*      mpxwrit - accepts a pointer value generated by mpxopen and
            a character string.  Writes the string as a record to
            the file indicated by the pointer.  The size of the
            record written will be the length of the character
            string.  */

/*      mpxclos - accepts a pointer value generated by mpxopen.
            Closes and deassigns the file indicated by the
            pointer.  */

#include "@SYSTEM(INCLUDE)STDIO.H"

#define M_ASSN(r1,r2) mpxsvc (0x2052,r1,r2)
#define M_WRIT(r1,r2) mpxsvc (0x1032,r1,r2)
#define M_READ(r1,r2) mpxsvc (0x1031,r1,r2)
#define M_WAIT(r1,r2) mpxsvc (0x103C,r1,r2)
#define M_CLSE(r1,r2) mpxsvc (0x1039,r1,r2)

#define TCW(count,ptr) ((count << 20) | (int) ptr | 0x80000)
#define RR_PATH   1                /* Assign by pathname  */
#define RR_TEMP   2                /* Assign to temp file */
#define RR_DEVC   3                /* Assign to device */
#define RR_LFC2   4                /* Assign to secondary LFC */
#define RR_SPACE  5                /* Assign by segment defination */
#define RR_RID    6                /* Assing by resource ID */
#define OPN_WRIT  (2 << 8)
#define OPN_APND  (5 << 8)
#define OPN_UPDAT (4 << 8)
#define OPN_BLKD  3
#define OPN_UNBLKD 2
#define INT(arb)  *((int *) (&(arb)))

typedef struct {    /* RRS structure */
    char    lfc[4];
unsigned type :
    8;
unsigned size :
    8;
unsigned spec :
    16;
    struct {    /* access specifications */
unsigned read  :
        1;
unsigned write :
        1;
unsigned modfy :
        1;
unsigned updat :
        1;
unsigned appnd :
        1;
unsigned resv1 :
        11;
unsigned shar  :
        1;
unsigned excl  :
        1;
unsigned resv2 :
        14;
    }
    accs;
    struct {   /* options specifications */
unsigned syc   :
        1;
unsigned sgo   :
        1;
unsigned slo   :
        1;
unsigned sbo   :
        1;
unsigned blk   :
        1;
unsigned unblk :
        1;
unsigned nomsg :
        1;
unsigned res1  :
        1;
unsigned open  :
        1;
unsigned buff  :
        1;
unsigned res2  :
        2;
unsigned nowt  :
        1;
unsigned publc :
        1;
unsigned vomm  :
        1;
unsigned sep   :
        1;
unsigned res3  :
        16;
    }
    opts;
    union {     /* Split into different RRS types */
        char rr_path [1];   /* Pathname -- Type 1 */
        char rr_volnm [1];  /* Volume Name -- Type 2 */
    }
    rr_end;
}
rrs;

typedef struct {  /* FCB structure */
    char lfc[4];
    char * tcw;
    struct {    /* control bits */
unsigned nowait :
        1;
unsigned noerr  :
        1;
unsigned dfi    :
        1;
unsigned nst    :
        1;
unsigned ran    :
        1;
unsigned bl     :
        1;
unsigned exp    :
        1;
unsigned iec    :
        1;
unsigned dfd    :
        1;
unsigned specctl:
        4;
unsigned racc   :
        19;
    }
    cntl;
    int devstat;
    int recl;
    int ioq;
    int errrtn;
    char * fataddr;
    char * bufaddr;
    int excount;
    int exracc;
    int statw1;
    int statw2;
    int nwnorm;
    int nwerrtn;
    char * blkbuff;
}
fcb;

typedef struct {        /* CNP structure */
    int timeout;
    int abrtn;
unsigned options :
    16;
unsigned status  :
    16;
    int resvd[2];
    fcb * parmlink;
}
cnp;

typedef struct {    /* combined file-control structure */
    fcb   file_fcb;
    cnp   file_cnp;
    rrs   file_rrs;    /* RRS must come last
                              to allow for variability */
}
file_ctl;

/*******************************************************************/

/* NOTE: This routine generates LFCs of the form Mnn, where nn
    varies from 00 to 99.  These LFCs are not reused, so no more
    than 100 calls to mpxopen can be safely made.  */

int *
mpxopen (pathnm,access)
char * pathnm;      /* Pathname of file to open */
char   access;      /* Access mode
                        'w' = write mode (overwrite previous data)
                        'a' = append mode */
{
    int i;
    file_ctl * fp;
    int rrsize;
    int svcregs[8];
    static lfc_cnt = 0;

    fp = (file_ctl *)malloc ((sizeof (file_ctl)) + strlen (pathnm));

    /* Initialize FCB and RRS LFC */

    fp->file_rrs.lfc[0] = fp->file_fcb.lfc[0] = '\0';
    fp->file_rrs.lfc[1] = fp->file_fcb.lfc[1] = 'M';
    fp->file_rrs.lfc[2] = fp->file_fcb.lfc[2]
        = ('0' + (lfc_cnt / 10));
    fp->file_rrs.lfc[3] = fp->file_fcb.lfc[3]
        = ('0' + (lfc_cnt % 10));

    lfc_cnt++;

    /* Set up RRS for assign and open */

    INT(fp->file_rrs.accs) = 0;
    INT(fp->file_rrs.opts) = 0;
    fp->file_rrs.opts.unblk = 1;
    fp->file_rrs.opts.open = 1;
    INT(fp->file_fcb.cntl) = 0;

    /* Now check for system files */

    fp->file_rrs.type = RR_TEMP;  /* Assume it's a temporary */
    rrsize = sizeof (rrs) - 1;             /* # of bytes */
    fp->file_rrs.spec = 0;  /* default initial file size */
    if (strcmp (pathnm,"SLO") == 0) {
        fp->file_rrs.opts.slo = 1;
        fp->file_fcb.cntl.dfi = 1;   /* Inhibit carriage-control */
    }
    else if (strcmp (pathnm,"SGO") == 0) {
        fp->file_rrs.opts.sgo = 1;
    }
    else if (strcmp (pathnm,"SBO") == 0) {
        fp->file_rrs.opts.sbo = 1;
    }
    else {
        fp->file_rrs.type = RR_PATH;
        rrsize += (sizeof (int)) - 1 + strlen (pathnm);
        fp->file_rrs.spec = strlen (pathnm) << 8;
        if (access == 'w') {
            fp->file_rrs.accs.write = 1;
        }
        else {
            fp->file_rrs.accs.updat = 1;
        } /* end if */
        strcpy (fp->file_rrs.rr_end.rr_path,pathnm);
        fp->file_cnp.options
            = ( (access == 'w') ? OPN_WRIT : OPN_UPDAT) + OPN_UNBLKD;
    } /* end if */
    fp->file_rrs.size = rrsize / sizeof (int);  /* in words */

    /* Now initialize FCB */

    fp->file_fcb.tcw = NULL;
    fp->file_fcb.cntl.nowait = 1; /* Nowait I/O */
    fp->file_fcb.cntl.ran = 1; /* Random access I/O */
    fp->file_fcb.cntl.exp = 1; /* Expanded FCB */
    fp->file_fcb.cntl.bl = 0; /* Unblocked I/O */
    fp->file_fcb.cntl.racc = 0;  /* start at sector 0 */
    fp->file_fcb.devstat = 0;
    fp->file_fcb.recl = 0;
    fp->file_fcb.ioq = 0;
    fp->file_fcb.errrtn = 0;
    fp->file_fcb.fataddr = fp->file_fcb.bufaddr = NULL;
    fp->file_fcb.excount = 0;
    fp->file_fcb.exracc = 0;
    fp->file_fcb.statw1 = 0;
    fp->file_fcb.statw2 = 0;
    fp->file_fcb.nwnorm = 0;
    fp->file_fcb.nwerrtn = 0;
    fp->file_fcb.blkbuff = NULL;

    /* Initialize CNP */

    fp->file_cnp.timeout = 1;   /* don't wait for resource */
    fp->file_cnp.abrtn = 0;
    fp->file_cnp.status = 0;
    fp->file_cnp.parmlink = &(fp->file_fcb);

    svcregs[1] = (int) &(fp->file_rrs);
    svcregs[7] = (int) &(fp->file_cnp);
    if (M_ASSN(svcregs,svcregs) >> 3) {
        free(fp);
        return (0);
    } /* endif */
    return ((int *) fp);
} /* end mpxopen */

/**********************************************************/

mpxwrite(fp, text, length, incr)
file_ctl * fp;  /* File to write to */
char * text;    /* Record to write */
int length, incr;
{
    int svcregs[8];

    svcregs[1] = (int) &(fp->file_fcb);

    M_WAIT(svcregs,svcregs);

    /* Set up TCW and record length in FCB
                -- rest of FCB carries over from mpxopen */

    fp->file_fcb.bufaddr = text;
    fp->file_fcb.excount = length;

    fp->file_fcb.cntl.nowait = 1;   /* set no wait mode */
    M_WRIT(svcregs,svcregs);
    if (incr == 1){
        fp->file_fcb.exracc += 1;
        M_WAIT(svcregs,svcregs);
    }
    fp->file_fcb.cntl.nowait = 1;   /* set wait mode */
} /* end mpxwrit */

/**********************************************************/

mpxread (fp, text, length)
file_ctl * fp;  /* File to read from*/
char * text;    /* Record to read */
int length;
{
    int svcregs[8];

    svcregs[1] = (int) &(fp->file_fcb);
 /*
   M_WAIT(svcregs,svcregs);
*/
    /* Set up TCW and record length in FCB
                -- rest of FCB carries over from mpxopen */

    fp->file_fcb.bufaddr = text;
    fp->file_fcb.excount = length;

    fp->file_fcb.cntl.nowait = 0;   /* set wait mode */
    M_READ(svcregs,svcregs);
    fp->file_fcb.exracc += 1;   /* point to next sector */
    fp->file_fcb.cntl.nowait = 0;   /* set no wait mode */
} /* end mpxwrit */

/********************************************************************/

mpxclos (fp)
file_ctl * fp;  /* File to read from*/
{
    int svcregs[8];
    int eof [2];

    eof [0] = eof [1] = -1;
    eof [2] = NULL;
    svcregs[1] = (int) &(fp->file_fcb);

    M_WAIT(svcregs,svcregs);

    fp->file_fcb.exracc += 1;
    fp->file_fcb.bufaddr = (char *) &eof;
    fp->file_fcb.excount = 8;

    fp->file_fcb.cntl.nowait = 1;   /* set wait mode */
    svcregs[1] = (int) &(fp->file_fcb);
    M_WAIT(svcregs,svcregs);
    M_WRIT(svcregs,svcregs);
    M_CLSE(svcregs,svcregs);
    free (fp);
} /* end mpxclos */

/********************************************************************/

mpxback (fp)
file_ctl * fp;  /* File to read from*/
{

    if (fp->file_fcb.exracc > 0)
        fp->file_fcb.exracc -= 1;   /* point to last sector */
} /* end mpxback */

/********************************************************************/
/* End of module */
