/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: pass2.c,v 5.2 88/08/30 01:11:09 pcc Released $";
#endif

/*
 *
 * Gould Base Register Assembler
 * Initial design & coding by R. J. Miller 
 * at Compion Corporation 
 * Thu Jun  9 20:01:37 CDT 1983
 */

#include <a.out.h>
#include "struct.h"
#include "operand.h"
#include "ophash.h"

#ifndef lint
#endif

char *Optypes[] =
{
    "NO OPERAND",
    "REGISTER",
    "BASE REG",
    "I CONSTANT",
    "MEMORY REF",
    "E CONSTANT",
    "INDEXED",
};

extern struct opcode optable[];

extern SYM tmpsym;
extern OPND oper[];
#ifdef POOL
extern OPND tmpopnd;
#endif POOL
/*
OPND t_oper;
*/

/*
NAME: bitnum
PURPOSE: convert power of two to bit position.
PRECONDITIONS:
parameter const is a constant that should be a power of two.
POSTCONDITIONS:
returns -1 if const is not a power of two.
returns number (0 - 31) of the bit position of the set bit in const.
return of zero means the most significant bit is set.
ALGORITHM:
first check for power of two; then beginning with bit 31, right shift as
necessary until the least significant bit is on.
if no bits were set in const, then the loop will fall through to the final
return (-1).
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial coding
15 Jun 84	Gould CSD 3423 put in power-of-two test & eliminated dependency
		on bit fill from left
*/

bitnum (const)
register const;
{
    register int bitoff;

    if ((const & (const -1)) == 0)
	for (bitoff = 31; bitoff >= 0; --bitoff) {
	    if (const & 1)
		return (bitoff);
	    const >>= 1;
	}
    return (-1);
}

/*
NAME: op_arith
PURPOSE: handle most arithmetic op codes.
PRECONDITIONS:
op is the usual optable pointer.
ALGORITHM:
Decides whether to emit a mem->reg, reg->mem, reg->reg, or immed->reg
instruction. Causes constant to be emitted, if necessary.
the ADR instruction is not used here due to speed considerations.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_arith (op)
register struct opcode *op;
{
    int i;

    switch (oper[0].type)
    {
	case IMMEDIATE:			/* we know it is 16-bit const */
	case CONSTANT:
	    if (op -> ocode == (short) 0xB800	/* it's an add */
		&& op -> olength == WORD	/* of a word */
		&& oper[1].type != REGISTER	/* do not use ABR for reg-reg */
		&& oper[0].haveseen == CONST	/* of a constant */
		&& oper[0].constval != 0
		&& ((oper[0].constval & (oper[0].constval-1)) == 0))
				/* and constant is nonzero power of 2 */
	    {				/* we'll use baddw */
		/* get opcode table entry */
		if ((i = uncollide ("baddw")) == MAX_HASH)
		{
		    error ("op_arith: can't find baddw");
		}
		else
		{
		    op = &optable[i];
		    do_align (4);	/* put on word bndy */
		    op_bits (op);	/* do this in other routine */
		}
		break;
	    }

	    if (oper[1].type != REGISTER)
	    {
		error ("op_arith: %s const to %s",
			op -> oname, Optypes[oper[1].type]);
	    }
	    else
	    {
		emit_rconst (op -> ocode,/* emit code to ref constant */
		    op -> ocodeim,	/* code for immediate op */
		    oper[1].regval,
		    &oper[0],
		    op -> olength);
	    }
	    break;

	case MEMORY:
	case INDEXED:
	    if (oper[1].type != REGISTER)
	    {
		error ("op_arith: mem to %s", Optypes[oper[1].type]);
	    }
	    else
	    {
		emit_rmem (op -> ocode,	/* emit code to ref memory */
		    oper[1].regval,
		    &oper[0],
		    op -> olength);
	    }
	    break;

	case REGISTER:
	    switch (oper[1].type)
	    {				/* might be add to mem */
		case MEMORY:
		case INDEXED:
		    if (op -> ocoderm == 0)
		    {
			error ("op_arith: No instruction for %s reg to mem",
			    op -> oname);
		    }
		    else
		    {
			emit_rmem (op -> ocoderm,	/* code for reg->mem */
			    oper[0].regval,
			    &oper[1],
			    op -> olength);
		    }
		    break;
		
		case REGISTER:
		    if (op -> ocoderr == 0)
		    {				/* is there such thing? */
			error ("op_arith: No instruction for %s reg to reg",
			    op -> oname);
		    }
		    else
		    {
			/* code for reg->reg */
			emit_half (op -> ocoderr | (oper[0].regval << 4) |
			    (oper[1].regval << 7));
		    }
		    break;

		default:
		    error ("op_arith: Illegal operand type %s",
			Optypes[oper[1].type]);
		    break;
	    }
	    break;

	case BASE:
	default:
	    error ("op_arith: Illegal operand type %s", Optypes[oper[0].type]);
	    break;
    }
}

/*
NAME: op_binc
PURPOSE: handle branch after increment opcodes
PRECONDITIONS:
op is the usual optable pointer.
these opcodes are not used by the Compion C compiler.
ALGORITHM:
the second operand cannot be indexed.
HISTORY:
15 Jul 83	D A Willcox of Compion	Initial coding
*/

op_binc (op)
struct opcode *op;
{
    if (oper[1].regval != 0)
    {
	error ("op_binc: Illegal operand type %s", Optypes[oper[1].type]);
    }
    else
    {
	emit_rmem (op -> ocode, oper[0].regval, &oper[1], op -> olength);
    }
}

/*
NAME: op_bits
PURPOSE: handle bclr, bset, btst, and badd op codes.
PRECONDITIONS:
op is the usual optable pointer.
ALGORITHM:
the first operand must be a power of two and must be appropriate to the
size of the second operand. if the size is a byte or halfword, the value
is stored in the high order byte or halfword. bitnum then returns a
value that is 3 bytes or 1 halfword too large.
the second operand may be a register, which results in a halfword
instruction, or memory-indexed, yielding a fullword.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_bits (op)
register struct opcode *op;
{
    register int inst;
    register int bitoff;

    if (oper[0].haveseen & NAME)
    {
	error ("op_bits: first oper must be constant");
	return;
    }

    bitoff = bitnum (oper[0].constval);	/* find bit number of set bit */

    if (bitoff < 0)
    {
	error ("op_bits: constant not power of two");
	return;
    }

    switch (oper[1].type)
    {					/* decide if reg or memory instr */
	case REGISTER:			/* [SZT]BR */
	    inst = op -> ocoderr;
	    inst |= oper[1].regval << 4;
	    inst |= (bitoff >> 3) & 3;
	    inst |= (bitoff & 0x7) << 7;
	    emit_half (inst);
	    break;
	
	case MEMORY:
	case INDEXED:			/* [SZT]BM */
	    switch (op -> olength)
	    {				/* must adjust for size */
		case BYTE:
		    bitoff -= 24;
		    break;
		
		case HALF:
		    bitoff -= 16;
		    break;
		
		case WORD:
		    break;

		default:
		    error ("op_bits: %s to register", op -> oname);
		    return;
	    }

	    if (bitoff < 0)
	    {				/* check for silly offset */
		error ("op_bits: silly bit offset %d", bitoff);
		bitoff = 0;
	    }

	    /* adjust for bit offset > 7 */
	    oper[1].constval += (bitoff >> 3) & 0x3;
	    bitoff &= 7;

	    /* bit instructions need a byte address, constval is */
	    /* set up correctly at this point for that requirement */
	    emit_rmem (op -> ocode, bitoff, &oper[1], BYTE);
	    break;

	default:
	    error ("op_bits: Illegal operand type %s", Optypes[oper[1].type]);
	    break;
    }
}

/*
NAME: op_bsub
PURPOSE: handle bsub opcode
PRECONDITIONS:
op is the usual optable pointer.
ALGORITHM:
this routine simply sets up arguments for routine emit_half in aout.c .
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_bsub (op)
struct opcode *op;
{
    register int regs;

    if (oper[0].type != BASE)
	error ("op_bsub: Illegal operand type %s", Optypes[oper[0].type]);

    regs = oper[0].baseval;

    emit_half (op -> ocode | (regs << 4));
}

/*
NAME: op_cvtcw
PURPOSE: handle the cvtcw opcode.
PRECONDITIONS:
op is the usual optable pointer.
the op -> ocode field for this instruction is not set to the instruction
code as it normally would be. the value is a flag value for the first
pass routine var_size in opcode.c .
POSTCONDITIONS:
cvtcw (convert character to word) is a two instruction opcode. the
character is circular shifted right 8 bits and then shifted right 24 bits
to get sign extension.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_cvtcw (op)
register struct opcode *op;
{
    register int mask;

    mask = oper[0].regval << 7;

    emit_half (0x2400 | 8 | mask);	/* rotrw #8,r */
    emit_half (0x1C00 | 24 | mask);	/* asrw #24,r */
}

/*
NAME: op_ex
PURPOSE: handle exr and exrr opcodes
PRECONDITIONS:
op is the usual optable pointer.
the ocoderr field distinguises between the exr and exrr opcodes. the exr
has a low order halfword of 0x0000, exrr has 0x0002.
HISTORY:
Mon Oct 10 20:11:58 EDT 1983 RJM
*/

op_ex (op)
register struct opcode *op;
{
    emit_word ((op -> ocode << 16) | op -> ocoderr |
	(oper[0].regval << 23));
}

/*
NAME: op_file
PURPOSE: handle the file opcode.
PRECONDITIONS:
op is the usual optable pointer.
POSTCONDITIONS:
there are four file instructions, LF, LFBR, STF, and STFBR. the
appropriate one is chosen using the operands.
a file is an 8 word (32 byte) block, except for these instructions (sigh).
consult the processor manual before using them.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_file (op)
register struct opcode *op;
{
    switch (oper[0].type)
    {					/* what moving from? */
	case BASE:
	    switch (oper[1].type)
	    {
		case MEMORY:
		case INDEXED:		/* do STFBR */
		    emit_rmem (0xDC08, oper[0].baseval, &oper[1], WORD);
		    break;

		default:
		    goto badtype;
	    }
	    break;

	case REGISTER:
	    switch (oper[1].type)
	    {
		case MEMORY:
		case INDEXED:		/* do STF */
		    emit_rmem (0xDC00, oper[0].regval, &oper[1], WORD);
		    break;

		default:
		    goto badtype;
	    }
	    break;

	case MEMORY:
	case INDEXED:
	    switch (oper[1].type)
	    {
		case BASE:
		    emit_rmem (0xCC08, oper[1].baseval, &oper[0], WORD);
					/* LFBR */
		    break;

		case REGISTER:
		    emit_rmem (0xCC00, oper[1].regval, &oper[0], WORD);
					/* LF */
		    break;

		default:
		    goto badtype;
	    }
	    break;

	case IMMEDIATE:
	case CONSTANT:
	default:
	  badtype:
	    error ("op_file: Illegal operand type %s %s",
		Optypes[oper[0].type], Optypes[oper[1].type]);
	    break;
    }
}

#ifndef OLDCALL
static int e_stack, e_rmsk, e_save, e_r2flg;
#endif NOT OLDCALL
/*
NAME: op_func
PURPOSE: handle the func opcode.
PRECONDITIONS:
op is the usual optable pointer.
note that the symbol "csav" must be defined for this opcode. csav is the
name of the register save routine.
the external structure tmpsym is initialized in aout.c for use when
emitting constants. the ONLY allowable operations on this structure are
#ifdef POOL
to set the svalue and sspace fields and to use its address in emit_rel calls.
#else not POOL
to set the svalue field and to use its address in emit_rel calls.
#endif POOL
POSTCONDITIONS:
func is a multi instruction opcode. the low order 3 bytes of register 3
is loaded with the address of the routine to be called and then a branch
and link to csav is performed.
there are several different ways to do the load depending upon the type
of the operand.
the first operand should be the argument count of the routine that is to
be called. the value is placed in the high byte of r3. this value is
used by the debuggers to unwind stacks for stack traces.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

#ifndef OLDCALL
extern SYM *curr_func_symp, *func_lab;

#endif NOT OLDCALL
op_func (op)
register struct opcode *op;
{
    OPND t_oper;
    int inst;
    int value;
    enum SEGS oldseg;
#ifdef POOL
    int reltype;
    enum SEGS cseg;
#endif POOL

    /* 2 opers, immediate arg count first */
    if (oper[0].haveseen & NAME)
    {
	/* relocatable constant too hard to do! */
	/* note that emit_rconst (used below) would trash address */
	error ("op_func: first operand must be a constant");
    }
    if ((unsigned) oper[0].constval > 0xFF)
    {
	oper[0].constval = 0;
	warning ("op_func: arg count too large, 0 is used");
    }
    else
    {
	oper[0].constval <<= 24;
    }

#ifndef OLDCALL
    do_align(4);
#ifdef R_SV
    if (e_save) {	/* save live regs across call */
	register int i, m;

	for (i = 6, m = 0xc0; i > 0; i -= 2, m >>= 2) {
	    if (!(value = (m & e_rmsk)))
		continue;
	    inst = 0xD4020000 | i << 23 | 4 * i ;
	    if (value == m)
		inst |= 2;
	    else if (value == (m & (m << 1)))
		inst += 0x00800004;
	    emit_word(inst);
	}
    }
#endif R_SV

		/* emit LA r2,.+4w */
    if (curr_func_symp)
	emit_word(0x51010000 + dot->svalue + 16 - curr_func_symp->svalue);
    else {
	tmpsym.svalue = dot->svalue+16;
	tmpsym.sspace = dot->sspace;
	inst = emit_rel(R_MEMREFINST, R_WORD, &tmpsym, 0);
	inst |= (unsigned) 0x51000000;
	emit_word (inst);
    }
#endif NOT OLDCALL

    switch (oper[1].type)
    {
	case MEMORY:
#ifndef OLDCALL
	    /* build a movw addr,b1 where addr is address of constant */
#else OLDCALL
	    /* build a movw addr,r3 where addr is address of constant */
#endif NOT OLDCALL
	    /* emitted by func_const, note no F or C bits as this is a word */
	    /* assure word alignment (in text space) or the assembler */
	    /* may dump core (but won't dump here, in patch_rel of aout.c) */

	    /* put the branch address in constant space */
#ifdef POOL
	    tmpopnd.haveseen = oper[0].haveseen | oper[1].haveseen;
	    tmpopnd.constval = oper[0].constval | oper[1].constval;
	    tmpopnd.symname = oper[1].symname;
	    tmpsym.svalue = pool_const(&tmpopnd, WORD, &reltype, &cseg);
	    tmpsym.sspace = cseg;
	    do_align (4);
	    inst = emit_rel (reltype, R_WORD, &tmpsym, 0);
#else not POOL
	    oldseg = dot -> sspace;
	    setseg (S_constant);

	    do_align (4);	/* on a word boundary */

	    if (oper[1].haveseen & NAME)
	    {
		value = emit_rel (R_ADDRLITERAL, R_WORD,
		    oper[1].symname, oper[1].constval);
	    }
	    else
	    {
		value = oper[1].constval;
	    }

#ifndef OLDCALL
	    emit_dword (value);
#else OLDCALL
	    emit_dword (oper[0].constval | value);
#endif NOT OLDCALL

	    /* remember where it is, caution required with alignment */
	    /* due to emit_dword, which may align before emitting */
	    tmpsym.svalue = dot -> svalue - 4;

	    setseg (oldseg);

	    do_align (4);
	    inst = emit_rel (R_MEMREFINST, R_WORD, &tmpsym, 0);
#endif POOL
#ifndef OLDCALL
	    inst |= (((unsigned) 0x5C000000) | ((unsigned) 0x1 << 23));
#else OLDCALL
	    inst |= (((unsigned) 0xAC000000) | ((unsigned) 0x3 << 23));
#endif NOT OLDCALL
	    emit_word (inst);

	    break;

	case INDEXED:
#ifndef OLDCALL
	/* if second operand is NOT [b1], then must use TRBR to load b1 */
	    if (!((oper[1].haveseen == BREG)
		    && (oper[1].baseval == 1)))
		emit_half(0x2c81 | (oper[1].regval << 4));	/* TRBR r#,b1 */
#else OLDCALL
	/* if second operand is NOT [r3], then must use LA to load r3 */
	    if (!((oper[1].haveseen == AREG) && (oper[1].regval == 3)))
	    {
		emit_rmem (0x5000, 3, &oper[1], WORD);
	    }

	/* must always emit ORW #oper[0].constval,r3 to get high byte set */
	    emit_rconst (0x8800, 0, 3, &oper[0], WORD);
#endif NOT OLDCALL
	    break;
	
	default:
	    error ("op_func: Illegal operand type %s",
		Optypes[oper[1].type]);
	    emit_word (0);
	    break;
    }

#ifndef OLDCALL
/*
    tmpopnd.haveseen = NAME | CONST;
    tmpopnd.constval = dot->svalue + 8 - curr_func_symp->svalue;
    tmpopnd.symname = curr_func_symp;
    tmpsym.svalue = pool_const(&tmpopnd, WORD, &reltype, &cseg);
    tmpsym.sspace = cseg;
    inst = emit_rel(reltype, R_WORD, &tmpsym, 0);
    inst |= 0xAD000000;
 */
    emit_word(0xD5020000);	/* STW r2,0[b2] */
    emit_word(0xEC010004);	/* BU 1W[b1] */
    if (curr_func_symp)
	inst = emit_rel(R_ADDRLITERAL, R_WORD, curr_func_symp, 0);
    else if (func_lab)
	inst = emit_rel(R_ADDRLITERAL, R_WORD, func_lab, 0);
    else
	inst = 0;
    emit_word(oper[0].constval + inst);
#else OLDCALL
    t_oper.haveseen = NAME;
    t_oper.regval = t_oper.baseval = 0;
    t_oper.constval = 0;
    if ((t_oper.symname = lookup ("csav")) == 0)
    {
	error ("op_func: csav not found");
    }
    else
    {
	emit_rmem (0xF880, 0, &t_oper, WORD);	/* BL csav */
    }
#endif NOT OLDCALL
#ifndef OLDCALL
#ifdef R_SV

    if (e_save) {
	register int i, m;

	for (i = 6, m = 0xc0; i > 0; i -= 2, m >>= 2) {
	    if (!(value = (m & e_rmsk)))
		continue;
	    inst = 0xAC020000 | i << 23 | 4 * i ;
	    if (value == m)
		inst |= 2;
	    else if (value == (m & (m << 1)))
		inst += 0x00800004;
	    emit_word(inst);
	}
    }
#endif R_SV
#endif NOT OLDCALL
}

#ifndef OLDCALL
/*
NAME: op_enter
PURPOSE: process enter pseudoinstruction
PRECONDITIONS:
op is the usual optable pointer.
ALGORITHM:
HISTORY:
85oct21 Gould CSD 3423
*/


op_enter(op)
{
    register int reg;

    do_align(4);

    e_stack = oper[0].constval;
    if (oper[0].symname)
       e_stack += oper[0].symname->svalue;
    e_rmsk = oper[1].constval;
    if (oper[1].symname)
       e_rmsk += oper[1].symname->svalue;
    e_r2flg = e_rmsk & 0x05;
    e_rmsk &= ~0x07;
    {
	register int m;

	for (e_save = 0, m = 0x80; m > 0x04; m >>= 1)
	    if (m & e_rmsk)
		++e_save;
    }

    emit_word(e_stack);
    if (e_stack) {
	emit_word(0x59000000 | e_stack);	/* SUABR b2,frame */
    }
#ifdef E_SV
    if (e_save) {	/* save register variables */
	register int i, m, value, inst;

	for (i = 6, m = 0xc0; i > 0; i -= 2, m >>= 2) {
	    if (!(value = (m & e_rmsk)))
		continue;
	    inst = 0xD4020000 | i << 23 | (4 * i + e_stack);
	    if (value == m)
		inst |= 2;
	    else if (value == (m & (m << 1)))
		inst += 0x00800004;
	    emit_word(inst);
	}
    }
#endif E_SV
}

/*
NAME: op_entry
PURPOSE: process entry psuedoinstruction
PRECONDITIONS:
op is the usual optable pointer.
ALGORITHM:
HISTORY:
85oct21 Gould CSD 3423
*/

op_entry(op)
{
    register int off;

    do_align(4);
    emit_word(e_stack);
    if ((off = dot->svalue - 4 - curr_func_symp->svalue) > 0)
	emit_word(0x58800000 + off);		/* SUABR b1,off */
    else if (off <= 0)
	emit_word(0x58890000 - off);		/* LABR b1,off[b1] */
    if (e_stack)
	emit_word(0x59000000 | e_stack);	/* SUABR b2,frame */
#ifdef E_SV
    if (e_save) {	/* save register variables */
	register int i, m, value, inst;

	for (i = 6, m = 0xc0; i > 0; i -= 2, m >>= 2) {
	    if (!(value = (m & e_rmsk)))
		continue;
	    inst = 0xD4020000 | i << 23 | (4 * i + e_stack);
	    if (value == m)
		inst |= 2;
	    else if (value == (m & (m << 1)))
		inst += 0x00800004;
	    emit_word(inst);
	}
    }
#endif E_SV
}

#endif NOT OLDCALL
/*
NAME: op_intr
PURPOSE: handle ai, ri, di, ei, and dai opcodes.
PRECONDITIONS:
op is the usual optable pointer.
these are privileged non F class I/O instructions. use of any of these
instructions by the faint of heart is NOT recommended.
HISTORY:
Bob Van Valzah, Mon Jun 20 16:24:18 EDT 1983
*/

op_intr(op)
register struct opcode *op;
{
	register int lev;

	if (oper[0].haveseen != CONST) {
		error("op_intr: expected constant operand to %s instruction",
		    op -> oname);
		return;
	}

	lev = oper[0].constval;
	if (lev<0 || lev>127) {
		error("op_intr: priority level of %d not in range 0 to 127",
		    lev);
		return;
	}

	emit_word((op -> ocode << 16) | (lev<<19));
}

/*
NAME: op_io
PURPOSE: handle F class I/O channel opcodes.
PRECONDITIONS:
op is the usual optable pointer.
POSTCONDITIONS:
the instructions emitted are privileged.
ALGORITHM:
note that the parser (yyparse) misbehaves in the case of a register
operand by setting the constval field to the register number (along with
the regval field).
HISTORY:
15 Jul 83	D A Willcox of Compion	Initial coding
*/

op_io (op)
register struct opcode *op;
{
    register int inst;

    inst = 0;

    switch (oper[0].type)
    {
	case MEMORY:
	case INDEXED:
	case REGISTER:
	case IMMEDIATE:
	    if (oper[0].haveseen & BREG)
	    {
		error ("op_io: Base reg not allowed");
		break;
	    }

	    if (oper[0].symname != 0)
	    {
		error ("op_io: Constant required");
		break;
	    }

	    if (oper[0].constval < 0
		|| oper[0].constval > 0x7fff)
	    {
		error ("op_io: constant out of range");
		break;
	    }

	    inst = op -> ocode << 16;
	    inst |= (oper[0].regval << 23);
	    /* yyparse sets constval = register number if REGISTER */
	    if (oper[0].type != REGISTER)
	    {
		inst |= oper[0].constval;
	    }
	    break;

	default:
	    error ("op_io: Illegal operand type %s", Optypes[oper[0].type]);
	    break;
    }

    emit_word (inst);
}

/*
NAME: op_iodev
PURPOSE: handle cd and td opcodes
PRECONDITIONS:
op is the usual optable pointer.
this instruction requires two numbers, the device address and the
command code. the device address must fit in a 7 bit field, the command
code is placed in the low order halfword as it appears on the input
line. note that the command code format is very different between td and
cd and that all 16 bits must be specified.
HISTORY:
Tue Oct 11 09:27:52 EDT 1983 RJM
*/

op_iodev (op)
register struct opcode *op;
{
    if ((oper[0].type == MEMORY) &&
        (oper[1].type == MEMORY) &&
        ((unsigned) oper[0].constval <= 0x7F) &&
        ((unsigned) oper[1].constval <= 0xFFFF))
    {
	emit_word ((op -> ocode << 16) | (oper[0].constval << 19) |
	    oper[1].constval);
    }
    else
    {
	error ("op_iodev: Illegal operand type %s %s",
	    Optypes[oper[0].type], Optypes[oper[1].type]);
    }
}

/*
NAME: op_jmp
PURPOSE: handle branch opcodes.
PRECONDITIONS:
op is the usual optable pointer.
the "D" field of the machine instruction is hard coded in op -> ocode.
POSTCONDITIONS:
unconditional BCT to a register operand uses the TRSW instruction.
any branch to [r0] (indexed on general register 0) also uses TRSW.
note that we have to force a NOP in this one case because indexed with
any other register is a full word (branch) instruction.
the jft opcode uses BFT, the other variants use BCF and BCT.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_jmp (op)
register struct opcode *op;
{
    OPND t_oper;
    switch (oper[0].type)
    {				/* different type operands, diff type jump */
	case IMMEDIATE:
	case CONSTANT:
	    emit_word ((unsigned) op -> ocode << 16
		| oper[0].constval);
	    break;

	case INDEXED:
	    if ((oper[0].regval == 0) && (oper[0].haveseen & AREG))
	    {
		if (oper[0].constval != 0)
		{
		    error ("op_jmp: indexed r0 with nonzero offset in jmp");
		    break;
		}
	    /* sigh, must look like emit_rmem did this */
		do_align (4);
		emit_half (0x2800);	/* TRSW r0 */
		emit_half (0x0002);	/* NOP */
		break;
	    }
	/* otherwise drop into next case */
	case MEMORY:
	    emit_rmem (op -> ocode, 0, &oper[0], WORD);
	    break;

	case REGISTER:
	    if (op -> ocode == (short) 0xEC00)
	    {				/* unconditional br to reg */
		emit_half (0x2800 | (oper[0].regval << 7)); /* trsw */
	    }
	    else
	    {
		if (oper[0].regval == 0)
		{
		    error ("op_jmp: cannot use r0 in jmp");
		    break;
		}

		t_oper = oper[0];
		t_oper.type = MEMORY;
		t_oper.constval = 0;	/* change to ref to mem */
		emit_rmem (op -> ocode, 0, &t_oper, WORD);
	    }
	    break;
	
	default:
	    error ("op_jmp: Illegal operand type %s", Optypes[oper[0].type]);
	    break;
    }
}

/*
NAME: op_mem
PURPOSE: handle single memory operand opcodes.
PRECONDITIONS:
op is the usual optable pointer.
currently, only exm, jwcs, lpsd and lpsdcm opcodes use this routine.
HISTORY:
14 Jul 83	D A Willcox of Compion	Initial coding
*/

op_mem (op)
register struct opcode *op;
{
    switch (oper[0].type)
    {
	case MEMORY:
	case INDEXED:
	    emit_rmem (op -> ocode, 0, &oper[0], op -> olength);
	    break;
	
	case IMMEDIATE:
	case CONSTANT:
	    emit_rconst (op -> ocode, 0, 0, &oper[0], op -> olength);
	    break;

	default:
	    error ("op_mem: Illegal operand type %s", Optypes[oper[0].type]);
	    break;
    }
}

/*
NAME: op_mov
PURPOSE: handle the mov opcode (in its many forms)
PRECONDITIONS:
op is the usual optable pointer.
the three operand form can be detected because of the check made in
routine op1 of opcode.c where the type field of the oper structure for
last + 1 operand is set to OPERROR.
POSTCONDITIONS:
the mov opcode has about 20 variants, depending upon the operands. except
for the 3 operand variant, all are done here. the masked and negative
machine instructions are not used.
movw on two base registers will emit a LABR because there is no machine
instruction to handle this case. B0 cannot be the source to a LABR due
to the way that base register addressing is implemented in the hardware.
movl and movd on two registers will emit two TRR instructions and
generally act like the LD instruction except that the condition codes
will only be set for the even register TRR operation. the source and
destination registers must be even numbered and the destination must be
a general purpose register due to machine instruction limitations.
BEFORE ADDING MORE VARIANTS, ASSURE THAT CONDITION CODES ARE COMPATIBLE.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_mov (op)
register struct opcode *op;
{
    register int inst;

    if (oper[2].type != OPERROR)
    {				/* it's a three-operand move */
	op_mov3 (op);
	return;
    }

    switch (oper[0].type)
    {					/* OK, so what are we moving from? */
	case REGISTER:
	    switch (oper[1].type)
	    {				/* mov from register to where? */
		case BASE:
		    if (op -> olength == LONG)
		    {
			error ("op_mov: Long move to base register");
		    }
		    emit_half (0x2C01 | (oper[0].regval << 4) |
			(oper[1].baseval << 7));	/* TRBR */
		    break;

		case REGISTER:
		    if (op -> olength == LONG)
		    {
		    /* pass 1 validate_op will check even reg */
			emit_half (0x2C00 | ((oper[0].regval + 1) << 4) |
			    ((oper[1].regval + 1) << 7));	/* TRR */
		    }
		    emit_half (0x2C00 | (oper[0].regval << 4) |
			(oper[1].regval << 7));		/* TRR */
		    break;

		case MEMORY:
		case INDEXED:
		    emit_rmem (0xD400, oper[0].regval, &oper[1], op -> olength);
					    /* STx R,mem */
		    break;

		default:
		    goto baddest;		/* mov r,const is bad */
	    }
	    break;

	case BASE:
	    switch (oper[1].type)
	    {				/* mov from breg to where? */
		case BASE:		/* must be done with load address */
		    if (op -> olength == LONG)
		    {
			error ("op_mov: Long move to base register");
		    }
	/* note there are no base to base machine instructions, and B0 */
	/* does not do the right thing with a LABR instruction, so ... */
		    if (oper[0].baseval == 0)
		    {
	    error ("op_mov: Illegal operand type, base reg 0 as source");
		    }
		    inst = 0x58080000;	/* LABR */
		    inst |= oper[0].baseval << 16;
		    inst |= oper[1].baseval << 23;
		    emit_word (inst);
		    break;

		case REGISTER:
		    if (op -> olength == LONG)
		    {
		    /* pass 1 validate_op will check even reg */
			emit_half (0x2C02 | ((oper[0].baseval + 1) << 4) |
			    ((oper[1].regval + 1) << 7));	/* TBRR */
		    }
		    emit_half (0x2C02 | (oper[0].baseval << 4) |
			(oper[1].regval << 7));	/* TBRR */
		    break;

		case MEMORY:
		case INDEXED:
		    emit_rmem (0x5400, oper[0].baseval, &oper[1],
			op ->olength);		/* STWBR B,mem */
		    break;

		default:
		    goto baddest;		/* mov r,const is bad */
	    }
	    break;

	case CONSTANT:
	case IMMEDIATE:
	    if ((oper[0].haveseen & NAME) == 0
		    && oper[0].constval == 0)
	    {				/* mov of const 0 is clear */
		switch (oper[1].type)
		{			/* what are we trying to clear? */
		    case REGISTER:
			emit_half (0x0C00 | (oper[1].regval << 4) |
			    (oper[1].regval << 7));	/* ZR R,R */
			if (op -> olength == LONG)
			{
			    emit_half (0x0C00 | ((oper[0].regval + 1) << 4) |
				((oper[1].regval + 1) << 7));
			}
			break;

		    case BASE:
			if (op -> olength == LONG)
			{
			    error ("op_mov: Long move to base register");
			}
			emit_word (0x58080000 	/* use labr */
			    | oper[1].baseval << 23);
			break;
		    
		    case MEMORY:
		    case INDEXED:
			emit_rmem (0xF800, 0, &oper[1], op -> olength);
			break;

		    default:
		      baddest:
	    error ("op_mov: Illegal operand type %s", Optypes[oper[1].type]);
			break;
		}
	    }
	    else
	    {				/* it is a real constant */
		switch (oper[1].type)
		{			/* what kind of destination? */
		    case REGISTER:
			emit_rconst (0xAC00,
				0xC800,
				oper[1].regval,
				&oper[0],
				op -> olength);
			break;

		    case BASE:
			if (op -> olength == LONG)
			{
			    error ("op_mov: Long move to base register");
			}
			if (oper[0].constval >= 0
			    && oper[0].constval <= 0xffff
			    && oper[0].haveseen == CONST)
			{			/* can use labr */
			    emit_word (0x58080000 
				| oper[1].baseval << 23
				| oper[0].constval);
			}
			else
			{
			    emit_rconst (0x5C00,	/* use lwbr */
				0,
				oper[1].baseval,
				&oper[0],
				op -> olength);
			}
			break;

		    default:
			goto baddest;
		}
	    }
	    break;
		    
	case MEMORY:
	case INDEXED:
	    switch (oper[1].type)
	    {					/* move memory to what? */
		case REGISTER:
		    emit_rmem (0xAC00, oper[1].regval, &oper[0], op -> olength);
						/* LW */
		    break;

		case BASE:
		    emit_rmem (0x5C00, oper[1].baseval, &oper[0],
		        op -> olength);		/* LWBR */
		    break;

		default:
		    goto baddest;
	    }
	    break;

	default:
	    error ("op_mov: Illegal operand type %s", Optypes[oper[0].type]);
	    break;
    }
}

/*
NAME: op_mov3
PURPOSE: handle the 3 operand mov opcode
PRECONDITIONS:
op is the usual optable pointer, but this routine is called from op_mov,
not from op1 in opcode.c as is normally done.
the first and third operands must be memory accesses, the second must be
a register.
POSTCONDITIONS:
this is a multi instruction opcode that generates 2 mov ops. the first
loads a register, the second loads a memory location from the register.
the first instruction could be a halfword if the first operand allows,
but this is determined by routine emit_rconst of aout.c .
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_mov3 (op)
register struct opcode *op;
{
    int load;
    int loadi;
    int store;
    int r;

    switch (oper[1].type)
    {					/* get instr's for load/store */
	case BASE:
	    if (op -> olength == LONG)
	    {
		error ("op_mov3: Long move to base register");
	    }
	    load = 0x5C00;
	    store = 0x5400;
	    loadi = 0;
	    r = oper[1].baseval;
	    break;

	case REGISTER:
	    load = 0xAC00;
	    if ((oper[0].haveseen == CONST) &&
		(oper[0].constval == 0))
	    {
		loadi = 0x0C00;	/* use a ZR */
	    }
	    else
	    {
		loadi = 0xC800;	/* use a load immediate */
	    }
	    store = 0xD400;
	    r = oper[1].regval;
	    break;

	default:
	  badop:
	    error ("op_mov3: Illegal operand type %s %s %s",
		Optypes[oper[0].type], Optypes[oper[1].type],
		Optypes[oper[2].type]);
	    return;
    }

    switch (oper[0].type)
    {					/* first, load source value */
	case MEMORY:
	case INDEXED:
	    emit_rmem (load, r, &oper[0], op -> olength);
	    break;

	case CONSTANT:
	case IMMEDIATE:
	    if (oper[1].type == BASE
		&& oper[0].constval >= 0
		&& oper[0].constval < 0x10000)
	    {				/* can use LABR */
		emit_word (0x58080000 
			| (r << 23)
			| (oper[0].constval));
	    }
	    else if (loadi == 0x0C00)	/* this is the ZR case */
	    {
		emit_half (0x0C00 | (r << 4) | (r << 7));	/* ZR R,R */
		if (op -> olength == LONG)
		{
		    emit_half (0x0C00 | ((r + 1) << 4) |
			((r + 1) << 7));
		}
	    }
	    else
	    {
		emit_rconst (load, loadi, r, &oper[0], op -> olength);
	    }
	    break;
	
	default:
	    goto badop;
    }

    switch (oper[2].type)
    {					/* now, store to dest */
	case MEMORY:
	case INDEXED:
	    emit_rmem (store, r, &oper[2], op -> olength);
	    break;

	default:
	    goto badop;
    }
}

/*
NAME: op_notyet
PURPOSE: handle the op codes we don't currently handle
PRECONDITIONS:
op is the usual optable pointer.
HISTORY:
part of initial coding RJM DAW
*/

op_notyet(op)
struct opcode *op;
{
    error ("op_notyet: %s not implemented", op -> oname);
    /* identical to SKIPEOS except parse.h not included here */
    while (yylinept && (*yylinept != '\n' && *yylinept != ';'))
    {
	yylinept++;
    }
}

/*
NAME: op_off
PURPOSE: handle the movea and subea opcodes.
PRECONDITIONS:
op is the usual optable pointer.
the ocode field of the optable is set on the assumption of use of a base
register destination.
POSTCONDITIONS:
for base register destination, LABR is used for movea and SUABR is used
for subea. for a register destination, LA is used for both cases.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial coding
*/

op_off (op)
register struct opcode *op;
{
    register int code;
    register int r;
    
    code = op -> ocode;

    if (oper[1].type == REGISTER)
    {
	code = 0x5000;		/* if gen register, fudge opcode */
	r = oper[1].regval;
    }
    else if (oper[1].type == BASE)
    {
	r = oper[1].baseval;
    }
    else
    {
	error ("op_off: Illegal operand type %s",
	    Optypes[oper[1].type]);
    }

    emit_rmem (code, r, &oper[0], WORD);
}

#ifndef OLDCALL
/*
NAME: op_retn
PURPOSE: process retn pseudoinstruction
PRECONDITIONS:
prior occurrence of enter pseudoinstruction has established
ALGORITHM:
this routine emits code for function return
HISTORY:
85oct21 Gould CSD 3423
*/

op_retn()
{
    do_align(4);
#ifdef E_SV

    if (e_save) {		/* restore register variables */
	register int i, m, value, inst;

	for (i = 6, m = 0xc0; i > 0; i -= 2, m >>= 2) {
	    if (!(value = (m & e_rmsk)))
		continue;
	    inst = 0xAC020000 | i << 23 | (4 * i + e_stack);
	    if (value == m)
		inst |= 2;
	    else if (value == (m & (m << 1)))
		inst += 0x00800004;
	    emit_word(inst);
	}
    }
#endif E_SV

    if (e_r2flg != 5)
	emit_word(0xAC000000 | 2 << 23 | 2 << 16 | e_stack);
    emit_word(0x5C000000 | 1 << 23 | 2 << 20);
    if (e_stack)
	emit_word(0x58080000 | 2 << 23 | 2 << 16 | e_stack);
    emit_word(0xEC000004 | 2 << 20);
}

#endif NOT OLDCALL
/*
NAME: op_rroper
PURPOSE: handle register to register operands
PRECONDITIONS:
op is the usual optable pointer.
ALGORITHM:
this routine simply sets up arguments for routine emit_half in aout.c .
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_rroper (op)
struct opcode *op;
{
    register int regs;
    register int regd;

    switch (oper[0].type)
    {
	case REGISTER:
	    regs = oper[0].regval;
	    break;

	case BASE:
	    regs = oper[0].baseval;
	    break;
	
	default:
	  badop:
	    error ("op_rroper: Illegal operand type %s", Optypes[oper[0].type]);
	    return;
    }

    if (oper[1].type == OPERROR)
    {
	regd = regs;
	regs = 0;
    }
    else
    {
	switch (oper[1].type)
	{
	    case REGISTER:
		regd = oper[1].regval;
		break;

	    case BASE:
		regd = oper[1].baseval;
		break;
	    
	    default:
		goto badop;
	}
    }

    emit_half (op -> ocode | (regs << 4) | (regd << 7));
}

/*
NAME: op_shift
PURPOSE: handle the shift opcodes.
PRECONDITIONS:
op is the usual optable pointer.
POSTCONDITIONS:
the register-register variant of each opcode is multi instruction. the
first operand register is massaged with an ANMH and ORMH so that it
contains the fullword shift instruction, which is then executed by EXRR.
the other variants are "normal" instructions.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial Coding
*/

op_shift (op)
register struct opcode *op;
{
    register int inst;
    static OPND andop = {CONSTANT, CONST, 0, 0, 0x1f, 0};
					/* used in ANMW r,#~0x1f */
    static OPND orop = {CONSTANT, CONST, 0, 0, 0, 0};
					/* used in ORMW r,#<op> */

    switch (oper[0].type)
    {				/* const or register shift? */
	case IMMEDIATE:
	case CONSTANT:			/* this is easy, const shift */
	    if (oper[0].haveseen & NAME)
	    {
		error ("op_shift: must shift by const");
	    }

	    if (oper[0].constval > 31 || oper[0].constval < 0)
	    {
		warning ("op_shift: shift is too long");
		oper[0].constval %= 32;
	    }

	    inst = op -> ocode;
	    inst |= oper[0].constval;
	    inst |= oper[1].regval << 7;
	    emit_half (inst);
	    break;
	
	case REGISTER:			/* this is nasty, register shift */
	    emit_rconst (0x8400, 0, oper[0].regval, &andop, HALF);
					/* ANMH r,#0x1f */
	    orop.constval = op -> ocode | (oper[1].regval << 7);
	    emit_rconst (0x8800, 0, oper[0].regval, &orop, HALF);
					/* ORMH r,#<shift op> */
	    emit_word (0xC8070002 | (oper[0].regval << 23));
					/* EXRR r */
	    break;

	default:
	    error ("op_shift: Illegal operand type %s", Optypes[oper[0].type]);
	    break;
    }
} 

/*
NAME: op_svc
PURPOSE: handle the svc opcode.
PRECONDITIONS:
op is the usual optable pointer.
HISTORY:
24 Jun 83	D A Willcox of Compion	Initial coding
*/

op_svc (op)
register struct opcode *op;
{
    if (oper[0].haveseen != CONST || oper[1].haveseen != CONST)
    {
	error ("op_svc: operands of %s must be constant", op -> oname);
    }
    else
    {
	if (oper[0].constval < 0 
	    || oper[0].constval > 0xf
	    || oper[1].constval < 0
	    || oper[1].constval > 0xfff)
	    error ("op_svc: operands must be 0-0xF,0-0xFFF");
    }
    
    emit_word (((unsigned) op -> ocode << 16)
	| (oper[0].constval << 12)
	| (oper[1].constval));
}

/*
NAME: op_xchg
PURPOSE: handle xchg opcode
PRECONDITIONS:
op is the usual optable pointer.
there are no machine instructions for exchanging base and general
purpose register values, hence the operands must be of the same
register type.
HISTORY:
16 Jun 83	D A Willcox of Compion	Initial coding
*/

op_xchg (op)
register struct opcode *op;
{
    if (oper[0].type != oper[1].type)
    {					/* type mismatch */
	error ("op_xchg: register type mismatch");
    }
    else
    {
	if (oper[0].type == BASE)
	{
	    emit_half (0x2802 | (oper[0].baseval << 4) |
		(oper[1].baseval << 7));	/* xcbr */
	}
	else
	{
	    if (oper[0].type == REGISTER)
		emit_half (0x2C05 | (oper[0].regval << 4) |
		    (oper[1].regval << 7));	/* xcr */
	    else
	    {
		error ("op_xchg: Illegal operand type %s",
		    Optypes[oper[0].type]);
	    }
	}
    }
}

/*
NAME: op_zero
PURPOSE: handle halfword aligned op codes that have no operands.
PRECONDITIONS:
op is the usual optable pointer.
*/

op_zero(op)
struct opcode *op;
{
    emit_half (op -> ocode);
}

/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */
