/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: builder.c,v 5.2 88/08/30 01:10:29 pcc Released $";
#endif

/*
NAME: main of builder
PURPOSE: build a hashed opcode table for the Gould assembler
PRECONDITIONS:
this is a standalone utility used to support the assembler.
the header file optab1.h should contain the raw opcode data in a similar
format to what this program emits.
the hashing function itself should be in procedure oplook in the file
ophash.c .
POSTCONDITIONS:
a file containing procedure declarations and the definition for a hashed
opcode table is output. the default filename is ops.h . this may be
changed by specifying a name on the command line.
includes in the assembler will access the file optable.h, which should be
the name given to the output of this program when a satisfactory hashing
function is found.
statistics on table size and hash collisions are output to stdout so that
the function may be tuned manually or thrown out completely if necessary.
HISTORY:
part of release 2 mods RJM
Sun Jul 24 21:27:08 EDT 1983
*/

#include <stdio.h>
#include "optab1.h"
#ifdef TEST
int MAX_HASH;
int HASH_BASE;
#else
#include "ophash.h"
#endif

/* MAX_TABLE used only by this program, size of hash_val and func_name */
#define MAX_TABLE 300

typedef unsigned char Tuchar;

Tuchar *outfile = (Tuchar *) "ops.h";
int count = 0;

struct collisions
{
    struct opcode *ptr;
    int collides;
} hash_val[MAX_TABLE];

Tuchar *func_name[MAX_TABLE];
int func_count = 0;
int pr_flag = 0;

FILE *new_table;

main (argc, argv)
char **argv;
{
    register char **ap;
    register struct opcode *opptr;
    int i;
    int chain;
    int avg_chain;
    int max_chain;
    int noncollide;
    int next_hash;

    /* handle command line arguments */
    for (ap = argv+1; *ap; ap++)
    {
	char *cp = *ap;
	if (*cp != '-')
	{
	    outfile = (Tuchar *) cp;
	}
	else
	{
	    switch (cp[1])
	    {
	    case 'p':
		pr_flag = 1;
		break;
	    default:
		fprintf (stderr, "bad command line option specified %s\n", cp);
		exit (1);
	    }
	}
    }

    /* determine size of optab1.h opcode table */
    for (opptr = optable, count = 0; opptr -> oname; opptr++, count++)
    {
    }

    printf ("builder: number of opcodes is %d\n\n", count);
#ifdef TEST
    for (MAX_HASH = count + 1; MAX_HASH < MAX_TABLE; MAX_HASH++)
	for (HASH_BASE = 7; HASH_BASE < 18; HASH_BASE++)
	{
#endif

    noncollide = 0;
    for (i = 0; i < MAX_HASH; i++)
    {
	hash_val[i].ptr = 0;
	hash_val[i].collides = 0;
    }

    if (pr_flag)
    {
	printf ("MAX_HASH %d HASH_BASE %d\n", MAX_HASH, HASH_BASE);
    }
    if (count > MAX_HASH)
    {
	fprintf (stderr, "builder: opcode count greater than MAX_HASH\n");
	exit (1);
    }

    /* set hash for each table entry and set up function declarations */
    for (opptr = optable; opptr -> oname; opptr++)
    {
#ifndef TEST
	save_func (opptr -> o2data);
#endif
	i = oplook (opptr -> oname);
	if (hash_val[i].ptr)
	{
	    if (strcmp (hash_val[i].ptr -> oname, opptr -> oname) == 0)
	    {
		fprintf (stderr, "builder: duplicate op codes found - %s\n",
		    opptr -> oname);
		exit (1);
	    }
	    hash_val[i].collides++;
	    next_hash = (i + 1 < MAX_HASH) ? i + 1 : 0;
	    while (hash_val[next_hash].ptr)
	    {
		if (strcmp (hash_val[next_hash].ptr -> oname,
		    opptr -> oname) == 0)
		{
		    fprintf (stderr,
			"builder: duplicate op codes found - %s\n",
			opptr -> oname);
		    exit (1);
		}
		next_hash = (++next_hash < MAX_HASH) ? next_hash : 0;
	    }
	    hash_val[next_hash].ptr = opptr;

	    if (pr_flag)
	    {
		printf ("builder: collision - %d - %s - %s\n",
		    i, hash_val[i].ptr -> oname, opptr -> oname);
	    }
	}
	else
	{
	    hash_val[i].ptr = opptr;
	    noncollide++;
	}
    }

    if (pr_flag)
    {
	printf ("\nbuilder: non-collision count - %d\n\n", noncollide);
	fflush (stdout);
    }

    /* check for secondary collisions caused by chaining into */
    /* another allocated spot in the array. */
    avg_chain = 0;
    max_chain = 0;
    for (opptr = optable; opptr -> oname; opptr++)
    {
	i = oplook (opptr -> oname);
		/* Kernel P bit */
	if (hash_val[i].ptr &&
		strcmp (hash_val[i].ptr -> oname, opptr -> oname))
	{
	    chain = 1;
	    if (hash_val[(i + 1 < MAX_HASH) ? i + 1 : 0].ptr &&
		(hash_val[(i + 1 < MAX_HASH) ? i + 1 : 0].ptr != opptr))
	    {
		if (pr_flag)
		{
		    printf ("builder: secondary collision - %d - %s\n",
			i, opptr -> oname);
		}
		while (hash_val[(++i < MAX_HASH) ? i : (i = 0)].ptr &&
		    (hash_val[i].ptr != opptr))
		{
		    chain++;
		    if (pr_flag)
		    {
			printf ("%s ", hash_val[i].ptr -> oname);
		    }
		}
		if (pr_flag)
		{
		    printf ("\n");
		    fflush (stdout);
		}
	    }
	    avg_chain += chain;
	    max_chain = (max_chain < chain) ? chain : max_chain;
	}
    }

    if (pr_flag)
    {
	printf ("\nbuilder: avg chain length - %.2f\n", 
	    (float) avg_chain / (count - noncollide));
	printf ("builder: max chain length - %d\n", max_chain);
	fflush (stdout);
    }

#ifdef TEST
	if ((noncollide > 125 && max_chain < 20) || max_chain < 5)
	{
	    printf ("MAX_HASH %d HASH_BASE %d\n", MAX_HASH, HASH_BASE);
	    printf ("builder: non-collision count - %d\n", noncollide);
	    printf ("builder: avg chain length - %.2f\n", 
		(float) avg_chain / (count - noncollide));
	    printf ("builder: max chain length - %d\n\n", max_chain);
	}
    }		/* end the sampling for loops */
#endif

    if (pr_flag)
    {
	printf ("\n");
	fflush (stdout);
	for (i = 0; i < MAX_HASH; i++)
	{
	    if (hash_val[i].collides)
		printf ("index %d collides %d\n", i, hash_val[i].collides);
	}
	printf ("\n");
	fflush (stdout);
    }
    
#ifndef TEST
    /* now create the new header file */
    new_table = fopen(outfile, "w");

    fprintf (new_table,
	"/* THIS TABLE IS MACHINE GENERATED, SEE builder.c */\n\n");

    func_print ();

    fprintf (new_table, "struct opcode optable[] =\n{\n");
    for (i = 0; i < MAX_HASH; i++)
    {
	opptr = hash_val[i].ptr;
	if (pr_flag)
	{
	    printf ("index %d is %s - oplook returns %d\n", i,
		(opptr) ? opptr -> oname : "EMPTY",
		(opptr) ? oplook(opptr -> oname) : 0);
	}
	if (opptr)
	{
	    fprintf (new_table,
	    "\t\"%s\", 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, %s,\n",
		opptr -> oname, opptr -> ocode & 0xFFFF,
		opptr -> ocoderr & 0xFFFF, opptr -> ocoderm & 0xFFFF,
		opptr -> ocodeim & 0xFFFF, opptr -> olength,
		opptr -> otype, opptr -> onopnd, opptr -> o1data,
		opptr -> o2data);
	}
	else
	{
	    fprintf (new_table, "\t0, 0, 0, 0, 0, 0, 0, 0, 0, 0,\n");
	}
    }
    fprintf (new_table, "\t0,\n};\n");
    fclose (new_table);
#endif

}

/*
NAME: func_print
PURPOSE: print the external function declarations to outfile
PRECONDITIONS:
this routine should only be called once, with the output placed just
after the "machine generated" banner line.
*/

func_print ()
{
    register int i;
    register int j = 0;
    register int need_decl = 1;

    for (i = 0; i < func_count; i++)
    {
	if (need_decl)
	{
	    fprintf (new_table, "extern int ");
	    need_decl = 0;
	}
	fprintf (new_table, "%s ()", func_name[i]);
	if ((++j < 5) && (i < (func_count - 1)))
	{
	     fprintf (new_table, ", ");
	}
	else
	{
	     fprintf (new_table, ";\n");
	     need_decl = 1;
	     j = 0;
	}
    }
    fprintf (new_table, "\n");
}

/*
NAME: save_func
PURPOSE: keep track of functions that must be declared in outfile
PRECONDITIONS:
parameter name is the name of a function used in the optable.
POSTCONDITIONS:
if the function name has not been seen before, put it into the func_name
table and increment func_count, otherwise just return.
*/

save_func (name)
Tuchar *name;
{
    register int i;

    for (i = 0; i < func_count; i++)
    {
	if (strcmp (name, func_name[i]) == 0)
	    return;
    }

    func_name[func_count] = name;
    func_count++;
}

/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */
