Module ELowSends;

{ *************************** Lowest Level Sends *************************** }
{
    These make no assumptions about where buffers come from or any of that
    stuff.  SendNoAck sends a packet and returns, not waiting for an
    acknowledgement.  It is used for sending acknowledgements and for
    sending to groups, where there no acknowledging is done.  It is pretty
    simple.  SendWAck is more complicated, since it sends a packet and waits
    for an acknowledgement.  It has three possible results:  packet was
    received, timeout, and send not allowed (send-within-a-send).  The latter
    error is recognized when someone is already waiting for an acknowledge.
    SendWAck communicates with the receiver through some global variables.

    ELS_Receive should be called by the receiver interrupt handler whenever
    an acknowledge or negative acknowledge is received.  It will deal with
    the sending stuff.
}

Exports

Imports Ether10IO from Ether10IO;

Type    SWA_type = (SWA_Received, SWA_Timeout, SWA_Error);

Procedure ELS_Init (Timeout, Retries: Integer);

Function SendWAck (header: pEtherHeader; buffer: pEtherBuffer;
                   status: pEtherStatus; size, id: Integer): SWA_type;

Procedure SendNoAck (header: pEtherHeader; buffer: pEtherBuffer;
                     status: pEtherStatus; size: Integer);

Procedure ELS_Receive (FromAddr: EtherAddress; ID: Integer; Pos_Neg: Boolean);

Private

Imports Timer from Timer;

Var     WaitFrom: EtherAddress;
        WaitID: Integer;
        State: (Idle, Waiting, RcvdPos, RcvdNeg);
        ELSTimeout, ELSRetries: Integer;

Procedure ELS_Init (Timeout, Retries: Integer);

        Begin
        State := Idle;
        if Timeout < 1
            then ELSTimeout := 1
            else ELSTimeout := Timeout;
        if Retries < 1
            then ELSRetries := 1
            else ELSRetries := Retries;
        End;

Procedure SendNoAck (header: pEtherHeader; buffer: pEtherBuffer;
                     status: pEtherStatus; size: Integer);

        Begin
        repeat
            E10WIO (ESend, header, buffer, status, size);
            until not status^ . SendError;
        End;

Function SendWAck (header: pEtherHeader; buffer: pEtherBuffer;
                   status: pEtherStatus; size, id: Integer): SWA_type;

        Var     tries: Integer;

        Begin
        if state = Idle then
            begin
            WaitFrom := header^ . Dest;
            WaitID := id;
            tries := 1;
            repeat                  { repeat when timed out or neg ack }
                state := Waiting;
                SendNoAck (header, buffer, status, size);
                StartTimeout (ELSTimeout);
                while (not Timeout) and (state = Waiting) do;
                if state = RcvdPos then     { Success! }
                    begin
                    SendWAck := SWA_Received;
                    state := Idle;
                    Exit (SendWAck);
                    end;
                if state = Waiting then     { Timed out }
                    tries := tries + 1;
                { otherwise received neg ack => try again }
                until tries > ELSRetries;
            SendWAck := SWA_Timeout;
            state := Idle;
            Exit (SendWAck);
            end;
        SendWAck := SWA_Error;
        End;

Procedure ELS_Receive (FromAddr: EtherAddress; ID: Integer; Pos_Neg: Boolean);

        Begin
        if state = Waiting then
            begin
            if (FromAddr . High = WaitFrom . High) and
                        (FromAddr . Mid = WaitFrom . Mid) and
                        (FromAddr . Low = WaitFrom . Low) and
                        (ID = WaitID) then
                begin
                if Pos_Neg
                    then state := RcvdPos
                    else state := RcvdNeg;
                end;
            end;
        End.
