module DiskIO;
{---------------------------------------------------------------------------
{
{Abstract:
{
{This module implements the basic low level operations to disk devices.
{It services the Hard Disk and the Floppy.  When dealing with the floppy
{here, the structures on the hard disk are mapped to the structures
{on the floppy.
{--------------------------------------------------------------------------}

{---------------------------------------------------------------------------
 Change history.


  12 Jan 82  BAM  V3.13 Fix bug in FloppyIO failure so deallocates storage.
  24 Jun 81  BAM  V3.12 Fix to prevent retry if floppy write and device not
                          writable.
  28 May 81  BAM  V3.11 Fix bug in floppy header die-On-error.
                        Add Recalibrate light to floppy
                        New light definitions
  26 May 81  JPS  V3.10 Use new Lights module.
  19 May 81  BAM  V3.9  Fixed position of Reset light
  12 May 81  BAM  V3.8  Removed print out for DiskErrors; more accurate print
                          out of errors when happen; blink "light" during
                          IOReset.
                        Added new Exceptions DiskFailure and BadDevice;
                        Removed Procedure DiskError;
                        Use new IO
   6 May 81  JPS  V3.7  Fix bug in DiskReset by using the new form of the
                          SetCylinder StartIO.
  20 Apr 81  JPS  V3.6  Use DiskSegment consistently.
                        Set DiskSegment as UnSwappable for virtual memory.
   9 Apr 81  BAM  V3.5  Fix Retry so no recal on last time; DiskError tell op.
                         Fixed bug in DoDiskIO exit
  30 Mar 81  BAM  V3.4  Added Retry count to TryDisk and Const NumTries.
  27 Mar 81  BAM  V3.3  Added comments and WriteLn to DiskError.
  19 Mar 81  BAM  V3.2  Combined JPS+GGR and BAM's changes.
  17 Mar 81  GGR  V3.1  Removed WinchDebug and FloppyDebug.
  17 Mar 81  JPS  V3.0  Changed MapAddr and UnMapAddr to handle 24 Mbyte
                          drives.
                        Changed harddisk interlace factor to one.
                        Removed partition kind 'node'.
                        Added TryDiskIO, and changed DiskError printout.
                        Changed FirstDiskBlock for harddisks and
                        LastDiskBlock for 24 Mbyte disks.
                        Improved retry and retry messages from DiskIO and
                          FloppyIO.
  16 Mar 81  BAM  V2.2  Changed directory to have extra bits
   5 Mar 81  BR   V2.1  Added comments
   5 Mar 81  BAM  V2.0  Changed Definitions and import FileDefs.
   3 Mar 81  JPS  V1.4  Fix DiskReset to agree with IO V4.4.
   1 Mar 81  BR   V1.3  Change FileDate to TimeStamp.
   1 Mar 81  JPS  V1.2  Export the DiskReset routine.
  28 Feb 81  JPS  V1.1  Get rid of the LogMapping parameter to MapAddr by
                          teaching UnitIO not to fool with a hard disk address.
---------------------------------------------------------------------------}

{******************} exports {***************************}

imports Arith from Arith;
imports FileDefs from FileDefs;
imports IOErrors from IOErrors;

const
    HARDNUMBER          = 0;             {device code of Shugart Disk}
    FLOPPYNUMBER        = 1;             {device code of FloppyDisk}

    {a Disk Address can be distinguished from a Segment Address by the
      upper two bits (in 32 bits).  These bits have a nonzero code to
      which disk the address is part of}

    RECORDIOBITS        = #140000;       {VirtualAddress upper 16 bits of disk}
    DISKBITS            = RECORDIOBITS + (HARDNUMBER*(#20000));
    FLOPBITS            = RECORDIOBITS + (FLOPPYNUMBER*(#20000));

    {The following definitions tell how many entries there are in the
     three pieces of the random index.  The first piece (Direct)
     are blocks whose DiskAddresses are actually contained in the
     Random Index (which is part of the FileInformationBlock).
     The second section has a list of blocks each of which contain 128 Disk
     Addresses of blocks in the file, forming a one level indirect addressing
     scheme.
     For very large files, the third section (DblInd) has DiskAddresses of
     blocks which point to other blocks which contain 128 DiskAddresses
     of blocks in the file, forming a two level indirect scheme.}

    DIRECTSIZE          =  64; { Entries in FIB of blocks directly accessable }
    INDSIZE             =  32; { Entries in FIB of 1 level indirect blocks }
    DBLINDSIZE          =   2; { Entries in FIB of 2 level indirect blocks }

    FILESPERDIRBLK      = 16;  { 256 / SizeOf(DirEntry) }
    NUMTRIES            = 15;  { number of tries at transfer before aborting }
    
type
    {Temporary segments go away when processes are destroyed,
     Permanent segments persist until explicitly destroyed
     Bad Segments are not well formed segments which are not
     readable by the Segment system}

    SpiceSegKind  = (Temporary, Permanent, Bad);
    PartitionType = (Root, UnUsed, Leaf); {A Root Partition is a device}
    DeviceType    = (Winch12, Winch24, FloppySingle, FloppyDouble,
                     UnUsed1, UnUsed2);

    MyDble = Array [0..1] of integer;
    
    DiskCheatType = record
                      case integer of
                        1: (
                             Addr    : DiskAddr
                           );
                        2: (
                             Dbl     : MyDble { should be IO.Double but don't
                                                import IO in export section }
                           );
                        3: (
                             Seg     : SegID
                           );
                        4: (
                             Lng     : FSBit32
                           )
                       end;
                       
    { A directory is an ordinary file which contains SegIDs of files
      along with their names.  Directories are hash coded by file name
      to make lookup fast.  They are often sparse files (ie contain
      unallocated blocks between allocated blocks).  The file name is a
      SimpleName, since a directory can only contain entries for
      files within the partition (and thus device) where the directory
      itself is located }

    DirEntry   = packed record 
                   InUse    : boolean; {true if this DirEntry is valid}
                   Deleted  : boolean; {true if entry deleted but not expunged}
                   Archived : boolean; {true if entry is on backup tape}
                   UnUsed   : 0..#17777;   {reserved for later use}
                   ID       : SegID;
                   Filename : SimpleName
                 end;
                 

    DiskBuffer = packed record
                   case integer of
                     1: (
                          Addr : array [0..(DISKBUFSIZE div 2)-1] of DiskAddr
                        );
                     2: (
                          IntData : array [0..DISKBUFSIZE-1] of FSBit16
                        );
                     3: (
                          ByteData : packed array [0..DISKBUFSIZE*2-1] of FSBit8
                        );
  {4 is format of the FileInformationBlock; the FIB has Logical Block -1 }
                     4: ( 
                          
                          FSData     : FSDataEntry;

                          {The Random Index is a hint of the DiskAddresses
                           of the blocks that form the file.
                           It has three parts as noted above.  Notice
                           that all three parts are always there, so
                           that even in a very large file, the first
                           DIRECTSIZE blocks can be located quickly
                           The blocks in the Random index have logical
                           block numbers that are negative.  The logical
                           block number of Indirect[0] is -2 (the FIB is -1)
                           the last possible block's number is 
                           -(INDSIZE+DBLINBDSIZE+1)}

                          Direct     : array [0..DIRECTSIZE-1] of DiskAddr;
                          Indirect   : array [0..INDSIZE-1]    of DiskAddr;
                          DblInd     : array [0..DBLINDSIZE-1] of DiskAddr;
                          
                          SegKind    : SpiceSegKind;
                          
                          NumBlksInUse : integer; {segments can have gaps, 
                                                  block n may exist when block 
                                                  n-1 has never been allocated.
                                                  NumBlksInUse says how many
                                                  data blocks are actually used
                                                  by the segment}
                          LastBlk    : FSBit16;  {Logical Block Number of 
                                                  largest block allocated}
                          LastAddr   : DiskAddr; {DiskAddr of LastBlk }
                          LastNegBlk : FSBit16;  {Logical Block Number of
                                                  largest pointer block
                                                  allocated}
                          LastNegAddr: DiskAddr  {Block number of LastNegBlk}
                        );
 {5 is the format of the DiskInformationBlock}
                      5: (
                          
                          {The Free List is a chain of free blocks linked
                          by their headers }

                          FreeHead   : DiskAddr; {Hint of Block Number of the
                                                  head of the free list}
                          FreeTail   : DiskAddr; {Hint of Block Number of the
                                                  tail of the free list}
                          NumFree    : FSBit32;  {Hint of how many blocks
                                                  are on the free list}
                          RootDirID  : SegID;    {where to find the Root
                                                  Directory}
                          BadSegID   : SegID;    {where the bad segment is}

                          {when booting, the boot character is indexed into
                           the following tables to find where code to be
                           boot loaded is found }

                          BootTable  : array [0..25] of DiskAddr; {qcode}
                          InterpTable: array [0..25] of DiskAddr; {microcode}
                          PartName   : packed array [1..8] of char;
                          PartStart  : DiskAddr;
                          PartEnd    : DiskAddr;
                          SubParts   : array [0..63] of DiskAddr;
                          PartRoot   : DiskAddr;
                          PartKind   : PartitionType;
                          PartDevice : DeviceType
                        );
 {6 is the format of a block of a Directory}
                     6: (
                          Entry      : array [0..FILESPERDIRBLK-1] of DirEntry
                        )
                  end;
                  
    ptrDiskBuffer = ^DiskBuffer;
    
    Header     = packed record           {format of a block header}
                   SerialNum : DiskAddr; {Actually has the SegID of the file}
                   LogBlock  : integer;  {logical block number}
                   Filler    : integer;  {holds a hint to a candidate
                                          for the FreeHead}
                   PrevAdr   : DiskAddr; {Disk Address of the next block in
                                          this segment}
                   NextAdr   : DiskAddr; {Disk Address of the previous block in
                                          this segment}
                 end;

    ptrHeader  = ^Header;
    
    DiskCommand= (DskRead, DskWrite, DskFirstWrite, DskReset, DskHdrRead,
                      DskHdrWrite);  {last ones for error reporting}

var
    DiskSegment : integer;               {a memory segment for DiskIO}
                     
procedure InitDiskIO;  {initialize DiskIO, called at boot time}

procedure ZeroBuffer(ptr : ptrDiskBuffer);  {write zeroes in all words of the
                                            buffer. When reading an unallocated
                                            block, Zeros are returned in the
                                            buffer}

function WhichDisk(addr : DiskAddr) : integer;  {Tells you which disk number a
                                                 DiskAddr is on}

function AddrToField(addr : DiskAddr) : integer;  {gives you a one word short
                                                   address by taking the lower
                                                   byte of the upper word and
                                                   the upper byte of the lower
                                                   word.  The upper byte of the
                                                   upper word can't have any
                                                   significant bits for the 12
                                                   or 24 megabyte disks.  The
                                                   lower byte of the lower word
                                                   is always zero (since a disk
                                                   address is a page address,
                                                   which is 256 words
                                                  }


function FieldToAddr(disk: integer; fld : integer) : DiskAddr;
                                                  { Makes a DiskAddr out of a
                                                    short address and a disk
                                                    number
                                                  }


procedure DiskIO(addr : DiskAddr; ptr : ptrDiskBuffer; 
                 hptr : ptrHeader; dskcommand : DiskCommand); {Do a disk
                                                               operation, if
                                                               errors occur,
                                                               exits via
                                                               DiskError}


function LogAddrToPhysAddr(addr : DiskAddr) : DiskAddr;
                                      {translate a Logical Disk Address (used
                                       throughout the system) to and from a
                                       physical Disk Address (the kind the disk
                                       contoller sees) Logical Disk Addresses
                                       use a sequential numbering system
                                       Physical Disk Addresses have a
                                       Cylinder-Head-Sector system This routine
                                       calls MapAddr (a private routine which
                                       does the translation) Map Addr
                                       implements interlace algorithm}

function PhysAddrToLogAddr(disk : integer; addr : DiskAddr) : DiskAddr;

function LastDiskAddr(DevType : DeviceType) : DiskAddr; {Gets the Disk Address
                                                         of the last possible
                                                         page on the device}


function NumberPages(DevType : DeviceType) : FSBit32;  {Return the number of
                                                        pages on a device}


procedure DiskReset;  {Reset the disk controller and recalibrate the actuater}

function TryDiskIO(addr : DiskAddr;  ptr : ptrDiskBuffer;
                   hptr : ptrHeader;  dskcommand : DiskCommand;
                   numTries: integer) : boolean;
                                      {Try a disk operation, but, return false
                                       if error occurred
                                      }

Exception DiskFailure(msg: String; operation: DiskCommand; addr: DiskAddr;
                       softStat: integer);
Exception DiskError(msg: String);
Exception BadDevice;


Var ErrorCnt : Array[IOEFirstError..IOELastError] of integer;


{******************} private {***************************}
