module InsertDeleteCmds;

{*****************************************}
{
{        Pepper:  Spice Interim Editor
{        Insertion and Deletion Commands
{        Richard Cohn
{        April 1, 1981
{
{*****************************************}


exports

imports RegionEditor from RegionEdit;
imports Perq_String from Perq_String;

procedure ChangeMode;
procedure InsertWhiteSpace (startP: Position; nSpaces: integer);
procedure InsertAtPosition;
procedure TwiddleChars;
procedure MakeOpenSpace;
procedure YankBuffer (B:  BufRange);
procedure DeleteSelection;
procedure DelForChar (count: integer);
procedure DelBackChar (count: integer);
procedure DelForWord (count: integer);
procedure DelBackWord (count: integer);
procedure DelEndLine (count: integer);
function GetBegLine (P:  Position):  Position;
procedure DelBegLine;
procedure DelToWhiteSpace;
procedure ModifyKillRing (backward: boolean);

procedure StrToBuffer (s: PString; b: BufRange);
procedure InsertString (p: Position; s: PString);
procedure DeleteString (first, last: Position);
procedure IncrNumber (incrSize: integer);
procedure FillCurParagraph;
procedure FillSelection;
procedure CenterCurLine;
procedure CenterSelection;
procedure indentBlock;
procedure undentBlock;


private

imports ScreenUtilities from EdScreen;
imports TextUtilities from EdText;
imports MoveCommands from EdMove;
imports ReplayUtilities from EdReplay;
imports Canvas from EdCanvas;
imports Screen from Screen;


{****************************************************************}

procedure ChangeMode;

begin
InsertMode := not InsertMode;
if InsertMode then 
    begin
    SelectKeyTranslation (InsertTable);
    CmdPrompt := InsModePrompt
    end
else
    begin
    SelectKeyTranslation (CmdTable);
    CmdPrompt := CmdModePrompt
    end;
ImmedPrompt := true;
NeedPrompt := true
end; { ChangeMode }


{************************************************************************}

procedure InsertWhiteSpace (startP: Position; nSpaces: integer);

{ Add white space before startP for nSpaces. }

var i:  integer;

begin
with CurWin^ do
    begin
    RightPart := startP;
    LeftPart  := Add (RightPart, -1);
    if SelectWindow = CurWin then
        if LE (RightPart, Bufary [SelectB].Last) then
            if LE (Bufary [SelectB].First, RightPart) then
                UnSelect;
    Split (RightPart);
    Attach(Cursor1,EmptyFirst,WriteCursor);
    Join(LeftPart,EmptyFirst);
    Join(EmptyLast,RightPart);
    if Ln [CurLine].Length = 1  then
        begin
        Ln [CurLine].Start := EmptyFirst;
        if CurLine = 0 then
            ScreenFirst := EmptyFirst
        end;
    for i := 1 to nSpaces do
        begin
        Cursor1.Ch := ' ';
        if EQ (Cursor1.Pos, EmptyLast) then
            begin
            Split (RightPart);
            CreateEmptyPage;
            Join (EmptyLast, RightPart)
            end;
        Add1C (Cursor1)
        end;
    Ln [CurLine].Length := Ln [CurLine].Length + nSpaces;
    EmptyFirst := Cursor1.Pos;
    Detach (Cursor1);
    Tmp := Add (EmptyFirst, -1);
    Split (EmptyFirst);
    Split (RightPart);
    Join (Tmp, RightPart)
    end { with }
end; { InsertWhiteSpace }

 
{************************************************************************}

procedure FillToCursor;

{ Insert spaces from end of line to cursor. }

begin
with curWin^ do
    InsertWhiteSpace (Add (Ln [CurLine].Start, Ln [CurLine].Length - 1),
        curCol - Ln [curLine].length + 1);
end; 

 
{************************************************************************}

procedure ShiftLeft (L:  LineIndex; C:  ColumnIndex);

begin
JoinScreen (RightPart, L, C)
end; 

{************************************************************************}

procedure ShiftRight (L:  LineIndex; C:  ColumnIndex);

var NewL:  LineIndex;

begin
with CurWin^ do
    begin
    EraseChar;
    NewL := L + (C+1) div nColumn;
    if NewL <= LastLine then
        JoinScreen (RightPart, NewL, (C+1) mod nColumn);
    MovePencil (L,C)
    end
end;

 
{************************************************************************}

procedure DoDelete (PF, PL:  Position);

var L: LineIndex;
    C: ColumnIndex;
    N: integer;
    Prepend, Append:  boolean;
    Redraw:  boolean;

begin { DoDelete }
with CurWin^ do begin
if LE (PF, PL) then
if not Eot(PF) then
    begin
    PFirst := PF;
    PLast := PL;
    Prepend := EQ (PLast, PrevKill);
    Append := EQ (PFirst, NextKill);
    LeftPart := Add (PFirst, -1);
    if not Append then
        PrevKill := LeftPart;
    if Eot(PLast) then
        begin
        if not Prepend then
            NextKill  := NilPosition;
        RightPart := PLast;
        Sub1 (PLast)
        end
    else
        begin
        RightPart := Add (PLast,1);
        if not Prepend then
            NextKill := RightPart
        end;
    if OnScreen (PFirst, 0, LastLine) then
        begin
        Redraw := true;
        GetLC (PFirst, L, C)
        end
    else
        begin
        Redraw := OnScreen (RightPart, 0, LastLine);
        if not Redraw then
            if LT (PFirst, ScreenFirst) then
                if LT (ScreenLast, PLast) then
                    Redraw := true;
        if Redraw then
            begin
            L := 0;
            C := 0
            end
        end;
    if SelectWindow = curWin then { was <> nil }
        with Bufary [SelectB] do
            if LT (First, PFirst) then
                begin
                if GE (Last, PFirst) then
                    UnSelect
                end
            else
                if LE (First, PLast) then
                    UnSelect;
    if NE (Mark, NilPosition) then
        if LE (PFirst, Mark) then
            if LE (Mark, PLast) then
                Mark := add(pLast, 1);        
    Split (PFirst);
    Split (RightPart);
    Join(LeftPart,RightPart);
    if Prepend then
        begin
        Join (PLast, Bufary [KillB].First);
        Bufary [KillB].First := PFirst;
        end
    else if Append then
        begin
        Join (Bufary [KillB].Last, PFirst);
        Bufary [KillB].Last := PLast;
        end
    else
        begin
        if KillB = KillBMin then 
            KillB := KillBMax
        else
            KillB := KillB - 1;
        with Bufary [KillB] do
            begin
            Collect (First, Last);
            First := PFirst;
            Last := PLast
            end;
        end;
    DrawLn (KillB);
    if Redraw then
        JoinScreen(RightPart,L,C);
    if NewEvent.Cmd <> DelSelection then
        Show (RightPart, 0, LastLine);
    UpdateThumbBar 
    end
end; { with }
end { DoDelete };

{****************************************************************}

procedure InsertString (p: Position; s: PString);

{ Insert the given string before position p.  The
{ cursor is not moved.  The screen is not updated.  }

begin { InsertString }
StrToBuffer (s, tempB);
RightPart := p;
LeftPart := Add (RightPart, -1);
if SelectWindow = CurWin then
    if LT (Bufary [SelectB].First, RightPart) then
        if GT (Bufary [SelectB].Last, LeftPart) then
            UnSelect;
Split (RightPart);
Join (LeftPart, Bufary[tempB].First);
Join (Bufary[tempB].Last, RightPart);
end; { InsertString }

{************************************************************************}


procedure InsertEndOfLine (p: Position);

{ Insert an end of line before position p.  The
{ cursor is not moved.  The screen is not updated.  }

var
    s:  PString;

begin { InsertEndOfLine }
s [0] := chr (2);  { set string length }
s [1] := CR;
s [2] := LF;
InsertString (p, s);
end; { InsertEndOfLine }

{****************************************************************}

procedure DeleteString (first, last: Position);

{ Delete the text in the current window pointed to by first and last.
{ The text is unselected and the mark is moved if necessary.  However,
{ the text is not saved in the kill buffer, and the screen is not
{ refreshed.  DoDelete does these functions.  }

begin  { DeleteString }
if GT (first, last) then
    exit (DeleteString);
pFirst := first;
pLast := last;
leftPart := Add (pFirst, -1);
rightPart := Add (pLast, 1);

if SelectWindow = curWin then
    if LT (Bufary [SelectB].First, PFirst) then
        begin
        if GE (Bufary [SelectB].Last, PFirst) then
            UnSelect
        end
    else
        if LE (Bufary [SelectB].First, PLast) then
            UnSelect;
if NE (curWin^.Mark, NilPosition) then
    if LE (PFirst, curWin^.Mark) then
        if LE (curWin^.Mark, PLast) then
            curWin^.Mark := add(pLast, 1);

Split (PFirst);
Split (RightPart);
Collect (PFirst, PLast);
Join (LeftPart, RightPart);
end;  { DeleteString }


{****************************************************************}

procedure InsertAtPosition;

{ Start inserting characters at the current position.  }

   var L: LineIndex;
       C: ColumnIndex;
       Blanks: integer;
       CountingBlanks: boolean;
       MouseMoved:  boolean;
       LenRemain:  integer;
       LRemain:  LineIndex;
       CRemain:  ColumnIndex;
       Indent:  boolean;
       SFeqRP:  boolean;
       i:  integer;

    procedure EchoChar;
    var LeadingBlanks: integer;
        Done: Boolean;
        Spaces, i:  integer;
        QuotedCh: Char;
        
        
     procedure CountLeadingBlanks;
     begin { CountLeadingBlanks }
      with CurWin^ do begin
      Attach(Cursor2,Ln[L].Start,ReadCursor);
      LeadingBlanks := 0;
      while Cursor2.Ch = ' ' do
       begin Add1C(Cursor2); LeadingBlanks := LeadingBlanks + 1 end;
      Detach(Cursor2)
      end; { with }
     end { CountLeadingBlanks };
        
        
     procedure NextLine;
     var OldLen, LRemain, CRemain:  integer;
         Width:  integer;
         ExtraChars:  integer;
         OldX, OldY, NewX, NewY:  integer;
         JoinLines:  boolean;
     begin { NextLine }
      with CurWin^ do begin
      PLast := Ln [L].Finish;
      Ln[L].Finish := Add(Cursor1.Pos,-1);
      OldLen := Ln[L].Length;
      Ln[L].Length := C;
      PFirst := RightPart;
      CRemain := C - 1;
      if L > LastLine div 2 then
          begin
          ScrollUp(0,L,1);
          if L = LastLine then
              ClearLine (LastLine,0,0)
          end
      else 
          begin
          L := L + 1; 
          ScrollDown(L,LastLine,1)
          end;
      LRemain := L - 1; 
      C := 0;
      Ln[L].Start := PFirst;
      if Indent then
          begin
          C := LeadingBlanks;
          if LeadingBlanks > 0 then
              Ln[L].Start := Cursor1.Pos;
          while LeadingBlanks > 0 do
              begin Cursor1.Ch := ' ';
              if EQ(Cursor1.Pos,EmptyLast) then
                  begin Split(RightPart);
                  CreateEmptyPage;
                  Join(EmptyLast,RightPart)
                  end;
              Add1C(Cursor1);
              LeadingBlanks := LeadingBlanks - 1
              end;
          MovePencil(L,C)
          end; { Indent }
      ExtraChars := OldLen - Ln[LRemain].Length;
      if (ExtraChars > 0) and (Ln[LRemain].Length <> nColumn) then
          begin
          Width := CharWidth * ExtraChars + 1;
          OldX := CRemain * CharWidth + xHome - 1;
          OldY := LRemain * LineHeight + yHome - 1;
          NewX := C * CharWidth + xHome - 1;
          NewY := L * LineHeight + yHome - 1;
          RasterOp (RRpl, Width, LineHeight, NewX, NewY, SScreenW, SScreenP,
              OldX, OldY, SScreenW, SScreenP);
          RasterOp (RXor, Width, LineHeight, OldX, OldY, SScreenW, SScreenP,
              OldX, OldY, SScreenW, SScreenP);
          end;
      Ln[L].Finish := PLast;
      Ln[L].Length := C + ExtraChars;
      if Ln[L].Length < nColumn then
          begin
          Attach (Cursor2, Ln[L].Finish, ReadCursor);
          if Cursor2.Ch = LF then
              begin
              Sub1C (Cursor2);
              JoinLines := Cursor2.Ch <> CR
              end
          else
              JoinLines := not Eot (Cursor2.Pos);
          if JoinLines then
              begin
              ReAttach (Cursor2, Ln[L].Finish);
              Tmp := Add (Cursor2.Pos, 1);
              Detach (Cursor2);
              JoinScreen (Tmp, L, Ln[L].Length)
              end
          else
              Detach (Cursor2)
          end
      end; {with }
     end { NextLine };
     
    procedure wrapWord;
    label
      1;
    var
      pf, pl, dStart: position;
      foundSpace, spanning: boolean;
      lineLen, charNum, spacePos: integer;
      dRow: lineIndex;
      dCol: columnIndex;
    begin { wrapWord }
      with curWin^ do
        begin
          savedPos := cursor1.pos;
          sub1C(cursor1);
          lineLen := 0;
          while cursor1.ch <> lf do
            begin
              lineLen := lineLen + 1;
              sub1C(cursor1);
            end;
          if lineLen < curWin^.rightMargin then
            begin
              reAttach(cursor1, savedPos);
              exit(wrapWord);
            end;
          { heal the split at the insert position }
          EmptyFirst := savedPos;
          Tmp := Add (EmptyFirst, -1);
          Split(EmptyFirst);
          Split(RightPart);
          Join (Tmp, RightPart);
          savedPos := rightPart;
          dStart := nilPosition;
          repeat
            { look for the last space string on the first line }
            foundSpace := false;
            spanning := false;
            charNum := 0;
            while ((charNum < curWin^.rightMargin) or
                   spanning or
                   (not foundSpace)) and
                  (charNum < lineLen) do
              begin
                add1C(cursor1);
                charNum := charNum + 1;
                if cursor1.ch = ' ' then
                  if spanning then
                    begin
                      pl := cursor1.pos;
                      spacePos := charNum;
                    end
                  else
                    begin
                      pf := cursor1.pos;
                      pl := cursor1.pos;
                      spanning := true;
                      spacePos := charNum;
                      foundSpace := true;
                    end
                else
                  spanning := false;
              end;
            if not foundSpace then
              goto 1; { abort }
            lineLen := lineLen - spacePos;
            reAttach(cursor1, add(pl, 1));
            if dStart = nilPosition then
              begin
                dStart := add(pf, -1);
                getLC(pf, dRow, dCol);
              end;
            { delete the space string }
            deleteString(pf, pl);
            { insert the EOL }
            justP := cursor1.pos;  { save cursor1 so strtobuffer can use it }
            detach(cursor1);
            insertEndOfLine(justP);
            attach(cursor1, justP, writeCursor);
            justP := nilPosition;
          until lineLen < curWin^.rightMargin;
       1:
          if dStart <> nilPosition then
            begin
              add1(dStart);
              draw(dStart, filledLast, dRow, dCol);
              if not onScreen(savedPos, 0, lastLine) then
                show(savedPos, 0, lastLine);
            end;
          getLC(savedPos, L, C);
          { split at the insert position }
          RightPart := savedPos;
          LeftPart := Add(RightPart, -1);
          Split (RightPart);
          reAttach(cursor1, emptyFirst);
          Join(LeftPart,EmptyFirst);
          Join(EmptyLast,RightPart);
        end;
    end { wrapWord };

       
  begin { EchoChar }
  with CurWin^ do begin
      if C = 0 then
          Ln[L].Start := Cursor1.Pos;

  Indent := (NewEvent.Cmd = IndentNewLine) or (NewEvent.Cmd = LFCmd);
  if (NewEvent.Cmd = MakeNewLine) or (NewEvent.Cmd = CRCmd) or Indent then
      begin
      if Replay <> NotReplaying then
          begin
          if not (land (#177, ord (NewEvent.Ch)) in
             [OpenSpace, OpenIndent]) then
              CheckReplay (ReplaySingleStep);
          if Replay = NotReplaying then
              begin
              NewEvent.Cmd := NullCmd;
              exit (EchoChar)
              end
          end
      else
          FlushTranscript;
      Cursor1.Ch := CR;
      if EQ (Cursor1.Pos, EmptyLast) then
          begin Split(RightPart);
          CreateEmptyPage;
          Join(EmptyLast,RightPart)
          end;
      Add1C (Cursor1);
      Cursor1.Ch := LF;
      if EQ(Cursor1.Pos,EmptyLast) then
          begin Split(RightPart);
          CreateEmptyPage;
          Join(EmptyLast,RightPart)
          end;
      Add1C(Cursor1);
      if Indent then
          CountLeadingBlanks;
      Ln [L].Length := Ln [L].Length + 1;
      C := C + 1;
      NextLine
      end
  else {text}
      begin
      if NewEvent.Cmd in [InsTab, TabCmd] then
          begin
          Spaces := TabSpaces - C mod TabSpaces;
          if C + Spaces > nColumn then
              Spaces := nColumn - C;
          for i := 1 to Spaces do
              begin
              Cursor1.Ch := ' ';
              Ln[L].Length := Ln[L].Length + 1;
              if EQ (Cursor1.Pos, EmptyLast) then
                  begin
                  Split (RightPart);
                  CreateEmptyPage;
                  Join (EmptyLast, RightPart)
                  end;
              Add1C (Cursor1);
              ShiftRight (L,C);
              WriteChar (' ');
              C := C + 1
              end
          end
      else 
          begin
          if NewEvent.Cmd = InsSpace then
              Cursor1.Ch := ' '
          else if NewEvent.Cmd = QuoteChar then
              begin
              Prompt ('Quote next character:');
              NeedPrompt := true;
              ImmedPrompt := true;
              if Replay = NotReplaying then
                  begin
                  repeat LookForCommand (CanvDflt)
                  until NewEvent.Cmd <> NullCmd;
                  QuotedCh := NewEvent.Ch;
                  SendTrnByte (QuotedCh)
                  end
              else QuotedCh := GetTrnByte;
              if LAND (ord (QuotedCh), #200) <> 0 then
                  { convert to Ascii control character }
                  Cursor1.Ch := chr (LAND (ord (QuotedCh), #037))
              else
                  Cursor1.Ch := QuotedCh
              end;
          Ln[L].Length := Ln[L].Length + 1;
          if EQ(Cursor1.Pos,EmptyLast) then
              begin 
              Split(RightPart);
              CreateEmptyPage;
              Join(EmptyLast,RightPart)
              end;
          ShiftRight (L,C);
          WriteChar(Cursor1.Ch);
          Add1C(Cursor1);
          C := C + 1
          end;
      if wordWrap then
        wrapWord;
      if C = NColumn then
          NextLine
      end;

  MovePencil(L,C);
  MoveTextPointer (L,C)
  end; { with }
  end { EchoChar };
    
    
{******************************}

begin { InsertAtPosition }
with CurWin^ do begin
RightPart := GetPosition (CurLine, CurCol);
if RightPart = FilledLast then
    begin
    GetLC (RightPart, L, C);
    MoveTextPointer (L,C)
    end
else if CurCol > Ln [CurLine].Length - 1 then
    FillToCursor;
TBarOn := false;
RightPart := GetPosition (CurLine, CurCol);
LeftPart := Add (RightPart, -1);
GetLC (RightPart, L, C);
MoveTextPointer (L,C);
MovePencil(L,C);
if SelectWindow = CurWin then
    if LE (RightPart, Bufary [SelectB].Last) then
        if LT (Bufary [SelectB].First, RightPart) then
            UnSelect;
Split (RightPart);
PFirst := EmptyFirst;
Attach(Cursor1,EmptyFirst,WriteCursor);
Join(LeftPart,EmptyFirst);
Join(EmptyLast,RightPart);
if C = 0 then
    begin
    Ln[L].Start := PFirst;
    if L = 0 then 
        ScreenFirst := PFirst
    end;
MouseMoved := false;
repeat
    if ImmedPrompt or NeedPrompt then
        Prompt (CmdPrompt);
    Cursor1.Ch := NewEvent.Ch;
    EchoChar;
    if (SaveAfter > 0) and (nChanges > SaveAfter) then
        begin
        NewEvent.Cmd := BackFile;
        nChanges := 0
        end
    else
        if RepeatCount > 1 then
            RepeatCount := RepeatCount - 1
        else
            begin
            repeat
                LookForCommand (CanvDflt);
                DisplPointer (xPointer, yPointer);
                FilterMouseCommand
            until NewEvent.Cmd <> NullCmd;
            MouseMoved := (CurLine <> L) or (CurCol <> C);
            if (Replay = NotReplaying) and (NewEvent.Cmd in InsCmds)
                and not MouseMoved then
                SendTrnByte (NewEvent.Ch)
            end
until not (NewEvent.Cmd in InsCmds) or MouseMoved;
NewEvent.x := 0;
NewEvent.y := 0;
if Replay = NotReplaying then
    begin
    ReturnKey (inputQueue, newEvent);
    NewEvent.Cmd := CloseInsert;
    SendKey (NewEvent)
    end;
EmptyFirst := Cursor1.Pos;
Detach(Cursor1);
Tmp := Add (EmptyFirst, -1);
Split(EmptyFirst);
Split(RightPart);
Join (Tmp, RightPart);
TBarOn := true;
UpdateThumbBar
end; { with }
end { Insert };
 

{************************************************************************}

procedure TwiddleChars;

var PFIsEoln, PLIsEoln:  boolean;
    L:  LineIndex;
    C:  ColumnIndex;


{********************************}

procedure CopyChar;

{ Make a real copy of the character in Cursor1 at Cursor2 }

begin
Cursor2.Ch := Cursor1.Ch;
if EQ (Cursor1.Pos, EmptyLast) then
    begin
    Split (RightPart);
    CreateEmptyPage;
    Join (EmptyLast, RightPart)
    end;
Add1C (Cursor2)
end;


{********************************}

begin
with CurWin^ do
if CurCol <= Ln [CurLine].Length then
    begin
    RightPart := GetPosition (CurLine, CurCol);
    if not Bot (RightPart) then
        begin
        PFirst := Add (RightPart, -1);
        if not Bot (PFirst) then
            begin
            Attach (Cursor1, PFirst, ReadCursor);
            PFIsEoln := false;
            if Cursor1.Ch = LF then
                begin
                Sub1C (Cursor1);
                if Cursor1.Ch <> CR then
                    Add1C (Cursor1)
                else
                    PFIsEoln := true
                end;
            PFirst := Cursor1.Pos;
            Sub1C (Cursor1);
            PLIsEoln := false;
            if not Bot (Cursor1.Pos) and (Cursor1.Ch = LF) then
                begin
                Sub1C (Cursor1);
                if Cursor1.Ch <> CR then
                    Add1C (Cursor1)
                else
                    PLIsEoln := true
                end;
            PLast := Cursor1.Pos;
            GetLC (PLast, L,C);
            Sub1C (Cursor1);
            LeftPart := Cursor1.Pos;
            if GE (Bufary [SelectB].Last, PLast) then
                if LE (Bufary [SelectB].First, PFirst) then
                    UnSelect;
            Split (PLast);
            Split (RightPart);
            Join (LeftPart, EmptyFirst);
            Join (EmptyLast, RightPart);
            Attach (Cursor2, EmptyFirst, WriteCursor);
            ReAttach (Cursor1, PFirst);
            CopyChar;
            if PFIsEoln then
                begin
                Add1C (Cursor1);
                CopyChar
                end;
            ReAttach (Cursor1, PLast);
            PLast := Cursor2.Pos;
            CopyChar;
            if PLIsEoln then
                begin
                Add1C (Cursor1);
                CopyChar
                end;
            PFirst := EmptyFirst;
            EmptyFirst := Cursor2.Pos;
            Detach (Cursor1);
            Detach (Cursor2);
            Tmp := Add (EmptyFirst, -1);
            Split (EmptyFirst);
            Split (RightPart);
            Join (Tmp, RightPart);
            if L >= 0 then
                JoinScreen (PFirst, L, C)
            else
                if PLIsEoln then
                    JoinScreen (RightPart, 0,0)
                else
                    Draw (PLast, FilledLast, 0,0);
            GetLC (RightPart, L, C);
            MoveTextPointer (L,C)
            end { if not Bot (PFirst) }
        end { if not Bot (RightPart) }
    end { with CurWin^ }
end; { TwiddleChars }
                

{************************************************************************}

procedure MakeOpenSpace;

var L:  LineIndex;
    C:  ColumnIndex;

begin
with CurWin^ do begin
if CurCol > Ln [CurLine].Length - 1 then
    FillToCursor;
L := CurWin^.CurLine;
C := CurWin^.CurCol;
if Replay = NotReplaying then
    ReturnKey (inputQueue, newEvent);  { to insure return from Insert }
if NewEvent.Cmd = OpenIndent then
    NewEvent.Cmd := IndentNewLine
else
    NewEvent.Cmd := MakeNewLine;
InsertAtPosition;
if Replay = NotReplaying then
    NewEvent := GetKey (inputQueue);  { to remove fake event }
if L > LastLine div 2 then
    L := L - 1;
MoveTextPointer (L, C)
end { with }
end; { MakeOpenSpace }


{************************************************************************}

procedure YankBuffer (B:  BufRange);

{ Insert contents of buffer B into the current window before the current
{ position.  }

var L:  LineIndex;
    C:  ColumnIndex;
    MoveSelect:  boolean;

begin { YankBuffer }
if EQ (Bufary [B].First, NilPosition) or
    ((B = SelectB) and (SelectWindow = nil)) then
    Warn ('Can''t insert:  No text in buffer')
else
    with CurWin^ do
        begin
        if CurCol > Ln [CurLine].Length - 1 then
            FillToCursor;
        RightPart := GetPosition (CurLine, CurCol);
        LeftPart := Add (RightPart, -1);
        GetLC (RightPart, L, C);
        MoveTextPointer (L,C);
        MoveSelect := EQ (Bufary [SelectB].First, RightPart);
        if SelectWindow = CurWin then
            if LT (Bufary [SelectB].First, RightPart) then
                if GT (Bufary [SelectB].Last, LeftPart) then
                    UnSelect;
        Copy (Bufary[B].First, Bufary[B].Last);
        Split (RightPart);
        Join (LeftPart, PFirst);
        Join (PLast, RightPart);
        if MoveSelect then
            Bufary [SelectB].First := RightPart;
        JoinScreen (pFirst, l, c);
        UpdateThumbBar;
        ImmedPrompt := true;
        NeedPrompt := true
        end { with }
end { YankBuffer };


{************************************************************************}


procedure DelForChar (count: integer);

var P, Q:  Position;
    OldPosCR:  boolean;

begin
if DEBUG [2] then Status ('Enter DelForChar');
if CurWin^.CurCol > CurWin^.Ln [CurWin^.CurLine].Length - 1 then
    FillToCursor;
P := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
Attach (Cursor1, P, ReadCursor);
while (count > 1) and not Eot (Cursor1.Pos) do
    begin
    count := count - 1;
    OldPosCR := Cursor1.Ch = CR;
    Add1C (Cursor1);
    if OldPosCR and (Cursor1.Ch = LF) then
        Add1C (Cursor1);
    end;
Q := Cursor1.Pos;
Detach (Cursor1);
CheckCRLF (P,Q);
DoDelete (P,Q);
if DEBUG [2] then Status ('Exit DelForChar')
end; { DelForChar }


{************************************************************************}

procedure DelBackChar (count: integer);

var P, Q:  Position;
    OldPosCR:  boolean;
    First:  boolean;

begin
if DEBUG [2] then Status ('Enter DelBackChar');
if CurWin^.CurCol > CurWin^.Ln [CurWin^.CurLine].Length - 1 then
    FillToCursor;
P := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
if not Bot (P) then
    begin
    First := true;
    Attach (Cursor1, P, ReadCursor);
    while (count > 0) and not Bot (Cursor1.Pos) do
        begin
        count := count - 1;
        Sub1C (Cursor1);
        if First then
            begin
            First := false;
            Q := Cursor1.Pos
            end;
        if Cursor1.Ch = LF then
            begin
            Sub1C (Cursor1);
            if Cursor1.Ch <> CR then
                Add1C (Cursor1)
            end;
        end;
    P := Cursor1.Pos;
    Detach (Cursor1);
    CheckCRLF (P,Q);
    DoDelete (P,Q)
    end;
if DEBUG [2] then Status ('Exit DelBackChar')
end; { DelBackChar }


{************************************************************************}

procedure DelForWord (count: integer);

var P, Q:  Position;
    
begin
if CurWin^.CurCol > CurWin^.Ln [CurWin^.CurLine].Length - 1 then
    FillToCursor;
P := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
Q := GetForWords (P, count);
if not Bot (Q) then
    begin
    Attach (Cursor1, Q, ReadCursor);
    if Cursor1.Ch = LF then
        begin
        Sub1C (Cursor1);
        if Cursor1.Ch <> CR then
            Add1C (Cursor1)
        end;
    Sub1C (Cursor1);
    Q := Cursor1.Pos;
    Detach (Cursor1)
    end;
DoDelete (P,Q)
end; { DelForWord }


{************************************************************************}

procedure DelBackWord (count: integer);

var P, Q, R:  Position;
    
begin
Q := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
if not Bot (Q) then
    begin
    P := GetBackWords (Q, count);
    Attach (Cursor1, Q, ReadCursor);
    if Cursor1.Ch = LF then
        begin
        Sub1C (Cursor1);
        if Cursor1.Ch <> CR then
            Add1C (Cursor1)
        end;
    Sub1C (Cursor1);
    Q := Cursor1.Pos;
    Detach (Cursor1);
    DoDelete (P,Q)
    end;
end; { DelBackWord } 


{************************************************************************}

procedure DelEndLine (count: integer);

var P, Q:  Position;
    Done:  boolean;

begin
with CurWin^ do
    begin
    if CurCol > Ln [CurLine].Length - 1 then
        FillToCursor;
    P := GetPosition (CurLine, CurCol)
    end;
if not Eot (P) then
    begin
    Attach (Cursor1, P, ReadCursor);
    Q := P;
    while (count > 0) and not Eot (Cursor1.Pos) do
        begin
        count := count - 1;
        Done := false;
        if Cursor1.Ch = CR then
            begin
            Add1C (Cursor1);
            Done := Cursor1.Ch = LF;
            if Done then
                Add1C (Cursor1)
            end;
        while not (Done or Eot (Cursor1.Pos)) do
            if Cursor1.Ch = CR then
                begin
                Add1C (Cursor1);
                Done := Cursor1.Ch = LF;
                if Done then
                    Sub1C (Cursor1)
                end
            else
                Add1C (Cursor1)
        end; { while (count... }
    Sub1C (Cursor1);
    Q := Cursor1.Pos;
    Detach (Cursor1);
    DoDelete (P,Q);
    end { if not Eot (P) }
end; { DelEndLine }


{************************************************************************}

function GetBegLine (P :  Position):  Position;

var Done:  boolean;

begin
if not Bot (P) then
    begin
    Attach (Cursor1, P, ReadCursor);
    if Cursor1.Ch = LF then
        begin
        Sub1C (Cursor1);
        if Cursor1.Ch = CR then
            Sub1C (Cursor1)
        end
    else
        Sub1C (Cursor1);
    Done := false;
    repeat
        if Cursor1.Ch = LF then
            begin
            Sub1C (Cursor1);
            Done := (Cursor1.Ch = CR);
            if Done then
                AddC (Cursor1, 2)
            end
        else
            Sub1C (Cursor1)
    until Done or Bot (Cursor1.Pos);
    GetBegLine := Cursor1.Pos;
    Detach (Cursor1);
    end
else
    GetBegLine := P;
end; { GetBegLine }



{************************************************************************}

procedure DelBegLine;

var P, Q:  Position;

begin
Q := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
if not Bot (Q) then
    begin
    P := GetBegLine (Q);
    Attach (Cursor1, Q, ReadCursor);
    if Cursor1.Ch = LF then
        begin
        Sub1C (Cursor1);
        if Cursor1.Ch = CR then
            Sub1C (Cursor1)
        end
    else
        Sub1C (Cursor1);
    Q := Cursor1.Pos;
    Detach (Cursor1);
    DoDelete (P,Q)
    end;
if GT (P,Q) then { no deletion done }
    MoveTextPointer (CurWin^.CurLine, 0)
end; { DelBegLine }


{************************************************************************}

procedure DelToWhiteSpace;

var P, Q:  Position;

begin

Q := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
if not Bot (Q) then
    begin
    P := GetBegLine (Q);
    Attach (Cursor1, P, ReadCursor);
    while Cursor1.Ch = ' ' do
        Add1C (Cursor1);
    P := Cursor1.Pos;
    ReAttach (Cursor1, Q);
    if Cursor1.Ch = LF then
        begin
        Sub1C (Cursor1);
        if Cursor1.Ch = CR then
            Sub1C (Cursor1)
        end
    else
        Sub1C (Cursor1);
    Q := Cursor1.Pos;
    Detach (Cursor1);
    DoDelete (P,Q)
    end;
end; { DelBegLine }


{************************************************************************}

procedure DeleteSelection;

var OldWindow:  pTextWindow;
    L:  LineIndex;
    C:  ColumnIndex;

begin
if SelectWindow = nil then
    Warn ('No selection:  can''t delete')
else
    begin
    Tmp := GetPosition (CurWin^.CurLine, CurWin^.CurCol);
    with Bufary [SelectB] do
        begin
        if SelectWindow = CurWin then
            if LE (First, Tmp) then
                if LE (Tmp, Last) then
                    Tmp := Add (Last, 1);
        OldWindow := CurWin;
        ChangeTextWindow (SelectWindow);
        DoDelete (First, Last)
        end;
    ChangeTextWindow (OldWindow);
    GetLC (Tmp, L, C);
    MoveTextPointer (L, C)
    end
end; { DeleteSelection }


{****************************************************************}

procedure ModifyKillRing (backward: boolean);

{ Either move backward or forward one step in the kill ring.  Backward
{ is defined to be to put the current top at the bottom and put the next
{ most recent kill at the top.  }

begin  { ModifyKillRing }
if backward then
    if KillB < KillBMax then
        KillB := KillB + 1
    else
        KillB := KillBMin
else
    if KillB > KillBMin then
        KillB := KillB - 1
    else
        KillB := KillBMax;
DrawLn (KillB);
PrevKill := NilPosition;
NextKill := NilPosition;
end;  { ModifyKillRing }


{****************************************************************}

procedure StrToBuffer (s: PString; b: BufRange);

{ Convert the string to a buffer.  }

var
    i:  integer;

begin { StrToBuffer }
Attach (cursor1, emptyFirst, writeCursor);
for i := 1 to Length (s) do
    begin
    Cursor1.Ch := s [i];
    if EQ (Cursor1.Pos, EmptyLast) then
        begin
        Split (RightPart);
        CreateEmptyPage;
        Join (EmptyLast, RightPart)
        end;
    Add1C (Cursor1)
    end;
Bufary [b].First := EmptyFirst;
EmptyFirst := Cursor1.Pos;
Detach (Cursor1);        
Bufary [b].Last := Add (EmptyFirst, -1);
Split (EmptyFirst);
end; { StrToBuffer }


{****************************************************************}

procedure InsertSpaces (p: Position; nSpaces: integer);

{ Insert nSpaces spaces before position p.  The
{ cursor is not moved.  The screen is not updated.  nSpaces MUST be
{ less than 255.  }

var
    s:  PString;

begin { InsertSpaces }
{ s set initially to 254 spaces }
if nSpaces <= 0 then
    exit (InsertSpaces);
s := '                                                                                                                                                                                                                                                              ';
s [0] := chr (nSpaces);  { set string length }
InsertString (p, s);
end; { InsertSpaces }


{****************************************************************}

procedure InsertMargin (first, nonWhite: Position; nSpaces: integer);

{ Insert nSpaces spaces at the left margin of the line containing
{ first and nonWhite.  first is the very first character of the line,
{ while nonWhite is the first non-white character on the line.  If
{ too many spaces exist then some are deleted, otherwise some are added. }

var
    s:  PString;
    oldSpaces:  integer;

begin { InsertMargin }
oldSpaces := Subtract (nonWhite, first);
if oldSpaces < nSpaces then
    InsertSpaces (first, nSpaces - oldSpaces)
else if oldSpaces > nSpaces then
    begin
    nonWhite := Add (first, oldSpaces - nSpaces - 1);
    DeleteString (first, nonWhite);
    end;
end; { InsertMargin }


{****************************************************************}

procedure IncrNumber (incrSize: integer);

{ See if the string to the right of the cursor is a number.  If so, increment
{ it by incrSize.  Else, do nothing.  incrSize can be negative, but the
{ number to be incremented cannot.  }

var
    num:  integer;  { number to right of cursor }

begin
with curWin^ do
    begin
    pFirst := GetPosition (curLine, curCol);
    tmp := Add (pFirst, -1);
    Attach (cursor1, pFirst, readCursor);
    pLast := pFirst;
    num := 0;
    while cursor1.ch in numbers do
        begin
        if num >= maxint div 10 then
            { don't want to overflow! }
            begin
            Warn ('Number too large to increment.');
            exit (IncrNumber);
            end;
        num := num * 10 + Ord (cursor1.ch) - ord ('0');
        pLast := cursor1.pos;
        Add1C (cursor1);
        end;  { while }
    rightPart := cursor1.pos;  
    Detach (cursor1);

    if EQ (rightPart, pFirst) then
        begin
        Warn ('Not a number.');
        exit (IncrNumber);
        end;

    if num < 0 then
        begin
        Warn ('Can''t decrement 0');
        exit (IncrNumber);
        end;

    { now delete old number and add new one }
    InsertString (pFirst, IntToStr (num + incrSize));
    DeleteString (PFirst, PLast);
    Add1 (tmp);
    JoinScreen (tmp, curLine, curCol);
    end;  { with }
end; { IncrNumber }


{****************************************************************}

function GetNonWhiteSpace (first: Position):  Position;

{ Starting at first, look for the next character that is not a space or a
{ CR-LF pair.  Single CRs or LFs are considered to be non-white space.
{ Returns first non-space.  Points to CR if last space is a CR-LF pair. }

label 1;

begin  { GetNonWhiteSpace }
Attach (cursor1, first, readCursor);
if cursor1.ch = LF then
    begin
    Sub1C (cursor1);
    if cursor1.ch = CR then
        Add1C (cursor1);
    Add1C (cursor1);
    end;  { if }
while (cursor1.ch = ' ') or (cursor1.ch = CR) do
    begin
    if cursor1.ch = CR then
        begin
        Add1C (cursor1);
        if cursor1.ch <> LF then
            begin
            Sub1C (cursor1);
            goto 1;
            end;
        end;
    Add1C (cursor1);
    end;  { while }
1:
GetNonWhiteSpace := cursor1.pos;
Detach (cursor1);
if DEBUG [5] then
    StatusCh ('GetNonWhiteSpace:  ch = ', cursor1.ch);
end;  { GetNonWhiteSpace }


{****************************************************************}

procedure GetWhiteSpace (first: Position; var whiteP: Position;
    var breakCh: char; var isSpace: boolean; var nScanned: integer);

{ Starting at first, look for the next character that is a space or a
{ CR-LF pair.  Single CRs or LFs are considered to be non-white space.
{ Returns first non-space.  Returns last non-white character in breakCh.
{ nScanned is the number of characters it looked at.  (CR-LF's count as
{ one.)  isSpace is true if whiteP points to a space.  }

label 1;

begin  { GetWhiteSpace }
nScanned := 0;
breakCh := ' ';
Attach (cursor1, first, readCursor);
while not Eot (cursor1.pos) and (cursor1.ch <> ' ') do
    begin
    if cursor1.ch = CR then
        begin
        Add1C (cursor1);
        if cursor1.ch = LF then
            begin
            Sub1C (cursor1);
            goto 1;
            end;
        end;
    breakCh := cursor1.ch;
    Add1C (cursor1);
    nScanned := nScanned + 1;
    end;  { while }
1:
whiteP := cursor1.pos;
isSpace := cursor1.ch = ' ';
Detach (cursor1);
if DEBUG [5] then
    StatusCh (Concat (Concat ('GetWhiteSpace:  isSp, nSc, bCh = ',
    IntToStr (ord (isSpace))), IntToStr (nScanned)), breakCh);
end;  { GetWhiteSpace }              


{****************************************************************}

function GetBegParagraph (p: Position):  Position;

{ Return the position pointing to the beginning of the paragraph
{ containing p.  The paragraph begins at the beginning of the file
{ or after two blank lines.  A blank line is a CR-LF pair alone or
{ preceded by spaces.  }

var
    first:  Position;  { beginning of paragraph }

label 1;

begin  { GetBegParagraph }
Attach (cursor1, p, readCursor);
first := cursor1.pos;
while not Bot (first) do
    begin
    Sub1C (cursor1);
    if cursor1.ch = LF then
        begin
        Sub1C (cursor1);
        if cursor1.ch = CR then
            begin
            repeat
                Sub1C (cursor1);
            until Bot (cursor1.pos) or (cursor1.ch <> ' ');
            if cursor1.ch = LF then
                begin
                Sub1C (cursor1);
                if cursor1.ch = CR then
                    goto 1;
                end;  { if }
            end;  { if }
        end;  { if }
    first := cursor1.pos;
    end;  { while }
1:
Detach (cursor1);
GetBegParagraph := first;
end;  { GetBegParagraph }


{****************************************************************}

function GetEndParagraph (p: Position):  Position;

{ Return the position pointing to the end of the paragraph
{ containing p.  The paragraph ends at the position before the end of
{ the file or before two blank lines.  A blank line is a CR-LF pair alone or
{ preceded by spaces.  }

var
    last:  Position;  { end of paragraph }

label 1;

begin  { GetEndParagraph }
Attach (cursor1, p, readCursor);
if Bot (cursor1.pos) or Eot (cursor1.pos) then
    last := p
else
    last := Add (p, -1);  { want character before CR-LF }
while not Eot (cursor1.pos) do
    begin
    if cursor1.ch = CR then
        begin
        Add1C (cursor1);
        if cursor1.ch = LF then
            { found EOL--is next line a blank line? }
            begin
            repeat
                Add1C (cursor1);
            until cursor1.ch <> ' ';
            if cursor1.ch = CR then
                begin
                Add1C (cursor1);
                if cursor1.ch = LF then
                    goto 1;
                end;  { if }
            end;  { if }
        end;  { if }
    last := cursor1.pos;
    if not Eot (cursor1.pos) then
        Add1C (cursor1);
    end;  { while }

{ this executes only if EOT reached }
ReAttach (cursor1, Add (last, -1));
if cursor1.ch = LF then
    begin
    Sub1C (cursor1);
    if cursor1.ch = CR then
        last := Add (cursor1.pos, -1);
    end;  { if }
1:
Detach (cursor1);
GetEndParagraph := last;
end;  { GetEndParagraph }


{****************************************************************}

procedure FillParagraph (first, last: Position);

{ Fill the paragraph from first to last.  Assumes that first is the first
{ character of the line.  Strips out all blank space and replaces each
{ with a single space except following periods and colons.  The text is
{ placed between the left and right margins.  Assumes last is not a space
{ or CR-LF pair but inserts one after last.  }

var
    curC:  integer;       { current column number }
    wordLength:  integer; { size of last word scanned }
    beforeP:   Position;  { last position before first }
    lineEndP:  Position;  { space before last character on line }
    nSpaces:  integer;    { number of spaces to add after word }
    breakCh:  char;       { last character of word }
    isSpace:  boolean;    { true if white space after word is a space }

begin  { FillParagraph }
sourceFirst := first;
sourceLast := last;
if SelectWindow = curWin then
    if LT (Bufary [SelectB].First, sourceFirst) then
        begin
        if GE (Bufary [SelectB].Last, sourceFirst) then
            UnSelect
        end
    else
        if LE (Bufary [SelectB].First, sourceLast) then
            UnSelect;

with curWin^ do
    begin
    repeat
        lineEndP := sourceFirst;

        { insert margin }
        tmp := GetNonWhiteSpace (sourceFirst);
        if GT (tmp, sourceLast) then
            tmp := sourceLast;
        InsertMargin (sourceFirst, tmp, leftMargin);
        sourceFirst := tmp;
        curC := leftMargin;

        while (curC < rightMargin) and LT (sourceFirst, sourceLast) do
            begin

            { search to word break; leave tmp at space }
            GetWhiteSpace (sourceFirst, tmp, breakCh, isSpace, wordLength);
            if (curC = leftMargin) and (curC + wordLength >= rightMargin) then
                lineEndP := tmp;
            curC := curC + wordLength;
            
            if LE (tmp, sourceLast) and
                ((curC < rightMargin) or (curC - wordLength = leftMargin)) then
                { second case if word is bigger than space between margins }
                { insert 1 or 2 spaces and strip white space }
                begin
                sourceFirst := tmp;  { after last non-space on line so far }
                lineEndP := Add (sourceFirst, -1);
                if (breakCh = '.') or (breakCh = ':') then
                    nSpaces := 2
                else
                    nSpaces := 1;
                if isSpace then
                    begin
                    if nSpaces = 2 then
                        InsertString (sourceFirst, ' ');
                    Add1 (sourceFirst);
                    end
                else
                    if nSpaces = 1 then
                        InsertString (sourceFirst, ' ')
                    else
                        InsertString (sourceFirst, '  ');
                Add1 (lineEndP);  { after last non-space on line so far }
                curC := curC + nSpaces;
                tmp := GetNonWhiteSpace (sourceFirst);
                if GT (tmp, sourceLast) then
                    begin
                    DeleteString (sourceFirst, sourceLast);
                    sourceLast := lineEndP;
                    tmp := lineEndP;
                    end
                else
                    DeleteString (sourceFirst, Add (tmp, -1));
                end; { if }
            sourceFirst := tmp; { next non-space }
            end;  { while }

        if curC >= rightMargin then
            begin
            if DEBUG [5] then Status ('Inserting end of line');
            sourceFirst := lineEndP;  { first character of new line }
            InsertEndOfLine (lineEndP);
            end;
    until GE (sourceFirst, sourceLast);
    end;  { with }
end;  { FillParagraph }


{****************************************************************}

procedure FillCurParagraph;

{ Fill the paragraph that contains the cursor.  A paragraph is delimited
{ by one or more blank lines.  }

var
    first, last:  Position;  { delimits text to be filled }
    beforeP:   Position;  { last position before first }
    curP:  Position;  { current position }
    drawL:  LineIndex;  { redraw screen from here down }
    drawC:  ColumnIndex;

begin  { FillCurParagraph }
Prompt ('Filling paragraph ...');
needPrompt := true;
immedPrompt := true;
with curWin^ do
    begin
    curP := GetPosition (curLine, curCol);
    first := GetBegParagraph (curP);
    GetLC (first, drawL, drawC);
    if DEBUG [5] then Show (first, 0, lastLine);
    if DEBUG [5] then Status ('Beginning of paragraph:    ');
    last := GetEndParagraph (curP);
    if DEBUG [5] then Show (last, 0, lastLine);
    if DEBUG [5] then Status ('End of paragraph:          ');
    beforeP := Add (first, -1);
    FillParagraph (first, last);
    Add1 (beforeP);
    Draw (beforeP, filledLast, drawL, -1);
    end;  { with }
end;  { FillCurParagraph }


{****************************************************************}

procedure FillSelection;

{ Fill the paragraphs that contain the selection.  A paragraph is delimited
{ by one or more blank lines.  }

var
    first:  Position;  { first and justP delimit text to be filled }
        { justP is a fixed up position }
    beforeP:   Position;  { last position before first }
    curP:  Position;  { current position }
    drawL:  LineIndex;  { redraw screen from here down }
    drawC:  ColumnIndex;

label 1;

begin  { FillSelection }
if selectWindow <> curWin then
    begin
    Warn ('No selection in this window');
    exit (FillSelection);
    end;
Prompt ('Filling selection ...');
needPrompt := true;
immedPrompt := true;
with curWin^ do
    begin
    first := GetBegParagraph (bufary [selectB].first);
    display := bufary [selectB].last;
    GetLC (first, drawL, drawC);

    { save a position that won't be deleted }
    justP := Add (GetEndParagraph (first), 1);
    if DEBUG [5] then
        begin
        Show (first, 0, lastLine);
        Status ('Beginning of paragraph:    ');
        Show (justP, 0, lastLine);
        Status ('End of paragraph:          ');
        end;  { if }
    { save a position that won't be deleted }
    beforeP := Add (first, -1);
    FillParagraph (first, Add (justP, -1));
    Add1 (beforeP);

    while LT (justP, display) do
        begin
        first := GetNonWhiteSpace (justP);
        if GT (first, display) then
            goto 1;
        justP := Add (GetEndParagraph (first), 1);
        first := GetBegParagraph (first);
        if DEBUG [5] then
            begin
            Show (first, 0, lastLine);
            Status ('Beginning of paragraph:    ');
            Show (justP, 0, lastLine);
            Status ('End of paragraph:          ');
            end;  { if }
        FillParagraph (first, Add (justP, -1));
        end;  { while }
1:
    Draw (beforeP, filledLast, drawL, -1);
    end;  { with }
end;  { FillSelection }


{****************************************************************}

procedure indentBlock;

{ indent the selected block by the prompted-for value }

var
    first:  Position;  { first and savedPos delimit text to be indented }
    beforeP:   Position;  { last position before first }
    drawL:  LineIndex;  { redraw screen from here down }
    drawC:  ColumnIndex;
    numIndent: integer;

begin  { indentBlock }
  if selectWindow <> curWin then
    begin
      Warn ('No selection in this window');
      exit (indentBlock);
    end;
  prompt('How Many Spaces? ');
  repeat LookForCommand (CanvDflt)
  until NewEvent.Cmd <> NullCmd;
  SendKey (NewEvent);
  if NewEvent.Ch in Numbers then
    numIndent := getNumber(true)
  else
    begin
      prompt('Indent Aborted');
      needPrompt := true;
      exit(indentBlock);
    end;
  MoveTextPointer (CurWin^.CurLine, CurWin^.CurCol);
  if numIndent > 100 then
    begin
      prompt('Absurd Number - Indent Aborted');
      needPrompt := true;
      exit(indentBlock);
    end;
  Prompt ('Indenting Block ...');
  needPrompt := true;
  immedPrompt := true;
  with curWin^ do
    begin
      first := bufary[selectB].first;
      beforeP := add(first, -1);
      attach(cursor2, first, readCursor);
      sub1C(cursor2);
      if cursor2.ch <> LF then
        add1C(cursor2);
      display := bufary[selectB].last;
      GetLC (first, drawL, drawC);
      savedPos := Add(bufary[selectB].last, 1);
      while LT(cursor2.pos, savedPos) do
        begin
          if cursor2.ch = LF then
            begin
              add1C(cursor2);
              if (not EOT(cursor2.pos)) and
                 (cursor2.ch <> CR) and
                 LT(cursor2.pos, savedPos) then
                insertSpaces(cursor2.pos, numIndent);
            end
          else
            add1C(cursor2);
        end;
      detach(cursor2);
      add1(beforeP);
      Draw (beforeP, filledLast, drawL, drawC);
    end;  { with }
end;  { indentBlock }


{****************************************************************}

procedure undentBlock;

{ undent the selected block by the prompted-for value }

var
    first:  Position;  { first and savedPos delimit text to be indented }
    beforeP:   Position;  { last position before first }
    start:  position;  { start of space string to delete }
    drawL:  LineIndex;  { redraw screen from here down }
    drawC:  ColumnIndex;
    numUndent, count: integer;

begin  { undentBlock }
  if selectWindow <> curWin then
    begin
      Warn ('No selection in this window');
      exit (undentBlock);
    end;
  prompt('How Many Spaces? ');
  repeat LookForCommand (CanvDflt)
  until NewEvent.Cmd <> NullCmd;
  SendKey (NewEvent);
  if NewEvent.Ch in Numbers then
    numUndent := getNumber(true)
  else
    begin
      prompt('Undent Aborted');
      needPrompt := true;
      exit(undentBlock);
    end;
  MoveTextPointer (CurWin^.CurLine, CurWin^.CurCol);
  if numUndent > 100 then
    begin
      prompt('Absurd Number - Undent Aborted');
      needPrompt := true;
      exit(undentBlock);
    end;
  Prompt ('Undenting Block ...');
  needPrompt := true;
  immedPrompt := true;
  with curWin^ do
    begin
      first := bufary[selectB].first;
      beforeP := add(first, -1);
      attach(cursor2, first, readCursor);
      sub1C(cursor2);
      if cursor2.ch <> LF then
        add1C(cursor2);
      display := bufary[selectB].last;
      GetLC (first, drawL, drawC);
      savedPos := Add(bufary[selectB].last, 1);
      while LT(cursor2.pos, savedPos) do
        begin
          if cursor2.ch = LF then
            begin
              add1C(cursor2);
              start := cursor2.pos;
              count := 0;
              while (cursor2.ch = ' ') and
                    (count < numUndent) and
                     LT(cursor2.pos, savedPos) do
                begin
                  add1C(cursor2);
                  count := count + 1;
                end;
              if count > 0 then
                deleteString(start, add(cursor2.pos, -1));
            end
          else
            add1C(cursor2);
        end;
      detach(cursor2);
      add1(beforeP);
      Draw (beforeP, filledLast, drawL, drawC);
    end;  { with }
end;  { undentBlock }


{****************************************************************}

procedure CenterLine (first, last: Position);

{ Center the line from first to last using the left and right margins.
{ Assumes that first points to the first character of the paragraph
{ and that last points to the CR of the end of line.  }

var
    newSpaces:  integer;  { number of spaces needed to center }

begin  { CenterLine }
if GE (first, last) then
    exit (CenterLine);
if SelectWindow = curWin then
    if LT (Bufary [SelectB].First, first) then
        begin
        if GE (Bufary [SelectB].Last, first) then
            UnSelect
        end
    else
        if LE (Bufary [SelectB].First, last) then
            UnSelect;
sourceFirst := GetNonWhiteSpace (first);
if GE (sourceFirst, last) then
    { blank line }
    begin
    DeleteString (first, Add (last, -1));
    exit (CenterLine);
    end;
newSpaces := (curWin^.leftMargin + curWin^.rightMargin -
    Subtract (last, sourceFirst)) div 2;
if newSpaces < curWin^.leftMargin then
    newSpaces := curWin^.leftMargin;
InsertMargin (first, sourceFirst, newSpaces);
end;  { CenterLine }


{****************************************************************}
(*  ALREADY DECLARED IN EDINSERT--SUBSTITUTE THIS VERSION FOR THAT ONE!
function GetBegLine (p: Position);

{ Get the first position of the line that contains p.  }

label 1;

begin  { GetBegLine }
Attach (cursor1, p, readCursor);
while not Bot (cursor1.pos) do
    begin
    Sub1C (cursor1);
    if cursor1.ch = LF then
        begin
        Sub1C (cursor1);
        if cursor1.ch = CR then
            begin
            AddC (cursor1, 2);
            goto 1;
            end;
        Add1C (cursor1);
        end;
    end; { while }
1:
GetBegLine := cursor1.pos;
Detach (cursor1);
end;  { GetBegLine }
*)

{****************************************************************}

function GetEndLine (p: Position):  Position;

{ Get the last position of the line that contains p.  It is the CR
{ of the end of line.  }

label 1;

begin  { GetEndLine }
Attach (cursor1, p, readCursor);
while not Eot (cursor1.pos) do
    begin
    if cursor1.ch = CR then
        begin
        Add1C (cursor1);
        if cursor1.ch = LF then
            begin
            Sub1C (cursor1);
            goto 1;
            end;
        Sub1C (cursor1);
        end;
    Add1C (cursor1);
    end; { while }
1:
GetEndLine := cursor1.pos;
Detach (cursor1);
end;  { GetEndLine }


{****************************************************************}

procedure CenterCurLine;

{ Center the current line using the left and right margins.  }

var
    first, last:  Position;  { delimits text to be filled }
    beforeP:   Position;  { last position before first }
    curP:  Position;  { current position }
    drawL:  LineIndex;  { redraw screen from here down }
    drawC:  ColumnIndex;

begin  { CenterCurLine }
with curWin^ do
    begin
    curP := GetPosition (curLine, curCol);
    first := GetBegLine (curP);
    GetLC (first, drawL, drawC);
    last := GetEndLine (curP);
    beforeP := Add (first, -1);
    CenterLine (first, last);
    Add1 (beforeP);
    JoinScreen (beforeP, drawL, drawC);
    end;  { with }
end;  { CenterCurLine }


{****************************************************************}

procedure CenterSelection;

{ Center the lines contained in the current selection.  }

var
    first:  Position;  { first and justP delimit text to be filled }
        { justP is a fixed up position }
    beforeP:   Position;  { last position before first }
    curP:  Position;  { current position }
    drawL:  LineIndex;  { redraw screen from here down }
    drawC:  ColumnIndex;

label 1;

begin  { CenterSelection }
if selectWindow <> curWin then
    begin
    Warn ('No selection in this window');
    exit (CenterSelection);
    end;
with curWin^ do
    begin
    first := GetBegLine (bufary [selectB].first);
    display := bufary [selectB].last;
    GetLC (first, drawL, drawC);
    justP := GetEndLine (first);
    beforeP := Add (first, -1);
    CenterLine (first, justP);
    Add1 (beforeP);
    while LT (justP, display) do
        begin
        first := Add (justP, 2);  { goto first char of next line }
        if GT (first, display) then
            goto 1;
        justP := GetEndLine (first);
        CenterLine (first, justP);
        end;  { while }
1:
    Draw (beforeP, filledLast, drawL, -1);
    end;  { with }
end.  { CenterSelection }
