module Stream;


{-----------------------------------------------------------------------------
{
{       Stream - Perq Pascal stream package.
{       John Strait              ca. Jan 80.
{       Copyright (C) Three Rivers Computer Corporation, 1980.
{
{
{ Abstract:
{       This module implements the low-level Pascal I/O.  It is not
{       intended for use directly by user programs, but rather the
{       compiler generates calls to these routines when a Reset,
{       Rewrite, Get, or Put is encountered.  Higher-level character
{       I/O functions (Read and Write) are implemented by the two
{       modules Reader and Writer.
{
{       In this module, the term "file buffer variable" refers to F^ for
{       a file variable F.
{
{-----------------------------------------------------------------------------}

{-----------------------------------------------------------------------------
{     Change log:

      V1.20  BAM  27 Jan 82  Turn RSX back on.

      V1.19a SLB  26 Jan 82  Add exception definition NotReal.

      V1.18 SLB  07 Jan 82  Add exception definition RealWriteError.

      V1.17 SLB  30 Dec 81  Add an exception definition LargeReal, and change
                            exception definition SmallNumber to SmallReal.
                                  
      V1.16 BAM  17 Dec 81  Turn off RSX switch.
                            Make regular OOPs work.
      
      V1.15 SLB  24 Nov 81  Add an exception definition SmallNumber.
      
      V1.14 JPS  25 Sep 81  Fix bug in control-BS for keyboard input.
      
      V1.13 JPS  27 Jul 81  1) Export the keyboard buffer variables.
                            2) Change line editing to be like the Editor.
                            3) Make Readln work for files with no Eol before
                               the Eof.

      V1.12 BAM   6 Jul 81  Check for IOECBF as well as IOEIOB in RS: output

      V1.11 BAM   1 Jul 81  Changed IO Screen output device name to ScreenOut

      V1.10 JPS  24 Jun 81  1) Remove reverse video control character code.
                            2) Fixed bug in FullLn.
                            3) Made EofCh (^Z) echo at Console.
      
      V1.9  JPS   4 Jun 81  Add Virgil headers and comments.
      
      V1.8  JPS  22 May 81  Make StreamKeyBoardReset accept a Text parameter.
      
      V1.7  BAM  19 May 81  StreamKeyBoardReset turn cursor off.
      
      V1.6  JPS  12 May 81  Use exceptions instead of StreamError.
                            Fix bug when EofCh is typed at console.
                            Use new control-C exceptions.
      
      V1.5  BAM  19 Mar 81  PERQ_String
      
      V1.4  BAM  17 Mar 81  1) Change to use FSIsFSDev on open
                            2) Change to use new Screen ClearChar routine
      
      V1.3  JPS  16 Feb 81  1) Implement FullLn predicate which tells you if
                               there is a full line in the keyboard input
                               buffer.
                            2) Fix bug in reporting "file opened to $ is not
                               a text file" error.
                            3) Import the new Loader module.
                            4) Move $R- to private part.
                            5) Shrink the error routine.
      
      V1.2  DCF  11 Feb 81  Conforms to new PString; compatable with
                            new System and Compiler.
                            
                                  
      V1.1  JPS   6 Nov 80  1) Add RSX: as an output device, RSX:FileName
                               generates
                                        ^QPip FileName=TI:
                               For each character:
                                 a) Flush RS232 input buffer.
                                 b) Send RS232 character.
                                 c) Wait for RS232 echo--in the case of
                                    sending a carriage return, wait for
                                    a line feed.
                               Send ^Z at end of file.
                            2) Add RSX: as an input device, RSX:FileName
                               generates
                                        ^QPip TI:=FileName
                                        ^S
                               For each line of RS232 input:
                                 a) Send a ^Q.
                                 b) Read from RS232 until end of line.
                                 c) Send a ^S.
                               End of file is indicated by a '>' after an
                               end of line, and the ^S is not sent after
                               the end of file is received.
                            3) In Reset and Rewrite allow filenames of the
                               form <device name>:.  This allows dumb
                               programs to unconditionally concatenate
                               extensions onto filenames typed by the
                               user (e.g. Console:.Map, RSX:.List).
                            4) Clean up control character processing some.
                            5) Make RSX: optional at compile time.  See
                               the CompileRSX constant.
-----------------------------------------------------------------------------}

exports


imports FileDefs from FileDefs;


const StreamVersion = '1.20';

      IdentLength = 8;          { significant characters in an identifier }

 
type pStreamBuffer = ^StreamBuffer;

     StreamBuffer = record case integer of  { element size: }
      0: (W: array[0..255] of integer);       { 1 or more words, or > 8 bits }
      1: (B1: packed array[0..0] of 0..1);    { 1 bit }
      2: (B2: packed array[0..0] of 0..3);    { 2 bits }
      3: (B3: packed array[0..0] of 0..7);    { 3 bits }
      4: (B4: packed array[0..0] of 0..15);   { 4 bits }
      5: (B5: packed array[0..0] of 0..31);   { 5 bits }
      6: (B6: packed array[0..0] of 0..63);   { 6 bits }
      7: (B7: packed array[0..0] of 0..127);  { 7 bits }
      8: (B8: packed array[0..0] of 0..255);  { 8 bits }
      9: (C: packed array[0..255] of char);   { for character structured }
      end;
     

     ControlChar = 0..#37;                { ordinal of an ASCII control
                                            character }

     FileKind = (BlockStructured, CharacterStructured);

     FileType = { file of Thing }
      packed record
       Flag: packed record case integer of
        0: (CharReady  : boolean;           { character is in file window }
            FEoln      : boolean;           { end of line flag }
            FEof       : boolean;           { end of file }
            FNotReset  : boolean;           { false if a Reset has been
                                              performed on this file }
            FNotOpen   : boolean;           { false if file is open }
            FNotRewrite: boolean;           { set false if a Rewrite has been
                                              performed on this file }
            FExternal  : boolean;           { not used - will be permanent/temp
                                              file flag }
            FBusy      : boolean;           { IO is in progress }
            FKind      : FileKind);
        1: (skip1     : 0..3;
            ReadError : 0..7);
        2: (skip2     : 0..15;
            WriteError: 0..3)
        end;
       EolCh, EofCh, EraseCh, NoiseCh: ControlChar; {self explanatory}
       OmitCh        : set of ControlChar;
       FileNum       : integer;      { POS file number }
       Index         : integer;      { current word in buffer for un-packed
                                       files, current element for packed
                                       files }
       Length        : integer;      { length of buffer in words for un-
                                       packed files, in elements for packed
                                       files }
       BlockNumber   : integer;      { next logical block number }
       Buffer        : pStreamBuffer;{ I/O buffer }
       LengthInBlocks: integer;      { file length in blocks }
       LastBlockLength:integer;      { last block length in bits }
       SizeInWords   : integer;      { element size in words, 0 means
                                       packed file }
       SizeInBits    : 0..16;        { element size in bits for packed
                                       files }
       ElsPerWord    : 0..16;         { elements per word for packed files }
       Element: { Thing } record case integer of  {The File window}
        1: (C: char);
        2: (W: array[0..0] of integer)
        end
       end;



     ChArray = packed array[1..1] of char; {For read/write character array}


     Identifier = string[IdentLength];
     IdentTable = array[0..1] of Identifier;
     
     
     
     
var StreamSegment: integer;     { Segment buffer for I/O buffers }
    KeyBuffer: packed array[0..255] of char;
    KeyNext, KeyLength: integer;


     
     
 procedure StreamInit( var F: FileType; WordSize, BitSize: integer;
                       CharFile: boolean );
 procedure StreamOpen( var F: FileType; var Name: PathName;
                       WordSize, BitSize: integer; CharFile: boolean;
                       OpenWrite: boolean );
 procedure StreamClose( var F: FileType );
 procedure GetB( var F: Filetype );
 procedure PutB( var F: Filetype );
 procedure GetC( var F: Filetype );
 procedure PutC( var F: FileType );
 procedure PReadln( var F: Filetype );
 procedure PWriteln( var F: Filetype );
 procedure InitStream;
 function StreamName( var F: FileType ): PathName;
 function FullLn( var F: Text ): Boolean;
 procedure StreamKeyBoardReset( var F: Text );


 exception ResetError( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when unable to reset a file--usually file not found but
{       also could be ill-formatted name or bad device name.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception RewriteError( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when unable to rewrite a file--usually file unknown device
{       or partition but also could be ill-formatted name or bad device name.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception NotTextFile( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to open a non-text file to a
{       character-structured device.
{
{ Parameters:
{       FileName - name of the device.
{
{-----------------------------------------------------------------------------}


 exception NotOpen;
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to use a file which is not open.
{
{-----------------------------------------------------------------------------}


 exception NotReset( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to read a file which is open but
{       has not been reset.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception NotRewrite( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to write a file which is open but
{       has not been
{       rewritten.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception PastEof( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to read past the end of the file.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception UnitIOError( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when IOCRead or IOCWrite returns an error status.
{
{ Parameters:
{       FileName - name of the device.
{
{-----------------------------------------------------------------------------}


 exception TimeOutError( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a device times out.
{
{ Parameters:
{       FileName - name of the device.
{
{-----------------------------------------------------------------------------}


 exception UndfDevice;
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to reference a file which is open
{       to a character-structured device, but the device number is bad.
{       In the current system (lacking automatic initialization of file
{       variables), this may be caused by referencing a file which has
{       never been opened.
{
{-----------------------------------------------------------------------------}


 exception NotIdentifier( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an identifier is expected on a file, but something
{       else is encountered.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception NotBoolean( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a boolean is expected on a file, but something
{       else is encountered.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception BadIdTable( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised by ReadIdentifier when the identifier table is bad.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception IdNotUnique( FileName: PathName; Id: Identifier );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when non-unique identifier is read.
{
{ Parameters:
{       FileName - name of the file or device.
{       Id - the identifier which was read.
{
{-----------------------------------------------------------------------------}


 exception IdNotDefined( FileName: PathName; Id: Identifier );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an undefined identifier is read.
{
{ Parameters:
{       FileName - name of the file or device.
{       Id - the identifier which was read.
{
{-----------------------------------------------------------------------------}


 exception NotNumber( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a number is expected on a file, but something
{       else is encountered.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception LargeNumber( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a number is read from a file, but it is too large.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception SmallReal( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a real number is read from a file, but it is too small.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception BadBase( FileName: PathName; Base: Integer );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to read a number with a numeric base
{       that is not in the range 2..36.
{
{ Parameters:
{       FileName - name of the file or device.
{       Base - numeric base (which is not in the range 2..36).
{
{-----------------------------------------------------------------------------}


 exception LargeReal( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a real number is read from a file, but it is too large.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception RealWriteError( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when an attempt is made to write a real number which is 
{       invalid.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}


 exception NotReal( FileName: PathName );
{-----------------------------------------------------------------------------
{
{ Abstract:
{       Raised when a real number is expected on a file, but something
{       else is encountered.
{
{ Parameters:
{       FileName - name of the file or device.
{
{-----------------------------------------------------------------------------}



private
