/*
 * localtime.c
 * Original Author:	Adapted from tzcode maintained by Arthur David Olson.
 *
 * Converts the calendar time pointed to by tim_p into a broken-down time
 * expressed as local time. Returns a pointer to a structure containing the
 * broken-down time.
 */

/*
FUNCTION
<<localtime>>---convert time to local representation

INDEX
	localtime

ANSI_SYNOPSIS
	#include <time.h>
	struct tm *localtime(time_t *<[timep]>);

TRAD_SYNOPSIS
	#include <time.h>
	struct tm *localtime(<[timep]>)
	time_t *<[timep]>;

DESCRIPTION
<<localtime>> converts the time at <[timep]> into local time, then
converts its representation from the arithmetic representation to the
traditional representation defined by <<struct tm>>.

<<localtime>> constructs the traditional time representation in static
storage; each call to <<gmtime>> or <<localtime>> will overwrite the
information generated by previous calls to either function.

<<mktime>> is the inverse of <<localtime>>.

RETURNS
A pointer to the traditional time representation (<<struct tm>>).

PORTABILITY
ANSI C requires <<localtime>>.

<<localtime>> requires no supporting OS subroutines.
*/

#include <stdlib.h>
#include <time.h>

#define SECSPERMIN	60
#define MINSPERHOUR	60
#define HOURSPERDAY	24
#define SECSPERHOUR	(SECSPERMIN * MINSPERHOUR)
#define SECSPERDAY	(SECSPERHOUR * HOURSPERDAY)
#define DAYSPERWEEK	7
#define MONSPERYEAR	12

#define YEAR_BASE	1900
#define EPOCH_YEAR      1970
#define EPOCH_WDAY      4

#define isleap(y) ((((y) % 4) == 0 && ((y) % 100) != 0) || ((y) % 400) == 0)

static _CONST int mon_lengths[2][MONSPERYEAR] = {
  {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
  {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}
} ;

static _CONST int year_lengths[2] = {
  365,
  366
} ;


/* Shouldn't this appear in the reent struct? */ 
static struct tm tim_s;

struct tm *
_DEFUN (localtime, (tim_p),
	_CONST time_t * tim_p)
{
  long days, rem;
  int y;
  int yleap;
  _CONST int *ip;

  days = ((long) *tim_p) / SECSPERDAY;
  rem = ((long) *tim_p) % SECSPERDAY;
  while (rem < 0) 
    {
      rem += SECSPERDAY;
      --days;
    }
  while (rem >= SECSPERDAY)
    {
      rem -= SECSPERDAY;
      ++days;
    }
 
  /* compute hour, min, and sec */  
  tim_s.tm_hour = (int) (rem / SECSPERHOUR);
  rem %= SECSPERHOUR;
  tim_s.tm_min = (int) (rem / SECSPERMIN);
  tim_s.tm_sec = (int) (rem % SECSPERMIN);

  /* compute day of week */
  if ((tim_s.tm_wday = ((EPOCH_WDAY + days) % DAYSPERWEEK)) < 0)
    tim_s.tm_wday += DAYSPERWEEK;

  /* compute year & day of year */
  y = EPOCH_YEAR;
  if (days >= 0)
    {
      for (;;)
	{
	  yleap = isleap(y);
	  if (days < year_lengths[yleap])
	    break;
	  y++;
	  days -= year_lengths[yleap];
	}
    }
  else
    {
      do
	{
	  --y;
	  yleap = isleap(y);
	  days += year_lengths[yleap];
	} while (days < 0);
    }

  tim_s.tm_year = y - YEAR_BASE;
  tim_s.tm_yday = days;
  ip = mon_lengths[yleap];
  for (tim_s.tm_mon = 0; days >= ip[tim_s.tm_mon]; ++tim_s.tm_mon)
    days -= ip[tim_s.tm_mon];
  tim_s.tm_mday = days + 1;

  /* set daylight saving time flag */
  tim_s.tm_isdst = -1;
  
  return (&tim_s);
}
