
#include "defs.h"
#include "gdbcmd.h"
#include "serial.h"



static int
dos_getc (port)
     volatile struct dos_ttystate *port;
{
    int c;

    if (port->count == 0)
      return -1;

    c = port->cbuf[port->first];
    disable ();
    port->first = (port->first + 1) & (CBSIZE - 1);
    port->count--;
    enable ();
    return c;
}
    

static int 
dos_putc (c, port)
     int c;
     struct dos_ttystate *port;
{
    if (port->count >= CBSIZE - 1)
	return -1;
    port->cbuf[(port->first + port->count) & (CBSIZE - 1)] = c;
    port->count++;
    return 0;
}



static void
dos_comisr (irq)
     int irq;
{
  struct dos_ttystate *port;
  unsigned char iir, lsr, c;

  disable ();			/* Paranoia */
  outportb (ICU_OCW2, 0x20);	/* End-Of-Interrupt */
#ifdef DOS_STATS
  ++intrcnt;
#endif

  port = intrupts[irq].port;
  if (!port) 
    {
      COUNT (CNT_STRAY);
      return;		/* not open */
    }

  while (1)
    {
      iir = inb (port, com_iir) & IIR_IMASK;
      switch (iir) 
	{
	  
	case IIR_RLS:
	  lsr = inb (port, com_lsr);
	  goto rx;
	  
	case IIR_RXTOUT:
	case IIR_RXRDY:
	  lsr = 0;
	  
      rx:
	  do 
	    {
	      c = inb (port, com_data);
	      if (lsr & (LSR_BI | LSR_FE | LSR_PE | LSR_OE))
		{
		  if (lsr & (LSR_BI | LSR_FE))
		    port->ferr++;
		  else if (lsr & LSR_PE)
		    port->perr++;
		  if (lsr & LSR_OE)
		    port->oflo++;
		}

	      if (dos_putc (c, port) < 0)
		{
		  COUNT (CNT_ORUN);
		}
	      else
		{
		  COUNT (CNT_RX);
		}
	    }
	  while ((lsr = inb (port, com_lsr)) & LSR_RXRDY);
	  break;
	  
	case IIR_MLSC:
	  /* could be used to flowcontrol Tx */
	  port->msr = inb (port, com_msr);
	  break;
	  
	case IIR_TXRDY:
	  port->txbusy = 0;
	  break;

	case IIR_NOPEND:
	  /* no more pending interrupts, all done */
	  return;

	default:
	  /* unexpected interrupt, ignore */
	  break;
	}
      COUNT (iir);
    } 
}

#ifdef __STDC__
#define ISRNAME(x) dos_comisr##x
#else
#define ISRNAME(x) dos_comisr/**/x
#endif
#define ISR(x) static void ISRNAME(x)() {dos_comisr(x);}

ISR(0) ISR(1) ISR(2) ISR(3)
ISR(4) ISR(5) ISR(6) ISR(7)

typedef void (*isr_t)();

static isr_t isrs[NINTR] = {
  ISRNAME(0), ISRNAME(1), ISRNAME(2), ISRNAME(3),
  ISRNAME(4), ISRNAME(5), ISRNAME(6), ISRNAME(7)
};



static struct intrupt *
dos_hookirq (irq)
     unsigned int irq;
{
  struct intrupt *intr;
  unsigned int vec;
  isr_t isr;

  if (irq >= NINTR)
    return 0;

  intr = &intrupts[irq];
  if (intr->inuse)
    return 0;
  
  vec = 0x08 + irq;
  isr = isrs[irq];

  /* setup real mode handler */
  _go32_dpmi_get_real_mode_interrupt_vector (vec, &intr->old_rmhandler);

  intr->new_rmhandler.pm_selector = _go32_my_cs();
  intr->new_rmhandler.pm_offset = (u_long)isr;
  if (_go32_dpmi_allocate_real_mode_callback_iret (&intr->new_rmhandler,
						   &intr->regs))
    {
      return 0;
    }

  if (_go32_dpmi_set_real_mode_interrupt_vector (vec, &intr->new_rmhandler))
    {
      return 0;
    }
      
  /* setup protected mode handler */
  _go32_dpmi_get_protected_mode_interrupt_vector(vec, &intr->old_pmhandler);

  intr->new_pmhandler.pm_selector = _go32_my_cs();
  intr->new_pmhandler.pm_offset = (u_long)isr;
  _go32_dpmi_allocate_iret_wrapper (&intr->new_pmhandler);

  if (_go32_dpmi_set_protected_mode_interrupt_vector(vec, &intr->new_pmhandler))
    {
      return 0;
    }

  /* setup interrupt controller mask */
  disable ();
  outportb (ICU_MASK, inportb (ICU_MASK) & ~(1 << irq));
  enable ();

  intr->inuse = 1;
  return intr;
}


static void
dos_unhookirq (intr)
     struct intrupt *intr;
{
  unsigned int irq, vec;
  unsigned char mask;

  irq = intr - intrupts;
  vec = 0x08 + irq;

  /* restore old interrupt mask bit */
  mask = 1 << irq;
  disable ();
  outportb (ICU_MASK, inportb (ICU_MASK) | (mask & icu_oldmask));
  enable ();

  /* remove real mode handler */
  _go32_dpmi_set_real_mode_interrupt_vector (vec, &intr->old_rmhandler);
  _go32_dpmi_free_real_mode_callback (&intr->new_rmhandler);
      
  /* remove protected mode handler */
  _go32_dpmi_set_protected_mode_interrupt_vector (vec, &intr->old_pmhandler);
  _go32_dpmi_free_iret_wrapper (&intr->new_pmhandler);
  intr->inuse = 0;
}



static int
dos_open (scb, name)
     serial_t scb;
     const char *name;
{
  struct dos_ttystate *port;
  int fd, i;

  if (strncasecmp (name, "/dev/", 5) == 0)
    name += 5;
  else if (strncasecmp (name, "\\dev\\", 5) == 0)
    name += 5;

  if (strlen (name) != 4 || strncasecmp (name, "com", 3) != 0)
    {
      errno = ENOENT;
      return -1;
    }

  if (name[3] < '1' || name[3] > '4')
    {
      errno = ENOENT;
      return -1;
    }

  fd = name[3] - '1';
  port = &ports[fd];
  if (port->refcnt++ > 0)
    {
      /* Device already opened another user.  Just point at it. */
      scb->fd = fd;
      return 0;
    }

  /* force access to ID reg */
  outb(port, com_cfcr, 0);
  outb(port, com_iir, 0);
  for (i = 0; i < 17; i++) {
    if ((inb(port, com_iir) & 0x38) == 0)
      goto ok;
    (void) inb(port, com_data); /* clear recv */
  }
  errno = ENODEV;
  return -1;

ok:
  /* disable all interrupts in chip */
  outb(port, com_ier, 0);

  /* tentatively enable 16550 fifo, and see if it responds */
  outb(port, com_fifo, FIFO_ENABLE|FIFO_RCV_RST|FIFO_XMT_RST|FIFO_TRIGGER);
  sleep(1);
  port->fifo = ((inb(port, com_iir) & IIR_FIFO_MASK) == IIR_FIFO_MASK);

  /* clear pending status reports. */
  (void) inb(port, com_lsr);
  (void) inb(port, com_msr);

  /* enable external interrupt gate (to avoid floating IRQ) */
  outb(port, com_mcr, MCR_IENABLE);

  /* hook up interrupt handler and initialise icu */
  port->intrupt = dos_hookirq (port->irq);
  if (!port->intrupt)
    {
      outb(port, com_mcr, 0);
      outb(port, com_fifo, 0);
      errno = ENODEV;
      return -1;
    }

  disable ();

  /* record port */
  port->intrupt->port = port; 
  scb->fd = fd;

  /* clear rx buffer, tx busy flag and overflow count */
  port->first = port->count = 0;
  port->txbusy = 0;
  port->oflo = 0;

  /* set default baud rate and mode: 9600,8,n,1 */
  i = dos_baudconv (port->baudrate = 9600);
  outb(port, com_cfcr, CFCR_DLAB);
  outb(port, com_dlbl, i & 0xff);
  outb(port, com_dlbh, i >> 8);
  outb(port, com_cfcr, CFCR_8BITS);

  /* enable all interrupts */
  outb(port, com_ier, IER_ETXRDY | IER_ERXRDY | IER_ERLS | IER_EMSC);

  /* enable DTR & RTS */
  outb(port, com_mcr, MCR_DTR | MCR_RTS | MCR_IENABLE);

  enable ();

  return 0;
}


static void
dos_close (scb)
     serial_t scb;
{
    struct dos_ttystate *port;
    struct intrupt *intrupt;

    if (!scb)
      return;

    port = &ports[scb->fd];

    if (port->refcnt-- > 1)
      return;

    if (!(intrupt = port->intrupt))
      return;

    /* disable interrupts, fifo, flow control */
    disable ();
    port->intrupt = 0;
    intrupt->port = 0;
    outb(port, com_fifo, 0);
    outb(port, com_ier, 0);
    enable ();

    /* unhook handler, and disable interrupt gate */
    dos_unhookirq (intrupt);
    outb(port, com_mcr, 0);

    /* Check for overflow errors */
    if (port->oflo)
      {
	fprintf_unfiltered (gdb_stderr,
			    "Serial input overruns occurred.\n");
	fprintf_unfiltered (gdb_stderr, "This system %s handle %d baud.\n",
			    port->fifo ? "cannot" : "needs a 16550 to",
			    port->baudrate);
      }
}



static int
dos_noop (scb)
     serial_t scb;
{
  return 0;
}

static void
dos_raw (scb)
     serial_t scb;
{
  /* Always in raw mode */
}

static int
dos_readchar (scb, timeout)
     serial_t scb;
     int timeout;
{
  struct dos_ttystate *port = &ports[scb->fd];
  long then;
  int c;

  then = rawclock() + (timeout * RAWHZ);
  while ((c = dos_getc (port)) < 0)
    {
      if (timeout >= 0 && (rawclock () - then) >= 0)
	return SERIAL_TIMEOUT;
      notice_quit ();
    }

  return c;
}


static serial_ttystate
dos_get_tty_state (scb)
     serial_t scb;
{
  struct dos_ttystate *port = &ports[scb->fd];
  struct dos_ttystate *state;

  state = (struct dos_ttystate *) xmalloc (sizeof *state);
  *state = *port;
  return (serial_ttystate) state;
}

static int
dos_set_tty_state (scb, ttystate)
     serial_t scb;
     serial_ttystate ttystate;
{
  struct dos_ttystate *state;

  state = (struct dos_ttystate *) ttystate;
  dos_setbaudrate (scb, state->baudrate);
  return 0;
}

static int
dos_noflush_set_tty_state (scb, new_ttystate, old_ttystate)
     serial_t scb;
     serial_ttystate new_ttystate;
     serial_ttystate old_ttystate;
{
  struct dos_ttystate *state;

  state = (struct dos_ttystate *) new_ttystate;
  dos_setbaudrate (scb, state->baudrate);
  return 0;
}

static int
dos_flush_input (scb)
     serial_t scb;
{
  struct dos_ttystate *port = &ports[scb->fd];
  disable();
  port->first = port->count = 0;
  if (port->fifo)
    outb(port, com_fifo, FIFO_ENABLE|FIFO_RCV_RST|FIFO_TRIGGER);
  enable();
}

static void
dos_print_tty_state (scb, ttystate)
     serial_t scb;
     serial_ttystate ttystate;
{
  /* Nothing to print */
  return;
}

static int
dos_baudconv (rate)
     int rate;
{
  long x, err;
  
  if (rate <= 0) 
    return -1;

#define divrnd(n, q)	(((n) * 2 / (q) + 1) / 2) /* divide and round off */
  x = divrnd(COMTICK, rate);
  if (x <= 0)
    return -1;
  
  err = divrnd(1000 * COMTICK, x * rate) - 1000;
  if (err < 0)
    err = -err;
  if (err > SPEED_TOLERANCE)
    return -1;
#undef divrnd
  return x;
}


static int
dos_setbaudrate (scb, rate)
     serial_t scb;
     int rate;
{
    struct dos_ttystate *port = &ports[scb->fd];

    if (port->baudrate != rate) 
      {
	int x;
	unsigned char cfcr;

	x = dos_baudconv (rate);
	if (x <= 0)
	  {
	    fprintf_unfiltered (gdb_stderr, "%d: impossible baudrate\n", rate);
	    errno = EINVAL;
	    return -1;
	  }

	disable ();
	cfcr = inb (port, com_cfcr);

	outb(port, com_cfcr, CFCR_DLAB);
	outb(port, com_dlbl, x & 0xff);
	outb(port, com_dlbh, x >> 8);
	outb(port, com_cfcr, cfcr);
	port->baudrate = rate;
	enable ();
      }

    return 0;
}

static int
dos_setstopbits (scb, num)
     serial_t scb;
     int num;
{
    struct dos_ttystate *port = &ports[scb->fd];
    unsigned char cfcr;

    disable ();
    cfcr = inb (port, com_cfcr);

    switch (num)
      {
      case SERIAL_1_STOPBITS:
	outb (port, com_cfcr, cfcr & ~CFCR_STOPB);
	break;
      case SERIAL_1_AND_A_HALF_STOPBITS:
      case SERIAL_2_STOPBITS:
	outb (port, com_cfcr, cfcr | CFCR_STOPB);
	break;
      default:
	enable ();
	return 1;
      }
    enable ();

    return 0;
}

static int
dos_write (scb, str, len)
     serial_t scb;
     const char *str;
     int len;
{
  volatile struct dos_ttystate *port = &ports[scb->fd];
  int fifosize = port->fifo ? 16 : 1;
  long then;
  int cnt;

   while (len > 0) 
     {
	/* send the data, fifosize bytes at a time */
	cnt = fifosize > len ? len : fifosize;
	port->txbusy = 1;
	outportsb (port->base + com_data, str, cnt);
	str += cnt;
	len -= cnt;
#ifdef DOS_STATS
	cnts[CNT_TX] += cnt;
#endif
	/* wait for transmission to complete (max 1 sec) */
	then = rawclock() + RAWHZ;
	while (port->txbusy)
	  {
	    if ((rawclock () - then) >= 0)
	      {
		  errno = EIO;
		  return SERIAL_ERROR;
	      }
	  }
    }
  return 0;
}


static int
dos_sendbreak (scb)
     serial_t scb;
{
  volatile struct dos_ttystate *port = &ports[scb->fd];
  unsigned char cfcr;
  long then;

  cfcr = inb(port, com_cfcr);
  outb(port, com_cfcr, cfcr | CFCR_SBREAK);

  /* 0.25 sec delay */
  then = rawclock () + RAWHZ / 4;
  while ((rawclock () - then) < 0)
    continue;

  outb(port, com_cfcr, cfcr);
  return 0;
}


static struct serial_ops dos_ops =
{
  "hardwire",
  0,
  dos_open,
  dos_close,
  dos_readchar,
  dos_write,
  dos_noop,			/* flush output */
  dos_flush_input,
  dos_sendbreak,
  dos_raw,
  dos_get_tty_state,
  dos_set_tty_state,
  dos_print_tty_state,
  dos_noflush_set_tty_state,
  dos_setbaudrate,
  dos_setstopbits,
};


static void
dos_info (arg, from_tty)
     char *arg;
     int from_tty;
{
  struct dos_ttystate *port;
  int i;

  for (port = ports; port < &ports[4]; port++) 
    {
      if (port->baudrate == 0)
	continue;
      printf_filtered ("Port:\tCOM%d (%sactive)\n", port - ports + 1,
		       port->intrupt ? "" : "not ");
      printf_filtered ("Addr:\t0x%03x (irq %d)\n", port->base, port->irq);
      printf_filtered ("16550:\t%s\n", port->fifo ? "yes" : "no");
      printf_filtered ("Speed:\t%d baud\n", port->baudrate);
      printf_filtered ("Errs:\tframing %d parity %d overflow %d\n\n", 
		       port->ferr, port->perr, port->oflo);
    }

#ifdef DOS_STATS
  printf_filtered ("\nTotal interrupts: %d\n", intrcnt);
  for (i = 0; i < NCNT; i++)
    if (cnts[i])
      printf_filtered ("%s:\t%d\n", cntnames[i], cnts[i]);
#endif
}


void
_initialize_ser_dos ()
{
  struct cmd_list_element *c;

  serial_add_interface (&dos_ops);

  /* Save original interrupt mask register. */
  icu_oldmask = inportb (ICU_MASK);

  /* Mark fixed motherboard irqs as inuse. */
  intrupts[0].inuse =		/* timer tick */
    intrupts[1].inuse =		/* keyboard */
      intrupts[2].inuse = 1;	/* slave icu */
    
  add_show_from_set (
    add_set_cmd ("com1base", class_obscure, var_zinteger,
		 (char *) &ports[0].base,
		 "Set COM1 base i/o port address.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com1irq", class_obscure, var_zinteger,
		 (char *) &ports[0].irq,
		 "Set COM1 interrupt request.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com2base", class_obscure, var_zinteger,
		 (char *) &ports[1].base,
		 "Set COM2 base i/o port address.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com2irq", class_obscure, var_zinteger,
		 (char *) &ports[1].irq,
		 "Set COM2 interrupt request.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com3base", class_obscure, var_zinteger,
		 (char *) &ports[2].base,
		 "Set COM3 base i/o port address.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com3irq", class_obscure, var_zinteger,
		 (char *) &ports[2].irq,
		 "Set COM3 interrupt request.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com4base", class_obscure, var_zinteger,
		 (char *) &ports[3].base,
		 "Set COM4 base i/o port address.",
		 &setlist),
	&showlist);

  add_show_from_set (
    add_set_cmd ("com4irq", class_obscure, var_zinteger,
		 (char *) &ports[3].irq,
		 "Set COM4 interrupt request.",
		 &setlist),
	&showlist);

  add_info ("serial", dos_info,
	    "Print DOS serial port status.");
}
