/* Low level Alpha interface, for GDB when running native.
   Copyright 1993, 1995 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "defs.h"
#include "inferior.h"
#include "gdbcore.h"
#include "target.h"
#include <sys/ptrace.h>
#include <machine/reg.h>
#include <machine/frame.h>
#include <string.h>

/* Size of elements in jmpbuf */

#define JB_ELEMENT_SIZE 8

/* The definition for JB_PC in machine/reg.h is wrong.
   And we can't get at the correct definition in setjmp.h as it is
   not always available (eg. if _POSIX_SOURCE is defined which is the
   default). As the defintion is unlikely to change (see comment
   in <setjmp.h>, define the correct value here.  */

#undef JB_PC
#define JB_PC 2

/* Figure out where the longjmp will land.
   We expect the first arg to be a pointer to the jmp_buf structure from which
   we extract the pc (JB_PC) that we will land at.  The pc is copied into PC.
   This routine returns true on success. */

int
get_longjmp_target (pc)
     CORE_ADDR *pc;
{
  CORE_ADDR jb_addr;
  char raw_buffer[MAX_REGISTER_RAW_SIZE];

  jb_addr = read_register(A0_REGNUM);

  if (target_read_memory(jb_addr + JB_PC * JB_ELEMENT_SIZE, raw_buffer,
			 sizeof(CORE_ADDR)))
    return 0;

  *pc = extract_address (raw_buffer, sizeof(CORE_ADDR));
  return 1;
}

/* Extract the register values out of the core file and store
   them where `read_register' will find them.

   CORE_REG_SECT points to the register values themselves, read into memory.
   CORE_REG_SIZE is the size of that area.
   WHICH says which set of registers we are handling (0 = int, 2 = float
         on machines where they are discontiguous).
   REG_ADDR is the offset from u.u_ar0 to the register values relative to
            core_reg_sect.  This is used with old-fashioned core files to
	    locate the registers in a large upage-plus-stack ".reg" section.
	    Original upage address X is at location core_reg_sect+x+reg_addr.
 */

/* XXX From alpha /sys/arch/alpha/alpha/vm_machdep.c */
struct cpustate {
	struct trapframe regs;
	struct fpreg fpstate;
};
#define	ogr(name) \
	    offsetof(struct cpustate, regs.tf_regs[__CONCAT(FRAME_,name)])
#define og(elem) \
	    offsetof(struct cpustate, regs.__CONCAT(tf_,elem))
#define ofr(num) \
	    offsetof(struct cpustate, fpstate.fpr_regs[num])

void
fetch_core_registers (core_reg_sect, core_reg_size, which, reg_addr)
     char *core_reg_sect;
     unsigned core_reg_size;
     int which;
     unsigned reg_addr;
{
  register int regno;
  register int addr;
  int bad_reg = -1;
  static char zerobuf[MAX_REGISTER_RAW_SIZE] = {0};
  int regoff[NUM_REGS] = {
      ogr(V0), ogr(T0), ogr(T1), ogr(T2), ogr(T3), ogr(T4), ogr(T5), ogr(T6),
      ogr(T7), ogr(S0), ogr(S1), ogr(S2), ogr(S3), ogr(S4), ogr(S5), ogr(S6),
      og(a0), og(a1), og(a2),  ogr(A3), ogr(A4), ogr(A5), ogr(T8), ogr(T9),
      ogr(T10), ogr(T11), ogr(RA), ogr(T12), ogr(AT), og(gp), ogr(SP), -1,
      ofr(0),  ofr(1),  ofr(2),  ofr(3),  ofr(4),  ofr(5),  ofr(6),  ofr(7),
      ofr(8),  ofr(9),  ofr(10), ofr(11), ofr(12), ofr(13), ofr(14), ofr(15),
      ofr(16), ofr(17), ofr(18), ofr(19), ofr(20), ofr(21), ofr(22), ofr(23),
      ofr(24), ofr(25), ofr(26), ofr(27), ofr(28), ofr(29), ofr(30), ofr(31),
      og(pc), -1,
  };

  for (regno = 0; regno < NUM_REGS; regno++)
    {
      if (CANNOT_FETCH_REGISTER (regno))
	{
	  supply_register (regno, zerobuf);
	  continue;
	}
      addr = regoff[regno];
      if (addr < 0 || addr >= core_reg_size)
	{
	  if (bad_reg < 0)
	    bad_reg = regno;
	}
      else
	{
	  supply_register (regno, core_reg_sect + addr);
	}
    }
  if (bad_reg >= 0)
    {
      error ("Register %s not found in core file.", reg_names[bad_reg]);
    }
}

register_t
rrf_to_register(regno, reg, fpreg)
	int regno;
	struct reg *reg;
	struct fpreg *fpreg;
{

	if (regno < 0)
		abort();
	else if (regno < FP0_REGNUM)
		return (reg->r_regs[regno]);
	else if (regno == PC_REGNUM)
		return (reg->r_regs[R_ZERO]);
	else if (regno >= FP0_REGNUM)
		return (fpreg->fpr_regs[regno - FP0_REGNUM]);
	else
		abort();
}

void
fetch_inferior_registers (regno)
	int regno;
{
	struct reg reg;
	struct fpreg fpreg;
	register_t regval;
	static char zerobuf[MAX_REGISTER_RAW_SIZE] = {0};
	char *rp;

	ptrace(PT_GETREGS, inferior_pid, (PTRACE_ARG3_TYPE)&reg, 0);
	ptrace(PT_GETFPREGS, inferior_pid, (PTRACE_ARG3_TYPE)&fpreg, 0);

	if (regno < 0) {
		for (regno = 0; regno < NUM_REGS; regno++) {
			if (CANNOT_FETCH_REGISTER (regno))
				rp = zerobuf;
			else {
				regval = rrf_to_register(regno, &reg, &fpreg);
				rp = (char *)&regval;
			}
			supply_register(regno, rp);
		}
	} else {
		if (CANNOT_FETCH_REGISTER (regno))
			rp = zerobuf;
		else {
			regval = rrf_to_register(regno, &reg, &fpreg);
			rp = (char *)&regval;
		}

		supply_register(regno, rp);
	}
}

void
register_into_rrf(val, regno, reg, fpreg)
	register_t val;
	int regno;
	struct reg *reg;
	struct fpreg *fpreg;
{

	if (regno < 0)
		abort();
	else if (regno < FP0_REGNUM)
		reg->r_regs[regno] = val;
	else if (regno == PC_REGNUM)
		reg->r_regs[R_ZERO] = val;
	else if (regno >= FP0_REGNUM)
		fpreg->fpr_regs[regno - FP0_REGNUM] = val;
	else
		abort();
}

void
store_inferior_registers (regno)
	int regno;
{
	struct reg reg;
	struct fpreg fpreg;
	register_t regval;

	if (regno < 0) {
		for (regno = 0; regno < NUM_REGS; regno++) {
			if (CANNOT_STORE_REGISTER (regno))
				continue;
	
			if (REGISTER_RAW_SIZE (regno) != sizeof regval)
				abort();
			memcpy(&regval, &registers[REGISTER_BYTE (regno)],
			    REGISTER_RAW_SIZE (regno));
			register_into_rrf(regval, regno, &reg, &fpreg);
		}
	} else {
		ptrace(PT_GETREGS, inferior_pid, (PTRACE_ARG3_TYPE)&reg, 0);
		ptrace(PT_GETFPREGS, inferior_pid, (PTRACE_ARG3_TYPE)&fpreg, 0);
		
		memcpy(&regval, &registers[REGISTER_BYTE (regno)],
		    REGISTER_RAW_SIZE (regno));
		register_into_rrf(regval, regno, &reg, &fpreg);
	}
	
	ptrace(PT_SETREGS, inferior_pid, (PTRACE_ARG3_TYPE)&reg, 0);
	ptrace(PT_SETFPREGS, inferior_pid, (PTRACE_ARG3_TYPE)&fpreg, 0);
}

void 
child_resume (pid, step, signal)
	int pid;
	int step;
	enum target_signal signal;
{    

  errno = 0;

  if (pid == -1)
    /* Resume all threads.  */
    /* I think this only gets used in the non-threaded case, where "resume
       all threads" and "resume inferior_pid" are the same.  */
    pid = inferior_pid;

  /* An address of (PTRACE_ARG3_TYPE)1 tells ptrace to continue from where
     it was.  (If GDB wanted it to start some other way, we have already
     written a new PC value to the child.)

     If this system does not support PT_STEP, a higher level function will
     have called single_step() to transmute the step request into a
     continue request (by setting breakpoints on all possible successor
     instructions), so we don't have to worry about that here.  */

  if (step)
	abort();
  else
    ptrace (PT_CONTINUE, pid, (PTRACE_ARG3_TYPE) 1,
	    target_signal_to_host (signal));

  if (errno)
    perror_with_name ("ptrace");
}
